/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "linsolve_umfpack.h"
#include "assembly.h"
#include "assem_conv_diff.h"
#include "gen_aux.h"
#include "mesh_deform.h"
#include "lame_adj.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

//#include <unistd.h>
#include <string.h>

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }

int main(int argc, char *argv[])
{

  struct sparse K;
  struct projector1 P;
  
  struct vector rhs, x;

  struct mesh msh1;
  struct mesh msh2;

  struct solver_settings set;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t0, ti;

  int  err;
  FIDX dim, vx_nr, i, j;

  FIDX level0, lvlm, lvl;

  double *vx_new;
  FIDX vx_max;

  double eps, maxPe;

  double *vx_tmp;
  
  int iter=0;
  double resi=0.0;

  char *buffer;
  int solver;
  
#ifdef HAVE_OPENMP
#pragma omp parallel
  {
#pragma omp master
    {
      printf("OpenMP num_threads=%d\n",omp_get_num_threads());
      printf("  should be number of phyisical cores, use environment variable "
	     " OMP_NUM_THREADS to modify,\n");
      printf("  e.g. (bash shell): \"export OMP_NUM_THREADS=4\" \n");
      printf("  e.g. (most other): \"set OMP_NUM_THREADS 4\" \n");
    }
  }
#endif
  
  if (argc>1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* set the goals and methods */
  level0  =  set.refine_ini;
  lvlm    =  set.refine_steps;

  dim=msh1.dim;

  eps=msh1.para[0];

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }


  /* initialise the deformation */
  err=mesh_deform_t1_t2(&msh1, lvlm, &vx_new, &vx_max, 0, NULL);
  FUNCTION_FAILURE_HANDLE( err, mesh_deform_t2_t1, main);

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

  /* refine */
  for (lvl=0; lvl<lvlm; lvl++)
    {
      err=mesh_refine_uniform_t2( &msh2 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
    }


  vx_nr=msh2.vx_nr; 

  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);



  /* allocate memory for matrix K and the vectors */
  err=sparse_flex_alloc( &K, vx_nr, FEINS_SPARSE_COLS_P_ROW_T2);
  err=projector1_alloc( &P, vx_nr );
  err=vector_alloc( &x, vx_nr );
  err=vector_alloc( &rhs, vx_nr );
  
  /* set vectors to zero */
  for (i=0; i<vx_nr; i++)
    {
      x.V[i]    =0.0;
      rhs.V[i]  =0.0;
    }

  vx_tmp=msh2.vertex;
  msh2.vertex=vx_new;

  /* assemble the system on the fine mesh */
  {
    //double eps[]={1e-1, 1e-6, 1e-6, 1e-7};
    double eps[]={1e-4, 0, 0, 1e-4};
    err=assem_conv_diff_tx( NULL, &K,  &rhs, NULL, &maxPe, eps,
			    NULL, NULL, 0.0, 0.0, 0.0, 0.0,
			    &msh2, 1, 0.0, 2);
    FUNCTION_FAILURE_HANDLE( err, assem_conv_diff_tx, main);
  }

  /* write matrix for debuging */
  /* err=sparse_mat_write_file( &K,  "visual/conv_diff_mat.txt");
     FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file, main); /* */


  /* set the Dirichlet boundary conditions */
  err=assem_Dirichlet_BC_tx( &P, &msh2, 1, &x, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_Dirichlet_BC_tx, main);

  

  /* solve the equation system */
  solver=3;
  switch (solver)
    {
    case 1: /* variant 1: direct solve with cmat (good for coarse mesh) */
      {
	struct coarse_mat K_inv;
	struct vector helpv;

	TIMEGET;
	t0=ti;

	err=vector_alloc(&helpv, x.len);
	FUNCTION_FAILURE_HANDLE( err, vector_alloc, main );

	/* invert the stiffness matrix */
	err=coarse_mat_set( &K, P.len, P.V,
			    2, &K_inv );
	FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	
	/* modify the rhs */
	/* helpv=K*x */
	err=sparse_mul_mat_vec( &K, &x, &helpv);
	FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, main);  
	/* rhs=rhs-helpv */
	for (i=0; i<rhs.len; i++)
	  {
	    rhs.V[i]-=helpv.V[i];
	  }

	/* project rhs */
	for (i=0; i<P.len; i++)
	  {
	    FIDX dof = P.V[i];
	    rhs.V[dof]=0.0;
	  }

	/* solve the projected system */
	err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &helpv);
	FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	/* apply the solution */
	for (i=0; i<vx_nr; i++)
	  {
	    x.V[i]+=helpv.V[i];
	  }
	TIMEGET;
	printf("solver cmat,    size=%6"dFIDX", t_solver=%10.3e\n",
	       rhs.len, ti-t0);


	coarse_mat_free(&K_inv);
	vector_free(&helpv);
      }
      break;
    case 2:/* variant 2: PCG */
      {
	TIMEGET;
	t0=ti;

	err=FAIL; /* PCG( 10000, 2, set.solver_atol,
		    set.solver_ini_rtol, 1, &nodem, &resi, &iter, 
		    sparse_mul_mat_vec, gen_proj_MG_tx,
		    &Ks[lvlm], &rhs, &mg); /* */
	FUNCTION_FAILURE_HANDLE( err, PCG, main);


	printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
	TIMEGET;
	printf("solver PCG MG,  size=%6"dFIDX", t_solver=%10.3e\n",
	       rhs.len, ti-t0);
      }
      break;
    case 3:/* variant 3: direct solver UMFPACK */
      /* variant 3: direct solver UMFPACK */
      {
	TIMEGET;
	t0=ti;
    
	err=sparse_solve_UMFPACK( &x, &K, &rhs, &P); /* */
	FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK, main);
	
	TIMEGET;
	printf("solver UMFPACK, size=%6"dFIDX", t_solver=%10.3e\n", 
	       rhs.len, ti-t0);
      }
      break;
    default:
      fprintf(stderr,"main: unknown solver requested\n");
      return FAIL;
    }

  /* write the solution for paraview visualisation */
  err=mesh_write_solution_vtk_tx( &msh2, NULL, 0, &x, 1,
				  21, "visual/conv_diff_vtk_" );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_tx,
			   main); /* */



  /* clean up */
  sparse_free(&K);

  /* free(vx_new) is not necessary, cause of mesh_free(&msh2) where the target of vx_new is included
     so only free(vx_tmp) is necessary */
  free(vx_tmp);

  projector1_free(&P);

  vector_free(&rhs);
  vector_free(&x);
  mesh_free(&msh2);
  mesh_free(&msh1);

  free(buffer);

  return SUCCESS;


}
