/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#ifndef SPARSE_STRUCT_H
#define SPARSE_STRUCT_H

// config (by cmake)
#include "config.h"
#include "linsolve_struct.h" /* for UMFPACK coarse_mat, if available */

/* sparse matrix data structure */
/* 23.03.2007 rework to compressed row structure,
   matrix will have two stages of buildup:
   1. ultra flexible datastructure that allows arbitrary addition and
      deletion of entries, based on simply connected lists
   2. compressed row structure, once the matrix is assembled it will
      be compressed to reduce indirect adressing issues 
      
   4.9.2011 adding grouping (coloring) of dofs to allow parallel 
   gauss-seidel smoother 
*/
#define SP_TYPE_UNDEF -1 
#define SP_TYPE_FLEX 1 
#define SP_TYPE_COMPROW 2 

/* minimum (assymptotic) values, actual minimum (no new violations) value:
   #define FEINS_SPARSE_COLS_P_ROW_T1 7,  7
   #define FEINS_SPARSE_COLS_P_ROW_T2 12, 19
   #define FEINS_SPARSE_COLS_P_ROW_E1 15, 20--23
   #define FEINS_SPARSE_COLS_P_ROW_E2 29, 65--105 */
/* values used: */
#define FEINS_SPARSE_COLS_P_ROW_T1 8
#define FEINS_SPARSE_COLS_P_ROW_T2 20
#define FEINS_SPARSE_COLS_P_ROW_E1 21
#define FEINS_SPARSE_COLS_P_ROW_E2 70

struct sparse_flex_col_block {
  FIDX   *cols_idx;
  double *cols_data;
                      /* arrays which store the column indices and
			 the data (matrix entry) for this column, 
			 the arrays are of size 
			 sparse.flex_row_col_max, known to the parent
			 of the blocks, the parent also knows how many
			 entries in total are valid:
			 sparse.flex_row_col_nr[i]
		      */ 
  struct sparse_flex_col_block *next;
                      /* next block, if sparse.flex_row_col_max did
			 not suffice for sparse.flex_row_col_nr[i], 
			 NULL as long as it is not used */
};

struct sparse_flex_reserve_list {
  FIDX  nr;           /* number of entries in use */
  FIDX  max;          /* number of entries allocated */
  struct sparse_flex_col_block *blocks;
                      /* the reserve blocks of this list entry */
  struct sparse_flex_reserve_list *next;
                      /* the next entry of the reserve list,
			 NULL if none */
};

struct sparse {
  int  type;          /* describes the actual storage type of the
			 current matrix:
			 type==SP_TYPE_FLEX: flexible type 
                                (for assembly, as long as connectivity
                                 of matrix is not yet known),
			 type==SP_TYPE_COMPROW: compressed row
			 storage, faster for calculations */
  FIDX row_nr;        /* actual number of rows */

  /* data structure for compressed row sotrage, only used when type==2 */
  FIDX *rows;         /* array containing the index of the begin of
			 each row in both cols and A, length= (row_nr+1),
			 meaning:
			 k:=rows[i], m:=rows[i+1], i<row_nr
			 => i-th row is stored in cols[k...m-1] and
			    A[k..m-1] */
  FIDX *cols;         /* array containing the columns to which entries
			 in the matrix correspond,
			 length=rows[row_nr], meaning:
			 j:=cols[rows[i]+k], k<rows[i+1]-rows[i],
			                     i<row_nr,
			 => A[rows[i]+k] contains A(i,j) */
  double *A;          /* array containing the actual matrix
			 entries, length= (rows[row_nr]), 
			 interpretation as described above in
			 comment on cols (and rows) */


  /* flexible storage stage, only used when type==1 */
  FIDX  *flex_row_col_nr;
                      /* array storing the number of columns currently 
			 in each row */
  FIDX  flex_row_col_max;
                      /* maximum number of entries (memory allocated)
			 of each colums block */
  struct sparse_flex_col_block *flex_row_cols;
                      /* array of one column block for each row,
			 which store the column indices and
			 the data (matrix entry) for this row+column, 
			 the diagonal entry is always first
			 flex_row_cols[i] is a bock of size 
			 flex_row_col_max, of which the first 
			 flex_row_col_nr[i] entries are valid. If
			 these are more than flex_row_col_max, then
			 flex_row_cols.next points to the next block,
			 which may point to the block again and so on.
		      */
  int nr_processors;  /* number of processors, */
  struct sparse_flex_reserve_list *flex_reserve;
                      /* each processor has its own reserve blocks,
			 which it appends tp flex_row_cols[i].next if
			 necessary, these reserve blocks are stored in
			 a list, the list is extended if they don't
			 suffice */

  FIDX DD_proc_nr;    /* number of processors to be used (DD) */
  FIDX DD_waves_nr;   /* number of waves of DD_proc_nr domains (DD) */
  FIDX DD_nr;         /* number of subdomains in Domain Decomposition
			 (DD), should be DD_proc_nr*DD_waves_nr */
  FIDX *DD_rows_idx;  /* DD_rows_idx[i] index of start of subdomain i
			 in DD_rows (length DD_nr+1) */
  FIDX *DD_rows;      /* list of rows in each subdomain,
			 DD_rows[DD_rows_idx[i]]
			 ... DD_rows[DD_rows_idx[i+1]-1] */

  
#ifdef HAVE_OPENMP
  /* coloring of dofs in matrix */
  FIDX n_col_buckets; /* number of colors in the coloring of dofs */
  FIDX *color_buckets;/* index into color_dofs defining the begin and
			 end of each color: 
			 i=color_buckets[j] gives the first entry (i) 
			    of color_dofs of color j, all following are
			    also of color j, until the k-th:
			 k=color_buckets[j+1]-1, so k marks the end of
			 this color */
  FIDX *color_dofs;   /* vector if dofs that belong to the colors, 
			 ordered by color */  
#endif
};


/* simply connected list of storing one FIDX and one double */
struct int_double_list
{
  FIDX   ix;          /* FIDX datum */
  double dx;          /* double data */
  struct int_double_list *succ;
                      /* successor element in the list,
		         if ==NULL no successor yet */
};


/* vector data structure */
struct vector {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  double *V;          /* pointer to array of length n_max, such that 
			 V[i] gives the i-th component of the vector
		      */
};

/* simple projector data structure */
struct projector1 {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  FIDX *V;            /* pointer to array of length n_max, such that 
			 V[i] tells which components of a vector the
			 projector modifies (sets to zero),
			 V[i]=j ==> j-th component is set to zero
		      */
};

/* M-orthogonal to Z projector data structure */
struct projectorMorthoZ {
  FIDX num;           /* number of vectors Z */
  struct vector **Z;  /* pointer to array num vectors, such that 
			 Z[i] is a vector to which the result shall be
			 M-orthogonal
		      */
  struct sparse *M;   /* corresponding Mass matrix */
  struct vector My;   /* vector for storing M*y */
};

/* general transpose / notranspose stuff */
enum transposetype {
  NoTrans=0,  /* no transpose */
  Trans=1     /* transpose    */
};



/* data structure for coarse grid matrix factorizations */
struct coarse_mat
{
  int nr;             /* number of degrees of freedom (DOFs) covered
			 by this matrix */
  int band_w;         /* bandwidht of the matrix */
  int symm;           /* marking the symmetry properties of the
			 matrix */
  FIDX *nodes;        /* pointer to an array of node numbers to which
			 the DOFs correspond, represents a
			 permutation, which is used to reduce fill in
			 in the trinagulation */
  double *A;          /* the triangular factors of the matrix
			 themself, 
			 if symm==1 stored as band matrix of dimension
			       nr, with bandwidth band_w, stored in
			       the format as produced by the LAPACK
			       routine dpbtrf with upload='U',
			       LDAB=KD+1 and KD=band_w, note that the
			       i-th DOF of the system corresponds to
			       the node nodes[i] 
			 if symm==2 stored as by  LAPACK
			       routine dgbtrf with KU=KL=band_w,
			       LDAB=2*KL+KU+1, note that the i-th DOF
			       of the system corresponds to the node
			       nodes[i] */ 
  double *help;       /* a help vector of size dofs */
  int *ipiv;          /* integer vector of length dofs, to store the
			 pivot indices, only used if symm=2, otherwise
			 the pointer will be ==NULL */

  /* stuff related to HypreAMG variant of coarse_mat: */
  int HYPREamgIsInit;
#ifdef HAVE_HYPRE_NO_MPI
  void *AMGdata;
#endif


  /* stuff related to UMFPACK variant of coarse_mat: */
  int UMFPACKisinit;
#ifdef HAVE_UMFPACK
  struct linsolver UMFPACKdata;
  struct projector1 P;
  FIDX *UMFPACKisdiri;
  struct vector UMFPACKhelp;
  /* rhs is not modified by solver, if input x=0 */
#endif

};


/* data structure for adjency-, level- and similar lists of integers */
struct int_list
{
  FIDX ixn;           /* actual length of the index vector ix */
  FIDX ixnmax;        /* maximal length of the index vector ix */
  FIDX *ix;           /* index vector, j=ix[i] marks the begin of the
			 i-th part of el, k=ix[i+1]-1 marks the end of
			 the i-th part of el, i<ixn<=ixnmax */
  FIDX elnmax;        /* maximal length of el, therefore 
			 ix[ixn+1]-1<=elnmax has to be asured */
  FIDX *el;           /* list of elements, with j and k as described
			 in the ix part, el[j+t] is the t-th element
			 of the i-th part of the list, t<=k-j-1 */
};



#endif
