static inline int sparse_get_entry(struct sparse *K, FIDX row, FIDX col,
		     int create, double **entry
/* if K contains an entry K(row,col) a pointer to this entry is
   returned, such that 
             (*entry)=K(row,col)
   if it doesn't contain the entry yet and create==1, it is created
   and a pointer to the entry is returned as before.

   Input:   row    - row index into the sparse matrix K
            col    - column index into the sparse matrix K
	    create - if ==1 entry will be created (if possible)
	             otherwise, no entry will be created, only
	             existing ones are returned

   In/Out:  K      - sparse matrix, eventually the entry K(row,col) is
                     created

   Output:  entry  - (by reference) pointer to a double such that 
                     (*entry)=K(row,col)

   Return: SUCCESS - success,
           FAIL    - failure, entry not found and not allowed to
                     create,
		     if allowed, creation failed, see error message
*/
		     ){
  if ((row>=(*K).row_nr)||(row<0))
    {
      fprintf( stderr, "sparse_get_entry: "
	       "row=%"dFIDX" >= K.row_nr=%"dFIDX"  !\n",
	        row,  (*K).row_nr);
      return FAIL;
    }

  switch ( (*K).type )
    {
    case SP_TYPE_FLEX:
      {
	if (row==col)
	  {
	    /* the diagonal entry is always the first in the list */
	    (*entry) = &( (*K).flex_row_cols[row].cols_data[0] );
	    return SUCCESS;
	  }
	else
	  {
	    struct sparse_flex_col_block *block, *block_old;
	    FIDX col_max=(*K).flex_row_col_max;
	    FIDX j;
	    
	    /* loop over the collumns of this row */
	    block=&(*K).flex_row_cols[row];
	    block_old=NULL;
	    /* try to find the entry, can't be first entry */
	    for (j=1; j<(*K).flex_row_col_nr[row]; j++)
	      {
		if ( (*block).cols_idx[j%col_max] == col)
		  {
		    (*entry) = &( (*block).cols_data[j%col_max] );
		    return SUCCESS;
		  }
		/* at end of block */
		if ( (j%col_max)==col_max-1 )
		  {
		    /* switch to next block */
		    block_old=block;
		    block=(*block).next;
		  }
	      }

	    /* not found? ==> need new entry */
	    if (create!=1)
	      {
		/* not allowed to create */
		(*entry)=NULL;
		return FAIL;
	      }

	    /***********************************************/
	    /****  create entry                         ****/
	    /***********************************************/

            /* at end of block and no new block ? */
            if ( block==NULL )
	      {
		/* get new reserve block */

		int my_id;
		struct sparse_flex_reserve_list *this, *head, *next;

#ifdef HAVE_OPENMP
		my_id      = omp_get_thread_num();
#else
		my_id      = 0;
#endif
		/* traverse this processors reserve list to find free
		   column block */
		head=&(*K).flex_reserve[my_id];
		this=head;
		while ( (*this).nr == (*this).max )
		  {
		    /* this reserve block is full, get next one */
		    next=(*this).next;
		    if (next==NULL)
		      {
			FIDX j;
			/* no next reserve list yet, alloc one */
			TRY_MALLOC( (*this).next, 1,
				    struct sparse_flex_reserve_list,
				    sparse_get_entry);
			next=(*this).next;

			/* init new reserve list entry */
			TRY_MALLOC( (*next).blocks, (*this).max,
				    struct sparse_flex_col_block,
				    sparse_get_entry);
			(*next).max  = (*this).max;
			(*next).nr   = 0;
			(*next).next = NULL;
			/* init blocks of new reserve */
			for (j=0; j<(*next).max; j++)
			  {
			    struct sparse_flex_col_block *new_block;
			    new_block=&(*next).blocks[j];
			    TRY_MALLOC( (*new_block).cols_idx,
					(*K).flex_row_col_max, FIDX, 
					sparse_flex_alloc);
			    TRY_MALLOC( (*new_block).cols_data,
					(*K).flex_row_col_max, double,
					sparse_flex_alloc);
			    (*new_block).next = NULL;
			  }
		      } /* end if (next==NULL) */
		    this=next;
		  } /* end if (this.nr==this.max), no free block in 
		       this list entry */

		/* we (now) have room in this list entry to store the
		   new column block */
		(*block_old).next=&(*this).blocks[(*this).nr];
		(*this).nr++;
		block=(*block_old).next;
	      } /* end if (block==NULL), no new block */
	    
	    /* now block definitely has space for the new entry,
	       create it */
	    (*block).cols_idx[(*K).flex_row_col_nr[row]%col_max]  = col;
	    (*block).cols_data[(*K).flex_row_col_nr[row]%col_max] = 0.0;
	    (*entry)= &( (*block).cols_data[(*K).flex_row_col_nr[row]%col_max
					    ] );
	    (*K).flex_row_col_nr[row]++;
	    return SUCCESS;
	  } /* end else, this is not the diagonal entry of K */
      } /* end case SP_TYPE_FLEX */
      break;
    case SP_TYPE_COMPROW:
      {
	FIDX j, endrow;
	FIDX *Acols;

	Acols = (*K).cols;
	endrow = (*K).rows[row+1];
	j= (*K).rows[row];
	while ((j<endrow)&&(Acols[j]!=col)) j++;
	if (j>=endrow)
	  {
	    if (create==1)
	      {
		fprintf( stderr, "sparse_get_entry: "
			 "K(%"dFIDX",%"dFIDX") does not exist \n"
			 "can not be created in compressed row mode!\n",
			  row,  col);
	      }
	    (*entry)=NULL;
	    return FAIL;
	  }
	else 
	  {
	  /* is found */
	    *entry = &( (*K).A[j] );
	    return SUCCESS;
	  }
      }
      break;
    default:
      /* unknown type of matrix */
      fprintf(stderr,"sparse_get_entry: type %d not implemented\n",
	      (*K).type );
      return FAIL;
    }
  return FAIL;
}



static inline int sparse_add_local(struct sparse *K, enum transposetype trans,
		     FIDX m, FIDX row[], FIDX n,
		     FIDX col[], double local[], FIDX LDlocal
/* adds the matrix local to the by row and col defined submatrix of K,
   if trans==NoTrans:
   K(row[i],col[j])+=local[j*LDlocal+i];

   if trans==Trans:
   K(row[i],col[j])+=local[i*LDlocal+j];

   Input:  trans   - ==NoTrans  => normal add
                     ==Trans    => Transposed add
           m       - number of rows in local, length of row
           row     - array of row indices into K, length m
           n       - number of collumns in local, length of col
           col     - array of collumn indices into K, length n
	   local   - array to be added to K, size m*n
	   LDlocal - leading dimension of local

   In/Out: K       - sparse matrix to which values are added

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		     ){
  int  err;
  FIDX i,j;
  double *entry;

  if (trans==NoTrans)
    {
      for (i=0; i<m; i++)
	for (j=0; j<n; j++)
	  {
	    err=sparse_get_entry( K, row[i], col[j], 1, &entry);
	    if (err!=SUCCESS)
	      {
		fprintf(stderr, "sparse_add_local: "
			"coulnd get/create entry in K for "
			"row[%"dFIDX"] col[%"dFIDX"] ",  i,  j);
		return FAIL;
	      }
	    (*entry)+=local[j*LDlocal+i];
	  }
    }
  else if (trans==Trans)
    {
      for (i=0; i<m; i++)
	for (j=0; j<n; j++)
	  {
	    err=sparse_get_entry( K, row[i], col[j], 1, &entry);
	    if (err!=SUCCESS)
	      {
		fprintf(stderr, "sparse_add_local: "
			"coulnd get/create entry in K for "
			"row[%"dFIDX"] col[%"dFIDX"] ",  i,  j);
		return FAIL;
	      }
	    (*entry)+=local[i*LDlocal+j];
	  }
    }
  else
    {
      fprintf(stderr,
	      "sparse_add_local: wrong transpose value!\n");
      return FAIL;
    }

  return SUCCESS;
}
