/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
navstoassem.h
*/
#ifndef __NAVSTOASSEM_H__
#define __NAVSTOASSEM_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"
#include "navsto_struct.h"


int navsto_C_assem_t21(struct navsto_matrix *K, struct vector *rhs,
		       double *maxPe, struct vector *rhs_stokes,
		       struct vector *u_old,
		       struct mesh *m,
		       FIDX lvl, int Fsonly, 
		       int lintype
/* performs the assembly of the changing part Fs of the stiffness
   matrix K and the right hand side vector rhs which result from
   linearised finite element discretisation of the Navier Stokes
   equation on the T2 mesh m, such that 

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

         -nu Laplace(u) + u * grad u + grad p = f
                                        div u = 0

   two different linearisations are implemented here, (see Gunzburger,
   Finite Element Methods for Viscous Incompressible Flows, Chapter
   6.1 and 6.3) Newtons Method and the Picard (/Simple) Iteration

   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  rhs_stokes
                   - the righthand side of the Stokes system, as
                     obtained by navsto_MG_init_assem_t21
	   u_old   - the previous iterate solution
	   m       - the mesh
	   lvl     - the level who's Fs is written
	   Fsonly  - to indicate if (Fsonly==1) only the matrix is to
	             be assembled, or (Fsonly==0) the righthand side
	             as well
	   lintype - the type of linearisation for which Fs and the
        	     rhs are wanted, lintype==1 ==> Newtons Method,
		     lintype==2 ==> Picard iteration
           
   In/Out: K       - stiffness matrix in specialised storage, the
                     structure K has to be initialised by
                     navsto_MG_init_assem_t21 once, and afterwards it is
                     just updated (the Fs part, which is the only part
                     that changes between iterations) in this routine

   Output: rhs     - righthand side vector, has to be initialised by
                     navsto_MG_init_assem_t21 once, and afterwards it
                     is just updated in this routine
	   maxPe   - max element Peclet number Pe (maxPe>1 means the
 	             FE discretisation on this mesh is unstable unless
 	             stabelised)


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_MG_init_assem_t21(struct mesh *m1, FIDX lvl,
			     struct navsto_matrix *K, 
			     struct vector *rhs_stokes,
			     struct vector *rhs,
			     struct vector *u0,
			     struct mesh *m2,
			     double *vx_new, FIDX vx_max
/* performs the assembly of the non-changing parts of the stiffness
   matrix K and the right hand side vector rhs which result from
   linearised finite element discretisation of the Navier Stokes
   equation on the T2 mesh m2 (build from T1 mesh m1 by refinement),
   such that 

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

         -nu Laplace(u) + u * grad u + grad p = f
                                        div u = 0

   with boundary conditions as given in the mesh m    
   (p2 triangles)

   The linear terms are identical to the discretisation of a
   Stokes system, so mainly the Stokes assembly is called
   here. However, initialisation of the remaining parts is performed
   here as well. 

   Input:  m1      - the coarse mesh (T1)
           lvl     - (new) current level
	   vx_new  - overriding set of node positions for T2 meshes
                     ignored if ==NULL
	   vx_max  - maximum number of nodes for use of vx_max
                     ignored if vc_new==NULL


   Output: K       - stiffness matrix in specialised storage, the
                     structure K is initialised here, memory
                     allocated and during successive iterations
                     navsto_C_assem_MG_t21 has to be used to update
                     the changing parts of the system
           rhs_stokes
                   - the righthand side of the Stokes system, only the
                     vector data structure needs to be given, it is
		     initialised here
           rhs     - the righthand side of the linearised Navier
                     Stokes system, only the vector data structure
                     needs to be given, it is initialised here, on
                     exit it will be identical to rhs_stokes
	   u0      - a initial guess for the solution which fulfils
         	     the boundary conditions
	   m2      - the refined T2 mesh



   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_MG_C_assem_t21(struct mesh *m1, FIDX lvl,
			  struct navsto_matrix *K, struct vector *rhs,
			  double *maxPe,
			  struct vector *rhs_stokes,
			  struct vector *u_old,
			  struct mesh *m2, double *vx_new, FIDX vx_max,
			  int lintype
/* repeatetly calls navsto_C_assem_t21 to perform the assembly of the
   changing part Fs of the stiffness matrix K and all its coarse
   grid versions and the right hand side vector rhs which result from
   linearised finite element discretisation of the Navier Stokes
   equation on the T2 mesh m2, which results from refinement of the T1
   mesh m1, such that  

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

   for the description of the linearisations see navsto_C_assem_t21

   Input:  m1      - the coarse mesh (T1)
           lvl     - the current refinement level
           rhs_stokes
                   - the righthand side of the Stokes system, as
                     obtained by navsto_init_assem_t21
	   u_old   - the previous iterate solution
	   vx_new  - overriding set of node positions for T2 meshes
                     ignored if ==NULL
	   vx_max  - maximum number of nodes for use of vx_max
                     ignored if vc_new==NULL
	   lintype - the type of linearisation, see navsto_C_assem_t21
	             for description
           
   In/Out: K       - stiffness matrix in specialised storage, the
                     structure K has to be initialised by
                     navsto_init_assem_t21 once, and afterwards it is
                     just updated (the Fs part, which is the only part
                     that changes between iterations) in this routine

   Output: m2      - the fine mesh (T2)
	   rhs     - righthand side vector, has to be initialised by
                     navsto_init_assem_t21 once, and afterwards it is
                     just updated in this routine
	   maxPe   - max element Peclet number Pe for the finest mesh
	             (maxPe>1 means the FE discretisation on this mesh
	             is unstable)



   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_MG_ref_intp_ass_t21(struct mesh *m1, FIDX lvl,
			       struct mesh *m2,
			       double *vx_new, FIDX vx_max,
			       struct navsto_matrix *K, 
			       struct vector *rhs_stokes,
			       struct vector *rhs,
			       struct vector *uc,
			       struct vector *uf
/* performs uniform refinement of the mesh, reinitialises all data for
   the finer mesh, navsto_init_assem_t21 on the finer and all coarse
   meshes, interpolates the given coarse solution uc to a finer
   solution uf which fulsfils the boundary conditions of the finer
   mesh

   Input:  m1      - the coarse mesh
           lvl     - the new level for the fine mesh
	   vx_new  - overriding set of node positions for T2 meshes
	   vx_max  - maximum number of nodes for use of vx_max

   In/Out: m2      - the mesh, will be refined
           K       - stiffness matrix in specialised storage, the
                     structure K is reinitialised here, memory
                     reallocated and during successive iterations
                     navsto_MG_C_assem_t21 has to be used to update the
                     changing parts of the system
           rhs_stokes
                   - the righthand side of the Stokes system, it is
		     reinitialised here
           rhs     - the righthand side of the linearised Navier
                     Stokes system, it is reinitialised here, on
                     exit it will be identical to rhs_stokes
	   uc      - the solution on the current mesh m, is
               	     reinitialised to size for the new mesh
           uf      - the solution uc interpolated to a solution on the
                     new fine mesh, adjusted to fulfil the boundary
                     conditions, the given vector is reinitialised to
                     the new size



   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_maxnu_init_t21(struct mesh *m1,
			  double *nu_max
/* solves the Navier-Stokes equations on the coarse grid (with low
   accuracy) to get an estimate for the highest possible Reynolds
   number on the coarse mesh

   Input:  m1      - the coarse mesh (T1)

   Output: nu_max  - estimate for the highest Re for which the
                     discretisation on the coarse mesh is stable



   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_velssorter_init_t21(struct mesh *m,  struct multilvl *ml, 
			       struct mgdata *mg
/* initialises the sorterl part of the mgdata struct

   Input:  m       - the mesh, needs to be the finest level of the
                     hierarchy 
           ml      - multilevel data struct
	   
   In/Out: mg      - multigrid data struct, has to be initialised
                     already sorterl part is set here

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_dofsorter_t21(struct mesh *m,  struct multilvl *ml, 
			 FIDX dir1,
			 FIDX *sortperm
/* initialises the defines a permutation on of the dofs on each level
   of the mesh that sorts the nodes according to their coordinates in
   the mesh, first by spacial direction dir1 (e.g. dir1==0, sort by x
   first), then by the remaining

   Input:  m       - the mesh, needs to be the finest level of the
                     hierarchy 
           ml      - multilevel data struct, used to define the
                     multilevel sorter array
           dir1    - first direction to be sorted by, i.e. dir1==0:
                     sort by increasing x coordinates, dir1==1: sort
                     by increasing y coordinate
	   
   Out:    sortperm- integer array, size ml.nlevl[0], has to be
                     allocated by the calling routine, defines
                     permutation that sorts the dofs in the desired
                     order 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_assem_SDFEM_t21(struct navsto_matrix *K, struct vector *rhs,
			   struct vector *u_old, struct mesh *m,
			   FIDX lvl 
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from linearised SDFEM (streamline
   diffusion finite element method) discretisation of the Navier
   Stokes equation on the T2 mesh m, such that

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

         -nu Laplace(u) + u * grad u + grad p = f
                                        div u = 0

   SDFEM is described in L. Tobiska and R. Verfuerth, Analysis of a
   Streamline Diffusion Finite Element Method for The Stokes and
   Navier-Stokes equations, SIAM J. Numer. Anal., Vol. 33, No. 1,
   pp. 107--127, 1996

   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  u_old   - the previous iterate solution
	   m       - the mesh
	   lvl     - the level who's Fs is written
	   
   Out:    K       - stiffness matrix in specialised storage, the
                     structure K has to be initialised by
                     navsto_MG_init_assem_SDFEM_t21 once, and
                     afterwards it is updated in this routine

   Output: rhs     - righthand side vector, has to be initialised by
                     navsto_MG_init_assem_t21 once, and afterwards it
                     is just updated in this routine


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_MG_assem_SDFEM_t21(struct mesh *m1, FIDX lvl,
			      struct navsto_matrix *K, 
			      struct vector *rhs,
			      struct vector *u0,
			      struct mesh *m2, double *vx_new, FIDX vx_max
/* repeatetly calls navsto_assem_SDFEM_t21 to perform the assembly of the
   changing parts of the stiffness matrix K and all its coarse
   grid versions and the right hand side vector rhs which result from
   linearised finite element discretisation of the Navier Stokes
   equation on the T2 mesh m2, which results from refinement of the T1
   mesh m1, such that  

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

   Input:  m1      - the coarse mesh (T1)
           lvl     - the current refinement level
	   u0      - the previous iterate solution
	   vx_new  - overriding set of node positions for T2 meshes
	   vx_max  - maximum number of nodes for use of vx_max
           
   In/Out: K       - stiffness matrix in specialised storage, the
                     structure K has to be initialised by
                     navsto_init_MG_assem_SDFEM_t21 once, and
                     afterwards it is just updated (the Fs part, which
                     is the only part that changes between iterations)
                     in this routine 

   Output: m2      - the fine mesh (T2)
	   rhs     - righthand side vector, has to be initialised by
                     navsto_init_assem_t21 once, and afterwards it is
                     just updated in this routine


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_MG_init_assem_SDFEM_t21(struct mesh *m1, FIDX lvl,
				   struct navsto_matrix *K, 
				   struct vector *rhs,
				   struct vector *u0,
				   struct mesh *m2,
				   double *vx_new, FIDX vx_max
/* initialises the non-changing parts of the navsto_matrix struct K,
   the initial solution u0 and the right hand side vector rhs which
   result from linearised finite element discretisation of the Navier
   Stokes equation on the T2 mesh m2 (build from T1 mesh m1 by
   refinement), such that

           K x = rhs

   defines the new iterate (approximate) solution x=(u_x, u_y, p) of
   the Navier Stokes equation

         -nu Laplace(u) + u * grad u + grad p = f
                                        div u = 0

   with boundary conditions as given in the mesh m    
   (p2 triangles)

   The terms initialised are mainly projector data for the Dirichlet
   boundary conditions and for the zero mean pressure condition, which
   are identical to what they are in the Stokes case, thus we use the
   same routines to initialise them. However, initialisation of the
   remaining parts is performed here as well.

   Input:  m1      - the coarse mesh (T1)
           lvl     - (new) current level
	   vx_new  - overriding set of node positions for T2 meshes
	   vx_max  - maximum number of nodes for use of vx_max

   Output: K       - stiffness matrix in specialised storage, the
                     structure K is initialised here, memory
                     allocated and during successive iterations
                     navsto_MG_assem_SDFEM_t21 has to be used to
                     update the changing parts of the system
           rhs     - the righthand side of the linearised Navier
                     Stokes system, only the vector data structure
                     needs to be given, it is initialised here
	   u0      - a initial guess for the solution which fulfils
         	     the boundary conditions
	   m2      - the refined T2 mesh



   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_pdof_vels_init_t21(struct mesh *m, FIDX lvl,
			      struct navsto_matrix *K
/* initialises the pdof_vels part of the matrix struct

   Input:  m       - the mesh
	   lvl     - the level who's Fs is written
	   
   In/Out: K       - stiffness matrix in specialised storage, the
                     structure K has to be initialised by
                     stokes_pdof_init_t21 and stokes_diri_bc_init_t2
                     for the current level already, the pdof_vels[lvl]
                     part is set here 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
