/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
navsto_aux.h
*/
#ifndef __NAVSTO_AUX_H__
#define __NAVSTO_AUX_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"
#include "navsto_struct.h"


int navsto_write_line_u_t2( struct mesh *m, struct vector *usol,
			    int xy, double dxy,
			    char *name
/* writes the values of the solution usol along a line specified
   (x=dxy or y=dxy),
   the values are stored for a set of points which will be sorted in
   descending order (of the y resp. x component) and written together
   with the velocity vector in a file named name
   
   Input:  m         - the mesh
           usol      - vector with containing the velocity field,
                       stored as usol[j*vx_nr+i] gives the j-th
                       velocity component in the i-th node of the mesh
	   xy        - specifies if dxy shall be matched by the x or y
	               component of the coordinates,
		       xy=0  ==> for x==dxy
		       xy=1  ==> for y==dxy
           dxy       - the position of the line
	   name      - name of the file

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_matrix_init(struct navsto_matrix *K, FIDX dim, FIDX lvlmax
/* allocates memory in K
   
   Input:  dim     - problem dimension (d=2 or 3)
	   lvlmax  - maximal number of levels for which the stiffness
	             matrices are stored

   Output: K       - (K is given by reference), memory in K is
                     allocated, dimensions of K are initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_matrix_reinit(struct navsto_matrix *K, FIDX vx_nr
/* reallocates memory in K as necessary if the mesh changes
   (e.g. refined)
   
   Input:  vx_nr   - number of vertices in the corresponding mesh

   Output: K       - (K is given by reference), memory in K is
                     reallocated, dimensions of K are reinitialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void navsto_matrix_free(struct navsto_matrix *K
/* memory allocated in K is freed, so that K can be freed

   Output: K       - (K is given by reference) inside memory is
                     released 
*/);

int navsto_projector_no_precon(void *arg1, struct vector *in,
			       void *notused,
			       struct vector *out
/* performs the boudary condition projection for stokes problems,
     out = P*I*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there) and projects the pressure such that it has zero
   mean on the domain
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner" 

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_matrix_tim_vec(void *arg1, struct vector *vec,
			  struct vector *out
/* multiplies the matrix K from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - linearised Navier-Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_matrices_write_files(struct navsto_matrix *K, char *basename
/* write all the top-level matrices into files
   (useful to analyse stuff in matlab)

   Input:  K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
           basename- the names of the individual files are derived
                     from this by appending "_XX.txt", where XX
                     specifies which of the matrices it is, i.e. one
                     of F, B1, B2, A_p, C_p, M_p, w, Dir
		     (F_p = A_p + C_p)

   Output: (the files)

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_projector_w_precon(void *arg1, struct vector *in,
			      void *notused,
			      struct vector *out
/* performs preconditioning and the boudary condition projection for
   the linearizations of Navier-Stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there) and projects the pressure such that it has zero
   mean 

   C^-1 is the Wathen block preconditioner:
        
   C^-1=[I                   ] [ I   ] [F^-1  ] velocity space
        [  (Mp^-1)*Fp*(Ap^-1)]*[-B  I]*[     I] pressure space

   where F is the advection diffusion operator, Mp is the pressure
   space mass matrix, Fp the pressure space advection diffusion
   operator, Ap the pressure space laplace operator
   (F, Fp and Ap are discretisations of these operators)

   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_advecdiff_tim_vec(void *arg1, struct vector *vec,
			     struct vector *out
/* multiplies the stiffness matrix K of the advection diffusion
   operator from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - Navier-Stokes matrix struct
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_projector_vel(void *arg1, struct vector *in,
			 void *arg3, struct vector *out
/* performs the boudary condition projection for the advection
   diffusion subproblems of the Wathen block preconditioner for
   Navier-Stokes problems, 
     out = P*I*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there)
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
           in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner" 

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_projector_vel_MG(void *arg1, struct vector *in,
			    void *arg3, struct vector *out
/* performs the boudary condition projection and multigrid
   preconditioning for the advection diffusion subproblems of the
   Wathen block preconditioner for Navier Stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there), and C^-1 repressents the action of the multigrid
   preconditioning, that is one V cycle
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
           in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_projector_vel_W_cycle(void *arg1, struct vector *in,
				 void *arg3, struct vector *out
/* performs the boudary condition projection and multigrid
   preconditioning for the advection diffusion subproblems of the
   Wathen block preconditioner for Navier Stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there), and C^-1 repressents the action of the multigrid
   preconditioning, that is one W cycle
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
           in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_projector_vel_W_CGC(struct navsto_matrix *K,
			       FIDX lvl, FIDX lvlmin, FIDX W_arms
/* forms a W cycle multigrid scheme by recursively doing

   loop W_arms times:
      smooth_bwd
          calc residual, restrict
               navsto_projector_vel_W_CGC 
          inject solution, add to current level
      smooth_fwd

   

   
   Input:  K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
           lvl     - current level
           lvlmin  - lowest level of the hierarchy to be used
	   W_arms  - number of CGC loops, see description above

   In/Out: K.mg    - the lvl part contains in and output: bl=rhs(in),
                     xl=solution(in(should be initialised,e.g.=0)+out),
		     dxl is modified, ALL LOWER LEVELS ARE MODIFIED

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_precon_p_lapl_bpx(void *notused, struct vector *in,
			     void *arg3, struct vector *out
/* performs BPX preconditioning for the pressure space Laplacian
   subproblems which occure in the Wathen block preconditioner for
   Navier Stokes problems, 
     out = C^{-1}* in
   where C^{-1} is the BPX preconditioner sum(l=level)Q_l*D_l*Q_l^T 
   
   Input:  notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner" 
           in      - input vector
	   arg3=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_SDFEM_precon(void *arg1, struct vector *in,
			   void *notused,
			   struct vector *out
/* performs preconditioning and the boudary condition projection for
   the linearizations of Navier-Stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there) and projects the pressure such that it has zero
   mean 

   C^-1 is a W-multigrid cycle 

   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_Wcycle_SDFEM(struct navsto_matrix *K, struct vector *bh,
			   struct vector *xh, struct vector *rh, FIDX lvl
/* performs a W-multigrid cycle to approximate the solution of 
     K*xl = bl
   bl and xl are stored in mg and mg1: mg->velocities, mg1->pressure,
   read and write of these vectors should be done with
   navsto_mg_pull_vec_SDFEM and navsto_mg_push_vec_SDFEM respectively

   the W-cycle is performed by recursively
      1. setting xl=0
      2. smoothing 
      3. if lvl>0
           a. restrict
           b. W-cycle on lvl-1
           c. interpolate
      4. smoothing
      5. define resulting xl, store in dxl.

   Input:  lvl     - the current level

   In/Out: K       - navsto_matrix struct,
                     the help vectors in mg and mg1 are modified,
                     otherwise only input
           bh, xh, rh
                   - help vectors used for interaction with the matrix
                     on th current level

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_smoother_SDFEM(struct vector *x, struct vector *rhs,
			     int dir, FIDX lvl, struct navsto_matrix *K
/* performes a smoothing sweep using the box smoother (aka Vanka
   smoother), used in multigrid schemes to solve
     K*x = rhs

   The box smoother comprises of solving local equation systems
   corresponding to small patches of the mesh in a Gauss-Seidel like
   manner. (It is a generalisation of the Gauss-Seidel scheme to
   coupled equations.)

   Our patches are defined by the support of the pressure basis
   functions (linear). On each such patch we solve a local dirichlet
   problem, adjusting only those velocity dofs which are not on the
   boundary of the patch and the pressure dof that defines the patch.
   
   Input:  rhs     - the right hand side vector
           dir     - direction of the sweep,
	             dir=+1 -> forward, dir=-1 -> backward,
	   lvl     - level of the mesh that the given vectors
	             correspond to
           K       - navsto_matrix struct, holds the necessary data

   In/Out: x       - the solution vector, it is updated according to
                     the smoothing scheme
           
   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_push_vec_SDFEM(struct vector *vec, double *vml, double *pml,
			     FIDX lvl, struct navsto_matrix *K
/* copies the vector vec into the multilevel vectors, seperated in the
   velocity (vml) and pressure parts (pml)

   Input:  vec     - vector to be copied
           lvl     - level of the mesh that the given vector
	             corresponds to
           K       - navsto_matrix struct, holds the necessary data

   Output: vml     - in level lvl part, the velocities from vec
           pml     - in level lvl part, the pressure from vec

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_pull_vec_SDFEM(struct vector *vec, double *vml, double *pml,
			     FIDX lvl, struct navsto_matrix *K
/* copies a vector stored in multilevel vectors, seperated in the
   velocity (vml) and pressure parts (pml), into the vector vec

   Input:  vml     - in level lvl part, the velocities for vec
           pml     - in level lvl part, the pressure for vec
	   lvl     - level of the mesh that the given vector
	             corresponds to
           K       - navsto_matrix struct, holds the necessary data

   Output: vec     - vector to be copied into
           
   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_copy_vec_SDFEM(double *vmlin, double *pmlin,
			     double *vmlout, double *pmlout,
			     FIDX lvl, struct navsto_matrix *K
/* copies multilevel vectors, seperated in the
   velocity (vml) and pressure parts (pml)

   Input:  vmlin   - in level lvl part, the velocities
           pmlin   - in level lvl part, the pressure
	   lvl     - level of the mesh that the given vector
	             corresponds to
           K       - navsto_matrix struct, holds the necessary data

   Output: vmlout  - in level lvl part, the velocities 
           pmlout  - in level lvl part, the pressure

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_projector_trans_SDFEM( struct vector *vec, 
				     FIDX lvl, struct navsto_matrix *K
/* performs application of the transpose of the projector used to
   implement the Dirichlet boundary conditions and the zero mean
   pressure condition

   Input:  lvl     - level of the mesh that the given vector
	             corresponds to
           K       - navsto_matrix struct, holds the necessary data
           
   In/Out: vec     - out = P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_mg_projector_SDFEM( struct vector *vec, 
			       FIDX lvl, struct navsto_matrix *K
/* performs application of the projector used to implement the
   Dirichlet boundary conditions and the zero mean pressure condition

   Input:  lvl     - level of the mesh that the given vector
	             corresponds to
           K       - navsto_matrix struct, holds the necessary data
           
   In/Out: vec     - out = P * in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);
#endif
