/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE mesh3d.c
HEADER mesh3d.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"
#include "sparse_struct.h"
#include "mesh.h"
*/




/* prototypes of external functions */
#include <math.h>
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
//#ifndef __USE_GNU
//#define __USE_GNU
#endif
#include <string.h>


/* function prototypes */
#include "mesh3d.h"

#include "sparse.h"



/*FUNCTION*/
int mesh_write_solution_vtk_e1( struct mesh *m, struct vector *vec_x,
				FIDX vcompo, struct vector *scalar_x,
				FIDX scompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a vtk readable
   file for visualisation, 
   this can for example be used with paraview
           http://www.paraview.org/

   Input:  m         - the mesh
           vec_x     - vector with data associated to the nodes,
	               for representing a vector field of dimension
	               vcompo, ignored if vec_x==NULL
	   vcompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector x have to be numbered such that
	               vec_x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh, has to be TWO for t1
	               mesh, ignored if vec_x==NULL
           scalar_x  - vector with data associated to the nodes,
	               for representing scompo scalar fields,
		       for each scalar field a separate part is
		       written in the file, ignored if scalar_x==NULL
	   scompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector scalar_x have to be numbered such that
	               saclar_x[j*vx_nr+i] gives the j-th scalar
	               variable in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  int err;
  FIDX i, j, vx_w, vx_nr, el_w, el_nr;

  double *vertex;
  struct vector my_vec_x, my_scalar_x;

  char *vtkname;
  FILE *vtkfile;

  int elem_nodes, elem_type_vtk;

  vx_w =(*m).vx_w;
  vx_nr=(*m).vx_nr;
  el_w =(*m).el_w;
  el_nr=(*m).el_nr;

  if ((vec_x!=NULL)&&(vcompo!=3))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_e1: "
	       "dimension of vector vec_x must be 3 for 3D mesh!\n");
      return FAIL;
    }
  if ((vec_x!=NULL)&&(vcompo*vx_nr!=(*vec_x).len))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_e1: vx_nr in mesh and "
	       "dimension of vector vec_x must agree!\n");
      return FAIL;
    }

  if ((scalar_x!=NULL)&&(scompo*vx_nr!=(*scalar_x).len))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_e1: vx_nr in mesh and "
	       "dimension of vector scalar_x must agree!\n");
      return FAIL;
    }

  /* correct hierarchical nodepos to nodal-positions if m.type==32,
     same for hierachical data in solution vectors */
  if ( (*m).type==32 )
    {
      elem_nodes=10;
      elem_type_vtk=24;
      TRY_MALLOC(vertex, vx_w*vx_nr, double, mesh_write_solution_vtk_e1);
      /* copy first */
      for (i=0; i<vx_w*vx_nr; i++)
	{
	  vertex[i]=(*m).vertex[i];
	}
      if (vec_x!=NULL)
	{
	  err=vector_alloc(&my_vec_x, (*vec_x).len );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc,
				   mesh_write_solution_vtk_e1);
	  for (i=0; i<my_vec_x.len; i++)
	    {
	      my_vec_x.V[i]=(*vec_x).V[i];
	    }
	}
      if (scalar_x!=NULL)
	{
	  err=vector_alloc(&my_scalar_x, (*scalar_x).len );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc,
				   mesh_write_solution_vtk_e1);
	  for (i=0; i<my_scalar_x.len; i++)
	    {
	      my_scalar_x.V[i]=(*scalar_x).V[i];
	    }
	}
      /* check all edges if their midnode is of type 2 */
      for (i=0; i<(*m).eg_nr; i++)
	{
	  FIDX nodem=(*m).edge[i*(*m).eg_w+MCE2EGNODM];
	  if (vertex[nodem*vx_w+MCEXVXTYPE]==2.0)
	    {
	      /* if type=2, modify to type=0,
		 i.e. switch from hierarchical basis to nodal basis
	       */
	      FIDX nod0=(*m).edge[i*(*m).eg_w+MCE1EGNOD1];
	      FIDX nod1=(*m).edge[i*(*m).eg_w+MCE1EGNOD1+1];
	      
	      for (j=0; j<3; j++)
		{
		  vertex[nodem*vx_w+MCE1VXSTRT+j] +=
		    0.5*(vertex[nod0*vx_w+MCE1VXSTRT+j] 
			 + vertex[nod1*vx_w+MCE1VXSTRT+j]);
		}
	      
	      vertex[nodem*vx_w+MCEXVXTYPE]=0.0;

	      /* do the same with the solution vectors */
	      if (vec_x!=NULL)
		{
		    for (j=0; j<vcompo; j++)
		      {
			my_vec_x.V[j*vx_nr+nodem] +=
			  0.5*(my_vec_x.V[j*vx_nr+nod0] 
			       + my_vec_x.V[j*vx_nr+nod1]);
		      }
		}
	      if (scalar_x!=NULL)
		{
		    for (j=0; j<scompo; j++)
		      {
			my_scalar_x.V[j*vx_nr+nodem] +=
			  0.5*(my_scalar_x.V[j*vx_nr+nod0] 
			       + my_scalar_x.V[j*vx_nr+nod1]);
		      }
		}
  	    }
	}
      /* check orientation of elements */
      /* turned out not to be important 
	 for (i=0; i<el_nr; i++)
	{
	  double vec01[3], vec02[3], vec03[3], cross[3];
	  double sprod;
	  FIDX nod0=(*m).elem[i*el_w+MCE1ELNOD1];
	  FIDX nod1=(*m).elem[i*el_w+MCE1ELNOD1+1];
	  FIDX nod2=(*m).elem[i*el_w+MCE1ELNOD1+2];
	  FIDX nod3=(*m).elem[i*el_w+MCE1ELNOD1+3];
	  for (j=0; j<3; j++)
	    {
	      vec01[j]=vertex[nod1*vx_w+MCE1VXSTRT+j] 
		- vertex[nod0*vx_w+MCE1VXSTRT+j];
	      vec02[j]=vertex[nod2*vx_w+MCE1VXSTRT+j] 
		- vertex[nod0*vx_w+MCE1VXSTRT+j];
	      vec03[j]=vertex[nod3*vx_w+MCE1VXSTRT+j] 
		- vertex[nod0*vx_w+MCE1VXSTRT+j];
	    }
	  for (j=0; j<3; j++)
	    {
	      cross[j]=vec01[(j+1)%3]*vec02[(j+2)%3]
		-vec01[(j+2)%3]*vec02[(j+1)%3];
	    }
	  sprod=0.0;
	  for (j=0; j<3; j++)
	    {
	      sprod+=cross[j]*vec03[j];
	    }
	  if (sprod<0.0)
	    {
	      //int norder[]={0,1,2,3,4,5,6,7,8,9};
	      int norder[]={1,0,2,3,4,6,5,8,7,9};
	      FIDX oldel[10];
	      for (j=0; j<10; j++)
		{
		  oldel[j]=(*m).elem[i*el_w+MCE1ELNOD1+j];
		}
	      for (j=0; j<10; j++)
		{
		  (*m).elem[i*el_w+MCE1ELNOD1+j]=oldel[norder[j]];
		}
	    }
	    }*/
      
    }
  else
    {
      elem_nodes=4;
      elem_type_vtk=10;
      vertex=(*m).vertex;
      if (vec_x!=NULL)
	{
	  my_vec_x.V=(*vec_x).V;
	  my_vec_x.len=(*vec_x).len;
	}
      if (scalar_x!=NULL)
	{
	  my_scalar_x.V=(*scalar_x).V;
	  my_scalar_x.len=(*scalar_x).len;
	}
    }

  /* prepare the names */
  TRY_MALLOC( vtkname, (namlen+5), char, 
	      mesh_write_solution_vtk_e1 );

  strncpy( vtkname, name, namlen );

  /* make sure no overread happens */
  vtkname[namlen-1]='\0';

  /* append ".pyr" resp. ".lat" */
  strcat( vtkname, ".vtk" );
  
  /* open the lat file */
  vtkfile=fopen(vtkname, "w");
  if (vtkfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_vtk_e1: error opening file %s\n",
	      vtkname);
      return FAIL;
    }

  /* write the header */
  fprintf(vtkfile, "# vtk DataFile Version 3.0\n"); /* version */
  fprintf(vtkfile, "file %s generated by feins\n", vtkname); /* description */
  fprintf(vtkfile, "ASCII\n"); /* ASCII or BINARY */
  fprintf(vtkfile, "DATASET UNSTRUCTURED_GRID\n"); /* Type of mesh */

  fprintf(vtkfile, "POINTS %8d double\n", (int) vx_nr); /* no of nodes */
  /* write the node positions */
  for (i=0; i<vx_nr; i++)
    {
      fprintf(vtkfile, "   %18.10e  %18.10e  %18.10e \n",
	      vertex[i*vx_w+MCE1VXSTRT  ],
	      vertex[i*vx_w+MCE1VXSTRT+1],
              vertex[i*vx_w+MCE1VXSTRT+2]);
    }

  fprintf(vtkfile, "\n");
  /* nr of elements, and number of list entries (=(1+elem_nodes)*el_nr) */
  fprintf(vtkfile, "CELLS %8"dFIDX" %8"dFIDX"\n", el_nr, (1+elem_nodes)*el_nr); 
  /* write the tetrahedra */
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(vtkfile, "   %2d ", elem_nodes);
      for (j=0; j<elem_nodes; j++)
	{
	  fprintf(vtkfile, " %8d ",
		  (int) (*m).elem[i*el_w+MCE1ELNOD1+j]);      
	}
      fprintf(vtkfile, "\n");
    }


  fprintf(vtkfile, "\n");
  /* define cell type for each element */
  fprintf(vtkfile, "CELL_TYPES %8d\n", (int) el_nr); 
  /* all elements are tetrahedra, type=10, or quad_tetrahedra, type=24 */
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(vtkfile, "   %d\n", elem_type_vtk);      
    }

  fprintf(vtkfile, "\n");
  /* switch to point data */
  fprintf(vtkfile, "POINT_DATA %8d\n", (int) vx_nr); 

  /* now write the vector data */
  if (vec_x!=NULL)
    {
      fprintf(vtkfile, "VECTORS feins_vector double\n"); 
      for (i=0; i<vx_nr; i++)
	{
	  fprintf(vtkfile, "  %15.7e  %15.7e %15.7e \n", 
		  my_vec_x.V[0*vx_nr+i], my_vec_x.V[1*vx_nr+i], my_vec_x.V[2*vx_nr+i]);
	}
      
    }



  /* now write the scalar data */
  if (scalar_x!=NULL)
    {
      for (j=0; j<scompo; j++)
	{
	  fprintf(vtkfile, "\nSCALARS feins_scalar_%02"dFIDX" double 1\n", j); 
	  fprintf(vtkfile, "LOOKUP_TABLE default \n"); 
	  for (i=0; i<vx_nr; i++)
	    {
	      fprintf(vtkfile, "  %15.7e\n",my_scalar_x.V[j*vx_nr+i]);
	    }
	}
      
    }
  
  /* close vtkfile */
  fclose(vtkfile);

  free(vtkname);

  if ( (*m).type==32 )
    {
      free(vertex);
      if (vec_x!=NULL)
	{
	  vector_free(&my_vec_x);
	}
      if (scalar_x!=NULL)
	{
	  vector_free(&my_scalar_x);
	}
    }

  return SUCCESS;
}

/*FUNCTION*/
int mesh_read_file_e1( struct mesh *m, struct solver_settings *set, char *name
/* checks the type of mesh file and calls the appropriate reader

   input:   name    - string containing the file name of the file to
                      be read, should end in ".f1m" (FEINS mesh)
		      or ".msh" (gmsh, corresponding files will
		      automatically be looked for as well)
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  int err; 

  if (strstr(name,".f1m")!=NULL)
    {
      err = mesh_read_file_f1m_e1( m, set, name);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_file_f1m_e1,
			      mesh_read_file_e1);
    }
  else if (strstr(name,".msh")!=NULL)
    {
      err = mesh_read_file_gmsh_e1( m, set, name);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_file_gmsh_e1,
			      mesh_read_file_e1);
    }
  else
    {
      fprintf(stderr,
	      "mesh_read_file_e1: unknown type of mesh file %s\n",
	      name);
      return FAIL;
    }

  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_file_f1m_e1( struct mesh *m,
			   struct solver_settings *set, 
			   char *name
/* reads a mesh file, for a description of the file format, see the
   example file "mesh_example.f1m"

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  int err;
  long secstart;
  FIDX format, dim, problem, meshgen, vx_nr, el_nr, bd_nr, holes,
    ps_nr, pv_nr, pc_nr, fn_nr, pa_nr, sg_nr, sp_nr; 
  char buff[1024];

  FILE *in;

  mesh_init(m);

  /* open the file */
  in = fopen( name, "r" );
  if (in == NULL)
    {
      fprintf(stderr, "mesh_read_file_f1m_e1: couldn't open file: %s\n",
	      name);
      return FAIL;	        
    }

  /* bin all the crap before <header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"<header>")!=0));
  if (strcmp(buff,"<header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_f1m_e1: file has no header: %s\n",
	      name);
      return FAIL;
    }

  /* save the begin of the header to secstart */
  secstart = ftell(in);

  /* get each of the header fields */
  err=mesh_read_header_key_int(in, "format", &format);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  if (format!=0)
    {
      fprintf(stderr, 
	      "mesh_read_file_f1m_e1: format of file unknown: %s\n",
	      name);
      return FAIL;
    } 

  err=mesh_read_header_key_int(in, "dim", &dim);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "problem", &problem);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "meshgen", &meshgen);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET); 
  (*m).meshgen = meshgen;
  (*m).meshgenDATA = NULL;

  err=mesh_read_header_key_int(in, "vertex", &vx_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "elements", &el_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "boundary", &bd_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "holes", &holes);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pcsurf", &ps_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pcvol", &pv_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pccrit", &pc_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "function", &fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "parameter", &pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "shape_seg", &sg_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "shape_par", &sp_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,
			  mesh_read_file_f1m_e1);
  fseek(in, secstart, SEEK_SET);

  /* jump to </header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"</header>")!=0));
  if (strcmp(buff,"</header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_f1m_e1: file corrupt header: %s\n",
	      name);
      return FAIL;
    }

  /* save the end of the header to secstart */
  secstart = ftell(in);

  /* header successfully read, now set the mesh */
  (*m).dim     = dim;
  (*m).type    = 31;
  (*m).problem = problem;

  /* set width of fields */
  (*m).vx_w = MCE1VXLN;
  (*m).el_w = MCE1ELLN;
  (*m).eg_w = MCE1EGLN;
  (*m).fc_w = MCE1FCLN;
  (*m).vo_w = MCE1VOLN;
  (*m).hi_w = MCE1HILN;
  (*m).bd_w = MCE1BDLN;
  (*m).ps_w = MCXXPSLN;
  (*m).pv_w = MCXXPVLN;
  (*m).pc_w = MCXXPCLN;
  (*m).fu_w = MC3XFULN;
  (*m).pa_w = MC3XPALN;
  (*m).sg_w = MC3XSGLN;
  (*m).sp_w = MC3XSPLN;

  err=mesh_read_solver_settings( in, set);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_solver_settings, 
			  mesh_read_file_f1m_e1);

  /* rewind in case no settings are given */
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_vertex( in, m, vx_nr, MCE1VXSTRT);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_vertex,
			  mesh_read_file_f1m_e1);

  err=mesh_read_elements_e1( in, m, el_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_elements_e1,
			  mesh_read_file_f1m_e1);

  err=mesh_read_boundary_e1( in, m, bd_nr, sg_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_boundary_e1, 
			  mesh_read_file_f1m_e1);

  err=mesh_remove_unused_vertices_e1( m);
  FUNCTION_FAILURE_HANDLE(err, mesh_remove_unused_vertices_e1, 
			  mesh_read_file_f1m_e1);
  
 
  /* save the position to secstart */
  secstart = ftell(in);
  //err=mesh_read_pccrit( in, m, pc_nr);
  //FUNCTION_FAILURE_HANDLE(err, mesh_read_pccrit,mesh_read_file_f1m_e1);

  /* rewind to read the pcsurf part */
  fseek(in, secstart, SEEK_SET);
  //err=mesh_read_pcsurf( in, m, ps_nr);
  //FUNCTION_FAILURE_HANDLE(err, mesh_read_pcsurf,mesh_read_file_f1m_e1);

  //err=mesh_read_pcvol( in, m, pv_nr);
  //FUNCTION_FAILURE_HANDLE(err, mesh_read_pcsurf,mesh_read_file_f1m_e1);

  /* read the shape segments and shape parameter */
  /* if (sg_nr>0)
    {
      err=mesh_read_sseg( in, m, sg_nr, sp_nr);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_sseg,mesh_read_file_f1m_e1);

      err=mesh_read_spar( in, m, sp_nr);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_spar,mesh_read_file_f1m_e1);
    } */

  /* rewind to read the other stuff */
  fseek(in, secstart, SEEK_SET);

  /* create an empty hierarchy part */
  (*m).hi_max=10;
  (*m).hi_nr =0;
  TRY_MALLOC( (*m).hier, (*m).hi_max*(*m).hi_w, FIDX, 
	      mesh_read_file_f1m_e1);

  /* create empty st */
  (*m).st_max=-1;
  (*m).st_nr =-1;
  (*m).st    = NULL;
  
  /* call the appropriate mesh-generator-wrapper (reading the holes
     section is left to that)  */
  switch(meshgen)
    {
    case 0: /* no mesh generator */
      if (sg_nr>0)
	{
	  /* adjust the position of nodes on the parametric boundary */
	  /*                    msh, insert, nI, dIdx, dIdF, type */
	  err=mesh_sseg_adjust_tx( m, 0,     0, NULL, NULL, 1); 
	  FUNCTION_FAILURE_HANDLE(err, mesh_sseg_adjust_tx,mesh_read_file_f1m_e1);
	}	  
      break;
        default: /* unknown, cry */
      fprintf(stderr, 
	      "mesh_read_file_f1m_e1: mesh generator unknown: %d\n",
	      (int) meshgen);
      return FAIL;
    }
    
  err=mesh_read_compute_conns_e1(m);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_compute_conns_e1,
			  mesh_read_file_f1m_e1);
	    
  err=mesh_read_function( in, m, fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_function, mesh_read_file_f1m_e1); 

  err=mesh_read_parameter( in, m, pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_parameter, mesh_read_file_f1m_e1);

  fclose(in);

  err=mesh_init_DD_elem( m );
  FUNCTION_FAILURE_HANDLE(err, mesh_init_DD_elem, mesh_read_file_f1m_e1);

  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_elements_e1( FILE *in, struct mesh *m, FIDX el_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            el_nr   - the number of elements in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the elem and face
                      part are allocated here, but only elem set!
                      in face only the RHSF part is defined and PCVL
                      is initiated
   

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  int  ihlp;
  FIDX i, j;
  char buff[1024];

  TRY_MALLOC((*m).elem, el_nr * (*m).el_w , FIDX,
	     mesh_read_elements_e1);
  (*m).el_max = el_nr;

  TRY_MALLOC((*m).vols, el_nr * (*m).vo_w , FIDX,
	     mesh_read_elements_e1);
  (*m).vo_max = el_nr;

  /* find the begin of the elements block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<elements>",buff)!=0));
  if (strcmp("<elements>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_elements_e1: file has no <elements>\n");
      return FAIL;
    }

  /* read all the elements one by one */
  for (i=0; i<el_nr; i++)
    {
      if ((fscanf(in, "%d", &ihlp) !=1)||(FIDX) (ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_elements_e1: order of entries wrong!\n");
	  return FAIL;
	}
      if ((fscanf(in, "%d", &ihlp) !=1)||(ihlp!=31))
	{
	  fprintf(stderr,
		  "mesh_read_elements_e1: element %d not e1!\n",(int) i);
	  return FAIL;
	}

      if (fscanf(in, "%d", &ihlp) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_elements_e1: entry %d deffect\n",(int) i);
	  return FAIL;
	}
      (*m).vols[i*(*m).vo_w+MCE1VORHSF ]= (FIDX) ihlp;
      (*m).vols[i*(*m).vo_w+MCE1VOFRED ]= -1;
      (*m).vols[i*(*m).vo_w+MCE1VOPCVL ]= -1;
 
      for (j=0; j<4; j++)
	{
	  if (fscanf(in, "%d", &ihlp) !=1)
	    {
	      fprintf(stderr,
		      "mesh_read_elements_e1: error reading entry %d!\n",
		      (int) i);
	      return FAIL;
	    }
	  (*m).elem[i*(*m).el_w+MCE1ELNOD1+j] = (FIDX) ihlp;
	}
      /* init DD subdomain of the element */
      (*m).elem[i*(*m).el_w+MCXXELDDPR] = -1;

    }
  (*m).el_nr  = el_nr;
  (*m).fc_nr  = el_nr;

  /* check if this is really the end of the elements block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</elements>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_elements_e1: "
	      "spurious end of elements block (no </elements> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_boundary_e1( FILE *in, struct mesh *m, FIDX bd_nr, FIDX sg_nr
/* reads the boundary part of a mesh file, for a description of the
   file format, see the example file "mesh_example.f1m" (!!!! should be 
   replaced by an 3D example file)

   input:   in      - a file open for reading
            bd_nr   - the number of boundary entries in the file, as
                      given by the header
            sg_nr   - the number of shape segment entries in the file,
	             		 as given by the header
   
   output:  m       - (given by reference) a mesh, the face and bound
                      part are allocated and set here, but in face
                      only those faces are defined which are part of
                      the boundary!

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  int  ihlp, err;
  FIDX i, j, nodes[3], nod1, nod2, eg, found;
  char buff[1024];
  struct ilist *vx2eg; /* an array of list anchors, where each list
			  entry is the id of an edge the vertex of
			  the lists anchor belongs to */
  struct ilist *pntr, *hlp;
                       /* auxiliary list pointers */
  
   
  TRY_MALLOC((*m).face, bd_nr * (*m).fc_w , FIDX,
	     mesh_read_boundary_e1);
  (*m).fc_max = bd_nr;

  TRY_MALLOC((*m).bound, bd_nr * (*m).bd_w , FIDX,
	     mesh_read_boundary_e1);
  (*m).bd_max = bd_nr;

  /*collect all edges of boundary elements in an array of list anchors*/	
  TRY_MALLOC(vx2eg, (*m).vx_nr , struct ilist, 
	     mesh_read_boundary_e1);

   /* initialise all the lists as empty */
  for (i=0; i<(*m).vx_nr; i++)
    {
      vx2eg[i].next=NULL;
      vx2eg[i].data=-1;
    }
	

 /* find the begin of the boundary block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<boundary>",buff)!=0));
  if (strcmp("<boundary>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_boundary_e1: file has no <boundary>\n");
      return FAIL;
    }

  /* read all the boundary one by one */
  for (i=0; i<bd_nr; i++)
    {
      if ((fscanf(in, "%d", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_boundary_e1: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%d", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_e1: "
		  "read error bound %d \n",(int) i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCE1BDTYPE]= (FIDX) ihlp;

      if ((fscanf(in, "%d", &ihlp) !=1)||(ihlp!=0))
	{
	  fprintf(stderr, "mesh_read_boundary_e1: "
		  "boundary element %d not t1!\n", (int) i);
	  return FAIL;
	}

      if (fscanf(in, "%d", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_e1: "
		  "read error bc type, boundary %d \n", (int) i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCE1BDFNCT]= (FIDX) ihlp;

      if (fscanf(in, "%d", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_e1: "
		  "read error bound %d \n", (int) i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCE1BDORIE]= (FIDX) ihlp;

      if (fscanf(in, "%d", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_e1: "
		  "read error bound %d \n", (int) i);
	  return FAIL;
	}
      if ((ihlp!=-1)&&(ihlp>=sg_nr))
	{
	  fprintf(stderr,"mesh_read_boundary_e1: "
		  "shape segment out of bounds, bound %d \n", (int) i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCE1BDSSEG]= (FIDX) ihlp;
 
      /*read nodes of a boundary element and store to nodes[3]*/
      for (j=0; j<3; j++)
	{
	  if (fscanf(in, "%d", &ihlp) !=1)
	    {
	      fprintf(stderr, "mesh_read_boundary_e1: "
		      "error reading entry %d!\n", (int) i);
	      return FAIL;
	    }
	      nodes[j] = (FIDX) ihlp;
	}

      for (j=0; j<3; j++)
	{
	  nod1= nodes[j];
	  nod2= nodes[(j+1)%3];

	  /* check if an edge eg with nod1 and nod2 exists */
	  found=0;
	  pntr=vx2eg[nod1].next;
	  hlp =&vx2eg[nod1];
	  while ((found==0)&&(pntr!=NULL))
	    {
	      eg=pntr->data;
	      if ((nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ])||
		  (nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]))
		{
		  found=1;
		}
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  
	   /* if not, create one */
	  if (found==0)
	    {
	      /* check if enough space is available */
	      if ((*m).eg_max<(*m).eg_nr+1) 
		{ 
		  err=mesh_more_edges( m, 3);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,
					   mesh_read_boundary_e1);
		}
	      /* create the edge */
	      eg=(*m).eg_nr;
	      
	      
	      (*m).eg_nr++;

	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ]=nod1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]=nod2;
	      (*m).edge[eg*(*m).eg_w+MCE1EGCHL1  ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGGEO   ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGLVL   ]= -1;

	      /* register the new edge in vx2eg */
	      /* append an entry to the end of the nod1 list */
	      TRY_MALLOC(hlp->next, 1, struct ilist,  
			 mesh_read_boundary_e1);
	      pntr       = hlp->next;
	      pntr->data = eg;
	      pntr->next = NULL;

	       /* append an entry to the end of the nod2 list */
	      pntr=&vx2eg[nod2];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			mesh_read_boundary_e1);
	      pntr       = pntr->next;
	      pntr->data = eg;
	      pntr->next = NULL;
	      
	    }

	  /* now we have the edge eg which holds both nodes, make it
	     edge of the face corresponding to the i-th boundary element */
	  (*m).face[i*(*m).fc_w+MCE1FCEDG1+j]=eg;

		

 	}
    

 
     /* complete the definition of the face and bound entry */
      (*m).bound[i*(*m).bd_w+MCE1BDFACE]=  i;
      (*m).face[i*(*m).fc_w+MCE1FCCHL1] = -1;
      (*m).face[i*(*m).fc_w+MCE1FCBND ] =  i;
      (*m).face[i*(*m).fc_w+MCE1FCGEO ] = -1;
      (*m).face[i*(*m).fc_w+MCE1FCPCSU] = -1;
      (*m).face[i*(*m).fc_w+MCE1FCLVL]  = -1;
	
    }
  (*m).fc_nr   = bd_nr;
  (*m).bd_nr   = bd_nr;
  
  /* check if this is really the end of the boundary block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</boundary>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_boundary_e1: "
	      "spurious end of boundary block (no </boundary> ?)\n");
      return FAIL;
    }

  /* free memory that is no longer required */
  for (i=0; i<(*m).vx_nr; i++)
    {
      ilist_free(&vx2eg[i].next);
    }
  free(vx2eg);

  return SUCCESS;
}



/*FUNCTION*/
int mesh_remove_unused_vertices_e1( struct mesh *m
/* removes unused vertices, by renumbering all vertices

   In/Out:  m       - (given by reference) a mesh, the vertex, edge
                      and elements parts are modified if necessary

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX i, j, el, eg, vx_nr, el_nr, eg_nr, vx_w, el_w, eg_w, new_vx_nr;

  int *vx_used;   /* vector telling if vertex[i] is used elsewhere
		     (==1)  or not (==0) */
  FIDX *newnumber;/* vector storing the renumbering of the vertices */

  vx_nr = (*m).vx_nr;
  el_nr = (*m).el_nr;
  eg_nr = (*m).eg_nr;

  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;
  eg_w  = (*m).eg_w;

  TRY_MALLOC(vx_used, vx_nr, int, 
	     mesh_remove_unused_vertices_e1);
  TRY_MALLOC(newnumber, vx_nr , FIDX, 
	     mesh_remove_unused_vertices_e1);

  /* first initialise all nodes as not used */
  for (i=0; i<vx_nr; i++)
    {
      vx_used[i]=0;
    }

  /* go through all elements, mark vertices if used */
  for (el=0; el<el_nr; el++)
    for (i=0; i<4; i++)
      {
	vx_used[(*m).elem[el*el_w+MCE1ELNOD1+i]]=1;
      }
  /* go through all edges, mark vertices if used */
  for (eg=0; eg<eg_nr; eg++)
    for (i=0; i<2; i++)
      {
	vx_used[(*m).edge[eg*eg_w+MCE1EGNOD1+i]]=1;
      }

  /* now all that are in used should be marked, create the
     renumbering and renumber the vertices */
  new_vx_nr=0;
  for (i=0; i<vx_nr; i++)
    {
      if (vx_used[i]==1)
	{
	  newnumber[i]=new_vx_nr;
	  for (j=0; j<vx_w; j++)
	    {
	      (*m).vertex[new_vx_nr*vx_w+j]=(*m).vertex[i*vx_w+j];
	    }
	  new_vx_nr++;
	}
      else
	{
	  /* give unused ones an invalid number, to be save */
	  newnumber[i]=-1;
	}
    }

  /* go through all elements, renumber vertices */
  for (el=0; el<el_nr; el++)
    for (i=0; i<4; i++)
      {
	(*m).elem[el*el_w+MCE1ELNOD1+i]=newnumber[(*m).elem[el*el_w+MCE1ELNOD1+i]];
      }
  /* go through all edges, mark vertices if used */
  for (eg=0; eg<eg_nr; eg++)
    for (i=0; i<2; i++)
      {
	(*m).edge[eg*eg_w+MCE1EGNOD1+i]=newnumber[(*m).edge[eg*eg_w+MCE1EGNOD1+i]];
      }

  if (vx_nr!=new_vx_nr)
    {
      fprintf(stderr, "\nwarning: there where %"dFIDX" unused vertices removed \n"
	      "by mesh_remove_unused_vertices_e1, new vx_nr=%"dFIDX"\n\n",
	      vx_nr-new_vx_nr, new_vx_nr);
    }

  (*m).vx_nr=new_vx_nr;

  /* free memory */
  free(newnumber);
  free(vx_used);

  return SUCCESS;
  
}

/*FUNCTION*/
int mesh_read_compute_conns_e1( struct mesh *m
/* builds the part of the connectivity data of the internal mesh
   format that is not covered by the mesh files (see the example file
   "mesh_example.f1m")

   In/Out:  m       - (given by reference) a mesh, the edge and face
                      parts are completed

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX i, j, el, eg, fc, nod1, nod2, edge[3], found;
  int  err;

  FIDX sort_vx2eg[6*2]={ 0,1, 1,2, 0,2, 0,3, 1,3, 2,3 };
  FIDX sort_eg2fc[4*3]={ 0,1,2, 0,3,4, 2,3,5, 1,4,5 };
  FIDX local_edges[6];

  struct ilist *vx2eg; /* an array of list anchors, where each list
			  entry is the id of an edge the vertex of
			  the lists anchor belongs to */
  struct ilist *eg2fc; /* an array of list anchors, where each list
			  entry is the id of a face the edge of
			  the lists anchor belongs to */
  struct ilist *pntr, *hlp;
                       /* auxiliary list pointers */

  TRY_MALLOC(vx2eg, (*m).vx_nr , struct ilist, 
	     mesh_read_compute_conns_e1);
  /* number of edges still unknown, use upper bound for alloc, 
     eg_nr <= 6*el_nr */
  TRY_MALLOC(eg2fc, 6*(*m).el_nr, struct ilist, 
	     mesh_read_compute_conns_e1);
  /* initialise all the lists as empty */
  for (i=0; i<(*m).vx_nr; i++)
    {
      vx2eg[i].next=NULL;
      vx2eg[i].data=-1;
    }
    
  for (i=0; i<6*(*m).el_nr; i++)
    {
      eg2fc[i].next=NULL;
      eg2fc[i].data=-1;
    }

  /* take all the known edges (from the boundary) into the list */
  for (i=0; i<(*m).eg_nr; i++)
    {
      /* both vertices of the edge */
      for (j=0; j<2; j++)
	{
	  pntr=&vx2eg[(*m).edge[i*(*m).eg_w+MCE1EGNOD1+j]];
	  /* find the end of the list of this vertex */
	  while (pntr->next!=NULL) pntr=pntr->next;
	  /* append an entry to the list */
	  TRY_MALLOC(pntr->next, 1, struct ilist,  
		     mesh_read_compute_conns_e1);
	  pntr       = pntr->next;
	  pntr->data = i;
	  pntr->next = NULL;
	}
    }
    
  /* take all the known faces (from the boundary) into the list */
  for (i=0; i<(*m).fc_nr; i++)
    {
      /* all three edges of the face */
      for (j=0; j<3; j++)
	{
	  pntr=&eg2fc[(*m).face[i*(*m).fc_w+MCE1FCEDG1+j]];
	  /* find the end of the list of this edge */
	  while (pntr->next!=NULL) pntr=pntr->next;
	  /* append an entry to the list */
	  TRY_MALLOC(pntr->next, 1, struct ilist,  
		     mesh_read_compute_conns_e1);
	  pntr       = pntr->next;
	  pntr->data = i;
	  pntr->next = NULL;
	}
    }    

  /* now for each element */
  for (el=0; el<(*m).el_nr; el++)
    {
      /* First: Check if the edges exist and create them if necessary */
      for (j=0; j<6; j++)
	{
	  nod1=(*m).elem[el*(*m).el_w+MCE1ELNOD1+ sort_vx2eg[2*j  ] ];
	  nod2=(*m).elem[el*(*m).el_w+MCE1ELNOD1+ sort_vx2eg[2*j+1] ];

	  /* check if an edge eg with nod1 and nod2 exists */
	  found=0;
	  pntr=vx2eg[nod1].next;
	  hlp =&vx2eg[nod1];
	  while ((found==0)&&(pntr!=NULL))
	    {
	      eg=pntr->data;
	      if ((nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ])||
		  (nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]))
		{
		  found=1;
		}
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  
	  /* if not, create one */
	  if (found==0)
	    {
	      /* check if enough space is available */
	      if ((*m).eg_max<(*m).eg_nr+1) 
		{ 
		  err=mesh_more_edges( m, 3);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,
					   mesh_read_compute_conns_e1);
		}
	      /* create the edge */
	      eg=(*m).eg_nr;
	      (*m).eg_nr++;

	      /* fprintf(stderr," new edge %d\n", eg); /* */

	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ]=nod1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]=nod2;
	      (*m).edge[eg*(*m).eg_w+MCE1EGCHL1  ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGGEO   ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGLVL   ]= -1;

	      /* register the new edge in vx2eg */
	      /* append an entry to the end of the nod1 list */
	      TRY_MALLOC(hlp->next, 1, struct ilist,  
			 mesh_read_compute_conns_e1);
	      pntr       = hlp->next;
	      pntr->data = eg;
	      pntr->next = NULL;

	      /* append an entry to the end of the nod2 list */
	      pntr=&vx2eg[nod2];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			 mesh_read_compute_conns_e1);
	      pntr       = pntr->next;
	      pntr->data = eg;
	      pntr->next = NULL;
	      
	    }
	  local_edges[j]=eg;
	}
      /* Second: Check if the faces exist and create them if necessary*/
      for (j=0; j<4; j++)
	{
	  for (i=0; i<3; i++)
	    {
	      edge[i]=local_edges[sort_eg2fc[3*j+i]];
	    }
	  
	  /* check if a face fc with edge[0], edge[1] and edge[2], exists */
	  found=0;
	  pntr=eg2fc[edge[0]].next;
	  hlp =&eg2fc[edge[0]];
	  while ((found==0)&&(pntr!=NULL))
	    {
	      fc=pntr->data;
	      if ((edge[1]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1  ])||
		  (edge[1]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1+1])||
		  (edge[1]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1+2]))
		{
	          if ((edge[2]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1  ])||
		      (edge[2]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1+1])||
		      (edge[2]==(*m).face[fc*(*m).fc_w+MCE1FCEDG1+2]))
		    {
		      found=1;
		    }
		  else
		    {
		      fprintf(stderr, "mesh_read_compute_conns_e1: "
	      	       "two faces found that share only two edges\n");
                      return FAIL;
		    }
		}
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  
	  /* if not, create one */
	  if (found==0)
	    {
	      /* check if enough space is available */
	      if ((*m).fc_max<(*m).fc_nr+1) 
		{ 
		  err=mesh_more_faces( m, 3);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_faces,
					   mesh_read_compute_conns_e1);
		}
	      /* create the face */
	      fc=(*m).fc_nr;
	      (*m).fc_nr++;

	      for (i=0; i<3; i++)
	        {
		  (*m).face[fc*(*m).fc_w+MCE1FCEDG1+i] = edge[i];
		}
	      (*m).face[fc*(*m).fc_w+MCE1FCCHL1  ]= -1;
	      (*m).face[fc*(*m).fc_w+MCE1FCBND   ]= -1;
	      (*m).face[fc*(*m).fc_w+MCE1FCGEO   ]= -1;
	      (*m).face[fc*(*m).fc_w+MCE1FCLVL   ]= -1;
	      (*m).face[fc*(*m).fc_w+MCE1FCPCSU  ]= -1;

	      /* register the new face in eg2fc */
	      /* append an entry to the end of the edge[0] list */
	      pntr=&eg2fc[edge[0]];
	      /* find the end of the list of this vertex */
	      TRY_MALLOC(hlp->next, 1, struct ilist,  
			 mesh_read_compute_conns_e1);
	      pntr       = hlp->next;
	      pntr->data = fc;
	      pntr->next = NULL;


	      /* append an entry to the end of the edge[1] list */
	      pntr=&eg2fc[edge[1]];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			 mesh_read_compute_conns_e1);
	      pntr       = pntr->next;
	      pntr->data = fc;
	      pntr->next = NULL;

	      /* append an entry to the end of the edge[2] list */
	      pntr=&eg2fc[edge[2]];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			 mesh_read_compute_conns_e1);
	      pntr       = pntr->next;
	      pntr->data = fc;
	      pntr->next = NULL;	      
	    }
	  (*m).vols[el*(*m).vo_w+MCE1VOFAC1+ j]=fc;

	}
    }
  /* now announce that the number of volumes is the same as number of
     elements */
  (*m).vo_nr=(*m).el_nr;


  /* check the connectivity of the pcsurf entries, which faces they
     belong to */
  for (i=0; i<(*m).ps_nr; i++)
    {
      FIDX face, found;

      fprintf(stderr, "mesh_read_compute_conns_e1: pcsurf still not adapted to 3D\n");
      return FAIL;

      face= (*m).pcsurf[i*(*m).ps_w+MCXXPSSURF];

      /* search elements containing this edge */
      (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE+0]= -1;
      (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE+1]= -1;
      found=0;
      j=0;
      while ((found<2)&&(j<(*m).vo_nr))
	{
	  FIDX elface;
	  for (elface=0; elface<3; elface++)
	    {
	      if (face==(*m).vols[j*(*m).vo_w+MCE1VOFAC1+elface])
		{
		  /* this face contains the edge */
		  (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE]= j;
		  found++;
		}
	    }
	  j++;
	} /* end search element for this edge */

      if (found==-1)
	{
	  fprintf(stderr,
		  "mesh_read_compute_conns_e1: no face found with "
		  "psurf edge, pcsruf entry %d \n", (int) i);
	  return FAIL;
	}
    }

  /* clean up */
  /* empty the vx2eg list */
  for (i=0; i<(*m).vx_nr; i++)
    {
      pntr=vx2eg[i].next;
      while (pntr!=NULL)
	{
	  hlp  = pntr;
	  pntr = pntr->next;
	  free(hlp);
	}
    }
  free(vx2eg);

  /* empty the eg2fc list */
  for (i=0; i<(*m).eg_nr; i++)
    {
      pntr=eg2fc[i].next;
      while (pntr!=NULL)
	{
	  hlp  = pntr;
	  pntr = pntr->next;
	  free(hlp);
	}
    }
  free(eg2fc);

  return SUCCESS;
}

/*FUNCTION*/
int mesh_read_file_gmsh_e1( struct mesh *m,
			    struct solver_settings *set, 
			    char *name
/* reads a mesh file generated with gmsh

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  int err; 
  int reported;
  long secstart;
  FIDX format, dim, problem, meshgen, curved, 
    vx_nr, eg_nr, fc_nr, vo_nr, el_nr, 
    ps_nr, pv_nr, pc_nr, fn_nr, pa_nr, gmsh_elnr; 
  FIDX ihlp, i, j, max_geo_tag, rhsf_nr, fst_bd_nr;
  FIDX *geo_to_bnd, *bnd_fst; 
  FIDX *geo_to_rhsf, *rhsf_fst; 
  double dhlp;
  char buff[1024];
  char *fst_name, *name_end_pos;

  struct ilist *vx2eg; /* an array of list anchors, where each list
			  entry is the id of an edge the vertex of
			  the lists anchor belongs to */
  struct ilist *pntr, *hlp;
                       /* auxiliary list pointers */
  
  FILE *in, *fst;
  

  mesh_init(m);

  /* set type of meshgen */
  (*m).meshgen = 2; /* type gmsh */

  dim = 3;
  (*m).dim = dim;
  (*m).type    = 31;

  /* set width of fields */
  (*m).vx_w = MCE1VXLN;
  (*m).el_w = MCE1ELLN;
  (*m).eg_w = MCE1EGLN;
  (*m).fc_w = MCE1FCLN;
  (*m).vo_w = MCE1VOLN;
  (*m).hi_w = MCE1HILN;
  (*m).bd_w = MCE1BDLN;
  (*m).ps_w = MCXXPSLN;
  (*m).pv_w = MCXXPVLN;
  (*m).pc_w = MCXXPCLN;
  (*m).fu_w = MC3XFULN;
  (*m).pa_w = MC3XPALN;
  (*m).sg_w = MC3XSGLN;
  (*m).sp_w = MC3XSPLN;

  /* get the .fst filename */
  TRY_MALLOC( fst_name, strlen(name)+5, char, mesh_read_file_gmsh_e1);
  strcpy(fst_name, name);
  name_end_pos = strstr(fst_name, ".msh");
  if (name_end_pos==NULL)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: filename should end in .msh : %s\n",
	      fst_name);
      return FAIL;	        
    }
  strcpy(name_end_pos, ".fst\0");

  /* open the files */
  in = fopen( name, "r" );
  if (in == NULL)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: couldn't open file: %s\n",
	      name);
      return FAIL;	        
    }
  fst = fopen( fst_name, "r" );
  if (fst == NULL)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: couldn't open file: %s\n",
	      fst_name);
      return FAIL;	        
    }
  
 
  /*****************************************************************
   *****************************************************************
   *********** read the msh file  **********************************
   *****************************************************************
   *****************************************************************/

  /* bin all the crap before $Nodes */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"$Nodes")!=0));
  if (strcmp(buff,"$Nodes")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file has no \"$Nodes\" : %s\n",
	      name);
      return FAIL;
    }
  /* read the number of nodes/vertices */
  if (fscanf(in, "%"dFIDX"", &vx_nr)!=1)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: error reading node number,"
	      " file : %s\n", name);
      return FAIL;
    }
  TRY_MALLOC((*m).vertex, vx_nr * (*m).vx_w , double,
	     mesh_read_file_gmsh_e1);
  (*m).vx_max = vx_nr;

  /* read the vertices */
  for (i=0; i<vx_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(ihlp!=i+1))
	{
	  fprintf(stderr, "mesh_read_file_gmsh_e1: "
		  "order of node entries wrong!\n");
	  return FAIL;
	}
      for (j=0; j<dim; j++)
	{
	  if (fscanf(in, "%lg", &dhlp) !=1)
	    {
	      fprintf(stderr, "mesh_read_file_gmsh_e1: "
		      "error reading node entry %"dFIDX"!\n",
		       i);
	      return FAIL;
	    }
	  (*m).vertex[i*(*m).vx_w+MCE1VXSTRT+j] = dhlp;
	}
    } /* end loop reading vertices */
      
    /* check if this is really the end of the vertex block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("$EndNodes",buff)!=0))
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: spurious end of nodes block "
	      "(no $EndNodes ?)\n");
      return FAIL;
    }
  (*m).vx_nr=vx_nr;

  /* bin all the crap before $Elements */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"$Elements")!=0));
  if (strcmp(buff,"$Elements")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file has no \"$Elements\" : %s\n",
	      name);
      return FAIL;
    }
  /* read the number of gmsh-elements */
  if (fscanf(in, "%"dFIDX"", &gmsh_elnr)!=1)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: error reading element number,"
	      " file : %s\n", name);
      return FAIL;
    }
  /* gmsh elements is more tricky, as points, lines, faces and volumes
     all count as elements, 

     thus we initialise the maximal size of edges, faces, volumes and
     elements to the specified number of elements, (so we allocate
     more memory than needed) */

  TRY_MALLOC((*m).edge, gmsh_elnr * (*m).eg_w , FIDX,
	     mesh_read_file_gmsh_e1);
  (*m).eg_max = gmsh_elnr;
  eg_nr = 0;

  TRY_MALLOC((*m).face, gmsh_elnr * (*m).fc_w , FIDX,
	     mesh_read_file_gmsh_e1);
  (*m).fc_max = gmsh_elnr;
  fc_nr = 0;

  TRY_MALLOC((*m).vols, gmsh_elnr * (*m).vo_w , FIDX,
	     mesh_read_file_gmsh_e1);
  (*m).vo_max = gmsh_elnr;
  vo_nr = 0;

  TRY_MALLOC((*m).elem, gmsh_elnr * (*m).el_w , FIDX,
	     mesh_read_file_gmsh_e1);
  (*m).el_max = gmsh_elnr;
  el_nr = 0;

  max_geo_tag=-1;

  /* read all the gmsh-elements one by one */
  reported = 0;
  for (i=0; i<gmsh_elnr; i++)
    {
      int type;
      int numtags;
      FIDX geotag;
      FIDX neweg, newfc, newel;
      FIDX nod1, nod2, nod3, nod4;

      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(FIDX) (ihlp!=i+1))
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: order of entries wrong!\n");
	  return FAIL;
	}
      if (fscanf(in, "%d", &type) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading type of element %"dFIDX" !\n", i+1);
	  return FAIL;
	}

      if (fscanf(in, "%d", &numtags) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading numtags of element %"dFIDX" !\n", i+1);
	  return FAIL;
	}
      geotag=-1;
      for (j=0; j<numtags; j++)
	{
	  if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	    {
	      fprintf(stderr,
		      "mesh_read_file_gmsh_e1: "
		      "error reading %"dFIDX"-th tag of element %"dFIDX" !\n", 
		      j+1, i+1);
	      return FAIL;
	    }
	  /* only keep last nonzero geotag */
	  if (ihlp!=0)
	    {
	      geotag=ihlp;
	    }
	}
      if ((numtags > 2)&&(reported<5))
	{
	  reported++;
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "warning, element %"dFIDX" has more than two geometry-tags,"
		  " only last nonzero is used (tag=%"dFIDX")! (report only first 5 warnings)\n", 
		  i+1, geotag);
	}
      if (geotag>max_geo_tag)
	{
	  max_geo_tag = geotag;
	}

      switch(type)
	{
	case 15: /* point -> ignore (but read) */
	  if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	    {
	      fprintf(stderr,
		      "mesh_read_file_gmsh_e1: "
		      "point entry %"dFIDX" deffect\n", i+1);
	      return FAIL;
	    }
	  break;
	case 1:  /* line */
	  /* create new edge */
	  neweg=eg_nr;
	  if (fscanf(in, "%"dFIDX" %"dFIDX"", 
		     &nod1, &nod2) !=2)
	    {
	      fprintf(stderr,
		      "mesh_read_file_gmsh_e1: "
		      "line entry %"dFIDX" deffect\n", i+1);
	      return FAIL;
	    }
	  (*m).edge[neweg*(*m).eg_w+MCE1EGNOD1  ]= nod1-1;
	  (*m).edge[neweg*(*m).eg_w+MCE1EGNOD1+1]= nod2-1;
	  (*m).edge[neweg*(*m).eg_w+MCE1EGCHL1  ]= -1;
	  (*m).edge[neweg*(*m).eg_w+MCE1EGGEO   ]= geotag;
	  (*m).edge[neweg*(*m).eg_w+MCE1EGLVL   ]= -1;
	  eg_nr++;
	  break;
	case 2:  /* face */
	  /* create new face */
	  newfc=fc_nr;
          /* writing node numbers in face, will be corrected afterwards */
	  if (fscanf(in, "%"dFIDX" %"dFIDX" %"dFIDX"", 
		     &nod1, &nod2, &nod3) !=3)
	    {
	      fprintf(stderr,
		      "mesh_read_file_gmsh_e1: "
		      "face entry %"dFIDX" deffect\n", i+1);
	      return FAIL;
	    }
	  (*m).face[newfc*(*m).fc_w+MCE1FCEDG1  ]= nod1-1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCEDG1+1]= nod2-1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCEDG1+2]= nod3-1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCBND ]= -1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCGEO ]= geotag;
	  (*m).face[newfc*(*m).fc_w+MCE1FCPCSU]= -1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCCHL1]= -1;
	  (*m).face[newfc*(*m).fc_w+MCE1FCLVL ]= -1;
	  fc_nr++;
	  break;
	case 4:  /* volume */
	  /* create new face */
	  newel=el_nr;
	  if (fscanf(in, "%"dFIDX" %"dFIDX" %"dFIDX" %"dFIDX"", 
		     &nod1, &nod2, &nod3, &nod4) !=4)
	    {
	      fprintf(stderr,
		      "mesh_read_file_gmsh_e1: "
		      "tetra entry %"dFIDX" deffect\n", i+1);
	      return FAIL;
	    }
	  (*m).elem[newel*(*m).el_w+MCE1ELNOD1  ]= nod1-1;
	  (*m).elem[newel*(*m).el_w+MCE1ELNOD1+1]= nod2-1;
	  (*m).elem[newel*(*m).el_w+MCE1ELNOD1+2]= nod3-1;
	  (*m).elem[newel*(*m).el_w+MCE1ELNOD1+3]= nod4-1;
	  /* init DD subdomain of the element */
	  (*m).elem[newel*(*m).el_w+MCXXELDDPR] = -1;

	  /* writing geotag in MCE1VORHSF, will be corrected afterwards */
	  (*m).vols[newel*(*m).vo_w+MCE1VORHSF]= geotag;
	  (*m).vols[newel*(*m).vo_w+MCE1VOFRED]= -1;
	  (*m).vols[newel*(*m).vo_w+MCE1VOPCVL]= -1;
	  el_nr++;
	  vo_nr=el_nr;
	  break;
	default:
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "entry %"dFIDX" unknown type %d\n", i+1, type);
	  return FAIL;
	} /* switch type */
    } /* loop elements */

  /* check if this is really the end of the elements block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("$EndElements",buff)!=0))
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: spurious end of elements block "
	      "(no $EndElements ?)\n");
      return FAIL;
    }
  (*m).eg_nr = eg_nr;
  (*m).fc_nr = fc_nr;
  (*m).el_nr = el_nr;
  (*m).vo_nr = vo_nr;

  fclose(in);

  /*****************************************************************
   *****************************************************************
   *********** read the fst file  **********************************
   *****************************************************************
   *****************************************************************/

  /* bin all the crap before <header> */
  while ((fscanf(fst, "%1023s", buff)==1)&&(strcmp(buff,"<header>")!=0));
  if (strcmp(buff,"<header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file has no header: %s\n",
	      fst_name);
      return FAIL;
    }

  /* save the begin of the header to secstart */
  secstart = ftell(fst);

  /* get each of the header fields */
  err=mesh_read_header_key_int(fst, "format", &format);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  if (format!=0)
    {
      fprintf(stderr, 
	      "mesh_read_file_gmsh_e1: format of file unknown: %s\n",
	      fst_name);
      return FAIL;
    } 

  err=mesh_read_header_key_int(fst, "dim", &dim);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "problem", &problem);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "meshgen", &meshgen);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET); 
  if (meshgen!=2)
    {
      fprintf(stderr, 
	      "mesh_read_file_gmsh_e1: meshgen should be 2 (gmsh) file: %s\n",
	      fst_name);
      return FAIL;
    } 

  err=mesh_read_header_key_int(fst, "curved", &curved);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET); 
  if (curved!=0)
    {
      /* save the file name for later use */
      TRY_MALLOC( (*m).meshgenDATA, strlen(name)+1, char, mesh_read_file_gmsh_e1);
      strcpy( (*m).meshgenDATA, name);
      (*m).curved=curved;
    } 

  err=mesh_read_header_key_int(fst, "rhsf", &rhsf_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "boundary", &fst_bd_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);


  err=mesh_read_header_key_int(fst, "pcsurf", &ps_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "pcvol", &pv_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "pccrit", &pc_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "function", &fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  err=mesh_read_header_key_int(fst, "parameter", &pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_gmsh_e1);
  fseek(fst, secstart, SEEK_SET);

  /* jump to </header> */
  while ((fscanf(fst, "%1023s", buff)==1)&&(strcmp(buff,"</header>")!=0));
  if (strcmp(buff,"</header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file corrupt header: %s\n",
	      name);
      return FAIL;
    }

  /* save the end of the header to secstart */
  secstart = ftell(fst);

  err=mesh_read_solver_settings( fst, set);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_solver_settigs, 
			  mesh_read_file_gmsh_e1);

  /* rewind in case no settings are given */
  fseek(fst, secstart, SEEK_SET);

  /* read the boundary block */
  TRY_MALLOC(geo_to_bnd, max_geo_tag+1, FIDX, mesh_read_file_gmsh_e1);
  for (i=0; i<=max_geo_tag; i++)
    {
      geo_to_bnd[i]=-1;
    }

#define FSTBDTYPE (0)
#define FSTBDFUNR (FSTBDTYPE+1)
#define FSTBDLN   (FSTBDFUNR+1)
  TRY_MALLOC(bnd_fst, fst_bd_nr*FSTBDLN, FIDX, mesh_read_file_gmsh_e1);
  /* seek to <boundary> */
  while ((fscanf(fst, "%1023s", buff)==1)&&(strcmp(buff,"<boundary>")!=0));
  if (strcmp(buff,"<boundary>")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file has no <boundary>: %s\n",
	      fst_name);
      return FAIL;
    }

  for (i=0; i<fst_bd_nr; i++)
    {
      FIDX type, fun, geo;
      if ((fscanf(fst, "%"dFIDX"", &ihlp) !=1)||(ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "fst boundary order wrong, entry %"dFIDX"\n",
		  i);
	  return FAIL;
	}
      if (fscanf(fst, "%"dFIDX" %"dFIDX" %"dFIDX"", &type, &fun, &geo) !=3)
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading boundary entry %"dFIDX" of %s\n", 
		  i, fst_name);
	  return FAIL;
	}
      bnd_fst[i*FSTBDLN+FSTBDTYPE]=type;
      bnd_fst[i*FSTBDLN+FSTBDFUNR]=fun;
      if (geo<=max_geo_tag)
	{
	  geo_to_bnd[geo]=i;
	}
      else
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading boundary entry %"dFIDX" of %s\n"
		  "geo=%"dFIDX" max_geo_tag=%"dFIDX" in %s", 
		  i, fst_name, geo, max_geo_tag, name);
	  return FAIL;
	}
    }
  /* check for </boundary> */
  if ((fscanf(fst, "%1023s", buff)!=1)||(strcmp(buff,"</boundary>")!=0))
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: "
	      "spurious end of </boundary> in file %s\n",
	      fst_name);
      return FAIL;
    }




  /* read the rhsf block */
  TRY_MALLOC(geo_to_rhsf, max_geo_tag+1, FIDX, mesh_read_file_gmsh_e1);
  for (i=0; i<=max_geo_tag; i++)
    {
      geo_to_rhsf[i]=-1;
    }

#define FSTRHSFTYPE (0)
#define FSTRHSFFUNR (FSTRHSFTYPE+1)
#define FSTRHSFLN   (FSTRHSFFUNR+1)
  TRY_MALLOC(rhsf_fst, rhsf_nr*FSTRHSFLN, FIDX, mesh_read_file_gmsh_e1);
  /* seek to <rhsf> */
  while ((fscanf(fst, "%1023s", buff)==1)&&(strcmp(buff,"<rhsf>")!=0));
  if (strcmp(buff,"<rhsf>")!=0)
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: file has no <rhsf>: %s\n",
	      fst_name);
      return FAIL;
    }

  for (i=0; i<rhsf_nr; i++)
    {
      FIDX type, fun, geo;
      if ((fscanf(fst, "%"dFIDX"", &ihlp) !=1)||(ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "fst rhsf order wrong, entry %"dFIDX"\n",
		  i);
	  return FAIL;
	}
      if (fscanf(fst, "%"dFIDX" %"dFIDX" %"dFIDX"", &type, &fun, &geo) !=3)
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading rhsf entry %"dFIDX" of %s\n", 
		  i, fst_name);
	  return FAIL;
	}
      rhsf_fst[i*FSTBDLN+FSTBDTYPE]=type;
      rhsf_fst[i*FSTBDLN+FSTBDFUNR]=fun;
      if (geo<=max_geo_tag)
	{
	  geo_to_rhsf[geo]=i;
	}
      else
	{
	  fprintf(stderr,
		  "mesh_read_file_gmsh_e1: "
		  "error reading rhsf entry %"dFIDX" of %s\n"
		  "geo=%"dFIDX" max_geo_tag=%"dFIDX" in %s", 
		  i, fst_name, geo, max_geo_tag, name);
	  return FAIL;
	}
    }
  /* check for </rhsf> */
  if ((fscanf(fst, "%1023s", buff)!=1)||(strcmp(buff,"</rhsf>")!=0))
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: "
	      "spurious end of </rhsf> in file %s\n",
	      fst_name);
      return FAIL;
    }

  


  /* same procedure for pcsurf and pcvol, to be implemented later */
  if ((ps_nr!=0)||(pv_nr!=0))
    {
      fprintf(stderr, "mesh_read_file_gmsh_e1: "
	      "pcsurf and pcvol not yet implemented, file %s\n",
	      fst_name);
      return FAIL;
    }
  err=mesh_read_function( fst, m, fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_function, mesh_read_file_gmsh_e1); 

  err=mesh_read_parameter( fst, m, pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_parameter, mesh_read_file_gmsh_e1);

  fclose(fst);

  /* now fill in all bound entries */
  /* there will be at most fc_nr boudary entries now */
  TRY_MALLOC((*m).bound, fc_nr*(*m).bd_w, FIDX, mesh_read_file_gmsh_e1);
  (*m).bd_max=fc_nr;
  for (i=0; i<fc_nr; i++)
    {
      FIDX geo=(*m).face[i*(*m).fc_w+MCE1FCGEO];
      if (geo_to_bnd[geo]!=-1)
	{
	  /* create new bnd-entry */
	  FIDX newbd=(*m).bd_nr;
	  FIDX fst_bd_entry= geo_to_bnd[geo];

	  (*m).bound[newbd*(*m).bd_w+MCE1BDFACE]=  i;
	  (*m).bound[newbd*(*m).bd_w+MCE1BDTYPE]=
	    bnd_fst[fst_bd_entry*FSTBDLN+FSTBDTYPE];
	  (*m).bound[newbd*(*m).bd_w+MCE1BDFNCT]=
	    bnd_fst[fst_bd_entry*FSTBDLN+FSTBDFUNR];
	  (*m).bound[newbd*(*m).bd_w+MCE1BDORIE]=  1;
	  (*m).bound[newbd*(*m).bd_w+MCE1BDSSEG]= -1;

	  (*m).face[i*(*m).fc_w+MCE1FCBND ] = newbd;

	  (*m).bd_nr++;
	}
    }


  /* adjust the RHSF entry of all vols from geotag to RHSF */
  for (i=0; i<vo_nr; i++)
    {
      FIDX geo=(*m).vols[i*(*m).vo_w+MCE1VORHSF];
      if (geo_to_rhsf[geo]!=-1)
	{
	  FIDX fst_rhsf_entry= geo_to_rhsf[geo];
	  (*m).vols[i*(*m).vo_w+MCE1VORHSF] = 
	    rhsf_fst[fst_rhsf_entry*FSTRHSFLN+FSTRHSFFUNR];
	}
      else
	{
	  /* no rhsf */
	  (*m).vols[i*(*m).vo_w+MCE1VORHSF] = -1;
	}
    }


  free(rhsf_fst);
  free(geo_to_rhsf);
#undef FSTRHSFTYPE
#undef FSTRHSFFUNR
#undef FSTRHSFLN


  free(bnd_fst);
  free(geo_to_bnd);
#undef FSTBDTYPE 
#undef FSTBDFUNR 
#undef FSTBDLN


  /**********************************************************************
   *  now correct node numbers in faces to edge numbers                 *
   **********************************************************************/

  /*collect all edges of boundary elements in an array of list anchors*/	
  TRY_MALLOC(vx2eg, (*m).vx_nr , struct ilist, 
	     mesh_read_file_gmsh_e1);

   /* initialise all the lists as empty */
  for (i=0; i<(*m).vx_nr; i++)
    {
      vx2eg[i].next=NULL;
      vx2eg[i].data=-1;
    }

  /* fill in all existing edges */
  for (i=0; i<(*m).eg_nr; i++)
    {
      for (j=0; j<2; j++)
	{
	  FIDX nod;

	  nod = (*m).edge[i*(*m).eg_w + MCE1EGNOD1+j];

	  /* append an entry to the end of the nod1 list */
	  pntr=vx2eg[nod].next;
	  hlp =&vx2eg[nod];
	  while (pntr!=NULL)
	    {
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  TRY_MALLOC(hlp->next, 1, struct ilist,  
		 mesh_read_file_gmsh_e1);
	  pntr       = hlp->next;
	  pntr->data = i;
	  pntr->next = NULL;
	} /* for j, nodes of edge i */
    } /* for i, fill existin edges in vx2eg */

  /* now for all faces, modify node ids to edge ids */
  for (i=0; i<(*m).fc_nr; i++)
    {
      FIDX nodes[3], nod1, nod2;
      int found;
      for (j=0; j<3; j++)
	{
	  nodes[j] = (*m).face[i*(*m).fc_w+MCE1FCEDG1+j];
	}

      for (j=0; j<3; j++)
	{
	  FIDX eg;
	  nod1= nodes[j];
	  nod2= nodes[(j+1)%3];

	  /* check if an edge eg with nod1 and nod2 exists */
	  found=0;
	  pntr=vx2eg[nod1].next;
	  hlp =&vx2eg[nod1];
	  while ((found==0)&&(pntr!=NULL))
	    {
	      eg=pntr->data;
	      if ((nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ])||
		  (nod2==(*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]))
		{
		  found=1;
		}
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  
	   /* if not, create one */
	  if (found==0)
	    {
	      /* check if enough space is available */
	      if ((*m).eg_max<(*m).eg_nr+1) 
		{ 
		  err=mesh_more_edges( m, 3);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,
					   mesh_read_file_gmsh_e1);
		}
	      /* create the edge */
	      eg=(*m).eg_nr;
	      
	      
	      (*m).eg_nr++;

	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1  ]=nod1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGNOD1+1]=nod2;
	      (*m).edge[eg*(*m).eg_w+MCE1EGCHL1  ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGGEO   ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCE1EGLVL   ]= -1;

	      /* register the new edge in vx2eg */
	      /* append an entry to the end of the nod1 list */
	      TRY_MALLOC(hlp->next, 1, struct ilist,  
			 mesh_read_file_gmsh_e1);
	      pntr       = hlp->next;
	      pntr->data = eg;
	      pntr->next = NULL;

	       /* append an entry to the end of the nod2 list */
	      pntr=&vx2eg[nod2];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			mesh_read_file_gmsh_e1);
	      pntr       = pntr->next;
	      pntr->data = eg;
	      pntr->next = NULL;
	      
	    }

	  /* now we have the edge eg which holds both nodes, make it
	     edge of the face corresponding to the i-th boundary element */
	  (*m).face[i*(*m).fc_w+MCE1FCEDG1+j]=eg;

	} /* for j, edges of the node-face */
    } /* for i, faces, modify node ids to edge ids */
  
  /* free memory that is no longer required */
  for (i=0; i<(*m).vx_nr; i++)
    {
      ilist_free(&vx2eg[i].next);
    }
  free(vx2eg);
  free(fst_name);

  err=mesh_remove_unused_vertices_e1( m);
  FUNCTION_FAILURE_HANDLE(err, mesh_remove_unused_vertices_e1, 
			  mesh_read_file_f1m_e1);

  /* create an empty hierarchy part */
  (*m).hi_max=10;
  (*m).hi_nr =0;
  TRY_MALLOC( (*m).hier, (*m).hi_max*(*m).hi_w, FIDX, 
	      mesh_read_file_gmsh_e1);

  /* create empty st */
  (*m).st_max=-1;
  (*m).st_nr =-1;
  (*m).st    = NULL;

  err=mesh_read_compute_conns_e1(m);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_compute_conns_e1,
			  mesh_read_file_gmsh_e1);

  err=mesh_init_DD_elem( m );
  FUNCTION_FAILURE_HANDLE(err, mesh_init_DD_elem, mesh_read_file_gmsh_e1);

  return SUCCESS;
}

/*internalfunction*/
int mesh_gmsh_project_nodes_write_file_e1( struct mesh *m, 
					   struct ilist *head, 
					   FILE *qryfile,
					   FIDX *out_nr_nodes,
					   FIDX *node_edge,
					   int *vx_is_done,
					   double *char_length2
/* helper routine for mesh_gmsh_project_nodes_e1 below, writes points
   from the list to file and computes characteristic edge lengths for
   the edges which created them during bisection

   output: qryfile  - the file were the nodes are written to
           out_nr_nodes
                    - number of nodes written
	   node_edge- for each node written the associated edge 
           char_length2
                    - characteristic length, length of the edge 
		      associated with each node from the list
	   vx_is_done
                    - marker if a node is done already 

   input:  m        - the mesh,
           head     - list of new nodes that have a designated  
	              geometry number, three list elements per node:
	              first the node number, second the geometry tag/id, 
		      third an edge ID (for computation of characteristic 
		      length)

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
					   ){
  FIDX vx_w, eg_w, i;
  int dim;
  int err;

  struct ilist *pntr;

  FIDX nr_nodes=0;


  vx_w  = (*m).vx_w;
  eg_w  = (*m).eg_w;
  dim   = (*m).dim;

  pntr=head;

  while( pntr->next!=NULL )
    {
      FIDX node=pntr->data;
      pntr = pntr->next->next->next;
      /* only count if not done yet */
      if (vx_is_done[node]==0)
	{
	  vx_is_done[node]=1;
	  nr_nodes++;
	}	  
    }
  /* write the number of nodes to the file */
  fprintf(qryfile, "%"dFIDX"\n",  nr_nodes);
  /* write the node positions */
  pntr=head;
  while( pntr->next!=NULL )
    {
      FIDX node=pntr->data;
      FIDX geo= pntr->next->data;
      FIDX eg=  pntr->next->next->data;
      FIDX nd1, nd2;
      double egLen2=0.0;
      double vx1,vx2,vx3;

      /* only write if not done yet */
      if (vx_is_done[node]==1)
	{
	  vx_is_done[node]=2;

	  node_edge[node]=eg;

	  nd1=(*m).edge[eg*eg_w+MCE1EGNOD1  ];
	  nd2=(*m).edge[eg*eg_w+MCE1EGNOD1+1];
	  
	  if ( (*m).type==31 )
	    {
	      vx1=(*m).vertex[node*vx_w+MCE1VXSTRT  ];
	      vx2=(*m).vertex[node*vx_w+MCE1VXSTRT+1];
	      vx3=(*m).vertex[node*vx_w+MCE1VXSTRT+2];
	    }
	  else
	    {
	      if ( (*m).vertex[node*vx_w+MCEXVXTYPE]==0.0 )
		{
		  vx1=(*m).vertex[node*vx_w+MCE1VXSTRT  ];
		  vx2=(*m).vertex[node*vx_w+MCE1VXSTRT+1];
		  vx3=(*m).vertex[node*vx_w+MCE1VXSTRT+2];
		}
	      else
		{
		  vx1=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT  ]
			   +(*m).vertex[nd2*vx_w+MCE1VXSTRT  ])
		    +(*m).vertex[node*vx_w+MCE1VXSTRT  ];
		  vx2=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT+1]
			   +(*m).vertex[nd2*vx_w+MCE1VXSTRT+1])
		    +(*m).vertex[node*vx_w+MCE1VXSTRT+1];
		  vx3=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT+2]
			   +(*m).vertex[nd2*vx_w+MCE1VXSTRT+2])
		    +(*m).vertex[node*vx_w+MCE1VXSTRT+2];
		}

	    }

	  err=fprintf(qryfile, "%"dFIDX"  %"dFIDX"  "
		      "%25.16e  %25.16e  %25.16e \n",
		      node, geo, vx1, vx2, vx3 );
	  if (err<0)
	    {
	      fprintf(stderr,
		      "mesh_gmsh_project_nodes_write_file_e1: error "
		      "writing %"dFIDX"-th node to qryfile\n",
		      i);
	      return FAIL;
	    }

	  for (i=0; i<dim; i++)
	    {
	      egLen2 +=
		((*m).vertex[nd1*vx_w+MCE1VXSTRT+i]-
		 (*m).vertex[nd2*vx_w+MCE1VXSTRT+i])
		* ((*m).vertex[nd1*vx_w+MCE1VXSTRT+i]-
		   (*m).vertex[nd2*vx_w+MCE1VXSTRT+i]);
	    }
	  char_length2[node]=egLen2;
	}
      pntr = pntr->next->next->next;
    }

  *out_nr_nodes=nr_nodes;

  return SUCCESS;
}


/*internalfunction*/
int mesh_gmsh_project_nodes_read_file_e1( struct mesh *m, 
					  FILE *resfile,
					  FIDX nr_nodes,
					  FIDX *node_edge,
					  char *edge_or_face,
					  double *char_length2
/* helper routine for mesh_gmsh_project_nodes_e1 below, reads points
   from file, compares the distance they moved to the characteristic lengths
   and updates the node if the distance is not to large

   In/Out: m        - the mesh, is modified accordingly, 
                      only the coordinates of the vertices
		      in the file are modified
   input : resfile  - the file were the nodes are read from
           nr_nodes - number of nodes that should be read
	   node_edge- for each node to be read the associated edge 
           edge_or_face
	            - string used in error messages
           char_length2
                    - characteristic length, length of the edge 
		      associated with each node from the list

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
					   ){
  FIDX vx_w, vx_nr, eg_w, i;

  FIDX res_nr_nodes=0;

  vx_nr = (*m).vx_nr;
  vx_w  = (*m).vx_w;
  eg_w  = (*m).eg_w;


  /* read the number of edge nodes in the resfile */
  if ((fscanf(resfile, "%"dFIDX"\n",  &res_nr_nodes)!=1)
      || (res_nr_nodes!=nr_nodes))
    {
      fprintf(stderr,
	      "mesh_gmsh_project_nodes_read_file_e1: error, "
	      "expected %"dFIDX" %s nodes, but in resfile are %"dFIDX"\n",
	      nr_nodes, edge_or_face, res_nr_nodes);
      return FAIL;
    }
  /* read the new node positions */
  for (i=0; i<nr_nodes; i++)
    {
      FIDX node, eg, nd1, nd2;
      double new_vx1, new_vx2, new_vx3;
      double vx1, vx2, vx3;
      double mv1, mv2, mv3;
      double move_norm2;

      if (fscanf(resfile, "%"dFIDX"   %lg  %lg  %lg",
		 &node, &new_vx1, &new_vx2, &new_vx3)!=4)
	{
	  fprintf(stderr,
		  "mesh_gmsh_project_nodes_read_file_e1: error "
		  "reading %"dFIDX"-th %s-node in resfile\n",
		  i, edge_or_face);
	  return FAIL;
	}
      if ( (node<0)||(node>vx_nr) )
	{
	  fprintf(stderr,
		  "mesh_gmsh_project_nodes_read_file_e1: error, "
		  "%"dFIDX"-th %s-node in resfile out of range, "
		  "node=%"dFIDX"\n",
		  i, edge_or_face, node);
	  return FAIL;
	}
	
      eg=node_edge[node];
      nd1=(*m).edge[eg*eg_w+MCE1EGNOD1  ];
      nd2=(*m).edge[eg*eg_w+MCE1EGNOD1+1];

      /* check if this new position is plausible */
      /* old position */
      if ( (*m).type==31 )
	{
	  vx1=(*m).vertex[node*vx_w+MCE1VXSTRT  ];
	  vx2=(*m).vertex[node*vx_w+MCE1VXSTRT+1];
	  vx3=(*m).vertex[node*vx_w+MCE1VXSTRT+2];
	}
      else
	{
	  if ( (*m).vertex[node*vx_w+MCEXVXTYPE]==0.0 )
	    {
	      vx1=(*m).vertex[node*vx_w+MCE1VXSTRT  ];
	      vx2=(*m).vertex[node*vx_w+MCE1VXSTRT+1];
	      vx3=(*m).vertex[node*vx_w+MCE1VXSTRT+2];
	    }
	  else
	    {
	      vx1=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT  ]
		       +(*m).vertex[nd2*vx_w+MCE1VXSTRT  ])
		+(*m).vertex[node*vx_w+MCE1VXSTRT  ];
	      vx2=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT+1]
		       +(*m).vertex[nd2*vx_w+MCE1VXSTRT+1])
		+(*m).vertex[node*vx_w+MCE1VXSTRT+1];
	      vx3=0.5*((*m).vertex[nd1*vx_w+MCE1VXSTRT+2]
		       +(*m).vertex[nd2*vx_w+MCE1VXSTRT+2])
		+(*m).vertex[node*vx_w+MCE1VXSTRT+2];
	    }
	}

      mv1=new_vx1-vx1;
      mv2=new_vx2-vx2;
      mv3=new_vx3-vx3;
      move_norm2 = mv1*mv1 + mv2*mv2 + mv3*mv3;

      if (move_norm2< 0.5*char_length2[node])
	{
	  /* new position appears reasonable, move the point */
	  if ((*m).type==31)
	    {
	      (*m).vertex[node*vx_w+MCE1VXSTRT  ] = new_vx1;
	      (*m).vertex[node*vx_w+MCE1VXSTRT+1] = new_vx2;
	      (*m).vertex[node*vx_w+MCE1VXSTRT+2] = new_vx3;
	    }
	  else 
	    {
	      if ( (*m).vertex[node*vx_w+MCEXVXTYPE]==0.0)
		{
		  (*m).vertex[node*vx_w+MCE1VXSTRT  ] = new_vx1;
		  (*m).vertex[node*vx_w+MCE1VXSTRT+1] = new_vx2;
		  (*m).vertex[node*vx_w+MCE1VXSTRT+2] = new_vx3;
		}
	      else
		{
		  (*m).vertex[node*vx_w+MCE1VXSTRT  ] = mv1;
		  (*m).vertex[node*vx_w+MCE1VXSTRT+1] = mv2;
		  (*m).vertex[node*vx_w+MCE1VXSTRT+2] = mv3;
		}
	    }
	}
      else
	{
	  /* new position strange, ignore new position */
	  fprintf(stderr,"mesh_gmsh_project_nodes_read_file_e1: "
		  "warning, %s-node "
		  "%"dFIDX" moved to far by gmsh (%f>c*%f), "
		  "keep old position\n",
		  edge_or_face, node,
		  sqrt(move_norm2), sqrt(char_length2[node]));
	  return FAIL;
	}
    }
  return SUCCESS;
}

/*FUNCTION*/
int mesh_gmsh_project_nodes_e1( struct mesh *m, 
				struct ilist *edgeVL, 
				struct ilist *faceVL
/* calls gmsh_closest_points to move all nodes in edgeVL and faceVL to
   their geometry

   In/Out: m        - the mesh, is modified accordingly, 
                      only the coordinates of the vertices
		      in edgeVL and faceVL are modified

   input:  edgeVL   - list of new nodes that have a designated EDGE 
	              geometry number, three list elements per node:
	              first the node number, second the geometry tag/id, 
		      third an edge ID (for computation of characteristic 
		      length)
	   faceVL   - list of new nodes that have a designated FACE
	              geometry number, three list elements per node:
	              first the node number, second the geometry tag/id,
		      third an edge ID (for computation of characteristic 
		      length)

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  FIDX i, vx_nr, nr_eg_nodes, nr_fc_nodes;
  int err;
  int *vx_is_done; /* marker is a node is done already */
  double *char_length2;
  FIDX *node_edge;

  char *basename, *qryname, *resname, *commandString, *strpos;

  FILE *qryfile, *resfile;
  int fdqry = -1;

  /* #warning "removed geometry projection for tests"
     return SUCCESS; */

  vx_nr = (*m).vx_nr;

  TRY_MALLOC( vx_is_done, vx_nr, int, 
	      mesh_gmsh_project_nodes_e1 );
  TRY_MALLOC( node_edge, vx_nr, FIDX, 
	      mesh_gmsh_project_nodes_e1 );
  for (i=0; i<vx_nr; i++)
    {
      vx_is_done[i]=0;
      node_edge[i]=-1;
    }
  TRY_MALLOC( char_length2, vx_nr, double, 
	      mesh_gmsh_project_nodes_e1 );
  for (i=0; i<vx_nr; i++)
    {
      char_length2[i]=0.0;
    }


  /* prepare the names */
  TRY_MALLOC( basename, strlen((*m).meshgenDATA)+1, char, 
	      mesh_gmsh_project_nodes_e1 );
  strcpy(basename, (*m).meshgenDATA);
  strpos=strstr(basename, ".msh");
  if (strpos==NULL)
    {
      fprintf(stderr,
	      "mesh_gmsh_project_nodes_e1: basename has no .msh %s ?\n",
	      basename);
      return FAIL;
    }
  /* cut the basename string by setting the char at the position where
     ".msh" starts to '\0', terminating the string at this position */
  *strpos='\0';
  
  /* prepare the string with the ".qry" filename */
  /* debug version: 
     TRY_MALLOC( qryname, strlen(basename)+5, char,
     mesh_gmsh_project_nodes_e1 );
     strcpy( qryname, basename );
     strcat( qryname, ".qry" );
     qryfile=fopen(qryname, "w");
     if (qryfile==NULL)
     {
     fprintf(stderr,
     "mesh_gmsh_project_nodes_e1: error opening file %s\n",
     qryname);
     return FAIL;
     }
  */
  /* production version: */
#include <unistd.h>
#include <errno.h>
  TRY_MALLOC( qryname, strlen("/tmp/FEINS_gmsh_tmp_XXXXXX")+5, char,
	      mesh_gmsh_project_nodes_e1 );
  strcpy(qryname, "/tmp/FEINS_gmsh_tmp_XXXXXX");
  if ((fdqry = mkstemp(qryname)) == -1 ||
      (qryfile = fdopen(fdqry, "w+")) == NULL)
    {
      if (fdqry != -1) 
	{
	  unlink(qryname);
	  close(fdqry);
	}
      fprintf(stderr,
	      "mesh_gmsh_project_nodes_e1: error opening file %s error %s\n",
	      qryname, strerror(errno));
      return FAIL;
    }


  /* write the edge nodes */
  err=mesh_gmsh_project_nodes_write_file_e1( m, edgeVL, qryfile,
					     &nr_eg_nodes, node_edge,
					     vx_is_done, char_length2);
  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_write_file_e1,
			   mesh_gmsh_project_nodes_e1);

  /* write the face nodes */
  err=mesh_gmsh_project_nodes_write_file_e1( m, faceVL, qryfile,
					     &nr_fc_nodes, node_edge,
					     vx_is_done, char_length2);
  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_write_file_e1,
			   mesh_gmsh_project_nodes_e1);

  fclose(qryfile);

  /* call gmsh_closest_points or vtk_stl_closest_point */
  /* prepare the string with the ".res" filename */
  TRY_MALLOC( commandString, 4*(strlen(basename)+5)+300, char,
	      mesh_gmsh_project_nodes_e1 );
  switch ((*m).curved)
    {
    case 1:
      sprintf(commandString, "gmsh_closest_points %s.geo %s.msh %s %s.out >/dev/null",
	      basename, basename, qryname, qryname); /* */
      break;
    case 2:
      sprintf(commandString, "vtk_stl_closest_point %s_orig.stl %s %s.out",
	      basename, qryname, qryname);
      break;
    default:
      fprintf(stderr,"mesh_gmsh_project_nodes_e1: error "
	      "unknown m.curved=%d \n", (*m).curved);
      return FAIL;
    }
  
  /* fprintf(stderr,"DEBUG: commandString=\"%s\"\n", commandString); /* */
  
  err=system(commandString);
  if (err!=0)
    {
      fprintf(stderr,"mesh_gmsh_project_nodes_e1: error "
	      "executing the command \n"
	      "  \"%s\"\n", commandString);
      return FAIL;
    }

  /* now read the resulting file */

  /* prepare the string with the ".res" filename */
  TRY_MALLOC( resname, strlen(qryname)+5, char,
	      mesh_gmsh_project_nodes_e1 );
  strcpy( resname, qryname );
  strcat( resname, ".out" );

  /* open the res file */
  resfile=fopen(resname, "r");
  if (resfile==NULL)
    {
      fprintf(stderr,
	      "mesh_gmsh_project_nodes_e1: error opening file %s\n",
	      resname);
      return FAIL;
    }


  /* read the edge nodes */
  err=mesh_gmsh_project_nodes_read_file_e1( m, resfile, nr_eg_nodes,
					    node_edge,
					    "edge", char_length2);
  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_read_file_e1,
			   mesh_gmsh_project_nodes_e1);

  /* read the face nodes */
  err=mesh_gmsh_project_nodes_read_file_e1( m, resfile, nr_fc_nodes,
					    node_edge,
					    "face", char_length2);
  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_read_file_e1,
			   mesh_gmsh_project_nodes_e1);

  fclose(resfile);

  /* delete the temporary files */
  unlink(qryname);
  unlink(resname);
  

  free(resname);
  free(commandString);
  free(qryname);
  free(basename);

  free(char_length2);
  free(node_edge);
  free(vx_is_done);

  return SUCCESS;
}
 

/*FUNCTION*/
int mesh_switch_type_e1( struct mesh *m, int type
/* modifies the mesh so that all elements are of the polynomial degree
   given by type, all necessarry new parts are created thereby,
   

   In/Out: m       - the mesh, is modified accordingly, e.g.
                     elements, volumes, faces, edges, vertices, are modified,
		     if necessary the fields of the mesh are
		     reallocated to have space for the new stuff

   Input:  type    - type==31 for P1 tetrahedra
                     type==32 for P2 tetrahedra

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j;
  int err;

  if ( ((*m).type==31) && (type==32) )
    {
      /* switch from P1 to P2 */
      struct ilist head_geo_edge_vertex_list, *tail_GEVL;
      struct ilist head_geo_face_vertex_list, *tail_GFVL;

      head_geo_edge_vertex_list.data = -1;
      head_geo_edge_vertex_list.next = NULL;
      tail_GEVL = &head_geo_edge_vertex_list;

      head_geo_face_vertex_list.data = -1;
      head_geo_face_vertex_list.next = NULL;
      tail_GFVL = &head_geo_face_vertex_list;

      /* actually declare the new type */
      (*m).type=32;

      /* add type info to vertices if necessary */
      if ( (*m).vx_w < MCEXVXTYPE+1 )
	{
	  /* not yet, add it, init type to 0.0 for all existing vertices */
	  double *vertex_tmp;
	  FIDX old_vx_w, new_vx_w;

	  vertex_tmp=(*m).vertex;
	  old_vx_w = (*m).vx_w;

	  new_vx_w = MCEXVXTYPE+1;

	  TRY_MALLOC( (*m).vertex, (*m).vx_max*new_vx_w, double,
		      mesh_switch_type_e1);

	  for (i=0; i<(*m).vx_nr; i++)
	    {
	      for (j=0; j<old_vx_w; j++)
		{
		  (*m).vertex[i*new_vx_w+j] = vertex_tmp[i*old_vx_w+j];
		}
	      (*m).vertex[i*new_vx_w+MCEXVXTYPE] = 0.0;
	    }

	  (*m).vx_w =new_vx_w;
	  free(vertex_tmp);
	} /* end add vertex type info if necessary */

      /* add NODM info to edges if necessary */
      if ( (*m).eg_w < MCE2EGNODM+1 )
	{
	  FIDX *edge_tmp;
	  FIDX old_eg_w, new_eg_w;

	  edge_tmp=(*m).edge;
	  old_eg_w = (*m).eg_w;

	  new_eg_w = MCE2EGNODM+1;

	  TRY_MALLOC( (*m).edge, (*m).eg_max*new_eg_w, FIDX,
		      mesh_switch_type_e1);

	  for (i=0; i<(*m).eg_nr; i++)
	    {
	      for (j=0; j<old_eg_w; j++)
		{
		  (*m).edge[i*new_eg_w+j] = edge_tmp[i*old_eg_w+j];
		}
	      /* see if the edge was already refined, then just add
		 the common node of the new edges as midnode of the old */
	      if ( edge_tmp[i*old_eg_w + MCE1EGCHL1] != -1 )
		{
		  /* second node of child is midnode of parent */
		  FIDX child = edge_tmp[i*old_eg_w + MCE1EGCHL1];
		  (*m).edge[i*new_eg_w+MCE2EGNODM] = 
		    edge_tmp[child*old_eg_w + MCE1EGNOD1 +1];
		}
	      else
		{
		  /* make sure there is enough space */
		  if ( (*m).vx_nr+1 > (*m).vx_max )
		    {
		      err=mesh_more_vertices( m, 1);
		      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices,
					       mesh_switch_type_e1);
		    }
		  /* create the new node */
		  for (j=0; j<(*m).vx_w; j++)
		    {
		      (*m).vertex[(*m).vx_nr*(*m).vx_w+j]=0.0;
		    }
		  (*m).vertex[(*m).vx_nr*(*m).vx_w+MCEXVXTYPE]=2.0;
		  (*m).edge[i*new_eg_w+MCE2EGNODM] = (*m).vx_nr;
		  (*m).vx_nr++;
		}
	    }

	  (*m).eg_w =new_eg_w;
	  free(edge_tmp);
	} /* end add NODM info to edges if necessary */

      /* loop over all faces, to see if any of the edges or faces with
	 type 2 nodes have geometry tags */
      for (i=0; i<(*m).fc_nr; i++) 
	for (j=0; j<3; j++)
	  {
	    FIDX eg=(*m).face[i*(*m).fc_w+MCE1FCEDG1+j];
	    FIDX nodm=(*m).edge[eg*(*m).eg_w+MCE2EGNODM];
	    
	    /* if nodm is hierarchical node: */
	    if ( (*m).vertex[nodm*(*m).vx_w+MCEXVXTYPE]==2.0 )
	      {
		/* check if the edge is a geometry edge
		   or the face a geometry face, 
		   the edge has priority over the face */
		if ((*m).edge[eg*(*m).eg_w+MCE1EGGEO]!=-1)
		  {
		    /* is geometry edge, add node to edge list */
		    struct ilist *tmp = tail_GEVL;
		    TRY_MALLOC(tmp->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    /* first the node number, then geo tag/id, then edge */
		    tmp->data=nodm;
		    tmp->next->data=(*m).edge[eg*(*m).eg_w+MCE1EGGEO];
		    tmp->next->next->data= eg;
		    tmp->next->next->next->data= -1;
		    tmp->next->next->next->next= NULL;
		    tail_GEVL = tmp->next->next->next;
		  }
		else if ((*m).face[i*(*m).fc_w+MCE1FCGEO]!=-1)
		  {
		    /* is geometry face but not edge,
		       add node to face list */
		    struct ilist *tmp = tail_GFVL;
		    TRY_MALLOC(tmp->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			       mesh_switch_type_e1);
		    /* first the node number, then geo tag/id, then edge */
		    tmp->data=nodm;
		    tmp->next->data=(*m).face[i*(*m).fc_w+MCE1FCGEO];
		    tmp->next->next->data= eg;
		    tmp->next->next->next->data= -1;
		    tmp->next->next->next->next= NULL;
		    tail_GFVL = tmp->next->next->next;
		  }
	      }
	  } /* end loop over faces to check geometry tags */

      /* check if a mesh generator was used, if so, move the new points
	 onto their geometry */
      if ((tail_GEVL != &head_geo_edge_vertex_list)
	  ||(tail_GFVL != &head_geo_face_vertex_list))
	{
	  /* actually need to project */
	  if (((*m).meshgen==2)&&((*m).meshgenDATA!=NULL))
	    {
	      /* gmsh */
	      err=mesh_gmsh_project_nodes_e1( m, &head_geo_edge_vertex_list,
					      &head_geo_face_vertex_list );
	      FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_e1, 
				       mesh_switch_type_e1 ); 
	    }
	}
      
      ilist_free(&head_geo_edge_vertex_list.next);
      ilist_free(&head_geo_face_vertex_list.next);

      /* make space for the larger elements */
      if ( (*m).el_w<MCE1ELNOD1+10 ) 
	{
	  FIDX new_el_w;
	  free((*m).elem);
	  new_el_w = MCE1ELNOD1+10;
	  TRY_MALLOC( (*m).elem, (*m).el_max*new_el_w, FIDX,
		      mesh_switch_type_e1);
	  (*m).el_w =new_el_w;
	} /* end make space for the larger elements */
      
      /* re-compute elements */
      for (i=0; i<(*m).vo_nr; i++)
	{
	  /* init DD subdomain of the element */
	  (*m).elem[i*(*m).el_w+MCXXELDDPR] = -1;

	  err=mesh_elem_from_vols_e1( m, i);
	  FUNCTION_FAILURE_HANDLE( err, mesh_elem_from_vols_e1,
				   mesh_switch_type_e1);
	} /* end re-compute elements */
      fprintf(stderr,"DEBUG: mesh_switch_type_e1 31 to 32 done\n");
    }
  else if ( ((*m).type==32) && (type==31) )
    {
      /* switch from P2 to P1 */
      fprintf(stderr,"mesh_switch_type_e1: switch form P2 to P1 "
	      "not implemented yet\n");
      return FAIL;
    }
  else
    {
      fprintf(stderr,"mesh_switch_type_e1: unknow type switch "
	      "from m.type=%d to type=%d\n", (*m).type, type);
      return FAIL;
    }

  err=mesh_init_DD_elem( m );
  FUNCTION_FAILURE_HANDLE(err, mesh_init_DD_elem, mesh_switch_type_e1 );

  return SUCCESS;
}



/*FUNCTION*/
int mesh_refine_uniform_e1( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m       - the mesh, is modified accordingly, e.g. new
                     elements, volumes, faces, edges, vertices, boundary
                     elements and hierarchy entries are created,
		     the element el and the volume el are thereby
		     replaced by one of their children, the faces and edges
		     remain to allow other elements to stay intact,
		     if necessary the fields of the mesh are
		     reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX old_el_nr, i;
  int err;

  struct ilist head_geo_edge_vertex_list, *tail_GEVL;
  struct ilist head_geo_face_vertex_list, *tail_GFVL;

  head_geo_edge_vertex_list.data = -1;
  head_geo_edge_vertex_list.next = NULL;
  tail_GEVL = &head_geo_edge_vertex_list;

  head_geo_face_vertex_list.data = -1;
  head_geo_face_vertex_list.next = NULL;
  tail_GFVL = &head_geo_face_vertex_list;
 
  old_el_nr=(*m).el_nr;
  (*m).lvl++;

  if ( (*m).type==32 )
    {
      if ((*m).elhier==NULL ) 
	{
	  (*m).eh_w   = MCE2EHLN;
	  (*m).eh_max = (*m).el_nr;
	  (*m).eh_nr  = 0;
	  TRY_MALLOC( (*m).elhier,  (*m).eh_w*(*m).eh_max, FIDX, 
		      mesh_refine_uniform_e1);
	}
      if ( (*m).eh_max < (*m).eh_nr + (*m).el_nr )
	{
	  err=mesh_more_elhiers( m,  (*m).el_nr );
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, 
				   mesh_refine_uniform_e1 ); 
	}
    }


  /* loop over all elements */
  for (i=0; i< old_el_nr; i++)
    {
      err=mesh_refine_element_e1( m , i, &tail_GEVL, &tail_GFVL);
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_element_e1, 
			       mesh_refine_uniform_e1 ); 
    }

  /* check if a mesh generator was used, if so, move the new points
     onto their geometry */
  if ((tail_GEVL != &head_geo_edge_vertex_list)
      ||(tail_GFVL != &head_geo_face_vertex_list))
    {
      /* actually need to project */
      if (((*m).meshgen==2)&&((*m).meshgenDATA!=NULL))
	{
	  /* gmsh */
	  err=mesh_gmsh_project_nodes_e1( m, &head_geo_edge_vertex_list,
					  &head_geo_face_vertex_list );
	  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_e1, 
				   mesh_refine_uniform_e1 ); 
	}
    }

  ilist_free(&head_geo_edge_vertex_list.next);
  ilist_free(&head_geo_face_vertex_list.next);

  err=mesh_init_DD_elem( m );
  FUNCTION_FAILURE_HANDLE(err, mesh_init_DD_elem, mesh_refine_uniform_e1 );
 
  return SUCCESS;
}


/*FUNCTION*/
int mesh_refine_element_e1( struct mesh *m , FIDX el,
			    struct ilist **tail_GEVL,
			    struct ilist **tail_GFVL
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, volumes, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the faces and
		      edges remain to allow other elements to stay
		      intact, if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, vo_w, fc_w, el_w, pv_w, level, oct_diag_edge;
  FIDX old_DD_subdomain;
  int err;


  FIDX oldnod[4], nodnewface[4*3], nodopsnewedge[4*3],
    newinnerface[4], newoutercenterface[4], newvols[8];
  /* nodnewface[lnode*3+(this nodes face)]=the new face (global number) 
     nodopsnewedge[lnode*4+(this nodes face)]=the new edge oposing this
                                           node (global number) */
  FIDX oldel[10], newelindices[8];


  /* the refinement level of the new face */
  level=(*m).lvl;

  vo_w=(*m).vo_w;
  fc_w=(*m).fc_w;
  el_w=(*m).el_w;
  pv_w=(*m).pv_w;


  /* clear nodnewface */
  for (i=0; i<4; i++)
    for (j=0; j<3; j++)
      nodnewface[i*3+j]=-1;

  /* the DD_subdomain to which this element belongs */
  old_DD_subdomain=(*m).elem[el*el_w+MCXXELDDPR];

  /* get oldnod */
  for (i=0; i<4; i++)
    oldnod[i]=(*m).elem[el*el_w+MCE1ELNOD1+i];


  if ( (*m).type==32 )
    {
      for (i=0; i<10; i++)
	oldel[i]=(*m).elem[el*el_w+MCE1ELNOD1+i];
    }


  /* make sure the 4 faces are refined, store the new edges, midnodes, get
     nodedge */ 
  for (i=0; i<4; i++)
    {
      FIDX fc;

      fc=(*m).vols[el*vo_w+MCE1VOFAC1+i];

      err=mesh_split_face_e1( m, fc, oldnod, nodnewface, nodopsnewedge,
			      &newoutercenterface[i], 
			      tail_GEVL, tail_GFVL );
      FUNCTION_FAILURE_HANDLE( err, mesh_split_face_e1,
			       mesh_refine_element_e1); 
    }

  /* make sure we have enough space for the new diagonal edge in the inner
     of the element, 8=4+4 new faces, 7 new
     volumes and elements (the old are recycled as a 8th), 
     7 new pcvol if needed (the old is recycled as a 4th),
     1 new elhier (if m.type==32)  */ 
  if ((*m).eg_nr+1>(*m).eg_max)
    {
      err=mesh_more_edges( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_refine_element_e1 );
    }
  if ((*m).fc_nr+8>(*m).fc_max)
    {
      err=mesh_more_faces( m, 8);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_refine_element_e1 );
    }
  if ((*m).vo_nr+7>(*m).vo_max)
    {
      err=mesh_more_volumes( m, 7);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_volumes, mesh_refine_element_e1 );
    }
  if ((*m).el_nr+7>(*m).el_max)
    {
      err=mesh_more_elems( m, 7);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elems, mesh_refine_element_e1 );
    }

  if (((*m).vols[el*vo_w+MCE1VOPCVL]!=-1)&&((*m).pv_nr+7>(*m).pv_max))
    {
      err=mesh_more_pcvol( m, 7);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcvol, mesh_refine_element_e1 );
    }
  if ( (*m).type==32 )
    {
      if ( (*m).eh_nr+1>(*m).eh_max )
	{
	  err=mesh_more_elhiers( m,  (*m).el_nr );
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, 
				   mesh_refine_element_e1 ); 
	}
    }


  /* for all old corner nodes */
  for (i=0; i<4; i++)
    {
      FIDX newel, theinnerface;

      /* create new element/volume, containing this old node */
      newel=(*m).vo_nr;
      (*m).vo_nr++;
      
      /* first the three faces created by splitting the old faces of
	 the old volume */
      for (j=0; j<3; j++)
	{
	  (*m).vols[newel*vo_w+MCE1VOFAC1+j]=nodnewface[i*3+j];
	}
      /* create a new inner face, from the three new edges oposing
	 this node */
      theinnerface=(*m).fc_nr;
      (*m).fc_nr++;
      for (j=0; j<3; j++)
	{
	  (*m).face[theinnerface*fc_w+MCE1FCEDG1+j]=nodopsnewedge[i*3+j];
	}
      /* init remaining info for new face */
      (*m).face[theinnerface*fc_w+MCE1FCCHL1]= -1;
      (*m).face[theinnerface*fc_w+MCE1FCBND] = -1;
      (*m).face[theinnerface*fc_w+MCE1FCGEO] = -1;
      (*m).face[theinnerface*fc_w+MCE1FCLVL] =level;
      (*m).face[theinnerface*fc_w+MCE1FCPCSU]= -1;

      /* make this the fourth and last face of the new volume, store
	 it as newinnerface[i] */
      (*m).vols[newel*vo_w+MCE1VOFAC1+3]=theinnerface;
      newinnerface[i]=theinnerface;

      newvols[i]=newel;
    }

  /* create the new volumes in the inner part */
  err=mesh_refine_element_e1_innerels( m, el,
				       newinnerface, newoutercenterface, 
				       &newvols[4], &oct_diag_edge);
  FUNCTION_FAILURE_HANDLE( err, mesh_refine_element_e1_innerels,
			   mesh_refine_element_e1); 
  
  /* copy remaining information from old to new volume */
  for (i=0; i<7; i++)
    {
      (*m).vols[newvols[i]*vo_w+MCE1VORHSF]=(*m).vols[el*vo_w+MCE1VORHSF];
      (*m).vols[newvols[i]*vo_w+MCE1VOFRED]=-1;

      /* if necessary create a new pcvol entry */
      if ((*m).vols[el*vo_w+MCE1VOPCVL]!=-1)
	{
	  FIDX pcv, new;

	  pcv=(*m).vols[el*vo_w+MCE1VOPCVL];
	  new=(*m).pv_nr;
	  
	  (*m).pcvol[new*pv_w+MCXXPVVOLM]=newvols[i];
	  (*m).vols[newvols[i]*vo_w+MCE1VOPCVL]=new;
	  (*m).pcvol[new*pv_w+MCXXPVCRIT]=(*m).pcvol[pcv*pv_w+MCXXPVCRIT];

	  (*m).pv_nr++;
	}
      else
	{
	  (*m).vols[newvols[i]*vo_w+MCE1VOPCVL]=-1;
	}
    } /* for i, copy info from old volume to all new ones */

  /* create the new elements */
  for (i=0; i<7; i++)
    {
      FIDX newel=(*m).el_nr;
      (*m).el_nr++;
      
      newelindices[i]=newel;

      err=mesh_elem_from_vols_e1( m, newel);
      FUNCTION_FAILURE_HANDLE( err, mesh_elem_from_vols_e1,
			       mesh_refine_element_e1);

      /* set the DD_subdomain */
      (*m).elem[newel*el_w+MCXXELDDPR]=old_DD_subdomain;
    }

  /* overwrite old element with last new one */
  err=mesh_elem_from_vols_e1( m, el);
  FUNCTION_FAILURE_HANDLE( err, mesh_elem_from_vols_e1,
			   mesh_refine_element_e1); 
  /* for el DD_subdomain does not need to be set, as it is unchanged */
  newelindices[7]=el;

  /* create the new elhier entry */
  if ( (*m).type==32 )
    {
      /* define possible rotations, rot123 means the triangle 123 is
	 rotated to 231, leaving the opposite node 0 where it is,
	 stored is rot123[i]=j, the old index j that becomes the new
	 index i */
      FIDX tmp[10];
      /*  orig      0  1 2 3  4 5 6  7 8 9 */
      int rot012[]={1, 2,0,3, 5,6,4, 8,9,7};
      int rot021[]={2, 0,1,3, 6,4,5, 9,7,8};

      int rot123[]={0, 2,3,1, 6,9,7, 4,5,8};
      int rot132[]={0, 3,1,2, 7,8,4, 6,9,5};

      int rot013[]={1, 3,2,0, 8,9,5, 4,7,6};
      int rot031[]={3, 0,2,1, 7,6,9, 8,4,5};

      int rot023[]={2, 1,3,0, 5,8,9, 6,4,7};
      int rot032[]={3, 1,0,2, 8,4,7, 9,5,6};

      /* flip nodes 2 and 3 */
      int flip23[]={0, 1,3,2, 4,8,7, 6,5,9};

      int reference[]={0,4,6,7,
		       1,5,4,8,
		       2,6,5,9,
		       3,7,9,8};
      FIDX new_outer_els[4*10];

      int outernode;
      FIDX neweh;
      //FILE *fdbg; char fdbgname[]="/tmp/feins_elhier_debug.txt";
      
      /* rotate the old element to make sure the oct_diag_edge is 
	 4-9 */
      FIDX global4a=(*m).edge[oct_diag_edge*(*m).eg_w+MCE1EGNOD1  ];
      FIDX global4b=(*m).edge[oct_diag_edge*(*m).eg_w+MCE1EGNOD1+1];
      int new4=-1;
      for (j=4; j<=6; j++)
	{
	  if ( (oldel[j]==global4a) || (oldel[j]==global4b) )
	    new4=j;
	}
#ifdef DEBUGFEINS
      if (new4==-1)
	{
	  fprintf(stderr,"mesh_refine_element_e1: "
		  "elhier -> oct_diag_edge not found\n");
	  return FAIL;
	}
#endif
      // fdbg=fopen(fdbgname,"w");
      // if (fdbg==NULL) {fprintf(stderr,"DEBUG: fopen failed\n"); return FAIL;}
      // fprintf(fdbg,"DEBUG elhier el%03d ", el);
      // fprintf(fdbg,"DBG new4 %d 4a %3d 4b %3d   ", new4, global4a, global4b);
      switch (new4)
	{
	case 4: /* do nothing */
	  break;
	case 5: /* apply rot012 */
	  mesh_elem_reorder(oldel, rot012, tmp, 10);
	  break;
	case 6: /* apply rot021 */
	  mesh_elem_reorder(oldel, rot021, tmp, 10);
	  break;
	}
      /* now the oldel has node 4 as one endpoint of the diagonal edge
	 of the octahedron */
      /* now turn the 4 new elements at the corners of the original
	 tetrahedron into their reference configuration */
      for (outernode=0; outernode<4; outernode++)
	{
	  int found;
	  int is_new_el;
	  int is_node_in_new_el;
	  FIDX lookfor;

	  /* find the old corner node in the new elements */
	  found=0;
	  for (i=0; (i<4)&&(found==0); i++)
	    for (j=0; (j<4)&&(found==0); j++)
	      {
		if ( (*m).elem[newelindices[i]*el_w+MCE1ELNOD1+j]==oldel[outernode])
		  {
		    found=1;
		    is_new_el=i;
		    is_node_in_new_el=j;
		  }
	      }
#ifdef DEBUGFEINS
	  if (found==0)
	    {
	      fprintf(stderr,"mesh_refine_element_e1: "
		      "elhier -> corner node not found in newels\n");
	      return FAIL;
	    }
#endif
	  /* copy the new element, then rotate it such that the corner
	     node is the first in the element (see reference
	     configurations) */
	  for (i=0; i<10; i++)
	    {
	      new_outer_els[outernode*10+i]=
		(*m).elem[newelindices[is_new_el]*el_w+MCE1ELNOD1+i];
	    }
	  // fprintf(fdbg,"DBG 0is %d ", is_node_in_new_el);
	  switch (is_node_in_new_el)
	    {
	    case 0: /* do nothing */
	      break;
	    case 1: /* apply rot012 */
	      mesh_elem_reorder(&new_outer_els[outernode*10], rot012, tmp, 10);
	      break;
	    case 2: /* apply rot021 */
	      mesh_elem_reorder(&new_outer_els[outernode*10], rot021, tmp, 10);
	      break;
	    case 3: /* apply rot031 */
	      mesh_elem_reorder(&new_outer_els[outernode*10], rot031, tmp, 10);
	      break;
	    }
	  /* now the first node (0) is the first node (0) of the
	     reference, next make the second (1) fit */
	  lookfor=oldel[reference[outernode*4+1]];
	  found=0;
	  for (i=1; (i<4)&&(found==0); i++)
	    if ( new_outer_els[outernode*10+i]==lookfor )
	      {
		found=1;
		is_node_in_new_el=i;
	      }
#ifdef DEBUGFEINS
	  if (found==0)
	    {
	      fprintf(stderr,"mesh_refine_element_e1: "
		      "elhier -> second node of reference not found\n");
	      return FAIL;
	    }
#endif
	  // fprintf(fdbg,"DBG 1is %d ", is_node_in_new_el);
	  switch (is_node_in_new_el)
	    {
	    case 1: /* do nothing */
	      break;
	    case 2: /* apply rot123 */
	      mesh_elem_reorder(&new_outer_els[outernode*10], rot123, tmp, 10);
	      break;
	    case 3: /* apply rot132 */
	      mesh_elem_reorder(&new_outer_els[outernode*10], rot132, tmp, 10);
	      break;
	    }
	  /* first (0) and second (1) nodes fit, now make the third
	     (2) and fourth (3) fit, this can only require a flip
	     between (2) and (3)
	   */
	  lookfor=oldel[reference[outernode*4+2]];
	  if (new_outer_els[outernode*10+2]!=lookfor)
	    {
	      mesh_elem_reorder(&new_outer_els[outernode*10], flip23, tmp, 10);
	      // fprintf(fdbg,"DBG flip23 ");
	    }
#ifdef DEBUGFEINS
	  if (new_outer_els[outernode*10+2]!=lookfor)
	    {
	      fprintf(stderr,"mesh_refine_element_e1: "
		      "elhier -> third node of reference not found\n");
	      return FAIL;
	    }
	  if (new_outer_els[outernode*10+3]!=oldel[reference[outernode*4+3]])
	    {
	      fprintf(stderr,"mesh_refine_element_e1: "
		      "elhier -> fourth node of reference not found\n");
	      return FAIL;
	    }
#endif
	} /* for outernode */
      /* now all four new_outer_els are in reference configuration, 
	 generate the elhier entry */
      neweh=(*m).eh_nr;
      (*m).eh_nr++;
      // fprintf(fdbg,"\nDBG father ");
      for (i=0; i<10; i++)
	{
	  (*m).elhier[neweh*(*m).eh_w+MCE2EHFAT1+i]=oldel[i];
	  // fprintf(fdbg," %3d ", (*m).elhier[neweh*(*m).eh_w+MCE2EHFAT1+i]);
	}
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1   ]=new_outer_els[0*10+4];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 1]=new_outer_els[0*10+5];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 2]=new_outer_els[0*10+6];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 3]=new_outer_els[1*10+4];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 4]=new_outer_els[1*10+5];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 5]=new_outer_els[1*10+6];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 6]=new_outer_els[2*10+4];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 7]=new_outer_els[2*10+5];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 8]=new_outer_els[2*10+6];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 9]=new_outer_els[0*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+10]=new_outer_els[0*10+8];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+11]=new_outer_els[0*10+9];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+12]=new_outer_els[1*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+13]=new_outer_els[1*10+8];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+14]=new_outer_els[1*10+9];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+15]=new_outer_els[2*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+16]=new_outer_els[2*10+8];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+17]=new_outer_els[2*10+9];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+18]=new_outer_els[3*10+8];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+19]=new_outer_els[3*10+9];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+20]=new_outer_els[3*10+5];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+21]=new_outer_els[3*10+4];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+22]=new_outer_els[3*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+23]=new_outer_els[3*10+6];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+24]=
	(*m).edge[oct_diag_edge*(*m).eg_w+MCE2EGNODM];

      (*m).elhier[neweh*(*m).eh_w+MCE2EHLVL]=(*m).lvl;

      // fprintf(fdbg,"DBG child ");
      // for (i=0; i<25; i++)
	// {
	//   fprintf(fdbg," %3d ", (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+i]);
	// }
      // fprintf(fdbg,"DBG end\n");
      // fclose(fdbg);

    } /* if m.type==32 create elhier */

  /* that's it */

  return SUCCESS;
}

/*FUNCTION*/
void mesh_elem_reorder( FIDX *elem, int *order, FIDX *tmp, int nr
/* reorders the entries of elem according to order

   Input:  order    - array of nr indices (permutation), 
                      the new order of the old entries of elem
	   nr       - size of elem, order, and tmp

   In/Out: elem     - array to be reordered,
                      elem[j]=elem[order[j]]
           tmp      - help array
*/
			){
  int i;
  for (i=0; i<nr; i++)
    {
      tmp[i]=elem[i];
    }
  for (i=0; i<nr; i++)
    {
      elem[i]=tmp[order[i]];
    }
}


/*FUNCTION*/
int mesh_refine_element_e1_innerels( struct mesh *m, FIDX oldvol,
				     FIDX *faces1, FIDX *faces2, 
				     FIDX *newvols, FIDX *oct_diag_edge
/* the faces1 and faces2 together describe an octahedron, 
   for this the shortest diagonal is found, and introduced as new 
   edge, four new faces (triangles) are created with this diagonal
   edge, then three new volumes and the volume vo is overwritten,
   the mesh is modified accordingly, all necessarry new parts are
   created

   Input:  faces1   - array of 4 face indices, four arbitrary faces of
                      the octahedron
	   faces2   - array of 4 face indices, the remaining faces of
                      the octahedron
           oldvol   - original volume, which is refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      volumes, faces and edges are created,
		      of the mesh are reallocated to have space for
		      the new stuff 

   Output: newvols  - array of 3 volume indices, the ones which are
                      created new, the fourth new volume overwrites
                      the original volume vo
           oct_diag_edge
                    - index of the diagonal edge used in the above


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, k, r, nnodes, vo_w, fc_w, eg_w, vx_w, level;
  FIDX newedge, newface, newvol;
  FIDX faces[8];          /* combines faces1 ad faces2 */
  int  faceUsed[8];       /* if faceUsed[j]==0, then faces[j] has not
			     yet been used in the definition of a new
			     volume */
  FIDX nodes[6];          /* the 6 nodes inolved */
  FIDX node2edgeMat[6*6]; /* eg=node2edgeMat[i*6+j] means
			     edge eg connects nodes[j] and nodes[j],
			     eg=-1 if no connection yet (diagona nodes) */
  FIDX newfaces[4];       /* the new faces */
  FIDX locNewVols[4*4];   /* local representation of the four volumes
			     created in the octahedron
			     locNewVols[i*4+j]=fc, means fc is the
			     j-th face of new volume i */
  double length2_shortest;

  int shortest_i, shortest_j, n_newfaces, n_newvols, first_face;

  int dim, err;

  level = (*m).lvl;
  dim   = (*m).dim;

  vo_w  = (*m).vo_w;
  fc_w  = (*m).fc_w;
  eg_w  = (*m).eg_w;
  vx_w  = (*m).vx_w;

  /*** init ***/
  /* faces */
  for (i=0; i<4; i++)
    {
      faces[i  ]=faces1[i];
      faces[i+4]=faces2[i];
    }

  /* faceUsed */
  for (i=0; i<8; i++)
    {
      faceUsed[i]=0;
    }

  /* nodes and node2edgeMat */
  for (i=0; i<6*6; i++)
    node2edgeMat[i]=-1;

  nnodes=0;
  for (r=0; r<8; r++)
    { 
      FIDX fc=faces[r];
      for (i=0; i<3; i++)
	{ 
	  FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+i];
	  int  egLocNodes[2];
	  for (j=0; j<2; j++)
	    {
	      int notyetfound=1;
	      int check_nod=0;
	      FIDX node=(*m).edge[eg*eg_w+MCE1EGNOD1+j];
	      while ((check_nod<nnodes)&&notyetfound)
		{
		  if (nodes[check_nod]==node) 
		    {
		      notyetfound=0;
		    }
		  else
		    {
		      check_nod++;
		    }
		}
	      if (notyetfound)
		{
		  /* this is a new node */
#ifdef DEBUGFEINS
		  if (nnodes>=6)
		    {
		      fprintf(stderr,"mesh_refine_element_e1_innerels: error, "
			      "more than 6 nodes in octahedron!!\n");
		      return FAIL;
		    }
#endif
		  nodes[nnodes]=node;
		  nnodes++;
		}
	      egLocNodes[j]=check_nod;
	    } /* for j, node in edge */
	  /* mark edge in node2edgeMat */
	  node2edgeMat[egLocNodes[0]*6+egLocNodes[1]]=eg;
	  node2edgeMat[egLocNodes[1]*6+egLocNodes[0]]=eg;
	} /* for i, edge in face */
    } /* for r, faces */
  /* done, six nodes should have been found */
#ifdef DEBUGFEINS
  if (nnodes!=6)
    {
      fprintf(stderr,"mesh_refine_element_e1_innerels: error, "
	      "less than 6 nodes in octahedron!!\n");
      return FAIL;
    }
#endif 


  /* find shortest diagonal edge */
  length2_shortest=-1.0;
  shortest_i=-1;
  shortest_j=-1;

  for (i=0; i<5; i++)
    for (j=i+1; j<6; j++)
      {
	if (node2edgeMat[i*6+j]==-1)
	  {
	    /* this is a diagonal, compute its lenght */
	    double diag_length2=0.0;
	    for (k=0; k<dim; k++) 
	      {
		double diff_k;
		diff_k = 
		  (*m).vertex[nodes[i]*vx_w+MCE1VXSTRT+k]
		  - (*m).vertex[nodes[j]*vx_w+MCE1VXSTRT+k];

		diag_length2 += diff_k*diff_k;
	      }
	    if ((diag_length2<length2_shortest)||(length2_shortest<0))
	      {
		/* this is shorter than previous shortest edge */
		length2_shortest = diag_length2;
		shortest_i=i;
		shortest_j=j;
	      }
	  } /* if diagonal edge */
      } /* for i,j */

  /* make sure there is enough free memory for 1 new edge, 4 new fces,
     3 new volumes */
  if ((*m).eg_nr+1>(*m).eg_max)
    {
      err=mesh_more_edges( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_refine_element_e1_innerels );
    }
  if ((*m).fc_nr+4>(*m).fc_max)
    {
      err=mesh_more_faces( m, 4);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_refine_element_e1_innerels );
    }
  if ((*m).vo_nr+3>(*m).vo_max)
    {
      err=mesh_more_volumes( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_volumes, mesh_refine_element_e1_innerels );
    }

  /* create new edge */
  newedge=(*m).eg_nr;
  (*m).eg_nr++;

  *oct_diag_edge = newedge;

  (*m).edge[newedge*eg_w+MCE1EGNOD1  ]=nodes[shortest_i];
  (*m).edge[newedge*eg_w+MCE1EGNOD1+1]=nodes[shortest_j];
  (*m).edge[newedge*eg_w+MCE1EGCHL1  ]= -1;
  (*m).edge[newedge*eg_w+MCE1EGGEO   ]= -1;
  (*m).edge[newedge*eg_w+MCE1EGLVL   ]=level;

  /* if m.type==32, create new midnode */
  if ( (*m).type==32 )
    {
      FIDX newvx;

      if ((*m).vx_nr+1>(*m).vx_max)
	{
	  err=mesh_more_vertices( m, 1);
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices,
				   mesh_refine_element_e1_innerels );
	}
      newvx=(*m).vx_nr;
      (*m).vx_nr++;
      for (j=0; j<dim; j++)
	{
	  (*m).vertex[newvx*(*m).vx_w+MCE1VXSTRT+j]=0.0;
	}
      (*m).vertex[newvx*(*m).vx_w+MCEXVXTYPE]=2.0;
      /* the new inner edge is never a geo-edge */
      
      (*m).edge[newedge*eg_w+MCE2EGNODM]= newvx;
    }

  /* create new faces with this newedge */
  /* there are four nodes (connecting nodes) which are neither
     shortest_i nor shortest_j, each shares one edge with shortest_i
     and one with shortest_j, these two and the new edge make a new
     face */
  n_newfaces=0;
  for (i=0; i<6; i++)
    {
      if ((i!=shortest_i)&&(i!=shortest_j))
	{
	  /* this must be a connecting node */
	  FIDX eg_ic, eg_cj;

	  /* the two edges connecting connecting shortest_i via
	     connecting node to shortest_j */
	  eg_ic=node2edgeMat[shortest_i*6+i];
	  eg_cj=node2edgeMat[shortest_j*6+i];

	  /* create new face with these three edges */
	  newface=(*m).fc_nr;
	  (*m).fc_nr++;
	  
	  (*m).face[newface*fc_w+MCE1FCEDG1  ]=eg_ic;
	  (*m).face[newface*fc_w+MCE1FCEDG1+1]=eg_cj;
	  (*m).face[newface*fc_w+MCE1FCEDG1+2]=newedge;

	  (*m).face[newface*fc_w+MCE1FCBND]= -1;
	  (*m).face[newface*fc_w+MCE1FCGEO]= -1;

	  (*m).face[newface*fc_w+MCE1FCPCSU]= -1;
	  (*m).face[newface*fc_w+MCE1FCCHL1]= -1;
	  (*m).face[newface*fc_w+MCE1FCLVL]=level;
	
	  newfaces[n_newfaces]=newface;
	  n_newfaces++;
	} /* if connecting node */
    } /* for i, local nodes */
  
  /* now define the four new volumes (local definition) */
  n_newvols=0;
  /* choose one newface first (first_face), look for faces that share
     edges with first_face and a second newface */
  first_face=0;
  while (n_newvols<4)
    {
      /* find the two faces that share edges with newface[first_face]
	 and each other */
      FIDX eg1=(*m).face[newfaces[first_face]*fc_w+MCE1FCEDG1  ];
      FIDX eg2=(*m).face[newfaces[first_face]*fc_w+MCE1FCEDG1+1];
      int found_one_while=0;

      /* loop over outer faces */
      for (i=0; (i<8)&&(n_newvols<4); i++) 
	{
	  if (faceUsed[i]==0)
	    {
	      /* check if this face contains eg1 */
	      int sharesEdge1=-1;
	      for (j=0; j<3; j++)
		{
		  if ((*m).face[faces[i]*fc_w+MCE1FCEDG1+j]==eg1)
		    sharesEdge1=j;
		}
	      if (sharesEdge1!=-1)
		{
		  /* look for second face, that contains eg2
		     and shares an edge with faces[i] */
		  for (k=0; k<8; k++) 
		    {
		      if (faceUsed[k]==0)
			{
			  /* check if this face contains eg2 */
			  int sharesEdge2=-1;
			  for (j=0; j<3; j++)
			    {
			      if ((*m).face[faces[k]*fc_w+MCE1FCEDG1+j]==eg2)
				sharesEdge2=j;
			    }
			  if (sharesEdge2!=-1)
			    {
			      /* now see if faces[i] and faces[k] have
				 a common edge */
			      int commonEdge=0;
			      for (j=0; j<3; j++)
				{
				  FIDX eg3=(*m).face[faces[k]*fc_w+MCE1FCEDG1+j];
				  for (r=0; r<3; r++)
				    {
				      if ((*m).face[faces[i]*fc_w+MCE1FCEDG1+r]==eg3)
					{
					  commonEdge=1;
					}
				    } /* for r */
				} /* for j */
			      if (commonEdge==1)
				{
				  /* look for second newface, with a
				     common edge to faces[i] */
				  int second_face=-1;
				  int ts;
				  for (ts=first_face+1; 
				       (ts<4)&&(second_face==-1);
				       ts++)
				    for (j=0; j<3; j++)
				      {
					FIDX eg3=(*m).face[newfaces[ts]*fc_w
							   +MCE1FCEDG1+j];
					for (r=0; r<3; r++)
					  {
					    if ((*m).face[faces[i]*fc_w
							  +MCE1FCEDG1+r]==eg3)
					    {
					      second_face=ts;
					    }
					  } /* for r */
				      } /* for j */
				  if (second_face!=-1)
				    {
				      /* we now have four faces making
					 up a new volume */
				      locNewVols[n_newvols*4  ]
					=newfaces[first_face];
				      locNewVols[n_newvols*4+1]
					=newfaces[second_face];
				      locNewVols[n_newvols*4+2]
					=faces[i];
				      locNewVols[n_newvols*4+3]
					=faces[k];

				      /* mark outer faces as used */
				      faceUsed[i]=1;
				      faceUsed[k]=1;
				      
				      found_one_while++;

				      n_newvols++;
				    } /* if second face found */
				} /* if commonEdge */
			    } /* if sharesEdge2 */
			} /* if faceUsed[k] not yet */
		    } /* for k, second loop outer faces */
		} /* if sharesEdge1 */
	    } /* if faceUsed[i] not yet */
	} /* for i, loop outer faces */

      first_face++;
    } /* while (n_newvols<4) */

  /* now actually create the first three new volumes */
  for (i=0; i<3; i++)
    {
      newvol=(*m).vo_nr;
      (*m).vo_nr++;
      
      /* copy the four faces */
      for (j=0; j<4; j++)
	{
	  (*m).vols[newvol*vo_w+MCE1VOFAC1+j]=locNewVols[i*4+j];
	  /* remaining information regarding new volume is taken care
	     of by calling routine */
	}

      newvols[i]=newvol;
    } /* for i, loop 3 new vols */

  /* for the fourth, overwrite the old volume,
     copy the four faces */
  for (j=0; j<4; j++)
    {
      (*m).vols[oldvol*vo_w+MCE1VOFAC1+j]=locNewVols[3*4+j];
      /* remaining information regarding new volume is taken care
	 of by calling routine */
    }


  return SUCCESS;
}


/*FUNCTION*/
int mesh_elem_from_vols_e1( struct mesh *m , FIDX el
/* for volume el the element is redefined by finding the nodes that
   belong to the volume

   Input:  el       - the element to be redefined

   In/Out: m        - the mesh, is modified accordingly, only the elem
                      part corresponding to element el is overwritten

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, r, el_w, vo_w, fc_w, eg_w;


  el_w  = (*m).el_w;
  vo_w  = (*m).vo_w;
  fc_w  = (*m).fc_w;
  eg_w  = (*m).eg_w;

  if ( (*m).type==31 )
    {
      int nnodes;
      
      FIDX nodes[4];          /* the 4 nodes inolved */
#ifdef DEBUGFEINS
#define COLLECT_NODES_FROM_N_FACES 4
#define STOP_IF_N_NODES_FOUND 5
#else
#define COLLECT_NODES_FROM_N_FACES 2
#define STOP_IF_N_NODES_FOUND 4
#endif

      nnodes=0;
      for (r=0; r<COLLECT_NODES_FROM_N_FACES; r++)
	{ 
	  FIDX fc=(*m).vols[el*vo_w+MCE1VOFAC1+r];


	  for (i=0; (i<3)&&(nnodes<STOP_IF_N_NODES_FOUND); i++)
	    { 
#undef COLLECT_NODES_FROM_N_FACES
#undef STOP_IF_N_NODES_FOUND

	      FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+i];

	      for (j=0; j<2; j++)
		{
		  int notyetfound=1;
		  int check_nod=0;
		  FIDX node=(*m).edge[eg*eg_w+MCE1EGNOD1+j];
		  while ((check_nod<nnodes)&&notyetfound)
		    {
		      if (nodes[check_nod]==node) 
			{
			  notyetfound=0;
			}
		      else
			{
			  check_nod++;
			}
		    }
		  if (notyetfound)
		    {
		      /* this is a new node */
#ifdef DEBUGFEINS
		      if (nnodes>=4)
			{
			  fprintf(stderr,"mesh_elem_from_vols_e1: error, "
				  "more than 4 nodes in volume!!\n");
			  return FAIL;
			}
#endif
		      nodes[nnodes]=node;
		      nnodes++;
		    }
		} /* for j, node in edge */
	    } /* for i, edge in face */
	} /* for r, faces */
      /* done, four nodes should have been found */
#ifdef DEBUGFEINS
      if (nnodes!=4)
	{
	  fprintf(stderr,"mesh_elem_from_vols_e1: error, "
		  "less than 4 nodes in volume!!\n");
	  return FAIL;
	}
#endif 

      for (i=0; i<4; i++)
	{
	  (*m).elem[el*el_w+MCE1ELNOD1+i]=nodes[i];
	}
    } /* if type==31 */
  else if ( (*m).type==32 )
    {
      int nnodes, nedges;
      FIDX nodes[10];          /* the 10 nodes inolved */
      FIDX edges[6];           /* the 6 edges inolved */
      int edge2vertex[6*2];    /* edge2vertex[i*2+j]=k
				  means node[k] is in edges[i] as j-th node
			       */
      

      /* find 6 edges first */
      nedges=0;
      /* always need exactly 3 faces to find all 6 edges */
      for (r=0; r<3; r++) 
	{ 
	  FIDX fc=(*m).vols[el*vo_w+MCE1VOFAC1+r];

	  for (i=0; i<3; i++)
	    { 
	      FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+i];
	      int notfound=1;
	      int checkedge=0;
	      while (notfound&&(checkedge<nedges))
		{
		  if (edges[checkedge]==eg)
		    {
		      notfound=0;
		    }
		  else
		    {
		      checkedge++;
		    }
		}
	      if (notfound)
		{
		  /* new edge, ad to our list */
#ifdef DEBUGFEINS
		  if (nedges>=6)
		    {
		      fprintf(stderr,"mesh_elem_from_vols_e1: error, "
			      "more than 6 edges in volume!!\n");
		      return FAIL;
		    }
#endif
		  edges[nedges]=eg;
		  nedges++;
		}
	    } /* for edge i of face r */
	} /* for face r of vol, looking for 6 edges */
#ifdef DEBUGFEINS
      if (nedges<6)
	{
	  fprintf(stderr,"mesh_elem_from_vols_e1: error, "
		  "less than 6 edges in volume!!\n");
	  return FAIL;
	}
#endif
      /* now find nodes 0..4, build edge2vertex */
      nnodes=0;
      for (i=0; i<nedges; i++)
	{
	  FIDX eg=edges[i];
	  int k;
	  for (j=0; j<2; j++)
	    {
	      FIDX nd=(*m).edge[eg*eg_w+MCE1EGNOD1+j];
	      int found=-1;
	      
	      /* check if this node is known yet */
	      for (k=0; (k<nnodes)&&(found==-1); k++)
		{
		  if (nodes[k]==nd)
		    {
		      found=k;
		    }
		}
	      if (found==-1)
		{
#ifdef DEBUGFEINS
		  if (nnodes>=4)
		    {
		      fprintf(stderr,"mesh_elem_from_vols_e1: error, "
			      "more than 4 vertices in type==32 element\n");
		      return FAIL;
		    }
#endif
		  nodes[nnodes]=nd;
		  found=nnodes;
		  nnodes++;
		}
		
	      edge2vertex[i*2+j] = found;
	    } /* end for j node of edge i */
	} /* end for edge i */
      /* first three edges must be from first face, they defined nodes 0,1,2,
	 the first edge definitely connects 0-1 thus it its midnode is 4 */
      nodes[nnodes]=(*m).edge[edges[0]*eg_w+MCE2EGNODM]; nnodes++;
      /* the second edge may be 1-2 or 0-2 (the else below)
	 thus the midnodes of edges 1,2 will be nodes 5,6 or 6,5 */
      if ( (edge2vertex[1*2+0]==1)||(edge2vertex[1*2+1]==1) )
	{
	  nodes[nnodes]=(*m).edge[edges[1]*eg_w+MCE2EGNODM]; nnodes++;
	  nodes[nnodes]=(*m).edge[edges[2]*eg_w+MCE2EGNODM]; nnodes++;
	}
      else
	{
	  nodes[nnodes]=(*m).edge[edges[2]*eg_w+MCE2EGNODM]; nnodes++;
	  nodes[nnodes]=(*m).edge[edges[1]*eg_w+MCE2EGNODM]; nnodes++;
	}
      /* the other 3 edges each connect node 3 to one of 0,1,2,
	 according to which one of these, make their midnodes 7,8,9 */
      for (i=3; i<6; i++)
	{
	  int other_node;
	  if (edge2vertex[i*2+0]==3)
	    {
	      other_node=edge2vertex[i*2+1];
	    }
	  else
	    {
	      other_node=edge2vertex[i*2+0];
	    }

#ifdef DEBUGFEINS
	  if (other_node>=3)
	    {
	       fprintf(stderr,"mesh_elem_from_vols_e1: error, "
		  "other_node wrong in type==32 element\n");
	       return FAIL;
	    }
#endif	    
	  nodes[7+other_node]=(*m).edge[edges[i]*eg_w+MCE2EGNODM]; nnodes++;
	}
#ifdef DEBUGFEINS
      if (nnodes!=10)
	{
	  fprintf(stderr,"mesh_elem_from_vols_e1: error, "
		  "did not find 10 nodes in type==32 element\n");
	  return FAIL;
	}
      if (el_w<MCE1ELNOD1+10)
	{
	  fprintf(stderr,"mesh_elem_from_vols_e1: error, "
		  "el_w<%d for mesh type==32\n",(int) MCE1ELNOD1+10);
	  return FAIL;
	}
#endif

      for (i=0; i<10; i++)
	{
	  (*m).elem[el*el_w+MCE1ELNOD1+i]=nodes[i];
	}
    } /* if type==32 */
  else
    {
      fprintf(stderr,"mesh_elem_from_vols_e1: error, "
	      "unknown mesh type\n");
      return FAIL;
    }

  return SUCCESS;
}




/*FUNCTION*/
int mesh_split_face_e1( struct mesh *m , FIDX fc,
			FIDX *voldnod, 
			FIDX *nodnewface, FIDX *nodopsnewedge,
			FIDX *newcenterface,
			struct ilist **tail_GEVL,
			struct ilist **tail_GFVL
/* refines the face fc of the mesh m, if it is not refined already,
   all necessarry new parts are created

   Input:  fc       - the face to be refined
           voldnod  - array of 4 node indices, to which nodnewface and
                      nodopsnewedge correspond

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      old faces and edges remain to allow other
		      elements to stay intact, if necessary the fields
		      of the mesh are reallocated to have space for
		      the new stuff 
           nodnewface 
                    - array (4*3), modified such that
                      nodnewface[node*3+k]=newface
                      if newface has vertex voldnod[node], k is the
                      first still available of (0,1,2)
           nodopsnewedge 
                    - array (4*3), modified such that
                      nodopsnewedge[node*3+k]=newedge
                      if newedge is oposing vertex voldnod[node] in a
                      new face, k is the
                      first still available of (0,1,2)
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)

   Output: newcenterface
                    - the central face of the four new faces created
                      by splitting old face fc		      


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, fc_w, eg_w, bd_w, ps_w, level;
  FIDX newface;
  int dim;

  int err;


  FIDX newedge[3], foldnod[3], midnodes[3], nodedge[3][2], nodineg[3][2];
  /* nodedge[lnode][this nodes edge]=the edge (global number),
     nodineg[lnode][this nodes edge]=the node is 1st/2nd of the edge */

  fc_w=(*m).fc_w;
  eg_w=(*m).eg_w;
  bd_w=(*m).bd_w;
  ps_w=(*m).ps_w;
  dim=(*m).dim;

  /* the refinement level of the new edges and faces */
  level=(*m).lvl;


  /* clear nodedge */
  for (i=0; i<3; i++)
    for (j=0; j<2; j++)
      nodedge[i][j]=-1;

  /***************
   * get foldnod *
   ***************/
  { 
    int nfoldnod=0;
    for (i=0; i<3; i++)
      { 
	FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+i];
	for (j=0; j<2; j++)
	  {
	    int notyetfound=1;
	    int check_nod=0;
	    FIDX node=(*m).edge[eg*eg_w+MCE1EGNOD1+j];
	    while ((check_nod<nfoldnod)&&notyetfound)
	      {
		if (foldnod[check_nod]==node) notyetfound=0;
		check_nod++;
	      }
	    if (notyetfound)
	      {
		/* this is a new foldnod */
#ifdef DEBUGFEINS
		if (nfoldnod>=3)
		  {
		    fprintf(stderr,"mesh_split_face_e1: error, "
			    "more than 3 old nodes in face!!\n");
		    return FAIL;
		  }
#endif
		foldnod[nfoldnod]=node;
		nfoldnod++;
	      }
	  } /* for j, node in edge */
      } /* for i, edge in face */
    /* done, three nodes should have been found */
#ifdef DEBUGFEINS
    if (nfoldnod!=3)
      {
	fprintf(stderr,"mesh_split_face_e1: error, "
		"less than 3 old nodes in face!!\n");
	return FAIL;
      }
#endif
  } /* end block get foldnod */

  /* check if this face is already refined */
  if ((*m).face[fc*fc_w+MCE1FCCHL1]!=-1)
    {
      /* just return the connectivitiy pattern based on the rules:
	 - newface[i]= belongs to foldnod[i], has edges containing
	   foldnod[i] as two first edges,
	 - newface[3] is newcenterface */
      for (i=0; i<3; i++)
	{
	  int oldvnode=-1;
	  int k=0;
	  for (j=0; j<4; j++)
	    {
	      if (voldnod[j]==foldnod[i]) oldvnode=j;
	    }
	  if (oldvnode==-1)
	    {
	      fprintf(stderr,"mesh_split_face_e1: error, "
		      "already-ref-fc, foldnod not found in voldnod!!\n");
	      return FAIL;
	    }
	  
	  while ((nodnewface[oldvnode*3+k]!=-1)&&(k<3))
	    {
	      k++;
	    }
	  if (k>=3)
	    {
	      fprintf(stderr,"mesh_split_face_e1: error, "
		      "already-ref-fc, more than 3 times new face for one node??\n");
	      return FAIL;	    
	    }

	  /* now set the data regarding the old node and adjacent
	     newface and oposite new edge */
	  newface=(*m).face[fc*fc_w+MCE1FCCHL1+i];
	  nodnewface[oldvnode*3+k]=newface;
	  nodopsnewedge[oldvnode*3+k]=(*m).face[newface*fc_w+MCE1FCEDG1+2];
	} /* for i, foldnods */

      *newcenterface=(*m).face[fc*fc_w+MCE1FCCHL1+3];
      
      return SUCCESS;
    }



  /* make sure the 3 edges are refined, store the midnodes, get
     nodedge */ 
  for (i=0; i<3; i++)
    {
      FIDX eg, chld;

      eg=(*m).face[fc*fc_w+MCE1FCEDG1+i];

      /* make sure this edge is refined */
      if ((*m).edge[eg*eg_w+MCE1EGCHL1]==-1)
	{
	  err=mesh_split_edge_e1( m, eg, tail_GEVL);
	  FUNCTION_FAILURE_HANDLE( err, mesh_split_edge_e1,
				   mesh_split_face_e1); 
	}
      
      /* store the midnode */
      chld=(*m).edge[eg*eg_w+MCE1EGCHL1];
      /* second node of child edge is always the midnode of the old
	 edge */ 
      midnodes[i]=(*m).edge[chld*eg_w+MCE1EGNOD1+1]; 


      /* check if we are on a geometry FACE,
	 if so, add the new midnode to tail_GFVL */
      if ( (*m).face[fc*fc_w+MCE1FCGEO] != -1)
	{
	  /* check if the new node is on a geometry edge, then the
	     tail_GEVL should have changed */
          /* each face node may appear twice in tail_GFVL, we take
	     care of this in the projection routine, such that each
	     node will only be adjusted once */
	  if ((*m).edge[eg*eg_w+MCE1EGGEO]==-1)
	    {
	      struct ilist *tmp;
	      if ( (*m).type==31 )
		{
		  tmp= *tail_GFVL;
		  TRY_MALLOC(tmp->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  /* first the node number, then geo tag/id, then edge */
		  tmp->data=midnodes[i];
		  tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
		  tmp->next->next->data= eg;
		  tmp->next->next->next->data= -1;
		  tmp->next->next->next->next= NULL;
		  *tail_GFVL = tmp->next->next->next;
		}
	      else if ( (*m).type==32 )
		{
		  FIDX chld1=(*m).edge[eg*eg_w+MCE1EGCHL1  ];
		  FIDX chld2=(*m).edge[eg*eg_w+MCE1EGCHL1+1];
		  FIDX midn1=(*m).edge[chld1*eg_w+MCE2EGNODM];
		  FIDX midn2=(*m).edge[chld2*eg_w+MCE2EGNODM];

		  tmp= *tail_GFVL;
		  TRY_MALLOC(tmp->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  /* first the node number, then geo tag/id, then edge */
		  tmp->data=midn1;
		  tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
		  tmp->next->next->data= chld1;
		  tmp->next->next->next->data= -1;
		  tmp->next->next->next->next= NULL;
		  *tail_GFVL = tmp->next->next->next;

		  tmp= *tail_GFVL;
		  TRY_MALLOC(tmp->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			     mesh_split_face_e1);
		  /* first the node number, then geo tag/id, then edge */
		  tmp->data=midn2;
		  tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
		  tmp->next->next->data= chld2;
		  tmp->next->next->next->data= -1;
		  tmp->next->next->next->next= NULL;
		  *tail_GFVL = tmp->next->next->next;
		}
	    }
	}

      for (j=0; j<2; j++)
	{
	  FIDX amnodeedge, thenode, intnode;
	  thenode=(*m).edge[eg*eg_w+MCE1EGNOD1+j];
	  intnode=0;
	  /* find the apropriate internal node for this guy */
	  while ((intnode<3)&&(foldnod[intnode]!=thenode))
	    intnode++;
	  /* if not found, cry */
	  if (intnode>=3)
	    {
	      fprintf(stderr, "internal mesh consistency error in mesh_split_face_e1, file %s, line %d\n", __FILE__, __LINE__);
	      return FAIL;
	    }

	  /* find where in nodedge and nodineg this one goes to */
	  if (nodedge[intnode][0]==-1)
	    { amnodeedge=0; }
	  else
	    { amnodeedge=1; }
	  nodedge[intnode][amnodeedge]=eg;
	  nodineg[intnode][amnodeedge]=j;
	}
    }

  /* make sure we have enough space for the 3 new edges in the inner
     of the face, 4 new faces, 3 new boundary and 3 new pcsurf if needed
     (the old is recycled as a 4th) */ 
  if ((*m).eg_nr+3>(*m).eg_max)
    {
      err=mesh_more_edges( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_split_face_e1 );
    }
  if ( (*m).type==32 )
    if ((*m).vx_nr+3>(*m).vx_max)
      {
        err=mesh_more_vertices( m, 3);
        FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices, mesh_split_face_e1 );
      }
  
  if ((*m).fc_nr+4>(*m).fc_max)
    {
      err=mesh_more_faces( m, 4);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_split_face_e1 );
    }

  if (((*m).face[fc*fc_w+MCE1FCBND ]!=-1)&&((*m).bd_nr+3>(*m).bd_max))
    {
      err=mesh_more_boundary( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_boundary, mesh_split_face_e1 );
    }

  if (((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)&&((*m).ps_nr+3>(*m).ps_max))
    {
      err=mesh_more_pcsurf( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcsurf, mesh_split_face_e1 );
    }


  /* for all old nodes */
  for (i=0; i<3; i++)
    {
      FIDX chld0, chld1, midnod0, midnod1;

      /* get the child edges containig this node */
      chld0=(*m).edge[nodedge[i][0]*eg_w+MCE1EGCHL1+nodineg[i][0]];
      chld1=(*m).edge[nodedge[i][1]*eg_w+MCE1EGCHL1+nodineg[i][1]];
  
      /* get the corresponding midnodes */
      midnod0=(*m).edge[chld0*eg_w+MCE1EGNOD1+1]; 
      midnod1=(*m).edge[chld1*eg_w+MCE1EGNOD1+1];
      /* second node of child edge is allways the midnode of the old
	 edge */

      /* create a new edge from the midnodes */
      newedge[i]=(*m).eg_nr;
      (*m).eg_nr++;

      (*m).edge[newedge[i]*eg_w+MCE1EGNOD1  ]=midnod0;
      (*m).edge[newedge[i]*eg_w+MCE1EGNOD1+1]=midnod1;
      (*m).edge[newedge[i]*eg_w+MCE1EGCHL1  ]= -1;
      (*m).edge[newedge[i]*eg_w+MCE1EGGEO   ]= -1;
      (*m).edge[newedge[i]*eg_w+MCE1EGLVL   ]=level;

      /* if m.type==32, create new midnode */
      if ( (*m).type==32 )
	{
	  FIDX newvx;

	  newvx=(*m).vx_nr;
	  (*m).vx_nr++;
	  for (j=0; j<dim; j++)
	    {
	      (*m).vertex[newvx*(*m).vx_w+MCE1VXSTRT+j]=0.0;
	    }
	  (*m).vertex[newvx*(*m).vx_w+MCEXVXTYPE]=2.0;
	  /* the new inner edge is never a geo-edge, 
	     but we may be on a geo-face */
	  if ( (*m).face[fc*fc_w+MCE1FCGEO] != -1)
	    {
	      struct ilist *tmp = *tail_GFVL;
	      TRY_MALLOC(tmp->next, 1, struct ilist,  
			 mesh_split_face_e1);
	      TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			 mesh_split_face_e1);
	      TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			 mesh_split_face_e1);
	      /* first the node number, then geo tag/id, then edge */
	      tmp->data=newvx;
	      tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
	      tmp->next->next->data= newedge[i];
	      tmp->next->next->next->data= -1;
	      tmp->next->next->next->next= NULL;
	      *tail_GFVL = tmp->next->next->next;
	    }
      
	  (*m).edge[newedge[i]*eg_w+MCE2EGNODM]= newvx;
	}
      

      /* create a new face */
      newface=(*m).fc_nr;
      (*m).fc_nr++;

      (*m).face[fc*fc_w+MCE1FCCHL1+i]=newface;

      (*m).face[newface*fc_w+MCE1FCEDG1  ]=chld0;
      (*m).face[newface*fc_w+MCE1FCEDG1+1]=chld1;
      (*m).face[newface*fc_w+MCE1FCEDG1+2]=newedge[i];

      (*m).face[newface*fc_w+MCE1FCCHL1] = -1;
      (*m).face[newface*fc_w+MCE1FCGEO ] = (*m).face[fc*fc_w+MCE1FCGEO];

      (*m).face[newface*fc_w+MCE1FCLVL ]=level;

      /* if necessary create a new boundary entry */
      if ((*m).face[fc*fc_w+MCE1FCBND ]!=-1)
	{
	  FIDX new, old;

	  old=(*m).face[fc*fc_w+MCE1FCBND ];
	  new=(*m).bd_nr;

	  (*m).face[newface*fc_w+MCE1FCBND]=new;

	  (*m).bound[new*bd_w+MCE1BDFACE]=newface;
	  (*m).bound[new*bd_w+MCE1BDTYPE]=(*m).bound[old*bd_w+MCE1BDTYPE];
	  (*m).bound[new*bd_w+MCE1BDFNCT]=(*m).bound[old*bd_w+MCE1BDFNCT];
	  (*m).bound[new*bd_w+MCE1BDORIE]=(*m).bound[old*bd_w+MCE1BDORIE];
	  (*m).bound[new*bd_w+MCE1BDSSEG]=(*m).bound[old*bd_w+MCE1BDSSEG];

	  (*m).bd_nr++;
	}
      else
	{
	  (*m).face[newface*fc_w+MCE1FCBND]=-1;
	}

      /* if necessary create a new pcsurf entry */
      if ((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)
	{
	  FIDX pcs, new;

	  pcs=(*m).face[fc*fc_w+MCE1FCPCSU];
	  new=(*m).ps_nr;
	  
	  (*m).pcsurf[new*ps_w+MCXXPSSURF]=newface;
	  (*m).face[newface*fc_w+MCE1FCPCSU]=new;
	  (*m).pcsurf[new*ps_w+MCXXPSCRIT]=(*m).pcsurf[pcs*ps_w+MCXXPSCRIT];

	  (*m).ps_nr++;
	}
      else
	{
	  (*m).face[newface*fc_w+MCE1FCPCSU]=-1;
	}

      /* store the connectivity info for the calling routine */
      {
	int oldvnode=-1;
	int k=0;
	for (j=0; j<4; j++)
	  {
	    if (voldnod[j]==foldnod[i]) oldvnode=j;
	  }
	if (oldvnode==-1)
	  {
	    fprintf(stderr,"mesh_split_face_e1: error, "
		    "foldnod not found in voldnod!!\n");
	    return FAIL;
      	  }
	
	while ((nodnewface[oldvnode*3+k]!=-1)&&(k<3))
	  {
	    k++;
	  }
	if (k>=3)
	  {
	    fprintf(stderr,"mesh_split_face_e1: error, "
		    "more than 3 times new face for one node??\n");
	    return FAIL;	    
	  }

	/* now set the data regarding the old node and adjacent
	   newface and oposite new edge */
	nodnewface[oldvnode*3+k]=newface;
	nodopsnewedge[oldvnode*3+k]=newedge[i];
      }
      /* next node */
    }

  /* the outer 3 faces are correctly created now,
     now create the inner one, the 4-th */
  newface=(*m).fc_nr;
  (*m).fc_nr++;

  (*m).face[fc*fc_w+MCE1FCCHL1+i]=newface;

  for (i=0; i<3; i++)
    {
      /* the face */
      (*m).face[newface*fc_w+MCE1FCEDG1+i]=newedge[i];
    }

  (*m).face[newface*fc_w+MCE1FCCHL1]=-1;
  (*m).face[newface*fc_w+MCE1FCGEO ] = (*m).face[fc*fc_w+MCE1FCGEO];
  (*m).face[newface*fc_w+MCE1FCLVL]=level;

  /* if necessary overwrite the boundary entry */
  if ((*m).face[fc*fc_w+MCE1FCBND ]!=-1)
    {
      FIDX old;
      old=(*m).face[fc*fc_w+MCE1FCBND ];
      (*m).bound[old*bd_w+MCE1BDFACE]=newface;
    }
  /* in any case, set the new BND info */
  (*m).face[newface*fc_w+MCE1FCBND]=(*m).face[fc*fc_w+MCE1FCBND ];

  /* if necessary overwrite the pcsurf entry */
  if ((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)
    {
      FIDX pcs;

      pcs=(*m).face[fc*fc_w+MCE1FCPCSU];
      
      (*m).pcsurf[pcs*ps_w+MCXXPSSURF]=newface;
    }
  /* in any case, set the new PCSU info */
  (*m).face[newface*fc_w+MCE1FCPCSU]=(*m).face[fc*fc_w+MCE1FCPCSU];


  *newcenterface=newface;
  

  /* that's it */

  return SUCCESS;
}

/*FUNCTION*/
int mesh_split_edge_e1( struct mesh *m , FIDX eg,
			struct ilist **tail_GEVL
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring faces
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, dim, vx_w, eg_w, hi_w;
  int  err;
  FIDX newnode, nod1, nod2, newmidnode1, newmidnode2;
  FIDX neweg1, neweg2;
  FIDX newhi, level;

  dim=(*m).dim;
  vx_w=(*m).vx_w;
  eg_w=(*m).eg_w;
  hi_w=(*m).hi_w;


  if ((*m).edge[eg*eg_w+ MCE1EGCHL1 ]!= -1)
    {
      /* nothing to do, return */
      return SUCCESS;
    }
    

  /* first we have to make sure we have space for one or two new
     vertex entries (1 for m.type==31, 2 for m.type==32), two new
     edges, one hierarchy entry
  */
  if ((*m).vx_nr+2>(*m).vx_max)
    {
      err=mesh_more_vertices( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices,  mesh_split_edge_e1 );
    }
  if ((*m).eg_nr+2>(*m).eg_max)
    {
      err=mesh_more_edges( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,  mesh_split_edge_e1 );
    }
  if ((*m).hi_nr+1>(*m).hi_max)
    {
      err=mesh_more_hierarchy( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_hierarchy,  mesh_split_edge_e1 );
    }
  /* nodes of the edge */
  nod1=(*m).edge[eg*eg_w+ MCE1EGNOD1];
  nod2=(*m).edge[eg*eg_w+ MCE1EGNOD1+1];

  if ( (*m).type==31 )
    {
      /* create the new node */
      newnode= (*m).vx_nr;
      (*m).vx_nr++;
      
      for (i=0;i<dim; i++)
	{
	  (*m).vertex[newnode*vx_w+MCE1VXSTRT+i]=
	    0.5*(*m).vertex[nod1*vx_w+MCE1VXSTRT+i]+
	    0.5*(*m).vertex[nod2*vx_w+MCE1VXSTRT+i];
	} 
    }
  else if ( (*m).type==32 )
    {
      newnode= (*m).edge[eg*eg_w+ MCE2EGNODM];

      /* if necessary, convert the midnode */
      if ( (*m).vertex[newnode*vx_w+MCEXVXTYPE]==2.0 )
	{
	  for (i=0;i<dim; i++)
	    {
	      (*m).vertex[newnode*vx_w+MCE1VXSTRT+i]+=
		0.5*((*m).vertex[nod1*vx_w+MCE1VXSTRT+i]+
		     (*m).vertex[nod2*vx_w+MCE1VXSTRT+i]);
	    } 
	  (*m).vertex[newnode*vx_w+MCEXVXTYPE]=0.0;
        }
    }
    
  /* refinement level of the old edge */
  level=(*m).lvl;
  
  /* create the new edges */
  neweg1=(*m).eg_nr;
  neweg2=(*m).eg_nr+1;
  (*m).eg_nr+=2;
  /* child edge 1 */
  (*m).edge[neweg1*eg_w+MCE1EGNOD1  ]=nod1;
  (*m).edge[neweg1*eg_w+MCE1EGNOD1+1]=newnode;
  (*m).edge[neweg1*eg_w+MCE1EGCHL1  ]=-1;
  (*m).edge[neweg1*eg_w+MCE1EGGEO   ]=(*m).edge[eg*eg_w+MCE1EGGEO   ];
  (*m).edge[neweg1*eg_w+MCE1EGLVL   ]=level; 
  /* child edge 2 */
  (*m).edge[neweg2*eg_w+MCE1EGNOD1  ]=nod2;
  (*m).edge[neweg2*eg_w+MCE1EGNOD1+1]=newnode;
  (*m).edge[neweg2*eg_w+MCE1EGCHL1  ]=-1;
  (*m).edge[neweg2*eg_w+MCE1EGGEO   ]=(*m).edge[eg*eg_w+MCE1EGGEO   ];
  (*m).edge[neweg2*eg_w+MCE1EGLVL   ]=level;  
  /* make them children of eg */
  (*m).edge[eg*eg_w+MCE1EGCHL1  ]=neweg1;
  (*m).edge[eg*eg_w+MCE1EGCHL1+1]=neweg2;

  if ( (*m).type==32 )
    {
      /* create the new midnodes */
      newmidnode1= (*m).vx_nr;
      (*m).vx_nr++;
      newmidnode2= (*m).vx_nr;
      (*m).vx_nr++;
      
      for (i=0;i<dim; i++)
	{
	  (*m).vertex[newmidnode1*vx_w+MCE1VXSTRT+i]=0.0;
	  (*m).vertex[newmidnode2*vx_w+MCE1VXSTRT+i]=0.0;
	} 
      (*m).vertex[newmidnode1*vx_w+MCEXVXTYPE]=2.0;
      (*m).vertex[newmidnode2*vx_w+MCEXVXTYPE]=2.0;

      (*m).edge[neweg1*eg_w+MCE2EGNODM]=newmidnode1; 
      (*m).edge[neweg2*eg_w+MCE2EGNODM]=newmidnode2; 
    }

  /* check if we are on a geometry EDGE,
     if so, add the new node to tail_GEVL */
  if ( (*m).edge[eg*eg_w+MCE1EGGEO] != -1)
    {
      struct ilist *tmp;
      if ( (*m).type==31 )
	{
	  tmp = *tail_GEVL;
	  TRY_MALLOC(tmp->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  /* first the node number, then geo tag/id */
	  tmp->data=newnode;
	  tmp->next->data=(*m).edge[eg*eg_w+MCE1EGGEO];
	  tmp->next->next->data= eg;
	  tmp->next->next->next->data= -1;
	  tmp->next->next->next->next= NULL;
	  *tail_GEVL = tmp->next->next->next;
	}
      else if ( (*m).type==32 )
	{
	  tmp = *tail_GEVL;
	  TRY_MALLOC(tmp->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  /* first the node number, then geo tag/id */
	  tmp->data=newmidnode1;
	  tmp->next->data=(*m).edge[eg*eg_w+MCE1EGGEO];
	  tmp->next->next->data= neweg1;
	  tmp->next->next->next->data= -1;
	  tmp->next->next->next->next= NULL;
	  *tail_GEVL = tmp->next->next->next;

	  tmp = *tail_GEVL;
	  TRY_MALLOC(tmp->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
		     mesh_split_edge_e1);
	  /* first the node number, then geo tag/id */
	  tmp->data=newmidnode2;
	  tmp->next->data=(*m).edge[eg*eg_w+MCE1EGGEO];
	  tmp->next->next->data= neweg2;
	  tmp->next->next->next->data= -1;
	  tmp->next->next->next->next= NULL;
	  *tail_GEVL = tmp->next->next->next;
	}

    }
  

  /* new hierarchy entry */
  newhi = (*m).hi_nr;
  (*m).hi_nr++;
  (*m).hier[newhi*hi_w+MCE1HICHLD  ] = newnode;
  (*m).hier[newhi*hi_w+MCE1HIFAT1  ] = nod1;
  (*m).hier[newhi*hi_w+MCE1HIFAT1+1] = nod2;
  (*m).hier[newhi*hi_w+MCE1HILVL   ] = level;

  return SUCCESS;
}














/*********************************************************************
 *******                                                      ********
 *******   routines for adaptive refinement                   ********
 *******                                                      ********
 *********************************************************************/






/*FUNCTION*/
int mesh_split_face_green_e1( struct mesh *m , FIDX fc,
			      struct ilist **tail_GEVL,
			      struct ilist **tail_GFVL
/* refines the face fc of the mesh m with Baensch-Green
   refinement (bisection), all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  fc       - the face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the old faces and edges
		      remain to allow elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, fc_w, eg_w, bd_w, ps_w;
  int  err;


  FIDX newedge, oldedges[3], eg, eg2;
  FIDX cornernode, midnode, chld0, chld1, newface, 
	oldNode;

  fc_w=(*m).fc_w;
  eg_w=(*m).eg_w;
  bd_w=(*m).bd_w;
  ps_w=(*m).ps_w;

  /* get all old edges */
  for (i=0; i<3; i++)
    {
      oldedges[i]=(*m).face[fc*fc_w+MCE1FCEDG1+i];
    }
 
  /*the first edge is the refinement edge */ 
  eg=oldedges[0];
  
  /* make sure this edge is refined */
  if ((*m).edge[eg*eg_w+MCE1EGCHL1]==-1)
    {
      err=mesh_split_edge_e1( m, eg, 
			      tail_GEVL);
      FUNCTION_FAILURE_HANDLE( err, mesh_split_edge_e1,
			       mesh_split_face_green_e1); 
    } 

  /* check if we are on a geometry FACE,
     if so, add the new midnode to tail_GFVL */
  if ( (*m).face[fc*fc_w+MCE1FCGEO] != -1)
    {
      /* check if the new node is on a geometry edge, then the
	 tail_GEVL should have changed */

      /* store the midnode */
      chld0=(*m).edge[eg*eg_w+MCE1EGCHL1];
      /* second node of child edge is always the midnode of the old
	 edge */ 
      midnode=(*m).edge[chld0*eg_w+MCE1EGNOD1+1]; 

      /* each face node may appear twice in tail_GFVL, we take
	 care of this in the projection routine, such that each
	 node will only be adjusted once */
      if ((*m).edge[eg*eg_w+MCE1EGGEO]==-1)
	{
	  struct ilist *tmp;
	  if ( (*m).type==31 )
	    {
	      tmp= *tail_GFVL;
	      TRY_MALLOC(tmp->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      /* first the node number, then geo tag/id */
	      tmp->data=midnode;
	      tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
	      tmp->next->next->data= eg;
	      tmp->next->next->next->data= -1;
	      tmp->next->next->next->next= NULL;
	      *tail_GFVL = tmp->next->next->next;
	    }
	  else if ( (*m).type==32 )
	    {
	      FIDX midn0, midn1;
	      chld0=(*m).edge[eg*eg_w+MCE1EGCHL1  ];
	      chld1=(*m).edge[eg*eg_w+MCE1EGCHL1+1];
	      midn0=(*m).edge[chld0*eg_w+MCE2EGNODM];
	      midn1=(*m).edge[chld1*eg_w+MCE2EGNODM];

	      tmp= *tail_GFVL;
	      TRY_MALLOC(tmp->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      /* first the node number, then geo tag/id */
	      tmp->data=midn0;
	      tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
	      tmp->next->next->data= chld0;
	      tmp->next->next->next->data= -1;
	      tmp->next->next->next->next= NULL;
	      *tail_GFVL = tmp->next->next->next;

	      tmp= *tail_GFVL;
	      TRY_MALLOC(tmp->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
			 mesh_split_face_green_e1);
	      /* first the node number, then geo tag/id */
	      tmp->data=midn1;
	      tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
	      tmp->next->next->data= chld1;
	      tmp->next->next->next->data= -1;
	      tmp->next->next->next->next= NULL;
	      *tail_GFVL = tmp->next->next->next;
	    }
	}
    }

  
  /* make sure we have enough space for the new edge in the inner of
     the face, the two new faces, one new boundary and pcsurf if
     needed (the old is recycled as a 2nd), and one new node is m.type==32 */ 
  if ((*m).eg_nr+1>(*m).eg_max)
    {
      err=mesh_more_edges( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_split_face_green_e1 );
    }
  if ((*m).fc_nr+2>(*m).fc_max)
    {
      err=mesh_more_faces( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_split_face_green_e1 );
    }
  if (((*m).face[fc*fc_w+MCE1FCBND]!=-1)&&((*m).bd_nr+1>(*m).bd_max))
    {
      err=mesh_more_boundary( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_boundary, mesh_split_face_green_e1 );
    }
  if (((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)&&((*m).ps_nr+1>(*m).ps_max))
    {
      err=mesh_more_pcsurf( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcsurf, mesh_split_face_green_e1 );
    }
  if ( (*m).type==32 )
    {
      if ((*m).vx_nr+1>(*m).vx_max)
	{
	  err=mesh_more_vertices( m, 1);
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices, mesh_split_face_green_e1 );
	}
    }
    
  /* Find the cornernode which is opposite to the split edge */
  eg2 = (*m).face[fc*fc_w+ MCE1FCEDG1+1];
  cornernode = (*m).edge[eg2*eg_w+ MCE1EGNOD1  ];
  /* if this node happens to be also in edge eg, then just take the
     other one in eg2 */
  if ( (cornernode==(*m).edge[eg*eg_w+ MCE1EGNOD1  ])||
       (cornernode==(*m).edge[eg*eg_w+ MCE1EGNOD1+1]) )
    { 
      cornernode = (*m).edge[eg2*eg_w+ MCE1EGNOD1+1];
    }	    


  /* get the child edges of eg */
  chld0=(*m).edge[eg*eg_w+MCE1EGCHL1  ];
  chld1=(*m).edge[eg*eg_w+MCE1EGCHL1+1];
  
  /* get the midnode of the splitedge */
  midnode=(*m).edge[chld0*eg_w+MCE1EGNOD1+1]; 

  /* create a new edge from the midnode to the cornernode*/
  newedge=(*m).eg_nr;
  (*m).eg_nr++;

  (*m).edge[newedge*eg_w+MCE1EGNOD1  ]=cornernode;
  (*m).edge[newedge*eg_w+MCE1EGNOD1+1]=midnode;
  (*m).edge[newedge*eg_w+MCE1EGCHL1  ]=-1;
  (*m).edge[newedge*eg_w+MCE1EGGEO   ]=-1;
  (*m).edge[newedge*eg_w+MCE1EGLVL   ]=(*m).lvl;

  /* if m.type==32, create new midnode */
  if ( (*m).type==32 )
    {
      FIDX newvx;

      newvx=(*m).vx_nr;
      (*m).vx_nr++;
      for (j=0; j<(*m).dim; j++)
	{
	  (*m).vertex[newvx*(*m).vx_w+MCE1VXSTRT+j]=0.0;
	}
      (*m).vertex[newvx*(*m).vx_w+MCEXVXTYPE]=2.0;
      /* the new inner edge is never a geo-edge, 
	 but we may be on a geo-face */
      if ( (*m).face[fc*fc_w+MCE1FCGEO] != -1)
	{
	  struct ilist *tmp = *tail_GFVL;
	  TRY_MALLOC(tmp->next, 1, struct ilist,  
		     mesh_split_face_green_e1);
	  TRY_MALLOC(tmp->next->next, 1, struct ilist,  
		     mesh_split_face_green_e1);
	  TRY_MALLOC(tmp->next->next->next, 1, struct ilist,  
		     mesh_split_face_green_e1);
	  /* first the node number, then geo tag/id, then edge */
	  tmp->data=newvx;
	  tmp->next->data=(*m).face[fc*fc_w+MCE1FCGEO];
	  tmp->next->next->data= newedge;
	  tmp->next->next->next->data= -1;
	  tmp->next->next->next->next= NULL;
	  *tail_GFVL = tmp->next->next->next;
	}
      
      (*m).edge[newedge*eg_w+MCE2EGNODM]= newvx;
    }



  /* Find the acording edges to the two new elements */
  oldNode=(*m).edge[chld0*eg_w+MCE1EGNOD1];
  
  /* now make sure oldedges[1] belongs to oldNode  */
  if ( (oldNode!=(*m).edge[oldedges[1]*eg_w+MCE1EGNOD1  ])&&
       (oldNode!=(*m).edge[oldedges[1]*eg_w+MCE1EGNOD1+1])   )
    {
      /* oldedges [1] and [2] have to be swaped */
      j=oldedges[2];
      oldedges[2]=oldedges[1];
      oldedges[1]=j;
    }    

  
  /* the first new face */
  newface=(*m).fc_nr;
  (*m).fc_nr++;
          
  (*m).face[newface*fc_w+MCE1FCEDG1  ]=oldedges[1];
  (*m).face[newface*fc_w+MCE1FCEDG1+1]=newedge;
  (*m).face[newface*fc_w+MCE1FCEDG1+2]=chld0;

  (*m).face[newface*fc_w+MCE1FCCHL1] = -1;
  (*m).face[newface*fc_w+MCE1FCGEO ] = (*m).face[fc*fc_w+MCE1FCGEO];
  (*m).face[newface*fc_w+MCE1FCLVL]  = (*m).lvl;

  /* if necessary create a new boundary entry */
  if ((*m).face[fc*fc_w+MCE1FCBND ]!=-1)
    {
      FIDX new, old;
      
      old=(*m).face[fc*fc_w+MCE1FCBND ];
      new=(*m).bd_nr;

      (*m).face[newface*fc_w+MCE1FCBND]=new;
      
      (*m).bound[new*bd_w+MCE1BDFACE]=newface;
      (*m).bound[new*bd_w+MCE1BDTYPE]=(*m).bound[old*bd_w+MCE1BDTYPE];
      (*m).bound[new*bd_w+MCE1BDFNCT]=(*m).bound[old*bd_w+MCE1BDFNCT];
      (*m).bound[new*bd_w+MCE1BDORIE]=(*m).bound[old*bd_w+MCE1BDORIE];
      (*m).bound[new*bd_w+MCE1BDSSEG]=(*m).bound[old*bd_w+MCE1BDSSEG];
      
      (*m).bd_nr++;
    }
  else
    {
      (*m).face[newface*fc_w+MCE1FCBND]=-1;
    }
  
  /* if necessary create a new pcsurf entry */
  if ((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)
    {
      FIDX pcs, new;
      
      pcs=(*m).face[fc*fc_w+MCE1FCPCSU];
      new=(*m).ps_nr;
      
      (*m).pcsurf[new*ps_w+MCXXPSSURF]=newface;
      (*m).face[newface*fc_w+MCE1FCPCSU]=new;
      (*m).pcsurf[new*ps_w+MCXXPSCRIT]=(*m).pcsurf[pcs*ps_w+MCXXPSCRIT];

      (*m).ps_nr++;
    }
  else
    {
      (*m).face[newface*fc_w+MCE1FCPCSU]=-1;
    }


  
  /* store the face hierarchy */
  (*m).face[fc*fc_w+MCE1FCCHL1  ] = newface;



  /* the second new face */
  newface=(*m).fc_nr;
  (*m).fc_nr++;
          
  (*m).face[newface*fc_w+MCE1FCEDG1  ]=oldedges[2];
  (*m).face[newface*fc_w+MCE1FCEDG1+1]=newedge;
  (*m).face[newface*fc_w+MCE1FCEDG1+2]=chld1;

  (*m).face[newface*fc_w+MCE1FCCHL1] = -1;
  (*m).face[newface*fc_w+MCE1FCGEO ] = (*m).face[fc*fc_w+MCE1FCGEO];
  (*m).face[newface*fc_w+MCE1FCLVL]  = (*m).lvl;

  /* if necessary overwrite the boundary entry */
  if ((*m).face[fc*fc_w+MCE1FCBND ]!=-1)
    {
      FIDX old;
      old=(*m).face[fc*fc_w+MCE1FCBND ];
      (*m).bound[old*bd_w+MCE1BDFACE]=newface;
    }
  /* in any case, set the new BND info */
  (*m).face[newface*fc_w+MCE1FCBND]=(*m).face[fc*fc_w+MCE1FCBND ];

  /* if necessary overwrite the pcsurf entry */
  if ((*m).face[fc*fc_w+MCE1FCPCSU]!=-1)
    {
      FIDX pcs;

      pcs=(*m).face[fc*fc_w+MCE1FCPCSU];
      
      (*m).pcsurf[pcs*ps_w+MCXXPSSURF]=newface;
    }
  /* in any case, set the new PCSU info */
  (*m).face[newface*fc_w+MCE1FCPCSU]=(*m).face[fc*fc_w+MCE1FCPCSU];
  /* store the face hierarchy */
  (*m).face[fc*fc_w+MCE1FCCHL1+1] = newface;

  (*m).face[fc*fc_w+MCE1FCCHL1+2] = -1;
  (*m).face[fc*fc_w+MCE1FCCHL1+3] = -1;

  /* that's it */

  return SUCCESS;
}




/*FUNCTION*/
int mesh_refine_element_green_e1( struct mesh *m , FIDX el,
				  struct ilist **tail_GEVL,
				  struct ilist **tail_GFVL
/* refines the element and volume el of the mesh m with Baensch-Green
   refinement (bisection), all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  el       - the element to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the old faces and edges
		      remain to allow elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then three list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry,
		      third an edge ID (for computation of characteristic 
		      length)


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX fc_w, vo_w, el_w, pv_w;
  int i,j,k;
  int twice_1, twice_2, nr;
  FIDX e34, chldfc, newface, newvol, newel, fatherRed, currentRed;
  FIDX old_DD_subdomain;
  int  err;

  FIDX refedges[4], the_refedge;
  FIDX face_reo[4];
  FIDX old_faces[4];
  FIDX chld_face[2*2];

  FIDX new_edg[2];
  int common[2];

  FIDX oldel[10], newelindices[2];

  fc_w=(*m).fc_w;
  vo_w=(*m).vo_w;
  el_w=(*m).el_w;
  pv_w=(*m).pv_w;

  /* the DD_subdomain to which this element belongs */
  old_DD_subdomain=(*m).elem[el*el_w+MCXXELDDPR];

  if ( (*m).type==32 )
    {
      for (i=0; i<10; i++)
	oldel[i]=(*m).elem[el*el_w+MCE1ELNOD1+i];
    }


  /* get the four old faces and their refinement edges
     (first edge of each face) */
  for (i=0; i<4; i++)
    {
      FIDX fc=(*m).vols[el*vo_w+MCE1VOFAC1+i];

      old_faces[i] = fc;
      refedges[i]  = (*m).face[fc*fc_w+MCE1FCEDG1];
    }

  /* find one which appears twice, this will be the refinement edge of
     the element */
  for (i=0; i<3; i++)
    for (j=i+1; j<4; j++)
      {
	if (refedges[i]==refedges[j])
	  {
	    twice_1=i;
	    twice_2=j;
	  }
      }
  the_refedge=refedges[twice_1];

  /* reorder such that 
     twice_1 = 0, twice_2 = 1, other two = 2,3 */
  face_reo[0]=twice_1;
  face_reo[1]=twice_2;
  nr=2;
  for (i=0; (i<4)&&(nr<4); i++)
    {
      /* if you have uninitialised values here (twice_1 and twice_2)
	 than the mesh has probably not been prepared with 
	 mesh_prepare_adaptive_green
       */
      if ( (i!=twice_1) && (i!=twice_2) )
	{
	  face_reo[nr]=i;
	  nr++;
	}
    }

#ifdef DEBUGFEINS
  if (nr!=4)
    {
      fprintf(stderr,"mesh_refine_element_green_e1: error, "
	      "reordering of faces failed!\n");
      return FAIL;
    }
#endif

  /* in any case we will need the edge e34 (P3-P4 in the paper)
     which connects the faces face_reo[2] and face_reo[3] */
  e34 = -1;
  {
    FIDX fc2=old_faces[face_reo[2]];
    FIDX fc3=old_faces[face_reo[3]];
    for (i=0; (i<3)&&(e34<0); i++)
      { 
	FIDX eg2i = (*m).face[fc2*fc_w+MCE1FCEDG1+i];
	for (j=0; j<3; j++)
	  {
	    FIDX eg3j = (*m).face[fc3*fc_w+MCE1FCEDG1+j];
	    if (eg3j == eg2i)
	      {
		e34 = eg2i;
	      }
	  }
      }
  }

  /* to find out if it is a black or red tetrahedron (Baensch
     notation), count the refinement edges of face_reo[2:3] that are
     also an (arbitrary) edge of face_reo[0] or face_reo[1] */
  common[0]=0;
  common[1]=0; 
  for (k=0; k<2; k++)
    {
      FIDX fc=old_faces[face_reo[k]];
      for (j=1; j<3; j++)
	{
	  FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+j];
	  for (i=2; i<4 ; i++)
	    if (eg==refedges[face_reo[i]])
	      {
		common[k]++;
	      }
	}
    }

  /* now we execute the refinement of faces face_reo[0] and face_reo[1] */
  for (i=0; i<2; i++)
    {
      FIDX fc=old_faces[face_reo[i]];

      /* make sure this face is refined */
      if ((*m).face[fc*fc_w+MCE1FCCHL1]==-1)
	{
	  err=mesh_split_face_green_e1( m, fc,
					tail_GEVL, tail_GFVL );
	  FUNCTION_FAILURE_HANDLE( err, mesh_split_face_green_e1,
				   mesh_refine_element_green_e1); 
	} 

      chldfc=(*m).face[fc*fc_w+MCE1FCCHL1];

      /* the new edge (splitting the old face) is always the second
	 edge of the childfaces */
      new_edg[i]= (*m).face[chldfc*fc_w+MCE1FCEDG1+1];
    }

  /* make sure there is enough space for
     one more element and volume, one more face, if necessary one more
     pcvol and one elhier
  */
  if ((*m).el_nr+1>(*m).el_max)
    {
      err=mesh_more_elems( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elems,
			       mesh_refine_element_green_e1 );
    }
  if ((*m).vo_nr+1>(*m).vo_max)
    {
      err=mesh_more_volumes( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_volumes,
			       mesh_refine_element_green_e1 );
    }
  if ((*m).fc_nr+1>(*m).fc_max)
    {
      err=mesh_more_faces( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces,
			       mesh_refine_element_green_e1 );
    }
  if (((*m).vols[el*vo_w+MCE1VOPCVL]!=-1)&&((*m).pv_nr+1>(*m).pv_max))
    {
      err=mesh_more_pcvol( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcvol,
			       mesh_refine_element_green_e1 );
    }
  if ( (*m).type==32 )
    {
      if ( (*m).eh_nr+1>(*m).eh_max )
	{
	  err=mesh_more_elhiers( m,  (*m).el_nr );
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, 
				   mesh_refine_element_green_e1 ); 
	}
    }


  newface = (*m).fc_nr;
  (*m).fc_nr++;

  fatherRed = (*m).vols[el*vo_w + MCE1VOFRED];
  
  if ((common[0]==1)&&(common[1]==1)) /* is red */ 
    {
      currentRed=1;
    }
  else
    {
      currentRed=0;
    }

  if ( (currentRed==1)/* is red */ ||
       ( ((common[0]==2)||(common[1]==2))
	 &&(fatherRed==1) )/* (red,black) */ )
    {
      /* new face according to Figure 10 [Baensch] */
      (*m).face[newface*fc_w+MCE1FCEDG1  ]=e34;
      (*m).face[newface*fc_w+MCE1FCEDG1+1]=new_edg[0];
      (*m).face[newface*fc_w+MCE1FCEDG1+2]=new_edg[1];
    }
  /* else (black,black) or (none,black) or (none) */
  else if (common[0]==2)
    {
      /* is black, new face according to Figure 11 [Baensch] */
      (*m).face[newface*fc_w+MCE1FCEDG1  ]=new_edg[0];
      (*m).face[newface*fc_w+MCE1FCEDG1+1]=e34;
      (*m).face[newface*fc_w+MCE1FCEDG1+2]=new_edg[1];
    }
  else if (common[1]==2)
    {
      /* new face according to Figure 12 [Baensch] */
      (*m).face[newface*fc_w+MCE1FCEDG1  ]=new_edg[1];
      (*m).face[newface*fc_w+MCE1FCEDG1+1]=new_edg[0];
      (*m).face[newface*fc_w+MCE1FCEDG1+2]=e34;
    }
  else
    {
      /* (none), neither red nor black */
      /* new face according to Figure 14 [Baensch] */
      (*m).face[newface*fc_w+MCE1FCEDG1  ]=e34;
      (*m).face[newface*fc_w+MCE1FCEDG1+1]=new_edg[0];
      (*m).face[newface*fc_w+MCE1FCEDG1+2]=new_edg[1];
    }
  
  /* now complete the definition of the new face */
  (*m).face[newface*fc_w+MCE1FCCHL1] = -1;
  (*m).face[newface*fc_w+MCE1FCGEO ] = -1;
  (*m).face[newface*fc_w+MCE1FCLVL]  = (*m).lvl;
  (*m).face[newface*fc_w+MCE1FCBND ] = -1;
  (*m).face[newface*fc_w+MCE1FCPCSU] = -1;


  /* to define the child volumes, we need to know which child face of
     face_reo[0:1] shares an edge with face_reo[2:3]
     chld_face[r*2+k]=fc   means fc is a child of face_reo[k] and 
                           shares an edge with face_reo[r+2]
  */
  for (k=0; k<2; k++)
    for (i=0; i<2; i++)
      {
	int r;
	chldfc = (*m).face[old_faces[face_reo[k]]*fc_w+MCE1FCCHL1+i];
	/* the shared edge must be the refinement edge of the child
	   face */
	FIDX share_edge = (*m).face[chldfc*fc_w+MCE1FCEDG1  ];
	for (r=0; r<2; r++)
	  {
	    FIDX fc = old_faces[face_reo[r+2]];
	    for (j=0; j<3; j++)
	      {
		FIDX eg = (*m).face[fc*fc_w+MCE1FCEDG1+j];
		if (eg==share_edge)
		  {
		    chld_face[r*2+k] = chldfc;
		  }
	      }
	  }
      }

  /* define the child volumes */

  /* the first new volume */
  newvol=(*m).vo_nr;
  (*m).vo_nr++;

  (*m).vols[newvol*vo_w+MCE1VOFAC1  ]=chld_face[0*2+0];
  (*m).vols[newvol*vo_w+MCE1VOFAC1+1]=chld_face[0*2+1];
  (*m).vols[newvol*vo_w+MCE1VOFAC1+2]=old_faces[face_reo[2]];
  (*m).vols[newvol*vo_w+MCE1VOFAC1+3]=newface;
  (*m).vols[newvol*vo_w+MCE1VOFRED  ]=currentRed;

  /* copy remaining information from old to new volume */
  (*m).vols[newvol*vo_w+MCE1VORHSF  ]=(*m).vols[el*vo_w+MCE1VORHSF];
  /* if necessary create a new pcvol entry */
  if ((*m).vols[el*vo_w+MCE1VOPCVL]!=-1)
    {
      FIDX pcv, new;

      pcv=(*m).vols[el*vo_w+MCE1VOPCVL];
      new=(*m).pv_nr;
	  
      (*m).pcvol[new*pv_w+MCXXPVVOLM]=newvol;
      (*m).vols[newvol*vo_w+MCE1VOPCVL]=new;
      (*m).pcvol[new*pv_w+MCXXPVCRIT]=(*m).pcvol[pcv*pv_w+MCXXPVCRIT];

      (*m).pv_nr++;
    }
  else
    {
      (*m).vols[newvol*vo_w+MCE1VOPCVL]=-1;
    }


  /* the second overwrites the old volume */
  (*m).vols[el*vo_w+MCE1VOFAC1  ]=chld_face[1*2+0];
  (*m).vols[el*vo_w+MCE1VOFAC1+1]=chld_face[1*2+1];
  (*m).vols[el*vo_w+MCE1VOFAC1+2]=old_faces[face_reo[3]];
  (*m).vols[el*vo_w+MCE1VOFAC1+3]=newface;
  (*m).vols[el*vo_w+MCE1VOFRED  ]=currentRed;
  /* remaining information stays the same */

  
          
  /* create new element */
  newel=(*m).el_nr;
  (*m).el_nr++;

  newelindices[0]=newel;
  err=mesh_elem_from_vols_e1( m, newel);
  FUNCTION_FAILURE_HANDLE( err, mesh_elem_from_vols_e1,
			   mesh_refine_element_green_e1); 

  /* set the DD_subdomain  */
  (*m).elem[newel*el_w+MCXXELDDPR]=old_DD_subdomain;

  /* overwrite old element with last new one */
  newelindices[1]=el;
  err=mesh_elem_from_vols_e1( m, el);
  FUNCTION_FAILURE_HANDLE( err, mesh_elem_from_vols_e1,
			   mesh_refine_element_green_e1); 
  /* don't need to set the DD_subdomain for this one, as it is
     unchanged */

  /* create the new elhier entry */
  if ( (*m).type==32 )
    {
      /* define possible rotations, rot123 means the triangle 123 is
	 rotated to 231, leaving the opposite node 0 where it is,
	 stored is rot123[i]=j, the old index j that becomes the new
	 index i */
      FIDX tmp[10];
      /*  orig      0  1 2 3  4 5 6  7 8 9 */
      int rot012[]={1, 2,0,3, 5,6,4, 8,9,7};
      int rot021[]={2, 0,1,3, 6,4,5, 9,7,8};

      int rot123[]={0, 2,3,1, 6,9,7, 4,5,8};
      int rot132[]={0, 3,1,2, 7,8,4, 6,9,5};

      int rot013[]={1, 3,2,0, 8,9,5, 4,7,6};
      int rot031[]={3, 0,2,1, 7,6,9, 8,4,5};

      int rot023[]={2, 1,3,0, 5,8,9, 6,4,7};
      int rot032[]={3, 1,0,2, 8,4,7, 9,5,6};

      /* flip nodes 2 and 3 */
      int flip23[]={0, 1,3,2, 4,8,7, 6,5,9};

      int reference[]={3,0,2,8,
		       1,2,0,8 };
      FIDX new_els[2*10];

      int the_el;
      FIDX neweh;
      //FILE *fdbg; char fdbgname[]="/tmp/feins_elhier_debug.txt";
      
      /* rotate the old element to make sure the bisected edge is 1-3 */
      FIDX global8=(*m).edge[the_refedge*(*m).eg_w+MCE2EGNODM];
      int new8=-1;
      for (j=4; j<=9; j++)
	{
	  if (oldel[j]==global8)
	    new8=j;
	}
#ifdef DEBUGFEINS
      if (new8==-1)
	{
	  fprintf(stderr,"mesh_refine_element_green_e1: "
		  "elhier -> the_refedge not found\n");
	  return FAIL;
	}
#endif
      // fdbg=fopen(fdbgname,"w");
      // if (fdbg==NULL) {fprintf(stderr,"DEBUG: fopen failed\n"); return FAIL;}
      // fprintf(fdbg,"DEBUG elhier el%03d ", el);
      // fprintf(fdbg,"DBG new4 %d 4a %3d 4b %3d   ", new4, global4a, global4b);
      switch (new8)
	{
	case 8: /* do nothing */
	  break;
	case 4: /* apply rot031 */
	  mesh_elem_reorder(oldel, rot031, tmp, 10);
	  break;
	case 5: /* apply rot032 */
	  mesh_elem_reorder(oldel, rot032, tmp, 10);
	  break;
	case 6: /* apply rot021 and rot031 */
	  mesh_elem_reorder(oldel, rot021, tmp, 10);
	  mesh_elem_reorder(oldel, rot031, tmp, 10);
	  break;
	case 7: /* apply rot021 */
	  mesh_elem_reorder(oldel, rot021, tmp, 10);
	  break;
	case 9: /* apply rot012 */
	  mesh_elem_reorder(oldel, rot012, tmp, 10);
	  break;
	}
      /* now the oldel has node 8 as the midpoint of the refinement edge */
      /* now turn the 2 new elements into their reference configuration */
      for (the_el=0; the_el<2; the_el++)
	{
	  int found;
	  int is_new_el;
	  int is_node_in_new_el;
	  FIDX lookfor;

	  /* find the unique node in the new elements */
	  found=0;
	  for (i=0; (i<2)&&(found==0); i++)
	    for (j=0; (j<4)&&(found==0); j++)
	      {
		if ( (*m).elem[newelindices[i]*el_w+MCE1ELNOD1+j]
		     ==oldel[reference[the_el*4  ]])
		  {
		    found=1;
		    is_new_el=i;
		    is_node_in_new_el=j;
		  }
	      }
#ifdef DEBUGFEINS
	  if (found==0)
	    {
	      fprintf(stderr,"mesh_refine_element_green_e1: "
		      "elhier -> corner node not found in newels\n");
	      return FAIL;
	    }
#endif
	  /* copy the new element, then rotate it such that the unique 
	     node is the first in the element (see reference
	     configurations) */
	  for (i=0; i<10; i++)
	    {
	      new_els[the_el*10+i]=
		(*m).elem[newelindices[is_new_el]*el_w+MCE1ELNOD1+i];
	    }
	  // fprintf(fdbg,"DBG 0is %d ", is_node_in_new_el);
	  switch (is_node_in_new_el)
	    {
	    case 0: /* do nothing */
	      break;
	    case 1: /* apply rot012 */
	      mesh_elem_reorder(&new_els[the_el*10], rot012, tmp, 10);
	      break;
	    case 2: /* apply rot021 */
	      mesh_elem_reorder(&new_els[the_el*10], rot021, tmp, 10);
	      break;
	    case 3: /* apply rot031 */
	      mesh_elem_reorder(&new_els[the_el*10], rot031, tmp, 10);
	      break;
	    }
	  /* now the first node (0) is the first node (0) of the
	     reference, next make the second (1) fit */
	  lookfor=oldel[reference[the_el*4+1]];
	  found=0;
	  for (i=1; (i<4)&&(found==0); i++)
	    if ( new_els[the_el*10+i]==lookfor )
	      {
		found=1;
		is_node_in_new_el=i;
	      }
#ifdef DEBUGFEINS
	  if (found==0)
	    {
	      fprintf(stderr,"mesh_refine_element_green_e1: "
		      "elhier -> second node of reference not found\n");
	      return FAIL;
	    }
#endif
	  // fprintf(fdbg,"DBG 1is %d ", is_node_in_new_el);
	  switch (is_node_in_new_el)
	    {
	    case 1: /* do nothing */
	      break;
	    case 2: /* apply rot123 */
	      mesh_elem_reorder(&new_els[the_el*10], rot123, tmp, 10);
	      break;
	    case 3: /* apply rot132 */
	      mesh_elem_reorder(&new_els[the_el*10], rot132, tmp, 10);
	      break;
	    }
	  /* first (0) and second (1) nodes fit, now make the third
	     (2) and fourth (3) fit, this can only require a flip
	     between (2) and (3)
	   */
	  lookfor=oldel[reference[the_el*4+2]];
	  if (new_els[the_el*10+2]!=lookfor)
	    {
	      mesh_elem_reorder(&new_els[the_el*10], flip23, tmp, 10);
	      // fprintf(fdbg,"DBG flip23 ");
	    }
#ifdef DEBUGFEINS
	  if (new_els[the_el*10+2]!=lookfor)
	    {
	      fprintf(stderr,"mesh_refine_element_green_e1: "
		      "elhier -> third node of reference not found\n");
	      return FAIL;
	    }
	  if (new_els[the_el*10+3]!=oldel[reference[the_el*4+3]])
	    {
	      fprintf(stderr,"mesh_refine_element_green_e1: "
		      "elhier -> fourth node of reference not found\n");
	      return FAIL;
	    }
#endif
	} /* for the_el */
      /* now all four new_outer_els are in reference configuration, 
	 generate the elhier entry */
      neweh=(*m).eh_nr;
      (*m).eh_nr++;
      // fprintf(fdbg,"\nDBG father ");
      for (i=0; i<10; i++)
	{
	  (*m).elhier[neweh*(*m).eh_w+MCE2EHFAT1+i]=oldel[i];
	  // fprintf(fdbg," %3d ", (*m).elhier[neweh*(*m).eh_w+MCE2EHFAT1+i]);
	}
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1   ]=new_els[1*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 1]=new_els[0*10+7];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 2]=new_els[0*10+8];
      (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+ 3]=new_els[0*10+9];
      /* mark remaining children as not used */
      for (i=4; i<25; i++)
	{
	  (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+i]=-1;
	}
      (*m).elhier[neweh*(*m).eh_w+MCE2EHLVL]=(*m).lvl;

      // fprintf(fdbg,"DBG child ");
      // for (i=0; i<25; i++)
	// {
	//   fprintf(fdbg," %3d ", (*m).elhier[neweh*(*m).eh_w+MCE2EHCHL1+i]);
	// }
      // fprintf(fdbg,"DBG end\n");
      // fclose(fdbg);

    } /* if m.type==32 create elhier */



  /* that's it */

  return SUCCESS;
}




/*FUNCTION*/
int mesh_refine_adaptive_e1( struct mesh *m, FIDX *marked_elem
/* refines all marked elements of the mesh m with the
   Baensch-Green refinement, all necessarry new parts are
   created thereby,

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991


   In/Out: m           - the mesh, is modified accordingly, e.g. new
                         elements, volumes, faces, edges, vertices, boundary
                         elements and hierarchy entries are created,
  		         the element el and the volume el are thereby
		         replaced by one of their children, the faces
		         and edges remain to allow other elements to
		         stay intact, 
		         if necessary the fields of the mesh are
		         reallocated to have space for the new stuff
	   marked_elem - Vector holding which elements are to be
	                 refined (1 = to be refined)
   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
                         valid
*/
		     ){
  FIDX old_el_nr, new_el_nr, old_fc_nr, eg_w, fc_w, vo_w, i, j;
  int  err, refine, refine_loops;

  struct ilist head_geo_edge_vertex_list, *tail_GEVL;
  struct ilist head_geo_face_vertex_list, *tail_GFVL;

  head_geo_edge_vertex_list.data = -1;
  head_geo_edge_vertex_list.next = NULL;
  tail_GEVL = &head_geo_edge_vertex_list;

  head_geo_face_vertex_list.data = -1;
  head_geo_face_vertex_list.next = NULL;
  tail_GFVL = &head_geo_face_vertex_list;

  vo_w=(*m).vo_w;
  fc_w=(*m).fc_w;
  eg_w=(*m).eg_w;
  
  old_el_nr = (*m).el_nr;
  (*m).lvl++;

  if ( (*m).type==32 )
    {
      if ((*m).elhier==NULL ) 
	{
	  (*m).eh_w   = MCE2EHLN;
	  (*m).eh_max = (*m).el_nr;
	  (*m).eh_nr  = 0;
	  TRY_MALLOC( (*m).elhier,  (*m).eh_w*(*m).eh_max, FIDX, 
		      mesh_refine_adaptive_e1);
	}
      if ( (*m).eh_max < (*m).eh_nr + (*m).el_nr )
	{
	  err=mesh_more_elhiers( m,  (*m).el_nr );
	  FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, 
				   mesh_refine_adaptive_e1 ); 
	}
    }


  /* first loop over all elements, refine all marked elements */
  for (i=0; i< old_el_nr; i++)
    {
      if ( marked_elem[i]==1)      
        {
          err=mesh_refine_element_green_e1( m , i, &tail_GEVL, &tail_GFVL);
          FUNCTION_FAILURE_HANDLE( err, 
		mesh_refine_element_green_e1, mesh_refine_adaptive_e1 );
        }
    }
  new_el_nr = (*m).el_nr;
  /* we want to refine the initially marked elements three times,
     that is into eight new elements rather than just two 
     (to accellerate element growth) */  
  /* second loop over all elemtents again and all new elements */ 
  if(0==1)
    {
      /* two more loops */
      for (j=0; j<2; j++)
	{
	  for (i=0; i< old_el_nr; i++)
	    {
	      if ( marked_elem[i]==1)      
		{
		  err=mesh_refine_element_green_e1( m , i, 
						    &tail_GEVL, &tail_GFVL);
		  FUNCTION_FAILURE_HANDLE( err, 
					   mesh_refine_element_green_e1, 
					   mesh_refine_adaptive_e1 );
		}
	    }
	  for (i=old_el_nr; i< new_el_nr; i++)
	    {
	      err=mesh_refine_element_green_e1( m , i,
						&tail_GEVL, &tail_GFVL);
	      FUNCTION_FAILURE_HANDLE( err, 
				       mesh_refine_element_green_e1,
				       mesh_refine_adaptive_e1 );
	    }
	  new_el_nr = (*m).el_nr;
	}
    }

  /* loop over all elements, refine all elements with hanging nodes 
     while there can be hanging nodes (only after a refinement) */
  refine=1;
  refine_loops=0;
  
  while (refine!=0)
    {
      int *fc_hang;

      refine=0;
      refine_loops++;
      old_el_nr=(*m).el_nr;
      old_fc_nr= (*m).fc_nr;

      TRY_MALLOC(fc_hang, old_fc_nr, int, mesh_refine_adaptive_e1 );

      for (i=0; i< old_fc_nr; i++)
	{
	  if ( ( ((*m).face[i*fc_w+MCE1FCCHL1]!=-1)||
		 ((*m).edge[(*m).face[i*fc_w+MCE1FCEDG1  ]*eg_w+MCE1EGCHL1]!=-1) )||
	       ( ((*m).edge[(*m).face[i*fc_w+MCE1FCEDG1+1]*eg_w+MCE1EGCHL1]!=-1) ||
		 ((*m).edge[(*m).face[i*fc_w+MCE1FCEDG1+2]*eg_w+MCE1EGCHL1]!=-1) )
	       ) 
	    {
	      fc_hang[i]=1;
	    }
	  else
	    {
	      fc_hang[i]=0;
	    }
	}

      for (i=0; i< old_el_nr; i++)
       {
          /* test if element has hanging nodes 
             (if and only if a face was already split) */           
	 if ( ( (fc_hang[(*m).vols[i*vo_w+MCE1VOFAC1  ]]==1)||
		(fc_hang[(*m).vols[i*vo_w+MCE1VOFAC1+1]]==1)  )||
	      ( (fc_hang[(*m).vols[i*vo_w+MCE1VOFAC1+2]]==1)||
		(fc_hang[(*m).vols[i*vo_w+MCE1VOFAC1+3]]==1) ) )
	   {
	     err=mesh_refine_element_green_e1( m , i, &tail_GEVL, &tail_GFVL);
	     FUNCTION_FAILURE_HANDLE( err, mesh_refine_element_green_e1, 
				      mesh_refine_adaptive_e1 );
	     refine=1;
	   } 
       }
      /* has the be re-computed in next loop */
      free(fc_hang);
    }

  fprintf(stderr,"DEBUG: refine_loops = %d\n", refine_loops);
  
  /* Refinement is done, there are no hanging nodes anymore */  

  /* check if a mesh generator was used, if so, move the new points
     onto their geometry */
  if ((tail_GEVL != &head_geo_edge_vertex_list)
      ||(tail_GFVL != &head_geo_face_vertex_list))
    {
      /* actually need to project */
      if (((*m).meshgen==2)&&((*m).meshgenDATA!=NULL))
	{
	  /* gmsh */
	  err=mesh_gmsh_project_nodes_e1( m, &head_geo_edge_vertex_list,
					  &head_geo_face_vertex_list );
	  FUNCTION_FAILURE_HANDLE( err, mesh_gmsh_project_nodes_e1, 
				   mesh_refine_adaptive_e1 );
	}
    }

  ilist_free(&head_geo_edge_vertex_list.next);
  ilist_free(&head_geo_face_vertex_list.next);

  err=mesh_init_DD_elem( m );
  FUNCTION_FAILURE_HANDLE(err, mesh_init_DD_elem, mesh_refine_adaptive_e1 );
    
  return SUCCESS;
}




/*FUNCTION*/
int mesh_prepare_adaptive_green_e1( struct mesh *m
/* makes sure the connectivity of the mesh is suitable for adaptive
   green refinement (Baensch), i.e. that the longest edge in each face
   is the first edge of the face, with ties being broken by preferring
   the edge with higher number

   
   in/out:  m       - (given by reference) a e1 mesh,

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  FIDX i, fc, fc_nr, fc_w, eg_nr, eg_w, vx_w;
  double *edge_length;

  /* in each face, make sure the longest edge is first in the face,
     ties are broken by preferring the larger edge number
     (set data structure for Baensch-green refinement) */
  
  /* first compute all squares of edge lengths */
  fc_nr = (*m).fc_nr;
  fc_w  = (*m).fc_w;
  eg_nr = (*m).eg_nr;
  eg_w  = (*m).eg_w;
  vx_w  = (*m).vx_w;

  TRY_MALLOC( edge_length, eg_nr, double, mesh_prepare_adaptive_green_e1);

  for (i=0; i<eg_nr; i++)
    {
      int j;
      FIDX node1, node2;

      node1 = (*m).edge[i*eg_w+MCE1EGNOD1  ];
      node2 = (*m).edge[i*eg_w+MCE1EGNOD1+1];

      edge_length[i] = 0.0;
      for (j=0; j<3; j++)
	{
	  double diff;
	  diff = 
	    (*m).vertex[node2*vx_w+MCE1VXSTRT+j]
	    - (*m).vertex[node1*vx_w+MCE1VXSTRT+j];

	  edge_length[i] += diff*diff;
	}
    }

  /* now for each face, order the edges */
  for (fc=0; fc<fc_nr; fc++)
    {
      int longest, j;
      FIDX eg_longest;
      double l_longest;

      FIDX old_edges[3];

      /* find the longest */
      longest    = 0;
      eg_longest = (*m).face[fc*fc_w+MCE1FCEDG1  ] ;
      l_longest  = edge_length[eg_longest];
      for (j=1; j<3; j++)
	{
	  FIDX eg=(*m).face[fc*fc_w+MCE1FCEDG1+j] ;
	  if ( (edge_length[eg_longest]>l_longest) ||
	       ( (edge_length[eg_longest]==l_longest) &&
		 (eg>eg_longest) ) )
	    {
	      longest    = j;
	      eg_longest = eg ;
	      l_longest  = edge_length[eg_longest];
	    }
	}
      
      /* second and third are arbitrary */

      /* now write the new numbering */
      for (j=0; j<3; j++)
	{
	  old_edges[j]=(*m).face[fc*fc_w+MCE1FCEDG1+j];
	}
      for (j=0; j<3; j++)
	{
	  (*m).face[fc*fc_w+MCE1FCEDG1+j]=old_edges[(longest+j)%3];
	}

    } /* for fc */

  free(edge_length);

  return SUCCESS;
}
