/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
mesh.h
*/
#ifndef __MESH_H__
#define __MESH_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"

void mesh_init( struct mesh *m
/* initialises the mesh to be empty,

   Output: m        - an empty mesh, all inner pointers set to NULL,
                      array lengths etc. to zero
*/);

void mesh_info( FILE *outf, struct mesh *m
/* printf info on the mesh to file outf (e.g. outf=stdout),

   Input:  m        - a mesh
   Output: (to file outf)

*/);

int mesh_refine_uniform_t1( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m       - the mesh, is modified accordingly, e.g. new
                     elements, faces, edges, vertices, boundary
                     elements and hierarchy entries are created,
		     the element el and the face el are thereby
		     replaced by one of their children, the edges
		     remain to allow other elements to stay intact,
		     if necessary the fields of the mesh are
		     reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_element_t1( struct mesh *m , FIDX el
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_split_edge_t1( struct mesh *m , FIDX eg
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices, boundary elements and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring elements
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_solution_exp_t1( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a NAG IRIS EXPLORER readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_solution_vtk_tx( struct mesh *m, struct vector *vec_x,
				FIDX vcompo, struct vector *scalar_x,
				FIDX scompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a vtk readable
   file for visualisation, 
   this can for example be used with paraview
           http://www.paraview.org/

   Input:  m         - the mesh
           vec_x     - vector with data associated to the nodes,
	               for representing a vector field of dimension
	               vcompo, ignored if vec_x==NULL
	   vcompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector x have to be numbered such that
	               vec_x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh, has to be TWO for t1
	               mesh, ignored if vec_x==NULL
           scalar_x  - vector with data associated to the nodes,
	               for representing scompo scalar fields,
		       for each scalar field a separate part is
		       written in the file, ignored if scalar_x==NULL
	   scompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector scalar_x have to be numbered such that
	               saclar_x[j*vx_nr+i] gives the j-th scalar
	               variable in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_solution_vtk_warp_tx( struct mesh *m, struct vector *scalar_x,
				     FIDX namlen, char *name,
				     double *retMinData, double *retMaxData
/* writes the mesh and the solution into a vtk readable
   file for visualisation, 
   this can for example be used with paraview
           http://www.paraview.org/

   Input:  m         - the mesh
           scalar_x  - vector with data associated to the nodes,
	               for representing 1 scalar field,
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)
           retMinData - the minimum of the scalar_x (by reference)
           retMaxData - the maximum of the scalar_x (by reference)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_solution_femplot_t1( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a ?FEMplot? readable
   file for visualisation 
   FEMplot by project students (Lars Eiserbeck und Felix Prehl, 2007)
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_gnuplot_boundary_circles_t2( FIDX ncirc, double *Rs, double tol,
				      struct mesh *m,
				      struct vector *data, FIDX ncomp, int mode,
				      char *fname_base
/* assuming that we have ncirc circles around (0,0) as boundary
   (r=Rs[i]), plot the data over the boundary

   Input:  ncirc     - number of circles
           Rs        - array of length ncirc, defining the radius of
                       each circle, the data is written to a separate
                       file for each circle
	   tol       - tolerance in testing if a point is on a circle
	   m         - the mesh that the data corresponds to
           data      - vector containing the data
           ncomp     - number of components in the data,  
           mode      - defines if the components of the data are
                       written to the file (mode=0), or the norm of
                       the vector of components associated to each
                       node of the mesh (mode=1)
           fname_base- base of the filenames that is written to, the
                       files will be 
		         <fname_base>_R1.txt
		       to
		         <fname_base>_R<ncirc>.txt

   Output: (the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_bound_gnuplot_tx( struct mesh *m, 
				FIDX namlen, char *name,
				int type
/* writes the mesh into a GNUPLOT readable file (for visualisation),
   this is done by writing a line for each edge of the finest level
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, full name will be
	               <name>.dat
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2
                       in any case only straigth lines are written,
                       ignoring the possibility of curved edges for T2
                       meshes

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_mesh_gnuplot_tx( struct mesh *m, 
				FIDX namlen, char *name,
				int type
/* writes the mesh into a GNUPLOT readable file (for visualisation),
   this is done by writing a line for each edge of the finest level
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.dat
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2
                       in any case only straigth lines are written,
                       ignoring the possibility of curved edges for T2
                       meshes

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_ssegs_svg_tx( struct mesh *m, 
			     FIDX namlen, char *name,
			     int type
/* writes the shape-segments into a scalable vector graphics (SVG)
   file, which can for example be read by sodipodi or inkscape
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, full name will be
	               <name>.svg
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_sseg_svg_file_tx(FILE *datfile, FILE *datfile_mrk,
				struct mesh *m, FIDX seg, FIDX node1,
				double diam
/* writes the segement to the files specified by the file pointers,
   depending on the type and orientation of the segment

   input:   datfile - pointer to path file, it is written to the file in
                      this routine
	    datfile_mrk
                    - pointer to marker file, parameter markers are
                      written to the file in this routine
	    m       - a t1 or t2 mesh
            seg     - the segment to be written into the file
            node1   - first node of the segment, the orientation of
                      the segment is adjusted such that it really has
                      this node as first node
            diam    - diameter of bounding box, used for scaling the markers

   output:  (writes to the file)

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_write_sseg_bbox_tx(double *minx, double *maxx, double *miny, double *maxy,
			    struct mesh *m
/* find the bounding box for all shape segments in the file

   input:   m       - a t1 or t2 mesh
            
   output:  minx,maxx,miny,maxy
                    - the min and max of all coordinates of shape
                      segment control points

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_read_file_t1( struct mesh *m, struct solver_settings *set, char *name
/* reads a mesh file, for a descrition of the file format, see the
   example file "mesh_example.f1m"

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_read_header_key_int( FILE *in, char *keywd, FIDX *value
/* reads the keyword part of the header of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m" 
   here a entry 
     <keywd N > 
   is looked up, and the integer number N is returned in value 

   input:   in      - a file open for reading, reading position at the
                      begin of the header part
	    keywd   - keyword which shall be looked up
   
   output:  value   - (given by reference) the integer number found
                      with for the keyword

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_header_key_double( FILE *in, char *keywd, double *value
/* reads the keyword part of the header of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m" 
   here a entry 
     <keywd N > 
   is looked up, and the double number N is returned in value 

   input:   in      - a file open for reading, reading position at the
                      begin of the header part
	    keywd   - keyword which shall be looked up
   
   output:  value   - (given by reference) the double number found
                      with for the keyword

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_solver_settings( FILE *in, struct solver_settings *set
/* reads the solver settings of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading

   output:  set     - holding the settings for the solver
   
   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

void solver_settings_init(struct solver_settings *set 
/* initialises the solver settings to be empty and its valuse to be zero

   Output: set        - empty solver settings, all values are set to zero
*/);

int mesh_read_vertex( FILE *in, struct mesh *m, FIDX vx_nr, FIDX vxstrt
/* reads the vertex part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            vx_nr   - the number of vertices in the file, as given by
                      the header
	    vxstrt  - the position within the vertex blocks, where the
	              coordinates of the vertex start
   
   output:  m       - (given by reference) a mesh, the vertex part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_elements_t1( FILE *in, struct mesh *m, FIDX el_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            el_nr   - the number of elements in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the elem and face
                      part are allocated here, but only elem set!
                      in face only the RHSF part is defined and PCVL
                      is initiated
   

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_boundary_t1( FILE *in, struct mesh *m, FIDX bd_nr, FIDX sg_nr
/* reads the boundary part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            bd_nr   - the number of boundary entries in the file, as
                      given by the header
            sg_nr   - the number of shape segment entries in the file,
	              as given by the header
   
   output:  m       - (given by reference) a mesh, the edge and bound
                      part are allocated and set here, but in edge
                      only those edges are defined which are part of
                      the boundary!

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_triangle_wrapper( FILE *infile, struct mesh *m,
				FIDX hole_nr, FIDX rhsf
/* reads the hole part of a mesh file, then calls the mesh generator
   "triangle" to generate a mesh (mainly the elements, plus changes of
   the boundary), for a descrition of the file format, see the example
   file "mesh_example.f1m",
   for each volume criterion in pccrit a pcvol entry is created for
   each element  


   input:   infile  - a file open for reading
            hole_nr - the number of hole entries in the file, as
                      given by the header
            rhsf    - the index of the rhs function, set -1 to ignore
   
   output:  m       - (given by reference) a mesh, the vertex,
                      element, edge and bound parts are reallocated
                      and reset here, according to the mesh generated
                      by "triangle",  (in edge only those edges are
                      defined which are part of the boundary!)

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_holes( FILE *in, FIDX dim, FIDX hole_nr, 
		     double *holes
/* reads the holes part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            dim     - dimension of the hole entries
            hole_nr - number of holes, as specified in the header

   output:  holes   - vector of the points inside the holes,
                      holes[i*dim+j] gives the j-th component of the
                      i-th hole point

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_pcsurf( FILE *in, struct mesh *m, FIDX ps_nr
/* reads the performance criteria surfaces of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m"

   input:   in      - a file open for reading
            ps_nr   - the number of pcsurf entries in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the pcsurf part is
                      allocated and set here, the edge and pccrit
                      parts have to be set properly, as they will be
                      referenced 

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_pcvol( FILE *in, struct mesh *m, FIDX pv_nr
/* reads the performance criteria volumes of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m",
   if the triangle mesh generator is used no pcvol entries may be
   defined in the file, but for each volume criterion in pccrit a
   pcvol entry is created for each element 

   input:   in      - a file open for reading
            pv_nr   - the number of pcvol entries in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the pcvol part is
                      allocated and set here, the pccrit and face
                      parts have to be set properly, as they will be
                      referenced 

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_pccrit( FILE *in, struct mesh *m, FIDX pc_nr
/* reads the performance function part of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m"

   input:   in      - a file open for reading
            pc_nr   - the number of performance functions in the file,
                      as given by the header
   
   output:  m       - (given by reference) a mesh, the pccrit part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_function( FILE *in, struct mesh *m, FIDX fu_nr
/* reads the function part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            fu_nr   - the number of functions in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the function part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_rhsf( FILE *in, struct mesh *m, FIDX *rhsf
/* reads the rhsf part of a mesh file, if any,
   for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
   
   output:  m       - (given by reference) a mesh, the function part is
                      allocated and defined
            rhsf    - specified function, if one is specified in the mesh file, 
	              -1 otherwise (ignore indicator)

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_parameter( FILE *in, struct mesh *m, FIDX pa_nr
/* reads the parameter part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            pa_nr   - the number of parameters in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the parameter part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_sseg( FILE *in, struct mesh *m, FIDX sg_nr, FIDX sp_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            sg_nr   - the number of shape segments in the file, as
                      given by the header
            sp_nr   - the number of shape parameters in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the sseg
                      part is allocated and set here

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_spar( FILE *in, struct mesh *m, FIDX sp_nr
/* reads the shape_par part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            sp_nr   - the number of shape parameters in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the shape_par part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_compute_conns_t1( struct mesh *m
/* builds the part of the connectivity data of the internal mesh
   format that is not covered by the mesh files (see the example file
   "mesh_example.f1m")

   In/Out:  m       - (given by reference) a mesh, the edge and face
                      parts are completed

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_pcvol_compute_stripe(struct mesh *m,
			      FIDX crit_inner, FIDX crit_outer,
			      FIDX bd1, FIDX bd2
/* re-defines the performance criteria for all elements, based on
   whether they are between shape segments bd1 and bd2, or outside of
   those: inside the criterion is changed to crit_inner, outside to crit_outer
   
   IN/OUT:   m         - the mesh, pcvol is modified accordingly
   
   INPUT:    crit_inner- id of the pccrit entry for the elements between 
                         bd1 and bd2
	     crit_outer- id of the pccrit entry for the elements outside of 
	                 bd1 and bd2
             bd1
	     bd2       - shape segments that enclose the inner part of 
	                 the criterion
 
   RETURN:   SUCCESS   - success
	     FAIL      - failure, see error message, output will not be valid

*/);

int mesh_init_DD_elem( struct mesh *m
/* initialises/sets the DD_elem part of the mesh

   in/out:  m       - (given by reference) a mesh, the DD_* parts are
                      initialised or set 

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_refine_uniform_t2( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_element_t2( struct mesh *m , FIDX el
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_split_edge_t2( struct mesh *m , FIDX eg
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices, boundary elements and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring elements
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_t2_get_elem_from_face( struct mesh *m , FIDX i
/* extracts the nodes of the element i from the data in m.face and
   m.edge and stores the element i in m

   Input:  i        - the element to be build

   In/Out: m        - the mesh, is modified accordingly, 
                      only the elem part is modified

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_reorder_tx( struct mesh *m, int type
/* reorders the vertices, faces, elements, such that higher cache
   efficiency can be achied
   
   Input:  type    - the type of the mesh, type=1 linear, =2 quadratic

   In/Out: m       - the mesh, node numbers are overwritten

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void multilvl_free( struct multilvl *ml
/* empties the multilvl data structure, such that it can be freed

   Output: ml      - internal memory is released,
*/);

void multilvl_null_def( struct multilvl *ml
/* defines the values in multilvl data structure, such that it can be
   given to multilvl_free and give no errors, even before
   multilvl_init is called

   Output: ml      - is defined to be empty,
*/);

int multilvl_init_t2( struct mesh *m, FIDX datdim, struct multilvl *ml
/* initialises the multilvl data structure
   
   Input:  m       - the mesh (T2) to which the multilvl data shall
                     correspond (the important bit is the hierarchy
                     data) 
	   datdim  - the number of dofs per node for which the
	             multilvl data will be created, on each of the
	             levels dim*(this levels vx_nr) components will be
	             available

   Output: ml      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int multilvl_init_tx( struct mesh *m, FIDX datdim, struct multilvl *ml,
		      int type
/* initialises the multilvl data structure
   
   Input:  m       - the mesh to which the multilvl data shall
                     correspond (the important bit is the hierarchy
                     data) 
	   datdim  - the number of dofs per node for which the
	             multilvl data will be created, on each of the
	             levels dim*(this levels vx_nr) components will be
	             available
           type    - the type of the mesh, type=1 linear, =2 quadratic

   Output: ml      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void bpx_free( struct bpxdata *bpx
/* empties the data structure for the BPX preconditioner, such that it
   can be freed

   Output: bpx     - internal memory is released,
*/);

int bpx_init_tx( struct bpxdata *bpx, struct multilvl *ml
/* initialises the data structure for the BPX preconditioner
   
   Input:  msh     - the mesh to which the BPX data shall correspond
                     (the important bit is the hierarchy data)
           ml      - multilevel data struct

   Output: bpx     - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void mg_free( struct mgdata *mg
/* empties the multilvl data structure, such that it can be freed

   Output: mg      - internal memory is released,
*/);

void mg_null_def( struct mgdata *mg
/* defines the multilvl data structure, such that a call ti mg_free
   prodruces no errors even though mg_init has not been called yet

   Output: mg      - is defined to be empty,
*/);

int mg_init_tx( struct sparse *Ks, struct mesh *m, struct multilvl *ml,
		struct mgdata *mg, struct projector1 *P
/* initialises the data structure for the multigrid preconditioner
   
   Input:  Ks      - array of sparse matrices, K[i] is the stiffness
                     matrix for the mesh on the i-th level
           m       - the mesh to which the mgdata shall correspond
                     (the important bits are the hierarchy and element
		     hierarchy data)
	   ml      - multilevel data
	   P       - Projector for boundary conditions, give P=NULL if
	             not wanted

   Output: mg      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_restrict_t2( struct mgdata *mg,
		    FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, 
	             the lvlin part is read and modified, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_interpolate_t2( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout >= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mesh_refine_green_element_t2( struct mesh *m , FIDX el
/* refines the element el of the mesh m with the Baensch-Green
   refinement, all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_adaptive_t2( struct mesh *m, FIDX *marked_elem
/* refines all marked elements of the mesh m with the
   Baensch-Green refinement, all necessarry new parts are
   created thereby,

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991


   In/Out: m           - the mesh, is modified accordingly, e.g. new
                         elements, faces, edges, vertices, boundary
                         elements and hierarchy entries are created,
  		         the element el and the face el are thereby
		         replaced by one of their children, the edges
		         remain to allow other elements to stay intact,
		         if necessary the fields of the mesh are
		         reallocated to have space for the new stuff
	   marked_elem - Vector holding which elements are to be
	                 refined (1 = to be refined)
   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
                         valid
*/);

int mg_restrict_t1( struct mgdata *mg, FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector,
	             the lvlin part is read and modified, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_interpolate_t1( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_restrict_tx( struct mgdata *mg, FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_interpolate_tx( struct mgdata *mg, FIDX lvlin, FIDX lvlout,
		       double *mlv
/* interpolates the vector lvlin part of mlv to lvlout
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout >= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the interpolateion from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_restrict_t1_t2( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation,
   using the T1 hiriarchy of the T1 mesh associated with the T2 mesh

   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mg_interpolate_t1_t2( struct mgdata *mg,
			  FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   using the T1 hiriarchy of the T1 mesh associated with the T2 mesh
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int mesh_write_solution_exp_t2( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a NAG IRIS EXPLORER readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes,
                       usually only input, but if (xtype==2) then the
                       pressure values are corrected, such that the
		       edge-mid-nodes have propper values
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat
           
   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_write_solution_femplot_t2( struct mesh *m, struct vector *x,
				    FIDX ncompo,
				    FIDX namlen, char *name
/* writes the mesh and the solution into a ?FEMplot? readable
   file for visualisation 
   FEMplot by project students (Lars Eiserbeck und Felix Prehl, 2007)
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes,
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat
           
   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_t1_to_t2( struct mesh *m1, struct mesh *m2
/* creates the T2 mesh by converting the T1 mesh m1 into a T2 mesh,
   Input:  m1         - T1 mesh
   Output: m2         - T2 mesh, (empty mesh given via reference,
                        interior is set up)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_t2_to_t1( struct mesh *m2, struct mesh *m1
/* creates the T1 mesh by converting the T2 mesh m2 into a T1 mesh,
   where one element/face in m2 creates 4 elements/faces in m1, such
   that they have the same number of vertices

   Input:  m2         - T2 mesh
   Output: m1         - T1 mesh, (empty mesh given via reference,
                        interior is set up)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_func_eval ( struct mesh *m, FIDX tfu, double *point, double time,
		     FIDX reslen, double *result, double *gradf, double *dotf
/* evaluates a function as defined by the mesh

   Input:  m       - the mesh
           tfu     - id of the function, points to the function
                     within m.func
	   point   - point where the function has to be evaluated
	             (vector of length m.dim)
           time    - time when the function is to be evaluated,
                     just provide time=0.0 if stationary problem 
	   reslen  - length of the provided result vector,
	             if the function type doesn't match reslen
	             this function will return FAIL

   Output: result  - vector of length reslen (provided by reference),
                     holds the result of the function evaluation on
                     SUCCESSful return
           gradf   - gradient of the function, only modified if
                     gradf!=NULL, otherwise has to be array of length reslen*(m.dim)
		     gradf[i*dim+j] is the derivative of the i-th
		     component of the function wrt. the j-th variable
		     of space
	   dotf    - time derivative of the function, only modified if
	             dotf!=NULL, otherwise has to be vector of length reslen
                     
   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_sseg_adjust_tx( struct mesh *m, int insert,
			 FIDX nI, struct vector *dIdx,
			 struct vector *dIdF,
			 int type
/* Adjusts the shape segment parts of the boundary to match the
   current parameters or evaluates the derivatives of the node
   positions and applies the chain rule to evlauate dIdF where x=x(F),
   and dIdx is given, dIdF:=dIdx*dxdF.

   In/Out: m       - the mesh, vertices are adjusted (moved), 
                     only modified if dIdF==NULL

   Input:  insert  - wether or not points may be inserted on the
                     curved boundary, ==1 may instert, ==0 not,
		     ==1 only usefull if in the pre-mesh-generator
		     stage, as this could destroy the connectivity
		     the nodes are distributed such that the
		     t-values such that |x_i - x_(i-1)| <= 4*c_curve*r, 
		     where r is the local curvature radius of the spline,
		     c_curve << 0.5    (better <=1/10) 
		     and t_i-t_(i-1) <= delta_t_max

	   nI      - number of performance criteria, size of dIdF and
                     dIdx, ignored if dIdF==NULL
           dIdx    - derivatives of performance criteria wrt node
                     positions, i.e. dIdx[i].V[d*vx_nr+j] is
                     derivative of criterion I[i] (i=0..nI) wrt the
                     d-th coordinate of node j, ignored if dIdF==NULL
	   type    - type of the mesh, type==1 ==> T1, type==2 ==> T2

   Output: dIdF    - chain rule is applied as describe above, 
		     ignored if dIdF==NULL, otherwise, dIdF[i].V[j] is
		     the derivative of performance criterion I[i] wrt
		     spar[j]

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_sseg_adjust_nodes( struct mesh *m, FIDX seg, FIDX seg_nds,
			    FIDX *nodes, FIDX ignore1, 
			    FIDX ignore2, FIDX l_mcVXSTRT, 
			    FIDX nI, struct vector *dIdx,
			    struct vector *dIdF
/* Adjusts the nodes listed in order in nodes according to shape
   segment seg to match the current parameters or evaluates the
   derivatives of the node positions and applies the chain rule to
   evlauate dIdF where x=x(F), and dIdx is given. 

   In/Out: m       - the mesh, vertices are adjusted (moved), 
                     only modified if dIdF==NULL
           dIdF    - chain rule is applied, adding the contributions
                     of this segment, has to be initialised outside!,
		     ignored if dIdF==NULL, otherwise, dIdF[i].V[j] is
		     the derivative of performance criterion I[i] wrt
		     spar[j]

   Input:  seg     - id of the segment
	   seg_nds - number of nodes in the list nodes
           nodes   - an ordered list of nodes
	   ts      - t-value for each node
	   ignore1 - see ignore2
	   ignore2 - pointers into nodes for nodes which are to be
	             ignored, because they are set in another segment
	   l_mcVXSTRT
	           - start of the coordinates in a vertex entry in
	             this mesh
           nI      - number of performance criteria, size of dIdF and
                     dIdx, ignored if dIdF==NULL
           dIdx    - derivatives of performance criteria wrt node
                     positions, i.e. dIdx[i].V[d*vx_nr+j] is
                     derivative of criterion I[i] (i=0..nI) wrt the
                     d-th coordinate of node j, ignored if dIdF==NULL


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_sseg_build_tmesh_t1(double **tmesh, FIDX *nt,
			     struct mesh *m, FIDX seg,
			     double t_0, double t_end, 
			     double c_curve, double delta_t_max
/* creates a mesh of t-values 

      t_0<t_1<t_2...<t_(nt-1) 

   such that 

      |x_i - x_(i-1)| <= 4*c_curve*r,
   and 
      t_i-t_(i-1) <= delta_t_max

   where r is the local curvature radius of the spline, 

   Let d be the distance between the spline and the polygon of edges
   that will result from the t-mesh. The above constraint guarantees
      d < c_curve*|x_i - x_(i-1)|
   if 
      c_curve << 0.5 (better <= 1/10) 
   

   Output: tmesh   - (pointer to double given by reference)
                     vector of length nt, containig t_0,...,t_(nt-1),
                     memory is allocated here, has to be freed again
		     by the calling routine, (pointer give by reference)
           nt      - (FIDX given by reverence)
	             the number of mesh points

   Input:  m       - the mesh,
           seg     - id of the segment
	   t_0     - starting point of the t-mesh
	   t_end   - end point of the t-mesh
	   c_curve - constant for the constraint, see above
           delta_t_max
                   - bound on distance of successive t-values

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_sseg_tmesh_constraintfunc_t1(double *ftk,
				      struct mesh *m, FIDX seg, 
				      double tk, double c_curve
/* evaluates the constraint function 

     "f(tk)=4*c_curve*r(tk)", 
  
   at the given t-value tk, where r(t) is the local curvature radius
   of the spline
   

   OUTPUT: ftk	   - value of the constraint function f(tk)

   INPUT:  seg     - the shape segment
	   m	   - the mesh 
	   tk	   - t-value for the bezier spline
           c_curve - constant as described above
			 
   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_vertices ( struct mesh *m, FIDX need
/* increases the number of vertices that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional vertices that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_edges ( struct mesh *m , FIDX need
/* increases the number of edges that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional edges that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_faces ( struct mesh *m , FIDX need
/* increases the number of faces that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional faces that are
                        required
   In/Out: m          - the mesh 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_volumes ( struct mesh *m , FIDX need
/* increases the number of volumes that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional volumes that are
                        required
   In/Out: m          - the mesh 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_elems ( struct mesh *m , FIDX need
/* increases the number of elems that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional elems that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_boundary ( struct mesh *m , FIDX need
/* increases the number of boundary entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional boundary entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_pcsurf ( struct mesh *m , FIDX need
/* increases the number of pcsurf entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional pcsurf entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_pcvol ( struct mesh *m , FIDX need
/* increases the number of pcvol entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional pcvol entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_hierarchy ( struct mesh *m , FIDX need
/* increases the number of hierarchy entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional hierarchy entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_elhiers ( struct mesh *m , FIDX need
/* increases the number of elementwise hierarchy entries that can be
   held in the mesh by reallocating memory

   Input:  need       - the number of additional elementwise hierarchy
                        entries that are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_more_st ( struct mesh *m, FIDX need
/* increases the number of t-values for the shape segmants that can 
   be held in the mesh by reallocating memory

   Input:  need       - the number of additional vertices that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

void mesh_free ( struct mesh *m
/* frees the memory internally allocated in the mesh data struct */);

void ilist_free( struct ilist **head
/* empties ilist (deletes all entries and defines head to be NULL */);

int ilist_sorted_insert( struct ilist **head, FIDX newname
/* if the entry newname does not exist in the sorted list, it is
   inserted at the appropriate position */);

int comp_intdouble_d(const void *a1, const void *b1
/* helper function for qsort, compares the double part of the
   intdouble struct, designed for descending order

   Input:  a1      - pointer to intdouble a
           b1      - pointer to intdouble b

   Return: -1   if a.d >  b.d
            0   if a.d == b.d
            1   if a.d <  b.d
*/);

int comp_intdouble_i(const void *a1, const void *b1
/* helper function for qsort, compares the FIDX part of the
   intdouble struct, designed for ascending order

   Input:  a1      - pointer to intdouble a
           b1      - pointer to intdouble b

   Return: -1   if a.i <  b.i
            0   if a.i == b.i
            1   if a.i >  b.i
*/);

int mesh_write_solution_grape_t2(struct mesh *m,struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *pathname
/* writes the mesh and the solution into a GRAPE readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   pathname  - pathname for the grape file post0.dat


   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_spline_t_init_tx(struct mesh *m, int type
/* creates st (t-values (parameter of spline segment)
   for the shape parameters) for the nodes by distributing the
   t-values uniformly on [0,1] has to be applied before the
   adjust_sseg, 
   (so even before the mesh generator) 

   Input/Output 
           m       - the mesh, m.st will be generated

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_spline_t_fix_new_nodes(struct mesh *m, int type
/* After the mesh generator, there may be new nodes on the boundary
   without a valid t-value for the shape segments. The m.st for these 
   nodes will be set.


  Input/Output 
          m        - the mesh, m.st_t will be adjusted

  Input:  type     - type of the mesh

  Return: SUCCESS  - success
          FAIL     - failure, see error message, output will not be
                     valid
*/);

int mesh_spline_set_t( struct mesh *m, FIDX seg, FIDX n1, FIDX n2, FIDX nm,
		       int adjust, int type
/* determine the t-value for the bezier spline for the midnode of an edge
   by bisection to make sure that the node is near the mid (so the 
   interpolation/restriction will work better). The node position can be also
   adjusted

   INPUT/Output:
	    m		- the mesh: m.st[nm] and 
	                  (optional) m.vertex[nm] will be set
   INPUT:   seg	        - the shape segment
	    n1,
	    n2     	- start and end node of the edge under consideration
	    nm		- the midnode
	    adjust	- the vertex entry will be adjusted if adjust==1
			 
   Return:  SUCCESS 	- success
            FAIL   	- failure, see error message, output will not be
                          valid
*/);

int mesh_seg_node_eval( struct mesh *m, FIDX seg, 
			double t, double *x
/* evaluates the position of a point on the shape segment according to
   the given t-value

   INPUT:  seg     - the shape segment
	   m	   - the mesh 
	   t	   - t-value for the bezier spline
   OUTPUT: x	   - vector holding the coordinates
			 
   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
