/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
 * =====================================================================
 *
 *       Filename:  linsolve_umfpack.c
 *
 *    Description:  solve a linear system with UMFPACK  
 *
 *        Version:  1.0
 *        Created:  23.11.2009 10:44:26
 *
 *    Main-Author:  Martin Köhler, martin.koehler@s2005.tu-chemnitz.de
 *        Company:  TU-Chemnitz, MiIT
 *
 * =====================================================================
 */

/*
FILE linsolve_umfpack.c
HEADER linsolve_umfpack.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "sparse_struct.h"
#include "linsolve_struct.h"

*/


/* prototypes of external functions */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/* function prototypes */
#include "linsolve_umfpack.h"
#include "sparse.h"


/* locally used datatypes */
#include "datastruc.h"

/* try to remove again: */
#include "mesh.h"

#include "config.h"

#ifdef HAVE_UMFPACK
#include <umfpack.h>
#endif

/*FUNCTION*/
int linsolver_init(struct linsolver *S
/* initialises S to be empty, such that it can be emptied or deleted without
   problems 
   
   Input:  (none)

   Output: S       - (S is given by reference), is initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		   ){
  
  (*S).type      = LINSOLVER_TYPE_UNDEF;
  (*S).symbolic  = NULL;
  (*S).numeric   = NULL;

  return SUCCESS;
}

/*FUNCTION*/
int linsolver_free(struct linsolver *S  
/* frees memory in S 
   
   Input:  (none)

   Output: S       - (S is given by reference), is emptied/freed

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/

		   ){
#ifdef HAVE_UMFPACK
  if (S->type == LINSOLVER_TYPE_UMFPACK )
    {
      if ( sizeof (FIDX) == sizeof(long) )
	{
	  umfpack_dl_free_numeric(&(S->numeric));
	  umfpack_dl_free_symbolic(&(S->symbolic));
	} 
      else 
	{
	  umfpack_di_free_numeric(&(S->numeric));
	  umfpack_di_free_symbolic(&(S->symbolic));
	}
      S->type =LINSOLVER_TYPE_UNDEF;
    }
  else
    {
      fprintf(stderr,"linsolver_init: unknown solver type\n");
      return FAIL;
    }
  return SUCCESS;
#else
  fprintf(stderr,"linsolver_init: compiled without UMFPACK support\n");
  return FAIL;
#endif	
}

/*FUNCTION*/
int UMFPACK_factorize (struct sparse *A, struct linsolver *sol
/* calls UMFPACK to factorise the sparse matrix A, such equation
   systems with A can be solved by UMFPACK_solve 
   
   Input:  A       - sparse matrix, must be in compressed row storage,
                     and must be invertible

   Output: sol     - (sol is given by reference), 
                     sparse triangulation of A is computed and stored
                     in sol, neccessary fields in sol are allocated,

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		       ) {
#ifdef HAVE_UMFPACK
  int ret;
	
  if ( A->type != SP_TYPE_COMPROW ){
    fprintf(stderr, "UMFPACK_factorize: "
	    "matrix isn't in compressed row storage\n");
    return FAIL; 
  }
#ifdef FEINS_have_warning 
#warning "One of the two calls to UMFPACK will always produce a warning of" 
#warning "     -passing argument of incompatible pointer type-"
#warning "because there is one version for -long int- and one for -int-"
#warning "but only the correct one for the currently used _FIDX_ will be used"
#warning "so it is save to ignore the warning" 
#endif
  if ( sizeof(FIDX) == sizeof ( long )) 
    {
      umfpack_dl_defaults(sol->control);
      /* turn of iterative refinement */
      sol->control [ UMFPACK_IRSTEP] = 0;
      /* UMFPACK uses compressed column storage, by using our
	 compressed row storage, a LU-factorisation of A^T is computed */
      ret = umfpack_dl_symbolic(A->row_nr, A->row_nr, (long *)A->rows, 
				(long*) A->cols, A->A, &(sol->symbolic),
				sol->control, sol->info);
    if ( ret != UMFPACK_OK )
      {
	fprintf(stderr, "UMFPACK_factorize: "
		"UMFPACK dl symbolic phase returned with error ret=%d.\n",ret);
	return FAIL;
      }
    ret = umfpack_dl_numeric(A->rows, A->cols, A->A, 
			     sol->symbolic, &(sol->numeric),
			     sol->control, sol->info);
    if ( ret != UMFPACK_OK)
      {
	fprintf(stderr, "UMFPACK_factorize: "
		"UMFPACK dl numeric phase returned with error.\n");
	return FAIL;
      }	

    }
  else 
    {
      umfpack_di_defaults(sol->control);
      // turn of iterative refinement
      sol->control [ UMFPACK_IRSTEP] = 0;
      /* UMFPACK uses compressed column storage, by using our
	 compressed row storage, a LU-factorisation of A^T is computed */
      ret = umfpack_di_symbolic(A->row_nr, A->row_nr, (int*) A->rows, (int*)A->cols, A->A, &(sol->symbolic), sol->control, sol->info);
      if ( ret != UMFPACK_OK ) 
	{
	  fprintf(stderr, "UMFPACK_factorize: "
		  "UMFPACK di symbolic phase returned with error.\n");
	  return FAIL;
	}
      ret = umfpack_di_numeric(A->rows, A->cols, A->A,
			       sol->symbolic, &(sol->numeric),
			       sol->control, sol->info);
      if ( ret != UMFPACK_OK) 
	{
	  fprintf(stderr, "UMFPACK_factorize: "
		  "UMFPACK di numeric phase returned with error.\n");
	  return FAIL;
	}	
    }
  sol->type = LINSOLVER_TYPE_UMFPACK;
  sol->n    = A->row_nr;
  return SUCCESS;
#else
  fprintf(stderr,"UMFPACK_factorize: compiled without UMFPACK support\n");
  return FAIL;
#endif
}

/*FUNCTION*/
int UMFPACK_solve(struct linsolver *sol, struct vector *b, struct vector *x
/* calls UMFPACK to solve a linear system with a previously by
   UMFPACK_factorize factorised sparse matrix
         A x = b
   
   Input:  sol     - sparse triangulation of A as computed by
                     UMFPACK_factorize 
	   b       - right-hand-side vector


   Output: x       - (x is given by reference), 
                     solution vector x of Ax=b, has to have right size
                     beforehand

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		  ){
#ifdef HAVE_UMFPACK
  int ret;
  if ( sol->type != LINSOLVER_TYPE_UMFPACK )
    {
      fprintf(stderr, "UMFPACK_solve: solver is not created by umfpack\n");
      return FAIL;
    }

  if ( sol->n != x->len) 
    {
      fprintf(stderr,"UMFPACK_solve: output vector has the wrong size\n");
      return FAIL;
    }
  if ( sol->n != b->len) 
    {
      fprintf(stderr, "UMFPACK_solve: input vector has the wrong size\n");
      return FAIL;
    }
  if (sizeof(FIDX) == sizeof(long)) 
    {
      ret = umfpack_dl_solve(UMFPACK_At, NULL, NULL, NULL, x->V, b->V,
			     sol->numeric, sol->control, sol->info);
      if ( ret != UMFPACK_OK )
	{
	  fprintf(stderr, "UMFPACK_solve: "
		  "UMFPACK linear system isn't solved successfully\n");
	  return FAIL;
	}
    }
  else 
    {
      /* UMFPACK uses compressed column storage, by using our
	 compressed row storage, a LU-factorisation of A^T is computed */
      ret = umfpack_di_solve(UMFPACK_At, NULL, NULL, NULL, x->V, b->V,
			     sol->numeric, sol->control, sol->info);
      if ( ret != UMFPACK_OK )
	{
	  fprintf(stderr, "UMFPACK_solve: "
		  "UMFPACK linear system isn't solved successfully\n");
	  return FAIL;
	}
    }
  return SUCCESS;
#else
  fprintf(stderr,"UMFPACK_solve: compiled without UMFPACK support\n");
  return FAIL;
#endif
}

/*FUNCTION*/
int sparse_solve_UMFPACK(struct vector *x,
			 struct sparse *K, struct vector *rhs, 
			 struct projector1 *P
/* wrapper to solve the linear system
     K*x=rhs
   with Dirichlet conditions as defined by the projector P

   In/Out: x       - solution vector, the values at the Dirichlet DOFs
                     as defined by the projector P are left unchanged,
                     the rest is set to the solution of the system
                     under the Dirichlet conditions 
	   K       - Stiffness matrix, rows and columns corresponding
                     to Dirichlet DOFs (see projector P) are modified
                     (for simplicity of implementation),
		     K must have symmetric connectivity structure, but
		     entries may be non-symmetric, ideally K should be
		     still in flexible storage format,
	   rhs     - right-hand-side vector, is modified to implement
                     Dirichlet conditions (for simplicity of
                     implementation) 


   Input:  P       - projector defining the Dirichlet DOFs

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
			 ){
  FIDX *isdiri;
  int err;
  struct vector help;
  struct linsolver umfpackdata;

  /* init and solve */
  err=sparse_solve_UMFPACK_reuse(x, K, rhs,P, &umfpackdata,
				 &isdiri, &help, 1);
  FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK_reuse, sparse_solve_UMFPACK);  

  /* clean up */
  err=sparse_solve_UMFPACK_reuse(x, K, rhs,P, &umfpackdata,
				 &isdiri, &help, 9);
  FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK_reuse, sparse_solve_UMFPACK);  

  return SUCCESS;
}


/*FUNCTION*/
int sparse_solve_UMFPACK_reuse(struct vector *x,
			       struct sparse *K, 
			       struct vector *rhs, 
			       struct projector1 *P,
			       struct linsolver *umfpackdata,
			       FIDX **isdiri,
			       struct vector *help, 
			       int task
/* wrapper to solve the linear system
     K*x=rhs
   with Dirichlet conditions as defined by the projector P, and allow
   reuse of the triangular factors of K in case multiple systems with
   identical K have to be solved

   In/Out: x       - solution vector, the values at the Dirichlet DOFs
                     as defined by the projector P are left unchanged,
                     the rest is set to the solution of the system
                     under the Dirichlet conditions 
	   rhs     - right-hand-side vector, is modified to implement
                     Dirichlet conditions (for simplicity of
                     implementation) 
           umfpackdata
                   - internal data for UMFPACK linear solver, 
		     see description of task for input/output behaviour
           isdiri  - internal data specifying if nodes are Dirichlet, 
		     see description of task for input/output behaviour
           help    - internaly used vector, 
		     see description of task for input/output behaviour


   Input:  P       - projector defining the Dirichlet DOFs,
	   K       - Stiffness matrix, 
		     K must have symmetric connectivity structure, but
		     entries may be non-symmetric
           task    - specify task:
                     task==1 => initial solve, umfpackdata, isdiri and
                                help are created
                     task==2 => follow up solve, umfpackdata, isdiri and
                                help are used as provided 
                     task==9 => finish, umfpackdata, isdiri and help are
                                destroyed, memory is freed

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
			 ){
  FIDX ndofs, i;
  int err;
  FIDX   *tmp_cols;
  double *tmp_A;



  if (task==9)
    {
      /* clean up */
      err=linsolver_free(umfpackdata);
      FUNCTION_FAILURE_HANDLE( err, linsolver_free, sparse_solve_UMFPACK_reuse);  

      free(*isdiri);
      
      vector_free(help);

      return SUCCESS;
    }

  ndofs=x->len;

  if (task==1)
    { 
      /* initialise the solver */
      err=vector_alloc(help, ndofs);
      FUNCTION_FAILURE_HANDLE(err, vector_alloc, sparse_solve_UMFPACK_reuse);

      /* create the field isdiri which defines if an entry is
	 Dirichlet DOF or not */
      TRY_MALLOC( *isdiri, ndofs, FIDX, sparse_solve_UMFPACK_reuse);
      for (i=0; i<ndofs; i++) 
	{
	  (*isdiri)[i]=0;
	}
      for (i=0; i<P->len; i++)
	{
	  (*isdiri)[ P->V[i] ] =1;
	}
    }


  /* the remaining stuff is only task==1 and task==2 */

  /* if K is still provided, modify rhs to account for Dirichlet data */
  if (K!=NULL)
    {
      /* set all non-Dirichlet components in x to zero */
      for (i=0; i<ndofs; i++) 
	{
	  if ((*isdiri)[i]==0)
	    {
	      x->V[i]=0.0;
	    }
	}

      /* multiply K*x, and compute rhs=rhs-K*x */
      err=sparse_mul_mat_vec( K, x, help);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec,
			       sparse_solve_UMFPACK_reuse);  

      for (i=0; i<ndofs; i++)
	{
	  rhs->V[i] -= help->V[i];
	}
    } /* end if K!=NULL */

  /* project rhs values at dirichlet dofs */
  for (i=0; i<P->len; i++)
    {
      rhs->V[ P->V[i] ]=0.0;
    }


  if (task==1)
    { 
      /* now the Dirichlet rows and columns of K are replaced by unity */
      /*  if ( (*K).type == SP_TYPE_COMPROW )
	  {
	  fprintf(stderr,"sparse_solve_UMFPACK_reuse: "
	  "warning, called with matrix of type COMPROW,\n"
	  "this will be less efficient, as no entries can be removed\n"
	  "try to use type FLEX\n\n\n");
	  }
	  /* */
      /* copy initial data of K to temporary arrays, to allow 
	 modification and restoration after that */
      if ((*K).type != SP_TYPE_COMPROW)
	{
	  fprintf(stderr,"sparse_solve_UMFPACK_reuse: "
		  "assume COMPROW here "
		  "-> error\n");
	  return FAIL;
	}
      {
	FIDX nrdata=(*K).rows[(*K).row_nr];
	TRY_MALLOC(tmp_cols, nrdata, FIDX, sparse_solve_UMFPACK_reuse);
	TRY_MALLOC(tmp_A, nrdata, double, sparse_solve_UMFPACK_reuse);
	for (i=0; i<nrdata; i++)
	  {
	    tmp_cols[i]= (*K).cols[i];
	    tmp_A[i]= (*K).A[i];
	  }
      }


      for (i=0; i<P->len; i++)
	{
	  FIDX dirdof= P->V[i];

	  // fprintf(stderr,"debug: dirdof[%3d]=%6d\n", i, dirdof);

	  switch( (*K).type)
	    {
	    case SP_TYPE_FLEX:
	      {
		FIDX j;

		fprintf(stderr,"sparse_solve_UMFPACK_reuse: "
			"FLEX not repaired after last change of FLEX format "
			"-> error\n");
		return FAIL;
	    
		/* set the of-diagonal-entries of the column to zero first, 
		   using the row 
		   (flex_row_cols[row][0] is the diagonal entry) */ 
		/*for (j=1; j<(*K).flex_row_col_nr[dirdof]; j++)
		  {
		    FIDX col;
		    double *entry;
		
		    col = (*K).flex_row_cols[dirdof][j];
		    /* delete the dirdof column in row col */
		    /*err=sparse_get_entry( K, col, dirdof, 0, &entry);
		    if (err=FAIL)
		      {
			fprintf(stderr,"sparse_solve_UMFPACK_reuse: (FLEX)"
				"connectivity of matrix not symmetric -> error\n");
			return FAIL;
		      }
		    (*entry)=0;
		  }
		/* now delete of-diagonals of the row itself */
		/*(*K).flex_row_col_nr[dirdof] = 1;
		/* set diagonal entry to unity */
		/*(*K).flex_row_cols[dirdof][0] = dirdof;
		  (*K).flex_row_data[dirdof][0] = 1.0;*/
	      }
	      break;
	    case SP_TYPE_COMPROW:
	      {
		FIDX m, j;
	    
		m = (*K).rows[dirdof+1];
		for (j=(*K).rows[dirdof]; j<m; j++)
		  {
		    FIDX col=(*K).cols[j];
		    if ( col == dirdof )
		      {
			(*K).A[j]=1.0;
		      }
		    else
		      {
			FIDX k, endrow;
			int found=0;
			/* delete this entry here */
			(*K).A[j]=0.0;
			/* and delete dirdof column in row col */
			endrow = (*K).rows[col+1];
			for (k=(*K).rows[col]; k<endrow; k++)
			  {
			    if ( (*K).cols[k] == dirdof )
			      {
				(*K).A[k]=0.0;
				found=1;
			      }
			  }
			if (found==0)
			  {
			    fprintf(stderr,"sparse_solve_UMFPACK_reuse: (COMPROW)"
				    "connectivity of matrix not symmetric "
				    "-> error\n");
			    return FAIL;
			  }
		      }
		  }
	      }
	      break;
	    default:
	      fprintf(stderr, "sparse_solve_UMFPACK_reuse: invalid type!\n");
	      return FAIL;
	    } /* end switch */
      
	} /* end loop over dirdof=P.V[i] */
      /* so the Dirichlet rows and columns of K have been replaced by unity */
  

      /* sort columns in each row of K */
      for (i=0; i<ndofs; i++)
	{
	  FIDX j, rowlen;
	  struct intdouble *sorter;

	  if ((*K).type != SP_TYPE_COMPROW)
	    {
	      fprintf(stderr,"sparse_solve_UMFPACK_reuse: "
		      "assume COMPROW here "
		      "-> error\n");
	      return FAIL;
	    }


	  rowlen = (*K).rows[i+1]-(*K).rows[i];
	  TRY_MALLOC( sorter, rowlen, struct intdouble, sparse_solve_UMFPACK_reuse);

	  for (j=0; j<rowlen; j++)
	    {
	      sorter[j].i = (*K).cols[(*K).rows[i]+j];
	      sorter[j].d = (*K).A[   (*K).rows[i]+j];
	    }

	  /* sort */
	  qsort( sorter, rowlen, sizeof(struct intdouble), comp_intdouble_i );      


	  for (j=0; j<rowlen; j++)
	    {
	      (*K).cols[(*K).rows[i]+j] = sorter[j].i;
	      (*K).A[   (*K).rows[i]+j] = sorter[j].d;
	    }

	  free(sorter);
	}


      /* err=sparse_mat_write_file( K, "visual/UMFPACK_testMAT.txt");
	 FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file,
	 sparse_solve_UMFPACK_reuse); /* */   

      /* factorize the matrix K */
      err=linsolver_init(umfpackdata);
      FUNCTION_FAILURE_HANDLE( err, linsolver_init, sparse_solve_UMFPACK_reuse);  
      err=UMFPACK_factorize (K, umfpackdata );
      FUNCTION_FAILURE_HANDLE( err, UMFPACK_factorize, sparse_solve_UMFPACK_reuse);  

      /* restore K to undo modifications */
      {
	FIDX nrdata=(*K).rows[(*K).row_nr];
	for (i=0; i<nrdata; i++)
	  {
	    (*K).cols[i] = tmp_cols[i];
	    (*K).A[i] = tmp_A[i];
	  }
	free(tmp_A);
	free(tmp_cols);
      }

    } /* end if task==1 */

  /* solve with the factors of K, solution in help */
  err=UMFPACK_solve(umfpackdata, rhs, help);
  FUNCTION_FAILURE_HANDLE( err, UMFPACK_solve, sparse_solve_UMFPACK_reuse);  

  /* update all non-Dirichlet DOFs to the solution values stored in
     help */
  for (i=0; i<ndofs; i++) 
    {
      if ((*isdiri)[i]==0)
	{
	  x->V[i] = help->V[i];
	}
    }

  return SUCCESS;
}

  

  
