/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE linsolve_hypre_AMG.c
HEADER linsolve_hypre_AMG.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "sparse_struct.h"

*/


/* prototypes of external functions */
#include <math.h>
#include "feins_lapack.h"

/* function prototypes */
#include "linsolve_hypre_AMG.h"


#ifdef HAVE_HYPRE_NO_MPI
  /* we took the code of example ex5.c of hypre to modify it to our needs */

#include "_hypre_utilities.h"
#include "HYPRE_krylov.h"
#include "HYPRE.h"
#include "HYPRE_parcsr_ls.h"

struct FEINS_HYPRE_data {
  HYPRE_IJMatrix A;            /* the matrix in HYPRE format */
  HYPRE_ParCSRMatrix parcsr_A; /* the matrix in HYPRE CSR format */
  HYPRE_Solver precond;        /* the data structure for the preconditioner */
  HYPRE_IJVector b;            /* rhs vector in HYPRE format */
  HYPRE_ParVector par_b;       /* other version of rhs */
  HYPRE_IJVector x;            /* solution vector in HYPRE format */
  HYPRE_ParVector par_x;       /* other version of solution vector */

  int Flen;         /* size of the original matrix (FEINS) */
  int Hlen;         /* size of the HYPRE matrix */
  int *hypre_DOFS;  /* array of size len, hypre_DOFS[i]=j means FEINS
		       DOF i is DOF j of the hypre matrix */
  int *feins_DOFS;  /* array of size len, feins_DOFS[i]=j means HYPRE
		       DOF i is DOF j of FEINS */
  int *rows;        /* vector of indices 0..Hlen */
  double *vals;     /* vector same size as rows, used to copy data
		       into/out of HYPRE vectors */
};
#endif

/*FUNCTION*/
int gen_hypre_AMG_setup(struct sparse *K, 
			FIDX nr_diri, FIDX *diris, int diri_copy_dims,
			void **AMGdata, int symm
/* setup of hypre AMG solver for sparse matrix problem
   
   Input:  K       - the stiffness matrix
           nr_diri - number of degrees of freedom (DOFs) which are
                     subject to Dirichlet boundary conditions, such
                     that the according rows and columns of the
                     matrix have to be deleted 
           diris   - vector of length nr_diri, stating which DOFs
                     these are
           diri_copy_dims
                   - if all dofs are specified in diris, 
		     this should be set to 1, 
		     if only the node numbers are specified, and a
		     multidimensional quantity is used, this may be
		     set to dim, then vx_nr=K.row_nr/diri_copy_dims
		     and for i=0<nr_diri, for d=0<diri_copy_dims 
		       DOF=diris[i]+d*vx_nr
		     is also defined to be Dirichlet DOF
           symm    - if preco should use symmetric smoothers
                     ==0 use simple GS, ==1 use symmetric GS

   Output: AMGdata - (given by reference) on output it will contain
                     the data for the AMG solver/preconditioner

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
			){
#ifndef HAVE_HYPRE_NO_MPI
  fprintf(stderr, "gen_hypre_AMG_setup: compiled without hypre support!\n");
  return FAIL;  
#else
  FIDX i, nfeins, vx_nr;
  int d;
  int myid, num_procs;

  /*  FIDX sys_nr=2, sys_vx_nr; */

  int ilower, iupper, local_size;
  int nondirs;
  int *cols;
  double *vals;
  
  struct FEINS_HYPRE_data *Hdat;

  if (K->type!=SP_TYPE_COMPROW)
    {
      fprintf(stderr, "gen_hypre_AMG_setup: require matrix of type comprow!\n");
      return FAIL;
    }

  if (diri_copy_dims<=0)
    {
      fprintf(stderr, "gen_hypre_AMG_setup: diri_copy_dims makes no sense!\n");
      return FAIL;
    }


  TRY_MALLOC(Hdat, 1, struct FEINS_HYPRE_data, gen_hypre_AMG_setup);

  /* Initialize MPI */
  {
    int argc=0; 
    char **argv={NULL};
    MPI_Init(&argc, &argv);
  }
  MPI_Comm_rank(MPI_COMM_WORLD, &myid);
  MPI_Comm_size(MPI_COMM_WORLD, &num_procs);

  if (num_procs!=1)
    {
      fprintf(stderr, "gen_hypre_AMG_setup: only one processor for now!\n");
      MPI_Finalize();
      return FAIL;
    }

  /* prepare dofs association */
  nfeins = K->row_nr;
  /* sys_vx_nr=nfeins/sys_nr; */
  TRY_MALLOC(Hdat->hypre_DOFS, nfeins, int, gen_hypre_AMG_setup);
  TRY_MALLOC(Hdat->feins_DOFS, nfeins, int, gen_hypre_AMG_setup);

  /* first init to zero */
  for (i=0; i<nfeins; i++)
    {
      Hdat->hypre_DOFS[i]=0;
    }
  /* mark all dirichlet dofs */
  vx_nr=nfeins/diri_copy_dims;
  for (d=0; d<diri_copy_dims; d++)
    for (i=0; i<nr_diri; i++)
      {
	Hdat->hypre_DOFS[diris[i]+d*vx_nr]=1;
      }

  /* now build the renumbering vectors */
  nondirs=0;
  for (i=0; i<nfeins; i++)
    {
      if (Hdat->hypre_DOFS[i]==0)
	{
	  /* is a non-dirichlet DOF */
	  Hdat->hypre_DOFS[i]=nondirs;
	  Hdat->feins_DOFS[nondirs]=i;
	  nondirs++;
	}
      else
	{
	  /* is a dirichlet DOF */
	  Hdat->hypre_DOFS[i]=-1;
	}
    }
  Hdat->Hlen=nondirs;
  Hdat->Flen=nfeins;

  /* init some temporary storage */
  TRY_MALLOC(cols, nondirs, int, gen_hypre_AMG_setup);
  TRY_MALLOC(vals, nondirs, double, gen_hypre_AMG_setup);

  /* set the hypre matrix up */
  ilower = 0;
  iupper = nondirs-1;
  /* How many rows do I have? */
  local_size = iupper - ilower + 1;

  /* Create the matrix.
     Note that this is a square matrix, so we indicate the row partition
     size twice (since number of rows = number of cols) */
  HYPRE_IJMatrixCreate(MPI_COMM_WORLD, ilower, iupper, ilower, iupper, &(Hdat->A));

  /* Choose a parallel csr format storage (see the Hypre User's Manual) */
  HYPRE_IJMatrixSetObjectType(Hdat->A, HYPRE_PARCSR);

  /* Initialize before setting coefficients */
  HYPRE_IJMatrixInitialize(Hdat->A);

  


  /* set one row at a time */
  for (i = 0; i < nondirs; i++)
    {
      int row_nnz;
      int int_i=i;
      FIDX j;
      FIDX feinsRow;

      feinsRow=Hdat->feins_DOFS[i];
      row_nnz = 0;

      for (j=K->rows[feinsRow]; j<K->rows[feinsRow+1]; j++)
	{
	  FIDX feinsCol=K->cols[j];
	  int hypreCol=Hdat->hypre_DOFS[feinsCol];
	  if ( (hypreCol>=0)
	       /* && (feinsRow/sys_vx_nr == feinsCol/sys_vx_nr) */
	       )
	    {
	      cols[row_nnz]=hypreCol;
	      vals[row_nnz]=K->A[j];
	      row_nnz++;
	    }
	}

      /* Set the values for row i */
      HYPRE_IJMatrixSetValues(Hdat->A, 1, &row_nnz, &int_i, cols, vals);

    } /* for i rows */

  /* Assemble after setting the coefficients */
  HYPRE_IJMatrixAssemble(Hdat->A);

  /* Get the parcsr matrix object to use */
  HYPRE_IJMatrixGetObject(Hdat->A, (void**) &(Hdat->parcsr_A));

  /* Create the rhs and solution */
  HYPRE_IJVectorCreate(MPI_COMM_WORLD, ilower, iupper,&(Hdat->b));
  HYPRE_IJVectorSetObjectType(Hdat->b, HYPRE_PARCSR);
  HYPRE_IJVectorInitialize(Hdat->b);

  HYPRE_IJVectorCreate(MPI_COMM_WORLD, ilower, iupper,&(Hdat->x));
  HYPRE_IJVectorSetObjectType(Hdat->x, HYPRE_PARCSR);
  HYPRE_IJVectorInitialize(Hdat->x);

  /* init the rhs and solution to zero */
  for (i = 0; i < nondirs; i++)
    {
      cols[i]=i;
      vals[i]=0.0;
    }
  Hdat->rows=cols;
  Hdat->vals=vals;

  HYPRE_IJVectorSetValues(Hdat->b, local_size, Hdat->rows, Hdat->vals);
  HYPRE_IJVectorAssemble(Hdat->b);
  HYPRE_IJVectorGetObject(Hdat->b, (void **) &(Hdat->par_b));
  HYPRE_IJVectorSetValues(Hdat->x, local_size, Hdat->rows, Hdat->vals);
  HYPRE_IJVectorAssemble(Hdat->x);
  HYPRE_IJVectorGetObject(Hdat->x, (void **) &(Hdat->par_x));


  /* Create solver/preconditioner */
  HYPRE_BoomerAMGCreate(&(Hdat->precond));

  /* Set some parameters (See Reference Manual for more parameters) */
  HYPRE_BoomerAMGSetPrintLevel(Hdat->precond, 0);  /* 0: quiet, 1: solve info */
  /*HYPRE_BoomerAMGSetPrintLevel(Hdat->precond, 3);  /* 3: print solve info + parameters */
  /*HYPRE_BoomerAMGSetStrongThreshold(Hdat->precond, 0.25); /* --->
			    for 2D Laplace 0.25 (default)
			    for 3D Laplace 0.5--0.6
			    for Lame 0.9 or higher   */
  HYPRE_BoomerAMGSetCoarsenType(Hdat->precond, 6); /* Falgout coarsening */
  /*HYPRE_BoomerAMGSetCoarsenType(Hdat->precond, 21); /* 21 CGC, 22-CGC-E coarsening */
  HYPRE_BoomerAMGSetNumSweeps(Hdat->precond, 1);   /* Sweeeps on each level */
  HYPRE_BoomerAMGSetMaxLevels(Hdat->precond, 20);  /* maximum number of levels */
  HYPRE_BoomerAMGSetTol(Hdat->precond, 0.0);       /* conv. tolerance zero */
  /*HYPRE_BoomerAMGSetMaxIter(Hdat->precond, 1);     /* do only one iteration! */
  HYPRE_BoomerAMGSetMaxIter(Hdat->precond, 4);     /* four iterations */
  /*HYPRE_BoomerAMGSetCycleType(Hdat->precond, 1);     /* 1 V-cycle, 2-W-cycle */

  if (symm==0)
    {
      HYPRE_BoomerAMGSetRelaxType(Hdat->precond, 3);   /* G-S/Jacobi
							  hybrid
							  relaxation
							  */
    }
  else
    {
      HYPRE_BoomerAMGSetRelaxType(Hdat->precond, 6);   /* Sym
							  G.S./Jacobi
							  hybrid  */ 
    }


  /* AMG preconditioner  parameters
     HYPRE_BoomerAMGCreate(&precond);
     HYPRE_BoomerAMGSetPrintLevel(precond, 1); /* print amg solution info 
     HYPRE_BoomerAMGSetCoarsenType(precond, 6);
     HYPRE_BoomerAMGSetRelaxType(precond, 6); /* Sym G.S./Jacobi hybrid 
     HYPRE_BoomerAMGSetNumSweeps(precond, 1);
     HYPRE_BoomerAMGSetTol(precond, 0.0); /* conv. tolerance zero 
     HYPRE_BoomerAMGSetMaxIter(precond, 1); /* do only one iteration! */

  /*{
    int *eqIdx;
    int sysNr=3;
    FIDX vx_nr=nfeins/sysNr;
    TRY_MALLOC(eqIdx, nondirs, int,  gen_hypre_AMG_setup);
    for (i = 0; i < nondirs; i++)
      {
	int func;
	FIDX feinsDof;

	feinsDof=Hdat->feins_DOFS[i];
	func=feinsDof/vx_nr;
	eqIdx[i]=func;
	printf("DEBUG eq[%4d]=%d\n",i,func);
      }
    HYPRE_BoomerAMGSetNodal(Hdat->precond, 1);
    HYPRE_BoomerAMGSetNumFunctions(Hdat->precond, sysNr);
    HYPRE_BoomerAMGSetDofFunc(Hdat->precond, eqIdx);
    free(eqIdx);
    }/**/



  /* Now setup the preconditioner */
  HYPRE_BoomerAMGSetup(Hdat->precond, Hdat->parcsr_A, Hdat->par_b, Hdat->par_x);


  *AMGdata = Hdat;
  
  return SUCCESS;
#endif
}

/*FUNCTION*/
int gen_hypre_AMG_preco(void *notused, struct vector *in,
			void *arg3, struct vector *out
/* performs
     out = C^-1*in
   where C^-1 is AMG multigrid operator as defined in
   gen_hypre_AMG_setup, 

   Input:  notused - well, it is not used but included in the
                     interface to allow this function to be used as a
                     preconditioner,
           in      - input vector
	   arg3=
	   Hdat    - AMGdata as returned by gen_hypre_AMG_setup,

   Output: out    - (given by reference), the C^-1*in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
			){
#ifndef HAVE_HYPRE_NO_MPI
  fprintf(stderr, "gen_hypre_AMG_preco: compiled without hypre support!\n");
  return FAIL;  
#else
  struct FEINS_HYPRE_data *Hdat=arg3;
  int nondirs=Hdat->Hlen;
  int i, ilower, iupper, local_size;

#warning "only initial debug: H_final_relative_res, H_iter "
  double H_final_relative_res; int H_iter;

  if ((in->len!=Hdat->Flen)||(out->len!=Hdat->Flen))
    {
      fprintf(stderr, "gen_hypre_AMG_preco: vector lengths don't match AMGdata!\n");
      fprintf(stderr, " in.len=%"dFIDX"  out.len=%"dFIDX"  Hdat.Flen=%d\n",
	      in->len, out->len, Hdat->Flen);
      MPI_Finalize();
      return FAIL;
    }

  /* set the hypre matrix up */
  ilower = 0;
  iupper = nondirs-1;
  /* How many rows do I have? */
  local_size = iupper - ilower + 1;

  /* copy the non-dirichlet dofs to the RHS for HYPRE */
  HYPRE_IJVectorInitialize(Hdat->b);
  for (i = 0; i < nondirs; i++)
    {
      Hdat->vals[i]=in->V[Hdat->feins_DOFS[i]];
    }
  HYPRE_IJVectorSetValues(Hdat->b, local_size, Hdat->rows, Hdat->vals);
  HYPRE_IJVectorAssemble(Hdat->b);
  HYPRE_IJVectorGetObject(Hdat->b, (void **) &(Hdat->par_b));

  /* init x to zero */
  HYPRE_IJVectorInitialize(Hdat->x);
  for (i = 0; i < nondirs; i++)
    {
      Hdat->vals[i]=0.0;
    }
  HYPRE_IJVectorSetValues(Hdat->x, local_size, Hdat->rows, Hdat->vals);
  HYPRE_IJVectorAssemble(Hdat->x);
  HYPRE_IJVectorGetObject(Hdat->x, (void **) &(Hdat->par_x));


  /* call the HYPRE solver to effect C^-1 */
  HYPRE_BoomerAMGSolve(Hdat->precond, Hdat->parcsr_A, Hdat->par_b, Hdat->par_x);

  /* Run info - needed logging turned on */
  /*
  HYPRE_BoomerAMGGetNumIterations(Hdat->precond, &H_iter);
  HYPRE_BoomerAMGGetFinalRelativeResidualNorm(Hdat->precond, &H_final_relative_res);
  fprintf(stderr,"DEBUG: H_final_relative_res=%8.2e   H_iter=%2d \n",
	  H_final_relative_res, H_iter); /**/

  /* get the solution */
  HYPRE_IJVectorGetValues(Hdat->x, local_size, Hdat->rows, Hdat->vals);

  /* copy the result to the output vector */
  for (i=0; i<Hdat->Flen; i++)
    {
      if (Hdat->hypre_DOFS[i]>=0)
	{
	  out->V[i]=Hdat->vals[Hdat->hypre_DOFS[i]];
	}
      else
	{
	  out->V[i]=0.0;
	}
    }

  return SUCCESS;
#endif
}


/*FUNCTION*/
int gen_hypre_AMG_free(void **AMGdata
/* free the hypre AMG solver data
   
   In/Out: AMGdata - (given by reference) 
                     the data for the AMG solver/preconditioner,
		     all content will be freed, on return this will
		     hold a NULL pointer

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		       ){
#ifndef HAVE_HYPRE_NO_MPI
  fprintf(stderr, "gen_hypre_AMG_free: compiled without hypre support!\n");
  return FAIL;  
#else
  struct FEINS_HYPRE_data *Hdat=*AMGdata;
  /* Destroy solver */
  HYPRE_BoomerAMGDestroy(Hdat->precond); 

  free(Hdat->vals);
  free(Hdat->rows);
  free(Hdat->feins_DOFS);
  free(Hdat->hypre_DOFS);

   /* Clean up */
  HYPRE_IJMatrixDestroy(Hdat->A);
  HYPRE_IJVectorDestroy(Hdat->b);
  HYPRE_IJVectorDestroy(Hdat->x);

  free(*AMGdata);
  *AMGdata = NULL;

  /* Finalize MPI */
  MPI_Finalize();
  
  return SUCCESS;
#endif
}

