/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
lin_solver.h
*/
#ifndef __LIN_SOLVER_H__
#define __LIN_SOLVER_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"


int PCG( int it_max, int stoptype, double atol, double rtol, int use_initial_x,
	 struct vector *x, double *residual, int *iter,
	 int (*multfunc)(void *Mat, struct vector *vec,
			 struct vector *out), 
	 int (*precondi)(void *Mat, struct vector *vec, void *prec_dat,
			 struct vector *out),
	 void *Mat, struct vector *rhs, void *prec_dat
/* Preconditioned Conjugate Gradient solver for systems of linear
   equations with symmetric positive definite system matrices, 
   approximates a solution of 

          Mat*x = rhs

   Input:  it_max   - maximal number of PCG iterations
	   stoptype - type of the stopping criterion:
                      0 - no criterion, just do it_max iterations,
		      1 - absolute, stop when norm(residual)<atol,
		      2 - relative to the initial residual, stop when 
		          norm(residual) < rtol*norm(residual_0)
		      3 - absolute and relative, stop if either the
		          relative or absolute stopping criterion of
		          above is fulfilled
		      other result in FAIL return + error message
           atol     - value of the absolute tolerance for the stopping
                      criteria 1 and 3, see stoptype above
           rtol     - value of the relative tolerance for the stopping
                      criteria 2 and 3, see stoptype above
           use_initial_x
                    - states if the provided x shall be used as
                      initial iterate x_0 or not,
		      = 1 : use it,
		      = 0 : don't use it, in this case the zero vector
		            is used instead
	   multfunc - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      PCG and vec is a vector of the same type as the
		      solution vector x, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      system matrix Mat, e.g. C^{-1} can be a
		      cheap approximation of the inverse of Mat,
		      the quality of the preconditioner has huge
		      influence on the performance of PCG,
		      syntax:
		      int precondi(void *Mat, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to PCG, vec a vector of
		      same type as solution vector x, prec_dat as
		      provided to PCG, must return SUCCESS on success
		      and FAIL in case of error,
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
           Mat      - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in PCG
		      directly, but only through multfunc and precondi
	   rhs      - the right-hand side vector of the equation system
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in PCG itself, but only
	              in precondi

   In/Out: x        - on Input:  initial iterate, if use_initial_x==1,
                                 else not used as input,
                      on Output: approximate solution of the system

   Output: residual - norm of the residual of the final iterate,
                      provided by reference, not used if NULL pointer
                      is provided
           iter     - number of the last iteration,
                      provided by reference, not used if NULL pointer
                      is provided

   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
	   10       - itermax reached, but stopping criterion not fulfilled
*/);

int PCR( int it_max, int stoptype, double atol, double rtol, int use_initial_x,
	 struct vector *x, double *residual, int *iter,
	 int (*multfunc)(void *Mat, struct vector *vec,
			 struct vector *out), 
	 int (*precondi)(void *Mat, struct vector *vec, void *prec_dat,
			 struct vector *out),
	 void *Mat, struct vector *rhs, void *prec_dat
/* Preconditioned Conjugate Residual solver for systems of linear
   equations with symmetric (maybe indefinite) system matrices, 
   approximates a solution of 

          Mat*x = rhs

   Input:  it_max   - maximal number of PCR iterations
	   stoptype - type of the stopping criterion:
                      0 - no criterion, just do it_max iterations,
		      1 - absolute, stop when norm(residual)<atol,
		      2 - relative to the initial residual, stop when 
		          norm(residual) < rtol*norm(residual_0)
		      3 - absolute and relative, stop if either the
		          relative or absolute stopping criterion of
		          above is fulfilled
		      other result in FAIL return + error message
           atol     - value of the absolute tolerance for the stopping
                      criteria 1 and 3, see stoptype above
           rtol     - value of the relative tolerance for the stopping
                      criteria 2 and 3, see stoptype above
           use_initial_x
                    - states if the provided x shall be used as
                      initial iterate x_0 or not,
		      = 1 : use it,
		      = 0 : don't use it, in this case the zero vector
		            is used instead
	   multfunc - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      PCR and vec is a vector of the same type as the
		      solution vector x, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      system matrix Mat, e.g. C^{-1} can be a
		      cheap approximation of the inverse of Mat,
		      the quality of the preconditioner has huge
		      influence on the performance of PCR,
		      syntax:
		      int precondi(void *Mat, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to PCR, vec a vector of
		      same type as solution vector x, prec_dat as
		      provided to PCR, must return SUCCESS on success
		      and FAIL in case of error,
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
           Mat      - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in PCR
		      directly, but only through multfunc and precondi
	   rhs      - the right-hand side vector of the equation system
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in PCR itself, but only
	              in precondi

   In/Out: x        - on Input:  initial iterate, if use_initial_x==1,
                                 else not used as input,
                      on Output: approximate solution of the system

   Output: residual - norm of the residual of the final iterate,
                      provided by reference, not used if NULL pointer
                      is provided
           iter     - number of the last iteration,
                      provided by reference, not used if NULL pointer
                      is provided

   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
	   10       - itermax reached, but stopping criterion not fulfilled
*/);

int GMRES( int m, int re_max, 
	   int stoptype, double atol, double rtol, 
	   int use_initial_x, struct vector *x, 
	   double *residual, int *iter,
	   int (*multfunc)(void *Mat, struct vector *vec,
			   struct vector *out), 
	   int (*precondi)(void *Mat, struct vector *vec,
			   void *prec_dat, struct vector *out),
	   void *Mat, struct vector *rhs, void *prec_dat
/* Generalised Minimal Residual solver for systems of linear
   equations, see Y. Saad, "Iterative Methods for Sparse Linear
   Systems", (2nd edition, Jan 3rd 2000, pp. 157)
   approximates a solution of 

          Mat*x = rhs

   Input:  m        - number of steps before restart, also number of
                      search directions to be stored,
                      generally, better convergence has to be expected
                      for larger m,
		      however, memory requirements are proportional to
		      m (m vectors have to be stored), and up to m
		      inner products (dot products) have to be
		      performed in each iteration,
           re_max   - maximal number restarts, therefore the maximal
                      number of iterations is m*re_max
	   stoptype - type of the stopping criterion:
                      0 - no criterion, just do re_max restarts,
		      1 - absolute, stop when norm(residual)<atol,
		      2 - relative to the initial residual, stop when 
		          norm(residual) < rtol*norm(residual_0)
		      3 - absolute and relative, stop if either the
		          relative or absolute stopping criterion of
		          above is fulfilled
                      3 - do re_max iterations (!not restarts!), but
                          stop if norm(residual)<=atol
		      other result in FAIL return + error message
           atol     - value of the absolute tolerance for the stopping
                      criteria 1 and 3, see stoptype above
           rtol     - value of the relative tolerance for the stopping
                      criteria 2 and 3, see stoptype above
           use_initial_x
                    - states if the provided x shall be used as
                      initial iterate x_0 or not,
		      = 1 : use it,
		      = 0 : don't use it, in this case the zero vector
		            is used instead
           multfunc - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      this routine and vec is a vector of the same
		      type as the solution vector x, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      system matrix Mat, e.g. C^{-1} can be a
		      cheap approximation of the inverse of Mat,
		      the quality of the preconditioner has huge
		      influence on the performance of this solver,
		      syntax:
		      int precondi(void *Mat, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to this routine, vec a
		      vector of same type as solution vector x,
		      prec_dat as provided to this routine, must
		      return SUCCESS on success and FAIL in case of
		      error, 
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
           Mat      - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in this
		      routine directly, but only through multfunc and
		      precondi 
	   rhs      - the right-hand side vector of the equation system
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in this routine itself,
	              but only in precondi

   In/Out: x        - on Input:  initial iterate, if use_initial_x==1,
                                 else not used as input,
                      on Output: approximate solution of the system

   Output: residual - norm of the residual of the final iterate,
                      provided by reference, not used if NULL pointer
                      is provided
           iter     - number of the last iteration,
                      provided by reference, not used if NULL pointer
                      is provided

   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
	   10       - itermax reached, but stopping criterion not
	              fulfilled 
*/);

int chebyshev_semi_iteration(  
	   struct vector *x,
	   int iter,
	   int (*multfunc)(void *Mat, struct vector *vec,
			   struct vector *out), 
	   int (*precondi)(void *Mat, struct vector *vec,
			   void *prec_dat, struct vector *out),
	   void *Mat, struct vector *rhs, void *prec_dat,
	   double lambdaMin, double lambdaMax
/* Chebyshev semi iteration for solving systems of linear
   equations, see ???
   approximates a solution of 

          Mat*x = rhs

   Input:  iter     - the fixed number of iterations to be performed,
           multfunc - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      this routine and vec is a vector of the same
		      type as the solution vector x, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      system matrix Mat, e.g. C^{-1} can be a
		      cheap approximation of the inverse of Mat,
		      the quality of the preconditioner has huge
		      influence on the performance of this solver,
		      syntax:
		      int precondi(void *Mat, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to this routine, vec a
		      vector of same type as solution vector x,
		      prec_dat as provided to this routine, must
		      return SUCCESS on success and FAIL in case of
		      error, 
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
           Mat      - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in this
		      routine directly, but only through multfunc and
		      precondi 
	   rhs      - the right-hand side vector of the equation system
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in this routine itself,
	              but only in precondi
	   lambdaMin- minimal eigenvalue of the preconditioned system
	   lambdaMax- maximal eigenvalue of the preconditioned system

   Output: x        - approximate solution of the system
           
   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
*/);

int PBSDeig(int it_max, int stoptype, double atol, double rtol,
	    int use_initial_x,
	    FIDX len, FIDX num,
	    struct vector *x, struct vector *lambda,
	    double *residual, int *iter,
	    int (*multfuncMass)(void *MatMass, struct vector *vec,
				struct vector *out), 
	    int (*multfuncStif)(void *MatStif, struct vector *vec,
				struct vector *out), 
	    int (*precondi)(void *MatStif, struct vector *vec, void *prec_dat,
			    struct vector *out),
	    void *MatMass, void *MatStif,
	    void *prec_dat
/* Preconditioned Block Steepest Descent solver for eigenvalue problems

          MatStif*x = lambda*MatMass*x 
   
   as described in [M. Weise: Eigenfrequenzanalyse realer
   CAD-Geometrien mit NetGen/NGSolve, Modellierungsseminar,
   TU-Chemnitz, 2009]
   where [A. V. Knyazev, K. Neymeyr: Efficient solution of symmetric
   eigenvalue problems using multigrid preconditioners in the locally
   optimal block conjugate gradient method, ETNA, Vol. 15 2003;
   http://math.ucdenver.edu/~aknyazev/research/papers/etna03.pdf]
   is cited as reference 

   seeks to find the num eigenvalues of smallest absolute value 


   Input:  it_max   - maximal number of iterations
	   stoptype - type of the stopping criterion:
                      0 - no criterion, just do it_max iterations,
		      1 - absolute, stop when norm(residual)<atol,
		      2 - relative to the initial residual, stop when 
		          norm(residual) < rtol*norm(residual_0)
		      3 - absolute and relative, stop if either the
		          relative or absolute stopping criterion of
		          above is fulfilled
		      other result in FAIL return + error message
           atol     - value of the absolute tolerance for the stopping
                      criteria 1 and 3, see stoptype above
           rtol     - value of the relative tolerance for the stopping
                      criteria 2 and 3, see stoptype above
           use_initial_x
                    - states if the provided x shall be used as
                      initial iterate x_0 or not,
		      = 1 : use it,
		      = 0 : don't use it, in this case pseudo-random 
		            numbers are used instead
           len      - size of the system matrices, 
	              (length of compatible vectors)
           num      - number of (smallest) eigenvalues and corresponding
	              eigenvectors to be computed
	   multfuncMass AND 
	   multfuncStif 
                    - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      PBSDeig and vec is a vector of size len, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      stiffness matrix MatStif, e.g. C^{-1} can be a
		      cheap approximation of the inverse of MatStif,
		      the quality of the preconditioner has huge
		      influence on the performance of PBSDeig,
		      syntax:
		      int precondi(void *MatStif, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to PBSDeig, vec a vector of
		      size len, prec_dat as
		      provided to PBSDeig, must return SUCCESS on success
		      and FAIL in case of error,
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
           MatMass    AND
           MatStif  - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in PBSDeig
		      directly, but only through multfuncMass, 
		      multfuncStif and precondi
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in PCG itself, but only
	              in precondi

   In/Out: x        - on Input:  initial iterates for all eigenvectors,
                                 if use_initial_x==1,
                                 else not used as input (then initialised
				               with pseudo random numbers),
                      on Output: approximate eigenvectors,
		      size: len*num,
		      x.V[k*len+i] is the i-th component of the k-th 
		      eigenvector

   Output: lambda   - the eigenvalues
           residual - norm of the residual of the final iterate,
                      provided by reference, not used if NULL pointer
                      is provided
           iter     - number of the last iteration,
                      provided by reference, not used if NULL pointer
                      is provided

   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
	   10       - itermax reached, but stopping criterion not fulfilled
*/);

int LOBPCGeig(int it_max, int stoptype, double atol, double rtol,
	      int use_initial_x,
	      FIDX len, FIDX num, FIDX num_chk,
	      struct vector *x, struct vector *lambda,
	      double *residual, int *iter,
	      int (*multfuncMass)(void *MatMass, struct vector *vec,
				  struct vector *out), 
	      int (*multfuncStif)(void *MatStif, struct vector *vec,
				  struct vector *out), 
	      int (*precondi)(void *MatStif, struct vector *vec, void *prec_dat,
			      struct vector *out),
	      int (*projectr)(void *MatStif, struct vector *vec, void *proj_dat,
			      struct vector *out),
	      void *MatMass, void *MatStif,
	      void *prec_dat, double *prec_lambda_i, void *proj_dat
/* Locally Optimal Block Preconditioned Conjugate Gradient Method for
   eigenvalue problems

          MatStif*x = lambda*MatMass*x 
   
   as described in [M. Weise: Eigenfrequenzanalyse realer
   CAD-Geometrien mit NetGen/NGSolve, Modellierungsseminar,
   TU-Chemnitz, 2009]
   where [A. V. Knyazev, K. Neymeyr: Efficient solution of symmetric
   eigenvalue problems using multigrid preconditioners in the locally
   optimal block conjugate gradient method, ETNA, Vol. 15 2003;
   http://math.ucdenver.edu/~aknyazev/research/papers/etna03.pdf]
   is cited as reference 

   seeks to find the num eigenvalues of smallest absolute value 


   Input:  it_max   - maximal number of iterations
	   stoptype - type of the stopping criterion:
                      0 - no criterion, just do it_max iterations,
		      1 - absolute, stop when norm(residual)<atol,
		      2 - relative to the initial residual, stop when 
		          norm(residual) < rtol*norm(residual_0)
		      3 - absolute and relative, stop if either the
		          relative or absolute stopping criterion of
		          above is fulfilled
		      other result in FAIL return + error message
           atol     - value of the absolute tolerance for the stopping
                      criteria 1 and 3, see stoptype above
           rtol     - value of the relative tolerance for the stopping
                      criteria 2 and 3, see stoptype above
           use_initial_x
                    - states if the provided x shall be used as
                      initial iterate x_0 or not,
		      = 1 : use it,
		      = 0 : don't use it, in this case pseudo-random 
		            numbers are used instead
           len      - size of the system matrices, 
	              (length of compatible vectors)
           num      - number of (smallest) eigenvalues and corresponding
	              eigenvectors to be used in computation
           num_chk  - number of (smallest) eigenvectors to be used in the
                      stopping criterion, so only num_chk eigenvalues will
                      actually fulfill the tolerances, has to fulfill 
		      num_chk<=num 
	   multfuncMass AND 
	   multfuncStif 
                    - pointer to a function performing the matrix
                      vector multiplication Mat*vec, syntax:
		      int multfunc(void *Mat, struct vector *vec, 
		                   struct vector *out),
		      where Mat is the pointer provided in the call to
		      LOBPCGeig and vec is a vector of size len, 
		      must return SUCCESS on success and FAIL in case
		      of error 
	   precondi - pointer to a function performing the
                      preconditioning operation out=C^{-1}*vec,
		      ideally C should be spectral equivalent to the
		      stiffness matrix MatStif, e.g. C^{-1} can be a
		      cheap approximation of the inverse of MatStif,
		      the quality of the preconditioner has huge
		      influence on the performance of LOBPCGeig,
		      syntax:
		      int precondi(void *MatStif, struct vector *vec,
		                   void *prec_dat, struct vector *out),
		      where Mat as provided to LOBPCGeig, vec a vector of
		      size len, prec_dat as
		      provided to LOBPCGeig, must return SUCCESS on success
		      and FAIL in case of error,
		      if precondi==NULL then the function is not
		      called but C^{-1}=I is used (no preconditioning)
		      if the preconditioner needs the eigenvalue 
		      approximation, use prec_lambda_i, see below
	   projectr - pointer to a function performing the
                      projection operation out=P*vec,
		      we look for eigenvectors only in span(P)
		      syntax:
		      int precondi(void *MatStif, struct vector *vec,
		                   void *proj_dat, struct vector *out),
		      where Mat as provided to LOBPCGeig, vec a vector of
		      size len, prec_dat as
		      provided to LOBPCGeig, must return SUCCESS on success
		      and FAIL in case of error,
		      if projectr==NULL then the function is not
		      called but P=I is used (no projection)
           MatMass    AND
           MatStif  - pointer to data necessary to perform the
                      multiplication with the matrix, 
		      in general this should be some representation of
		      the matrix itself, but it is not used in LOBPCGeig
		      directly, but only through multfuncMass, 
		      multfuncStif and precondi
	   prec_dat - pointer to additional data for the
	              preconditioner, not used in LOBPCGeig itself, but only
	              in precondi, see also prec_lambda_i below
	   proj_dat - pointer to additional data for the
	              preconditioner, not used in LOBPCGeig itself, but only
	              in projectr

   In/Out: x        - on Input:  initial iterates for all eigenvectors,
                                 if use_initial_x==1,
                                 else not used as input (then initialised
				               with pseudo random numbers),
                      on Output: approximate eigenvectors,
		      size: len*num,
		      x.V[k*len+i] is the i-th component of the k-th 
		      eigenvector
	   prec_lambda_i
                    - pointer where the eigenvalue corresponding to a
                      residual vector may be written for the
                      preconditioner,
		      if ==NULL it is ignored

   Output: lambda   - the eigenvalues
           residual - norm of the residual of the final iterate,
                      provided by reference, not used if NULL pointer
                      is provided
           iter     - number of the last iteration,
                      provided by reference, not used if NULL pointer
                      is provided

   Return: SUCCESS  - stopping criterion is fulfilled
           FAIL     - general failure, see error messages
	   10       - itermax reached, but stopping criterion not fulfilled
*/);
#endif
