/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
lame_adj.h
*/
#ifndef __LAME_ADJ_H__
#define __LAME_ADJ_H__ 1



#include "feins_macros.h"

#include "datastruc.h"

int lame_dIdu_t2( struct mesh *m,
                  struct vector *u,
		  struct vector *dIdu,
		  double *Is,
		  double lambda,
		  double mu
/* evaluates the performance criteria (given with the mesh) for the
   supplied solution vector sol, stores the values in pcvec

   Input:  m       - the mesh
           u       - vector containing the solution of a lame problem, 
	             calculated using the T2/T1 element pair
	   lambda,
           mu      - lame parameters

   Output: dIdu    - vector containing the values of the derivates 
                     of Is with respect to the solution u, ignored if ==NULL
           Is      - the value of the performance criteria (by
                     reference), ignored if ==NULL


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_DIDx_t2( struct mesh *m,
                   double lambda,
                   double mu,
		   struct vector *psi,
                   struct vector *u,
                   struct vector *DIDx
    
/* evaluates the derivatives of the performance criteria (given with
   the mesh) with respect to the nodal coordinates, using the adjoint
   solution psi

     DI   dI         dR
     -- = -- - psi^T --
     Dx   ds         ds

   Input:  m         - the mesh
           lambda    - Lame' constant 
           mu        - Lame' constant 
	   psi       - vector containing the solution of the adjoint equation
           u         - vector containing the solution of a lame problem

   Output: DIDx     - total gradient of I with respect to the node positions

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_Allaire_et_al_2004__formula_8_t2( struct mesh *m,
                   double lambda,
                   double mu,
                   struct vector *u,
                   struct vector *DIDx
    
/* evaluates the derivatives of the compliance (energy) according to
   formula (8) of [Allaire et.al. 2004]
   simplification: assume rhs function f=0  (no volume forces)

   Input:  m         - the mesh
           lambda    - Lame' constant 
           mu        - Lame' constant 
           u         - vector containing the solution of a lame problem

   Output: DIDx      - total gradient of I with respect to the node
                       positions (non-zero only on boundary)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int curvature_2d_3points( double *x1, double *x2, double *x3,
			  double *curv
/* evaluates the curvature curv of a line segment through
   three points by determining a circle through these points and
   using 1/radius 

   the circle is determined as in 
     http://www.arndt-bruenner.de/mathe/scripts/kreis3p.htm
   by considering a non-linear equation system in the variables
     xm (center x),ym (center y), radius 
   this is simplified to a linear system using the auxiliary variables
     [A,B,C]
   such that
     xm = B/2, ym = C/2, und r^2 = xm^2 + ym^2 - A


   Input:  x1,x2,x3  - two dimensional vectors, the three points which
                       are given

   Output: curv      - curvature of the circle through the three
                       points, ore zero if the three points are on a
                       line 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_zz_interpolate_sigma_tx(struct mesh *m,
				int type,
				FIDX ncomp, struct vector *u,
				double lambda, double mu,
				struct vector *sigmu
/* averages the values of the stress tensor at the nodes
   of the mesh, thus interpolation of these provides smoothed
   stress tensors.

   
   Input:  m       - the mesh, has to be T1 or T2 
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles
           ncomp   - number of (solution) components in u, i.e. number
                     of variables per node in the mesh
	   u       - coefficient vector representing a function on
	             the current mesh (e.g. solution of PDE problem)
		     u.V[r*m->vx_nr+i] has to be the value of the r-th
		     component of the (possibly) vector valued
		     function u at node i of the mesh
	   lambda, mu
                   - Lame constants

   Output: sigmu   - the stress tensor (of u) averaged at the
                     nodes of the mesh, has to be of size
                     dim*ncomp*vx_nr,
		     sigmu.V[d*ncomp*vx_nr + r*vx_nr +i]
		     will be the averaged value of sigma_d,r at the
		     i-th node of the mesh 

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int lame_zz_interpolate_sigma_gamma_tx(struct mesh *m,
				      int type,
				      FIDX ncomp, struct vector *u,
				      double lambda, double mu,
				      struct vector *sigmGu
/* averages the values of the *surface* stress tensor at the nodes of
   the mesh, thus interpolation of these provides smoothed surface
   stress tensors.

   
   Input:  m       - the mesh, has to be T1 or T2 
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles
           ncomp   - number of (solution) components in u, i.e. number
                     of variables per node in the mesh
	   u       - coefficient vector representing a function on
	             the current mesh (e.g. solution of PDE problem)
		     u.V[r*m->vx_nr+i] has to be the value of the r-th
		     component of the (possibly) vector valued
		     function u at node i of the mesh
	   lambda, mu
                   - Lame constants

   Output: sigmGu  - the surface stress tensor (of u) averaged at the
                     nodes of the mesh, has to be of size
                     dim*ncomp*vx_nr,
		     sigmu.V[d*ncomp*vx_nr + r*vx_nr +i]
		     will be the averaged value of sigma_G_d,r at the
		     i-th node of the mesh 

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int lame_eppler_conjecture_2010_rem3_3( struct mesh *m,
					double lambda,
					double mu,
					struct vector *u,
					struct vector *sdifference    
/* evaluates the square of the norm of the difference between the
   left-hand-side and right-hand-side in Remark 3.3 of 
     [K. Eppler, Hadamard shape gradients, 2010] 
   (pointwise) on the Neumann boundary i.e.

   difference = (f+H*g)-(-D(Ce(u))[n]*n - div_Gamma{Ce(u)_Gamma}

   sdifference = difference^T*difference

   if the conjecture holds true, this difference should converge to
   zero with mesh refinement, at least for sufficiently smooth bounded
   domains

   the higher order derivatives are evaluated using a ZZ-like
   postprocessing (averaging of gradients + interpolation of the
   averaged). the pointwise results are evaluated at the mesh points,
   thus can be interpolated to give an approximation of the
   distributions (or hopefully their function representants if they
   are smooth enough) 

   Input:  m         - the mesh
           lambda    - Lame' constant 
           mu        - Lame' constant 
           u         - vector containing the solution of a lame problem

   Output: sdifference
                     - square of norm of difference as described above
                       (non-zero only on Neumann boundary)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
