/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

/* this file contains the definitions of the data structures */

/* don't do anything if allready included */
#ifndef DATASTRUC_H
#define DATASTRUC_H

#include "sparse.h"

/* definition of element types */
enum elemtype {
  NO_type,
  inter,    /* interval [0,1] */
  tria,     /* triangles      */
  quadr,    /* quadrangles    */
  tetra,    /* tetrahedra     */
  hexa,     /* hexahedra      */
  pyra      /* pyramids       */
};


/* basic data structure for the mesh */
struct mesh {
  int  dim;           /* space dimension (2 for 2d, 3 for 3d) */
  int type;           /* type of elements
                         1=  P1 triangles
                         2=  P2 triangles
			 31= P1 tetrahedra
			 32= P2 tetrahedra (p-hierarchical basis)
		      */
  FIDX problem;       /* identifies the problem,
                         1= Poison Equation
                         2= Stokes flow
                         3= Navier Stokes flow
                         4= Lame Equation (Linear Elasticity), 
                            in 2d plane strain
		      */
  int  curved;        /* in 2d ignored,
			 in 3d: 
			 0= not curved, polygonal domain
			 1= curved, use gmsh +.geo to move points to geometry,
			 2= curved, use .stl file to move points to geometry
		      */

  FIDX vx_w;          /* number of double entries per vertex, in other
			 words the width of a vertex entry   */
  FIDX vx_nr;         /* actual (current) number of vertices */
  FIDX vx_max;        /* maximal number of vertices for which memory
			 has been allocated */
  double *vertex;     /* vertices of the mesh stored as
			 vertex[i*vertex_w+j], the j-th component of the
			 i-th vertex, j<v_w, i<v_nr */
  FIDX el_w;          /* number of int entries per element, in other
			 words the width of a element entry  */
  FIDX el_nr;         /* actual (current) number of elements */
  FIDX el_max;        /* maximal number of elements for which memory
			 has been allocated */
  FIDX *elem;         /* elements of the mesh, should hold the vertices
			 of the element (for fast creation of the
			 stiffnessmatrix), element identifier
			 corresponds to face identifier (2d)
			 resp. volume identifier (3) */ 
 
  FIDX eg_w;          /* number of int entries per edge, in other
			 words the width of a edge entry  */
  FIDX eg_nr;         /* actual (current) number of edges */
  FIDX eg_max;        /* maximal number of edges for which memory
			 has been allocated */
  FIDX *edge;         /* edges of the mesh, should hold the vertices
			 of the edges, status of the edge (boundary,
			 parents, childs) (for fast refinement) */

  FIDX fc_w;          /* number of int entries per face, in other
			 words the width of a face entry  */
  FIDX fc_nr;         /* actual (current) number of faces */
  FIDX fc_max;        /* maximal number of faces for which memory
			 has been allocated */
  FIDX *face;         /* faces of the mesh, should hold the edges
			 of the face, status of the face (boundary,
			 parents, childs) (for fast refinement) */

  FIDX vo_w;          /* number of int entries per volume, in other
			 words the width of a volume entry  */
  FIDX vo_nr;         /* actual (current) number of volumes */
  FIDX vo_max;        /* maximal number of volumes for which memory
			 has been allocated */
  FIDX *vols;         /* volumes of the mesh, should hold the faces
			 of the volumes, status of the volume (parents,
			 childs) (for fast refinement) */
		     
  FIDX hi_w;          /* number of int entries per hierarchy data, in other
			 words the width of a hierarchy entry  */
  FIDX hi_nr;         /* actual (current) number of hierarchy entries */
  FIDX hi_max;        /* maximal number of hierarchy entries for which
			 memory has been allocated */
  FIDX *hier;         /* hierarchy of the the mesh, edge based (which
			 is sufficient for linear elements and certain
			 situations regarding other elements), should
			 hold the parent nodes for each child node
			 (for fast hierarchical operations) */

  FIDX eh_w;          /* number of int entries per element hierarchy
			 data, in other words the width of a element
			 hierarchy entry  */ 
  FIDX eh_nr;         /* actual (current) number of element hierarchy
			 entries */ 
  FIDX eh_max;        /* maximal number of element hierarchy entries
			 for which memory has been allocated */
  FIDX *elhier;       /* hierarchy of the the mesh, element based,
			 should hold all the data necessarry to
			 interpolate coarse mesh data on the finer
			 meshes accurately */
  FIDX lvl;           /* current level of the refinement 
			 (-1=no refinement, n = (n+1) times refined */

  FIDX bd_w;          /* number of int entries per boundary data, in
			 other words the width of a boundary entry */ 
  FIDX bd_nr;         /* actual (current) number of boundary entries */
  FIDX bd_max;        /* maximal number of boundary entries for which
			 memory has been allocated */
  FIDX *bound;        /* boundary data of the the mesh, should hold the
			 boundary edges/faces and specify the boundary
			 condition */

  FIDX ps_w;          /* number of int entries per surface performance
			 criterion (pcsurf) entry, in other words the
			 width of a pcsurf entry */  
  FIDX ps_nr;         /* actual (current) number of pcsurf entries */
  FIDX ps_max;        /* maximal number of pcsurf entries for which
			 memory has been allocated */
  FIDX *pcsurf;       /* surface performance criteria data of the the
			 mesh, should hold the edges(2d)/faces(3d) on
			 which the criteria are evaluated and specify
			 the id (pointer into pcrit) of the criterion
		      */ 

  FIDX pv_w;          /* number of int entries per volume performance
			 criterion (pcvol) entry, in other words the
			 width of a pcvol entry */  
  FIDX pv_nr;         /* actual (current) number of pcvol entries */
  FIDX pv_max;        /* maximal number of pcvol entries for which
			 memory has been allocated */
  FIDX *pcvol;        /* volume performance criteria data of the the
			 mesh, should hold the faces(2d)/volumes(3d)
			 on which the criteria are evaluated and
			 specify the id (pointer into pcrit) of the
			 criterion
		      */ 

  FIDX pc_w;          /* number of double entries per performance
			 criterion (pccrit) entry, in other words the
			 width of a pccrit entry */  
  FIDX pc_nr;         /* actual (current) number of pccrit entries */
  FIDX pc_max;        /* maximal number of pccrit entries for which
			 memory has been allocated */
  double *pccrit;     /* performance criteria data of the the mesh,
			 should define the type of the criterion
			 (e.g. body force, square deviation of
			 specified value), and parameters for this
			 type of criterion (e.g. the weight vector for
			 the force vector, to arrive at a single
			 value) */ 

  FIDX fu_w;          /* number of int entries per function data, in
			 other words the width of a function entry */ 
  FIDX fu_nr;         /* actual (current) number of function entries */
  FIDX fu_max;        /* maximal number of function entries for which
			 memory has been allocated */
  double *func;       /* function data for the the mesh, should hold the
			 simple function descriptions, e.g. for
			 boundary conditions or volume forces */ 

  FIDX pa_w;          /* number of int entries per parameter data, in
			 other words the width of a parameter entry */ 
  FIDX pa_nr;         /* actual (current) number of parameter entries */
  FIDX pa_max;        /* maximal number of parameter entries for which
			 memory has been allocated */
  double *para;       /* parameter data for the the mesh, should hold
			 physical constants, and scaling parameters
			 related to the problem */

  FIDX sg_w;          /* number of int entries per shape segment data, in
			 other words the width of a shape segment entry */ 
  FIDX sg_nr;         /* actual (current) number of shape segment entries */
  FIDX sg_max;        /* maximal number of shape segment entries for which
			 memory has been allocated */
  FIDX *sseg;         /* shape segment data for the the mesh, should hold
			 type of the segment and pointers to its
			 parameters in spar */
  FIDX st_nr;         /* actual (current) number of t-value */
  FIDX st_max;	      /* maximal number of t-values for which memory
			 has been allocated */
  double *st;	      /* st[i] is the t-value of the i-th node on a shape 
			 segment */
  FIDX sp_w;          /* number of int entries per shape parameter data, in
			 other words the width of a shape parameter entry */ 
  FIDX sp_nr;         /* actual (current) number of shape parameter entries */
  FIDX sp_max;        /* maximal number of shape parameter entries for which
			 memory has been allocated */
  double *spar;       /* shape parameter data for the the mesh, should hold
			 physical constants, and scaling shape parameters
			 related to the problem */

  int meshgen;        /* the mesh generator involved in the creation
			 of the mesh,
			 ==0: no mesh generator;
			 ==1: triangle (2d);
			 ==2: gmsh (2d + 3d);
		      */
  void *meshgenDATA;  /* pointer to data required by the meshgen
			 wrapper routines (e.g. to project new nodes
			 to the geometry */

  FIDX DD_proc_nr;    /* number of processors to be used (DD) */
  FIDX DD_waves_nr;   /* number of waves of DD_proc_nr domains (DD) */
  FIDX DD_nr;         /* number of subdomains in Domain Decomposition
			 (DD), should be DD_proc_nr*DD_waves_nr */
  FIDX *DD_elem_idx;  /* DD_elem_idx[i] index of start of subdomain i
			 in DD_elem (length DD_nr+1) */
  FIDX *DD_elem;      /* list of elements in each subdomain,
			 DD_elem[DD_elem_idx[i]]
			 ... DD_elem[DD_elem_idx[i+1]-1] */
  FIDX *DD_node_idx;  /* DD_node_idx[i] index of start of subdomain i
			 in DD_node (length DD_nr+1) */
  FIDX *DD_node;      /* list of nodes in each subdomain,
			 DD_node[DD_node_idx[i]]
			 ... DD_node[DD_node_idx[i+1]-1] */

  /* detailed values for the positions of objects within the mesh
     arrays are element type dependent and therefore defined with text
     macros in the include file "meshdetails.h" */
};
#include "meshdetails.h"


/* datastructure to operate with a vector containing the vector of
   nodal data on each individual level */
struct multilvl {
  FIDX vx_nr;         /* number of nodes in the finest level */
  FIDX dim;           /* number of components for each node */
  FIDX type;          /* type of the elements (1=linear,2=quad)*/
  FIDX lmax;          /* number of levels of the hierarchy */
  FIDX *nlevl;        /* pointer to vector of length lmax+2 holding
			 the end positions(+1) of each level in the
			 multilevel help vectors, therefore nlevl[0] is
			 the total length of the help vectors, and
			 nlevl[k+1] is the start position of the
			 help vector belonging to level k,
			 to get the real position of a node in the
			 help vectors use the macro MLVLFINDENTRY (see
			 bellow) */
  FIDX *levlvx;       /* pointer to vector of length lmax+1 holding
			 the number of veritces for each level,
			 levlvx[i] gives the number of vertices in
			 level i */
  FIDX *levlhindx;    /* pointer to a vector of length lmax+2 holding
			 the end positions(+1) of each level in the
			 element hierarchy field of the mesh,
			 therefore levlhindx[0] is the begin of level 0,
			 levlhindx[1] the begin of level 1 (== end of
			 level 0 +1) and so on, */
};
/* a macro to find the index of a given node in a certain level,
   usage:
   out   - the variable you want to operate with (can as well be ==in)
   in    - the nodal data component number (in the finest grid)
           (= component*vx_nr + node)
   l     - the level you want
   mlvl  - the name of the multilvl struct to be used, ( *name if a
           pointer is used)
*/
#define MLVLFINDENTRY(__out__,__in__,__l__,__mlvl__) {\
   FIDX mlvld_, mlvlnode_; \
   mlvlnode_=__in__; mlvld_=0; \
   while( mlvlnode_>=(__mlvl__).vx_nr) \
     { \
        mlvlnode_ -= (__mlvl__).vx_nr; \
        mlvld_++; \
     } \
   if ((mlvlnode_)<(__mlvl__).levlvx[__l__]) \
     { \
        __out__=mlvlnode_ + mlvld_* (__mlvl__).levlvx[__l__] \
                +(__mlvl__).nlevl[(__l__)+1]; \
     } \
   else	\
     { \
        __out__=-1; \
     } \
   }

/* data structure for the BPX preconditioner */
struct bpxdata
{
  double *hvec;       /* pointer to a vector of length ml.nlevl[0]
			 containing the BPX help vector itself */

  struct coarse_mat *cmat;
                      /* pointer to the factorised coarse grid matrix,
			 if cmat==NULL then this will not be used */
  struct mesh *msh;   /* pointer to the mesh */
  struct multilvl *ml;/* multilevel data struct */
  struct projector1 *P;/* projector for boundary conditions */
};



/* data structure for the multigrid preconditioner */
struct mgdata {
  FIDX smooths;       /* used to pass the number of smooths */
  FIDX vcycles;       /* used to pass the number of V-cycles,
		         if >0 then do the given number of V-cycles,
		         if <0 then do at most -vcycles V-cycles
		           until the goal stop_eps is reached */
  double stop_eps;    /* stoping criterion for the norm of the
			 residual, only used if vcycles<0, iteration
			 is stoped if the residual improved by a
			 factor stop_eps, compared to the initial
			 iterate x=0 (==norm(rhs)) */
  FIDX vccount;       /* counts the number of vcycles */
  FIDX vx_nr;         /* the number of vertices in the mesh this data
			 struct belongs to */
  FIDX dim;           /* number of components for each node */
  double *xl;         /* pointer to a vector of length ml.nlevl[0]
			 containing the help vector for x, the
			 approximation of the solution of the system
		      */ 
  double *dxl;        /* pointer to a vector of length ml.nlevl[0]
			 containing the help vector for dx, the update
			 step of the multigrid (between levels) */
  double *bl;         /* pointer to a vector of length ml.nlevl[0]
			 containing the help vector for the
			 righthandside on the different levels */
  FIDX *sorterl;      /* if !=NULL, contains a permutation of the dofs
			 on each level to be used for sorted GS
			 smooths,  vector of length ml.nlevl[0] */
  FIDX bas_n;         /* number of fater nodes per element hierarchy
			 entry */
  FIDX chl_n;         /* number of child nodes per element hierarchy
			 entry for red refinement 
			 (4 congruent triangles) */
  FIDX chl_n_bi;      /* number of child nodes per element hierarchy
			 entry for green refinement (bisection) */
  double *phih;       /* values of the father nodes basis functions at
			 the child nodes, array of length bas_n*chl_n,
			 phih[i+j*bas_n] is the value of basis
			 function i at child node j (red refinement) */
  double *phibi;      /* values of the father nodes basis functions at
			 the child nodes, array of length bas_n*chl_n_bi,
			 phibi[i+j*bas_n] is the value of basis
			 function i at child node j (green refinement) */
  FIDX *done;         /* integer vector of lenght dim*vx_nr, intended
			 for marking if a particular component has ben
			 taken care of allready, e.g. whilst
			 interpolating between meshes */
  double *invdiag;    /* pointer to a vector of length ml.nlevl[0]
			 containing the vector of diagonal entries of
			 the stiffnessmatrices on the different levels
			 of the hierarchy */
  struct mesh *msh;   /* the mesh associated with this data structure */
  struct multilvl *ml;/* the multilvl struct associated with this
			 data structure */
  struct sparse *Ks;  /* array os sparse matrices */
  struct projector1 *P;/* boundary projection data */
  struct coarse_mat *cmat;
                      /* pointer to the factorised coarse grid matrix,
			 if cmat==NULL then this will not be used */
  double CGC_scale;   /* defines scaling for the coarse grid updates,
			 either for damping purposes or to adjust for
			 different scaling of the residual on
			 different levels */ 
};

/* structure to contain data for preconditioner for eigenvalue problems */
struct eigP_data
{
  struct sparse *M;   /* the mass matrix */
  struct sparse *A;   /* the stiffness matrix */
  double lambda;      /* the eigenvalue under consideration */
  struct vector hlp;  /* help vector for the multiplies */
  struct mgdata *mg;  /* multigrid data structure */
};



/* structure to contain basis specific intergration data, these (or a
   vector of them should be attached to int_data, as userdat */
struct int_data_basis
{
  FIDX subtype;       /* sub-type of the element, e.g. 1 for P_1 triangles,
			 while 2 for P_2 triangle */
  FIDX num_basis;     /* number of basis functions for this subtype */
  double *phi;        /* array of dimension [num_points*num_basis]
			 containing the j-th basis function evaluated at
			 the i-th point as phi[i*num_basis+j] */
  double *gradphi;    /* array of dimension [num_points*num_basis*dim]
			 containing the gradient of phi,
			 gradphi[i*num_basis*dim +j*dim +k],
			 the k-th component of the gradient of the j-th
			 basis function evaluated at the i-th point,
			 provided via reference */
  double *hessphi;    /* array of dimension
			 [num_points*num_basis*dim*dim] containing the
			 Hessian of phi,
			 hessphi[i*num_basis*dim*dim 
			         +j*dim*dim +k*dim +l],
			 is the derivative of the j-th basis function
			 with respect to the k-th and l-th spacial
			 dimension, evaluated at the i-th point,
			 provided via reference */
};


/* basic data structure for numerical intergration formulas */
struct int_data
{
  FIDX dim;           /* dimension */
  enum elemtype type; /* type of domain/element, see element types */
  FIDX num_subtypes;  /* number of subtypes of the element (for basis
			 pairs, or single bases) */
  FIDX degree;        /* aglebraic degree of the formula */
  FIDX num_points;    /* number of cubature points */
  double *weights;    /* vector of weights */
  double *points;     /* array of dimension [num_points*dim]
	                 containing the vectors of the quadrature
	                 points, points[i*dim +j] is the j-th
	                 component of the i-th point */
  struct int_data_basis **bases;
                      /* array of length num_subtypes of pointers to
		         int_data_basis, which  then hold basis
		         functions and their derivatives evaluated at
		         the quadrature points */
  /* use free_intdata to free the interior pointers */
};



/* simple connected integer list */
struct ilist
{
  FIDX data;          /* the data part of the list element */
  struct ilist *next; /* the successor */
};

/* simple combination of a int and a double, can be used to sort by
   the double and thereby get a permutation of the integers that does
   a sort of the doubles (use qsort with comp_intdouble_d as compare
   fucntion) */
struct intdouble{
  FIDX i;
  double d;
};

/* structure with various parameters */
struct solver_settings
{
  FIDX refine_ini;		/* initial uniform refinement steps */
  FIDX refine_type;       	/* Type of refinement:
			   	   0 - uniform (red) mesh refinement
			   	   1 - adaptive (beansch green) mesh refinement
				       with a residual-based error estimator*/
  FIDX refine_steps;		/* (maximum) number of refinement steps */
  double refine_stop;		/* error tolerance for the refinement,
				   stop refinement if the global error
				   estimate globest<refine_stop */
  FIDX refine_max_vx;		/* max number vertices, refinement
				   atempts to stop just before this
				   vx_nr is exceeded */


  FIDX adap_mark;		/* Type of marking the element 
				   (that are to be refined) 
				   0 - all T with n_T > adap_mark_par* (max n_t)
				   1 - sort list and mark the
				       (adap_mark_par) largest */
  double adap_mark_par;	        /* Parameter for the marking */



  FIDX solver;	                /* Type of solver for Ku=f:
			   	   0 - direct solver
			     	   1 - PCG with multigrid
				   2 - PCG with BPX
			   	   3 - Sparse direct solver with UMFPACK */
  double solver_atol;	        /* absolute stop criteria in the solver
			           (stop crit. can't be lower than this) */
  double solver_ini_rtol;	/* relative stop criteria in the solver
				   for the first solving */
  double solver_ref_rtol;	/* relative stop criteria in the solver
				   during the refinement (for the
				   interpolated solution) */

  FIDX is_instat;               /* if the problem is instationary, all
				   the instat_* properties are only
				   set if is_instat==1
				   0 - problem is stationary, all
				       instat_* properties are not
				       used/set
				   1 - problem is not stationary */
  double instat_delta_t0;       /* time step on the initial (coarse)
                                   mesh, actual time step is computed
                                   refining the time step according to
                                   the order of the time integration
                                   method in comparison to the order
                                   of the spatial discretisation to
                                   get balancing of these error
                                   contributions
				*/
  double instat_delta_t_fix;    /* if instat_delta_t_fix>0.0 the fixed
				   time step size instat_delta_t_fix
				   is used, and NOT REDUCED during mesh
				   refinement */
  double instat_Time0;          /* initial time for the time dependent
				   solution */
  double instat_Tend;           /* final time, computation is stopped
				   if time>instat_Tend */
  FIDX instat_write_steps;      /* the solution is written every
				   instat_write_steps time steps
				   instat_delta_t0 or instat_delta_t_fix,
				   i.e. if not fixed step size is
				   used, the number of time steps
				   before writing the solution
				   increases with mesh refinement */

  double instat_write_delta;    /* time step for writing to file,
                                   overrides intat_timesteps if !=-1.0
				*/


  FIDX write_mesh;	        /* What to write to file (optional):
			           0 - nothing
				   1 - initial and deformed T2 mesh
				   2 - initial and refined T2 mesh
				   3 - initial and refined T1 mesh */
  FIDX write_ssegs;	        /* 1 - Writing svg-file with the
				       shape parameters (optional) */


};

/* end of datastruc.h */
#endif 
