/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

static inline int matrix_2x2_invert(double *A, int LDA, 
			     double *Ainv, int LDAinv,
			     double *detA
/* returns the inverse of 2x2 matrix A and the determinant of A 
 
   Input:  A       - the matrix A stored column wise, 
                     A(i,j):=A[j*LDA+i]
		     2 by 2 matrix
           LDA     - leading dimension of A
	   LDAinv  - leading dimension of Ainv

   Output: Ainv    - matrix of same size as A, the inverse of A,
                     it is save to use the same pointer as A, 
                     to overwrite A with Ainv.
           detA    - determinant of A ( det(A) ), given by reference


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid

*/
			     )
{
  double A00, A01, A10, A11;
  double detA_loc, detAm1;

  detA_loc=
    +A[0+0*LDA]*A[1+1*LDA]
    -A[0+1*LDA]*A[1+0*LDA];

  *detA=detA_loc;

  if (detA_loc==0.0)
    {
      fprintf(stderr,"matrix_2x2_invert: det(A)==0\n");
      return FAIL;
    }
  
  A00= A[1+1*LDA];
  A01= A[1+0*LDA];
    
  A10= A[0+1*LDA];
  A11= A[0+0*LDA];
    
  detAm1 = 1.0/detA_loc;

  Ainv[0+0*LDAinv]= +detAm1*A00;
  Ainv[1+0*LDAinv]= -detAm1*A01;

  Ainv[0+1*LDAinv]= -detAm1*A10;
  Ainv[1+1*LDAinv]= +detAm1*A11;

  return SUCCESS;
}

static inline int matrix_3x3_invert(double *A, int LDA, 
			     double *Ainv, int LDAinv,
			     double *detA
/* returns the inverse of 3x3 matrix A and the determinant of A 
 
   Input:  A       - the matrix A stored column wise, 
                     A(i,j):=A[j*LDA+i]
		     3 by 3 matrix
           LDA     - leading dimension of A
	   LDAinv  - leading dimension of Ainv

   Output: Ainv    - matrix of same size as A, the inverse of A,
                     it is save to use the same pointer as A, 
                     to overwrite A with Ainv.
           detA    - determinant of A ( det(A) ), given by reference


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid

*/
			     )
{
  double A00, A01, A02, A10, A11, A12, A20, A21, A22;
  double detA_loc, detAm1;

  detA_loc=
    +A[0+0*LDA]*A[1+1*LDA]*A[2+2*LDA]
    +A[0+1*LDA]*A[1+2*LDA]*A[2+0*LDA]
    +A[0+2*LDA]*A[1+0*LDA]*A[2+1*LDA]
    -A[0+2*LDA]*A[1+1*LDA]*A[2+0*LDA]
    -A[0+1*LDA]*A[1+0*LDA]*A[2+2*LDA]
    -A[0+0*LDA]*A[1+2*LDA]*A[2+1*LDA];

  *detA=detA_loc;

  if (detA_loc==0.0)
    {
      fprintf(stderr,"matrix_3x3_invert: det(A)==0\n");
      return FAIL;
    }
  
  A00= ( A[1+1*LDA]*A[2+2*LDA] - A[1+2*LDA]*A[2+1*LDA] );
  A01= ( A[1+0*LDA]*A[2+2*LDA] - A[1+2*LDA]*A[2+0*LDA] );
  A02= ( A[1+0*LDA]*A[2+1*LDA] - A[1+1*LDA]*A[2+0*LDA] );
    
  A10= ( A[0+1*LDA]*A[2+2*LDA] - A[0+2*LDA]*A[2+1*LDA] );
  A11= ( A[0+0*LDA]*A[2+2*LDA] - A[0+2*LDA]*A[2+0*LDA] );
  A12= ( A[0+0*LDA]*A[2+1*LDA] - A[0+1*LDA]*A[2+0*LDA] );
  
  A20= ( A[0+1*LDA]*A[1+2*LDA] - A[0+2*LDA]*A[1+1*LDA] );
  A21= ( A[0+0*LDA]*A[1+2*LDA] - A[0+2*LDA]*A[1+0*LDA] );
  A22= ( A[0+0*LDA]*A[1+1*LDA] - A[0+1*LDA]*A[1+0*LDA] );
    

  detAm1 = 1.0/detA_loc;

  Ainv[0+0*LDAinv]= +detAm1*A00;
  Ainv[1+0*LDAinv]= -detAm1*A01;
  Ainv[2+0*LDAinv]= +detAm1*A02;

  Ainv[0+1*LDAinv]= -detAm1*A10;
  Ainv[1+1*LDAinv]= +detAm1*A11;
  Ainv[2+1*LDAinv]= -detAm1*A12;

  Ainv[0+2*LDAinv]= +detAm1*A20;
  Ainv[1+2*LDAinv]= -detAm1*A21;
  Ainv[2+2*LDAinv]= +detAm1*A22;

  return SUCCESS;
}

/* this definition must come AFTER the other two */
static inline int matrix_dim_x_dim_invert(int dim, double *A, int LDA, 
				   double *Ainv, int LDAinv,
				   double *detA
/* returns the inverse of a 2x2 or 3x3 matrix A and the determinant of A 
 
   Input:  dim     - size of A, only 2 and 3 are allowed
           A       - the matrix A stored column wise, 
                     A(i,j):=A[j*LDA+i]
		     2-by-2 matrix or 3-by-3 matrix
           LDA     - leading dimension of A
	   LDAinv  - leading dimension of Ainv

   Output: Ainv    - matrix of same size as A, the inverse of A,
                     it is save to use the same pointer as A, 
                     to overwrite A with Ainv.
           detA    - determinant of A ( det(A) ), given by reference


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid

*/
				   )
{
  if (dim==2)
    {
      return matrix_2x2_invert(A, LDA, Ainv, LDAinv, detA);
    }
  else if (dim==3)
    {
      return matrix_3x3_invert(A, LDA, Ainv, LDAinv, detA);
    }
  else
    {
      fprintf(stderr,"matrix_dim_x_dim_invert: dim=%d not supported\n",
	      dim);
      return FAIL;
    }
}
