/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "mesh3d.h"
#include "assembly.h"
#include "sparse.h"
#include "lin_solver.h"
#include "gen_aux.h"

#include <unistd.h>
#include <string.h>

/* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }


int main(int argc, char *argv[])
{
  char solver[50]="no_solver_set";
  struct sparse *Ks;
  struct vector rhs, *x, *old_x;
  struct mesh msh;
  struct projector1 P;
  
  struct multilvl ml;
  struct mgdata mg;
  struct coarse_mat *cmat;

  struct solver_settings set;

  int  err, iter, lvl, lmax, level0;
  FIDX i, vx_nr, vx_nr_old;

  double resi, atol, rtol;
  double globest;
  FIDX nrmarked;
  int stop;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t0, ti, t0l;


  if (argc>1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_e1( &msh, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* mesh_info(stdout, &msh); /* */

  TIMEGET;
  t0=ti;


  /* set refinement levels */
  level0  = set.refine_ini;
  lmax    = set.refine_steps;

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_e1( &msh );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_e1, main);

      mesh_info(stdout, &msh);
    }

  /* reset hierarchy */
  msh.hi_nr=0;
  msh.lvl=-1;
  for (i=0; i<msh.eg_nr; i++)
    {
      msh.edge[i*msh.eg_w+MCE1EGLVL]=-1;
    }
  if (set.refine_type!=0)
    {
      err=mesh_prepare_adaptive_green_e1(&msh);
      FUNCTION_FAILURE_HANDLE( err, mesh_prepareadaptive_green_e1, main);
    }

  /* allocate memory for the sparse matrices  */
  TRY_MALLOC( Ks,  lmax+1, struct sparse, main);
  old_x=NULL;
  vx_nr_old=0;
  stop=0;
  lvl=-1;

  while ((stop==0)&&(lvl<=lmax))
    {
      vx_nr=msh.vx_nr;
      lvl++;
      
      /* allocate memory for x, rhs, K, P */
      TRY_MALLOC( x,  1, struct vector, main);
      err=vector_alloc( x, vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
      
      err=vector_alloc( &rhs, vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      err=sparse_flex_alloc( &Ks[lvl], vx_nr );
      FUNCTION_FAILURE_HANDLE( err, sparse_flex_alloc, main);

      err=projector1_alloc( &P, vx_nr );
      FUNCTION_FAILURE_HANDLE( err, sparse_flex_alloc, main);
      
      /* init multilvl data */
      err=multilvl_init_tx( &msh, 1, &ml, 31);
      FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);

      /* on coarse mesh, init x to zero, 
	 on finer meshes interpolate solution from old mesh to new */
      if (lvl==0)
	{
	  for (i=0; i<vx_nr; i++)
	    {
	      (*x).V[i]=0.0;
	    }
	}
      else
	{
	  FIDX node;

	  err=mg_init_tx( NULL, &msh, &ml, &mg, NULL);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
      
	  /* Remark: vx_nr_old is the number of vertices on the coarser
	     mesh (lvl-1), vx_nr is the number on the finer mesh (lvl) */
	  for (i=0; i<vx_nr_old; i++)
	    {
	      MLVLFINDENTRY(node, i, lvl-1, ml);
	      mg.xl[node]=(*old_x).V[i];
	    }
	  
	  err=mg_interpolate_tx( &mg, lvl-1, lvl, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_tx, main);  
	      
	  /* now copy back to the solution vector */    
	  for (i=0; i<vx_nr; i++)
	    {
	      MLVLFINDENTRY(node, i, lvl, ml);
	      (*x).V[i]=mg.xl[node];
	    }
	  mg_free(&mg);

	  /* if (lvl==lmax)
	     {
	     err= mesh_write_solution_vtk_e1(&msh, NULL, 0, x, 1, 17+1,
	     "visual/mesh3D" );
	     FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t1, main);
	     }/* */
	}

      TIMEGET;
      t0l=ti;

      /* assembly of equation system */
      err=assem_poisson_e1( &Ks[lvl], &rhs, x, &P, &msh );
      FUNCTION_FAILURE_HANDLE( err, assem_poisson_e1, main);

      /* err=sparse_mat_write_file(&Ks[lvl], "visual/3D_poisson_mat.txt");
      FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file, main); /* */


      /* define the multigrid data */
      err=mg_init_tx( Ks, &msh, &ml, &mg, &P);
      FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);

      /* use coarse grid solver */
      if (lvl==0)
	{
	  /* generate the coarse grid matrix */
	  FIDX *coarse_bc_nodes;
	  FIDX n_c_bc=0;
	  TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
	  TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
	  for (i=0; i<P.len; i++)
	    {
	      FIDX dof, child;
	      dof=P.V[i];
	      MLVLFINDENTRY(child, dof, 0, ml);
	      if (child>=0)
		{
		  coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
		  n_c_bc++;
		}
	    }
	  err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			      1, cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	  free(coarse_bc_nodes);
	}

      /* define solver parameter for the V-cycle */
      mg.vcycles=1;
      mg.smooths=1;
      mg.CGC_scale=1.0;
      mg.cmat=cmat;

      atol = set.solver_atol;
      if (lvl == 0)
	{
	  rtol = set.solver_ini_rtol;
	}
      else
	{
	  rtol = set.solver_ref_rtol;
	} /* */


      /* solve equation system */
      resi=0.0;
      iter=0;
      
      /* strcpy(solver,"PCG_no");
	 err=PCG( 10000, 2, atol, rtol, 1, x, &resi, &iter, sparse_mul_mat_vec, 
	 projector1_no_precon, &Ks[lvl], &rhs, &P );
      /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */

      strcpy(solver,"PCG_MG");
      err=PCG( 10000, 2, atol, rtol, 1, x, &resi, &iter, sparse_mul_mat_vec, 
	       gen_proj_MG_tx, &Ks[lvl], &rhs, &mg );
      FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */

      TIMEGET;
  
      printf("%s: %4d iterations, |res|=%8.2e, vx_nr= %9"dFIDX",  "
	     "time_level=%10.3e, time_total=%10.3e\n",
	     solver, iter, resi, vx_nr, ti-t0l, ti-t0); /* */


      mg.cmat=NULL; /* prevent it from beeing destroyed, so we don't
		       need to recompute it every time */
      mg_free(&mg);
      multilvl_free(&ml);


      projector1_free(&P);
      vector_free(&rhs);

      /* Apply mesh refinement */
      {
	FIDX *marker;
	TRY_MALLOC( marker, msh.el_nr, FIDX, main);
	switch(set.refine_type)
	  {
	  case 0:
	    /* Uniform mesh refinement */
	    globest=10*set.refine_stop; /* init to save value */
	    /* OPTIONAL: estimating the error */
	    /* err=lame_error_est_residual_t2(&msh, x, lambda, mu,
	       marker, &nrmarked, &globest,
	       &set, 2); 
	       FUNCTION_FAILURE_HANDLE( err, lame_error_est_residual_t2,
	       main);
	    */
	    /* err=gen_error_est_ZZ_tx(&msh, 1, x, 
	       marker, &nrmarked, &globest, &set, 1); 
	       FUNCTION_FAILURE_HANDLE( err, gen_error_est_ZZ_tx, main);/* */
	    
	    if ((lvl>=lmax)||
		((globest<set.refine_stop)||(8*vx_nr>set.refine_max_vx)))
	      {
		stop=1;
		lmax=lvl;
	      }
	    else
	      {
		/* refine the mesh */
		err=mesh_refine_uniform_e1( &msh);
		FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_e1, main);
	      }
	    break;
	    /* case 1: */
	    /* no residual estimator yet for this problem */
	    /* Adaptive mesh refinement , residual error estimator */
	    /* err=lame_error_est_residual_t2(&msh, x, lambda, mu,
	       marker, &nrmarked, &globest,
	       &set, 2); 
	       FUNCTION_FAILURE_HANDLE( err, lame_error_est_residual_t2, main);
	       
	       if ((lvl>=lmax)||
	       ((globest<set.refine_stop)||
	       (vx_nr+1*nrmarked>set.refine_max_vx)))
	       {
	       stop=1;
	       lmax=lvl;
	       }
	       else
	       {
	       /* refine the mesh */
	    /* err=mesh_refine_adaptive_e1( &msh, marker);
	       FUNCTION_FAILURE_HANDLE( err, mesh_refine_adaptive_e1, main);
	       }
	       break;*/
	  case 2:
	    /* Adaptive mesh refinement, ZZ error estimator */
	    err=gen_error_est_ZZ_tx(&msh, 1, x, 
				    marker, &nrmarked, &globest, &set, 1); 
	    FUNCTION_FAILURE_HANDLE( err, gen_error_est_ZZ_tx, main);

	    /* #warning "for tests, mark all elements"
	       for (i=0; i<msh.el_nr; i++)
	       marker[i]=1;
	       nrmarked=msh.el_nr; /* */
	    
	    if ((lvl>=lmax)||
		((globest<set.refine_stop)||
		 (vx_nr+1*nrmarked>set.refine_max_vx)))
	      {
		stop=1;
		lmax=lvl;
	      }
	    else
	      {
		/* refine the mesh */
		err=mesh_refine_adaptive_e1( &msh, marker);
		FUNCTION_FAILURE_HANDLE( err, mesh_refine_adaptive_e1, main);
	      }
	    break;
	  default:
	    printf("Error in main (test_assme3D.c): unknown refinement type!"
		   " type = %"dFIDX"\n",  set.refine_type);
	    return FAIL;
	  }
	free(marker);
	
	/* mesh_info(stdout, &msh); /* */
      }

      if (stop==1)
	{
	  err= mesh_write_solution_vtk_e1(&msh, NULL, 0, x, 1, 17+1,
					  "visual/solution" );
	  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t1, main);/* */
	}

      /* keep current solution vector as old solution */
      if (old_x!=NULL)
	{
	  vector_free(old_x);
	  free(old_x);
	}
      old_x=x;
      vx_nr_old=vx_nr;
      x=NULL;
    } /* for lvl */


  /* free remaining memory */
  vector_free(old_x);
  free(old_x);
  
  coarse_mat_free(cmat);
  free(cmat);

  for (lvl=lmax; lvl>=0; lvl--)
    {
      sparse_free(&Ks[lvl]);
    }
  free(Ks);

  mesh_free(&msh);

  return 1;
}
