/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
stokesassem.h
*/
#ifndef __STOKESASSEM_H__
#define __STOKESASSEM_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"
#include "navsto_struct.h"


int stokes_assem_t21(struct navsto_matrix *K, struct vector *rhs,
		     struct vector *u0, 
		     struct mesh *m, FIDX lvl, int noFs
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Stokes equation on the T2 mesh m, such that 
       K x = rhs
   defines the approximate solution x=(u_x, u_y, p)  of the Stokes equation
       -nu Laplace(u) + grad p = f
                         div u = 0

   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  m       - the mesh
           lvl     - specifying the current refinement level, used to
                     define where in K.Fs[] the matrices are stored
           noFs    - specifies if the the Fs blocks of K are to be
                     computed or not, noFs==1 ==> they are not,
                     noFs==0 they are computed

   Output: K       - stiffness matrix and preconditioner data in
                     specialised storage, the structure K needs
                     to be given with internal data initialised,
                     here the given level is initialised and memory
                     allocated as necessary
           rhs     - righthand side vector, only the structure rhs 
	             needs to be given, all internal data is
	             initialised here and memory allocated 
	   u0      - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC, only the
 	             structure u0 needs to be given, all internal
 	             data is initialised here and memory allocated 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/ );

int stokes_diri_bc_init_t2( struct mesh *m, struct vector *uinit, 
			    struct vector *rhs, struct navsto_matrix *K
/* initialises the solution vector uinit to fulfill the Dirichlet
   boundary conditions, it is verified that the boundary conditions
   fulfil balancing in and outflow

   In/Out: m       - the mesh (2d T2), mostly input, only the boundary
                     condition data in the hierarchy entries is corrected
           uinit   - the solution vector, stored as uinit[j*vx_nr+i]
                     gives the j-th velocity component in the i-th
                     node of the mesh, the velocity at dirichlet BC -
                     nodes is set to the bescribed value, 
		     if uinit==NULL it is ignored
           rhs     - rhighthandside vector, the entries in lines
                     corresponding to Dirichlet BC are set to zero,
		     if rhs==NULL it is ignored
           K       - stiffness matrix struct, only K.bn_nr and K.nodes
                     are set (list of Dirichlet BC nodes)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int stokes_pdof_init_t21( struct mesh *m, struct navsto_matrix *K
/* initialises the pdof part of K and the projector data for the
   pressure space (weights for the pressure nodes)

   Input:  m       - the mesh (2d T2), mostly input, only the boundary
                     condition data in the hierarchy entries is corrected
   Output: K       - stiffness matrix struct, K.pvx_nr, K.pdof and
                     K.weight are set

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int stokes_ass_streamfunc_poisson_t2(struct sparse *K, struct vector *rhs,
				    struct vector *psi0, struct projector1 *P,
				    struct mesh *m, struct vector *usol
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs for the recovery of the streamfunction form the
   velocty field uslo and the finite element discretisation of the
   equation
       -Laplace psi = dv/dx - du/dy
   on the mesh m, such that 
       K psi = rhs

   Input:  m       - the mesh (2d T2)
           usol    - the velocity field stored as usol[j*vx_nr+i]
                     gives the j-th velocity component in the i-th
                     node of the mesh 

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   psi0    - initial guess for the solution of the equation
 	             system which satisfies the Dirichlet BC
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
