/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#ifndef SPARSE_STRUCT_H
#define SPARSE_STRUCT_H


/* sparse matrix data structure */
/* 23.03.2007 rework to compressed row structure,
   matrix will have two stages of buildup:
   1. ultra flexible datastructure that allows arbitrary addition and
      deletion of entries, based on simply connected lists
   2. compressed row structure, once the matrix is assembled it will
      be compressed to reduce indirect adressing issues 
      
   4.9.2011 adding grouping (coloring) of dofs to allow parallel 
   gauss-seidel smoother 
*/
#define SP_TYPE_UNDEF -1 
#define SP_TYPE_FLEX 1 
#define SP_TYPE_COMPROW 2 
struct sparse {
  int  type;          /* describes the actual storage type of the
			 current matrix:
			 type==SP_TYPE_FLEX: flexible type 
                                (for assembly, as long as connectivity
                                 of matrix is not yet known),
			 type==SP_TYPE_COMPROW: compressed row
			 storage, faster for calculations */
  FIDX row_nr;        /* actual number of rows */

  /* data structure for compressed row sotrage, only used when type==2 */
  FIDX *rows;         /* array containing the index of the begin of
			 each row in both cols and A, length= (row_nr+1),
			 meaning:
			 k:=rows[i], m:=rows[i+1], i<row_nr
			 => i-th row is stored in cols[k...m-1] and
			    A[k..m-1] */
  FIDX *cols;         /* array containing the columns to which entries
			 in the matrix correspond,
			 length=rows[row_nr], meaning:
			 j:=cols[rows[i]+k], k<rows[i+1]-rows[i],
			                     i<row_nr,
			 => A[rows[i]+k] contains A(i,j) */
  double *A;          /* array containing the actual matrix
			 entries, length= (rows[row_nr]), 
			 interpretation as described above in
			 comment on cols (and rows) */


  /* flexible storage stage, only used when type==1 */
  struct int_double_list *flex_rows;
                      /* array containing the roots of the simply
			 connected lists for each row,
		         each row will be sorted as it is created,
		         with the root element being the dioagonal
		         entry */
#ifdef HAVE_OPENMP
  /* coloring of dofs in matrix */
  FIDX n_col_buckets; /* number of colors in the coloring of dofs */
  FIDX *color_buckets;/* index into color_dofs defining the begin and
			 end of each color: 
			 i=color_buckets[j] gives the first entry (i) 
			    of color_dofs of color j, all following are
			    also of color j, until the k-th:
			 k=color_buckets[j+1]-1, so k marks the end of
			 this color */
  FIDX *color_dofs;   /* vector if dofs that belong to the colors, 
			 ordered by color */  
#endif
};


/* simply connected list of storing one FIDX and one double */
struct int_double_list
{
  FIDX   ix;          /* FIDX datum */
  double dx;          /* double data */
  struct int_double_list *succ;
                      /* successor element in the list,
		         if ==NULL no successor yet */
};


/* vector data structure */
struct vector {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  double *V;          /* pointer to array of length n_max, such that 
			 V[i] gives the i-th component of the vector
		      */
};

/* simple projector data structure */
struct projector1 {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  FIDX *V;            /* pointer to array of length n_max, such that 
			 V[i] tells which components of a vector the
			 projector modifies (sets to zero),
			 V[i]=j ==> j-th component is set to zero
		      */
};


/* general transpose / notranspose stuff */
enum transposetype {
  NoTrans=0,  /* no transpose */
  Trans=1     /* transpose    */
};



/* data structure for coarse grid matrix factorizations */
struct coarse_mat
{
  int nr;             /* number of degrees of freedom (DOFs) covered
			 by this matrix */
  int band_w;         /* bandwidht of the matrix */
  int symm;           /* marking the symmetry properties of the
			 matrix */
  FIDX *nodes;        /* pointer to an array of node numbers to which
			 the DOFs correspond, represents a
			 permutation, which is used to reduce fill in
			 in the trinagulation */
  double *A;          /* the triangular factors of the matrix
			 themself, 
			 if symm==1 stored as band matrix of dimension
			       nr, with bandwidth band_w, stored in
			       the format as produced by the LAPACK
			       routine dpbtrf with upload='U',
			       LDAB=KD+1 and KD=band_w, note that the
			       i-th DOF of the system corresponds to
			       the node nodes[i] 
			 if symm==2 stored as by  LAPACK
			       routine dgbtrf with KU=KL=band_w,
			       LDAB=2*KL+KU+1, note that the i-th DOF
			       of the system corresponds to the node
			       nodes[i] */ 
  double *help;       /* a help vector of size dofs */
  int *ipiv;          /* integer vector of length dofs, to store the
			 pivot indices, only used if symm=2, otherwise
			 the pointer will be ==NULL */
};


/* data structure for adjency-, level- and similar lists of integers */
struct int_list
{
  FIDX ixn;           /* actual length of the index vector ix */
  FIDX ixnmax;        /* maximal length of the index vector ix */
  FIDX *ix;           /* index vector, j=ix[i] marks the begin of the
			 i-th part of el, k=ix[i+1]-1 marks the end of
			 the i-th part of el, i<ixn<=ixnmax */
  FIDX elnmax;        /* maximal length of el, therefore 
			 ix[ixn+1]-1<=elnmax has to be asured */
  FIDX *el;           /* list of elements, with j and k as described
			 in the ix part, el[j+t] is the t-th element
			 of the i-th part of the list, t<=k-j-1 */
};



#endif
