/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#ifndef NAVSTO_STRUCT_H
#define NAVSTO_STRUCT_H

#include "datastruc.h"
#include "sparse_struct.h"

/* Navier-Stokes matrices data structure, also used in the preconditioner */
struct navsto_matrix {
  FIDX dim;           /* problem dimension */
  /* stuff needed for the matrix multiplication itself */
  FIDX vx_nr;         /* number of vertices in the corresponding mesh
		      */
  FIDX pvx_nr;        /* number of pressure nodes */
  FIDX *pdof;         /* pointer to array of length vx_nr, such that
			 pdof[i] tells which (if any) pressure degree
			 of freedom (DOF) belongs to the node,
			 pdof[i]=j, j>=0 ==> j-th pressure dof belongs
			 to the i-th node, j<0 ==> no pressure dof
			 associate */
  FIDX lvl;           /* current level */
  FIDX lvlmax;        /* max number of levels */
  struct sparse *Fs;  /* pointer to array of sparse matrices, which
			 form the velocity-velocity block the stiffness
			 matrix for a given level, changes after each
			 iteration of the nonlinear outer solve,
			 let 
			   A  = a(:,:)
			   Cw = c(u^(m-1),:,:)
			   Cu = c(:,u^(m-1),:)
			 then
			   Fs[lvl] = A+Cw     if Picarditeration is used and
			   Fs[lvl] = A+Cw+Cu  if Newtons Method is used
		      */
  struct sparse *Bs;  /* pointer to array of sparse matrices which
			 form the B_i blocks of the stiffness matrix 
		      */

  /* boundary projection data */
  FIDX bn_nr;         /* number of boundary nodes */
  FIDX *nodes;        /* pointer to array of length vx_nr, such that 
			 nodes[i] tells which nodes belong to the
			 boundary, the projector sets the velocity
			 components for these nodes to zero, pressure
			 is projected to give zero mean over the
			 boundary
			 nodes[i]=j ==> j-th node is boundary node
			 bn_nr tells how many such entries i are valid
		      */
  double *weight;     /* pointer to array of length vx_nr, such that 
			 weight[i] denotes the weight which node
			 nodes[i] contributes to the to the mean
			 pressure on the boundary, 
			 bn_nr denotes the number of valid entries,
			 sum(weight[i]) has to be =1.0,
			 sum(weight[i]*p[i]) gives the mean pressure
			 on the boundary
		      */

  /* preconditioner data */
  struct sparse M;    /* the mass matrix for the pressure space times
			 nu */ 
  struct vector Minvdiag;
                      /* inverse of the diagonal of M */
  struct sparse Ap;   /* stiffness matrix of a pressure space Laplace
			 problem, needed for the Wathen block
			 preconditioner (for the systems in the Picard
			 iteration) */ 
  struct sparse Cp;   /* the advection part of the pressure space
			 advection diffusion operator, scaled by 1/nu
			 such that 
 			   M^-1 * (Ap + Cp) * Ap^-1 gives the
			 desired preconditioner
		     */

  struct coarse_mat cmat;
                      /* coarse grid solver data for the F part on the
			 coarsest mesh) */ 

  FIDX cmat_Ap_vxnr;
  FIDX cmat_Ap_pvxnr;
  struct coarse_mat *cmat_Ap;
  FIDX *cmat_Ap_pdof;
                      /* coarse grid solver data for the Ap (pressure
			 space Laplacian) on the coarsest mesh */


  struct mesh *msh;   /* pointer to the mesh (to pass hierarchy info
			 to the preconditioner) */

  struct multilvl *ml1, *mld;
                      /* datastructure to operate with a vector
			 containing the vector of nodal data on each
			 individual level, needed by BPX and MG, ml1
			 is for those parts which require only one
			 datum per node, mld is for those who require
			 dim parts per node */
  struct bpxdata *bpx;/* pointer to a structure holding the data for
			 the PBX preconditioner */
  struct mgdata *mg, *mg1;
                      /* pointer to structure holding data for the
			 multigrid preconditioner */

  void *AMGdataFs;    /* data for AMG preconditioner for Fs (if used) */
  void *AMGdataAp;    /* data for AMG preconditioner for A_p (if used) */

  int innercount[5];  /* counter for the inner iterations of the
			 preconditioner */
  double innereps;    /* stopping criterion for solvers inside the
			 preconditioner */
  int innersteps[5];  /* to specify fixed number of steps for each of
			 the inner solvers */
  double *MGscale;    /* to store the scaling factor, which is applied
			 to the coarse grid correction */

  FIDX **mlsorters;   /* two multilevel arrays, each sorting the dofs
			 by the coordinates of the mesh vertices,
			 mlsorters[0] sorts by x, mlsorters[1] by y */ 

  /* box-smoother data */
  FIDX *vx_nr_lvl;    /* vertex number for each level */
  struct ilist*** pdof_vels;
                      /* for each level and each pdof this array holds
			 a list of related velocity nodes,
		         pdof_vels[lvl][pdof] ist the pointer to the
		         head of the list of pdof of level lvl */  
  FIDX max_n_pdof_vels;/*maximum number of local velocities */
  FIDX* pvx_nr_lvl;   /* the number of pdofs for each level,
			 pvx_nr_lvl[lvl] gives the number of pdofs for
			 level lvl */ 
  FIDX** pdof_lvl;    /* the pdof field for each level,
			 pdof_lvl[lvl] gives the pdof array for level
			 lvl */ 
  double **weight_lvl;/* weight_lvl[i] give the weight vector for
			 level i (see above) */
  double **vel_weight;/* weights for the velocity dofs associated with
			 each node, these are chosen sucht that the
			 negative influence of multiple appearance of
			 a node is reduced */
  FIDX *bn_nr_lvl;    /* number of boundary nodes per level*/
  FIDX **nodes_lvl;   /* nodes_lvl[i] gives the nodes vector for
			 level i, lenght bn_nr_lvl[i]  */

  FIDX *pdof_sorter_n;/* lenght of the pdof_sorter arrays bellow, 
			 pdof_sorter_n[lvl] gives the lenght of 
			 pdof_sorter[lvl][.] */
  FIDX **pdof_sorter; /* for each level a permutation of the pdofs
			 that should idealy be a stream-wise ordering,
			 pdof_sorter[lvl][i] gives the i-th ordered
			 pdof for level lvl,
			 for initial test we do just sort them by
			 ascending x coordinates, then by y coordinates */

  /* only needed for elem-based-smoother */
  FIDX **elem_lvl;
  FIDX * el_nr_lvl;
  FIDX **bc_marker_lvl;
  FIDX *elem_sorter_n;/* lenght of the elem_sorter arrays bellow, 
			 elem_sorter_n[lvl] gives the lenght of 
			 elem_sorter[lvl][.]  */
  FIDX **elem_sorter; /* for each level a permutation of the elements
			 that should idealy be a stream-wise ordering,
			 elelm_sorter[lvl][i] gives the i-th ordered
			 element name for level lvl,
			 for initial test we do just sort them by
			 ascending x coordinates, then by y coordinates */

};


#endif
