/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
navsto_adj.h
*/
#ifndef __NAVSTO_ADJ_H__
#define __NAVSTO_ADJ_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"
#include "navsto_struct.h"


int navsto_adj_convert(struct navsto_matrix *K
/* converts the system matrix of the Navier-Stokes system into its 
   transpose, including all lower level matrices (for multi grid),
   such that it can be used for an adjoint solve,

   the converting is not reversible!
   
   K = K^T;

   In/Out: K       - linearised Navier-Stokes-problem stiffness
                     matrix, must be the Newton linearisation!

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_adjmat_tim_vec(void *arg1, struct vector *vec,
			  struct vector *out
/* multiplies the matrix K from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - linearised Navier-Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_adj_w_precon(void *arg1, struct vector *in,
			void *notused,
			struct vector *out
/* performs preconditioning for the adjoint equation of Navier-Stokes
   problems,  

     out = C^-1 * in

   where C^-1 is the Wathen block preconditioner:
        
   C^-1=[I                   ] [ I   ] [F^-1  ] velocity space
        [  (Mp^-1)*Fp*(Ap^-1)]*[-B  I]*[     I] pressure space

   where F is the advection diffusion operator, Mp is the pressure
   space mass matrix, Fp the pressure space advection diffusion
   operator, Ap the pressure space laplace operator
   (F, Fp and Ap are discretisations of these operators)

   Input:  arg1=
           K       - Navier-Stokes-matrix struct
           in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), C^-1* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_adj_vel_MG(void *arg1, struct vector *in,
		      void *notused, struct vector *out
/* performs the boudary condition projection and multigrid
   preconditioning for the advection diffusion subproblems of the
   Wathen block preconditioner for navier stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there), and C^-1 repressents the action of the multigrid
   preconditioning, that is one V cycle
   
   Input:  arg1=
           K       - navsto_matrix struct
           in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int navsto_Psi_dRdF_t21(struct vector *Psi_dRdF, 
			struct vector *Psi,
			struct vector *sol,
			struct navsto_matrix *K,
			struct mesh *msh
/* evaluates the Psi times the derivative of the residual with respect
   to the position, the residual is defined as the finite element
   discretisation (with P2-P1 triangles) of the Navier-Stokes equation

         -nu Laplace(u) + u * grad u + grad p = f
                                        div u = 0

   with boundary conditions as given in K,

   The evaluation of the derivatives via the discrete adjoint method 
   requires the evaluation of

     DI   dI         dR
     -- = -- - Psi^T -- .
     DF   dF         dF

   This routine provides the 

                     dR
               Psi^T --
                     dF

   part. (Note that the "-" is omitted!)

   In/Out: Psi     - the adjoint solution vector for the NS-system,
                     it is mostly read only, but the values at
                     dirichlet nodes are set to zero,
		     SO IT IS MODIFIED!
	   
   Input:  sol     - the solution vector of the NS-system (where the
                     system is to be linearized)
	   K       - the Navier-Stokes matrix struct, used to get the 
	             pressure dofs and boundary conditions
	   m       - the mesh, the shape defining node positions are
	             marked in there as well

   Output: Psi_dRdF- Psi times jacobian matrix of the residual with
                     respect to the node positions, such that
                     Psi_dRdF[d*vx_nr+k] presents derivative values
                     respect to the d-th component of the k-th node,
		     thus this vector has to be of length dim*vx_nr,


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int navsto_dIdX_t21( struct mesh *msh, struct navsto_matrix *K,
		     struct vector *sol,
		     FIDX pc_nr, struct vector *pcvec,
		     struct vector *dIdsol, 
		     struct vector *dIdF,
		     struct vector *ref_sol
/* evaluates the performance criteria (given with the mesh) for the
   supplied solution vector sol, stores the values in pcvec

   Input:  msh       - the mesh
           K         - navsto_matrix struct, the pdof part is needed
           sol       - vector containing the solution of a Stokes or
                       Navier-Stokes problem, calculated using the
                       T2/T1 element pair
	   pc_nr     - number of performance criteria, has to be the
	               same as in the mesh file, pcvec, dIdsol and
	               dIdF have to provide space for at least pc_nr
	               criteria 
           ref_sol   - vector containing the reference solution, must
                       be compatible with sol (see above, same mesh)
		       if not required, ref_sol=NULL may me specified

   Output: pcvec     - vector containing the values of the performance
	               criteria, has to be allocated by the
	               calling routine, length >= pc_nr
	   dIdsol    - derivatives of the performance with respect to
	               the solution vector, array of px_nr vectors of
	               length bigN (bigN=sol.len), dIdsol[i].V[j] is
	               the derivative of the i-th component of the
	               performance vector wrt the j-th component of
	               the solution vector, not used if ==NULL
           dIdF      - derivatives of the performance with respect to
	               the node positions, array of pc_nr vectors of
	               length dim*vx_nr, dIdF[i].V[d*vx_nr+j] is the
	               derivative of the i-th component of the
	               performance vector wrt the d-th coordinate of
	               the j-th node, not used if ==NULL

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
