/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
mesh3d.h
*/
#ifndef __MESH3D_H__
#define __MESH3D_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"
#include "mesh.h"

int mesh_write_solution_vtk_e1( struct mesh *m, struct vector *vec_x,
				FIDX vcompo, struct vector *scalar_x,
				FIDX scompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a vtk readable
   file for visualisation, 
   this can for example be used with paraview
           http://www.paraview.org/

   Input:  m         - the mesh
           vec_x     - vector with data associated to the nodes,
	               for representing a vector field of dimension
	               vcompo, ignored if vec_x==NULL
	   vcompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector x have to be numbered such that
	               vec_x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh, has to be TWO for t1
	               mesh, ignored if vec_x==NULL
           scalar_x  - vector with data associated to the nodes,
	               for representing scompo scalar fields,
		       for each scalar field a separate part is
		       written in the file, ignored if scalar_x==NULL
	   scompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector scalar_x have to be numbered such that
	               saclar_x[j*vx_nr+i] gives the j-th scalar
	               variable in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_read_file_e1( struct mesh *m, struct solver_settings *set, char *name
/* checks the type of mesh file and calls the appropriate reader

   input:   name    - string containing the file name of the file to
                      be read, should end in ".f1m" (FEINS mesh)
		      or ".msh" (gmsh, corresponding files will
		      automatically be looked for as well)
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_read_file_f1m_e1( struct mesh *m,
			   struct solver_settings *set, 
			   char *name
/* reads a mesh file, for a description of the file format, see the
   example file "mesh_example.f1m"

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_read_elements_e1( FILE *in, struct mesh *m, FIDX el_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            el_nr   - the number of elements in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the elem and face
                      part are allocated here, but only elem set!
                      in face only the RHSF part is defined and PCVL
                      is initiated
   

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_boundary_e1( FILE *in, struct mesh *m, FIDX bd_nr, FIDX sg_nr
/* reads the boundary part of a mesh file, for a description of the
   file format, see the example file "mesh_example.f1m" (!!!! should be 
   replaced by an 3D example file)

   input:   in      - a file open for reading
            bd_nr   - the number of boundary entries in the file, as
                      given by the header
            sg_nr   - the number of shape segment entries in the file,
	             		 as given by the header
   
   output:  m       - (given by reference) a mesh, the face and bound
                      part are allocated and set here, but in face
                      only those faces are defined which are part of
                      the boundary!

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_remove_unused_vertices_e1( struct mesh *m
/* removes unused vertices, by renumbering all vertices

   In/Out:  m       - (given by reference) a mesh, the vertex, edge
                      and elements parts are modified if necessary

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_compute_conns_e1( struct mesh *m
/* builds the part of the connectivity data of the internal mesh
   format that is not covered by the mesh files (see the example file
   "mesh_example.f1m")

   In/Out:  m       - (given by reference) a mesh, the edge and face
                      parts are completed

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 );

int mesh_read_file_gmsh_e1( struct mesh *m,
			    struct solver_settings *set, 
			    char *name
/* reads a mesh file generated with gmsh

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_gmsh_project_nodes_e1( struct mesh *m, 
				struct ilist *edgeVL, 
				struct ilist *faceVL
/* calls gmsh_closest_points to move all nodes in edgeVL and faceVL to
   their geometry

   In/Out: m        - the mesh, is modified accordingly, 
                      only the coordinates of the vertices
		      in edgeVL and faceVL are modified

   input:  edgeVL   - list of new nodes that have a designated EDGE 
	              geometry number, two list elements per node:
	              first the node number, second the geometry tag/id,
	   faceVL   - list of new nodes that have a designated FACE
	              geometry number, two list elements per node:
	              first the node number, second the geometry tag/id,

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);

int mesh_refine_uniform_e1( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m       - the mesh, is modified accordingly, e.g. new
                     elements, volumes, faces, edges, vertices, boundary
                     elements and hierarchy entries are created,
		     the element el and the volume el are thereby
		     replaced by one of their children, the faces and edges
		     remain to allow other elements to stay intact,
		     if necessary the fields of the mesh are
		     reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_element_e1( struct mesh *m , FIDX el,
			    struct ilist **tail_GEVL,
			    struct ilist **tail_GFVL
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, volumes, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the faces and
		      edges remain to allow other elements to stay
		      intact, if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_element_e1_innerels( struct mesh *m, FIDX oldvol,
				     FIDX *faces1, FIDX *faces2, 
				     FIDX *newvols
/* the faces1 and faces2 together describe an octahedron, 
   for this the shortest diagonal is found, and introduced as new 
   edge, four new faces (triangles) are created with this diagonal
   edge, then three new volumes and the volume vo is overwritten,
   the mesh is modified accordingly, all necessarry new parts are
   created

   Input:  faces1   - array of 4 face indices, four arbitrary faces of
                      the octahedron
	   faces2   - array of 4 face indices, the remaining faces of
                      the octahedron
           oldvol   - original volume, which is refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      volumes, faces and edges are created,
		      of the mesh are reallocated to have space for
		      the new stuff 

   Output: newvols  - array of 3 volume indices, the ones which are
                      created new, the fourth new volume overwrites
                      the original volume vo


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_elem_from_vols_e1( struct mesh *m , FIDX el
/* for volume el the element is redefined by finding the nodes that
   belong to the volume

   Input:  el       - the element to be redefined

   In/Out: m        - the mesh, is modified accordingly, only the elem
                      part corresponding to element el is overwritten

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_split_face_e1( struct mesh *m , FIDX fc,
			FIDX *voldnod, 
			FIDX *nodnewface, FIDX *nodopsnewedge,
			FIDX *newcenterface,
			struct ilist **tail_GEVL,
			struct ilist **tail_GFVL
/* refines the face fc of the mesh m, if it is not refined already,
   all necessarry new parts are created

   Input:  fc       - the face to be refined
           voldnod  - array of 4 node indices, to which nodnewface and
                      nodopsnewedge

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      old faces and edges remain to allow other
		      elements to stay intact, if necessary the fields
		      of the mesh are reallocated to have space for
		      the new stuff 
           nodnewface 
                    - array (4*3), modified such that
                      nodnewface[node*3+k]=newface
                      if newface has vertex voldnod[node], k is the
                      first still available of (0,1,2)
           nodopsnewedge 
                    - array (4*3), modified such that
                      nodopsnewedge[node*3+k]=newedge
                      if newedge is oposing vertex voldnod[node] in a
                      new face, k is the
                      first still available of (0,1,2)
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry

   Output: newcenterface
                    - the central face of the four new faces created
                      by splitting old face fc		      


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_split_edge_e1( struct mesh *m , FIDX eg,
			struct ilist **tail_GEVL
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring faces
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_split_face_green_e1( struct mesh *m , FIDX fc,
			      struct ilist **tail_GEVL,
			      struct ilist **tail_GFVL
/* refines the face fc of the mesh m with Baensch-Green
   refinement (bisection), all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  fc       - the face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the old faces and edges
		      remain to allow elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_element_green_e1( struct mesh *m , FIDX el,
				  struct ilist **tail_GEVL,
				  struct ilist **tail_GFVL
/* refines the element and volume el of the mesh m with Baensch-Green
   refinement (bisection), all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  el       - the element to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the old faces and edges
		      remain to allow elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff
	   tail_GEVL- list of new nodes that have a designated EDGE 
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry
	   tail_GFVL- list of new nodes that have a designated FACE
	              geometry number, if an node is produced during
	              refinement, then two list elements are appended:
	              first the node number, second the geometry tag/id,
		      later this list should be used to move the nodes
		      onto their geometry


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int mesh_refine_adaptive_e1( struct mesh *m, FIDX *marked_elem
/* refines all marked elements of the mesh m with the
   Baensch-Green refinement, all necessarry new parts are
   created thereby,

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991


   In/Out: m           - the mesh, is modified accordingly, e.g. new
                         elements, volumes, faces, edges, vertices, boundary
                         elements and hierarchy entries are created,
  		         the element el and the volume el are thereby
		         replaced by one of their children, the faces
		         and edges remain to allow other elements to
		         stay intact, 
		         if necessary the fields of the mesh are
		         reallocated to have space for the new stuff
	   marked_elem - Vector holding which elements are to be
	                 refined (1 = to be refined)
   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
                         valid
*/);

int mesh_prepare_adaptive_green_e1( struct mesh *m
/* makes sure the connectivity of the mesh is suitable for adaptive
   green refinement (Baensch), i.e. that the longest edge in each face
   is the first edge of the face, with ties being broken by preferring
   the edge with higher number

   
   in/out:  m       - (given by reference) a e1 mesh,

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/);
#endif
