/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE mesh.c
HEADER mesh.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"
#include "sparse_struct.h"
*/





/* prototypes of external functions */
#define _USE_MATH_DEFINES 
#include <math.h>
#ifndef __USE_GNU
//#define __USE_GNU
#endif
#include <string.h>

#include "sparse.h"

/* function prototypes */
#include "mesh.h"

/*FUNCTION*/
void mesh_init( struct mesh *m
/* initialises the mesh to be empty,

   Output: m        - an empty mesh, all inner pointers set to NULL,
                      array lengths etc. to zero
*/
		){
  (*m).dim     = 0;
  (*m).problem = 0;

  (*m).vx_w   = 0;
  (*m).vx_nr  = 0;
  (*m).vx_max = 0;
  (*m).vertex = NULL;
			
  (*m).el_w   = 0;
  (*m).el_nr  = 0;
  (*m).el_max = 0;
  (*m).elem   = NULL;

  (*m).eg_w   = 0;
  (*m).eg_nr  = 0;
  (*m).eg_max = 0;
  (*m).edge   = NULL;

  (*m).fc_w   = 0;
  (*m).fc_nr  = 0;
  (*m).fc_max = 0;
  (*m).face   = NULL;

  (*m).vo_w   = 0;
  (*m).vo_nr  = 0;
  (*m).vo_max = 0;
  (*m).vols   = NULL;

  (*m).hi_w   = 0;
  (*m).hi_nr  = 0;
  (*m).hi_max = 0;
  (*m).hier   = NULL;

  (*m).eh_w   = 0;
  (*m).eh_nr  = 0;
  (*m).eh_max = 0;
  (*m).elhier = NULL;
  
  (*m).lvl = -1;

  (*m).bd_w   = 0;
  (*m).bd_nr  = 0;
  (*m).bd_max = 0;
  (*m).bound = NULL;

  (*m).ps_w   = 0;
  (*m).ps_nr  = 0;
  (*m).ps_max = 0;
  (*m).pcsurf = NULL;

  (*m).pv_w   = 0;
  (*m).pv_nr  = 0;
  (*m).pv_max = 0;
  (*m).pcvol = NULL;

  (*m).pc_w   = 0;
  (*m).pc_nr  = 0;
  (*m).pc_max = 0;
  (*m).pccrit = NULL;

  (*m).fu_w   = 0;
  (*m).fu_nr  = 0;
  (*m).fu_max = 0;
  (*m).func   = NULL;

  (*m).pa_w   = 0;
  (*m).pa_nr  = 0;
  (*m).pa_max = 0;
  (*m).para   = NULL;

  (*m).sg_w   = 0;
  (*m).sg_nr  = 0;
  (*m).sg_max = 0;
  (*m).sseg   = NULL;
  
  (*m).st_nr  = 0;
  (*m).st_max = 0;
  (*m).st = NULL;

  (*m).sp_w   = 0;
  (*m).sp_nr  = 0;
  (*m).sp_max = 0;
  (*m).spar   = NULL;

  (*m).meshgen = -1;
  (*m).meshgenDATA = NULL;


  return;
}


/*FUNCTION*/
void mesh_info( FILE *outf, struct mesh *m
/* printf info on the mesh to file outf (e.g. outf=stdout),

   Input:  m        - a mesh
   Output: (to file outf)

*/
		){
  double MB=1024.0*1024.0;
  char formatstring[]="%15s:   %12"dFIDX" -> %8.1f(MB);    %12"dFIDX" -> %8.1f(MB)\n";
  char headerstring[]="%15s:   %12s -> %8s(MB);    %12s -> %8s(MB)\n";

  fprintf(outf, "\n");
  fprintf(outf, "-------------------------------------------------------\n");

  fprintf(outf, "mesh_info: dim=%d  problem=%d  lvl=%2d\n",
	  (*m).dim, (*m).problem, (*m).lvl);
  fprintf(outf, "           meshgen=%d   meshgenDATA=%p \n",
	  (*m).meshgen, (*m).meshgenDATA);

  fprintf(outf, "\n");

  fprintf(outf, headerstring, 
	  "Field sizes", 
	  "current(_nr)", "memory", "max(_max)", "memory");

  fprintf(outf, "-------------------------------------------------------\n");

  fprintf(outf, formatstring, 
	  "vertex   ", 
	  (*m).vx_nr, fmax(0,(*m).vx_nr)*(*m).vx_w*sizeof(double)/MB,
	  (*m).vx_max, fmax(0,(*m).vx_max)*(*m).vx_w*sizeof(double)/MB );

  fprintf(outf, formatstring, 
	  "elem     ", 
	  (*m).el_nr, fmax(0,(*m).el_nr)*(*m).el_w*sizeof(FIDX)/MB,
	  (*m).el_max, fmax(0,(*m).el_max)*(*m).el_w*sizeof(FIDX)/MB );
  
  fprintf(outf, formatstring, 
	  "edge     ", 
	  (*m).eg_nr, fmax(0,(*m).eg_nr)*(*m).eg_w*sizeof(FIDX)/MB,
	  (*m).eg_max, fmax(0,(*m).eg_max)*(*m).eg_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "face     ", 
	  (*m).fc_nr, fmax(0,(*m).fc_nr)*(*m).fc_w*sizeof(FIDX)/MB,
	  (*m).fc_max, fmax(0,(*m).fc_max)*(*m).fc_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "vols     ", 
	  (*m).vo_nr, fmax(0,(*m).vo_nr)*(*m).vo_w*sizeof(FIDX)/MB,
	  (*m).vo_max, fmax(0,(*m).vo_max)*(*m).vo_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "hierarchy", 
	  (*m).hi_nr, fmax(0,(*m).hi_nr)*(*m).hi_w*sizeof(FIDX)/MB,
	  (*m).hi_max, fmax(0,(*m).hi_max)*(*m).hi_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "el hierar", 
	  (*m).eh_nr, fmax(0,(*m).eh_nr)*(*m).eh_w*sizeof(FIDX)/MB,
	  (*m).eh_max, fmax(0,(*m).eh_max)*(*m).eh_w*sizeof(FIDX)/MB );
  
  fprintf(outf, formatstring, 
	  "boundary ", 
	  (*m).bd_nr, fmax(0,(*m).bd_nr)*(*m).bd_w*sizeof(FIDX)/MB,
	  (*m).bd_max, fmax(0,(*m).bd_max)*(*m).bd_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "pc surf  ", 
	  (*m).ps_nr, fmax(0,(*m).ps_nr)*(*m).ps_w*sizeof(FIDX)/MB,
	  (*m).ps_max, fmax(0,(*m).ps_max)*(*m).ps_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "pc vols  ", 
	  (*m).pv_nr, fmax(0,(*m).pv_nr)*(*m).pv_w*sizeof(FIDX)/MB,
	  (*m).pv_max, fmax(0,(*m).pv_max)*(*m).pv_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "perfcrit ", 
	  (*m).pc_nr, fmax(0,(*m).pc_nr)*(*m).pc_w*sizeof(double)/MB,
	  (*m).pc_max, fmax(0,(*m).pc_max)*(*m).pc_w*sizeof(double)/MB );

  fprintf(outf, formatstring, 
	  "function ", 
	  (*m).fu_nr, fmax(0,(*m).fu_nr)*(*m).fu_w*sizeof(double)/MB,
	  (*m).fu_max, fmax(0,(*m).fu_max)*(*m).fu_w*sizeof(double)/MB );

  fprintf(outf, formatstring, 
	  "parameter", 
	  (*m).pa_nr, fmax(0,(*m).pa_nr)*(*m).pa_w*sizeof(double)/MB,
	  (*m).pa_max, fmax(0,(*m).pa_max)*(*m).pa_w*sizeof(double)/MB );

  fprintf(outf, formatstring, 
	  "shapesegm", 
	  (*m).sg_nr, fmax(0,(*m).sg_nr)*(*m).sg_w*sizeof(FIDX)/MB,
	  (*m).sg_max, fmax(0,(*m).sg_max)*(*m).sg_w*sizeof(FIDX)/MB );

  fprintf(outf, formatstring, 
	  "shp t(vx)", 
	  (*m).st_nr, fmax(0,(*m).st_nr)*1*sizeof(double)/MB,
	  (*m).st_max, fmax(0,(*m).st_max)*1*sizeof(double)/MB );

  fprintf(outf, formatstring, 
	  "shape par", 
	  (*m).sp_nr, fmax(0,(*m).sp_nr)*(*m).sp_w*sizeof(double)/MB,
	  (*m).sp_max, fmax(0,(*m).sp_max)*(*m).sp_w*sizeof(double)/MB );

  fprintf(outf, "-------------------------------------------------------\n");
  fprintf(outf, "\n");

  return;
}





/********************************************************************/
/*                                                                  */
/*         T1 element stuff                                         */
/*                                                                  */
/********************************************************************/


/*FUNCTION*/
int mesh_refine_uniform_t1( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m       - the mesh, is modified accordingly, e.g. new
                     elements, faces, edges, vertices, boundary
                     elements and hierarchy entries are created,
		     the element el and the face el are thereby
		     replaced by one of their children, the edges
		     remain to allow other elements to stay intact,
		     if necessary the fields of the mesh are
		     reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX old_el_nr, i;
  int err;

  old_el_nr=(*m).el_nr;
  (*m).lvl++;

  /* loop over all elements */
  for (i=0; i< old_el_nr; i++)
    {
      err=mesh_refine_element_t1( m , i);
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_element_t1, 
			       mesh_refine_uniform_t1 ); 
    }
  return SUCCESS;
}


/*FUNCTION*/
int mesh_refine_element_t1( struct mesh *m , FIDX el
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, fc_w, el_w, eg_w, ps_w, pv_w;
  int err;


  FIDX newedge[3], midnodes[3], oldnod[3], nodedge[3][2], nodineg[3][2];
  /* nodedge[lnode][this nodes edge]=the edge (global number),
     nodineg[lnode][this nodes edge]=the node is 1st/2nd of the edge */

  fc_w=(*m).fc_w;
  el_w=(*m).el_w;
  eg_w=(*m).eg_w;
  ps_w=(*m).ps_w;
  pv_w=(*m).pv_w;

  /* clear nodedge */
  for (i=0; i<3; i++)
    for (j=0; j<2; j++)
      nodedge[i][j]=-1;

  /* get oldnod */
  for (i=0; i<3; i++)
    oldnod[i]=(*m).elem[el*el_w+MCT1ELNOD1+i];

  /* make sure the 3 edges are refined, store the midnodes, get
     nodedge */ 
  for (i=0; i<3; i++)
    {
      FIDX eg, chld;

      eg=(*m).face[el*fc_w+MCT1FCEDG1+i];

      /* make sure this edge is refined */
      if ((*m).edge[eg*eg_w+MCT1EGCHL1]==-1)
	{
	  err=mesh_split_edge_t1( m, eg);
	  FUNCTION_FAILURE_HANDLE( err, mesh_split_edge_t1,
				   mesh_refine_element_t1); 
	}
      
      /* store the midnode */
      chld=(*m).edge[eg*eg_w+MCT1EGCHL1];
      /* second node of child edge is always the midnode of the old
	 edge */ 
      midnodes[i]=(*m).edge[chld*eg_w+MCT1EGNOD1+1]; 

      for (j=0; j<2; j++)
	{
	  FIDX amnodeedge, thenode, intnode;
	  thenode=(*m).edge[eg*eg_w+MCT1EGNOD1+j];
	  intnode=0;
	  /* find the apropriate internal node for this guy */
	  while ((intnode<3)&&(oldnod[intnode]!=thenode))
	    intnode++;
	  /* if not found, cry */
	  if (intnode>=3)
	    {
	      fprintf(stderr, "internal mesh consistency error in mesh_refine_element_t1, file %s, line %d\n", __FILE__, __LINE__);
	      return FAIL;
	    }

	  /* find where in nodedge and nodineg this one goes to */
	  if (nodedge[intnode][0]==-1)
	    { amnodeedge=0; }
	  else
	    { amnodeedge=1; }
	  nodedge[intnode][amnodeedge]=eg;
	  nodineg[intnode][amnodeedge]=j;
	}
    }

  /* make sure we have enough space for the 3 new edges in the inner
     of the element, 3 new faces (the old is recycled as a 4th), 3 new
     elements (the old is recycled as a 4th), 3 new pcvol if needed
     (the old is recycled as a 4th) */ 
  if ((*m).eg_nr+3>(*m).eg_max)
    {
      err=mesh_more_edges( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_refine_element_t1 );
    }
  if ((*m).fc_nr+3>(*m).fc_max)
    {
      err=mesh_more_faces( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_refine_element_t1 );
    }
  if ((*m).el_nr+3>(*m).el_max)
    {
      err=mesh_more_elems( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elems, mesh_refine_element_t1 );
    }

  if (((*m).face[el*fc_w+MCT1FCPCVL]!=-1)&&((*m).pv_nr+3>(*m).pv_max))
    {
      err=mesh_more_pcvol( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcvol, mesh_refine_element_t1 );
    }

  /* for all old nodes */
  for (i=0; i<3; i++)
    {
      FIDX chld0, chld1, midnod0, midnod1, newface, newel, level;

      /* get the child edges containig this node */
      chld0=(*m).edge[nodedge[i][0]*eg_w+MCT1EGCHL1+nodineg[i][0]];
      chld1=(*m).edge[nodedge[i][1]*eg_w+MCT1EGCHL1+nodineg[i][1]];

      /* the refinement level of the new edges */
      level=(*m).lvl;
  
      /* get the corresponding midnodes */
      midnod0=(*m).edge[chld0*eg_w+MCT1EGNOD1+1]; 
      midnod1=(*m).edge[chld1*eg_w+MCT1EGNOD1+1];
      /* second node of child edge is allway the midnode of the old
	 edge */

      /* create a new edge from the midnodes */
      newedge[i]=(*m).eg_nr;
      (*m).eg_nr++;

      (*m).edge[newedge[i]*eg_w+MCT1EGNOD1  ]=midnod0;
      (*m).edge[newedge[i]*eg_w+MCT1EGNOD1+1]=midnod1;
      (*m).edge[newedge[i]*eg_w+MCT1EGCHL1  ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT1EGBND   ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT1EGPCSU  ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT1EGLVL   ]=level;

      /* create a new face */
      newface=(*m).fc_nr;
      (*m).fc_nr++;

      (*m).face[newface*fc_w+MCT1FCEDG1  ]=chld0;
      (*m).face[newface*fc_w+MCT1FCEDG1+1]=chld1;
      (*m).face[newface*fc_w+MCT1FCEDG1+2]=newedge[i];

      (*m).face[newface*fc_w+MCT1FCRHSF]=(*m).face[el*fc_w+MCT1FCRHSF];

      /* if necessary create a new pcvol entry */
      if ((*m).face[el*fc_w+MCT1FCPCVL]!=-1)
	{
	  FIDX pcv, new;

	  pcv=(*m).face[el*fc_w+MCT1FCPCVL];
	  new=(*m).pv_nr;
	  
	  (*m).pcvol[new*pv_w+MCXXPVVOLM]=newface;
	  (*m).face[newface*fc_w+MCT1FCPCVL]=new;
	  (*m).pcvol[new*pv_w+MCXXPVCRIT]=(*m).pcvol[pcv*pv_w+MCXXPVCRIT];

	  (*m).pv_nr++;
	}
      else
	{
	  (*m).face[newface*fc_w+MCT1FCPCVL]=-1;
	}

      /* check if the edges of the new face are pcsurf edges, if so,
	 let them know the name of the new face */
      for (j=0; j<3; j++)
	{
	  FIDX eg, pcsu;
	  eg=(*m).face[newface*fc_w+MCT1FCEDG1+j];
	  pcsu=(*m).edge[eg*eg_w+MCT1EGPCSU];
	  if (pcsu!=-1)
	    {
	      if ((*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]==-1)
		{
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]=newface;
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=-1;
		}
	      else
		{
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=newface;
		}
	    }
	}

      /* create a new element */
      newel=(*m).el_nr;
      (*m).el_nr++;

      (*m).elem[newel*el_w+MCT1ELNOD1  ]=oldnod[i];
      (*m).elem[newel*el_w+MCT1ELNOD1+1]=midnod0;
      (*m).elem[newel*el_w+MCT1ELNOD1+2]=midnod1;
      
      /* next node */
    }

  /* the outer 3 elements are correctly created now,
     so lets overwrite the old element with the inner one, the 4-th */

  for (i=0; i<3; i++)
    {
      /* the face */
      (*m).face[el*fc_w+MCT1FCEDG1+i]=newedge[i];
      /* ( rhs-function and pcvol stays the same ) */
      /* the edges of the inner face can not be pcsurf edges */

      /* the element */
      (*m).elem[el*el_w+MCT1ELNOD1+i]=midnodes[i];
    }

  /* that's it */

  return SUCCESS;
}

/*FUNCTION*/
int mesh_split_edge_t1( struct mesh *m , FIDX eg
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices, boundary elements and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring elements
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, dim, vx_w, eg_w, hi_w, bd_w, ps_w;
  int  err;
  FIDX newnode, nod1, nod2;
  FIDX neweg1, neweg2;
  FIDX oldbd, newbd1,newbd2;
  FIDX newhi, level;
  FIDX bd, seg;

  dim=(*m).dim;
  vx_w=(*m).vx_w;
  eg_w=(*m).eg_w;
  hi_w=(*m).hi_w;
  bd_w=(*m).bd_w;
  ps_w=(*m).ps_w;

  /* debug st_nr problems: */
  if ((*m).vx_nr>(*m).st_nr)
    {
      fprintf(stderr, "Error in mesh_split_edge_t1: vx_nr=%"dFIDX">st_nr=%"dFIDX"\n",
	      (*m).vx_nr,(*m).st_nr);
      return FAIL;
    }

  if ((*m).edge[eg*eg_w+ MCT1EGCHL1 ]!= -1)
    {
      /* nothing to do, return */
      return SUCCESS;
    }
    
  bd = (*m).edge[eg*eg_w+ MCT1EGBND];   
  if (bd!=-1) 
    {
      seg = (*m).bound[bd*bd_w+MCT1BDSSEG];
    }
  else 
    {
      seg = -1;
    }

  /* first we have to make sure we have space for one new vertex, two
     new edges, eventually 2 new boundary elements, eventually 2 new
     pcsurf elements, one hierarchy entry, t-value of new vertex 
  */
  if ((*m).vx_nr+1>(*m).vx_max)
    {
      err=mesh_more_vertices( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices,  mesh_split_edge_t1 );
    }
  if ((*m).eg_nr+2>(*m).eg_max)
    {
      err=mesh_more_edges( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,  mesh_split_edge_t1 );
    }
  if (((*m).edge[eg*eg_w+ MCT1EGBND]!= -1)&&((*m).bd_nr+2>(*m).bd_max))
    {
      err=mesh_more_boundary( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_boundary,  mesh_split_edge_t1 );
    }
  if (((*m).edge[eg*eg_w+ MCT1EGPCSU]!= -1)&&((*m).ps_nr+2>(*m).ps_max))
    {
      err=mesh_more_pcsurf( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcsurf,  mesh_split_edge_t1 );
    }
  if ((*m).hi_nr+1>(*m).hi_max)
    {
      err=mesh_more_hierarchy( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_hierarchy,  mesh_split_edge_t1 );
    }
  if ((*m).st_nr+1>(*m).st_max)
    {
      err=mesh_more_st( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_st,  mesh_split_edge_t1 );
    }
  /* nodes of the edge */
  nod1=(*m).edge[eg*eg_w+ MCT1EGNOD1];
  nod2=(*m).edge[eg*eg_w+ MCT1EGNOD1+1];

  /* create the new node */
  newnode= (*m).vx_nr;
  (*m).vx_nr++;
  (*m).st_nr++;
    
  for (i=0;i<dim; i++)
    {
      (*m).vertex[newnode*vx_w+MCT1VXSTRT+i]=
	0.5*(*m).vertex[nod1*vx_w+MCT1VXSTRT+i]+
	0.5*(*m).vertex[nod2*vx_w+MCT1VXSTRT+i];
    } 
  if (seg>-1)
    {/* get t-value for the new node  */
      err=mesh_spline_set_t( m, seg, nod1,nod2, newnode, 1, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_get_st,  mesh_split_edge_t1 );
    }
    
  /* refinement level of the old edge */
  level=(*m).lvl;
  
  /* create the new edges */
  neweg1=(*m).eg_nr;
  neweg2=(*m).eg_nr+1;
  (*m).eg_nr+=2;
  /* child edge 1 */
  (*m).edge[neweg1*eg_w+MCT1EGNOD1  ]=nod1;
  (*m).edge[neweg1*eg_w+MCT1EGNOD1+1]=newnode;
  (*m).edge[neweg1*eg_w+MCT1EGCHL1  ]=-1;
  (*m).edge[neweg1*eg_w+MCT1EGBND   ]=-1;
  (*m).edge[neweg1*eg_w+MCT1EGPCSU  ]=-1;    
  (*m).edge[neweg1*eg_w+MCT1EGLVL   ]=level; 
  /* child edge 2 */
  (*m).edge[neweg2*eg_w+MCT1EGNOD1  ]=nod2;
  (*m).edge[neweg2*eg_w+MCT1EGNOD1+1]=newnode;
  (*m).edge[neweg2*eg_w+MCT1EGCHL1  ]=-1;
  (*m).edge[neweg2*eg_w+MCT1EGBND   ]=-1;
  (*m).edge[neweg2*eg_w+MCT1EGPCSU  ]=-1;
  (*m).edge[neweg2*eg_w+MCT1EGLVL   ]=level;  
  /* make them children of eg */
  (*m).edge[eg*eg_w+MCT1EGCHL1  ]=neweg1;
  (*m).edge[eg*eg_w+MCT1EGCHL1+1]=neweg2;
  
  /* eventually create new boundary elements, correct boundary
     information on the children */
  if ((*m).edge[eg*eg_w+ MCT1EGBND]!= -1)
    {
      newbd1=(*m).bd_nr;
      newbd2=(*m).bd_nr+1;
      (*m).bd_nr+=2;
      oldbd=(*m).edge[eg*eg_w+ MCT1EGBND];
      /* newbd1 */
      (*m).bound[newbd1*bd_w+MCT1BDEDGE]=neweg1;
      (*m).bound[newbd1*bd_w+MCT1BDTYPE]=
	(*m).bound[oldbd*bd_w+MCT1BDTYPE]; /* same type as old */
      (*m).bound[newbd1*bd_w+MCT1BDFNCT]=
	(*m).bound[oldbd*bd_w+MCT1BDFNCT]; /* same function data as old
					   */
      (*m).bound[newbd1*bd_w+MCT1BDORIE]=
	(*m).bound[oldbd*bd_w+MCT1BDORIE]; /* first edge has same
					      orientation as the old
					      edge 
					   */
      (*m).bound[newbd1*bd_w+MCT1BDSSEG]=
	(*m).bound[oldbd*bd_w+MCT1BDSSEG]; /* same shape variability */
      (*m).edge[neweg1*eg_w+MCT1EGBND   ]=newbd1;
	    
      /* newbd2 */
      (*m).bound[newbd2*bd_w+MCT1BDEDGE]=neweg2;
      (*m).bound[newbd2*bd_w+MCT1BDTYPE]=
	(*m).bound[oldbd*bd_w+MCT1BDTYPE]; /* same type as old */
      (*m).bound[newbd2*bd_w+MCT1BDFNCT]=
	(*m).bound[oldbd*bd_w+MCT1BDFNCT]; /* same function data as old
					   */
      (*m).bound[newbd2*bd_w+MCT1BDORIE]=
	-(*m).bound[oldbd*bd_w+MCT1BDORIE];/* 2nd edge has reverse
					      orientation of the old
					      edge */
      (*m).bound[newbd2*bd_w+MCT1BDSSEG]=
	(*m).bound[oldbd*bd_w+MCT1BDSSEG]; /* same shape variability */
      (*m).edge[neweg2*eg_w+MCT1EGBND   ]=newbd2;
    }

  /* eventually create new pcsurf elements, correct pcsurf
     information on the children */
  if ((*m).edge[eg*eg_w+ MCT1EGPCSU]!= -1)
    {
      FIDX oldps, newps1, newps2;
      newps1=(*m).ps_nr;
      newps2=(*m).ps_nr+1;
      (*m).ps_nr+=2;
      oldps=(*m).edge[eg*eg_w+ MCT1EGPCSU];
      /* newps1 */
      (*m).pcsurf[newps1*ps_w+MCXXPSSURF]=neweg1;
      (*m).pcsurf[newps1*ps_w+MCXXPSVOLE]=-1;/* faces are set in face
						refinement */
      (*m).pcsurf[newps1*ps_w+MCXXPSCRIT]=
	(*m).pcsurf[oldps*ps_w+MCXXPSCRIT]; /* same criterion as old */
      (*m).pcsurf[newps1*ps_w+MCXXPSORIE]=
	(*m).pcsurf[oldps*ps_w+MCXXPSORIE]; /* first edge has same
					       orientation as the old
					       edge 
					   */
      (*m).edge[neweg1*eg_w+MCT1EGPCSU  ]=newps1;
      /* newps2 */
      (*m).pcsurf[newps2*ps_w+MCXXPSSURF]=neweg2;
      (*m).pcsurf[newps2*ps_w+MCXXPSVOLE]=-1;/* faces are set in face
						refinement */
      (*m).pcsurf[newps2*ps_w+MCXXPSCRIT]=
	(*m).pcsurf[oldps*ps_w+MCXXPSCRIT]; /* same criterion as old */
      (*m).pcsurf[newps2*ps_w+MCXXPSORIE]=
	-(*m).pcsurf[oldps*ps_w+MCXXPSORIE];/* 2nd edge has reverse
					       orientation of the old
					       edge 
					   */
      (*m).edge[neweg2*eg_w+MCT1EGPCSU  ]=newps2;
    }

  /* new hierarchy entry */
  newhi = (*m).hi_nr;
  (*m).hi_nr++;
  (*m).hier[newhi*hi_w+MCT1HICHLD  ] = newnode;
  (*m).hier[newhi*hi_w+MCT1HIFAT1  ] = nod1;
  (*m).hier[newhi*hi_w+MCT1HIFAT1+1] = nod2;
  (*m).hier[newhi*hi_w+MCT1HILVL   ] = level;

  return SUCCESS;
}


/*FUNCTION*/
int mesh_write_solution_exp_t1( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a NAG IRIS EXPLORER readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX i, j, lastslash, el_w, vx_w, vx_nr;

  char *pyrname, *latname, *latbasename;
  FILE *pyrfile, *latfile;


  el_w =(*m).el_w;
  vx_w =(*m).vx_w;
  vx_nr=(*m).vx_nr;

  if (ncompo*vx_nr!=(*x).len)
    {
      fprintf( stderr,
	       "\nmesh_write_solution_exp_t1: nodenr in mesh and "
	       "dimension of vector must agree!\n");
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( pyrname, (namlen+4), char, 
	      mesh_write_solution_exp_t1 );
  TRY_MALLOC( latname, (namlen+4), char, 
	      mesh_write_solution_exp_t1 );
  TRY_MALLOC( latbasename, (namlen+4), char, 
	      mesh_write_solution_exp_t1 );

  strncpy( pyrname, name, namlen );
  strncpy( latname, name, namlen );
  strncpy( latbasename, name, namlen );

  /* make sure no overread happens */
  pyrname[namlen-1]='\0';
  latname[namlen-1]='\0';
  latbasename[namlen-1]='\0';

  /* cut the leading path out of latname */
  lastslash=-1;
  for (i=0; i<namlen; i++)
    if (latbasename[i]=='/') lastslash=i;
  if (lastslash>=0)
    {
      lastslash++;
      for (i=0; i<namlen-lastslash; i++)
	{
	  latbasename[i]=latbasename[i+lastslash];
	}
    }
  
  /* append ".pyr" resp. ".lat" */
  strcat( pyrname, ".pyr" );
  strcat( latname, ".lat" );
  strcat( latbasename, ".lat" );
  
  /* open the lat file */
  latfile=fopen(latname, "w");
  if (latfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_exp_t1: error opening file %s\n",
	      latname);
      return FAIL;
    }

  /* write the header */
  fprintf(latfile, "#!/usr/explorer/bin/explorer cxLattice plain 1.0\n\n");
  fprintf(latfile, "# ndim\n%6d\n", 1); /* dimension of the latice */
  fprintf(latfile, "# n points\n%6"dFIDX"\n",  vx_nr); /* nr of nodes */
  fprintf(latfile, "# ndatavar\n%6"dFIDX"\n",  ncompo);/* nr data per node */
  fprintf(latfile, "# primtype (float) \n%6d\n",3); /* data type float */
  fprintf(latfile, "# coordtype cx_coord_curvilinear\n%6d\n",2); 
  fprintf(latfile, "# nsteps\n%6d\n",1); /* nr of data sets */
  fprintf(latfile, "# coords\n%6d\n\n",2); /* dimension of the coords
					      */
  /* write the node positions */
  fprintf(latfile, "# coords of the nodes\n"); 
  for (i=0; i<vx_nr; i++)
    {
      fprintf(latfile, "  %18.10e  %18.10e\n",
	      (*m).vertex[i*vx_w+MCT1VXSTRT  ],
	      (*m).vertex[i*vx_w+MCT1VXSTRT+1]);
    }

  /* write the data */
  fprintf(latfile, "\n\n# data (step= 1)\n");
  for (i=0; i<vx_nr; i++)
    {
      for (j=0; j<ncompo; j++)
	{
	  fprintf(latfile, "  %18.10e",(*x).V[j*vx_nr+i]);
	}
      fprintf(latfile, "\n");
    }
  
  fprintf(latfile, "\n\n# EOF\n");
  /* close lat file */
  fclose(latfile);


  /* open the pyr file */
  pyrfile=fopen(pyrname, "w");
  if (pyrfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_exp_t1: error opening file %s\n",
	      pyrname);
      return FAIL;
    }

  /* write the header */
  fprintf(pyrfile, "#!/usr/explorer/bin/explorer cxPyramid plain 1.0\n\n");
  fprintf(pyrfile, "include %s\n\n", latbasename);
  fprintf(pyrfile, "# n layers\n%6d\n",2); /* n layers in the file */
  fprintf(pyrfile, "# compression type cx_compress_unique\n%6d\n",1);
  fprintf(pyrfile, "# compression dict cx_pyramid_dict_triangle\n%6d\n\n",2);
  fprintf(pyrfile, "# layer 1 ?????\n%6d  %6d\n\n",0,0);
  fprintf(pyrfile, "# layer 2 n elements, n connections\n%6"dFIDX"  %6"dFIDX"\n\n",
	   (*m).el_nr,  (*m).el_nr*3 );

  /* element pointers */
  fprintf(pyrfile, "# element pointers\n");
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(pyrfile, "%6"dFIDX"\n",  (3*i) );      
    }
  fprintf(pyrfile, "# end pointer\n%6"dFIDX"\n\n",  (3*(*m).el_nr) );

  /* connectivity */
  fprintf(pyrfile, "# connectivity\n");
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(pyrfile, "%6"dFIDX" %6"dFIDX" %6"dFIDX"\n",
	       (*m).elem[i*el_w+MCT1ELNOD1  ],
	       (*m).elem[i*el_w+MCT1ELNOD1+1],
	       (*m).elem[i*el_w+MCT1ELNOD1+2]);      
    }
  fprintf(pyrfile, "\n# no data lattice at this layer\nNULL\n");
  fprintf(pyrfile, "\n# EOF\n");

  /* close pyr file */
  fclose(pyrfile); 

  free(pyrname);
  free(latname);
  free(latbasename);

  return SUCCESS;
}

/*FUNCTION*/
int mesh_write_solution_vtk_t1( struct mesh *m, struct vector *vec_x,
				FIDX vcompo, struct vector *scalar_x,
				FIDX scompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a vtk readable
   file for visualisation, 
   this can for example be used with paraview
           http://www.paraview.org/

   Input:  m         - the mesh
           vec_x     - vector with data associated to the nodes,
	               for representing a vector field of dimension
	               vcompo, ignored if vec_x==NULL
	   vcompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector x have to be numbered such that
	               vec_x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh, has to be TWO for t1
	               mesh, ignored if vec_x==NULL
           scalar_x  - vector with data associated to the nodes,
	               for representing scompo scalar fields,
		       for each scalar field a separate part is
		       written in the file, ignored if scalar_x==NULL
	   scompo    - dimension of the solution vector field per node
                       which shall be writen to the file, (or number
                       of solution components), the components of the
	               vector scalar_x have to be numbered such that
	               saclar_x[j*vx_nr+i] gives the j-th scalar
	               variable in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX i, j, vx_w, vx_nr, el_w, el_nr;

  char *vtkname;
  FILE *vtkfile;


  vx_w =(*m).vx_w;
  vx_nr=(*m).vx_nr;
  el_w =(*m).el_w;
  el_nr=(*m).el_nr;

  if ((vec_x!=NULL)&&(vcompo!=2))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_t1: "
	       "dimension of vector vec_x must be 2 for 2D mesh!\n");
      return FAIL;
    }
  if ((vec_x!=NULL)&&(vcompo*vx_nr!=(*vec_x).len))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_t1: vx_nr in mesh and "
	       "dimension of vector vec_x must agree!\n");
      return FAIL;
    }

  if ((scalar_x!=NULL)&&(scompo*vx_nr!=(*scalar_x).len))
    {
      fprintf( stderr,
	       "\nmesh_write_solution_vtk_t1: vx_nr in mesh and "
	       "dimension of vector scalar_x must agree!\n");
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( vtkname, (namlen+5), char, 
	      mesh_write_solution_vtk_t1 );

  strncpy( vtkname, name, namlen );

  /* make sure no overread happens */
  vtkname[namlen-1]='\0';

  /* append ".pyr" resp. ".lat" */
  strcat( vtkname, ".vtk" );
  
  /* open the lat file */
  vtkfile=fopen(vtkname, "w");
  if (vtkfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_vtk_t1: error opening file %s\n",
	      vtkname);
      return FAIL;
    }

  /* write the header */
  fprintf(vtkfile, "# vtk DataFile Version 3.0\n"); /* version */
  fprintf(vtkfile, "file %s generated by feins\n", vtkname); /* description */
  fprintf(vtkfile, "ASCII\n"); /* ASCII or BINARY */
  fprintf(vtkfile, "DATASET UNSTRUCTURED_GRID\n"); /* Type of mesh */

  fprintf(vtkfile, "POINTS %8"dFIDX" double\n",  vx_nr); /* no of nodes */
  /* write the node positions */
  for (i=0; i<vx_nr; i++)
    {
      fprintf(vtkfile, "   %18.10e  %18.10e  0.0\n",
	      (*m).vertex[i*vx_w+MCT1VXSTRT  ],
	      (*m).vertex[i*vx_w+MCT1VXSTRT+1]);
    }

  fprintf(vtkfile, "\n");
  /* nr of elements, and number of list entries (=4*el_nr) */
  fprintf(vtkfile, "CELLS %8"dFIDX" %8"dFIDX"\n",  el_nr,  4*el_nr); 
  /* write the triangles */
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(vtkfile, "   3 %8"dFIDX" %8"dFIDX" %8"dFIDX"\n",
	       (*m).elem[i*el_w+MCT1ELNOD1  ],
	       (*m).elem[i*el_w+MCT1ELNOD1+1],
	       (*m).elem[i*el_w+MCT1ELNOD1+2]);      
    }


  fprintf(vtkfile, "\n");
  /* define cell type for each element */
  fprintf(vtkfile, "CELL_TYPES %8"dFIDX"\n",  el_nr); 
  /* all elements are triangles, type=5 */
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(vtkfile, "   5\n");      
    }

  fprintf(vtkfile, "\n");
  /* switch to point data */
  fprintf(vtkfile, "POINT_DATA %8"dFIDX"\n",  vx_nr); 

  /* now write the vector data */
  if (vec_x!=NULL)
    {
      fprintf(vtkfile, "VECTORS feins_vector double\n"); 
      for (i=0; i<vx_nr; i++)
	{
	  fprintf(vtkfile, "  %15.7e  %15.7e 0.0\n", 
		  (*vec_x).V[0*vx_nr+i], (*vec_x).V[1*vx_nr+i]);
	}
      
    }



  /* now write the scalar data */
  if (scalar_x!=NULL)
    {
      for (j=0; j<scompo; j++)
	{
	  fprintf(vtkfile, "\nSCALARS feins_scalar_%02"dFIDX" double 1\n", j); 
	  fprintf(vtkfile, "LOOKUP_TABLE default"); 
	  for (i=0; i<vx_nr; i++)
	    {
	      fprintf(vtkfile, "  %15.7e\n",(*scalar_x).V[j*vx_nr+i]);
	    }
	}
      
    }
  
  /* close vtkfile */
  fclose(vtkfile);

  free(vtkname);

  return SUCCESS;
}


/*FUNCTION*/
int mesh_write_solution_femplot_t1( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a ?FEMplot? readable
   file for visualisation 
   FEMplot by project students (Lars Eiserbeck und Felix Prehl, 2007)
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.femp

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX i, j, vx_w, vx_nr, el_w, el_nr;

  char *fempname;
  FILE *fempfile;


  vx_w =(*m).vx_w;
  vx_nr=(*m).vx_nr;
  el_w =(*m).el_w;
  el_nr=(*m).el_nr;

  if (ncompo*vx_nr!=(*x).len)
    {
      fprintf( stderr,
	       "\nmesh_write_solution_femplot_t1: nodenr in mesh and "
	       "dimension of vector must agree!\n");
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( fempname, (namlen+5), char, 
	      mesh_write_solution_femplot_t1 );

  strncpy( fempname, name, namlen );

  /* make sure no overread happens */
  fempname[namlen-1]='\0';

  /* append ".pyr" resp. ".lat" */
  strcat( fempname, ".femp" );
  
  /* open the lat file */
  fempfile=fopen(fempname, "w");
  if (fempfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_femplot_t1: error opening file %s\n",
	      fempname);
      return FAIL;
    }

  /* write the header */
  fprintf(fempfile, "%8"dFIDX" # Nodes \n",  vx_nr); /* no of nodes */
  fprintf(fempfile, "%8"dFIDX" # Triangles \n",  el_nr); /* nr of elements */
  fprintf(fempfile, "%8"dFIDX" # components \n",  ncompo);/* nr data per node */

  /* write the node positions */
  fprintf(fempfile, "// Nodes\n");
  for (i=0; i<vx_nr; i++)
    {
      fprintf(fempfile, "   %18.10e  %18.10e\n",
	      (*m).vertex[i*vx_w+MCT1VXSTRT  ],
	      (*m).vertex[i*vx_w+MCT1VXSTRT+1]);
    }


  /* write the triangles */
  fprintf(fempfile, "// Triangles\n");
  for (i=0; i<(*m).el_nr; i++)
    {
      fprintf(fempfile, "   %8"dFIDX" %8"dFIDX" %8"dFIDX"\n",
	       (*m).elem[i*el_w+MCT1ELNOD1  ],
	       (*m).elem[i*el_w+MCT1ELNOD1+1],
	       (*m).elem[i*el_w+MCT1ELNOD1+2]);      
    }

  /* write the data */
  fprintf(fempfile, "// data\n");
  for (i=0; i<vx_nr; i++)
    {
      for (j=0; j<ncompo; j++)
	{
	  fprintf(fempfile, "  %18.10e",(*x).V[j*vx_nr+i]);
	}
      fprintf(fempfile, "\n");
    }
  
  /* close fempfile */
  fclose(fempfile);

  free(fempname);

  return SUCCESS;
}


//#include <unistd.h>
#include <string.h>
/*FUNCTION*/
int mesh_gnuplot_boundary_circles_t2( FIDX ncirc, double *Rs, double tol,
				      struct mesh *m,
				      struct vector *data, FIDX ncomp, int mode,
				      char *fname_base
/* assuming that we have ncirc circles around (0,0) as boundary
   (r=Rs[i]), plot the data over the boundary

   Input:  ncirc     - number of circles
           Rs        - array of length ncirc, defining the radius of
                       each circle, the data is written to a separate
                       file for each circle
	   tol       - tolerance in testing if a point is on a circle
	   m         - the mesh that the data corresponds to
           data      - vector containing the data
           ncomp     - number of components in the data,  
           mode      - defines if the components of the data are
                       written to the file (mode=0), or the norm of
                       the vector of components associated to each
                       node of the mesh (mode=1)
           fname_base- base of the filenames that is written to, the
                       files will be 
		         <fname_base>_R1.txt
		       to
		         <fname_base>_R<ncirc>.txt

   Output: (the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		       ){
  int err;
  FIDX bd_w, eg_w, vx_w, vx_nr, nwrite;

  FIDX i, j, bd;

  char *fname;
  struct vector *circle;
  struct intdouble **sorter;
  FIDX *nRs;
  FIDX *iRs;
  FIDX maxlen;
  vx_nr=m->vx_nr;
  vx_w=m->vx_w;
  eg_w=m->eg_w;
  bd_w=m->bd_w;

  switch(mode)
    {
    case 0:
      nwrite=1+ncomp;
      break;
    case 1:
      nwrite=1+1;
      break;
    default:
      fprintf(stderr,"mesh_gnuplot_boundary_circles_t2: "
	      "unknown nwrite for mode=%d\n", mode );
      return FAIL;
    }

  maxlen= strlen(fname_base)+100;

  TRY_MALLOC(fname, maxlen, char, mesh_gnuplot_boundary_circles_t2);

  TRY_MALLOC(circle, nwrite, struct vector, mesh_gnuplot_boundary_circles_t2);

  TRY_MALLOC(sorter, ncirc, struct intdouble *, mesh_gnuplot_boundary_circles_t2);
  TRY_MALLOC(nRs, ncirc, FIDX, mesh_gnuplot_boundary_circles_t2);
  TRY_MALLOC(iRs, ncirc, FIDX, mesh_gnuplot_boundary_circles_t2);

  /* init nRs */
  for (i=0; i<ncirc; i++)
    {
      nRs[i]=0;
    }


  /* count the points on each circle */
  for (bd=0; bd<m->bd_nr; bd++)
    {
      /* use only finest edges */
      if ((m->edge[m->bound[bd*bd_w+MCT2BDEDGE]*eg_w+MCT2EGCHL1]==-1))
	{
	  int found;
	  FIDX *nodes=&m->edge[m->bound[bd*bd_w+MCT2BDEDGE]*eg_w
			       +MCT2EGNOD1];

	  double tR=sqrt(m->vertex[nodes[0]*vx_w+MCT2VXSTRT+0]
			 *m->vertex[nodes[0]*vx_w+MCT2VXSTRT+0]
			 +m->vertex[nodes[0]*vx_w+MCT2VXSTRT+1]
			 *m->vertex[nodes[0]*vx_w+MCT2VXSTRT+1]);

	  /* check which circle this bd belongs to */
	  found=0;
	  for (i=0; i<ncirc; i++)
	    {
	      if ((tR>(1.0-tol)*Rs[i])&&(tR<(1.0+tol)*Rs[i]))
		{
		  nRs[i]++;
		  found=1;
		}
	    }
	  if (found==0)
	    {
	      fprintf(stderr,"mesh_gnuplot_boundary_circles_t2: bd with R=%f "
		      "does not fit the circles\n",
		      tR);
	      exit(FAIL);
	    }
	} /* end use only finest edge */
    } /* end count points on each circle */

	  
  /* allocate memory for each sorter field */
  for (i=0; i<ncirc; i++)
    {
      TRY_MALLOC(sorter[i], 3*nRs[i], struct intdouble, mesh_gnuplot_boundary_circles_t2);
      iRs[i]=0;
    }

  /* collect the points on each circle */
  for (bd=0; bd<m->bd_nr; bd++)
    {
      /* use only finest edges */
      if ((m->edge[m->bound[bd*bd_w+MCT2BDEDGE]*eg_w+MCT2EGCHL1]==-1))
	{
	  FIDX *nodes=&m->edge[m->bound[bd*bd_w+MCT2BDEDGE]*eg_w
			       +MCT2EGNOD1];

	  double tR=sqrt(m->vertex[nodes[0]*vx_w+MCT2VXSTRT+0]
			 *m->vertex[nodes[0]*vx_w+MCT2VXSTRT+0]
			 +m->vertex[nodes[0]*vx_w+MCT2VXSTRT+1]
			 *m->vertex[nodes[0]*vx_w+MCT2VXSTRT+1]);
		
	  double phi0=atan2(m->vertex[nodes[0]*vx_w+MCT2VXSTRT+0],
			    m->vertex[nodes[0]*vx_w+MCT2VXSTRT+1]);
	  double phi1=atan2(m->vertex[nodes[1]*vx_w+MCT2VXSTRT+0],
			    m->vertex[nodes[1]*vx_w+MCT2VXSTRT+1]);
	  double phi2=atan2(m->vertex[nodes[2]*vx_w+MCT2VXSTRT+0],
			    m->vertex[nodes[2]*vx_w+MCT2VXSTRT+1]);
		
	  /* check which circle this bd belongs to */
	  for (i=0; i<ncirc; i++)
	    {
	      if ((tR>(1.0-tol)*Rs[i])&&(tR<(1.0+tol)*Rs[i]))
		{
		  sorter[i][iRs[i]].i=nodes[0];
		  sorter[i][iRs[i]].d=phi0;			  
		  iRs[i]++;

		  sorter[i][iRs[i]].i=nodes[1];
		  sorter[i][iRs[i]].d=phi1;			  
		  iRs[i]++;

		  sorter[i][iRs[i]].i=nodes[2];
		  sorter[i][iRs[i]].d=phi2;			  
		  iRs[i]++;
		}
	    }
	} /* end only finest edge */
    } /* end collect the points on each circle */

  /* sort the fields */
  for (i=0; i<ncirc; i++)
    {
      qsort( sorter[i], 3*nRs[i], sizeof(struct intdouble), 
	     comp_intdouble_d); 
    }

  /* now collect the points into a vector to write for
     plotting, use only pints that are actually different,
     because the corner points of the elements should be
     twice in the sorter, due to the structure of
     m->boundary */
  for (j=0; j<ncirc; j++)
    {
      FIDX old, nr;

      for (i=0; i<nwrite; i++)
	{
	  err=vector_alloc(&circle[i],2*nRs[j]);
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc,
				   mesh_gnuplot_boundary_circles_t2);
	}

      old=-1;
      nr=0;
      for (i=0; i<3*nRs[j]; i++)
	{
	  double norm;
	  FIDX k;
	  FIDX node=sorter[j][i].i;
		      
	  /* only output to the new nodes to the file */
	  if (node!=old)
	    {

	      circle[0].V[nr]=sorter[j][i].d/M_PI; // the angle
	      switch(mode)
		{
		case 0:
		  /* the data itself */
		  for (k=0; k<ncomp; k++)
		    {
		      circle[1+k].V[nr]=data->V[node+k*vx_nr];
		    }
		  break;
		case 1:
		  /* the norm of the data */
		  norm=0.0;
		  for (k=0; k<ncomp; k++)
		    {
		      norm+= data->V[node+k*vx_nr]
			* data->V[node+k*vx_nr];
		    }
		  norm=sqrt(norm);
		  circle[1].V[nr]=norm; 
		  break;
		default:
		  fprintf(stderr,"mesh_gnuplot_boundary_circles_t2: "
			  "unknown mode=%d\n",mode);
		  exit(FAIL);
		}
			
	      old=node;
	      nr++;
	    } /* end if node!=old */
	} /* end loop over sorted nodes */

      /* cut of undefined parts of the vectors */
      for (i=0; i<nwrite; i++)
	{
	  circle[i].len=nr;
	}
		
      sprintf(fname,"%s_R%"dFIDX".txt",fname_base,j);
		
      err=vector_n_write_file(nwrite, circle, fname);
      FUNCTION_FAILURE_HANDLE( err, vector_n_write_file, 
			       mesh_gnuplot_boundary_circles_t2);
		
      for (i=0; i<nwrite; i++)
	{
	  vector_free(&circle[i]);
	}

      free(sorter[j]);
    } /* end loop over circles */


  free(fname);
  free(circle);

  free(sorter);
  free(nRs);
  free(iRs);

  return SUCCESS;
}


/*FUNCTION*/
int mesh_write_bound_gnuplot_tx( struct mesh *m, 
				FIDX namlen, char *name,
				int type
/* writes the mesh into a GNUPLOT readable file (for visualisation),
   this is done by writing a line for each edge of the finest level
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, full name will be
	               <name>.dat
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2
                       in any case only straigth lines are written,
                       ignoring the possibility of curved edges for T2
                       meshes

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX eg, bd;
  FIDX bd_w, bd_nr, eg_w, eg_nr, vx_w, vx_nr;
  FIDX l_mcBDEDGE, l_mcEGCHL1, l_mcEGNOD1, l_mcVXSTRT;

  char *datname;
  FILE *datfile;


  vx_w =(*m).vx_w;
  eg_w =(*m).eg_w;
  bd_w =(*m).bd_w;

  vx_nr=(*m).vx_nr;
  eg_nr=(*m).eg_nr;
  bd_nr=(*m).bd_nr;

  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT = MCT1VXSTRT;
      l_mcEGCHL1 = MCT1EGCHL1;
      l_mcEGNOD1 = MCT1EGNOD1;
      l_mcBDEDGE = MCT1BDEDGE;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT = MCT2VXSTRT;
      l_mcEGCHL1 = MCT2EGCHL1;
      l_mcEGNOD1 = MCT2EGNOD1; 
      l_mcBDEDGE = MCT2BDEDGE;
      break;
    default:
      fprintf(stderr,"mesh_write_bound_gnuplot_tx: "
	      "unknown type=%d\n",type);
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( datname, (namlen+4), char, 
	      mesh_write_bound_gnuplot_tx );

  strncpy( datname, name, namlen );

  /* make sure no overread happens */
  datname[namlen-1]='\0';

  /* append ".dat"  */
  strcat( datname, ".dat" );
  
  /* open the lat file */
  datfile=fopen(datname, "w");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_bound_gnuplot_tx: error opening file %s\n",
	      datname);
      return FAIL;
    }
  /* write the egdes */
  for (bd=0; bd<bd_nr; bd++)
    {
      eg=(*m).bound[bd*bd_w+l_mcBDEDGE];
      if ((*m).edge[eg*eg_w+l_mcEGCHL1]==-1) 
	{
	  /* only write edges which have not been refined further */
	  FIDX n1,n2;
	  n1=(*m).edge[eg*eg_w+l_mcEGNOD1  ];
	  n2=(*m).edge[eg*eg_w+l_mcEGNOD1+1];

	  /* write first point\n, second point\n, blank line */
	  fprintf(datfile,
		  "%18.10e  %18.10e\n"
		  "%18.10e  %18.10e\n"
		  "\n",
		  (*m).vertex[n1*vx_w+l_mcVXSTRT  ],
		  (*m).vertex[n1*vx_w+l_mcVXSTRT+1],
		  (*m).vertex[n2*vx_w+l_mcVXSTRT  ],
		  (*m).vertex[n2*vx_w+l_mcVXSTRT+1]);
	}
    }


  /* close dat file */
  fclose(datfile); 

  free(datname);

  return SUCCESS;
}



/*FUNCTION*/
int mesh_write_mesh_gnuplot_tx( struct mesh *m, 
				FIDX namlen, char *name,
				int type
/* writes the mesh into a GNUPLOT readable file (for visualisation),
   this is done by writing a line for each edge of the finest level
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the file, full name will be
	               <name>.dat
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2
                       in any case only straigth lines are written,
                       ignoring the possibility of curved edges for T2
                       meshes

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX i, el, el_w, el_nr, vx_w, vx_nr;
  FIDX l_mcELNOD1, l_mc_elnodes, l_mcVXSTRT;

  char *datname;
  FILE *datfile;


  vx_w =(*m).vx_w;
  el_w =(*m).el_w;

  vx_nr=(*m).vx_nr;
  el_nr=(*m).el_nr;

  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT = MCT1VXSTRT;
      l_mcELNOD1 = MCT1ELNOD1;
      l_mc_elnodes= 3;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT = MCT2VXSTRT;
      l_mcELNOD1 = MCT2ELNOD1;
      l_mc_elnodes= 3; /* with 6 we would have to correct the numbering
		       */
      break;
    default:
      fprintf(stderr,"mesh_write_mesh_gnuplot_tx: "
	      "unknown type=%d\n",type);
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( datname, (namlen+4), char, 
	      mesh_write_mesh_gnuplot_tx );

  strncpy( datname, name, namlen );

  /* make sure no overread happens */
  datname[namlen-1]='\0';

  /* append ".dat"  */
  strcat( datname, ".dat" );
  
  /* open the lat file */
  datfile=fopen(datname, "w");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_mesh_gnuplot_tx: error opening file %s\n",
	      datname);
      return FAIL;
    }
  /* write the elements */
  for (el=0; el<el_nr; el++)
    {
      for (i=0; i<l_mc_elnodes+1; i++)
	{
	  FIDX n;
	  n=(*m).elem[el*el_w+l_mcELNOD1+(i%l_mc_elnodes)];

	  /* write all points then a blank line */
	  fprintf(datfile,
		  "%18.10e  %18.10e\n",
		  (*m).vertex[n*vx_w+l_mcVXSTRT  ],
		  (*m).vertex[n*vx_w+l_mcVXSTRT+1]);

	}
      fprintf(datfile, "\n");
    }


  /* close dat file */
  fclose(datfile); 

  free(datname);

  return SUCCESS;
}



/*FUNCTION*/
int mesh_write_ssegs_svg_tx( struct mesh *m, 
			     FIDX namlen, char *name,
			     int type
/* writes the shape-segments into a scalable vector graphics (SVG)
   file, which can for example be read by sodipodi or inkscape
   
   Input:  m         - the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, full name will be
	               <name>.svg
           type      - type of the mesh, ==1 ==> T1, ==2 ==> T2

   Output: (writes the file)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  int  err;
  FIDX i;
  FIDX eg, seg, paths;
  FIDX bd_w, bd_nr, eg_w, eg_nr, vx_w, vx_nr, sg_w, sg_nr, sp_w;
  FIDX l_mcBDSSEG, l_mcBDEDGE, l_mcEGCHL1, l_mcEGNOD1, l_mcVXSTRT;

  struct ilist **eglist; /* list of edges for each segment */
  FIDX *nd_nbs;   /* nd_nbs[i*2+j]=k,  j=1,2:  node i has neigbour k
		     within the current segment */
  FIDX *seg_nds;  /* first and last node for each segment:
                     seg_nds[seg*2+0]=first, seg_nds[seg*2+1]=last */
  FIDX *nd_segs;  /* for each node the segments which contain it as
		     first or last node k=nd_segs[node*2+i], i=0,1,
		     if k==-1 => no segment, else seg=k, node is
		     contained in segment seg */
  int  *seg_done; /* marks if an segment is already contained in a
		     path */

  /* bounding box parameters */
  double minx,maxx,miny,maxy,diam;

  char *datname, *datname_mrk, *datname_all;
  FILE *datfile, *datfile_mrk, *datfile_all;


  vx_w =(*m).vx_w;
  eg_w =(*m).eg_w;
  bd_w =(*m).bd_w;
  sg_w =(*m).sg_w;
  sp_w =(*m).sp_w;

  vx_nr=(*m).vx_nr;
  eg_nr=(*m).eg_nr;
  bd_nr=(*m).bd_nr;
  sg_nr=(*m).sg_nr;

  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT = MCT1VXSTRT;
      l_mcEGCHL1 = MCT1EGCHL1;
      l_mcEGNOD1 = MCT1EGNOD1;
      l_mcBDEDGE = MCT1BDEDGE;
      l_mcBDSSEG = MCT1BDSSEG;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT = MCT2VXSTRT;
      l_mcEGCHL1 = MCT2EGCHL1;
      l_mcEGNOD1 = MCT2EGNOD1; 
      l_mcBDEDGE = MCT2BDEDGE;
      l_mcBDSSEG = MCT2BDSSEG;
      break;
    default:
      fprintf(stderr,"mesh_write_ssegs_svg_tx: "
	      "unknown type=%d\n",type);
      return FAIL;
    }

  /*************************************/
  /*                                   */
  /* stage 1: sort out segments        */
  /*                                   */
  /*************************************/
  
  TRY_MALLOC( eglist, sg_nr, struct ilist*, mesh_write_ssegs_svg_tx);
  /* init the list */
  for (i=0; i<sg_nr; i++)
    eglist[i]=NULL;
  
  /* build a list of edges for each segment */
  for (i=0; i<(*m).bd_nr; i++)
    {
      seg = m->bound[i*bd_w+l_mcBDSSEG];
      eg  = m->bound[i*bd_w+l_mcBDEDGE];

      if ((seg!=-1)&&(m->edge[eg*eg_w+l_mcEGCHL1]==-1))
	{
	  ilist_sorted_insert( &eglist[seg], eg);
	}
    }

  TRY_MALLOC( nd_nbs, 2*vx_nr, FIDX, mesh_write_ssegs_svg_tx);
  TRY_MALLOC( nd_segs, 2*vx_nr, FIDX, mesh_write_ssegs_svg_tx);
  TRY_MALLOC( seg_nds, 2*sg_nr, FIDX, mesh_write_ssegs_svg_tx);
  TRY_MALLOC( seg_done, sg_nr, int, mesh_write_ssegs_svg_tx);

  /* clear nd_segs */
  for (i=0; i<2*vx_nr; i++)
    nd_segs[i]=-1;

  for (seg=0; seg<sg_nr; seg++)
    {
      struct ilist *this;
      FIDX tnode, pnode;      

      /**********************************************************/
      /* for each node on the edges mark its neighbouring nodes */
      /**********************************************************/
      this=eglist[seg];

      /* clear nd_nbs */
      for (i=0; i<2*vx_nr; i++)
	nd_nbs[i]=-1;

      while (this!=NULL)
	{
	  FIDX eg=this->data;
	  
	  FIDX n1= m->edge[eg*eg_w+l_mcEGNOD1  ];
	  FIDX n2= m->edge[eg*eg_w+l_mcEGNOD1+1];

	  if (nd_nbs[n1*2]==-1)
	    nd_nbs[n1*2]=n2;
	  else if (nd_nbs[n1*2+1]==-1)
	    nd_nbs[n1*2+1]=n2;
	  else 
	    {
	      fprintf(stderr,"mesh_write_ssegs_svg_tx: "
		      "node multiplicity error\n");
	      return FAIL;
	    }

	  if (nd_nbs[n2*2]==-1)
	    nd_nbs[n2*2]=n1;
	  else if (nd_nbs[n2*2+1]==-1)
	    nd_nbs[n2*2+1]=n1;
	  else 
	    {
	      fprintf(stderr,"mesh_write_ssegs_svg_tx: "
		      "node multiplicity error\n");
	      return FAIL;
	    }
	  this=this->next;
	}

      /**********************************************************/
      /* find first and last node                               */
      /**********************************************************/
      pnode=-2;                           /* previous node */
      tnode=m->sseg[seg*sg_w+MC2XSGNOD1]; /* this node     */
      seg_nds[seg*2+0] = tnode;           /* mark the first node */
      while (tnode>=0)
	{
	  FIDX nb;

	  nb=nd_nbs[tnode*2  ];
	  /* if this neighbour is the previous node, use other
	     neighbour */
	  if (nb==pnode)
	    nb=nd_nbs[tnode*2+1];
	  
	  /* continue with the neighbour */
	  pnode=tnode;
	  tnode=nb;
	}
      tnode = pnode;
      seg_nds[seg*2+1] = tnode;           /* mark the last node */

      /* mark last and first node in nd_segs */
      /* last node: */
      if (nd_segs[tnode*2+0]==-1)
	nd_segs[tnode*2+0]=seg;
      else if (nd_segs[tnode*2+1]==-1)
	nd_segs[tnode*2+1]=seg;
      else 
	{
	  fprintf(stderr, "mesh_write_ssegs_svg_tx: "
		  "one node in >2 segments!\n");
	  return FAIL;
	}
      /* first node: */
      tnode = seg_nds[seg*2+0];
      if (nd_segs[tnode*2+0]==-1)
	nd_segs[tnode*2+0]=seg;
      else if (nd_segs[tnode*2+1]==-1)
	nd_segs[tnode*2+1]=seg;
      else 
	{
	  fprintf(stderr, "mesh_write_ssegs_svg_tx: "
		  "one node in >2 segments!\n");
	  return FAIL;
	}

      /* the edge-list is not needed anymore, free it */
      ilist_free(&eglist[seg]);
      eglist[seg]=NULL;
      /* next segment */
    }/* end loop segments */

  /* find bounding box */
  err=mesh_write_sseg_bbox_tx(&minx, &maxx, &miny, &maxy, m);
  FUNCTION_FAILURE_HANDLE(err, mesh_write_sseg_bbox_tx, 
			  mesh_write_ssegs_svg_tx);

  diam=sqrt((maxx-minx)*(maxx-minx)+(maxy-miny)*(maxy-miny));

  /*************************************/
  /*                                   */
  /* stage 2: find paths and write     */
  /*                                   */
  /*************************************/

  /* now we have begin and end nodes for all segments and the segments
     for each of these nodes, 

     we need to collect them into paths now, which are then written
     into the file 

     the paths are build by starting from one segment, going into one
     direction until the end is found or a loop detected, in either
     case the path is written to the file starting from the last
     segment of the first search, since this is an end of the path in
     any case. segments which are dealt with already are marked in
     seg_done
  */

  /* prepare the names */
  TRY_MALLOC( datname, (namlen+4), char, 
	      mesh_write_ssegs_svg_tx );
  strncpy( datname, name, namlen );
  /* make sure no overread happens */
  datname[namlen-1]='\0';
  /* append ".svg"  */
  strcat( datname, ".svg" );
  /* open the svg file */
  datfile=fopen(datname, "w");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_ssegs_svg_tx: error opening file %s\n",
	      datname);
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( datname_mrk, (namlen+4+4), char, 
	      mesh_write_ssegs_svg_tx );
  strncpy( datname_mrk, name, namlen );
  /* make sure no overread happens */
  datname_mrk[namlen-1]='\0';
  /* append "_mrk.svg"  */
  strcat( datname_mrk, "_mrk.svg" );
  /* open the marker file */
  datfile_mrk=fopen(datname_mrk, "w");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_ssegs_svg_tx: error opening file %s\n",
	      datname_mrk);
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( datname_all, (namlen+4+4), char, 
	      mesh_write_ssegs_svg_tx );
  strncpy( datname_all, name, namlen );
  /* make sure no overread happens */
  datname_all[namlen-1]='\0';
  /* append "_all.svg"  */
  strcat( datname_all, "_all.svg" );


  /* SVG file header, reverse-engineered from SODIPODI output */
  fprintf(datfile, 
	  "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
	  "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20010904//EN\"\n"
	  "\"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">\n");
  fprintf(datfile, 
	  "<svg\n"
	  "   xmlns=\"http://www.w3.org/2000/svg\"\n"
	  "   xmlns:xlink=\"http://www.w3.org/1999/xlink\"\n"
	  "   version=\"1.0\"\n");
  fprintf(datfile, 
	  "   width=\"%12.6f\"\n"
	  "   height=\"%12.6f\"\n"
	  "   viewBox=\"%12.6f  %12.6f  %12.6f  %12.6f\"\n"
	  "   id=\"svg611\"\n"
	  "   xml:space=\"preserve\"><defs\n"
	  "   id=\"defs604\" />\n",
	  (maxx-minx),
	  (maxy-miny),
	  100.0*(minx-0.01*diam),
	  100.0*(miny-0.01*diam),
	  100.0*(maxx-minx+0.02*diam),
	  100.0*(maxy-miny+0.02*diam)   );


  /* mark all segments as not done yet */
  for (i=0; i<sg_nr; i++)
    seg_done[i]=0;

  seg=0;
  paths=0;
  while (seg<sg_nr)
    {
      FIDX fnode, tnode, nnode, tseg;
      int found;
      FIDX par0, par1;

      if (seg_done[seg]==0)
	{
	  /* this segment is not part of an path yet, try to find a
	     path */
	  fnode = seg_nds[seg*2+0]; /* first node in this search */
	  tnode = seg_nds[seg*2+1]; /* current end node in this search */
	  tseg  = seg;              /* current end segment in this search */
	  found = 1;
	  while (found==1) 
	    {
	      /* try to find next node on the path */
	      FIDX nseg,nnode;

	      nseg = nd_segs[tnode*2+0];
	      /* if the attempted next segment is the current one, use
		 the other one */
	      if (nseg==tseg) 
		nseg = nd_segs[tnode*2+1];

	      /* see if we have a next segment, otherwise we are done */
	      if (nseg != -1)
		{
		  nnode=seg_nds[nseg*2+0];
		  if (nnode==tnode) 
		    nnode=seg_nds[nseg*2+1];

		  /* we have found the next segment and next node, 
		     if this next node is the first, we have a loop
		     and can stop looking, otherwise we have to try
		     again */
		  tnode = nnode;
		  tseg  = nseg;
		  
		  if (tnode==fnode)
		    found = 0; /* loop, stop looking for end of path */
		  else
		    found = 1; /* continue looking for end of path */
		  
		  if (found==0) printf("loop of shape segments detected\n");
		}
	      else
		{
		  found = 0;
		  printf("line of shape segments detected (not loop)\n"); /* */
		}
	    }

	  /* now tseg and tnode contain the end segment and end node
	     of a path, build the path from this info */
	  fnode = tnode;
	  nnode=seg_nds[tseg*2+0];
	  if (nnode==tnode) 
	    nnode=seg_nds[tseg*2+1];

	  /* write the segment to the file */
	  fprintf(datfile, "    <path\n    d=\"");

	  /* first point */
	  if (tnode==m->sseg[tseg*sg_w+MC2XSGNOD1])
	    { 
	      /* correct order of parameters */
	      par0=m->sseg[tseg*sg_w+MC2XSGPAR1+0];
	      par1=m->sseg[tseg*sg_w+MC2XSGPAR1+1];
	    }
	  else
	    { 
	      /* reverse order of parameters */
	      par0=m->sseg[tseg*sg_w+MC2XSGPAR1+2];
	      par1=m->sseg[tseg*sg_w+MC2XSGPAR1+3];
	    }
	  fprintf(datfile, "M %12.6f %12.6f ", 
		  100* m->spar[par0*sp_w], 100* m->spar[par1*sp_w]);

	  err = mesh_write_sseg_svg_file_tx(datfile, datfile_mrk,  m,
					    tseg, tnode, diam);
	  FUNCTION_FAILURE_HANDLE(err, mesh_write_sseg_svg_file_tx,
				  mesh_write_ssegs_svg_tx);
	  seg_done[tseg]=1;

	  tnode = nnode;
	  found = 1; 
	  while (found==1) 
	    {
	      /* try to find next node on the path */
	      FIDX nseg,nnode;

	      nseg = nd_segs[tnode*2+0];
	      /* if the attempted next segment is the current one, use
		 the other one */
	      if (nseg==tseg) 
		nseg = nd_segs[tnode*2+1];

	      /* see if we have a next segment, otherwise we are done */
	      if (nseg != -1)
		{
		  nnode=seg_nds[nseg*2+0];
		  if (nnode==tnode) 
		    nnode=seg_nds[nseg*2+1];

		  /* we have found the next segment and next node, 
		     write the next segment to the file 
		     if this next node is the first, we have a loop
		     and can stop looking, otherwise we have to try
		     again */
		  err = mesh_write_sseg_svg_file_tx(datfile, datfile_mrk,
						    m, nseg, tnode, diam);
		  FUNCTION_FAILURE_HANDLE(err, mesh_write_sseg_svg_file_tx,
					  mesh_write_ssegs_svg_tx);
		  seg_done[nseg]=1;

		  tnode = nnode;
		  tseg  = nseg;
		  
		  if (tnode==fnode)
		    found = 0; /* loop, stop looking for end of path */
		  else
		    found = 1; /* continue looking for end of path */
		}
	      else
		found = 0;
	    } /* end while found==1 */

	  /* path complete, write final properties */
	  fprintf(datfile, "\"\n"
		  "    style=\"font-size:12;fill:none;fill-rule:evenodd;"
		  "stroke:#000000;stroke-width:0.3;\"\n"
		  "    id=\"path%"dFIDX"\" />\n",  paths);
	  paths++;

	}
      /* now this segment has definitely been asigned to a path, try
	 the next one */
      seg++;
    }

  /* all paths written, close the files, copy datfile to datfile_all,
     append datfile_mrk */
  fclose(datfile); 
  fclose(datfile_mrk); 
  {
    char *command;
    
    /*              3 times( 2 filenames    + command + spaces)   */
#ifdef __linux // For linux-only code
    TRY_MALLOC( command, 3*(2*(namlen+4+4) + 8       + 6), char, 
		mesh_write_ssegs_svg_tx );
    sprintf(command,"cat %s > %s; cat %s >> %s ; rm %s", 
	    datname, datname_all,
	    datname_mrk, datname_all, datname_mrk);
#endif
#ifdef _WIN32 // For Windows-MSVC and Windows-Cygwin, but *not* Windows-mingw
    TRY_MALLOC( command, 3*(2*(namlen+4+4) + 12       + 6), char, 
		mesh_write_ssegs_svg_tx );
    sprintf(command,"copy /Y %s + %s %s\n del %s", 
	    datname, datname_all,
	    datname_mrk, datname_all, datname_mrk);
#endif

#ifdef __MINGW32__ // For Windows-mingw
    TRY_MALLOC( command, 3*(2*(namlen+4+4) + 12       + 6), char, 
		mesh_write_ssegs_svg_tx );
    sprintf(command,"copy /Y %s + %s %s\n del %s", 
	    datname, datname_mrk, datname_all, datname_mrk);
#endif

    /* fprintf(stderr," command:\n  %s\n", command); */
    
    if (0!=system(command)) 
      {
	fprintf(stderr,
		"mesh_write_ssegs_svg_tx: error executing command\n  \"%s\"\n",
		command);
	return FAIL;
      }

    free(command);
  }
  
  /* open the svg file */
  datfile=fopen(datname, "a");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_ssegs_svg_tx: error opening file %s for appending\n",
	      datname);
      return FAIL;
    }

  /* open the all file */
  datfile_all=fopen(datname_all, "a");
  if (datfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_ssegs_svg_tx: error opening file %s for appending\n",
	      datname_all);
      return FAIL;
    }

  /* write tail of the files, close them */
  fprintf(datfile, "</svg>\n");
  fclose(datfile); 
  fprintf(datfile_all, "</svg>\n");
  fclose(datfile_all); 

  /* free local data */
  free(eglist);
  free(nd_nbs);
  free(nd_segs);
  free(seg_nds);
  free(seg_done);

  free(datname);
  free(datname_mrk);
  free(datname_all);

  return SUCCESS;
}


/*FUNCTION*/
int mesh_write_sseg_svg_file_tx(FILE *datfile, FILE *datfile_mrk,
				struct mesh *m, FIDX seg, FIDX node1,
				double diam
/* writes the segement to the files specified by the file pointers,
   depending on the type and orientation of the segment

   input:   datfile - pointer to path file, it is written to the file in
                      this routine
	    datfile_mrk
                    - pointer to marker file, parameter markers are
                      written to the file in this routine
	    m       - a t1 or t2 mesh
            seg     - the segment to be written into the file
            node1   - first node of the segment, the orientation of
                      the segment is adjusted such that it really has
                      this node as first node
            diam    - diameter of bounding box, used for scaling the markers

   output:  (writes to the file)

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
				){
  FIDX sg_w, sg_nr, sp_w;
  FIDX type;
  int orie, i;
  double lpar[8];
  double lpoint[8];
  char line_mark_color[]="#ff00ff";
  char dir_line_color[]="#0000ff";
  char dir_mark_color[]="#ff0000";
	
  sg_w =(*m).sg_w;
  sp_w =(*m).sp_w;

  sg_nr=(*m).sg_nr;

  if (seg >= sg_nr)
    {
      fprintf(stderr,"mesh_write_sseg_svg_file_tx: "
	      "seg >= sg_nr\n");
      return FAIL;
    }

  /* decide about the orientation */
  if (node1==m->sseg[seg*sg_w+MC2XSGNOD1])
    orie=1;
  else
    orie=-1;

  type = m->sseg[seg*sg_w+MC2XSGTYPE];

  switch (type)
    {
    case 1: /* line */
      for (i=0; i<4; i++)
	lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

      if (orie==1)
	fprintf(datfile, "L %12.6f %12.6f ", 100*lpar[2], 100*lpar[3]);
      else
	fprintf(datfile, "L %12.6f %12.6f ", 100*lpar[0], 100*lpar[1]);

      /* independent of the orientation, both points are written as
	 marker (rectangle) to the marker file */
      fprintf(datfile_mrk, "<rect "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegrect%"dFIDX"\"\n"
	      "width=\"%12.6f\"\n"
	      "height=\"%12.6f\"\n"
	      "x=\"%12.6f\"\n"
	      "y=\"%12.6f\" />\n",
	      line_mark_color, 1+4*seg+0,
	      2.0*diam,2.0*diam,
	      100*lpar[0]-1.0*diam, -100*lpar[1]-1.0*diam);
      fprintf(datfile_mrk, "<rect "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegrect%"dFIDX"\"\n"
	      "width=\"%12.6f\"\n"
	      "height=\"%12.6f\"\n"
	      "x=\"%12.6f\"\n"
	      "y=\"%12.6f\" />\n",
	      line_mark_color, 1+4*seg+1,
	      2.0*diam,2.0*diam,
	      100*lpar[2]-1.0*diam, -100*lpar[3]-1.0*diam);
      break;
    case 2: /* b-spline */ 
      for (i=0; i<8; i++)
	lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];
	
      /* the sxi and syi need to be adjusted for easy evaluation */
      /* the first two direction need just adding the point, to give
	 control points */
      lpoint[4]=lpar[0]+lpar[4];
      lpoint[5]=lpar[1]+lpar[5];
      /* the later two require turning around and adding the point */
      lpoint[6]=lpar[2]-lpar[6];
      lpoint[7]=lpar[3]-lpar[7];
      /* the reversed direction points */
      lpoint[0]=lpar[0]-lpar[4];
      lpoint[1]=lpar[1]-lpar[5];
      lpoint[2]=lpar[2]+lpar[6];
      lpoint[3]=lpar[3]+lpar[7];

      
      if (orie==1)
	fprintf(datfile, "C %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f ",
		100*lpoint[4], 100*lpoint[5], 100*lpoint[6], 100*lpoint[7], 
		100*lpar[2], 100*lpar[3]);
      else
	fprintf(datfile, "C %12.6f %12.6f %12.6f %12.6f %12.6f %12.6f ",
		100*lpoint[6], 100*lpoint[7], 100*lpoint[4], 100*lpoint[5],
		100*lpar[0], 100*lpar[1]);

      /* independent of the orientation, both direction markers are
	 written as to the marker file (line, diamond, circle)*/
      /***** lines *****/
      fprintf(datfile_mrk, "<path "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegline%"dFIDX"\"\n"
	      "d=\"M %12.6f %12.6f L %12.6f %12.6f\"\n"
	      "/>\n",
	      dir_line_color, 1+4*seg+0,
	      100*lpoint[0],-100*lpoint[1],
	      100*lpoint[4],-100*lpoint[5]);
      fprintf(datfile_mrk, "<path "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegline%"dFIDX"\"\n"
	      "d=\"M %12.6f %12.6f L %12.6f %12.6f\"\n"
	      "/>\n",
	      dir_line_color, 1+4*seg+1,
	      100*lpoint[2],-100*lpoint[3],
	      100*lpoint[6],-100*lpoint[7]);
      
      /***** diamonds *****/
      fprintf(datfile_mrk, "<rect "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegrect%"dFIDX"\"\n"
	      "width=\"%12.6f\"\n"
	      "height=\"%12.6f\"\n"
	      "x=\"%12.6f\"\n"
	      "y=\"%12.6f\"\n"
	      "transform=\"matrix(0.707,0.707,-0.707,0.707,0,0)\"\n"
	      "/>\n",
	      dir_mark_color, 1+4*seg+0,
	      2.0*diam,2.0*diam,
	      (0.707*100*lpar[0]-0.707*100*lpar[1])-1.0*diam,
	      -(0.707*100*lpar[0]+0.707*100*lpar[1])-1.0*diam );
      fprintf(datfile_mrk, "<rect "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegrect%"dFIDX"\"\n"
	      "width=\"%12.6f\"\n"
	      "height=\"%12.6f\"\n"
	      "x=\"%12.6f\"\n"
	      "y=\"%12.6f\"\n"
	      "transform=\"matrix(0.707,0.707,-0.707,0.707,0,0)\"\n"
	      "/>\n",
	      dir_mark_color, 1+4*seg+1,
	      2.0*diam,2.0*diam,
	      (0.707*100*lpar[2]-0.707*100*lpar[3])-1.0*diam,
	      -(0.707*100*lpar[2]+0.707*100*lpar[3])-1.0*diam );

      /***** circles *****/
      fprintf(datfile_mrk, "<path "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegcirc%"dFIDX"\"\n"
	      "d=\"M %12.6f %12.6f A 1 1 0 1 0 %12.6f %12.6f "
	      "A 1 1 0 1 0  %12.6f %12.6f z\"\n"
	      "/>\n",
	      dir_mark_color, 1+4*seg+0,
	      100*lpoint[2]+1.0*diam,-100*lpoint[3],
	      100*lpoint[2]-1.0*diam,-100*lpoint[3],
	      100*lpoint[2]+1.0*diam,-100*lpoint[3] );

      fprintf(datfile_mrk, "<path "
	      "style=\"fill:none;stroke:%s;stroke-width:0.1;\"\n"
	      "id=\"ssegcirc%"dFIDX"\"\n"
	      "d=\"M %12.6f %12.6f A 1 1 0 1 0 %12.6f %12.6f "
	      "A 1 1 0 1 0  %12.6f %12.6f z\"\n"
	      "/>\n",
	      dir_mark_color, 1+4*seg+1,
	      100*lpoint[4]+1.0*diam,-100*lpoint[5],
	      100*lpoint[4]-1.0*diam,-100*lpoint[5],
	      100*lpoint[4]+1.0*diam,-100*lpoint[5] );
      break;
    default:
      fprintf(stderr,"mesh_write_sseg_svg_file_tx: "
	      "unknown segment type=%"dFIDX"\n",  type);
      return FAIL;
    }

  return SUCCESS;
}


/*FUNCTION*/
int mesh_write_sseg_bbox_tx(double *minx, double *maxx, double *miny, double *maxy,
			    struct mesh *m
/* find the bounding box for all shape segments in the file

   input:   m       - a t1 or t2 mesh
            
   output:  minx,maxx,miny,maxy
                    - the min and max of all coordinates of shape
                      segment control points

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/
				){
  FIDX seg, sg_w, sg_nr, sp_w;
  FIDX type;
  int i;
  double lpar[8];
  double lpoint[8];
	
  sg_w =(*m).sg_w;
  sp_w =(*m).sp_w;

  sg_nr=(*m).sg_nr;

  if (sg_nr<1)
    {
      fprintf(stderr,"mesh_write_sseg_bbox_tx: "
	      "warning, no shape segments, return default bbox\n");
      
      *minx=0;
      *maxx=1;
      *miny=0;
      *maxy=1;

      return SUCCESS;
    }
  

  for (seg=0; seg<sg_nr; seg++)
    {
      double lminx,lmaxx,lminy,lmaxy;
      
      type = m->sseg[seg*sg_w+MC2XSGTYPE];
      
      switch (type)
	{
	case 1: /* line */
	  for (i=0; i<4; i++)
	    lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

	  lminx=fmin(lpar[0],lpar[2]);
	  lminy=fmin(lpar[1],lpar[3]);

	  lmaxx=fmax(lpar[0],lpar[2]);
	  lmaxy=fmax(lpar[1],lpar[3]);
	  break;
	case 2: /* b-spline */ 
	  for (i=0; i<8; i++)
	    lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];
	
	  /* the sxi and syi need to be adjusted for easy evaluation */
	  /* the first two direction need just adding the point, to give
	     control points */
	  lpoint[4]=lpar[0]+lpar[4];
	  lpoint[5]=lpar[1]+lpar[5];
	  /* the later two require turning around and adding the point */
	  lpoint[6]=lpar[2]-lpar[6];
	  lpoint[7]=lpar[3]-lpar[7];
	  /* the reversed direction points */
	  lpoint[0]=lpar[0]-lpar[4];
	  lpoint[1]=lpar[1]-lpar[5];
	  lpoint[2]=lpar[2]+lpar[6];
	  lpoint[3]=lpar[3]+lpar[7];

	  /* find local min and max */
	  lminx=lpoint[0];
	  lmaxx=lpoint[0];
	  lminy=lpoint[1];
	  lmaxy=lpoint[1];
	  for (i=1; i<4; i++)
	    {
	      if (lpoint[i*2+0]<lminx) lminx=lpoint[i*2+0];
	      if (lpoint[i*2+1]<lminy) lminy=lpoint[i*2+1];

	      if (lpoint[i*2+0]>lmaxx) lmaxx=lpoint[i*2+0];
	      if (lpoint[i*2+1]>lmaxy) lmaxy=lpoint[i*2+1];
	    }
	  
	  break;
	default:
	  fprintf(stderr," mesh_write_sseg_bbox_tx: "
		  "unknown segment type=%"dFIDX"\n",  type);
	  return FAIL;
	}

      if (seg==0)
	{
	  *minx=lminx;
	  *maxx=lmaxx;
	  *miny=lminy;
	  *maxy=lmaxy;
	}
      else
	{
	  if (lminx<(*minx)) *minx=lminx;
	  if (lminy<(*miny)) *miny=lminy;

	  if (lmaxx>(*maxx)) *maxx=lmaxx;
	  if (lmaxy>(*maxy)) *maxy=lmaxy;
	}
      /* fprintf(stderr,"seg=%2d minx=%12.6f maxx=%12.6f miny=%12.6f maxy=%12.6f\n",
	 seg,*minx,*maxx,*miny,*maxy); */
    } /* end loop over segments */

  return SUCCESS;
}




/*FUNCTION*/
int mesh_read_file_t1( struct mesh *m, struct solver_settings *set, char *name
/* reads a mesh file, for a descrition of the file format, see the
   example file "mesh_example.f1m"

   input:   name    - string containing the file name of the file to
                      be read
   
   output:  m       - (given by reference) a t1 mesh, it is assumed to
                      be empty and is initialised in this routine
	    set     - holding solver_settings

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/
		       ){
  int err;
  long secstart;
  FIDX format, dim, problem, meshgen, vx_nr, el_nr, bd_nr, holes,
    ps_nr, pv_nr, pc_nr, fn_nr, pa_nr, sg_nr, sp_nr, i; 
  char buff[1024];

  FILE *in;

  mesh_init(m);

  /* open the file */
  in = fopen( name, "r" );
  if (in == NULL)
    {
      fprintf(stderr, "mesh_read_file_t1: couldn't open file: %s\n",
	      name);
      return FAIL;
    }

  /* bin all the crap before <header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"<header>")!=0));
  if (strcmp(buff,"<header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_t1: file has no header: %s\n",
	      name);
      return FAIL;
    }

  /* save the begin of the header to secstart */
  secstart = ftell(in);

  /* get each of the header fields */
  err=mesh_read_header_key_int(in, "format", &format);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  if (format!=0)
    {
      fprintf(stderr, 
	      "mesh_read_file_t1: format of file unknown: %s\n",
	      name);
      return FAIL;
    } 

  err=mesh_read_header_key_int(in, "dim", &dim);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "problem", &problem);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "meshgen", &meshgen);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET); 
  (*m).meshgen = meshgen;
  (*m).meshgenDATA = NULL;

  err=mesh_read_header_key_int(in, "vertex", &vx_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "elements", &el_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "boundary", &bd_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "holes", &holes);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pcsurf", &ps_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pcvol", &pv_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "pccrit", &pc_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "function", &fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "parameter", &pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "shape_seg", &sg_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "shape_par", &sp_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_header_key_int,mesh_read_file_t1);
  fseek(in, secstart, SEEK_SET);

  /* jump to </header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"</header>")!=0));
  if (strcmp(buff,"</header>")!=0)
    {
      fprintf(stderr, "mesh_read_file_t1: file corrupt header: %s\n",
	      name);
      return FAIL;
    }

  /* save the end of the header to secstart */
  secstart = ftell(in);

  /* header successfully read, now set the mesh */
  (*m).dim     = dim;
  (*m).problem = problem;

  /* set width of fields */
  (*m).vx_w = MCT1VXLN;
  (*m).el_w = MCT1ELLN;
  (*m).eg_w = MCT1EGLN;
  (*m).fc_w = MCT1FCLN;
  (*m).vo_w = MCT1VOLN;
  (*m).hi_w = MCT1HILN;
  (*m).bd_w = MCT1BDLN;
  (*m).ps_w = MCXXPSLN;
  (*m).pv_w = MCXXPVLN;
  (*m).pc_w = MCXXPCLN;
  (*m).fu_w = MC2XFULN;
  (*m).pa_w = MC2XPALN;
  (*m).sg_w = MC2XSGLN;
  (*m).sp_w = MC2XSPLN;

  err=mesh_read_solver_settings( in, set);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_solver_settigs, 
			  mesh_read_file_t1);

  /* rewind in case no settings are given */
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_vertex( in, m, vx_nr, MCT1VXSTRT);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_vertex,mesh_read_file_t1);

  err=mesh_read_elements_t1( in, m, el_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_elements_t1,mesh_read_file_t1);

  err=mesh_read_boundary_t1( in, m, bd_nr, sg_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_boundary_t1,mesh_read_file_t1);

  /* save the position to secstart */
  secstart = ftell(in);
  err=mesh_read_pccrit( in, m, pc_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_pccrit,mesh_read_file_t1);

  /* rewind to read the pcsurf part */
  fseek(in, secstart, SEEK_SET);
  err=mesh_read_pcsurf( in, m, ps_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_pcsurf,mesh_read_file_t1);

  err=mesh_read_pcvol( in, m, pv_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_pcsurf,mesh_read_file_t1);

  /* read the shape segments and shape parameter */
  if (sg_nr>0)
    {
      err=mesh_read_sseg( in, m, sg_nr, sp_nr);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_sseg,mesh_read_file_t1);

      err=mesh_read_spar( in, m, sp_nr);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_spar,mesh_read_file_t1);
    }

  /* rewind to read the other stuff */
  fseek(in, secstart, SEEK_SET);

  /* create an empty hierarchy part */
  (*m).hi_max=10;
  (*m).hi_nr =0;
  TRY_MALLOC( (*m).hier, (*m).hi_max*(*m).hi_w, FIDX, 
	      mesh_read_file_t1);

  /* create empty st */
  (*m).st_max=vx_nr;
  (*m).st_nr =vx_nr;
  TRY_MALLOC( (*m).st, (*m).st_max, double, 
	      mesh_read_file_t1);
  for (i=0; i<(*m).st_max; i++) (*m).st[i]=-1.0;
	    
  err=mesh_spline_t_init_tx(m,1);
  FUNCTION_FAILURE_HANDLE(err, mesh_spline_t_init_tx, mesh_read_file_t1);
  
  
  /* call the appropriate mesh-generator-wrapper (reading the holes
     section is left to that)  */
  switch(meshgen)
    {
    case 0: /* no mesh generator */
      if (sg_nr>0)
	{
	  /* adjust the position of nodes on the parametric boundary */
	  /*                    msh, insert, nI, dIdx, dIdF, type */
	  err=mesh_sseg_adjust_tx( m, 0,     0, NULL, NULL, 1); 
	  FUNCTION_FAILURE_HANDLE(err, mesh_sseg_adjust_tx,mesh_read_file_t1);
	}	  
      break;
    case 1: /* triangle, call the wrapper */
      if (sg_nr>0)
	{
	  /* adjust the position of nodes on the parametric boundary,
	     refine curved parts to give better resolution if necessary */
	  /*                    msh, insert, nI, dIdx, dIdF, type */
	  err=mesh_sseg_adjust_tx( m, 1,      0, NULL, NULL, 1); 
	  FUNCTION_FAILURE_HANDLE(err, mesh_sseg_adjust_tx,
		mesh_read_file_t1);
	}	  
      if (pv_nr!=0)
	{
	  /* if the triangle mesh generator is used no pcvol entries
	     may be defined in the file, but for each volume criterion
	     in pccrit a pcvol entry is created for each element  */
	  fprintf(stderr, "mesh_read_file_t1: "
		  "pv_nr!=0 incompatible with meshgen=%"dFIDX"\n",
		   meshgen);
	  return FAIL;
	}
      err=mesh_read_triangle_wrapper( in, m, holes);
      FUNCTION_FAILURE_HANDLE(err, mesh_read_triangle_wrapper,
	      mesh_read_file_t1);
      /* in case there are new nodes which need a t-value, check
	 sufficient size of m.st */
      if ( (*m).vx_nr > (*m).st_max )
	{
	  err=mesh_more_st (m, (*m).vx_nr-(*m).st_max);
	  FUNCTION_FAILURE_HANDLE(err, mesh_more_st, mesh_read_file_t1);
	}

      /* get the t-value for the new nodes on the segments */      
      err=mesh_spline_t_fix_new_nodes(m,1);
      FUNCTION_FAILURE_HANDLE(err, mesh_spline_t_fix_new_nodes, 
			      mesh_read_file_t1);
      break;
    default: /* unknown, cry */
      fprintf(stderr, 
	      "mesh_read_file_t1: mesh generator unknown: %"dFIDX"\n",
	       meshgen);
      return FAIL;
    }
    
  err=mesh_read_compute_conns_t1(m);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_compute_conns_t1, mesh_read_file_t1);

  /* Sort the edges in each element for green refinement (longest edge
     is first one), only for dim=2 */
  if (dim==2)
    {
      double length[3];
      double x;	    
      FIDX eg, i, j, k, node1, node2;	    
	    
      /* first, clear the vertices in the element data 
      for (i=0; i<(*m).el_nr; i++)
	for (j=0; j<MCT1ELLN; j++);
	  (*m).elem[i*(*m).el_w+MCT1ELNOD1+j]=-1; /* */

      /* Now compute the lengths of the edges on each face */
      for (i=0; i<(*m).fc_nr; i++)
        {
	  for (j=0; j<3; j++)
	    {
	      length[j] = 0.0;
	      eg = (*m).face[i*(*m).fc_w+MCT1FCEDG1+j];
	      node1 = (*m).edge[eg*(*m).eg_w+MCT1EGNOD1  ];
	      node2 = (*m).edge[eg*(*m).eg_w+MCT1EGNOD1+1];
	      for (k=0; k<dim; k++)
	        {      
	          x = (*m).vertex[node1*(*m).vx_w+MCT1VXSTRT+k]
		      -(*m).vertex[node2*(*m).vx_w+MCT1VXSTRT+k];
		  length[j] += x*x ;
		}
	    }
	  /* check if the first edge is not the longest edge */
	  if (length[1]>length[0])  
	    {
	      /* switch first and second edge */ 
	      eg = (*m).face[i*(*m).fc_w+MCT1FCEDG1  ];
	      (*m).face[i*(*m).fc_w+MCT1FCEDG1  ] =
		  (*m).face[i*(*m).fc_w+MCT1FCEDG1+1];
	      (*m).face[i*(*m).fc_w+MCT1FCEDG1+1] = eg;
	      /* also switch first and third vertix 	    */
	      node1= (*m).elem[i*(*m).el_w+MCT1ELNOD1  ];	    
	      (*m).elem[i*(*m).el_w+MCT1ELNOD1  ] =
		  (*m).elem[i*(*m).el_w+MCT1ELNOD1+2];		    
	      (*m).elem[i*(*m).el_w+MCT1ELNOD1+2] = node1;	/* */
	     length[0] = length[1];
	    }
	  if (length[2]>length[0])  
	    {
	      /* switch first and third edge */ 
	      eg = (*m).face[i*(*m).fc_w+MCT1FCEDG1  ];
	      (*m).face[i*(*m).fc_w+MCT1FCEDG1 ] =
		  (*m).face[i*(*m).fc_w+MCT1FCEDG1+2];
	      (*m).face[i*(*m).fc_w+MCT1FCEDG1+2] = eg;
	      /* also switch second and third vertix     */
	      node1= (*m).elem[i*(*m).el_w+MCT1ELNOD1+1];	    
	      (*m).elem[i*(*m).el_w+MCT1ELNOD1+1] =
		  (*m).elem[i*(*m).el_w+MCT1ELNOD1+2];		    
	      (*m).elem[i*(*m).el_w+MCT1ELNOD1+2] = node1;	/* */	    
	    }
	  /* The first edge is now the longest edge */
	}
    }	    
	    
  err=mesh_read_function( in, m, fn_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_function,mesh_read_file_t1);

  err=mesh_read_parameter( in, m, pa_nr);
  FUNCTION_FAILURE_HANDLE(err, mesh_read_parameter,mesh_read_file_t1);

  fclose(in);

  return SUCCESS;
}

/*FUNCTION*/
int mesh_read_header_key_int( FILE *in, char *keywd, FIDX *value
/* reads the keyword part of the header of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m" 
   here a entry 
     <keywd N > 
   is looked up, and the integer number N is returned in value 

   input:   in      - a file open for reading, reading position at the
                      begin of the header part
	    keywd   - keyword which shall be looked up
   
   output:  value   - (given by reference) the integer number found
                      with for the keyword

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX  ihlp;
  char buff[1024];
  char *xkey;

  TRY_MALLOC(xkey, (strlen(keywd)+3), char, mesh_read_header_key_int);
  
  strcpy(xkey, "<");
  strcat(xkey, keywd);

  strcpy(buff, ""); /* to avoid uninitialised buffer if no reading is
		       possible */

  /* jump to </header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,xkey)!=0));
  if (strcmp(buff, xkey)!=0)
    {
      fprintf(stderr, "mesh_read_header_key_int: key not found: %s\n",
	      xkey);

      free(xkey);
      return FAIL;
    }
  if ((fscanf(in, "%"dFIDX" %s", &ihlp, buff)!=2)||(strcmp(buff,">")!=0))
    {
      fprintf(stderr, 
	      "mesh_read_header_key_int: key found, but unable to read "
	      "value: %s\n", xkey);

      free(xkey);
      return FAIL;
    }
  *value = (FIDX) ihlp;

  free(xkey);
  return SUCCESS;
}

/*FUNCTION*/
int mesh_read_header_key_double( FILE *in, char *keywd, double *value
/* reads the keyword part of the header of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m" 
   here a entry 
     <keywd N > 
   is looked up, and the double number N is returned in value 

   input:   in      - a file open for reading, reading position at the
                      begin of the header part
	    keywd   - keyword which shall be looked up
   
   output:  value   - (given by reference) the double number found
                      with for the keyword

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  double dhlp;
  char buff[1024];
  char *xkey;

  TRY_MALLOC(xkey, (strlen(keywd)+3), char, mesh_read_header_key_double);
  
  strcpy(xkey, "<");
  strcat(xkey, keywd);

  strcpy(buff, ""); /* to avoid uninitialised buffer if no reading is
		       possible */

  /* jump to </header> */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,xkey)!=0));
  if (strcmp(buff, xkey)!=0)
    {
      fprintf(stderr, "mesh_read_header_key_double: key not found: %s\n",
	      xkey);

      free(xkey);
      return FAIL;
    }
  if ((fscanf(in, "%lg %s", &dhlp, buff)!=2)||(strcmp(buff,">")!=0))
    {
      fprintf(stderr, 
	      "mesh_read_header_key_double: key found, but unable to read "
	      "value: %s\n", xkey);

      free(xkey);
      return FAIL;
    }
  *value = (double) dhlp;

  free(xkey);
  return SUCCESS;
}

/*FUNCTION*/
int mesh_read_solver_settings( FILE *in, struct solver_settings *set
/* reads the solver settings of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading

   output:  set     - holding the settings for the solver
   
   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  int err;
  char buff[1024];
  double max_vx_d;
  long secstart;


  /* bin all the crap before <solver_settings> */
  while ((fscanf(in, "%1023s", buff)==1)
	 &&(strcmp(buff,"<solver_settings>")!=0));
  if (strcmp(buff,"<solver_settings>")!=0)
    {
      fprintf(stderr, "Warning: mesh_read_file_t1:"
	      " file has no solver_settings\n");
    }
  /* init solver settings */
  solver_settings_init( set );

  /* save the begin of this block to secstart */
  secstart = ftell(in);


  /* read all parameters one by one */
  err=mesh_read_header_key_int(in, "refine_ini", 
			       &((*set).refine_ini));
  if (err!=SUCCESS) 
    {
      (*set).refine_ini=0;
      fprintf(stderr, "Warning: 'refine_ini' not set, "
	      "using default value, refine_ini=%"dFIDX"\n", 
	      (*set).refine_ini);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "refine_type", 
			       &((*set).refine_type));
  if (err!=SUCCESS) 
    {
      (*set).refine_type=0;
      fprintf(stderr, "Warning: 'refine_type' not set, "
	      "using default value, refine_type=%"dFIDX"\n", 
	      (*set).refine_type);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "refine_steps", 
			       &((*set).refine_steps));
  if (err!=SUCCESS) 
    {
      (*set).refine_steps=0;
      fprintf(stderr, "Warning: 'refine_steps' not set, "
	      "using default value, refine_steps=%"dFIDX"\n", 
	      (*set).refine_steps);
    }  
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "refine_stop", 
			       &((*set).refine_stop));
  if (err!=SUCCESS) 
    {
      (*set).refine_stop=0.0;
      fprintf(stderr, "Warning: 'refine_stop' not set, "
	      "using default value, refine_stop=%e\n", 
	      (*set).refine_stop);
    }  
  fseek(in, secstart, SEEK_SET);

  max_vx_d = 1e4;
  err=mesh_read_header_key_double(in, "refine_max_vx", 
			       &max_vx_d);
  (*set).refine_max_vx = (FIDX) max_vx_d;
  if (err!=SUCCESS) 
    {
      fprintf(stderr, "Warning: 'refine_max_vx' not set, "
	      "using default value, refine_max_vx=%"dFIDX"\n", 
	      (*set).refine_max_vx);
    }  
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "adap_mark", 
			       &((*set).adap_mark));
  if (err!=SUCCESS) 
    {
      (*set).adap_mark=0;
      fprintf(stderr, "Warning: 'adap_mark' not set, "
	      "using default value, adap_mark=%"dFIDX"\n", 
	      (*set).adap_mark);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "adap_mark_par", 
				  &((*set).adap_mark_par));
  if (err!=SUCCESS) 
    {
      (*set).adap_mark_par=0.5;
      fprintf(stderr, "Warning: 'adap_mark_par' not set, "
	      "using default value, adap_mark_par=%e\n", 
	      (*set).adap_mark_par);
    }
  fseek(in, secstart, SEEK_SET);




  err=mesh_read_header_key_int(in, "solver", 
			       &((*set).solver));
  if (err!=SUCCESS) 
    {
      (*set).solver=0;
      fprintf(stderr, "Warning: 'solver' not set, "
	      "using default value, solver=%"dFIDX"\n", 
	      (*set).solver);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "solver_atol", 
				  &((*set).solver_atol));
  if (err!=SUCCESS) 
    {
      (*set).solver_atol=1.0e-12;
      fprintf(stderr, "Warning: 'solver_atol' not set, "
	      "using default value, solver_atol=%e\n", 
	      (*set).solver_atol);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "solver_ini_rtol",
				  &((*set).solver_ini_rtol));
  if (err!=SUCCESS) 
    {
      (*set).solver_ini_rtol=1.0e-6;
      fprintf(stderr, "Warning: 'solver_ini_rtol' not set, "
	      "using default value, solver_ini_rtol=%e\n", 
	      (*set).solver_ini_rtol);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "solver_ref_rtol", 
				  &((*set).solver_ref_rtol));
  if (err!=SUCCESS) 
    {
      (*set).solver_ref_rtol=1.0e-2;
      fprintf(stderr, "Warning: 'solver_ref_rtol' not set, "
	      "using default value, solver_ref_rtol=%e\n", 
	      (*set).solver_ref_rtol);
    }
  fseek(in, secstart, SEEK_SET);


  err=mesh_read_header_key_int(in, "is_instat", 
			       &((*set).is_instat));
  if (err!=SUCCESS) 
    {
      (*set).is_instat=0;
      fprintf(stderr, "Warning: 'is_instat' not set, "
	      "using default value, is_instat=%"dFIDX"\n", 
	      (*set).is_instat);
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "instat_delta_t0", 
				  &((*set).instat_delta_t0));
  if (err!=SUCCESS) 
    {
      (*set).instat_delta_t0=-1.0;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_delta_t0' not set "
		  " even though is_instat==1, "
		  "using default value, instat_delta_t0=%e\n", 
		  (*set).instat_delta_t0);
	}
    }
  fseek(in, secstart, SEEK_SET);


  err=mesh_read_header_key_double(in, "instat_delta_t_fix", 
				  &((*set).instat_delta_t_fix));
  if (err!=SUCCESS) 
    {
      (*set).instat_delta_t_fix=-1.0;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_delta_t_fix' not set "
		  " even though is_instat==1, "
		  "using default value, instat_delta_t_fix=%e\n", 
		  (*set).instat_delta_t_fix);
	}
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "instat_Time0", 
				  &((*set).instat_Time0));
  if (err!=SUCCESS) 
    {
      (*set).instat_Time0=0.0;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_Time0' not set "
		  " even though is_instat==1, "
		  "using default value, instat_Time0=%e\n", 
		  (*set).instat_Time0);
	}
    }
  fseek(in, secstart, SEEK_SET);


  err=mesh_read_header_key_double(in, "instat_Tend", 
				  &((*set).instat_Tend));
  if (err!=SUCCESS) 
    {
      (*set).instat_Tend=(*set).instat_Time0;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_Tend' not set "
		  " even though is_instat==1, "
		  "using default value, instat_Tend=%e\n", 
		  (*set).instat_Tend);
	}
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "instat_write_steps", 
				  &((*set).instat_write_steps));
  if (err!=SUCCESS) 
    {
      (*set).instat_write_steps=1;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_write_steps' not set "
		  " even though is_instat==1, "
		  "using default value, instat_write_steps=%"dFIDX"\n", 
		  (*set).instat_write_steps);
	}
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_double(in, "instat_write_delta", 
				  &((*set).instat_write_delta));
  if (err!=SUCCESS) 
    {
      (*set).instat_write_delta=-1.0;
      if ((*set).is_instat==1)
	{
	  fprintf(stderr, "Warning: 'instat_write_delta' not set "
		  " even though is_instat==1, "
		  "using default value, instat_write_delta=%e\n", 
		  (*set).instat_write_delta);
	}
    }
  fseek(in, secstart, SEEK_SET);


  err=mesh_read_header_key_int(in, "write_mesh", 
			       &((*set).write_mesh));
  if (err!=SUCCESS) 
    {
      (*set).write_mesh=0;
    }
  fseek(in, secstart, SEEK_SET);

  err=mesh_read_header_key_int(in, "write_ssegs", 
			       &((*set).write_ssegs));
  if (err!=SUCCESS) 
    {
      (*set).write_ssegs=0;
    }
  fseek(in, secstart, SEEK_SET);

  /* search the end of the solver_settings block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp(buff,"</solver_settings>")!=0));
  if (strcmp(buff, "</solver_settings>")!=0)
    {
      fprintf(stderr, "mesh_read_file_settings: spurious end of header block "
	      "(no </solver_settings> ?)\n");
    }
  return SUCCESS;
}

/*FUNCTION*/
void solver_settings_init(struct solver_settings *set 
/* initialises the solver settings to be empty and its valuse to be zero

   Output: set        - empty solver settings, all values are set to zero
*/
      		      ){
  (*set).refine_ini=0;
  (*set).refine_type=0;
  (*set).refine_steps=0;
  (*set).refine_stop=0.0;
  (*set).refine_max_vx=0;


  (*set).adap_mark=0;
  (*set).adap_mark_par=0.0;


  (*set).solver=0;
  (*set).solver_atol=0.0;
  (*set).solver_ini_rtol=0.0;
  (*set).solver_ref_rtol=0.0;


  (*set).is_instat=0;
  (*set).instat_delta_t0=-1.0;
  (*set).instat_delta_t_fix=-1.0;
  (*set).instat_Time0=0.0;
  (*set).instat_Tend=0.0;
  (*set).instat_write_steps=1;
  

  (*set).write_mesh=0;
  (*set).write_ssegs=0;
  return;
}


/*FUNCTION*/
int mesh_read_vertex( FILE *in, struct mesh *m, FIDX vx_nr, FIDX vxstrt
/* reads the vertex part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            vx_nr   - the number of vertices in the file, as given by
                      the header
	    vxstrt  - the position within the vertex blocks, where the
	              coordinates of the vertex start
   
   output:  m       - (given by reference) a mesh, the vertex part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX i, j;
  FIDX ihlp;
  double dhlp;
  char buff[1024];

  TRY_MALLOC((*m).vertex, vx_nr * (*m).vx_w , double,
	     mesh_read_vertex);
  (*m).vx_max = vx_nr;

  /* find the begin of the vertex block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<vertex>",buff)!=0));
  if (strcmp("<vertex>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_vertex: file has no <vertex>\n");
      return FAIL;
    }

  /* read all the vertices one by one */
  for (i=0; i<vx_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(ihlp!=i))
	{
	  fprintf(stderr, "mesh_read_vertex: order of entries wrong!\n");
	  return FAIL;
	}
      for (j=0; j<(*m).dim; j++)
	{
	  if (fscanf(in, "%lg", &dhlp) !=1)
	    {
	      fprintf(stderr,
		      "mesh_read_vertex: error reading entry %"dFIDX"!\n",
		       i);
	      return FAIL;
	    }
	  (*m).vertex[i*(*m).vx_w+vxstrt+j] = dhlp;
	}
    }
  (*m).vx_nr  = vx_nr;
  
  /* check if this is really the end of the vertex block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</vertex>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_vertex: spurious end of vertex block "
	      "(no </vertex> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}



/*FUNCTION*/
int mesh_read_elements_t1( FILE *in, struct mesh *m, FIDX el_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            el_nr   - the number of elements in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the elem and face
                      part are allocated here, but only elem set!
                      in face only the RHSF part is defined and PCVL
                      is initiated
   

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  FIDX ihlp;
  FIDX i, j;
  char buff[1024];

  TRY_MALLOC((*m).elem, el_nr * (*m).el_w , FIDX,
	     mesh_read_elements_t1);
  (*m).el_max = el_nr;

  TRY_MALLOC((*m).face, el_nr * (*m).fc_w , FIDX,
	     mesh_read_elements_t1);
  (*m).fc_max = el_nr;

  /* find the begin of the elements block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<elements>",buff)!=0));
  if (strcmp("<elements>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_elements_t1: file has no <elements>\n");
      return FAIL;
    }

  /* read all the elements one by one */
  for (i=0; i<el_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(FIDX) (ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_elements_t1: order of entries wrong!\n");
	  return FAIL;
	}
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(ihlp!=1))
	{
	  fprintf(stderr,
		  "mesh_read_elements_t1: element %"dFIDX" not t1!\n", i);
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_elements_t1: entry %"dFIDX" deffect\n", i);
	  return FAIL;
	}
      (*m).face[i*(*m).fc_w+MCT1FCRHSF ]= (FIDX) ihlp;
      (*m).face[i*(*m).fc_w+MCT1FCPCVL ]= -1;
 
      for (j=0; j<3; j++)
	{
	  if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	    {
	      fprintf(stderr,
		      "mesh_read_elements_t1: error reading entry %"dFIDX"!\n",
		       i);
	      return FAIL;
	    }
	  (*m).elem[i*(*m).el_w+MCT1ELNOD1+j] = (FIDX) ihlp;
	}
    }
  (*m).el_nr  = el_nr;
  (*m).fc_nr  = el_nr;

  /* check if this is really the end of the elements block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</elements>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_elements_t1: "
	      "spurious end of elements block (no </elements> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_boundary_t1( FILE *in, struct mesh *m, FIDX bd_nr, FIDX sg_nr
/* reads the boundary part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            bd_nr   - the number of boundary entries in the file, as
                      given by the header
            sg_nr   - the number of shape segment entries in the file,
	              as given by the header
   
   output:  m       - (given by reference) a mesh, the edge and bound
                      part are allocated and set here, but in edge
                      only those edges are defined which are part of
                      the boundary!

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  FIDX ihlp;
  FIDX i, j;
  char buff[1024];

  TRY_MALLOC((*m).edge, bd_nr * (*m).eg_w , FIDX,
	     mesh_read_boundary_t1);
  (*m).eg_max = bd_nr;

  TRY_MALLOC((*m).bound, bd_nr * (*m).bd_w , FIDX,
	     mesh_read_boundary_t1);
  (*m).bd_max = bd_nr;

  /* find the begin of the boundary block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<boundary>",buff)!=0));
  if (strcmp("<boundary>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_boundary_t1: file has no <boundary>\n");
      return FAIL;
    }

  /* read all the boundary one by one */
  for (i=0; i<bd_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_boundary_t1: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_t1: "
		  "read error bound %"dFIDX" \n", i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCT1BDTYPE]= (FIDX) ihlp;

      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||(ihlp!=0))
	{
	  fprintf(stderr, "mesh_read_boundary_t1: "
		  "boundary element %"dFIDX" not t1!\n",  i);
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_t1: "
		  "read error bound %"dFIDX" \n",  i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCT1BDFNCT]= (FIDX) ihlp;

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_t1: "
		  "read error bound %"dFIDX" \n",  i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCT1BDORIE]= (FIDX) ihlp;

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_boundary_t1: "
		  "read error bound %"dFIDX" \n",  i);
	  return FAIL;
	}
      if ((ihlp!=-1)&&(ihlp>=sg_nr))
	{
	  fprintf(stderr,"mesh_read_boundary_t1: "
		  "shape segment out of bounds, bound %"dFIDX" \n",  i);
	  return FAIL;
	}
      (*m).bound[i*(*m).bd_w+MCT1BDSSEG]= (FIDX) ihlp;
 
      for (j=0; j<2; j++)
	{
	  if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	    {
	      fprintf(stderr, "mesh_read_boundary_t1: "
		      "error reading entry %"dFIDX"!\n",  i);
	      return FAIL;
	    }
	  (*m).edge[i*(*m).eg_w+MCT1EGNOD1+j] = (FIDX) ihlp;
	}

      /* complete the definition of the edge and bound entry */
      (*m).bound[i*(*m).bd_w+MCT1BDEDGE]=  i;
      (*m).edge[i*(*m).eg_w+MCT1EGCHL1] = -1;
      (*m).edge[i*(*m).eg_w+MCT1EGBND ] =  i;
      (*m).edge[i*(*m).eg_w+MCT1EGPCSU] = -1;
      (*m).edge[i*(*m).eg_w+MCT1EGLVL]  = -1;
	
    }
  (*m).eg_nr   = bd_nr;
  (*m).bd_nr   = bd_nr;
  
  /* check if this is really the end of the boundary block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</boundary>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_boundary_t1: "
	      "spurious end of boundary block (no </boundary> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_triangle_wrapper( FILE *infile, struct mesh *m,
				FIDX hole_nr
/* reads the hole part of a mesh file, then calls the mesh generator
   "triangle" to generate a mesh (mainly the elements, plus changes of
   the boundary), for a descrition of the file format, see the example
   file "mesh_example.f1m",
   for each volume criterion in pccrit a pcvol entry is created for
   each element  


   input:   infile  - a file open for reading
            hole_nr - the number of hole entries in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the vertex,
                      element, edge and bound parts are reallocated
                      and reset here, according to the mesh generated
                      by "triangle",  (in edge only those edges are
                      defined which are part of the boundary!)

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
#define REAL double
#include "triangle/triangle.h"

  FIDX i, j, dim, bd_nr, bd_cnt;
  int  err;

  FIDX found[2], foundboth, k, r;

  struct triangulateio in, out;
  
  FIDX *bndtmp, *edgtmp;

  /* check if the compiled REAL size of "triangle.h" is the same as
     doulbe ! */

  dim= (*m).dim;

  /* generate data structure for "triangle" */
  /* 1. points */
  TRY_MALLOC(in.pointlist, (*m).vx_nr * dim, double,
	     mesh_read_triangle_wrapper);
  in.pointattributelist = NULL;
  in.pointmarkerlist   = NULL;
  in.numberofpointattributes = 0;
  for (i=0; i<(*m).vx_nr; i++)
    for (j=0; j<dim; j++)
      {
	in.pointlist[i*dim+j]=(*m).vertex[i*MCT1VXLN+MCT1VXSTRT+j];
      }
  in.numberofpoints    = (*m).vx_nr;
  /* 2. triangle stuff */
  in.trianglelist          = NULL;
  in.triangleattributelist = NULL;
  in.trianglearealist      = NULL;
  in.neighborlist          = NULL;
  in.numberoftriangles     = 0;
  in.numberofcorners       = 3;
  in.numberoftriangleattributes = 0;
  
  /* 3. holes, allocate space for the holes, then read them */
  TRY_MALLOC(in.holelist, hole_nr * dim, double,
	     mesh_read_triangle_wrapper);
  err=mesh_read_holes( infile, dim, hole_nr, in.holelist);
  FUNCTION_FAILURE_HANDLE( err, mesh_read_holes, 
			   mesh_read_triangle_wrapper);
  in.numberofholes = hole_nr;

  /* 4. regions */
  in.regionlist      = NULL;
  in.numberofregions = 0;

  /* 5. edges */
  in.edgelist        = NULL;
  in.edgemarkerlist  = NULL;
  in.normlist        = NULL;
  in.numberofedges   = 0;

  /* we saved the segment list to last, as it is the most work,
     6. segments,

     We need the segments which are defined by the boundary entries of
     the mesh file, and their marker. We define one marker for each
     original boundary edge (bd+2), so all (eventual) children of that
     get the type, boundary function and shape marker copied from
     their parent, the orientation is ignored, as it is regenerated by
     triangle anyway
  */
  TRY_MALLOC( in.segmentlist, (*m).bd_nr*2, int, 
	      mesh_read_triangle_wrapper);
  TRY_MALLOC( in.segmentmarkerlist, (*m).bd_nr, int, 
	      mesh_read_triangle_wrapper);
  /* copy the boundary edges */
  bd_cnt=0;
  for (i=0; i<(*m).bd_nr; i++)
    {
      FIDX eg;
      /* copy the edge */
      eg=(*m).bound[i*(*m).bd_w+MCT1BDEDGE];

      /* only add edge if it is of the finest level */
      if ((*m).edge[eg*(*m).eg_w+MCT1EGCHL1]==-1)
	{
	  in.segmentlist[bd_cnt*2+0]=(*m).edge[eg*(*m).eg_w+MCT1EGNOD1+0];
	  in.segmentlist[bd_cnt*2+1]=(*m).edge[eg*(*m).eg_w+MCT1EGNOD1+1];
	 
	  /* define the marker */
	  in.segmentmarkerlist[bd_cnt]=i+2;
	  
	  bd_cnt++;
	}
    }
  in.numberofsegments=bd_cnt;

  /* init the out pointers */
  out.pointlist             = NULL;
  out.pointattributelist    = NULL;
  out.pointmarkerlist       = NULL;
  out.trianglelist          = NULL;
  out.triangleattributelist = NULL;
  out.trianglearealist      = NULL;
  out.neighborlist          = NULL;
  out.segmentlist           = NULL;
  out.segmentmarkerlist     = NULL;
  out.holelist              = NULL;
  out.regionlist            = NULL;
  out.edgelist              = NULL;
  out.edgemarkerlist        = NULL;
  out.normlist              = NULL;

  /* everything is ready for triangle, call it */
  triangulate( "-pqza.125", &in, &out, NULL);

  /* triangle should have defined a valid mesh, we need to copy it in
     our format */

  /* 1. vertices */
  TRY_REALLOC( (*m).vertex, out.numberofpoints * MCT1VXLN, double,
	       mesh_read_triangle_wrapper);
  for (i=0; i<out.numberofpoints; i++)
    for (j=0; j<dim; j++)
      {
	(*m).vertex[i*(*m).vx_w+MCT1VXSTRT+j]=out.pointlist[i*dim+j];
      }
  (*m).vx_nr=out.numberofpoints;
  (*m).vx_max=out.numberofpoints;

  /* 2. elements */
  if (out.numberofcorners!=3)
    {
      fprintf(stderr, "mesh_read_triangle_wrapper: something went " 
	      "wrong in triangle, numberofcorners=%d (!=3)\n",
	      out.numberofcorners);
      return FAIL;
    }
  TRY_REALLOC( (*m).elem, out.numberoftriangles * (*m).el_w, FIDX,
	       mesh_read_triangle_wrapper);
  TRY_REALLOC( (*m).face, out.numberoftriangles * (*m).fc_w, FIDX,
	       mesh_read_triangle_wrapper);
  (*m).el_max = out.numberoftriangles;
  (*m).fc_max = out.numberoftriangles;
  for (i=0; i<out.numberoftriangles; i++)
    {
      for (j=0; j<3; j++)
	{
	  if ( (out.trianglelist[i*3+j]<0) ||
	       (out.trianglelist[i*3+j]>=(*m).vx_nr) )
	    {
	      fprintf(stderr," mesh_read_triangle_wrapper: "
		      "triangle returned invalid vertex!\n"
		      "triangle[%"dFIDX" *3 + %"dFIDX"]=%d, vx_nr=%"dFIDX"\n",
		       i,  j, 
		      out.trianglelist[i*3+j],  ((*m).vx_nr));
	      return FAIL;
	    }
	  (*m).elem[i*(*m).el_w+MCT1ELNOD1+j]=out.trianglelist[i*3+j];    
	}
      /* face rhs and PC-vol default to zero/nothing */
      (*m).face[i*(*m).fc_w+MCT1FCRHSF ]=-1;
      (*m).face[i*(*m).fc_w+MCT1FCPCVL ]=-1;
    }
  (*m).el_nr  = out.numberoftriangles;
  (*m).fc_nr  = out.numberoftriangles;

  /* 3. boundary info + edges */
  /* save a copy of the original boundary and edge as the properties
     of the boundary edges (BC-type, BC-function, shape defining,
     pcsurf) are stored there, */
  bndtmp = (*m).bound;
  edgtmp = (*m).edge;
  (*m).bound=NULL;
  (*m).edge=NULL;

  /* (re)allocate memory for the new version */
  bd_nr = out.numberofsegments;
  TRY_MALLOC((*m).edge, bd_nr * (*m).eg_w , FIDX,
	     mesh_read_triangle_wrapper);
  (*m).eg_max = bd_nr;
  TRY_MALLOC((*m).bound, bd_nr * (*m).bd_w , FIDX,
	     mesh_read_triangle_wrapper);
  (*m).bd_max = bd_nr;

  /* copy all the boundary one by one */
  for (i=0; i<bd_nr; i++)
    {
      FIDX oldbc;

      /* the segmentmarker (-2) defines the bc entry from the file */
      oldbc=out.segmentmarkerlist[i]-2;

      /* copy the properties of the old boundary:
	 type, function, shape marker */
      (*m).bound[i*(*m).bd_w+MCT1BDTYPE]=
	bndtmp[oldbc*(*m).bd_w+MCT1BDTYPE];
      (*m).bound[i*(*m).bd_w+MCT1BDFNCT]=
	bndtmp[oldbc*(*m).bd_w+MCT1BDFNCT];
      (*m).bound[i*(*m).bd_w+MCT1BDSSEG]=
	bndtmp[oldbc*(*m).bd_w+MCT1BDSSEG];

      /* copy the edge from triangle's output */
      for (j=0; j<2; j++)
	{
	  if ( (out.segmentlist[i*2+j]<0) ||
	       (out.segmentlist[i*2+j]>=(*m).vx_nr) )
	    {
	      fprintf(stderr," mesh_read_triangle_wrapper: "
		      "triangle returned invalid vertex!\n"
		      "segment[%"dFIDX" *2 + %"dFIDX"]=%d, vx_nr=%"dFIDX"\n",
		       i,  j, 
		      out.segmentlist[i*2+j],  ((*m).vx_nr));
	      return FAIL;
	    }
	  (*m).edge[i*(*m).eg_w+MCT1EGNOD1+j] = 
	    out.segmentlist[i*2+j];
	}

      /* orientation is defined by the orientation of the edge in its
	 element (triangle defines the nodes in counterclockwise
	 order) */
      /* find the edge in an triangle */
      k=0;
      foundboth=0;
      while ( (k<(*m).el_nr)&&(foundboth==0) )
	{
	  for(j=0; j<2; j++)
	    {
	      FIDX node;
	      node= out.segmentlist[i*2+j];
	      found[j]=-1;
	      for (r=0; r<3; r++)
		{
		  if ( (*m).elem[k*(*m).el_w+MCT1ELNOD1+r]==node )
		    {
		      found[j]=r;
		    }
		}
	    }
	  if ((found[0]!=-1)&&(found[1]!=-1))
	    {
	      int bma, n;
	      foundboth=1;
	      /* found it, now work out what orientation it defines */
	      bma=found[1]-found[0];
	      n=1;
	      if (bma<0)
		{
		  n=-n;
		  bma=-bma;
		}
	      /* now bma>0, test if >=2 */
	      if (bma>=2) n=-n;
	      (*m).bound[i*(*m).bd_w+MCT1BDORIE]=n;
	    }
	  k++;
	}
      if (foundboth !=1)
	{
	  fprintf(stderr,"mesh_read_triangle_wrapper: "
		  "triangel output segment not found in elements!\n"
		  "segment[%"dFIDX"]= %d %d \n",  i,
		  out.segmentlist[i*2+0], out.segmentlist[i*2+1]);
	  return FAIL;
	}

      /* complete the definition of the edge and bound entry */
      (*m).bound[i*(*m).bd_w+MCT1BDEDGE]=  i;
      (*m).edge[i*(*m).eg_w+MCT1EGCHL1] = -1;
      (*m).edge[i*(*m).eg_w+MCT1EGBND ] =  i;
      (*m).edge[i*(*m).eg_w+MCT1EGLVL]  = -1;
	
      /* copy pcsurf info */
      (*m).edge[i*(*m).eg_w+MCT1EGPCSU] =
	edgtmp[oldbc*(*m).eg_w+MCT1EGPCSU];
      /* if there was pcsurf info: */
      if ((*m).edge[i*(*m).eg_w+MCT1EGPCSU]!=-1)
	{
	  FIDX oldpcs;
	  oldpcs=(*m).edge[i*(*m).eg_w+MCT1EGPCSU];
	  /* check if the edge has changed name */
	  if(i!= oldbc)
	    {
	      /* if so, we need to create a new pcsurf entry */
	      FIDX newpcs;
	      newpcs=(*m).ps_nr;

	      if( (*m).ps_max< (*m).ps_nr+1 )
		{
		  err=mesh_more_pcsurf( m, 1);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_pcsurf, 
					   mesh_read_triangle_wrapper );
		}

	      /* copy the data of the old to the new entry */
	      (*m).pcsurf[newpcs*(*m).ps_w+MCXXPSCRIT]= 
		(*m).pcsurf[oldpcs*(*m).ps_w+MCXXPSCRIT];

	      /* the adjacent vols are determined later, together with
		 the connectivity */
	      (*m).pcsurf[newpcs*(*m).ps_w+MCXXPSVOLE+0]= -1;
	      (*m).pcsurf[newpcs*(*m).ps_w+MCXXPSVOLE+1]= -1;

	      /* point it to the new edge and vice versa */
	      (*m).pcsurf[newpcs*(*m).ps_w+MCXXPSSURF]=i;
	      (*m).edge[i*(*m).eg_w+MCT1EGPCSU]=newpcs;
	      
	      (*m).ps_nr++;
	      oldpcs=newpcs;
	    }
	  /* orientation is that of the boundary entry */
	  (*m).pcsurf[oldpcs*(*m).ps_w+MCXXPSORIE]=
	    (*m).bound[(*m).edge[i*(*m).eg_w+MCT1EGBND]
		       *(*m).bd_w+MCT1BDORIE];
	}
    }
  (*m).eg_nr   = bd_nr;
  (*m).bd_nr   = bd_nr;

  /* 4. for each volume type pccrit create an pcvol entry for each
     element */
  {
    FIDX cr, pccrit_vol, *is_vol; 

    TRY_MALLOC(is_vol, (*m).pc_nr, FIDX, mesh_read_triangle_wrapper);

    /* count the volume type pccrit entries */
    pccrit_vol=0;
    for (cr=0; cr<(*m).pc_nr; cr++)
      {
	FIDX type=(FIDX) (*m).pccrit[cr*(*m).pc_w+MCXXPCTYPE];

	/* set marker is_vol for this criterion */
	switch (type)
	  {
	    /* case 0: /* squared deviation from given function */
	    /* is_vol[cr]=1;
	       break; (removed from navsto code) */ 
	  case 1: /* surface force (stokes+navsto) */
	    is_vol[cr]=0;
	    break;
	  case 2: /* energy dissipation */
	    is_vol[cr]=1;
	    break;
	  case 3: /* mesh volume */
	    is_vol[cr]=1;
	    break;
	  case 4: /* mean value */
	    is_vol[cr]=1;
	    break;
	  case 5: /* square deviation from mean value */
	    is_vol[cr]=1;
	    break;
	  case 11: /* area */
	    is_vol[cr]=0;
	    break;
	  case 12: /* potential energy */
	    is_vol[cr]=0;
	    break;
	  case 13: /* displacement in a given point */
	    is_vol[cr]=0;
	    break;
	  default:
	    fprintf(stderr, "mesh_read_triangle_wrapper: "
		    "unknown pccrit type: type=%"dFIDX"\n",  type);
	    return FAIL;
	  }
	
	/* count if it is */
	pccrit_vol+=is_vol[cr];
      }

    if (pccrit_vol>1)
      {
	fprintf(stderr,"mesh_read_triangle_wrapper: "
		" >1 volume criteria in the mesh,\n"
		"that should cause problems as only one space in "
		"MCT2FCPCVL\n");
	return FAIL;
      }

    /* now allocate memory for pccrit_vol*fc_nr more pcvol entries */
    if (pccrit_vol>0)
      {
	FIDX new_pv_max;
	new_pv_max = ((*m).pv_nr+(*m).fc_nr*pccrit_vol);
	TRY_REALLOC( (*m).pcvol,
		     (*m).pv_w*new_pv_max,
		     FIDX, mesh_read_triangle_wrapper);
	(*m).pv_max=new_pv_max;

	/* generate an entry for each element and each volume criterion */
	for (cr=0; cr<(*m).pc_nr; cr++)
	  {
	    if (is_vol[cr]==1)
	      {
		for (i=0; i<(*m).fc_nr; i++)
		  {
		    (*m).pcvol[(*m).pv_nr*(*m).pv_w+MCXXPVCRIT] = cr;
		    (*m).pcvol[(*m).pv_nr*(*m).pv_w+MCXXPVVOLM] =  i;
		    (*m).face[i*(*m).fc_w+MCT1FCPCVL] = (*m).pv_nr;
		    (*m).pv_nr++;
		  }
	      }
	  }
      }
    
    /* free temporary data */
    free(is_vol);
  }

  
  /* free the copy of the boundary field */
  free(bndtmp);
  free(edgtmp);

  /* free triangles data */
  free(in.pointlist);
  free(in.pointattributelist);
  free(in.pointmarkerlist);
  free(in.trianglelist);
  free(in.triangleattributelist);
  free(in.trianglearealist);
  free(in.neighborlist);
  free(in.segmentlist);
  free(in.segmentmarkerlist);
  free(in.holelist);
  free(in.regionlist);
  free(in.edgelist);
  free(in.edgemarkerlist);
  free(in.normlist);

  free(out.pointlist);
  free(out.pointattributelist);
  free(out.pointmarkerlist);
  free(out.trianglelist);
  free(out.triangleattributelist);
  free(out.trianglearealist);
  free(out.neighborlist);
  free(out.segmentlist);
  free(out.segmentmarkerlist);
  free(out.edgelist);
  free(out.edgemarkerlist);
  /* these are said to may be ignored 
     free(out.holelist);
     free(out.regionlist);
     free(out.normlist); */
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_holes( FILE *in, FIDX dim, FIDX hole_nr, 
		     double *holes
/* reads the holes part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            dim     - dimension of the hole entries
            hole_nr - number of holes, as specified in the header

   output:  holes   - vector of the points inside the holes,
                      holes[i*dim+j] gives the j-th component of the
                      i-th hole point

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX ihlp;
  FIDX i, j;
  double dhlp;
  char buff[1024];

  /* find the begin of the holes block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<holes>",buff)!=0));
  if (strcmp("<holes>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_holes: file has no <holes>\n");
      return FAIL;
    }

  /* read all the points one by one */
  for (i=0; i<hole_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr, "mesh_read_holes: order of entries wrong!\n");
	  return FAIL;
	}
      for (j=0; j<dim; j++)
	{
	  if (fscanf(in, "%lg", &dhlp) !=1)
	    {
	      fprintf(stderr, "mesh_read_holes: "
		      "error reading entry %"dFIDX"!\n",  i);
	      return FAIL;
	    }
	  holes[i*dim+j] = dhlp;
	}
    }
  
  /* check if this is really the end of the holes block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</holes>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_holes: spurious end of holes block "
	      "(no </holes> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_pcsurf( FILE *in, struct mesh *m, FIDX ps_nr
/* reads the performance criteria surfaces of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m"

   input:   in      - a file open for reading
            ps_nr   - the number of pcsurf entries in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the pcsurf part is
                      allocated and set here, the edge and pccrit
                      parts have to be set properly, as they will be
                      referenced 

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  FIDX ihlp, ihlp2;
  FIDX i, j, edge, found;
  char buff[1024];

  if ((*m).dim!=2)
    {
      fprintf(stderr, "mesh_read_pcsurf: only for dim=2 yet!\n");
      return FAIL;
    }

  TRY_MALLOC((*m).pcsurf, ps_nr * (*m).ps_w , FIDX,
	     mesh_read_pcsurf);
  (*m).ps_max = ps_nr;

  /* find the begin of the pcsurf block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<pcsurf>",buff)!=0));
  if (strcmp("<pcsurf>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_pcsurf: file has no <pcsurf>\n");
      return FAIL;
    }

  /* read all the pcsurf one by one */
  for (i=0; i<ps_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||( (FIDX) ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_pcsurf: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_pcsurf: "
		  "read error pcsurf %"dFIDX" \n",  i);
	  return FAIL;
	}
      else if ((ihlp<0)||((FIDX) ihlp>=(*m).pc_nr))
	{
	  fprintf(stderr, "mesh_read_pcsurf: "
		  "pcsurf %"dFIDX", crit_id=%"dFIDX" out of range!\n",
		   i, ihlp);
	  return FAIL;
	}
      (*m).pcsurf[i*(*m).ps_w+MCXXPSCRIT]= (FIDX) ihlp;

      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((ihlp!=1)&&(ihlp!=-1)))
	{
	  fprintf(stderr, "mesh_read_pcsurf: "
		  "pcsurf %"dFIDX" orientation nonsense!\n",  i);
	  return FAIL;
	}
      (*m).pcsurf[i*(*m).ps_w+MCXXPSORIE]= (FIDX) ihlp;

      if (fscanf(in, "%"dFIDX" %"dFIDX"", &ihlp, &ihlp2) !=2)
	{
	  fprintf(stderr, "mesh_read_pcsurf: "
		  "read error pcsurf %"dFIDX" \n",  i);
	  return FAIL;
	}

      /* search a edge consisting of these 2 nodes */
      found=-1;
      j=0;
      while ((found==-1)&&(j<(*m).eg_nr))
	{
	  FIDX nod1, nod2;
	  nod1= (*m).edge[j*(*m).eg_w+MCT1EGNOD1  ];
	  if (( nod1 != (FIDX) ihlp) && (nod1 != (FIDX) ihlp2))
	    {
	      /* edge can not be the same, try next */
	      j++;
	    }
	  else
	    {
	      nod2= (*m).edge[j*(*m).eg_w+MCT1EGNOD1+1];
	      if (( nod2 != (FIDX) ihlp) && (nod2 != (FIDX) ihlp2))
		{
		  /* edge can not be the same, try next */
		  j++;
		}
	      else if (( nod1 != (FIDX) ihlp) && (nod2 != (FIDX) ihlp2))
		{
		  /* edge found and has same orientation */
		  found=j;
		}
	      else if (( nod1 != (FIDX) ihlp2) && (nod2 != (FIDX) ihlp))
		{
		  /* edge found but has oposite orientation */
		  found=j;
		  (*m).pcsurf[i*(*m).ps_w+MCXXPSORIE]*=-1;
		}
	    }
	} /* end search edge for this pcsurf */

      if (found==-1)
	{
	  fprintf(stderr,
		  "mesh_read_pcsurf: no edge found with specified "
		  "nodes, entry %"dFIDX" \n",  i);
	  return FAIL;
	}

      edge = found;

      /* the vols connected to the pcsurf are determined by the
	 connectivity_init */
      /* complete the definition of the entry */
      (*m).pcsurf[i*(*m).ps_w+MCXXPSSURF]=   edge;
      (*m).edge[edge*(*m).eg_w+MCT1EGPCSU]= i;

    }
  (*m).ps_nr  = ps_nr;
  
  /* check if this is really the end of the pcsurf block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</pcsurf>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_pcsurf: spurious end of pcsurf block "
	      "(no </pcsurf> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}
    


/*FUNCTION*/
int mesh_read_pcvol( FILE *in, struct mesh *m, FIDX pv_nr
/* reads the performance criteria volumes of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m",
   if the triangle mesh generator is used no pcvol entries may be
   defined in the file, but for each volume criterion in pccrit a
   pcvol entry is created for each element 

   input:   in      - a file open for reading
            pv_nr   - the number of pcvol entries in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the pcvol part is
                      allocated and set here, the pccrit and face
                      parts have to be set properly, as they will be
                      referenced 

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  FIDX ihlp;
  FIDX i;
  char buff[1024];

  if ((*m).dim!=2)
    {
      fprintf(stderr, "mesh_read_pcvol: only for dim=2 yet!\n");
      return FAIL;
    }

  TRY_MALLOC((*m).pcvol, pv_nr * (*m).pv_w , FIDX,
	     mesh_read_pcvol);
  (*m).pv_max = pv_nr;

  /* find the begin of the pcvol block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<pcvol>",buff)!=0));
  if (strcmp("<pcvol>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_pcvol: file has no <pcvol>\n");
      return FAIL;
    }

  /* read all the pcvol one by one */
  for (i=0; i<pv_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_pcvol: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_pcvol: read error pcvol %"dFIDX" \n", i);
	  return FAIL;
	}
      else if ((ihlp<0)||((FIDX) ihlp>=(*m).pc_nr))
	{
	  fprintf(stderr,
		  "mesh_read_pcvol: pcvol %"dFIDX", crit_id=%"dFIDX" out of "
		  "range!\n", i, ihlp);
	  return FAIL;
	}
      (*m).pcvol[i*(*m).pv_w+MCXXPVCRIT]= (FIDX) ihlp;

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr,
		  "mesh_read_pcvol: read error pcvol %"dFIDX" \n", i);
	  return FAIL;
	}
      else if ((ihlp<0)||((FIDX) ihlp>=(*m).fc_nr))
	{
	  fprintf(stderr,
		  "mesh_read_pcvol: element out of range, "
		  "entry %"dFIDX"\n", i);
	  return FAIL;
	}

      /* complete the definition of the entry */
      (*m).pcvol[i*(*m).pv_w+MCXXPVVOLM]  = (FIDX) ihlp;
      (*m).face[ihlp*(*m).fc_w+MCT1FCPCVL] = i;
    }
  (*m).pv_nr  = pv_nr;
  
  /* check if this is really the end of the pcvol block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</pcvol>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_pcvol: spurious end of pcvol block "
	      "(no </pcvol> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}
    


/*FUNCTION*/
int mesh_read_pccrit( FILE *in, struct mesh *m, FIDX pc_nr
/* reads the performance function part of a mesh file, for a
   descrition of the file format, see the example file
   "mesh_example.f1m"

   input:   in      - a file open for reading
            pc_nr   - the number of performance functions in the file,
                      as given by the header
   
   output:  m       - (given by reference) a mesh, the pccrit part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX ihlp;
  FIDX i, k, type, fnr;
  double dhlp;
  char buff[1024];


  if ((*m).dim!=2)
    {
      fprintf(stderr, "mesh_read_pccrit: only for dim=2 yet!\n");
      return FAIL;
    }

  TRY_MALLOC((*m).pccrit, pc_nr * (*m).pc_w , double,
	     mesh_read_pccrit);
  (*m).pc_max = pc_nr;


  /* find the begin of the pccrit block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<pccrit>",buff)!=0));
  if (strcmp("<pccrit>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_pccrit: file has no <pccrit>\n");
      return FAIL;
    }

  /* read all the pccrit entries one by one */
  for (i=0; i<pc_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr, "mesh_read_pccrit: order of entries wrong! "
		  "(%"dFIDX" != %"dFIDX")\n",(FIDX) ihlp, i);
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_pccrit: error reading type!"
		  "(entry %"dFIDX")\n",  i);
	  return FAIL;
	}
      type = ihlp;

      /* decode the type */
      /* type of components */
      switch (type)
	{
	  /*case 0:  /* squared deviation from given function, 
	    data= component_nr, function
	    component_nr - the component for which the square
	    deviation is to be considered
	    function     - the reference function, pointer
	    into the function block of the mesh
	    (removed from navsto code)
	    fnr=2;
	    break; */
	case 1:  /* surface force (combinations of drag and lift),
		    handled by stokes_perfcrit_t21 in file stokes_aux.c,
		    data= weight_0, ..., weight_dim
		    weight_i - weight for the i-th component of the
		               force vector, e.g. if the weight vector
		               is orthogonal to the mainstream
		               velocity this is +-lift, if it is
		               parallel it is +-drag
		 */
	  fnr=(*m).dim;
	  break;
 	case 2:  /* energy dissipation,
		    handled by stokes_perfcrit_t21 in file stokes_aux.c,
		    no data required
		 */
	  fnr=0;
	  break;
	case 3:  /* volume of the mesh,
		    no data required */
	  fnr=0;
	  break;
	case 4:  /* mean value, 
		    data= component_nr, coef
		    component_nr - the component for which the mean
		                   value is to be considered
		    coef         - a coefficient by which the result is
                                   multiplied
		 */
	  fnr=2;
	  break;
	case 5:  /* squared deviation from mean value, 
		    data= component_nr, coef
		    component_nr - the component for which the square
		                   deviation is to be considered
		    coef         - a coefficient by which the result is
                                   multiplied
		 */
	  fnr=2;
	  break;
	case 6:  /* squared deviation from reference solution, 
		    data= component_nr1, component_nr2
		    component_nr*- the components for which the square
		                   deviation is to be considered,
		                   i.e. components 
				   component_nr1 to component_nr2
		 */
	  fnr=2;
	  break;
	case 11:  /* volume (or area) of Omega
		     data= coef
		     coef         - a coefficient by which the result is
                                    multiplied */
	  fnr=1;
	  break;
	case 12:  /* potential energy over Omega 
		     data= coef
		     coef         - a coefficient by which the result is
                                    multiplied */
	  fnr=1;
	  break;
	case 13:  /* square of the norm of the solution in a given node
		     data= node, coef
		     node         - node in which |u|^2 is evaluated
		     coef         - a coefficient by which the result is
                                    multiplied */
	  fnr=2;
	  break;
	default: 
	  fprintf(stderr, "mesh_read_pccrit: "
		  "unknown pccrit type!(entry %"dFIDX", type %"dFIDX")\n",  i, type);
	  return FAIL;
	}
      (*m).pccrit[i*(*m).pc_w+MCXXPCTYPE] = type;

      for (k=0; k<fnr; k++)
	{
	  if (fscanf(in, "%lg", &dhlp) !=1)
	    {
	      fprintf(stderr, "mesh_read_pccrit: "
		      "error reading entry %"dFIDX"!\n",  i);
	      return FAIL;
	    }
	  (*m).pccrit[i*(*m).pc_w+MCXXPCDAT1+k] = dhlp;
	}
      /* set the remaining slots to -1 to know which slots are used and which not */
      for (k=fnr+MCXXPCDAT1; k<(*m).pc_w; k++)
	{
	  (*m).pccrit[i*(*m).pc_w+k] = -1;
	}
	
    }
  (*m).pc_nr  = pc_nr;
  
  /* check if this is really the end of the pccrit block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</pccrit>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_pccrit: spurious end of pccrit block "
	      "(no </pccrit> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}


/*FUNCTION*/
int mesh_read_function( FILE *in, struct mesh *m, FIDX fu_nr
/* reads the function part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            fu_nr   - the number of functions in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the function part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX ihlp;
  FIDX i, j, k, type, fdim, fnr;
  double dhlp;
  char buff[1024];

  TRY_MALLOC((*m).func, fu_nr * (*m).fu_w , double,
	     mesh_read_function);
  (*m).fu_max = fu_nr;

  if (((*m).dim!=2)&&((*m).dim!=3))
    {
      fprintf(stderr, "mesh_read_function: only for dim=2 and dim=3 so far\n");
      return FAIL;
    }


  /* find the begin of the function block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<function>",buff)!=0));
  if (strcmp("<function>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_function: file has no <function>\n");
      return FAIL;
    }

  /* read all the function entries one by one */
  for (i=0; i<fu_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr, "mesh_read_function: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_function: error reading type!"
		  "(entry %"dFIDX")\n",  i);
	  return FAIL;
	}
      type = (FIDX) ihlp;

      /* decode the type */
      /* scalar / vector */
      if (type>=100)
	{ fdim=(*m).dim; }
      else
	{ fdim=1; }
      /* type of components */
      switch (type%100)
	{
	case 0:
	  fnr=1;
	  break;
	case 1:
	  fnr=(*m).dim+1;
	  break;
	case 2: 
	  fnr=6;
	  break;
	case 8: 
	  fnr=5;
	  break;
	case 9: 
	  fnr=4;
	  break;
	case 10: 
	  fnr=5;
	  break;
	case 31:
	  fnr=2;
	  break;
	default: 
	  fprintf(stderr, "mesh_read_function: "
		  "unknown function type! (entry %"dFIDX", type %"dFIDX", type%%100 %"dFIDX")\n",
		   i,  type, (type%100) );
	  return FAIL;
	}
      (*m).func[i*(*m).fu_w+MC2XFUTYPE] = type;

      for (j=0; j<fdim; j++)
	{
	  for (k=0; k<fnr; k++)
	    {
	      if (fscanf(in, "%lg", &dhlp) !=1)
		{
		  fprintf(stderr, "mesh_read_function: "
			  "error reading entry %"dFIDX"!\n",  i);
		  return FAIL;
		}
	      if ((*m).dim==2)
		(*m).func[i*(*m).fu_w+MC2XFUDAT1+j*MC2XFUDATX+k] = dhlp;
	      else
		(*m).func[i*(*m).fu_w+MC3XFUDAT1+j*MC3XFUDATX+k] = dhlp;
	    }
	}
    }
  (*m).fu_nr  = fu_nr;
  
  /* check if this is really the end of the function block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</function>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_function: spurious end of function block "
	      "(no </function> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}



/*FUNCTION*/
int mesh_read_parameter( FILE *in, struct mesh *m, FIDX pa_nr
/* reads the parameter part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            pa_nr   - the number of parameters in the file, as given by
                      the header
   
   output:  m       - (given by reference) a mesh, the parameter part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX i;
  double dhlp;
  char buff[1024];

  TRY_MALLOC((*m).para, pa_nr*(*m).pa_w , double,
	     mesh_read_parameter);
  (*m).pa_max = pa_nr;

  /* if (pa_nr>(*m).pa_w)
     {
     fprintf(stderr, "mesh_read_parameter: "
     "number of parameter to large!\n");
     return FAIL;
     }
  */


  /* find the begin of the parameter block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<parameter>",buff)!=0));
  if (strcmp("<parameter>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_parameter: file has no <parameter>\n");
      return FAIL;
    }

  /* read all the parameter data one by one */
  for (i=0; i<pa_nr; i++)
    {
      if (fscanf(in, "%lg", &dhlp) !=1)
	{
	  fprintf(stderr, "mesh_read_parameter: "
		  "error reading datum %"dFIDX"!\n",  i);
	  return FAIL;
	}
      (*m).para[i*(*m).pa_w +0] = dhlp;
    }
  (*m).pa_nr  = pa_nr;
  
  /* check if this is really the end of the parameter block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</parameter>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_parameter: spurious end of parameter block "
	      "(no </parameter> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}





/*FUNCTION*/
int mesh_read_sseg( FILE *in, struct mesh *m, FIDX sg_nr, FIDX sp_nr
/* reads the elements part of a mesh file, for a descrition of the
   file format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            sg_nr   - the number of shape segments in the file, as
                      given by the header
            sp_nr   - the number of shape parameters in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the sseg
                      part is allocated and set here

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
			   ){
  FIDX ihlp;
  FIDX i, j, type, type_par;
  char buff[1024];

  TRY_MALLOC((*m).sseg, sg_nr * (*m).sg_w , FIDX,
	     mesh_read_sseg);
  (*m).sg_max = sg_nr;

  /* find the begin of the shape segments block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<shape_seg>",buff)!=0));
  if (strcmp("<shape_seg>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_sseg: file has no <shape_seg>\n");
      return FAIL;
    }

  /* read all the entries one by one */
  for (i=0; i<sg_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr,
		  "mesh_read_sseg: order of entries wrong!\n");
	  return FAIL;
	}
   
      if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	{
	  fprintf(stderr, "mesh_read_sseg: "
		  "error reading type shape_seg %"dFIDX"\n",  i);
	  return FAIL;
	}
      type = ihlp;

      switch (type)
	{
	case 1: /* line */
	  type_par = 4; /* 4 parameters, x0,y0, x1,y1 */
	  break;
	case 2: /* b-spline */
	  type_par = 8; /* 8 parameters, x0,y0, x1,y1, sx0,sy0, sx1,sy1 */
	  break;
	default: /* error */
	  fprintf(stderr, "mesh_read_sseg: "
		  "unknown type, shape_seg %"dFIDX"\n",  i);
	  return FAIL;
	}
      (*m).sseg[i*(*m).sg_w+MC2XSGTYPE]=type;


      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||
	  ((ihlp<0)||((FIDX) ihlp>=(*m).vx_nr)))
	{
	  fprintf(stderr, "mesh_read_sseg: "
		  "starting vertex out of range, shape_seg %"dFIDX"\n",  i);
	  return FAIL;
	}
      (*m).sseg[i*(*m).sg_w+MC2XSGNOD1]= (FIDX) ihlp;



      if (type_par>MC2XSGPMAX)
	{
	  fprintf(stderr, "mesh_read_sseg: "
		  "type_par>=MC2XGPMAX would trouble!\n");
	  return FAIL;
	}

      /* read the parameters */
      for (j=0; j<type_par; j++)
	{
	  if (fscanf(in, "%"dFIDX"", &ihlp) !=1)
	    {
	      fprintf(stderr,"mesh_read_sseg: "
		      "error reading entry %"dFIDX"\n",  i);
	      return FAIL;
	    }
	  if ((ihlp<0)||((FIDX) ihlp>=sp_nr))
	    {
	      fprintf(stderr,"mesh_read_sseg: "
		      "parameter out of range, shape_seg entry %"dFIDX"\n",
		       i);
	      return FAIL;
	    }
	  (*m).sseg[i*(*m).sg_w+MC2XSGPAR1+j]= (FIDX) ihlp;
	}
    }
  (*m).sg_nr  = sg_nr;

  /* check if this is really the end of the shape_seg block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</shape_seg>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_sseg: spurious end of shape_seg block "
	      "(no </shape_seg> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}
    




/*FUNCTION*/
int mesh_read_spar( FILE *in, struct mesh *m, FIDX sp_nr
/* reads the shape_par part of a mesh file, for a descrition of the file
   format, see the example file "mesh_example.f1m"

   input:   in      - a file open for reading
            sp_nr   - the number of shape parameters in the file, as
                      given by the header
   
   output:  m       - (given by reference) a mesh, the shape_par part is
                      allocated and defined

   retun:   SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX ihlp;
  FIDX i;
  double dhlp;
  char buff[1024];

  TRY_MALLOC((*m).spar, sp_nr*(*m).sp_w , double,
	     mesh_read_spar);
  (*m).sp_max = (*m).sp_nr;

  /* find the begin of the shape_par block */
  while ((fscanf(in, "%1023s", buff)==1)&&(strcmp("<shape_par>",buff)!=0));
  if (strcmp("<shape_par>",buff)!=0)
    {
      fprintf(stderr, "mesh_read_spar: file has no <shape_par>\n");
      return FAIL;
    }

  /* read all the shape_par data one by one */
  for (i=0; i<sp_nr; i++)
    {
      if ((fscanf(in, "%"dFIDX"", &ihlp) !=1)||((FIDX) ihlp!=i))
	{
	  fprintf(stderr, "mesh_read_spar: order of entries wrong!\n");
	  return FAIL;
	}

      if (fscanf(in, "%lg", &dhlp) !=1)
	{
	  fprintf(stderr, "mesh_read_spar: "
		  "error reading shape parameter %"dFIDX"!\n",  i);
	  return FAIL;
	}
      (*m).spar[i] = dhlp;
    }
  (*m).sp_nr  = sp_nr;
  
  /* check if this is really the end of the shape_par block */
  if ((fscanf(in, "%1023s", buff)!=1)||(strcmp("</shape_par>",buff)!=0))
    {
      fprintf(stderr, "mesh_read_spar: spurious end of shape_par block "
	      "(no </shape_par> ?)\n");
      return FAIL;
    }
  
  return SUCCESS;
}






/*FUNCTION*/
int mesh_read_compute_conns_t1( struct mesh *m
/* builds the part of the connectivity data of the internal mesh
   format that is not covered by the mesh files (see the example file
   "mesh_example.f1m")

   In/Out:  m       - (given by reference) a mesh, the edge and face
                      parts are completed

   return:  SUCCESS - success,
            FAIL    - failure, see error message 
*/			 
      		      ){
  FIDX i, j, el, eg, nod1, nod2, found;
  int  err;

  struct ilist *vx2eg; /* an array of list anchors, where each list
			  entry is the id of an edge the vertex of
			  the lists anchor belongs to */
  struct ilist *pntr, *hlp;
                       /* auxiliary list pointers */

  TRY_MALLOC(vx2eg, (*m).vx_nr , struct ilist, 
	     mesh_read_compute_conns_t1);

  /* initialise all the lists as empty */
  for (i=0; i<(*m).vx_nr; i++)
    {
      vx2eg[i].next=NULL;
      vx2eg[i].data=-1;
    }

  /* take all the known edges (from the boundary) into the list */
  for (i=0; i<(*m).eg_nr; i++)
    {
      /* both vertices of the edge */
      for (j=0; j<2; j++)
	{
	  pntr=&vx2eg[(*m).edge[i*(*m).eg_w+MCT1EGNOD1+j]];
	  /* find the end of the list of this vertex */
	  while (pntr->next!=NULL) pntr=pntr->next;
	  /* append an entry to the list */
	  TRY_MALLOC(pntr->next, 1, struct ilist,  
		     mesh_read_compute_conns_t1);
	  pntr       = pntr->next;
	  pntr->data = i;
	  pntr->next = NULL;
	}
    }

  /* now for each element: check if the 3 edges exist, if not create
     them, build a face from these edges */
  for (el=0; el<(*m).el_nr; el++)
    {
      for (j=0; j<3; j++)
	{
	  nod1=(*m).elem[el*(*m).el_w+MCT1ELNOD1+ j     ];
	  nod2=(*m).elem[el*(*m).el_w+MCT1ELNOD1+(j+1)%3];

	  /* check if an edge eg with nod1 and nod2 exists */
	  found=0;
	  pntr=vx2eg[nod1].next;
	  hlp =&vx2eg[nod1];
	  while ((found==0)&&(pntr!=NULL))
	    {
	      eg=pntr->data;
	      if ((nod2==(*m).edge[eg*(*m).eg_w+MCT1EGNOD1  ])||
		  (nod2==(*m).edge[eg*(*m).eg_w+MCT1EGNOD1+1]))
		{
		  found=1;
		}
	      hlp  = pntr;
	      pntr = pntr->next;
	    }
	  
	  /* if not, create one */
	  if (found==0)
	    {
	      /* check if enough space is available */
	      if ((*m).eg_max<(*m).eg_nr+1) 
		{ 
		  err=mesh_more_edges( m, 3);
		  FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,
					   mesh_read_compute_conns_t1);
		}
	      /* create the edge */
	      eg=(*m).eg_nr;
	      (*m).eg_nr++;

	      (*m).edge[eg*(*m).eg_w+MCT1EGNOD1  ]=nod1;
	      (*m).edge[eg*(*m).eg_w+MCT1EGNOD1+1]=nod2;
	      (*m).edge[eg*(*m).eg_w+MCT1EGCHL1  ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCT1EGBND   ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCT1EGPCSU  ]= -1;
	      (*m).edge[eg*(*m).eg_w+MCT1EGLVL   ]= -1;

	      /* register the new edge in vx2eg */
	      /* append an entry to the end of the nod1 list */
	      TRY_MALLOC(hlp->next, 1, struct ilist,  
			 mesh_read_compute_conns_t1);
	      pntr       = hlp->next;
	      pntr->data = eg;
	      pntr->next = NULL;

	      /* append an entry to the end of the nod2 list */
	      pntr=&vx2eg[nod2];
	      /* find the end of the list of this vertex */
	      while (pntr->next!=NULL) pntr=pntr->next;
	      /* append an entry to the list */
	      TRY_MALLOC(pntr->next, 1, struct ilist,  
			 mesh_read_compute_conns_t1);
	      pntr       = pntr->next;
	      pntr->data = eg;
	      pntr->next = NULL;
	      
	    }

	  /* now we have the edge eg which holds both nodes, make it
	     edge of the face corresponding to the element el */
	  (*m).face[el*(*m).fc_w+MCT1FCEDG1+j]=eg;
	}
    }

  /* check the connectivity of the pcsurf entries, which faces they
     belong to */
  for (i=0; i<(*m).ps_nr; i++)
    {
      FIDX edge, found;

      edge= (*m).pcsurf[i*(*m).ps_w+MCXXPSSURF];

      /* search elements containing this edge */
      (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE+0]= -1;
      (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE+1]= -1;
      found=0;
      j=0;
      while ((found<2)&&(j<(*m).fc_nr))
	{
	  FIDX eledge;
	  for (eledge=0; eledge<3; eledge++)
	    {
	      if (edge==(*m).face[j*(*m).fc_w+MCT1FCEDG1+eledge])
		{
		  /* this face contains the edge */
		  (*m).pcsurf[i*(*m).ps_w+MCXXPSVOLE]= j;
		  found++;
		}
	    }
	  j++;
	} /* end search element for this edge */

      if (found==-1)
	{
	  fprintf(stderr,
		  "mesh_read_compute_conns_t1: no face found with "
		  "psurf edge, pcsruf entry %"dFIDX" \n",  i);
	  return FAIL;
	}
    }

  /* clean up */
  /* empty the list */
  for (i=0; i<(*m).vx_nr; i++)
    {
      pntr=vx2eg[i].next;
      while (pntr!=NULL)
	{
	  hlp  = pntr;
	  pntr = pntr->next;
	  free(hlp);
	}
    }
  free(vx2eg);

  return SUCCESS;
}


/********************************************************************/
/*                                                                  */
/*         T2 element stuff                                         */
/*                                                                  */
/********************************************************************/


/*FUNCTION*/
int mesh_refine_uniform_t2( struct mesh *m 
/* refines all elements of the mesh m, all necessarry new parts are
   created thereby, increases mesh.lvl

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX old_el_nr, i;
  int  err;

  old_el_nr=(*m).el_nr;
  (*m).lvl++;

  /* loop over all elements */
  for (i=0; i< old_el_nr; i++)
    {
      err=mesh_refine_element_t2( m , i);
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_element_t2, 
			       mesh_refine_uniform_t2 ); 
    }
  return SUCCESS;
}


/*FUNCTION*/
int mesh_refine_element_t2( struct mesh *m , FIDX el
/* refines the element el of the mesh m, all necessarry new parts are
   created thereby

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, vx_w, fc_w, el_w, eg_w, ps_w, pv_w, hi_w, eh_w, elhi, eho;
  int  err;


  FIDX newedge[3], midnodes[3], oldnod[3], nodedge[3][2], nodineg[3][2];
  /* nodedge[lnode][this nodes edge]=the edge (global number),
     nodineg[lnode][this nodes edge]=the node is 1st/2nd of the edge */

  vx_w=(*m).vx_w;
  fc_w=(*m).fc_w;
  el_w=(*m).el_w;
  eg_w=(*m).eg_w;
  ps_w=(*m).ps_w;
  pv_w=(*m).pv_w;
  hi_w=(*m).hi_w;
  eh_w=(*m).eh_w;

  /* clear nodedge */
  for (i=0; i<3; i++)
    for (j=0; j<2; j++)
      nodedge[i][j]=-1;

  /* get oldnod */
  for (i=0; i<3; i++)
    oldnod[i]=(*m).elem[el*el_w+MCT2ELNOD1+i];

  /* make sure the 3 edges are refined, store the midnodes, get
     nodedge */ 
  for (i=0; i<3; i++)
    {
      FIDX eg, chld;

      eg=(*m).face[el*fc_w+MCT2FCEDG1+i];

      /* make sure this edge is refined */
      if ((*m).edge[eg*eg_w+MCT2EGCHL1]==-1)
	{
	  err=mesh_split_edge_t2( m, eg);
	  FUNCTION_FAILURE_HANDLE( err, mesh_split_edge_t2,
				   mesh_refine_element_t2); 
	}
      
      /* store the midnode */
      chld=(*m).edge[eg*eg_w+MCT2EGCHL1];
      /* second node of child edge is always the midnode of the old
	 edge */ 
      midnodes[i]=(*m).edge[chld*eg_w+MCT2EGNOD1+1]; 

      for (j=0; j<2; j++)
	{
	  FIDX amnodeedge, thenode, intnode;
	  thenode=(*m).edge[eg*eg_w+MCT2EGNOD1+j];
	  intnode=0;
	  /* find the apropriate internal node for this guy */
	  while ((intnode<3)&&(oldnod[intnode]!=thenode))
	    intnode++;
	  /* if not found, cry */
	  if (intnode>=3)
	    {
	      fprintf(stderr, "internal mesh consistency error in "
		      "mesh_refine_element_t2, file %s, line %d\n",
		      __FILE__, __LINE__);
	      return FAIL;
	    }

	  /* find where in nodedge and nodineg this one goes to */
	  if (nodedge[intnode][0]==-1)
	    { amnodeedge=0; }
	  else
	    { amnodeedge=1; }
	  nodedge[intnode][amnodeedge]=eg;
	  nodineg[intnode][amnodeedge]=j;
	}
    }

  /* make sure we have enough space for the 3 new edges in the inner
     of the element, the 3 new vertices and their hierarchy entires
     who come with the new edges, 3 new faces (the old is recycled as
     a 4th), 3 new elements (the old is recycled as a 4th), 3 new
     pcvol if needed (the old is recycled as a 4th), one element
     hierarchy entry */ 
  if ((*m).eg_nr+3>(*m).eg_max)
    {
      err=mesh_more_edges( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_refine_element_t2 );
    }
  if ((*m).vx_nr+3>(*m).vx_max)
    {
      err=mesh_more_vertices( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices, mesh_refine_element_t2 );
    }
  if ((*m).hi_nr+3>(*m).hi_max)
    {
      err=mesh_more_hierarchy( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_hierarchy, mesh_refine_element_t2 );
    }
  if ((*m).fc_nr+3>(*m).fc_max)
    {
      err=mesh_more_faces( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_refine_element_t2 );
    }
  if ((*m).el_nr+3>(*m).el_max)
    {
      err=mesh_more_elems( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elems, mesh_refine_element_t2 );
    }
  if (((*m).face[el*fc_w+MCT2FCPCVL]!=-1)&&((*m).pv_nr+3>(*m).pv_max))
    {
      err=mesh_more_pcvol( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcvol, mesh_refine_element_t2 );
    }
  if ((*m).eh_nr+1>(*m).eh_max)
    {
      err=mesh_more_elhiers( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, mesh_refine_element_t2 );
    }
  if ((*m).st_nr+3>(*m).st_max)
    {
      err=mesh_more_st( m, 3);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_st, mesh_refine_element_t2 );
    }
  /* create the new elhier */
  elhi=(*m).eh_nr;
  (*m).eh_nr++;
  /* level = level of the new child edges (=lvl of edge hierarchy entry */
  (*m).elhier[elhi*eh_w+MCT2EHLVL]= (*m).lvl;

  /* for all old corner nodes */
  for (i=0; i<3; i++)
    {
      FIDX chld0, chld1, midnod0, midnod1, newface, newel, newnode,
	old1, old2, newhier;

      /* check if the first or the second edge points to the successor
	 node */
      if ((nodedge[i][0]==nodedge[(i+1)%3][0])||
	  (nodedge[i][0]==nodedge[(i+1)%3][1]))
	{ /* first edge positive */
	  eho=0; }
      else
	{ /* second edge must be positive */
	  eho=1; }

      /* get the child edges containig this node */
      chld0=(*m).edge[nodedge[i][0]*eg_w+MCT2EGCHL1+nodineg[i][0]];
      chld1=(*m).edge[nodedge[i][1]*eg_w+MCT2EGCHL1+nodineg[i][1]];
  
      /* get the corresponding midnodes */
      midnod0=(*m).edge[chld0*eg_w+MCT2EGNOD1+1]; 
      midnod1=(*m).edge[chld1*eg_w+MCT2EGNOD1+1];
      /* second node of child edge is allways the midnode of the old
	 edge */

      /* create a new edge from the midnodes */
      newedge[i]=(*m).eg_nr;
      (*m).eg_nr++;

      (*m).edge[newedge[i]*eg_w+MCT2EGNOD1  ]=midnod0;
      (*m).edge[newedge[i]*eg_w+MCT2EGNOD1+1]=midnod1;
      (*m).edge[newedge[i]*eg_w+MCT2EGCHL1  ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT2EGBND   ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT2EGPCSU  ]=-1;
      (*m).edge[newedge[i]*eg_w+MCT2EGLVL   ]=(*m).lvl;

      /* create the midnode of the new edge */
      newnode=(*m).vx_nr;
      (*m).vx_nr++;
      (*m).st_nr++;
      
      old1= (*m).edge[newedge[i]*eg_w + MCT2EGNOD1  ];
      old2= (*m).edge[newedge[i]*eg_w + MCT2EGNOD1+1];

      (*m).vertex[newnode*vx_w+MCT2VXSTRT  ]= 0.5*
	( (*m).vertex[old1*vx_w+MCT2VXSTRT  ]
	  + (*m).vertex[old2*vx_w+MCT2VXSTRT  ]);
      (*m).vertex[newnode*vx_w+MCT2VXSTRT+1]= 0.5*
	( (*m).vertex[old1*vx_w+MCT2VXSTRT+1]
	  + (*m).vertex[old2*vx_w+MCT2VXSTRT+1]);
      
      (*m).edge[newedge[i]*eg_w + MCT2EGNODM]=newnode;

      /* the acording hierarchy entry to the midnode */
      newhier=(*m).hi_nr;
      (*m).hi_nr++;

      (*m).hier[newhier*hi_w+ MCT2HICHLD  ]=newnode;
      (*m).hier[newhier*hi_w+ MCT2HIFAT1  ]=old1;
      (*m).hier[newhier*hi_w+ MCT2HIFAT1+1]=old2;
      (*m).hier[newhier*hi_w+ MCT2HILVL   ]=(*m).lvl;

      /* create a new face */
      newface=(*m).fc_nr;
      (*m).fc_nr++;

      (*m).face[newface*fc_w+MCT2FCEDG1  ]=chld0;
      (*m).face[newface*fc_w+MCT2FCEDG1+1]=chld1;
      (*m).face[newface*fc_w+MCT2FCEDG1+2]=newedge[i];

      (*m).face[newface*fc_w+MCT2FCRHSF]=(*m).face[el*fc_w+MCT2FCRHSF];

      /* if necessary create a new pcvol entry */
      if ((*m).face[el*fc_w+MCT2FCPCVL]!=-1)
	{
	  FIDX pcv, new;

	  pcv=(*m).face[el*fc_w+MCT2FCPCVL];
	  new=(*m).pv_nr;
	  
	  (*m).pcvol[new*pv_w+MCXXPVVOLM]=newface;
	  (*m).face[newface*fc_w+MCT2FCPCVL]=new;
	  (*m).pcvol[new*pv_w+MCXXPVCRIT]=(*m).pcvol[pcv*pv_w+MCXXPVCRIT];

	  (*m).pv_nr++;
	}
      else
	{
	  (*m).face[newface*fc_w+MCT2FCPCVL]=-1;
	}

      /* check if the edges of the new face are pcsurf edges, if so,
	 let them know the name of the new face */
      for (j=0; j<3; j++)
	{
	  FIDX eg, pcsu;
	  eg=(*m).face[newface*fc_w+MCT2FCEDG1+j];
	  pcsu=(*m).edge[eg*eg_w+MCT2EGPCSU];
	  if (pcsu!=-1)
	    {
	      if ((*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]==-1)
		{
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]=newface;
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=-1;
		}
	      else
		{
		  (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=newface;
		}
	    }
	}


      /* create a new element */
      newel=(*m).el_nr;
      (*m).el_nr++;

      err=mesh_t2_get_elem_from_face( m, newel);
      FUNCTION_FAILURE_HANDLE( err, mesh_t2_get_elem_from_face,
			       mesh_refine_element_t2 );

      /* define this node part of the new elhier */
      /* fathers */
      (*m).elhier[elhi*eh_w+MCT2EHFAT1+i]=(*m).elem[el*el_w+MCT2ELNOD1+i];
      (*m).elhier[elhi*eh_w+MCT2EHFAT1+3+(i+    eho*2)%3]=midnod0;
      (*m).elhier[elhi*eh_w+MCT2EHFAT1+3+(i+(1-eho)*2)%3]=midnod1;
      /* children */
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+i*3+    eho*2]=
	(*m).edge[chld0*eg_w+MCT2EGNODM];
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+i*3+(1-eho)*2]=
	(*m).edge[chld1*eg_w+MCT2EGNODM];
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+i*3+        1]=newnode;

      /* next node */
    }

  /* the outer 3 elements are correctly created now,
     so lets overwrite the old element with the inner one, the 4-th */

  for (i=0; i<3; i++)
    {
      /* the face */
      (*m).face[el*fc_w+MCT2FCEDG1+i]=newedge[i];
      /* ( rhs-function, pcvol stays the same ) */
      /* the edges of the inner face can not be pcsurf edges */
    }
  /* the element */
  err=mesh_t2_get_elem_from_face( m, el);
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_get_elem_from_face,
			   mesh_refine_element_t2 );

  /* that's it */

  return SUCCESS;
}

/*FUNCTION*/
int mesh_split_edge_t2( struct mesh *m , FIDX eg
/* refines the edge eg of the mesh m, so that the first node of eg is
   the first node of the first child edge, the second node is the
   first node of the second child edge and the new middle node is the
   second node of both child edges,
   all necessarry new parts are created

   Input:  eg       - the edge to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      edges, vertices, boundary elements and hierarchy
                      entries are created, 
		      thereby the edge remains to allow neigbouring elements
		      to stay intact, 
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, dim, vx_w, eg_w, hi_w, bd_w, ps_w;
  int  err;
  FIDX newm1, newm2, nod1, nod2, nodm;
  FIDX neweg1, neweg2;
  FIDX oldbd, newbd1,newbd2;
  FIDX newhi1, newhi2;
  FIDX bd, seg;

  dim=(*m).dim;
  vx_w=(*m).vx_w;
  eg_w=(*m).eg_w;
  hi_w=(*m).hi_w;
  bd_w=(*m).bd_w;
  ps_w=(*m).ps_w;

  if ((*m).edge[eg*eg_w+ MCT2EGCHL1 ]!= -1)
    {
      /* nothing to do, return */
      return SUCCESS;
    }

  bd = (*m).edge[eg*eg_w+ MCT2EGBND];   
  if (bd!=-1) 
    {
      seg = (*m).bound[bd*bd_w+MCT2BDSSEG];
    }
  else
    {
      seg = -1;    
    }
    
  /* first we have to make sure we have space for two new vertices, two
     new edges, eventually new boundary elements, two hierarchy entries
  */
  if ((*m).vx_nr+2>(*m).vx_max)
    {
      err=mesh_more_vertices( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices,  mesh_split_edge_t2 );
    }
  if ((*m).eg_nr+2>(*m).eg_max)
    {
      err=mesh_more_edges( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges,  mesh_split_edge_t2 );
    }
  if (((*m).edge[eg*eg_w+ MCT2EGBND]!= -1)&&((*m).bd_nr+2>(*m).bd_max))
    {
      err=mesh_more_boundary( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_boundary,  mesh_split_edge_t2 );
    }
  if (((*m).edge[eg*eg_w+ MCT2EGPCSU]!= -1)&&((*m).ps_nr+2>(*m).ps_max))
    {
      err=mesh_more_pcsurf( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcsurf,  mesh_split_edge_t2 );
    }
  if ((*m).hi_nr+2>(*m).hi_max)
    {
      err=mesh_more_hierarchy( m, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_hierarchy,  mesh_split_edge_t2 );
    }
    if ((*m).st_nr+2>(*m).st_max)
      {
        err=mesh_more_st( m, 2);
        FUNCTION_FAILURE_HANDLE( err, mesh_more_st,  mesh_split_edge_t2 );
      }
  /* nodes of the edge */
  nod1=(*m).edge[eg*eg_w+ MCT2EGNOD1];
  nod2=(*m).edge[eg*eg_w+ MCT2EGNOD1+1];
  nodm=(*m).edge[eg*eg_w+ MCT2EGNODM];

  /* create the new nodes */
  newm1= (*m).vx_nr;
  (*m).vx_nr++;
  (*m).st_nr++;
  newm2= (*m).vx_nr;
  (*m).vx_nr++;
  (*m).st_nr++;
  /* set position of the new node */
  for (i=0;i<dim; i++)
    {
      (*m).vertex[newm1*vx_w+MCT2VXSTRT+i]=
	0.5*(*m).vertex[nod1*vx_w+MCT2VXSTRT+i]+
	0.5*(*m).vertex[nodm*vx_w+MCT2VXSTRT+i];
      (*m).vertex[newm2*vx_w+MCT2VXSTRT+i]=
	0.5*(*m).vertex[nod2*vx_w+MCT2VXSTRT+i]+
	0.5*(*m).vertex[nodm*vx_w+MCT2VXSTRT+i];
    } 
  if (seg>-1)
    {/* get t-value for the new midnodes  */
      err=mesh_spline_set_t( m, seg, nod1, nodm, newm1, 0, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_get_st,  mesh_split_edge_t2 );
      if ((*m).st[newm1]<0.0) 
	{
	  fprintf(stderr, "Error in mesh_split_edge_t2: mesh_spline_set_t did not yield"
		  " a valid t-value for the spline segment!\n");
	    return FAIL;
	}
      err=mesh_spline_set_t( m, seg, nod2, nodm, newm2, 0, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_get_st,  mesh_split_edge_t2 );
      if ((*m).st[newm2]<0.0) 
	{
	  fprintf(stderr, "Error in mesh_split_edge_t2: mesh_spline_set_t did not yield"
		  " a valid t-value for the spline segment!\n");
	  return FAIL;
	}
    }

    
    
  /* create the new edges */
  neweg1=(*m).eg_nr;
  neweg2=(*m).eg_nr+1;
  (*m).eg_nr+=2;
  /* child edge 1 */
  (*m).edge[neweg1*eg_w+MCT2EGNOD1  ]=nod1;
  (*m).edge[neweg1*eg_w+MCT2EGNOD1+1]=nodm;
  (*m).edge[neweg1*eg_w+MCT2EGNODM  ]=newm1;
  (*m).edge[neweg1*eg_w+MCT2EGCHL1  ]=-1;
  (*m).edge[neweg1*eg_w+MCT2EGBND   ]=-1;
  (*m).edge[neweg1*eg_w+MCT2EGPCSU  ]=-1;
  (*m).edge[neweg1*eg_w+MCT2EGLVL   ]=(*m).lvl;
  /* child edge 2 */
  (*m).edge[neweg2*eg_w+MCT2EGNOD1  ]=nod2;
  (*m).edge[neweg2*eg_w+MCT2EGNOD1+1]=nodm;
  (*m).edge[neweg2*eg_w+MCT2EGNODM  ]=newm2;
  (*m).edge[neweg2*eg_w+MCT2EGCHL1  ]=-1;
  (*m).edge[neweg2*eg_w+MCT2EGBND   ]=-1;
  (*m).edge[neweg2*eg_w+MCT2EGPCSU  ]=-1;
  (*m).edge[neweg2*eg_w+MCT2EGLVL   ]=(*m).lvl;
  /* make them children of eg */
  (*m).edge[eg*eg_w+MCT2EGCHL1  ]=neweg1;
  (*m).edge[eg*eg_w+MCT2EGCHL1+1]=neweg2;
  
  /* if required, create new boundary elements, correct boundary
     information on the children */
  if ((*m).edge[eg*eg_w+ MCT2EGBND]!= -1)
    {
      newbd1=(*m).bd_nr;
      newbd2=(*m).bd_nr+1;
      (*m).bd_nr+=2;
      oldbd=(*m).edge[eg*eg_w+ MCT2EGBND];
	    
      /* newbd1 */
      (*m).bound[newbd1*bd_w+MCT2BDEDGE]=neweg1;
      (*m).bound[newbd1*bd_w+MCT2BDTYPE]=
	(*m).bound[oldbd*bd_w+MCT2BDTYPE]; /* same type as old */
      (*m).bound[newbd1*bd_w+MCT2BDFNCT]=
	(*m).bound[oldbd*bd_w+MCT2BDFNCT]; /* same function data as old
					   */
      (*m).bound[newbd1*bd_w+MCT2BDORIE]=
	(*m).bound[oldbd*bd_w+MCT2BDORIE]; /* first edge has same
					      orientation as the old
					      edge */
      (*m).bound[newbd1*bd_w+MCT2BDSSEG]=
	(*m).bound[oldbd*bd_w+MCT2BDSSEG]; /* same shape variability */
      (*m).edge[neweg1*eg_w+MCT2EGBND   ]=newbd1;
	
      /* newbd2 */
      (*m).bound[newbd2*bd_w+MCT2BDEDGE]=neweg2;
      (*m).bound[newbd2*bd_w+MCT2BDTYPE]=
	(*m).bound[oldbd*bd_w+MCT2BDTYPE]; /* same type as old */
      (*m).bound[newbd2*bd_w+MCT2BDFNCT]=
	(*m).bound[oldbd*bd_w+MCT2BDFNCT]; /* same function data as old
					   */
      (*m).bound[newbd2*bd_w+MCT2BDORIE]=
	-(*m).bound[oldbd*bd_w+MCT2BDORIE];/* second edge has reverse
					      orientation as the old
					      edge */
      (*m).bound[newbd2*bd_w+MCT2BDSSEG]=
	(*m).bound[oldbd*bd_w+MCT2BDSSEG]; /* same shape variability */
      (*m).edge[neweg2*eg_w+MCT2EGBND   ]=newbd2;
    }
    
  /* if required, create new pcsurf elements, correct pcsurf
     information on the children */
  if ((*m).edge[eg*eg_w+ MCT2EGPCSU]!= -1)
    {
      FIDX oldps, newps1, newps2;
      newps1=(*m).ps_nr;
      newps2=(*m).ps_nr+1;
      (*m).ps_nr+=2;
      oldps=(*m).edge[eg*eg_w+ MCT2EGPCSU];
      /* newps1 */
      (*m).pcsurf[newps1*ps_w+MCXXPSSURF]=neweg1;
      (*m).pcsurf[newps1*ps_w+MCXXPSVOLE]=-1;/* faces are set in face
						refinement */
      (*m).pcsurf[newps1*ps_w+MCXXPSCRIT]=
	(*m).pcsurf[oldps*ps_w+MCXXPSCRIT]; /* same criterion as old */
      (*m).pcsurf[newps1*ps_w+MCXXPSORIE]=
	(*m).pcsurf[oldps*ps_w+MCXXPSORIE]; /* first edge has same
					       orientation as the old
					       edge 
					   */
      (*m).edge[neweg1*eg_w+MCT2EGPCSU  ]=newps1;
      /* newps2 */
      (*m).pcsurf[newps2*ps_w+MCXXPSSURF]=neweg2;
      (*m).pcsurf[newps2*ps_w+MCXXPSVOLE]=-1;/* faces are set in face
						refinement */
      (*m).pcsurf[newps2*ps_w+MCXXPSCRIT]=
	(*m).pcsurf[oldps*ps_w+MCXXPSCRIT]; /* same criterion as old */
      (*m).pcsurf[newps2*ps_w+MCXXPSORIE]=
	-(*m).pcsurf[oldps*ps_w+MCXXPSORIE];/* 2nd edge has reverse
					       orientation of the old
					       edge 
					   */
      (*m).edge[neweg2*eg_w+MCT2EGPCSU  ]=newps2;
    }

  /* new hierarchy entries */
  newhi1 = (*m).hi_nr;
  (*m).hi_nr++;
  (*m).hier[newhi1*hi_w+MCT2HICHLD  ] = newm1;
  (*m).hier[newhi1*hi_w+MCT2HIFAT1  ] = nod1;
  (*m).hier[newhi1*hi_w+MCT2HIFAT1+1] = nodm;
  (*m).hier[newhi1*hi_w+MCT2HILVL   ] = (*m).lvl;
  newhi2 = (*m).hi_nr;
  (*m).hi_nr++;
  (*m).hier[newhi2*hi_w+MCT2HICHLD  ] = newm2;
  (*m).hier[newhi2*hi_w+MCT2HIFAT1  ] = nod2;
  (*m).hier[newhi2*hi_w+MCT2HIFAT1+1] = nodm;
  (*m).hier[newhi2*hi_w+MCT2HILVL   ] = (*m).lvl;

  return SUCCESS;
}




/*FUNCTION*/
int mesh_t2_get_elem_from_face( struct mesh *m , FIDX i
/* extracts the nodes of the element i from the data in m.face and
   m.edge and stores the element i in m

   Input:  i        - the element to be build

   In/Out: m        - the mesh, is modified accordingly, 
                      only the elem part is modified

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  FIDX j, fc_w, el_w, eg_w;
  FIDX nodes[3], edges[3];

  fc_w=(*m).fc_w; el_w=(*m).el_w; eg_w=(*m).eg_w;

  /* the nodes of the element have to be ordered such that 0,1,2
     are the corners of the triangle, 3 is between 0 and 1, 4
     between 1 and 2, 5 between 2 and 0
  */


  for (j=0; j<3; j++)
    edges[j]=(*m).face[i*fc_w+MCT2FCEDG1+j];


  /* first, get the edges into order,
     first edge stays first and defines the first two points
  */
  nodes[0]=(*m).edge[edges[0]*eg_w+MCT2EGNOD1];
  nodes[1]=(*m).edge[edges[0]*eg_w+MCT2EGNOD1+1];
  
  /* now see which of the two other edges has node[1] as well */
  if ((nodes[1]!=(*m).edge[edges[1]*eg_w+MCT2EGNOD1])&&
      (nodes[1]!=(*m).edge[edges[1]*eg_w+MCT2EGNOD1+1]))
    {
      /* edges 2 and 3 have to be swaped */
      j=edges[2];
      edges[2]=edges[1];
      edges[1]=j;
      (*m).face[i*fc_w+MCT2FCEDG1+2]=
	(*m).face[i*fc_w+MCT2FCEDG1+1];
      (*m).face[i*fc_w+MCT2FCEDG1+1]=j;    
    }
  /* get the 3rd node */
  if (nodes[1]==(*m).edge[edges[1]*eg_w+MCT2EGNOD1  ])
    {
      /* ok, the second node of the second edge is the 3rd node of
	 the face
      */
      nodes[2]=(*m).edge[edges[1]*eg_w+MCT2EGNOD1+1];
    }
  else if (nodes[1]==(*m).edge[edges[1]*eg_w+MCT2EGNOD1+1])
    {
      /* ok, the first node of the second edge is the 3rd node of
	 the face
      */
      nodes[2]=(*m).edge[edges[1]*eg_w+MCT2EGNOD1  ];
    }
  else
    { 
      /* cry */
      fprintf(stderr,
      "mesh_t2_get_elem_from_face: face with one node only in one edge!\n");
      return FAIL;
    }
  
  for (j=0; j<3; j++)
    {
      /* nodes 0,1,2 */
      (*m).elem[i*el_w + j]=nodes[j];
      /* nodes 3,4,5 are midnodes of the edges */
      (*m).elem[i*el_w+3+j]=(*m).edge[edges[j]*eg_w+MCT2EGNODM];
    }

  return SUCCESS;
}




/*FUNCTION*/
int mesh_reorder_tx( struct mesh *m, int type
/* reorders the vertices, faces, elements, such that higher cache
   efficiency can be achied
   
   Input:  type    - the type of the mesh, type=1 linear, =2 quadratic

   In/Out: m       - the mesh, node numbers are overwritten

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		     ){
  FIDX dim, vx_w, vx_nr, eg_w, eg_nr, bd_w, bd_nr, el_w, el_nr, fc_w, 
    fc_nr, ps_w, pv_w, sg_w;
  FIDX i, j, el, eg, check_from_node, next_node_new, next_elem_new,
    curr_old, second_old, found_el, curr_el, found_next,
    node_curr_first, node_curr_last, elem_curr_first, elem_curr_last;

  FIDX l_mctxelnod1, l_mctxegnod1, l_mctxbdedge, l_mctxfcedg1,
    l_mctxegchl1, l_mc_elnodes;

  FIDX *node_new;          /* stores new number of node */
  FIDX *node_old;          /* stores old number of node */

  FIDX *elem_new;          /* stores new number of elem */
  FIDX *elem_old;          /* stores old number of elem */

  struct ilist *node2elem; /* nede2elem[i] lists all elements that
			      contain node i */
  struct ilist *curr;      /* pointer to navigate ilists */

  FIDX *edge2face;         /* edge2face[i*2+j] gives the j-th
			      element containig edge i */ 
  FIDX *node_bd_neighbours;/* to each node, if it is boundary_node,
			      store neigbouring nodes,
			      k=node_bd_neighbours[i*4+j]:
                              for j=0 and j=1 means node k is
                              neigbouring endpoint of an edge,
			      for t2 elements j=2 and j=3 store the
			      midpoints of neigbouring edges, if i
			      itself is an endpoint */
  FIDX *node2string;       /* tel for each node to which string of
			      nodes it belongs, if any */
  FIDX string_nr, longest_string, longest_strlen, longest_start;

  double *pvertex;         /* (temporary) array for reordering vertex*/
  FIDX *pelem;             /* (temporary) array for reordering elem */
  FIDX *pface;             /* (temporary) array for reordering face */
  double *pst;             /* (temporary) array for reordering st (t-values) */

  dim  = (*m).dim;

  vx_nr= (*m).vx_nr;
  vx_w = (*m).vx_w;
  eg_nr= (*m).eg_nr;
  eg_w = (*m).eg_w;
  bd_nr= (*m).bd_nr;
  bd_w = (*m).bd_w;
  el_nr= (*m).el_nr;
  el_w = (*m).el_w;
  fc_nr= (*m).fc_nr;
  fc_w = (*m).fc_w;

  ps_w = (*m).ps_w;
  pv_w = (*m).pv_w;
  sg_w = (*m).sg_w;


  switch (type)
    {
    case 1: /* T1 mesh */
      l_mctxelnod1 = MCT1ELNOD1;
      l_mctxegnod1 = MCT1EGNOD1;
      l_mctxbdedge = MCT1BDEDGE;
      l_mctxfcedg1 = MCT1FCEDG1;
      l_mctxegchl1 = MCT1EGCHL1;
      l_mc_elnodes = 3;
      break;
    case 2: /* T2 mesh */
      l_mctxelnod1 = MCT2ELNOD1;
      l_mctxegnod1 = MCT2EGNOD1;
      l_mctxbdedge = MCT2BDEDGE;
      l_mctxfcedg1 = MCT2FCEDG1;
      l_mctxegchl1 = MCT2EGCHL1;
      l_mc_elnodes = 6;
      break;
    default:
      fprintf(stderr,"mesh_reorder_tx: unknown type=%d\n",type);
      return FAIL;
    }

  /* init node_new, node_old, elem_new, elem_old */
  TRY_MALLOC(node_new, vx_nr, FIDX, mesh_reorder_tx);
  for (i=0; i<vx_nr; i++)
    { 
      node_new[i]=-1;
    }
  TRY_MALLOC(node_old, vx_nr, FIDX, mesh_reorder_tx);

  TRY_MALLOC(elem_new, el_nr, FIDX, mesh_reorder_tx);
  for (i=0; i<el_nr; i++)
    { 
      elem_new[i]=-1;
    }
  TRY_MALLOC(elem_old, el_nr, FIDX, mesh_reorder_tx);

  /*************************************************************
   *** 1st Stage: find strings of boundary nodes,            ***
   *** find longest string  (later to contain first nodes)   ***
   *************************************************************/

  /* set node_bd_neighbours */
  TRY_MALLOC(node_bd_neighbours, (4*vx_nr), FIDX, mesh_reorder_tx);
  /* init */
  for (i=0; i<vx_nr*4; i++)
    {
      node_bd_neighbours[i]=-1;
    }
      
  /* set */
  for (i=0; i<bd_nr; i++)
    {
      FIDX eg, node1, node2;
      eg= m->bound[i*bd_w+l_mctxbdedge];

      if (m->edge[eg*eg_w+l_mctxegchl1]==-1)
	{
	  node1 = m->edge[eg*eg_w+l_mctxegnod1  ];
	  node2 = m->edge[eg*eg_w+l_mctxegnod1+1];
	  if (     node_bd_neighbours[node1*4  ]==-1)
	    {
	      node_bd_neighbours[node1*4  ]=node2;
	    }
	  else if (node_bd_neighbours[node1*4+1]==-1)
	    {
	      node_bd_neighbours[node1*4+1]=node2;
	    }
	  else 
	    {
	      fprintf(stderr,"mesh_reorder_tx: edge %"dFIDX" has "
		      "node1= %"dFIDX" which is part of more than 2 boundary "
		      "edges\n", eg, node1);
	      return FAIL;
	    }
	  /* same for node2 */
	  if (     node_bd_neighbours[node2*4  ]==-1)
	    {
	      node_bd_neighbours[node2*4  ]=node1;
	    }
	  else if (node_bd_neighbours[node2*4+1]==-1)
	    {
	      node_bd_neighbours[node2*4+1]=node1;
	    }
	  else 
	    {
	      fprintf(stderr,"mesh_reorder_tx: edge %"dFIDX" has "
		      "node2= %"dFIDX" which is part of more than 2 boundary "
		      "edges\n", eg, node2);
	      return FAIL;
	    }

	  if (type==2)
	    { 
	      FIDX nodem;
	      nodem = m->edge[eg*eg_w+MCT2EGNODM];
	      if (     node_bd_neighbours[node1*4+2]==-1)
		{
		  node_bd_neighbours[node1*4+2]=nodem;
		}
	      else if (node_bd_neighbours[node1*4+3]==-1)
		{
		  node_bd_neighbours[node1*4+3]=nodem;
		}
	      else 
		{
		  fprintf(stderr,"mesh_reorder_tx: edge %"dFIDX" has "
			  "node1= %"dFIDX" which is part of more than 2 boundary "
			  "edges (nodem!) \n", eg, node1);
		  return FAIL;
		}
	      /* same for node2 */
	      if (     node_bd_neighbours[node2*4+2]==-1)
		{
		  node_bd_neighbours[node2*4+2  ]=nodem;
		}
	      else if (node_bd_neighbours[node2*4+3]==-1)
		{
		  node_bd_neighbours[node2*4+3]=nodem;
		}
	      else 
		{
		  fprintf(stderr,"mesh_reorder_tx: edge %"dFIDX" has "
			  "node2= %"dFIDX" which is part of more than 2 boundary "
			  "edges (nodem!) \n", eg, node2);
		  return FAIL;
		}

	      node_bd_neighbours[nodem*4  ]=node1;
	      node_bd_neighbours[nodem*4+1]=node2;
	    } /* if type==2 */
	} /* if edge has no children */
    } /* loop over boundaries */

  /* init node2string */
  TRY_MALLOC(node2string, vx_nr, FIDX, mesh_reorder_tx);
  for (i=0; i<vx_nr; i++)
    {
      node2string[i]=-1;
    }

  /* now collect strings of boundary nodes, store their lenght */
  string_nr=0;
  longest_string=-1; longest_strlen=0; longest_start= -1;
  for (check_from_node=0; check_from_node<vx_nr; check_from_node++)
    {
      if ((type==2)&&(node_bd_neighbours[check_from_node*4  ]!=-1)
	  &&(node_bd_neighbours[check_from_node*4+2]==-1))
	{
	  /* first non-endpoint reached, break the for loop */
	  break;
	}

      if (node_bd_neighbours[check_from_node*4  ]!=-1)
	{
	  if (node2string[check_from_node]==-1)
	    {
	      /* is starting node of new string */
	      FIDX current, last, strlen;

	      strlen=0;
	      current=check_from_node;
	      last=-1;
	      while ((current!=-1)&&(node2string[current]==-1))
		{
		  //fprintf(stderr,"  string %d, node %"dFIDX" \n", string_nr, current);

		  /* add node to string */
		  node2string[current]=string_nr;
		  strlen++;
		  /* figure out next node */
		  if (last!=node_bd_neighbours[current*4  ])
		    {
		      last=current;
		      current=node_bd_neighbours[current*4  ];
		    }
		  else
		    {
		      last=current;
		      current=node_bd_neighbours[current*4+1];
		    }
		}
	      /* end off current string/loop */
	      //fprintf(stderr,"reorder    string=%"dFIDX" strlen=%"dFIDX" start=%"dFIDX"\n", string_nr, strlen, check_from_node);
	      if (strlen>longest_strlen)
		{
		  longest_string=string_nr;
		  longest_strlen=strlen;
		  longest_start=check_from_node;
		}
	      string_nr++;
	      /* search next starting point of string */
	    } /* if not already on sting */
	} /* if node on boundary */
    } /* loop check_from_node */

  //fprintf(stderr,"reorder    longest_string=%"dFIDX" longest_strlen=%"dFIDX" longest_start=%"dFIDX"\n", longest_string, longest_strlen, longest_start);

  /*************************************************************
   *** 1st Stage done                                        ***
   *** longest string of nodes defined                       ***
   *************************************************************/
  




  /*************************************************************
   *** 2nd Stage:                                            ***
   *** simplify access to neighbours                         ***
   *************************************************************/
  
  TRY_MALLOC(node2elem, vx_nr, struct ilist, mesh_reorder_tx);
  for (i=0; i<vx_nr; i++)
    {
      node2elem[i].data=-1;
      node2elem[i].next=NULL;
    }

  for (el=0; el<el_nr; el++)
    {
      for (i=0; i<3; i++)
	{
	  FIDX node;
	  struct ilist *curr;
	  node=m->elem[el*el_w+l_mctxelnod1+i];
	  curr=&node2elem[node];
	  while (curr->next != NULL)
	    {
	      curr = curr->next;
	    }
	  curr->data=el;
	  TRY_MALLOC(curr->next, 1, struct ilist, mesh_reorder_tx);
	  curr->next->data=-1;
	  curr->next->next=NULL;
	}
    } /* end loop el */

  TRY_MALLOC(edge2face, 2*eg_nr, FIDX, mesh_reorder_tx);
  for (i=0; i<2*eg_nr; i++)
    {
      edge2face[i]=-1;
    }
  for (el=0; el<el_nr; el++)
    {
      for (i=0; i<3; i++)
	{
	  FIDX eg;
	  eg=m->face[el*fc_w+l_mctxfcedg1+i];
	  if (edge2face[eg*2  ]==-1)
	    {
	      edge2face[eg*2  ]=el;
	    }
	  else if (edge2face[eg*2+1]==-1)
	    {
	      edge2face[eg*2+1]=el;
	    }
	  else 
	    {
	      fprintf(stderr,"mesh_reorder_tx: edge %"dFIDX" in more "
		     "than 2 elements\n", eg);
	      return FAIL;
	    }
	} 
    } /* end loop eg */

  /*************************************************************
   *** 2nd Stage finished                                    ***
   *** simplify access to neighbours                         ***
   *************************************************************/


  /*************************************************************
   *** 3rd Stage:                                            ***
   *** follow string of boundary nodes to define first       ***
   *** string of elements and number nodes                   ***
   *************************************************************/

  next_node_new=0;
  next_elem_new=0;
  curr_old=longest_start;
  /* give this node a new number */
  node_new[curr_old]=next_node_new;
  node_old[next_node_new]=curr_old;
  next_node_new++;

  /* give the second node a new number */
  second_old=node_bd_neighbours[curr_old*4];
  node_new[second_old]=next_node_new;
  node_old[next_node_new]=second_old;
  next_node_new++;


  //fprintf(stderr,"first 2 nodes: node1=%"dFIDX"  node2=%"dFIDX"\n", 
  //		  curr_old, second_old);

  /* give all elements that contain this node a new number, start with
     the element that contains the two first nodes, then all with the
     second node */
  /* find element with the first two nodes */
  curr=&node2elem[curr_old];
  found_el=0;
  //fprintf(stderr,"reorder  el_nr=%"dFIDX"\n", el_nr);
  while ((curr!=NULL)&&(found_el==0))
    {
      FIDX el;
      int belongs_string;

      el=curr->data;
      if (el==-1)
	{
	  fprintf(stderr,"mesh_reorder_tx: element with first two nodes "
		  "not found, node1=%"dFIDX"  node2=%"dFIDX"\n", 
		  curr_old, second_old);
	  return FAIL;
	}


      //fprintf(stderr,"reorder    el=%"dFIDX"  ", el);
      belongs_string=0;
      for (i=0; i<3; i++)
	{
	  
	  //fprintf(stderr,"  %"dFIDX, m->elem[el*el_w+l_mctxelnod1+i]);
	  belongs_string+=(m->elem[el*el_w+l_mctxelnod1+i]==curr_old)
	    +(m->elem[el*el_w+l_mctxelnod1+i]==second_old);
	}
      //fprintf(stderr," belongs=%d\n", belongs_string);
      if (belongs_string==2)
	{
	  found_el=1;
	}
      else
	{
	  curr=curr->next;
	}
    }
  if (found_el==0)
    {
      fprintf(stderr,"mesh_reorder_tx: no element found with first two "
	      "nodes, node1=%"dFIDX" node2=%"dFIDX"\n",curr_old, second_old);
      return FAIL;
    }
  curr_el=curr->data;

  /* found the first element */
  elem_new[curr_el]=next_elem_new;
  elem_old[next_elem_new]=curr_el;
  next_elem_new++;
  
  /* now look for the next elements containig this node */
  curr_old=second_old;

  found_next=1;
  while (found_next==1)
    {
      //fprintf(stderr," reoder: curr_old %"dFIDX" \n", curr_old); 

      found_next=0;

      /* give all elements that contain this node a new number, unless
	 they already got one */
      
      /* loop over all elements that neighbour the current element */
      for (i=0; i<3; i++)
	{
	  FIDX other_el;
	  eg=m->face[curr_el*fc_w+l_mctxfcedg1+i];
	  if (edge2face[eg*2  ]!=curr_el)
	    {
	      other_el=edge2face[eg*2  ];
	    }
	  else
	    {
	      other_el=edge2face[eg*2+1];
	    }
	  /* now check if this element has a new number already and if
	     it contains our node */
	  if ((other_el>=0)&&(elem_new[other_el]==-1))
	    {
	      /* no new number yet, check for our node */
	      int nodeinel=0;
	      for (j=0; j<3; j++)
		{
		  nodeinel+=
		    (m->elem[other_el*el_w+l_mctxelnod1+j]==curr_old);
		}
	      if (nodeinel!=0)
		{
		  /* this is our next element in the string */
		  elem_new[other_el]=next_elem_new;
		  elem_old[next_elem_new]=other_el;
		  next_elem_new++;

		  //fprintf(stderr," reoder:   element %"dFIDX" \n", other_el); 

		  /* there can only be one neighbouring element with
		     the same node, so continue from this new element */ 
		  curr_el=other_el;
		  found_next=1;
		  break; /* loop i */
		}
	    } /* other el has no new number yet */
	}/* loop i */

      /* if we haven't found a new element with this node, look for
	 the next node, this should be a node of the current element */
      if (found_next==0)
	{
	  FIDX next_node=-1;
	  for (i=0; i<3; i++)
	    {
	      FIDX node=m->elem[curr_el*el_w+l_mctxelnod1+i];
	      if ((node!=curr_old)&&(node2string[node]==longest_string))
		{
		  next_node=node;
		}
	    }
	  
	  if (next_node!=-1)
	    {
	      /* if this has no number yet, it is our new node, if it
		 already has a new number, we finished the first loop */
	      if (node_new[next_node]==-1)
		{
		  /* give this node a new number */
		  curr_old=next_node;
		  node_new[curr_old]=next_node_new;
		  node_old[next_node_new]=curr_old;
		  next_node_new++;
		  found_next=1;
		}
	    }
	} /* end looking for new node in current element */

    } /* as long as new element or node is found */


  /*************************************************************
   *** 3rd Stage finished                                    ***
   *** follow string of boundary nodes to define first       ***
   *** string of elements and number nodes                   ***
   *************************************************************/



  elem_curr_first = 0;
  elem_curr_last  = next_elem_new;

  while (elem_curr_first!=elem_curr_last)
    {
      /*************************************************************
       *** 4th Stage:                                            ***
       *** follow elements to define next string of nodes        ***
       *************************************************************/

      node_curr_first = next_node_new;

      for (i=elem_curr_first; i<elem_curr_last; i++)
	{
	  FIDX el=elem_old[i];
	  
	  /* test all corner nodes of this element if they have a new
	     number already, if not give them one, prefer boundary
	     nodes */
	  for (j=0; j<3; j++)
	    {
	      FIDX node=m->elem[el*el_w+l_mctxelnod1+j];
	      if ((node_new[node]==-1)&&( node_bd_neighbours[node*4  ]!=-1))
		{
		  node_new[node]=next_node_new;
		  node_old[next_node_new]=node;
		  next_node_new++;
		}
	    }
	  for (j=0; j<3; j++)
	    {
	      FIDX node=m->elem[el*el_w+l_mctxelnod1+j];
	      if (node_new[node]==-1)
		{
		  node_new[node]=next_node_new;
		  node_old[next_node_new]=node;
		  next_node_new++;
		}
	    }
	}

      node_curr_last  = next_node_new;

      /*************************************************************
       *** 4th Stage finished                                    ***
       *** follow elements to define next string of nodes        ***
       *************************************************************/

      /*************************************************************
       *** 5th Stage:                                            ***
       *** follow nodes to define next string of elements,       ***
       *** start with the new element furthest down the new node ***
       *** numbers                                               ***
       *************************************************************/

      
      /* printf("elem_curr_first =%"dFIDX" elem_curr_last =%"dFIDX"\n", 
	 elem_curr_first, elem_curr_last); /*/

      elem_curr_first = next_elem_new;

      /* non-standard for loop for starting element strings,
	 check_from_node may be increased within a step */
      for (check_from_node=node_curr_first; 
	   check_from_node<node_curr_last;
	   check_from_node++)
	{
	  FIDX dead_end1, dead_end2, dead_end1_hn, dead_end2_hn, last_el;
	  int keep_searching;

	  /* look for first element that contains check_from_node but
	     has no new number yet */
	  curr=&node2elem[node_old[check_from_node]];
	  found_el=0;
	  while ((curr!=NULL)&&(found_el==0))
	    {
	      FIDX el;

	      el=curr->data;
	      if ((el!=-1) && (elem_new[el]==-1))
		{
		  found_el=1;
		}
	      else
		{
		  curr=curr->next;
		}
	    }

	  /* check if node belongs to element without new number */
	  if (curr!=NULL)
	    {
	      /* determine the most downward element based on this
		 one */
	      curr_el=curr->data;

	      /*fprintf(stderr," reodebug:  start looking at
		curr_el=%"dFIDX"\n",curr_el); /*/
	  
	      /* first "walk" the elements in one direction along the
		 check_from_node,
	 
		 if we find an element with a new node other than
		 check_from_node and (check_from_node+1), then stop,
		 this is the starting element

		 if no connecting element can be found any more, store
		 the current one as candidate (dead-end)

		 if no definite starting element has been found, turn
		 around, "walk" in other direction where either of the
		 following cases must happen:

		 a) we find the starting element according to the
		    condition above,

		 b) find an element with (check_from_node+1), then we
		    use the stored dead-end as starting element

		 c) another dead-end -> we may use either dead-end as
		    starting element
	      */
	      keep_searching =2;
	      dead_end1      =-1;
	      dead_end2      =-1;
	      dead_end1_hn   =-1;
	      dead_end2_hn   =-1;
	      last_el        =-1;
	      while (keep_searching!=0)
		{
		  /* fprintf(stderr," reodebug:  am at
		     curr_el=%"dFIDX"\n",curr_el); /*/

		  /* try to find an other element without new number,
		     different from the current and the last, but with
		     check_from_node as one of its nodes */
		  if (keep_searching!=0)
		    {
		      FIDX high_nb=-1;

		      found_next=0;
		      
		      /* loop over all elements that neighbour the
			 current element */
		      for (i=0; i<3; i++)
			{
			  FIDX other_el;

			  eg=m->face[curr_el*fc_w+l_mctxfcedg1+i];
			  if (edge2face[eg*2  ]!=curr_el)
			    {
			      other_el=edge2face[eg*2  ];
			    }
			  else
			    {
			      other_el=edge2face[eg*2+1];
			    }

			  /* compute highest name of neighbouring new
			     element */
			  if ((other_el>=0)&&(elem_new[other_el]>high_nb))
			    {
			      high_nb=elem_new[other_el];
			    }

			  /* now check if this element has a new
			     number already, if it is different from
			     last_el and if it contains our node */
			  if ((other_el>=0)&&(elem_new[other_el]==-1)
			      &&(other_el!=last_el))
			    {
			      /* indeed potential new element, check
				 for our node */
			      int nodeinel=0;
			      for (j=0; j<3; j++)
				{
				  nodeinel+=
				    (node_new[m->elem[other_el*el_w
						      +l_mctxelnod1+j]]
				     ==check_from_node);
				}
			      if (nodeinel!=0)
				{
				  /* advance to this element */
				  last_el=curr_el;
				  curr_el=other_el;
				  found_next=1;
				  break; /* loop over i */
				}
			    } /* if potential new element */
			} /* end loop over edges of element, looking
			     for next element */
		      if (found_next==0)
			{
			  /* fprintf(stderr," reodebug:  hit dead_end "
			     "at curr_el=%"dFIDX"\n",curr_el); /*/

			  /* no other element in this direction found,
			     mark as dead-end, allow reverse of
			     direction for searching (but at most one
			     time) */
			  if (dead_end1==-1)
			    {
			      dead_end1    = curr_el;
			      dead_end1_hn = high_nb;
			    }
			  else
			    {
			      dead_end2    = curr_el;
			      dead_end2_hn = high_nb;
			    }
		      
			  last_el  = -1;    /* change direction from
					       this dead-end */
			  keep_searching--; /* 2->1->0, 0=stop */
			}
		    } /* end if keep searching (new neighbour) */
		} /* while keep_searching */
	  
	      if ((dead_end2!=-1)&&(dead_end1!=dead_end2))
		{
		  /* found two different dead_ends, use the one with
		     highers new neighbour */
		  
		  /* fprintf(stderr," reodebug:  dead_end1=%"dFIDX" hn=%"dFIDX"\n",dead_end1, dead_end1_hn);
		     fprintf(stderr," reodebug:  dead_end2=%"dFIDX"
		     hn=%"dFIDX"\n",dead_end2, dead_end2_hn); /*/
		  

		  /* prefer nodes which are actual dead-ends,
		     i.e. have high_nb=-1 */
		  if (dead_end1_hn==-1)
		    curr_el=dead_end1;
		  else if (dead_end2_hn==-1)
		    curr_el=dead_end2;
		  else if (dead_end1_hn>dead_end2_hn)
		    curr_el=dead_end1;
		  else
		    curr_el=dead_end2;
		}
      
	      /* found the first new element */
	      elem_new[curr_el]=next_elem_new;
	      elem_old[next_elem_new]=curr_el;
	      next_elem_new++;

	      /* now collect all elements connected to new_node
		 check_from_node and (check_from_node+1), starting
		 from this element if it contains (check_from_node+1)
		 then increment check_from_node */
	      found_next=1;
	      while (found_next==1)
		{
		  found_next=0;
		  /* loop over all elements that neighbour the current
		     element */
		  for (i=0; i<3; i++)
		    {
		      FIDX other_el;
		      eg=m->face[curr_el*fc_w+l_mctxfcedg1+i];
		      if (edge2face[eg*2  ]!=curr_el)
			{
			  other_el=edge2face[eg*2  ];
			}
		      else
			{
			  other_el=edge2face[eg*2+1];
			}
		      /* now check if this element has a new number
			 already, and if it contains our node */
		      if ((other_el>=0)&&(elem_new[other_el]==-1))
			{
			  /* indeed potential new element, check for
			     our node */
			  int nodeinel=0;
			  int node_p1=0;
			  for (j=0; j<3; j++)
			    {
			      nodeinel+=
				(node_new[m->elem[other_el*el_w+l_mctxelnod1+j]]
				 ==check_from_node);
			      node_p1+=
				(node_new[m->elem[other_el*el_w+l_mctxelnod1+j]]
				 ==check_from_node+1);
			    }
			  if (nodeinel!=0)
			    {
			      /* advance to this element, mark it as
				 new element */
			      curr_el=other_el;
			      found_next=1;
			      elem_new[curr_el]=next_elem_new;
			      elem_old[next_elem_new]=curr_el;
			      next_elem_new++;

			      /* next node also contained?, increment
				 check_from_node */
			      if (node_p1!=0)
				{
				  check_from_node++;
				}
			  
			      break; /* loop over i */
			    }
			} /* if potential new element */
		    } /* end loop i over edges of element, looking for
			 next element */
		} /* while found_next */
	    } /* end if node belongs to element without new number */
	} /* end for check_from_node */

      elem_curr_last  = next_elem_new;
      
      /*************************************************************
       *** 5th Stage finished                                    ***
       *** follow nodes to define next string of elements,       ***
       *** start with the new element furthest down the new node ***
       *** numbers                                               ***
       *************************************************************/
    } /* while found new elements */

  /* check that we actually have all elements */
  if (next_elem_new!=el_nr)
    { 
      fprintf(stderr,"mesh_reorder_tx: not all elements renumbered by "
	      "reordering algorithm, find out why!\n "
	      "el_nr=%"dFIDX"  renumbered el_nr=%"dFIDX"\n",
	      el_nr, next_elem_new);
      return FAIL;
    }

  /* finally, for type=2 meshes collect the quadratic dofs */
  if (type==2)
    {
      for (i=0; i<el_nr; i++)
	{
	  FIDX el=elem_old[i];
	  
	  /* test all edge midpoints of this element if they have a
	     new number already, if not give them one */
	  for (j=0; j<3; j++)
	    {
	      FIDX node=m->elem[el*el_w+l_mctxelnod1+3+j];
	      if (node_new[node]==-1)
		{
		  node_new[node]=next_node_new;
		  node_old[next_node_new]=node;
		  next_node_new++;
		}
	    }
	}
    }

  /* now all nodes should have a new number */
  if (next_node_new!=vx_nr)
    { 
      int l_mctxvxstrt;
      if (type==1)
	l_mctxvxstrt=MCT1VXSTRT;
      else
	l_mctxvxstrt=MCT2VXSTRT;

      fprintf(stderr,"mesh_reorder_tx: not all nodes renumbered by "
	      "reordering algorithm, find out why!\n "
	      "vx_nr=%"dFIDX"  renumbered vx_nr=%"dFIDX"\n",
	      vx_nr, next_node_new);
      for (i=0; i<vx_nr; i++)
	{
	  if (node_new[i]==-1)
	    {
	      fprintf(stderr,"  node %"dFIDX"   (%e,%e)\n", i,
		      m->vertex[i*vx_w+l_mctxvxstrt  ],
		      m->vertex[i*vx_w+l_mctxvxstrt+1]);
	    }
	}

      return FAIL;
    }


  /* debug_output */
  if (0==1)
  {
    FILE *outfile, *outfile2;

    printf("el_nr =%"dFIDX"   el_nr_new =%"dFIDX"\n", el_nr, next_elem_new);
    printf("vx_nr =%"dFIDX"   vx_nr_new =%"dFIDX"\n", vx_nr, next_node_new);

    printf("check point sequence:\n");

    outfile=fopen("/tmp/reorder_nd.txt","w");
    for (i=0; i<next_node_new; i++)
      {
	//printf("  %"dFIDX"  ", node_old[i]);
	fprintf(outfile," %e %e %"dFIDX" \n", m->vertex[node_old[i]*m->vx_w  ],
		m->vertex[node_old[i]*m->vx_w+1], i);
      }
    fclose(outfile);

    outfile=fopen("/tmp/reorder_nd_old.txt","w");
    for (i=0; i<next_node_new; i++)
      {
	fprintf(outfile," %e %e %"dFIDX" \n", m->vertex[i*m->vx_w  ],
		m->vertex[i*m->vx_w+1], i);
      }
    fclose(outfile);

    outfile=fopen("/tmp/reorder_el.txt","w");
    outfile2=fopen("/tmp/reorder_el_vx.txt","w");
    for (i=0; i<next_elem_new; i++)
      {
	int d;
	//printf("  %"dFIDX"  ", node_old[i]);
	double vx[2]={0.0, 0.0};
	for (j=0; j<3; j++)
	  {
	    for (d=0; d<2; d++)
	      {
		double vxd=m->vertex[m->elem[elem_old[i]*el_w+l_mctxelnod1+j]*m->vx_w+d];
		vx[d]+=vxd;
		fprintf(outfile2," %e ", vxd);
	      }
	    fprintf(outfile2,"\n");
	  }
	j=0;
	for (d=0; d<2; d++)
	  {
	    double vxd=m->vertex[m->elem[elem_old[i]*el_w+l_mctxelnod1+j]*m->vx_w+d];
	    fprintf(outfile2," %e ", vxd);
	  }
	fprintf(outfile2,"\n\n");
		
	fprintf(outfile," %e %e %"dFIDX" \n", vx[0]/3.0, vx[1]/3.0, i);
      }
    fclose(outfile);
    fclose(outfile2);


    outfile=fopen("/tmp/reorder_el_old.txt","w");
    outfile2=fopen("/tmp/reorder_el_old_vx.txt","w");
    for (i=0; i<next_elem_new; i++)
      {
	int d;
	double vx[2]={0.0, 0.0};
	for (j=0; j<3; j++)
	  {
	    for (d=0; d<2; d++)
	      {
		double vxd=m->vertex[m->elem[i*el_w+l_mctxelnod1+j]*m->vx_w+d];
		vx[d]+=vxd;
		fprintf(outfile2," %e ", vxd);
	      }
	    fprintf(outfile2,"\n");
	  }
	j=0;
	for (d=0; d<2; d++)
	  {
	    double vxd=m->vertex[m->elem[i*el_w+l_mctxelnod1+j]*m->vx_w+d];
	    fprintf(outfile2," %e ", vxd);
	  }
	fprintf(outfile2,"\n\n");
		
	fprintf(outfile," %e %e %"dFIDX" \n", vx[0]/3.0, vx[1]/3.0, i);
      }
    fclose(outfile);
    fclose(outfile2);
    printf("end \n");
    printf("ploting ordering in gnuplot, press 'q' or mouse button or CTRL-'c' to terminate GNUPLOT\n");
    system("(echo \'plot \"/tmp/reorder_nd.txt\" using 1:2 with lines, \"/tmp/reorder_nd.txt\" using 1:2:3  with labels; pause mouse\' |gnuplot) & \n "
	   "(echo \'plot \"/tmp/reorder_el.txt\" using 1:2 with lines, \"/tmp/reorder_el.txt\" using 1:2:3  with labels, \"/tmp/reorder_el_vx.txt\" using 1:2 with lines; pause mouse\' |gnuplot)"); 
    return FAIL;
  } /* end debug output */


  /* intermediate clean up */
  free(edge2face);
  for (i=0; i<vx_nr; i++)
    {
      ilist_free(&node2elem[i].next);
    }
  free(node2elem);
  free(node2string);
  free(node_bd_neighbours);



  printf("debug: test sanity order 1...vx_nr forth and back\n");
  {
    FIDX *nrs, *pnrs;
    TRY_MALLOC(nrs, vx_nr, FIDX, mesh_reorder_tx);
    TRY_MALLOC(pnrs, vx_nr, FIDX, mesh_reorder_tx);
    
    /* init nrs */
    for (i=0; i<vx_nr; i++)
      {
	nrs[i]=i;
      }

    /* pnrs=node_new(nrs) */
    for (i=0; i<vx_nr; i++)
      {
	pnrs[i]=node_new[nrs[i]];
      }
    
    /* nrs=node_old(pnrs) */
    for (i=0; i<vx_nr; i++)
      {
	nrs[i]=node_old[pnrs[i]];
      }

    /* check sequence */
    for (i=0; i<vx_nr; i++)
      {
	if (i!=nrs[i])
	  {
	    printf("order wrong, i=%"dFIDX"\n", i);
	  }
      }

    free(pnrs);
    free(nrs);
  }



  /*************************************************************
   ***                                                       ***
   *** perform the actual permutation                        ***
   ***                                                       ***
   *************************************************************/

  /* allocate memory for permuted arrays */
  TRY_MALLOC(pvertex, m->vx_max*vx_w, double,  mesh_reorder_tx);
  TRY_MALLOC(pelem,   m->el_max*el_w, FIDX,    mesh_reorder_tx);
  TRY_MALLOC(pface,   m->fc_max*fc_w, FIDX,    mesh_reorder_tx);
  TRY_MALLOC(pst,     m->st_max,      double,  mesh_reorder_tx);
  
  /* vertex */
  for (i=0; i<vx_nr; i++)
    {
      memcpy(&pvertex[i*vx_w],&(m->vertex[node_old[i]*vx_w]),
	     vx_w*sizeof(double));
    }
  /* face */
  for (i=0; i<fc_nr; i++)
    {
      memcpy(&pface[i*fc_w],&(m->face[elem_old[i]*fc_w]),
	     fc_w*sizeof(FIDX));
    }

  /* elem */
  for (i=0; i<el_nr; i++)
    {
      for (j=0; j<l_mc_elnodes; j++)
	{
	  pelem[i*el_w+l_mctxelnod1+j]=
	    node_new[m->elem[elem_old[i]*el_w+l_mctxelnod1+j]];
	}
    }

  /* now all places where node or element numbers appear */
  /* pcsurf */
  for (i=0; i<m->ps_nr; i++)
    {
      m->pcsurf[i*ps_w+MCXXPSVOLE]=elem_new[m->pcsurf[i*ps_w+MCXXPSVOLE]];
    }
  /* pcvol */
  for (i=0; i<m->pv_nr; i++)
    {
      m->pcvol[i*pv_w+MCXXPVVOLM]=elem_new[m->pcvol[i*pv_w+MCXXPVVOLM]];
    }
  /* sseg */
  for (i=0; i<m->sg_nr; i++)
    {
      m->sseg[i*sg_w+MC2XSGNOD1]=node_new[m->sseg[i*sg_w+MC2XSGNOD1]];
    }

  /* edges */
  for (i=0; i<m->eg_nr; i++)
    {
      m->edge[i*eg_w+l_mctxegnod1  ]=node_new[m->edge[i*eg_w+l_mctxegnod1  ]];
      m->edge[i*eg_w+l_mctxegnod1+1]=node_new[m->edge[i*eg_w+l_mctxegnod1+1]];
    }
  if (type==2)
    {
      for (i=0; i<m->eg_nr; i++)
	{
	  m->edge[i*eg_w+MCT2EGNODM]=node_new[m->edge[i*eg_w+MCT2EGNODM]];
	}
    }

  printf("debug: st_nr=%"dFIDX"  vx_nr=%"dFIDX"\n", m->st_nr, m->vx_nr);

  /* st (t-values) */
  for (i=0; i<m->st_nr; i++)
    {
      pst[node_new[i]]=m->st[i];
    }


#warning "mesh_reorder_tx: edge hierarchy not treated yet"
#warning "mesh_reorder_tx: element hierarchy not treated yet"


  /* replace array by reordered versions */
  free(m->vertex);
  m->vertex=pvertex;

  free(m->elem);
  m->elem=pelem;

  free(m->face);
  m->face=pface;
  
  free(m->st);
  m->st=pst;


  /* final clean up */
  free(elem_old);
  free(elem_new);
  free(node_old);
  free(node_new);

  printf("mesh_reoder_tx: done\n");

  return SUCCESS;
}



/*FUNCTION*/
void multilvl_free( struct multilvl *ml
/* empties the multilvl data structure, such that it can be freed

   Output: ml      - internal memory is released,
*/
	       ){
  (*ml).vx_nr = 0; 
  (*ml).dim   = 0; 
  (*ml).lmax  = 0; 

  free((*ml).nlevl);
  (*ml).nlevl=NULL;

  free((*ml).levlvx);
  (*ml).levlvx=NULL;

  free((*ml).levlhindx);
  (*ml).levlhindx=NULL;

  return;
}


/*FUNCTION*/
void multilvl_null_def( struct multilvl *ml
/* defines the values in multilvl data structure, such that it can be
   given to multilvl_free and give no errors, even before
   multilvl_init is called

   Output: ml      - is defined to be empty,
*/
	       ){
  (*ml).vx_nr = 0; 
  (*ml).dim   = 0; 
  (*ml).lmax  = 0; 

  (*ml).nlevl=NULL;
  (*ml).levlvx=NULL;
  (*ml).levlhindx=NULL;

  return;
}

/*FUNCTION*/
int multilvl_init_t2( struct mesh *m, FIDX datdim, struct multilvl *ml
/* initialises the multilvl data structure
   
   Input:  m       - the mesh (T2) to which the multilvl data shall
                     correspond (the important bit is the hierarchy
                     data) 
	   datdim  - the number of dofs per node for which the
	             multilvl data will be created, on each of the
	             levels dim*(this levels vx_nr) components will be
	             available

   Output: ml      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  int err;

  /* call the general code for type=2 */
  err=multilvl_init_tx( m, datdim, ml, 2); 
  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, multilvl_init_t2);
  
  return SUCCESS;
}




/*FUNCTION*/
int multilvl_init_tx( struct mesh *m, FIDX datdim, struct multilvl *ml,
		      int type
/* initialises the multilvl data structure
   
   Input:  m       - the mesh to which the multilvl data shall
                     correspond (the important bit is the hierarchy
                     data) 
	   datdim  - the number of dofs per node for which the
	             multilvl data will be created, on each of the
	             levels dim*(this levels vx_nr) components will be
	             available
           type    - the type of the mesh, type=1 linear, =2 quadratic

   Output: ml      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, dim, vx_nr, hi_w, hi_nr, eh_w, eh_nr, eg_w, eg_nr;
  FIDX level, lvl; 

  FIDX l_mctxhilvl, l_mctxhichld,  l_mctxhifat1, l_mctxehlvl,
    l_mctxegnod1, l_mctxegnods, l_mctxeglvl;

  vx_nr= (*m).vx_nr;
  dim  = (*m).dim;
  eg_nr= (*m).eg_nr;
  eg_w = (*m).eg_w;
  hi_nr= (*m).hi_nr;
  hi_w = (*m).hi_w;
  eh_nr= (*m).eh_nr;
  eh_w = (*m).eh_w;

  /* basic init */
  (*ml).vx_nr = vx_nr; 
  (*ml).dim   = datdim; 

  switch (type)
    {
    case 1: /* T1 mesh */
      l_mctxhilvl  = MCT1HILVL;
      l_mctxhichld = MCT1HICHLD;
      l_mctxhifat1 = MCT1HIFAT1;
      l_mctxegnod1 = MCT1EGNOD1;
      l_mctxegnods = 2;
      l_mctxeglvl  = MCT1EGLVL;
      l_mctxehlvl  = 0;
      break;
    case 2: /* T2 mesh */
      l_mctxhilvl  = MCT2HILVL;
      l_mctxhichld = MCT2HICHLD;
      l_mctxhifat1 = MCT2HIFAT1;
      l_mctxegnod1 = MCT2EGNOD1;
      l_mctxegnods = 3;
      l_mctxeglvl  = MCT2EGLVL;
      l_mctxehlvl  = MCT2EHLVL;
      break;
    case 31: /* E1 mesh */
      l_mctxhilvl  = MCE1HILVL;
      l_mctxhichld = MCE1HICHLD;
      l_mctxhifat1 = MCE1HIFAT1;
      l_mctxegnod1 = MCE1EGNOD1;
      l_mctxegnods = 2;
      l_mctxeglvl  = MCE1EGLVL;
      l_mctxehlvl  = 0;
      break;
    default:
      fprintf(stderr,"multilvl_init_tx: unknown type=%d\n",type);
      return FAIL;
    }

  (*ml).type=type;
    

  /* check for the highest level */
  (*ml).lmax=0; 
  for (i=0; i<hi_nr; i++)
    {
      level=(*m).hier[i*hi_w+l_mctxhilvl]+1;
      if (level>(*ml).lmax) (*ml).lmax=level;
      if (level<0) 
	{
	  fprintf(stderr,"multilvl_init_tx: level<0 not allowed!\n");
	  return FAIL;
	}
    }
 /* fprintf(stderr,"multilvl_init_tx: lmax=%"dFIDX"!\n", (*ml).lmax); /* */

  /* allocate memory for nlevl and levlhindx */
  TRY_MALLOC( (*ml).nlevl, (*ml).lmax+2, FIDX, multilvl_init_tx);
  TRY_MALLOC( (*ml).levlvx, (*ml).lmax+1, FIDX, multilvl_init_tx);
  TRY_MALLOC((*ml).levlhindx, (*ml).lmax+2, FIDX, multilvl_init_tx);
  

  /* search the highest nodennumber in each level */
  for (i=0; i<=(*ml).lmax+1; i++) (*ml).nlevl[i]=-1;
  /* for adaptive refinements the level=0 has to be done differently */
  level=-1;
  for (i=0; (i<eg_nr)&&(level==-1); i++)
    {
      level=(*m).edge[i*eg_w + l_mctxeglvl];
      if (level==-1)
	{
	  FIDX j;
	  for (j=0; j<l_mctxegnods; j++)
	    {
	      if ((*ml).nlevl[0]<(*m).edge[i*eg_w + l_mctxegnod1+j])
		(*ml).nlevl[0]=(*m).edge[i*eg_w + l_mctxegnod1+j];
	    }
	}
    }
  /*fprintf(stderr,"DEBUG: 1: nlevl[0]=%3"dFIDX"\n",(*ml).nlevl[0]);*/
  /* other levels based on hierarchy */
  for (i=0; i<hi_nr; i++)
    {
      level=(*m).hier[i*hi_w + l_mctxhilvl];
      if ((*ml).nlevl[level+1]<(*m).hier[i*hi_w + l_mctxhichld])
	(*ml).nlevl[level+1]=(*m).hier[i*hi_w + l_mctxhichld];
      if ((*ml).nlevl[level+1]<(*m).hier[i*hi_w + l_mctxhifat1])
	(*ml).nlevl[level+1]=(*m).hier[i*hi_w + l_mctxhifat1];
      if ((*ml).nlevl[level+1]<(*m).hier[i*hi_w + l_mctxhifat1+1])
	(*ml).nlevl[level+1]=(*m).hier[i*hi_w + l_mctxhifat1+1];
    }
  /* if no hierarchy given, use current vertex number */
  if (hi_nr==0) (*ml).nlevl[0]=(*m).vx_nr-1;
  /*fprintf(stderr,"DEBUG: 2: nlevl[0]=%3"dFIDX"\n",(*ml).nlevl[0]);*/

  /* now correct from highest index to number of entries and copy it
     to levlvx */
  for (i=0; i<=(*ml).lmax+1; i++)
    {
      (*ml).nlevl[i]+=1;
      /*fprintf(stderr,"DEBUG: 3: nlevl[%d]=%3"dFIDX"\n",i,(*ml).nlevl[i]);*/
    }
  for (i=0; i<=(*ml).lmax; i++)
    {
      (*ml).levlvx[i]=(*ml).nlevl[i];
    }

  
  if ((*ml).nlevl[(*ml).lmax]!=(*m).vx_nr)
    {
      fprintf(stderr, 
	      "multilvl_init_tx: something is wrong with vx_nr=%"dFIDX", "
	      "max_vx=%"dFIDX"\n",  (*m).vx_nr,
	       (*ml).nlevl[(*ml).lmax]);
      return FAIL;
    }

  /* now define the end positions of each level in the help vectors,
     the finest level lmax comes first, then..., end position of one
     vector=start position of the next */
  for (i=(*ml).lmax;   i>=0; i--) (*ml).nlevl[i]*=datdim;
  for (i=(*ml).lmax-1; i>=0; i--) (*ml).nlevl[i]+=(*ml).nlevl[i+1];
  (*ml).nlevl[(*ml).lmax+1]=0;


  /* set levelhindx for each level */
  (*ml).levlhindx[0]=0;
  for (lvl=0; lvl<=(*ml).lmax; lvl++)
    {
      (*ml).levlhindx[lvl+1]=0;
      if (type==2) /* element hierarchy used for T2 meshes */
	{    
	  for (i=0; i<eh_nr; i++)
	    {
	      if ((*m).elhier[i*eh_w+l_mctxehlvl]==lvl-1)
		{
		  (*ml).levlhindx[lvl+1]=i+1;
		}
	    }
	}
      else if ((type==1)||(type==31)) /* hierarchy used for T1 and E1 meshes */
	{    
	  for (i=0; i<hi_nr; i++)
	    {
	      if ((*m).hier[i*hi_w+MCT1HILVL]==lvl-1)
		{
		  (*ml).levlhindx[lvl+1]=i+1;
		}
	    }
	}
    }
  return SUCCESS;
}




/*FUNCTION*/
void bpx_free( struct bpxdata *bpx
/* empties the data structure for the BPX preconditioner, such that it
   can be freed

   Output: bpx     - internal memory is released,
*/
	       ){
  free((*bpx).hvec);
  (*bpx).hvec=NULL;

  if ((*bpx).cmat!=NULL) 
    {
      coarse_mat_free((*bpx).cmat);
      free((*bpx).cmat);
      (*bpx).cmat=NULL;  
    }

  /* it is not upon us to delete the mesh, the ml, or the projector
     but we can forget about them */
  (*bpx).msh=NULL;
  (*bpx).ml=NULL;
  (*bpx).P=NULL;

  return;
}





/*FUNCTION*/
int bpx_init_tx( struct bpxdata *bpx, struct multilvl *ml
/* initialises the data structure for the BPX preconditioner
   
   Input:  msh     - the mesh to which the BPX data shall correspond
                     (the important bit is the hierarchy data)
           ml      - multilevel data struct

   Output: bpx     - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  /* allocate memory for the help vectors */
  TRY_MALLOC((*bpx).hvec,  (*ml).nlevl[0], double,
	     bpx_init);

  /* initialise the coarse grid as non-existend */
  (*bpx).cmat=NULL;

  /* the mesh, if needed, should be set elsewere */
  (*bpx).msh =NULL; 

  /* the multilevel struct */
  (*bpx).ml  =ml; 

  /* the projector, if needed, should be set elsewere */
  (*bpx).P =NULL; 

  return SUCCESS;
}



/*FUNCTION*/
void mg_free( struct mgdata *mg
/* empties the multilvl data structure, such that it can be freed

   Output: mg      - internal memory is released,
*/
	       ){
  (*mg).vx_nr=0; 
  (*mg).dim=0; 

  free((*mg).xl);
  (*mg).xl=NULL;

  free((*mg).dxl);
  (*mg).dxl=NULL;

  free((*mg).bl);
  (*mg).bl=NULL;

  free((*mg).sorterl);
  (*mg).sorterl=NULL;

  (*mg).bas_n=0;
  (*mg).chl_n=0;
  (*mg).chl_n_bi=0;

  free((*mg).phih);
  (*mg).phih=NULL;

  free((*mg).phibi);
  (*mg).phibi=NULL;

  free((*mg).done);
  (*mg).done=NULL;

  if ((*mg).invdiag != NULL)
    {
      free((*mg).invdiag);
      (*mg).invdiag=NULL;
    }
  /* it is not upon us to delete msh, ml, Ks, P, as we only have
     copies of each of them, but set them to NULL */
  (*mg).msh  = NULL;
  (*mg).ml   = NULL;
  (*mg).Ks   = NULL;
  (*mg).P    = NULL;

  if ((*mg).cmat != NULL)
    {
      coarse_mat_free((*mg).cmat);
      free((*mg).cmat);
    }
  (*mg).cmat = NULL;


  return;
}


/*FUNCTION*/
void mg_null_def( struct mgdata *mg
/* defines the multilvl data structure, such that a call ti mg_free
   prodruces no errors even though mg_init has not been called yet

   Output: mg      - is defined to be empty,
*/
	       ){
  (*mg).vx_nr=0; 
  (*mg).dim=0; 

  (*mg).xl=NULL;
  (*mg).dxl=NULL;
  (*mg).bl=NULL;
  (*mg).sorterl=NULL;

  (*mg).bas_n=0;
  (*mg).chl_n=0;
  (*mg).chl_n_bi=0;

  (*mg).phih=NULL;
  (*mg).phibi=NULL;		       
  (*mg).done=NULL;
  (*mg).invdiag=NULL;
  
  (*mg).msh  = NULL;
  (*mg).ml   = NULL;
  (*mg).Ks   = NULL;
  (*mg).P    = NULL;
  (*mg).cmat = NULL;

  (*mg).CGC_scale= 1.0;

  return;
}


#include "elements.h"
/*FUNCTION*/
int mg_init_tx( struct sparse *Ks, struct mesh *m, struct multilvl *ml,
		struct mgdata *mg, struct projector1 *P
/* initialises the data structure for the multigrid preconditioner
   
   Input:  Ks      - array of sparse matrices, K[i] is the stiffness
                     matrix for the mesh on the i-th level
           m       - the mesh to which the mgdata shall correspond
                     (the important bits are the hierarchy and element
		     hierarchy data)
	   ml      - multilevel data
	   P       - Projector for boundary conditions, give P=NULL if
	             not wanted

   Output: mg      - internal data is initialised, memory allocated
                     for the arrays

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		){
  FIDX i;
  int  err;
  FIDX vx_nr;
  FIDX lvl, lvl_vx;

  double *entry;
  double *invdiag;

  FIDX bas_n;
  double *phih, *phibi, *gradphi, *hessphi;
  double points[9*2]={ 0.25, 0.0,
		       0.25, 0.25,
		       0.0,  0.25,
		       0.75, 0.25,
		       0.5,  0.25,
		       0.75, 0.0,
		       0.0,  0.75,
		       0.25, 0.5,
 		       0.25, 0.75 };
  double points_bi[3*2]={ 0.25, 0.0,
		       0.25, 0.5,
		       0.75,  0.00 };
		       
  FIDX *done;
  mg_null_def(mg);

  vx_nr=(*m).vx_nr;
  (*mg).vx_nr=(*m).vx_nr;
  (*mg).dim=(*ml).dim;

  if ((*ml).type==2)
    {
      /* initialise interpolation info for bisection and uniform refinement */
      err= eval_basis( 2, tria, 2,
		       9, points,
		       &bas_n, &phih, &gradphi, &hessphi);
      FUNCTION_FAILURE_HANDLE( err, eval_basis, mg_init_tx);
      free( gradphi );
      free( hessphi );
      (*mg).phih=phih;
      (*mg).bas_n=bas_n;
      (*mg).chl_n=9;
      err= eval_basis( 2, tria, 2,
		       3, points_bi,
		       &bas_n, &phibi, &gradphi, &hessphi);
      FUNCTION_FAILURE_HANDLE( err, eval_basis, mg_init_tx);
      free( gradphi );
      free( hessphi );
      (*mg).phibi=phibi;
      (*mg).chl_n_bi=3; 
    }
  else
    {
      /* linear elements don't use the element hierarchy but the edge
	 hierarchy */
      (*mg).phih=NULL;
      (*mg).bas_n=0;
      (*mg).chl_n=0;
    }


  /* allocate mg.xl, mg.dxl, mg.bl */
  TRY_MALLOC((*mg).xl,  (*ml).nlevl[0], double, mg_init_tx);
  TRY_MALLOC((*mg).dxl, (*ml).nlevl[0], double, mg_init_tx);
  TRY_MALLOC((*mg).bl,  (*ml).nlevl[0], double, mg_init_tx);

  /* allocate mg.done */
  TRY_MALLOC((*mg).done, (*mg).dim*vx_nr, FIDX, mg_init_tx);
  done=(*mg).done;

  /* if the matrices are given, set invdiag */
  if (Ks != NULL)
    {
      /* allocate mg.invdiag */
      TRY_MALLOC((*mg).invdiag, (*ml).nlevl[0], double, mg_init_tx);
      invdiag=(*mg).invdiag;
      
      /* for each level, finest to coarsest */
      for (lvl=(*ml).lmax; lvl>=0; lvl--)
	{
	  lvl_vx=(*ml).nlevl[lvl]-(*ml).nlevl[lvl+1];
	  /* printf("debug lvl=%"dFIDX"  lvl_vx=%4d\n",  lvl,  lvl_vx); /* */
	  for (i=0; i<lvl_vx; i++)
	    {
	      err=sparse_get_entry( &Ks[lvl], i, i, 0, &entry);
	      FUNCTION_FAILURE_HANDLE(err, sparse_get_entry, mg_init_tx);
#ifdef DEBUGFEINS
	      if ((*entry==0.0)||(isnan(*entry)))
		{
		  fprintf(stderr, "mg_init_tx: error, "
			  "Ks[%"dFIDX"](%"dFIDX",%"dFIDX")=%e, "
			  "init invdiag fails",
			  lvl,i,i,*entry);
		  return FAIL;
		}
#endif
	      invdiag[i+(*ml).nlevl[lvl+1]]=1.0/(*entry); 
	    }
	  /* if the projector is given project invdiag */
	  if (P!=NULL)
	    {
	      for (i=0; i<(*P).len; i++)
		{
		  FIDX child,dof;
		  dof=(*P).V[i];
		  MLVLFINDENTRY(child, dof, lvl, *ml);
		  if (child >=0)
		    {
		      invdiag[child]=0.0;
		    }
		}
	    }
	}/* end of loop over levels */
    }

  (*mg).sorterl=NULL;

  (*mg).msh  = m;
  (*mg).ml   = ml;
  (*mg).Ks   = Ks;
  (*mg).P    = P;
  (*mg).cmat = NULL;

  (*mg).CGC_scale= 1.0;

  return SUCCESS;
}

/*FUNCTION*/
int mg_restrict_t2( struct mgdata *mg,
		    FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, 
	             the lvlin part is read and modified, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		){
  FIDX d, i, j, k;
  FIDX mldim, vx_nr, eh_w, eh_nr, nchildren;
  FIDX node, fath, fathl, fathlp1, chldlp1;
  FIDX lvl;

  FIDX bas_n, chl_n,chl_n_bi;
  double *phi;
  FIDX *done;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  vx_nr = (*mg).vx_nr;
  mldim = (*ml).dim;
  eh_w  = (*m).eh_w;
  eh_nr = (*m).eh_nr;
  bas_n = (*mg).bas_n;
  chl_n = (*mg).chl_n;
  chl_n_bi = (*mg).chl_n_bi;
  done  = (*mg).done;

  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr,
	      "mg_restrict_t2: mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout>lvlin)
    {
      fprintf(stderr,
	      "mg_restrict_t2: %"dFIDX"=lvlout>lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlin>(*ml).lmax)||(lvlout<0))
    {
      fprintf(stderr,
	      "mg_restrict_t2: lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range "
	      "(0<lvl<%"dFIDX"=maxl)\n",  lvlout,  lvlin,
	       (*ml).lmax);
      return FAIL;
    }
  
      
  /* now, for each lvl, lvlin-1 to lvlout */
  for (lvl=lvlin-1; lvl>=lvlout; lvl--)
    {
      /* mark all nodes as undone */
      for (i=0; i<mldim*vx_nr; i++) done[i]=0;

      /* compute mlv[lvl]= (level interpolation)^T * mlv[lvl+1] */

      for (i=(*ml).levlhindx[lvl+2]-1; i>=(*ml).levlhindx[lvl+1]; i--)
	{
#ifdef DEBUGFEINS
	  if ((*m).elhier[i*eh_w+MCT2EHLVL]!=lvl)
	    {
	      fprintf(stderr,
		      "mg_restrict_t2: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  /* Test if the element was refined by bisection or uniform 
    	       (bisection has only 3 children) */
	  if ((*m).elhier[i*eh_w+MCT2EHCHL1+3] >-1)
	    {
	      /* Uniform (9 child nodes) */
	      nchildren= chl_n;
	      phi  = (*mg).phih;
            }
	  else
	    {
	      /* Bisection (3 child nodes) */
	      nchildren = chl_n_bi;
	      phi  = (*mg).phibi;
	    }
	    
	  for (j=0; j<nchildren; j++)
	    for (d=0; d<mldim; d++)
	      {
		/* check out the child nodes */
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHCHL1+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(chldlp1, node, lvl+1, *ml);
		    for (k=0; k<bas_n; k++)
		      {
			/* father += alpha*child */
			fath=d*vx_nr
			  +(*m).elhier[i*eh_w+MCT2EHFAT1+k];
			/* we look up the father on the fine level
			   lvl+1, this avoids problems with fathers
			   being existing only on the fine level,
			   which may happen in adaptive refinement,
			   where one face may be split multiple times
			   during one logical refinement step,
			   
			   this implies that the input vector
			   has to be modified */
			MLVLFINDENTRY(fathl, fath, lvl+1, *ml);
			mlv[fathl]+=phi[j*bas_n+k]*mlv[chldlp1];
		      }
	      	  } /* end check out child nodes */
	      }   
	} /* (children done, 
	     new fine fathers = old fine fathers + fine children) */

	/*copy the father of the finer mesh to the coarser mesh */
	for (node=0; node<(*ml).levlvx[lvl]; node++)
	  for (d=0; d<mldim; d++)
	    {
	      /* copy them only if they are on the coarser mesh */  
	      MLVLFINDENTRY(fathl, node+d*vx_nr, lvl, *ml);
	      if (fathl>-1)
		{
		  MLVLFINDENTRY(fathlp1, node+d*vx_nr, lvl+1, *ml);
		  mlv[fathl]=mlv[fathlp1];
		}
	      else return FAIL;
	    }
    }/* end of loop over levels */

  return SUCCESS;
}


/*FUNCTION*/
int mg_interpolate_t2( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout >= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		){
  FIDX d, i, j, k;
  FIDX mldim, vx_nr, eh_w, eh_nr;
  FIDX node, fath, fathl, fathlm1, chldl;
  FIDX lvl;

  FIDX bas_n, chl_n, chl_n_bi, children;
  double *phi;
  FIDX *done;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  vx_nr = (*mg).vx_nr;
  mldim = (*ml).dim;
  eh_w  = (*m).eh_w;
  eh_nr = (*m).eh_nr;
  bas_n = (*mg).bas_n;
  chl_n = (*mg).chl_n;
  chl_n_bi = (*mg).chl_n_bi;
  done  = (*mg).done;


  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr,
	      "mg_interpolate_t2: mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout<lvlin)
    {
      fprintf(stderr,
	      "mg_interpolate_t2: %"dFIDX"=lvlout<lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlout>(*ml).lmax)||(lvlin<0))
    {
      fprintf(stderr, "mg_interpolate_t2: "
	      "lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range (0<lvl<%"dFIDX"=lmax)\n",
	       lvlout,  lvlin,  (*ml).lmax);
      return FAIL;
    }
 

    
  /* now, for each lvl, lvlin+1 to lvlout */
  for (lvl=lvlin+1; lvl<=lvlout; lvl++)
    {
      /* mark all nodes as undone */
      for (i=0; i<mldim*vx_nr; i++) done[i]=0;

     /*copy the father of the coarser mesh to the finer mesh */
     for (node=0; node<(*ml).levlvx[lvl-1]; node++)
	for (d=0; d<mldim; d++)
	  {
	    /* Copy them if they are on the coarser mesh */  
	    MLVLFINDENTRY(fathlm1, node+d*vx_nr, lvl-1, *ml);
	    if (fathlm1>-1)
	      {
		done[node]=1;
		MLVLFINDENTRY(fathl, node+d*vx_nr, lvl, *ml);
		mlv[fathl]=mlv[fathlm1];
	      }
	    else return FAIL;
	  }
      /* compute mlv[lvl]= (level interpolation) * mlv[lvl-1] */
      for (i=(*ml).levlhindx[lvl]; i<(*ml).levlhindx[lvl+1]; i++)
	{
#ifdef DEBUGFEINS
	  if ((*m).elhier[i*eh_w+MCT2EHLVL]!=lvl-1)
	    {
	      fprintf(stderr,
		      "mg_interpolate_t2: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  /* Test if the element was refined by bisection or uniform 
    	       (bisection has only 3 children) */

	  if ((*m).elhier[i*eh_w+MCT2EHCHL1+3] >-1)
	    {
	     /* Uniform (9 child nodes) */
	     children= chl_n;
	     phi  = (*mg).phih;
            }
	   else
	    {
	      /* Bisection (3 child nodes) */
	      children = chl_n_bi;
	      phi  = (*mg).phibi;
	    }	    
	  for (j=0; j<children; j++)
	    for (d=0; d<mldim; d++)
	      {
		/* check out the child nodes */
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHCHL1+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(chldl, node, lvl, *ml);
		    mlv[chldl]=0.0;
		    for (k=0; k<bas_n; k++)
		      {
			/* child += alpha*father */
			fath=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1+k];
			/* we look up the father on the fine level
			   lvl, this avoids problems with fathers
			   being existing only on the fine level,
			   which may happen in adaptive refinement,
			   where one face may be split multiple times
			   during one logical refinement step,

			   this is OK because the coarse dofs have
			   already been copied to the fine lvl */
			MLVLFINDENTRY(fathlm1, fath, lvl, *ml);
			mlv[chldl]+=phi[j*bas_n+k]*mlv[fathlm1];
		      }
		  }
		/* end check out child nodes */
	      }
	}
    }/* end of loop over levels */

  return SUCCESS;
}

/*FUNCTION*/
int mesh_refine_green_element_t2( struct mesh *m , FIDX el
/* refines the element el of the mesh m with the Baensch-Green
   refinement, all necessarry new parts are
   created thereby

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991

   Input:  el       - the element/face to be refined

   In/Out: m        - the mesh, is modified accordingly, e.g. new
                      elements, faces, edges, vertices, boundary
                      elements and hierarchy entries are created,
		      the element el and the face el are thereby
		      replaced by one of their children, the edges
		      remain to allow other elements to stay intact,
		      if necessary the fields of the mesh are
		      reallocated to have space for the new stuff

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, vx_w, fc_w, el_w, eg_w, ps_w, pv_w, hi_w, eh_w, elhi;
  int  err;


  FIDX newedge, oldedges[3], oldnodes[6], eg;
  /* nodedge[lnode][this nodes edge]=the edge (global number),
     nodineg[lnode][this nodes edge]=the node is 1st/2nd of the edge */
  FIDX cornernode, midnode;
  FIDX chld0, chld1, newface, newel, newnode,
	old1, old2, newhier;

  vx_w=(*m).vx_w;
  fc_w=(*m).fc_w;
  el_w=(*m).el_w;
  eg_w=(*m).eg_w;
  ps_w=(*m).ps_w;
  pv_w=(*m).pv_w;
  hi_w=(*m).hi_w;
  eh_w=(*m).eh_w;


  /* get all old nodes and edges */
  for (i=0; i<6; i++)
    {
      oldnodes[i]=(*m).elem[el*el_w+MCT2ELNOD1+i];
    }
  for (i=0; i<3; i++)
    {
      oldedges[i]=(*m).face[el*fc_w+MCT2FCEDG1+i];
    }

  /*the first edge is the refinement edge */ 

  eg=oldedges[0];
    
 if ( ((*m).edge[eg*eg_w+MCT2EGNOD1]!=oldnodes[0])&& 
	((*m).edge[eg*eg_w+MCT2EGNOD1+1]!=oldnodes[0]) )
    {
      fprintf(stderr, "internal mesh consistency error in "
        "mesh_refine_green_element_t2, file %s, line %d\n",
        __FILE__, __LINE__);
      return FAIL;
    } 
    
  /* make sure this edge is refined */
    if ((*m).edge[eg*eg_w+MCT2EGCHL1]==-1)
	  {
	    err=mesh_split_edge_t2( m, eg);
	    FUNCTION_FAILURE_HANDLE( err, mesh_split_edge_t2,
				     mesh_refine_green_element_t2); 
	  } 
	  
  /* make sure we have enough space for the new edge in the inner
     of the element, the new vertice and its hierarchy entire
     who come with the new edges, one new face (the old is recycled as
     a 2nd), one new element (the old is recycled as a 2nd), one new
     pcvol if needed (the old is recycled as a 2nd), one element
     hierarchy entry */ 
  if ((*m).eg_nr+1>(*m).eg_max)
    {
      err=mesh_more_edges( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_edges, mesh_refine_green_element_t2 );
    }
  if ((*m).vx_nr+1>(*m).vx_max)
    {
      err=mesh_more_vertices( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_vertices, mesh_refine_green_element_t2 );
    }
  if ((*m).hi_nr+1>(*m).hi_max)
    {
      err=mesh_more_hierarchy( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_hierarchy, mesh_refine_green_element_t2 );
    }
  if ((*m).fc_nr+1>(*m).fc_max)
    {
      err=mesh_more_faces( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_faces, mesh_refine_green_element_t2 );
    }
  if ((*m).el_nr+1>(*m).el_max)
    {
      err=mesh_more_elems( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elems, mesh_refine_green_element_t2 );
    }
  if (((*m).face[el*fc_w+MCT2FCPCVL]!=-1)&&((*m).pv_nr+3>(*m).pv_max))
    {
      err=mesh_more_pcvol( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_pcvol, mesh_refine_green_element_t2 );
    }
  if ((*m).eh_nr+1>(*m).eh_max)
    {
      err=mesh_more_elhiers( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_elhiers, mesh_refine_green_element_t2 );
    }
  if ((*m).st_nr+1>(*m).st_max)
    {
      err=mesh_more_st( m, 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_more_st, mesh_refine_green_element_t2 );
    }
    
  /* Find the cornernode which is opposite to the split edge */
    
  /* make sure that oldnodes[2] is the cornernode */  
  cornernode = oldnodes[2];
  if ( (cornernode==(*m).edge[eg*eg_w+ MCT2EGNOD1  ])||
            (cornernode==(*m).edge[eg*eg_w+ MCT2EGNOD1+1]) )
    { 
      fprintf(stderr, "internal mesh consistency error in "
	"mesh_refine_green_element_t2, file %s, line %d\n",
	__FILE__, __LINE__);
      return FAIL;  
    }	    


  /* get the child edges containig this node */
  chld0=(*m).edge[eg*eg_w+MCT2EGCHL1+0];
  chld1=(*m).edge[eg*eg_w+MCT2EGCHL1+1];

  /* the correct refinement level of the new edges */
  (*m).edge[chld0*eg_w + MCT2EGLVL]=(*m).lvl;
  (*m).edge[chld1*eg_w + MCT2EGLVL]=(*m).lvl;
  
  /* get the midnode of the splitedge */
  midnode=(*m).edge[eg*eg_w+MCT2EGNODM]; 

  /* create a new edge from the midnode to the cornernode*/
  newedge=(*m).eg_nr;
  (*m).eg_nr++;

  (*m).edge[newedge*eg_w+MCT2EGNOD1  ]=cornernode;
  (*m).edge[newedge*eg_w+MCT2EGNOD1+1]=midnode;
  (*m).edge[newedge*eg_w+MCT2EGCHL1  ]=-1;
  (*m).edge[newedge*eg_w+MCT2EGBND   ]=-1;
  (*m).edge[newedge*eg_w+MCT2EGPCSU  ]=-1;
  (*m).edge[newedge*eg_w+MCT2EGLVL   ]=(*m).lvl;

  /* create the midnode of the new edge */
  newnode=(*m).vx_nr;
  (*m).vx_nr++;
  (*m).st_nr++;
      
  old1= (*m).edge[newedge*eg_w + MCT2EGNOD1  ];
  old2= (*m).edge[newedge*eg_w + MCT2EGNOD1+1];

  (*m).vertex[newnode*vx_w+MCT2VXSTRT  ]= 
        0.5*( (*m).vertex[old1*vx_w+MCT2VXSTRT  ]
            + (*m).vertex[old2*vx_w+MCT2VXSTRT  ] );
  (*m).vertex[newnode*vx_w+MCT2VXSTRT+1]= 
        0.5*( (*m).vertex[old1*vx_w+MCT2VXSTRT+1]
	    + (*m).vertex[old2*vx_w+MCT2VXSTRT+1] );
  
  (*m).edge[newedge*eg_w + MCT2EGNODM]=newnode;

  /* the according hierarchy entry to the midnode */
  newhier=(*m).hi_nr;
  (*m).hi_nr++;

  (*m).hier[newhier*hi_w+ MCT2HICHLD  ]=newnode;
  (*m).hier[newhier*hi_w+ MCT2HIFAT1  ]=old1;
  (*m).hier[newhier*hi_w+ MCT2HIFAT1+1]=old2;
  (*m).hier[newhier*hi_w+ MCT2HILVL   ]=(*m).lvl;

  /* create a new face */
  newface=(*m).fc_nr;
  (*m).fc_nr++;

  /* Find the acording edges to the two new elements */
  i=(*m).edge[chld0*eg_w+MCT2EGNOD1];
  
  /* In case i is the midnode of the refinement edge */
  if ( i==(*m).edge[eg*eg_w+MCT2EGNODM] )
    {
      printf ("Warning mesh_refine_green_element_t2:\n"
	    "mesh_split_edge_t2 should order the first node as the exterior node\n");
      i=(*m).edge[chld0*eg_w+MCT2EGNOD1+1];
    }

  /* now see which of the two old edges belongs to which new element */
  if ( (i!=(*m).edge[oldedges[1]*eg_w+MCT2EGNOD1  ])&&
       (i!=(*m).edge[oldedges[1]*eg_w+MCT2EGNOD1+1])   )
    {
      /* edges 2 and 3 have to be swaped */
      j=oldedges[2];
      oldedges[2]=oldedges[1];
      oldedges[1]=j;
    }    
  else
    {
      fprintf(stderr, "internal mesh consistency error in "
        "mesh_refine_green_element_t2, file %s, line %d\n",
        __FILE__, __LINE__);
      return FAIL;
    }
    
  (*m).face[newface*fc_w+MCT2FCEDG1  ]=oldedges[1];
  (*m).face[newface*fc_w+MCT2FCEDG1+1]=chld0;
  (*m).face[newface*fc_w+MCT2FCEDG1+2]=newedge;

  (*m).face[newface*fc_w+MCT2FCRHSF]=(*m).face[el*fc_w+MCT2FCRHSF];

  /* if necessary create a new pcvol entry */
  if ((*m).face[el*fc_w+MCT2FCPCVL]!=-1)
    {
      FIDX pcv, new;

      pcv=(*m).face[el*fc_w+MCT2FCPCVL];
      new=(*m).pv_nr;
	  
      (*m).pcvol[new*pv_w+MCXXPVVOLM]=newface;
      (*m).face[newface*fc_w+MCT2FCPCVL]=new;
      (*m).pcvol[new*pv_w+MCXXPVCRIT]=(*m).pcvol[pcv*pv_w+MCXXPVCRIT];

      (*m).pv_nr++;
    }
  else
    {
      (*m).face[newface*fc_w+MCT2FCPCVL]=-1;
    }

  /* check if the edges of the new face are pcsurf edges, if so,
    let them know the name of the new face */
  for (j=0; j<3; j++)
    {
      FIDX eg, pcsu;
      eg=(*m).face[newface*fc_w+MCT2FCEDG1+j];
      pcsu=(*m).edge[eg*eg_w+MCT2EGPCSU];
      if (pcsu!=-1)
	{
	  if ((*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]==-1)
	    {
	      (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE]=newface;
	      (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=-1;
	    }
	  else
	    {
	      (*m).pcsurf[pcsu*ps_w+MCXXPSVOLE+1]=newface;
	    }
	}
    }

  /* create the new elhier */
  elhi=(*m).eh_nr;
  (*m).eh_nr++;
  /* level = level of the new child edges +1 (=lvl of edge hierarchy entry */
  (*m).elhier[elhi*eh_w+MCT2EHLVL]= (*m).lvl; 
    
  /* create a new element */
  newel=(*m).el_nr;
  (*m).el_nr++;

  err=mesh_t2_get_elem_from_face( m, newel);
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_get_elem_from_face,
			       mesh_refine_green_element_t2 );

  /* lets overwrite the old element with the second one */
  (*m).face[el*fc_w+MCT2FCEDG1  ]=oldedges[2];
  (*m).face[el*fc_w+MCT2FCEDG1+1]=newedge;
  (*m).face[el*fc_w+MCT2FCEDG1+2]=chld1;

  /* the element */
  err=mesh_t2_get_elem_from_face( m, el);
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_get_elem_from_face,
			   mesh_refine_green_element_t2 );
    
  /* define the new elhier */
  /* fathers */
  for (i=0; i<6; i++)
    {
      (*m).elhier[elhi*eh_w+MCT2EHFAT1+i]=
	oldnodes[i];
    }
	
  /* children */
  i=oldnodes[0];
  if ( (i!=(*m).edge[chld0*eg_w+MCT2EGNOD1  ])&&
       (i!=(*m).edge[chld0*eg_w+MCT2EGNOD1+1])   )
    {
      if ( (i!=(*m).edge[chld1*eg_w+MCT2EGNOD1  ])&&
	   (i!=(*m).edge[chld1*eg_w+MCT2EGNOD1+1])   )
	{
	  printf("Error in mesh_refine_green_element_t2: children don't fit\n");
	  return FAIL;
	}  

      /* the childrens need to be swaped */
      (*m).elhier[elhi*eh_w+MCT2EHCHL1  ]=
	(*m).edge[chld1*eg_w+MCT2EGNODM]; 
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+1]=
	(*m).edge[newedge*eg_w+MCT2EGNODM];    
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+2]=
	(*m).edge[chld0*eg_w+MCT2EGNODM];
    }   
  else
    {	  
      (*m).elhier[elhi*eh_w+MCT2EHCHL1  ]=
	(*m).edge[chld0*eg_w+MCT2EGNODM]; 
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+1]=
	(*m).edge[newedge*eg_w+MCT2EGNODM];    
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+2]=
	(*m).edge[chld1*eg_w+MCT2EGNODM];
    }
  for (i=3;i<9;i++)
    {
      (*m).elhier[elhi*eh_w+MCT2EHCHL1+i]=-1;
    }	    
  /* that's it */

  return SUCCESS;
}




/*FUNCTION*/
int mesh_refine_adaptive_t2( struct mesh *m, FIDX *marked_elem
/* refines all marked elements of the mesh m with the
   Baensch-Green refinement, all necessarry new parts are
   created thereby,

   method according to E. B\"ansch, Local Mesh Refinement in 2 and 3
   Dimensions, Impact of Computing in Science and Engineering, Vol. 3,
   pp. 181--191, 1991


   In/Out: m           - the mesh, is modified accordingly, e.g. new
                         elements, faces, edges, vertices, boundary
                         elements and hierarchy entries are created,
  		         the element el and the face el are thereby
		         replaced by one of their children, the edges
		         remain to allow other elements to stay intact,
		         if necessary the fields of the mesh are
		         reallocated to have space for the new stuff
	   marked_elem - Vector holding which elements are to be
	                 refined (1 = to be refined)
   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
                         valid
*/
		     ){
  FIDX old_el_nr, new_el_nr, eg_w, fc_w, i;
  int  err, refine;
  fc_w=(*m).fc_w;
  eg_w=(*m).eg_w;
  
  old_el_nr = (*m).el_nr;
  (*m).lvl++;
  /* first loop over all elements, refine all marked elements */
  for (i=0; i< old_el_nr; i++)
    {
      if ( marked_elem[i]==1)      
        {
          err=mesh_refine_green_element_t2( m , i);
          FUNCTION_FAILURE_HANDLE( err, 
		mesh_refine_green_element_t2, mesh_refine_adaptive_t2 );
        }
    }
  new_el_nr = (*m).el_nr;
  /* we want to refine the initially marked elements twice,
    that is into four elements rather than just splitted (to increase element growth) */  
  /* second loop over all elemtents again andall new elements */ 
  if(0==1)
    {
      for (i=0; i< old_el_nr; i++)
        {
	  if ( marked_elem[i]==1)      
            {
              err=mesh_refine_green_element_t2( m , i);
              FUNCTION_FAILURE_HANDLE( err, 
		mesh_refine_green_element_t2, mesh_refine_adaptive_t2 );
            }
	}
      for (i=old_el_nr; i< new_el_nr; i++)
	{
	  err=mesh_refine_green_element_t2( m , i);
	  FUNCTION_FAILURE_HANDLE( err, 
	    mesh_refine_green_element_t2, mesh_refine_adaptive_t2 );
	}		
    }
  /* loop over all elements, refine all elements with hanging nodes 
     while there can be hanging nodes (only after a refinement) */

  old_el_nr=(*m).el_nr;
  refine=1;
  
  while (refine!=0)
    {
      refine=0;
      old_el_nr=(*m).el_nr;

      for (i=0; i< old_el_nr; i++)
       {
          /* test if element has hanging nodes 
             (if and only if an edge was already splitted) */           
      if ( ((*m).edge[((*m).face[i*fc_w+MCT2FCEDG1+0])*eg_w+MCT2EGCHL1]!=-1)||
	   ((*m).edge[((*m).face[i*fc_w+MCT2FCEDG1+1])*eg_w+MCT2EGCHL1]!=-1)||
           ((*m).edge[((*m).face[i*fc_w+MCT2FCEDG1+2])*eg_w+MCT2EGCHL1]!=-1) )
	    {
              err=mesh_refine_green_element_t2( m , i);
              FUNCTION_FAILURE_HANDLE( err, mesh_refine_green_element_t2, 
			       mesh_refine_adaptive_t2 );
              refine=1;
            } 
       }
    }
#
  
  /* Refinement is done, there are no hanging nodes anymore */  
    
  return SUCCESS;
}














/*FUNCTION*/
int mg_restrict_t1( struct mgdata *mg, FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector,
	             the lvlin part is read and modified, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		    ){
  FIDX i, j, d, lvl, child, fath1, fath2;
  FIDX hi_w, mldim, vx_nr;

  FIDX l_mctxhilvl, l_mctxhichld,  l_mctxhifat1;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  mldim = (*ml).dim;
  vx_nr = (*mg).vx_nr;

  hi_w  = (*m).hi_w;

  switch ((*ml).type)
    {
    case 1: /* T1 mesh */
      l_mctxhilvl  = MCT1HILVL;
      l_mctxhichld = MCT1HICHLD;
      l_mctxhifat1 = MCT1HIFAT1;
      break;
    case 31: /* E1 mesh */
      l_mctxhilvl  = MCE1HILVL;
      l_mctxhichld = MCE1HICHLD;
      l_mctxhifat1 = MCE1HIFAT1;
      break;
    default:
      fprintf(stderr,"mg_restrict_t1: unknown ml.type=%"dFIDX"\n",
	      (*ml).type);
      return FAIL;
    }
  
  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr,
	      "mg_restrict_t1: mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout>lvlin)
    {
      fprintf(stderr,
	      "mg_restrict_t1: %"dFIDX"=lvlout>lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlin>(*ml).lmax)||(lvlout<0))
    {
      fprintf(stderr,
	      "mg_restrict_t1: lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range "
	      "(0<lvl<%"dFIDX"=maxl)\n",  lvlout,  lvlin, 
	       (*ml).lmax);
      return FAIL;
    }

  /* now, for each lvl, lvlin-1 to lvlout */
  for (lvl=lvlin-1; lvl>=lvlout; lvl--)
    {
      FIDX l_vx_nr=((*ml).nlevl[lvl+1]-(*ml).nlevl[lvl+2])/mldim;
      
      /* compute mlv[lvl]= (level interpolation)^T * mlv[lvl+1]
	 modifications on fine level first
       */
      for (i=(*ml).levlhindx[lvl+2]-1; i>=(*ml).levlhindx[lvl+1]; i--)
	{
#ifdef DEBUGFEINS
	  if ((*m).hier[i*hi_w+l_mctxhilvl]!=lvl)
	    {
	      fprintf(stderr,
		      "mg_restrict_t1: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif

	  for (d=0; d<mldim; d++)
	    {
	      child=(*m).hier[i*hi_w+ l_mctxhichld  ]+d*vx_nr;
	      fath1=(*m).hier[i*hi_w+ l_mctxhifat1  ]+d*vx_nr;
	      fath2=(*m).hier[i*hi_w+ l_mctxhifat1+1]+d*vx_nr;

	      /* we look up the father on the fine level
		 lvl+1, this avoids problems with fathers
		 being existing only on the fine level,
		 which may happen in adaptive refinement,
		 where one face may be split multiple times
		 during one logical refinement step,
		 
		 this implies that the input vector
		 has to be modified */
	      MLVLFINDENTRY(fath1, fath1, lvl+1, *ml);
	      MLVLFINDENTRY(fath2, fath2, lvl+1, *ml);
	      MLVLFINDENTRY(child, child, lvl+1,   *ml);

	      /* the interpolation part */
	      mlv[fath1] += 0.5*mlv[child];
	      mlv[fath2] += 0.5*mlv[child]; 
	    }
	} /* end loop hierarchy entries */

      /* copy the father nodes */
      for (j=0; j<l_vx_nr; j++)
	for (d=0; d<mldim; d++)
	  {
	    /* get the entry number of the coarser level */
	    child = j+d*vx_nr;
	    MLVLFINDENTRY(fath2, child, lvl, *ml);
	    if (fath2>=0) 
	      {
		MLVLFINDENTRY(child, child, lvl+1, *ml);
		/* the I (identity) part, r_l = r_(l+1) */
		mlv[fath2]=mlv[child];
	      }
	  }


    } /* end loop levels */
  
  return SUCCESS;
}


/*FUNCTION*/
int mg_interpolate_t1( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		       ){
  FIDX i, j, d, lvl, child, fath1, fath2;
  FIDX hi_w, vx_nr, mldim;

  FIDX l_mctxhilvl, l_mctxhichld,  l_mctxhifat1;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  mldim = (*ml).dim;
  vx_nr = (*mg).vx_nr;

  hi_w  = (*m).hi_w;

  switch ((*ml).type)
    {
    case 1: /* T1 mesh */
      l_mctxhilvl  = MCT1HILVL;
      l_mctxhichld = MCT1HICHLD;
      l_mctxhifat1 = MCT1HIFAT1;
      break;
    case 31: /* E1 mesh */
      l_mctxhilvl  = MCE1HILVL;
      l_mctxhichld = MCE1HICHLD;
      l_mctxhifat1 = MCE1HIFAT1;
      break;
    default:
      fprintf(stderr,"mg_interpolate_t1: unknown ml.type=%"dFIDX"\n",
	      (*ml).type);
      return FAIL;
    }

  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr, "mg_interpolate_t1: "
	      "mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout<lvlin)
    {
      fprintf(stderr, "mg_interpolate_t1: "
	      "%"dFIDX"=lvlout<lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlout>(*ml).lmax)||(lvlin<0))
    {
      fprintf(stderr, "mg_interpolate_t1: "
	      "lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range (0<lvl<%"dFIDX"=lmax)\n",
	       lvlout,  lvlin,  (*ml).lmax);
      return FAIL;
    }
  
  /* now, for each lvl, lvlin+1 to lvlout */
  for (lvl=lvlin+1; lvl<=lvlout; lvl++)
    {
      FIDX lm1_vx_nr=((*ml).nlevl[lvl-1]-(*ml).nlevl[lvl])/mldim;
      
      /* initialise the new (finer) level */
      /*copy the father of the coarser mesh to the finer mesh */
      for (j=0; j<lm1_vx_nr; j++)
	for (d=0; d<mldim; d++)
	  {
	    /* get the entry number of the coarser level */
	    fath2 = j+d*vx_nr;
	    MLVLFINDENTRY(child, fath2, lvl, *ml);
	    MLVLFINDENTRY(fath2, fath2, lvl-1, *ml);
	    /* the I (identity) part */
	    mlv[child]=mlv[fath2];
	  }

      /* compute mlv[lvl]= (level interpolation) * mlv[lvl-1] */
      for (i=(*ml).levlhindx[lvl]; i<(*ml).levlhindx[lvl+1]; i++)
	{
#ifdef DEBUGFEINS
	  if ((*m).hier[i*hi_w+l_mctxhilvl]!=lvl-1)
	    {
	      fprintf(stderr,
		      "mg_interpolate_t1: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  for (d=0; d<mldim; d++)
	    {
	      child=(*m).hier[i*hi_w+l_mctxhichld  ]+d*vx_nr;
	      fath1=(*m).hier[i*hi_w+l_mctxhifat1  ]+d*vx_nr;
	      fath2=(*m).hier[i*hi_w+l_mctxhifat1+1]+d*vx_nr;

	      /* we look up the father on the fine level
		 lvl, this avoids problems with fathers
		 being existing only on the fine level,
		 which may happen in adaptive refinement,
		 where one face may be split multiple times
		 during one logical refinement step,
		 
		 this is OK because the coarse dofs have
		 already been copied to the fine lvl */
      	      MLVLFINDENTRY(fath1, fath1, lvl, *ml);
	      MLVLFINDENTRY(fath2, fath2, lvl, *ml);
	      MLVLFINDENTRY(child, child, lvl, *ml);

	      /* the interpolation part */
	      mlv[child] =0.5*( mlv[fath1] + mlv[fath2] );
	    }
	} /* end loop hierarchy entries */
    } /* end loop levels */
  
  return SUCCESS;
}






/*FUNCTION*/
int mg_restrict_tx( struct mgdata *mg, FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation
   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		    ){
  int err;

  if (((*(*mg).ml).type == 1)||((*(*mg).ml).type == 31))
    {
      /* T1 and E1 meshes are handled the same way */
      err = mg_restrict_t1( mg, lvlin, lvlout, mlv);
      FUNCTION_FAILURE_HANDLE(err, mg_restrict_t1, mg_restrict_tx);
    }
  else if ( (*(*mg).ml).type == 2 )
    {
      err = mg_restrict_t2( mg, lvlin, lvlout, mlv);
      FUNCTION_FAILURE_HANDLE(err, mg_restrict_t2, mg_restrict_tx);
    }
  else
    {
      fprintf(stderr,"mg_restrict_tx: unknown multilevel type!\n");
      return FAIL;
    }

  return SUCCESS;
}


/*FUNCTION*/
int mg_interpolate_tx( struct mgdata *mg, FIDX lvlin, FIDX lvlout,
		       double *mlv
/* interpolates the vector lvlin part of mlv to lvlout
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout >= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the interpolateion from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		    ){
  int err;

  if (((*(*mg).ml).type == 1)||((*(*mg).ml).type == 31))
    {
      /* T1 and E1 meshes are handled the same way */
      err = mg_interpolate_t1( mg, lvlin, lvlout, mlv);
      FUNCTION_FAILURE_HANDLE(err, mg_interpolate_t1, mg_interpolate_tx);
    }
  else if ( (*(*mg).ml).type == 2 )
    {
      err = mg_interpolate_t2( mg, lvlin, lvlout, mlv);
      FUNCTION_FAILURE_HANDLE(err, mg_interpolate_t2, mg_interpolate_tx);
    }
  else
    {
      fprintf(stderr,"mg_interpolate_tx: unknown multilevel type!\n");
      return FAIL;
    }

  return SUCCESS;
}






/****************************************************************************/












/*FUNCTION*/
int mg_restrict_t1_t2( struct mgdata *mg,
		       FIDX lvlin, FIDX lvlout, double *mlv
/* restricts the vector lvlin part of mlv to lvlout, by applying the
   transpose operation of interpolation,
   using the T1 hiriarchy of the T1 mesh associated with the T2 mesh

   
   Input:  lvlin   - the level from which the restriction shall start
	   lvlout  - the level to which the restriction shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin>lvl>=lvlout are
	             overwritten by the restriction from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		){
  FIDX d, i, j, k;
  FIDX dim, vx_nr, eh_w, eh_nr;
  FIDX node, fath, fathl, fathlp1, chldlp1;
  FIDX lvl;

  FIDX *done;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  vx_nr = (*mg).vx_nr;
  dim   = (*mg).dim;
  eh_w  = (*m).eh_w;
  eh_nr = (*m).eh_nr;
  done  = (*mg).done;

  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr, "mg_restrict_t1_t2: "
	      "mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout>lvlin)
    {
      fprintf(stderr,
	      "mg_restrict_t1_t2: %"dFIDX"=lvlout>lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlin>(*ml).lmax)||(lvlout<0))
    {
      fprintf(stderr, "mg_restrict_t1_t2: "
	      "lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range (0<lvl<%"dFIDX"=maxl)\n",
	       lvlout,  lvlin,  (*ml).lmax);
      return FAIL;
    }
  
      
  /* now, for each lvl, lvlin-1 to lvlout */
  for (lvl=lvlin-1; lvl>=lvlout; lvl--)
    {
      /* mark all nodes as undone */
      for (i=0; i<dim*vx_nr; i++) done[i]=0;

      /* compute mlv[lvl]= (level interpolation)^T * mlv[lvl+1] */
      /* copy the father nodes */
      for (i=(*ml).levlhindx[lvl+1]; i<(*ml).levlhindx[lvl+2]; i++)
	{
#ifdef DEBUGFEINS
	  if ((*m).elhier[i*eh_w+MCT2EHLVL]!=lvl)
	    {
	      fprintf(stderr,
		      "mg_restrict_t2_t1: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  for (j=0; j<3; j++)
	    for (d=0; d<dim; d++)
	      {
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(fathl, node, lvl, *ml);
		    MLVLFINDENTRY(fathlp1, node, lvl+1, *ml);
		    /* printf("node=%3d   fathl=%3d    fathlp1=%3d\n",
		       node, fathl, fathlp1); /* */
		    mlv[fathl]=mlv[fathlp1];
		  }
	      }
	}
      for (i=(*ml).levlhindx[lvl+1]; i<(*ml).levlhindx[lvl+2]; i++)
	{
#ifdef DEBUGFEINS
	  if ((*m).elhier[i*eh_w+MCT2EHLVL]!=lvl)
	    {
	      fprintf(stderr,
		      "mg_restrict_t2_t1: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  for (j=0; j<3; j++)
	    for (d=0; d<dim; d++)
	      {
		/* check out the child nodes */
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1+3+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(chldlp1, node, lvl+1, *ml);
		    for (k=0; k<2; k++)
		      {
			/* father += alpha*child */
			fath=d*vx_nr
			  +(*m).elhier[i*eh_w+MCT2EHFAT1+
				       (j+k)%3];
			MLVLFINDENTRY(fathl, fath, lvl, *ml);
			mlv[fathl]+= 0.5*mlv[chldlp1];
		      }
		  }
		/* end check out child nodes */
	      }
	}
    }/* end of loop over levels */

  return SUCCESS;
}


/*FUNCTION*/
int mg_interpolate_t1_t2( struct mgdata *mg,
			  FIDX lvlin, FIDX lvlout, double *mlv
/* interpolates the vector lvlin part of mlv to lvlout, 
   using the T1 hiriarchy of the T1 mesh associated with the T2 mesh
   
   Input:  lvlin   - the level from which the interpolation shall start
	   lvlout  - the level to which the interpolation shall take
	             place, lvlout <= lvlin


   In/Out: mg      - multigrid data, the done part is used and therby
                     overwritten, other parts may be used read only
                     (phih)
	   mlv     - a multilevel vector, the lvlin part is read, and
	             all levels lvl with lvlin<lvl<=lvlout are
	             overwritten by the interpolation from lvlin

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		){
  FIDX d, i, j, k;
  FIDX dim, vx_nr, eh_w, eh_nr;
  FIDX node, fath, fathl, fathlm1, chldl;
  FIDX lvl;

  FIDX *done;

  struct mesh *m;
  struct multilvl *ml;
  
  m     = (*mg).msh;
  ml    = (*mg).ml;

  vx_nr = (*mg).vx_nr;
  dim   = (*mg).dim;
  eh_w  = (*m).eh_w;
  eh_nr = (*m).eh_nr;
  done  = (*mg).done;


  if (vx_nr!=(*m).vx_nr)
    {
      fprintf(stderr, "mg_interpolate_t1_t2: "
	      "mesh doesn't match mgdata (vx_nr test)!\n");
      return FAIL;
    }

  if (lvlout<lvlin)
    {
      fprintf(stderr, "mg_interpolate_t1_t2: "
	      "%"dFIDX"=lvlout<lvlin=%"dFIDX", makes no sense!\n",
	       lvlout,  lvlin);
      return FAIL;
    }

  if ((lvlout>(*ml).lmax)||(lvlin<0))
    {
      fprintf(stderr, "mg_interpolate_t1_t2: "
	      "lvlout=%"dFIDX" or lvlin=%"dFIDX", out of range (0<lvl<%"dFIDX"=lmax)\n",
	       lvlout,  lvlin,  (*ml).lmax);
      return FAIL;
    }
  
  /* now, for each lvl, lvlin+1 to lvlout */
  for (lvl=lvlin+1; lvl<=lvlout; lvl++)
    {
      /* mark all nodes as undone */
      for (i=0; i<dim*vx_nr; i++) done[i]=0;

      /* compute mlv[lvl]= (level interpolation) * mlv[lvl-1] */
      for (i=(*ml).levlhindx[lvl]; i<(*ml).levlhindx[lvl+1]; i++)
	{
#ifdef DEBUGFEINS
	  if ((*m).elhier[i*eh_w+MCT2EHLVL]!=lvl-1)
	    {
	      fprintf(stderr,
		      "mg_interpolate_t1_t2: ml.levlhindx wrong?\n");
	      return FAIL;
	    }
#endif
	  /* copy the father nodes */
	  for (j=0; j<3; j++)
	    for (d=0; d<dim; d++)
	      {
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(fathl, node, lvl, *ml);
		    MLVLFINDENTRY(fathlm1, node, lvl-1, *ml);
		    mlv[fathl]=mlv[fathlm1];
		  }
	      }
	  for (j=0; j<3; j++)
	    for (d=0; d<dim; d++)
	      {
		/* check out the child nodes */
		node=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1+3+j];
		if (done[node]==0)
		  {
		    done[node]=1;
		    MLVLFINDENTRY(chldl, node, lvl, *ml);
		    mlv[chldl]=0.0;
		    for (k=0; k<2; k++)
		      {
			/* father += alpha*child */
			fath=d*vx_nr+(*m).elhier[i*eh_w+MCT2EHFAT1
						 +(j+k)%3];
			MLVLFINDENTRY(fathlm1, fath, lvl-1, *ml);
			mlv[chldl]+= 0.5 * mlv[fathlm1];
		      }
		  }
		/* end check out child nodes */
	      }
	}
    }/* end of loop over levels */

  return SUCCESS;
}














/****************************************************************************/





















/*FUNCTION*/
int mesh_write_solution_exp_t2( struct mesh *m, struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *name
/* writes the mesh and the solution into a NAG IRIS EXPLORER readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes,
                       usually only input, but if (xtype==2) then the
                       pressure values are corrected, such that the
		       edge-mid-nodes have propper values
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat
           
   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  int err;
  struct mesh m_t1;

  if ((*x).len!=ncompo*(*m).vx_nr)
    {
      /* cry */
      fprintf(stderr, "mesh_write_solution_exp_t2: "
	      "size of x doesn't match ncompo=%"dFIDX" and vx_nr=%"dFIDX"!\n",
	       ncompo,  (*m).vx_nr );
      return FAIL;
    }

  mesh_init(& m_t1);

  err=mesh_t2_to_t1( m, &m_t1 );
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_to_t1,
			   mesh_write_solution_exp_t2);

  err=mesh_write_solution_exp_t1( &m_t1, x, ncompo, namlen, name );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t1,
			   mesh_write_solution_exp_t2);

  mesh_free( &m_t1 );

  return SUCCESS;
}





/*FUNCTION*/
int mesh_write_solution_femplot_t2( struct mesh *m, struct vector *x,
				    FIDX ncompo,
				    FIDX namlen, char *name
/* writes the mesh and the solution into a ?FEMplot? readable
   file for visualisation 
   FEMplot by project students (Lars Eiserbeck und Felix Prehl, 2007)
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes,
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat
           
   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  int err;
  struct mesh m_t1;

  if ((*x).len!=ncompo*(*m).vx_nr)
    {
      /* cry */
      fprintf(stderr, "mesh_write_solution_femplot_t2: "
	      "size of x doesn't match ncompo=%"dFIDX" and vx_nr=%"dFIDX"!\n",
	       ncompo,  (*m).vx_nr );
      return FAIL;
    }

  mesh_init(& m_t1);

  err=mesh_t2_to_t1( m, &m_t1 );
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_to_t1,
			   mesh_write_solution_femplot_t2);

  err=mesh_write_solution_femplot_t1( &m_t1, x, ncompo, namlen, name );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_femplot_t1,
			   mesh_write_solution_femplot_t2);

  mesh_free( &m_t1 );

  return SUCCESS;
}





/*FUNCTION*/
int mesh_write_solution_vtk_t2( struct mesh *m,
				struct vector *vec_x, FIDX vcompo,
				struct vector *scalar_x, FIDX scompo,
				FIDX namlen, char *name
/* converts mesh to type t1 and uses mesh_write_solution_vtk_t1, see
   there for description,
   mesh m has to be of type t2
*/
				){
  int err;
  struct mesh m_t1;

  mesh_init(& m_t1);

  err=mesh_t2_to_t1( m, &m_t1 );
  FUNCTION_FAILURE_HANDLE( err, mesh_t2_to_t1,
			   mesh_write_solution_vtk_t2);

  err=mesh_write_solution_vtk_t1( &m_t1, vec_x, vcompo, scalar_x, scompo, 
				  namlen, name );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t1,
			   mesh_write_solution_vtk_t2);

  mesh_free( &m_t1 );

  return SUCCESS;
}






/*FUNCTION*/
int mesh_t1_to_t2( struct mesh *m1, struct mesh *m2
/* creates the T2 mesh by converting the T1 mesh m1 into a T2 mesh,
   Input:  m1         - T1 mesh
   Output: m2         - T2 mesh, (empty mesh given via reference,
                        interior is set up)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX somany, i, j, len, insane;
  int  err;

  FIDX vx_w1, el_w1, eg_w1, fc_w1, hi_w1, bd_w1,
    ps_w1, pv_w1, pc_w1, fu_w1, pa_w1, sg_w1, sp_w1;

  const FIDX   vx_w2 = MCT2VXLN, eg_w2 = MCT2EGLN, 
    fc_w2 = MCT2FCLN, el_w2 = MCT2ELLN, vo_w2 = MCT2VOLN,
    eh_w2 = MCT2EHLN, hi_w2 = MCT2HILN, 
    bd_w2 = MCT2BDLN, ps_w2 = MCXXPSLN, pv_w2 = MCXXPVLN,
    pc_w2 = MCXXPCLN, fu_w2 = MC2XFULN, pa_w2 = MC2XPALN,
    sg_w2 = MC2XSGLN, sp_w2 = MC2XSPLN;

  /* set dimension */
  (*m2).dim = (*m1).dim;

  /* set width of fields */
  (*m2).vx_w = vx_w2;
  (*m2).el_w = el_w2;
  (*m2).eg_w = eg_w2;
  (*m2).fc_w = fc_w2;
  (*m2).vo_w = vo_w2;
  (*m2).hi_w = hi_w2;
  (*m2).eh_w = eh_w2;
  (*m2).bd_w = bd_w2;
  (*m2).ps_w = ps_w2;
  (*m2).pv_w = pv_w2;
  (*m2).pc_w = pc_w2;
  (*m2).fu_w = fu_w2;
  (*m2).pa_w = pa_w2;
  (*m2).sg_w = sg_w2;
  (*m2).sp_w = sp_w2;

  vx_w1 = (*m1).vx_w; el_w1 = (*m1).el_w; eg_w1 = (*m1).eg_w;
  fc_w1 = (*m1).fc_w; hi_w1 = (*m1).hi_w; bd_w1 = (*m1).bd_w;
  ps_w1 = (*m1).ps_w; pv_w1 = (*m1).pv_w; pc_w1 = (*m1).pc_w;
  fu_w1 = (*m1).fu_w; pa_w1 = (*m1).pa_w; sg_w1 = (*m1).sg_w;
  sp_w1 = (*m1).sp_w;

  /* initialise arrays */
  somany=(*m1).vx_nr+(*m1).eg_nr+100;
  TRY_MALLOC( (*m2).vertex, (*m2).vx_w * somany , double,
	      mesh_t1_to_t2 );
  (*m2).vx_max=somany;
  (*m2).vx_nr=0;

  somany=(*m1).el_nr+100;
  TRY_MALLOC( (*m2).elem, (*m2).el_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).el_max=somany;
  (*m2).el_nr=0;

  somany=(*m1).eg_nr+100;
  TRY_MALLOC( (*m2).edge, (*m2).eg_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).eg_max=somany;
  (*m2).eg_nr=0;

  somany=(*m1).fc_nr+100;
  TRY_MALLOC( (*m2).face, (*m2).fc_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).fc_max=somany;
  (*m2).fc_nr=0;

  (*m2).vols= NULL;
  (*m2).vo_max=0;
  (*m2).vo_nr=0;

  somany=(*m1).hi_nr+(*m1).eg_nr+100;
  TRY_MALLOC( (*m2).hier, (*m2).hi_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).hi_max=somany;
  (*m2).hi_nr=0;
  
  (*m2).lvl=(*m1).lvl;

  somany=(*m1).fc_nr;
  TRY_MALLOC( (*m2).elhier, (*m2).eh_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).eh_max=somany;
  (*m2).eh_nr=0;

  somany=(*m1).bd_nr+100;
  TRY_MALLOC( (*m2).bound, (*m2).bd_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).bd_max=somany;
  (*m2).bd_nr=0;

  somany=(*m1).ps_nr+100;
  TRY_MALLOC( (*m2).pcsurf, (*m2).ps_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).ps_max=somany;
  (*m2).ps_nr=0;

  somany=(*m1).pv_nr+100;
  TRY_MALLOC( (*m2).pcvol, (*m2).pv_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).pv_max=somany;
  (*m2).pv_nr=0;

  somany=(*m1).pc_nr+100;
  TRY_MALLOC( (*m2).pccrit, (*m2).pc_w * somany , double,
	      mesh_t1_to_t2 );
  (*m2).pc_max=somany;
  (*m2).pc_nr=0;

  somany=(*m1).fu_nr+8;
  TRY_MALLOC( (*m2).func, (*m2).fu_w * somany , double,
	      mesh_t1_to_t2 );
  (*m2).fu_max=somany;
  (*m2).fu_nr=0;

  somany=(*m1).pa_nr;
  TRY_MALLOC( (*m2).para, (*m2).pa_w * somany , double,
	      mesh_t1_to_t2 );
  (*m2).pa_max=somany;
  (*m2).pa_nr=0;

  somany=(*m1).sg_nr;
  TRY_MALLOC( (*m2).sseg, (*m2).sg_w * somany , FIDX,
	      mesh_t1_to_t2 );
  (*m2).sg_max=somany;
  (*m2).sg_nr=0;

  somany=(*m1).sp_nr;
  TRY_MALLOC( (*m2).spar, (*m2).sp_w * somany , double,
	      mesh_t1_to_t2 );
  (*m2).sp_max=somany;
  (*m2).sp_nr=0;
  
  somany=(*m2).vx_max;
  TRY_MALLOC( (*m2).st, somany , double,
	      mesh_t1_to_t2 );
  (*m2).st_max=somany;
  (*m2).st_nr=0;


  /* for debuging purposes, initialise the arrays with insane data */
  insane=-1523123;

#ifdef DEBUGFEINS
  len=(*m2).vx_w*(*m2).vx_max;
  for (i=0;i<len; i++)
    (*m2).vertex[i]=(double)insane;

  len=(*m2).el_w*(*m2).el_max;
  for (i=0;i<len; i++)
    (*m2).elem[i]=insane;

  len=(*m2).eg_w*(*m2).eg_max;
  for (i=0;i<len; i++)
    (*m2).edge[i]=insane;

  len=(*m2).fc_w*(*m2).fc_max;
  for (i=0;i<len; i++)
    (*m2).face[i]=insane;

  len=(*m2).hi_w*(*m2).hi_max;
  for (i=0;i<len; i++)
    (*m2).hier[i]=insane;

  len=(*m2).eh_w*(*m2).eh_max;
  for (i=0;i<len; i++)
    (*m2).elhier[i]=insane;

  len=(*m2).bd_w*(*m2).bd_max;
  for (i=0;i<len; i++)
    (*m2).bound[i]=insane;

  len=(*m2).fu_w*(*m2).fu_max;
  for (i=0;i<len; i++)
    (*m2).func[i]=(double)insane;

  len=(*m2).pa_w*(*m2).pa_max;
  for (i=0;i<len; i++)
    (*m2).para[i]=(double)insane;

  len=(*m2).sg_w*(*m2).sg_max;
  for (i=0;i<len; i++)
    (*m2).sseg[i]=insane;

  len=(*m2).sp_w*(*m2).sp_max;
  for (i=0;i<len; i++)
    (*m2).spar[i]=(double)insane;
  
  len=(*m2).st_max;
  for (i=0;i<len; i++)
    (*m2).st[i]=-1.0; 
#endif
  
  /* now copy the mesh */

  /* vertices */
  for(i=0;i<(*m1).vx_nr;i++)
    {
      (*m2).vertex[i*vx_w2+MCT2VXSTRT  ] =
	(*m1).vertex[i*vx_w1+MCT1VXSTRT  ];
      (*m2).vertex[i*vx_w2+MCT2VXSTRT+1] =
	(*m1).vertex[i*vx_w1+MCT1VXSTRT+1];
    }
  (*m2).vx_nr=(*m1).vx_nr;
    
  /* hierarchy */
  for (i=0; i<(*m1).hi_nr; i++)
    {
      (*m2).hier[i*hi_w2 + MCT2HICHLD] =
	(*m1).hier[i*hi_w1 + MCT1HICHLD];

      (*m2).hier[i*hi_w2 + MCT2HIFAT1  ] =
	(*m1).hier[i*hi_w1 + MCT1HIFAT1  ];
      (*m2).hier[i*hi_w2 + MCT2HIFAT1+1] =
	(*m1).hier[i*hi_w1 + MCT1HIFAT1+1];

      (*m2).hier[i*hi_w2 + MCT2HILVL] =
	(*m1).hier[i*hi_w1 + MCT1HILVL];
    }
  (*m2).hi_nr=(*m1).hi_nr;

  /* no element hierarchy to be copied */

  /* edges */
  /* fist copy as much as possible */
  for (i=0; i<(*m1).eg_nr; i++)
    {
      /* both end nodes of the edge */
      (*m2).edge[i*eg_w2 + MCT2EGNOD1   ] =
	(*m1).edge[i*eg_w1 + MCT1EGNOD1   ];
      (*m2).edge[i*eg_w2 + MCT2EGNOD1+1 ] =
	(*m1).edge[i*eg_w1 + MCT1EGNOD1+1 ];
      /* child data */
      (*m2).edge[i*eg_w2 + MCT2EGCHL1   ] =
	(*m1).edge[i*eg_w1 + MCT1EGCHL1   ];
      (*m2).edge[i*eg_w2 + MCT2EGCHL1+1 ] =
	(*m1).edge[i*eg_w1 + MCT1EGCHL1+1 ];
      /* boundary data */
      (*m2).edge[i*eg_w2 + MCT2EGBND    ] =
	(*m1).edge[i*eg_w1 + MCT1EGBND    ];
      /* pcsurf data */
      (*m2).edge[i*eg_w2 + MCT2EGPCSU   ] =
	(*m1).edge[i*eg_w1 + MCT1EGPCSU   ];
      /* level data */
      (*m2).edge[i*eg_w2 + MCT2EGLVL    ] =
	(*m1).edge[i*eg_w1 + MCT1EGLVL    ];
    }
    
  /* boundary */
  /* just copy */
  for (i=0; i<(*m1).bd_nr; i++)
    {
      (*m2).bound[i*bd_w2 + MCT2BDEDGE]=
	(*m1).bound[i*bd_w1 + MCT1BDEDGE];
      (*m2).bound[i*bd_w2 + MCT2BDTYPE]=
	(*m1).bound[i*bd_w1 + MCT1BDTYPE];
      (*m2).bound[i*bd_w2 + MCT2BDFNCT]=
	(*m1).bound[i*bd_w1 + MCT1BDFNCT];
      (*m2).bound[i*bd_w2 + MCT2BDORIE]=
	(*m1).bound[i*bd_w1 + MCT1BDORIE];
      (*m2).bound[i*bd_w2 + MCT2BDSSEG]=
	(*m1).bound[i*bd_w1 + MCT1BDSSEG];    
    }
  (*m2).bd_nr=(*m1).bd_nr;
  
  /* shape segments, just copy */
  for (i=0; i<(*m1).sg_nr; i++)
    {
      for (j=0; j<MC2XSGLN; j++)
	{
	  (*m2).sseg[i*sg_w2 + j] = (*m1).sseg[i*sg_w1 + j];
	}
    }
  (*m2).sg_nr=(*m1).sg_nr;

  for (i=0; i<(*m1).st_nr; i++)
    (*m2).st[i]=(*m1).st[i];
  (*m2).st_nr=(*m1).st_nr;

  /* shape parameters, just copy */
  for (i=0; i<(*m1).sp_nr; i++)
    {
      for (j=0; j<MC2XSPLN; j++)
	{
	  (*m2).spar[i*sp_w2 + j] = (*m1).spar[i*sp_w1 + j];
	}
    }
  (*m2).sp_nr=(*m1).sp_nr;   
    
  (*m2).eg_nr=(*m1).eg_nr;
  /* now build the midnodes and the hierarchy entires for them */
  for (i=0; i<(*m2).eg_nr; i++)
    {
      /* if the T1 edge had no child we have to create the midnode */ 
      if ((*m2).edge[i*eg_w2 + MCT2EGCHL1]==-1)
	{
	  FIDX old1, old2, newnode, newhier, level, bd, seg;

	  old1= (*m2).edge[i*eg_w2 + MCT2EGNOD1  ];
	  old2= (*m2).edge[i*eg_w2 + MCT2EGNOD1+1];

	  /* level of the edge */
	  level= (*m2).edge[i*eg_w2 + MCT2EGLVL];

	  /* create new vertex, enough space is assured (vx_nr1+eg_nr1) */
	  newnode=(*m2).vx_nr;
	  (*m2).vx_nr++;
	  (*m2).st_nr++;

	  bd = (*m2).edge[i*eg_w2+ MCT2EGBND];   
	  if (bd!=-1) seg = (*m2).bound[bd*bd_w2+MCT2BDSSEG];
	  else seg = -1;   
		  
	  /* node is not on a shape segment */
	  for (j=0; j<2; j++)
	    {
	      (*m2).vertex[newnode*vx_w2+MCT2VXSTRT+j]=
		0.5*(*m2).vertex[old1*vx_w2+MCT2VXSTRT+j]+
		0.5*(*m2).vertex[old2*vx_w2+MCT2VXSTRT+j];
	    } 

	  if(seg>-1)
	    {/* adjust shape segment  */
	      err=mesh_spline_set_t( m2, seg, old1, old2, newnode, 0, 2);
	      FUNCTION_FAILURE_HANDLE( err, 
		    mesh_spline_set_t, mesh_t1_to_t2); 
	    }
 
	  (*m2).edge[i*eg_w2 + MCT2EGNODM]=newnode;

	  /* create new hierarchy entry */
	  newhier=(*m2).hi_nr;
	  (*m2).hi_nr++;

	  (*m2).hier[newhier*hi_w2+MCT2HICHLD  ]=newnode;
	  (*m2).hier[newhier*hi_w2+MCT2HIFAT1  ]=old1;
	  (*m2).hier[newhier*hi_w2+MCT2HIFAT1+1]=old2;
	  (*m2).hier[newhier*hi_w2+MCT2HILVL   ]=level;
	}
      else
	{
	  FIDX chld;
	  /* the midnode is the second node of the child edge */
	  chld=(*m2).edge[i*eg_w2 + MCT2EGCHL1];
	  (*m2).edge[i*eg_w2 + MCT2EGNODM]=
	    (*m2).edge[chld*eg_w2 + MCT2EGNOD1+1];
	}
    }
  
  /* faces */
  /* just copy, no difference */
  for (i=0; i<(*m1).fc_nr; i++)
    {
      for (j=0; j<3; j++)
	{
	  (*m2).face[i*fc_w2 + MCT2FCEDG1+j]=
	    (*m1).face[i*fc_w1 + MCT1FCEDG1+j];
	}
      (*m2).face[i*fc_w2 + MCT2FCRHSF]=
	(*m1).face[i*fc_w1 + MCT1FCRHSF];
      (*m2).face[i*fc_w2 + MCT2FCPCVL]=
	(*m1).face[i*fc_w1 + MCT1FCPCVL];
    }
  (*m2).fc_nr=(*m1).fc_nr;

  /* no volumes */

  /* pcsurf */
  /* just copy */
  for (i=0; i<(*m1).ps_nr; i++)
    {
      for (j=0; j<MCXXPSLN; j++)
	{
	  (*m2).pcsurf[i*ps_w2 + j] = (*m1).pcsurf[i*ps_w1 + j];
	}
    }
  (*m2).ps_nr=(*m1).ps_nr;

  /* pcvol */
  /* just copy */
  for (i=0; i<(*m1).pv_nr; i++)
    {
      for (j=0; j<MCXXPVLN; j++)
	{
	  (*m2).pcvol[i*pv_w2 + j] = (*m1).pcvol[i*pv_w1 + j];
	}
    }
  (*m2).pv_nr=(*m1).pv_nr;

  /* pccrit */
  /* just copy */
  for (i=0; i<(*m1).pc_nr; i++)
    {
      for (j=0; j<MCXXPCLN; j++)
	{
	  (*m2).pccrit[i*pc_w2 + j] = (*m1).pccrit[i*pc_w1 + j];
	}
    }
  (*m2).pc_nr=(*m1).pc_nr;

  /* function, just copy */
  for (i=0; i<(*m1).fu_nr; i++)
    {
      for (j=0; j<MC2XFULN; j++)
	{
	  (*m2).func[i*fu_w2 + j] = (*m1).func[i*fu_w1 + j];
	}
    }
  (*m2).fu_nr=(*m1).fu_nr;


  /* parameter, just copy */
  for (i=0; i<(*m1).pa_nr; i++)
    {
	(*m2).para[i] = (*m1).para[i];
    }
  (*m2).pa_nr=(*m1).pa_nr;

  /* shape segments, just copy */
  for (i=0; i<(*m1).sg_nr; i++)
    {
      for (j=0; j<MC2XSGLN; j++)
	{
	  (*m2).sseg[i*sg_w2 + j] = (*m1).sseg[i*sg_w1 + j];
	}
    }
  (*m2).sg_nr=(*m1).sg_nr;


  /* shape parameters, just copy */
  for (i=0; i<(*m1).sp_nr; i++)
    {
      for (j=0; j<MC2XSPLN; j++)
	{
	  (*m2).spar[i*sp_w2 + j] = (*m1).spar[i*sp_w1 + j];
	}
    }
  (*m2).sp_nr=(*m1).sp_nr;


  /* elements */
  for(i=0; i<(*m1).el_nr; i++)
    {
      err=mesh_t2_get_elem_from_face( m2, i );
      FUNCTION_FAILURE_HANDLE( err, mesh_t2_get_elem_from_face,
			       mesh_t1_to_t2);
    }
  (*m2).el_nr = (*m1).el_nr;

  (*m2).meshgen = (*m1).meshgen;
  (*m2).meshgenDATA = (*m1).meshgenDATA;


  return SUCCESS;
}





/*FUNCTION*/
int mesh_t2_to_t1( struct mesh *m2, struct mesh *m1
/* creates the T1 mesh by converting the T2 mesh m2 into a T1 mesh,
   where one element/face in m2 creates 4 elements/faces in m1, such
   that they have the same number of vertices

   Input:  m2         - T2 mesh
   Output: m1         - T1 mesh, (empty mesh given via reference,
                        interior is set up)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX somany, i, j, elj;

  FIDX   vx_w2, el_w2, eg_w2, fc_w2, hi_w2, bd_w2,
    ps_w2, pv_w2, pc_w2, fu_w2, pa_w2, sg_w2, sp_w2;

  const FIDX   vx_w1 = MCT1VXLN, eg_w1 = MCT1EGLN, 
    fc_w1 = MCT1FCLN, el_w1 = MCT1ELLN, vo_w1 = MCT1VOLN,
    eh_w1 = MCT1EHLN, hi_w1 = MCT1HILN, 
    bd_w1 = MCT1BDLN, ps_w1 = MCXXPSLN, pv_w1 = MCXXPVLN,
    pc_w1 = MCXXPCLN, fu_w1 = MC2XFULN, pa_w1 = MC2XPALN,
    sg_w1 = MC2XSGLN, sp_w1 = MC2XSPLN;

  /* set dimension */
  (*m1).dim = (*m2).dim;

  /* set width of fields */
  (*m1).vx_w = vx_w1;
  (*m1).el_w = el_w1;
  (*m1).eg_w = eg_w1;
  (*m1).fc_w = fc_w1;
  (*m1).vo_w = vo_w1;
  (*m1).hi_w = hi_w1;
  (*m1).eh_w = eh_w1;
  (*m1).bd_w = bd_w1;
  (*m1).ps_w = ps_w1;
  (*m1).pv_w = pv_w1;
  (*m1).pc_w = pc_w1;
  (*m1).fu_w = fu_w1;
  (*m1).pa_w = pa_w1;
  (*m1).sg_w = sg_w1;
  (*m1).sp_w = sp_w1;

  vx_w2 = (*m2).vx_w; el_w2 = (*m2).el_w; eg_w2 = (*m2).eg_w;
  fc_w2 = (*m2).fc_w; hi_w2 = (*m2).hi_w; bd_w2 = (*m2).bd_w;
  ps_w2 = (*m2).ps_w; pv_w2 = (*m2).pv_w; pc_w2 = (*m2).pc_w;
  fu_w2 = (*m2).fu_w; pa_w2 = (*m2).pa_w; sg_w2 = (*m2).sg_w;
  sp_w2 = (*m2).sp_w;

  /* initialise arrays */
  somany=(*m2).vx_nr;
  TRY_MALLOC( (*m1).vertex, (*m1).vx_w * somany , double,
	      mesh_t2_to_t1 );
  (*m1).vx_max=somany;
  (*m1).vx_nr=0;

  somany=4*(*m2).el_nr; /* one face to 4 */
  TRY_MALLOC( (*m1).elem, (*m1).el_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).el_max=somany;
  (*m1).el_nr=0;

  somany=3*(*m2).eg_nr+3*(*m2).fc_nr; /* all edges without children
					 have to be split + each face
					 generates 3 child edges */
  TRY_MALLOC( (*m1).edge, (*m1).eg_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).eg_max=somany;
  (*m1).eg_nr=0;

  somany=4*(*m2).fc_nr; /* one face to 4 */
  TRY_MALLOC( (*m1).face, (*m1).fc_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).fc_max=somany;
  (*m1).fc_nr=0;

  (*m1).vols= NULL;
  (*m1).vo_max=0;
  (*m1).vo_nr=0;

  somany=(*m2).hi_nr+(*m2).eg_nr;
  TRY_MALLOC( (*m1).hier, (*m1).hi_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).hi_max=somany;
  (*m1).hi_nr=0;

  somany=3*(*m2).bd_nr; /* due to split edges */
  TRY_MALLOC( (*m1).bound, (*m1).bd_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).bd_max=somany;
  (*m1).bd_nr=0;

  somany=3*(*m2).ps_nr; /* due to split edges */
  TRY_MALLOC( (*m1).pcsurf, (*m1).ps_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).ps_max=somany;
  (*m1).ps_nr=0;

  somany=4*(*m2).pv_nr; /* one face to 4 */
  TRY_MALLOC( (*m1).pcvol, (*m1).pv_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).pv_max=somany;
  (*m1).pv_nr=0;

  somany=(*m2).pc_nr;
  TRY_MALLOC( (*m1).pccrit, (*m1).pc_w * somany , double,
	      mesh_t2_to_t1 );
  (*m1).pc_max=somany;
  (*m1).pc_nr=0;

  somany=(*m2).fu_nr;
  TRY_MALLOC( (*m1).func, (*m1).fu_w * somany , double,
	      mesh_t2_to_t1 );
  (*m1).fu_max=somany;
  (*m1).fu_nr=0;

  somany=(*m2).pa_nr;
  TRY_MALLOC( (*m1).para, (*m1).pa_w * somany , double,
	      mesh_t2_to_t1 );
  (*m1).pa_max=somany;
  (*m1).pa_nr=0;

  somany=(*m2).sg_nr;
  TRY_MALLOC( (*m1).sseg, (*m1).sg_w * somany , FIDX,
	      mesh_t2_to_t1 );
  (*m1).sg_max=somany;
  (*m1).sg_nr=0;

  somany=(*m2).sp_nr;
  TRY_MALLOC( (*m1).spar, (*m1).sp_w * somany , double,
	      mesh_t2_to_t1 );
  (*m1).sp_max=somany;
  (*m1).sp_nr=0;

  somany=(*m2).vx_nr;
  TRY_MALLOC( (*m1).st, somany , double,
	      mesh_t2_to_t1 );
  (*m1).st_max=somany;
  (*m1).st_nr=0;

  (*m1).eh_nr  = 0;
  (*m1).eh_max = 0;
  (*m1).elhier = NULL;

  /* now copy the mesh */

  /* vertices */
  for(i=0;i<(*m2).vx_nr;i++)
    {
      (*m1).vertex[i*vx_w1+MCT1VXSTRT  ] =
	(*m2).vertex[i*vx_w2+MCT2VXSTRT  ];
      (*m1).vertex[i*vx_w1+MCT1VXSTRT+1] =
	(*m2).vertex[i*vx_w2+MCT2VXSTRT+1];
    }
  (*m1).vx_nr=(*m2).vx_nr;


  /* hierarchy */
  for (i=0; i<(*m2).hi_nr; i++)
    {
      (*m1).hier[i*hi_w1 + MCT1HICHLD] =
	(*m2).hier[i*hi_w2 + MCT2HICHLD];

      (*m1).hier[i*hi_w1 + MCT1HIFAT1  ] =
	(*m2).hier[i*hi_w2 + MCT2HIFAT1  ];
      (*m1).hier[i*hi_w1 + MCT1HIFAT1+1] =
	(*m2).hier[i*hi_w2 + MCT2HIFAT1+1];

      (*m1).hier[i*hi_w1 + MCT1HILVL] =
	(*m2).hier[i*hi_w2 + MCT2HILVL];
    }
  (*m1).hi_nr=(*m2).hi_nr;

  /* boundary, just copy */
  for (i=0; i<(*m2).bd_nr; i++)
    {
      (*m1).bound[i*bd_w1 + MCT1BDEDGE]=
	(*m2).bound[i*bd_w2 + MCT2BDEDGE];
      (*m1).bound[i*bd_w1 + MCT1BDTYPE]=
	(*m2).bound[i*bd_w2 + MCT2BDTYPE];
      (*m1).bound[i*bd_w1 + MCT1BDFNCT]=
	(*m2).bound[i*bd_w2 + MCT2BDFNCT];
      (*m1).bound[i*bd_w1 + MCT1BDORIE]=
	(*m2).bound[i*bd_w2 + MCT2BDORIE];
      (*m1).bound[i*bd_w1 + MCT1BDSSEG]=
	(*m2).bound[i*bd_w2 + MCT2BDSSEG];
    }
  (*m1).bd_nr=(*m2).bd_nr;

  /* pcsurf, just copy */
  for (i=0; i<(*m2).ps_nr; i++)
    {
      for (j=0; j<MCXXPSLN; j++)
	{
	  (*m1).pcsurf[i*ps_w1 + j]= (*m2).pcsurf[i*ps_w2 + j];
	}
    }
  (*m1).ps_nr=(*m2).ps_nr;

  /* edges */
  /* fist copy as much as possible */
  for (i=0; i<(*m2).eg_nr; i++)
    {
      /* both end nodes of the edge */
      (*m1).edge[i*eg_w1 + MCT1EGNOD1   ] =
	(*m2).edge[i*eg_w2 + MCT2EGNOD1   ];
      (*m1).edge[i*eg_w1 + MCT1EGNOD1+1 ] =
	(*m2).edge[i*eg_w2 + MCT2EGNOD1+1 ];
      /* child data */
      (*m1).edge[i*eg_w1 + MCT1EGCHL1   ] =
	(*m2).edge[i*eg_w2 + MCT2EGCHL1   ];
      (*m1).edge[i*eg_w1 + MCT1EGCHL1+1 ] =
	(*m2).edge[i*eg_w2 + MCT2EGCHL1+1 ];
      /* boundary data */
      (*m1).edge[i*eg_w1 + MCT1EGBND    ] =
	(*m2).edge[i*eg_w2 + MCT2EGBND    ];
      /* pcsurf data */
      (*m1).edge[i*eg_w1 + MCT1EGPCSU   ] =
	(*m2).edge[i*eg_w2 + MCT2EGPCSU   ];
      /* level data */
      (*m1).edge[i*eg_w1 + MCT1EGLVL    ] =
	(*m2).edge[i*eg_w2 + MCT2EGLVL    ];
    }
  (*m1).eg_nr=(*m2).eg_nr;
  /* now handle the midnodes */
  for (i=0; i<(*m2).eg_nr; i++)
    {
      /* if the T2 edge had no child we have to create two */
      if ((*m2).edge[i*eg_w2 + MCT2EGCHL1]==-1)
	{
	  FIDX newe1, newe2, level;
	  newe1=(*m1).eg_nr; (*m1).eg_nr++;
	  newe2=(*m1).eg_nr; (*m1).eg_nr++;

	  level=(*m2).edge[i*eg_w2 + MCT2EGLVL];

	  (*m1).edge[newe1*eg_w1+MCT1EGNOD1  ]=
	    (*m2).edge[i*eg_w2 + MCT2EGNOD1  ]; /* 1st child = 1st
						   node */
	  (*m1).edge[newe1*eg_w1+MCT1EGNOD1+1]=
	    (*m2).edge[i*eg_w2 + MCT2EGNODM  ]; /* 2n node = midnode */

	  (*m1).edge[newe2*eg_w1+MCT1EGNOD1  ]=
	    (*m2).edge[i*eg_w2 + MCT2EGNOD1+1]; /* 2nd child = 2nd
						   node */
	  (*m1).edge[newe2*eg_w1+MCT1EGNOD1+1]=
	    (*m2).edge[i*eg_w2 + MCT2EGNODM  ]; /* 2n node = midnode */

	  (*m1).edge[newe1*eg_w1+MCT1EGCHL1]=-1; /* no children */
	  (*m1).edge[newe2*eg_w1+MCT1EGCHL1]=-1; /* no children */

	  (*m1).edge[i*eg_w1+MCT1EGCHL1  ]=newe1; /* the children */
	  (*m1).edge[i*eg_w1+MCT1EGCHL1+1]=newe2; 

	  (*m1).edge[newe1*eg_w1+MCT1EGBND]=-1; /* no boundary */
	  (*m1).edge[newe2*eg_w1+MCT1EGBND]=-1; /* by default  */
	  /* if boundary .. */
	  if ((*m2).edge[i*eg_w2+MCT2EGBND]!=-1)
	    {
	      FIDX newbd1, newbd2, oldbd;

	      oldbd=(*m2).edge[i*eg_w2+MCT2EGBND];

	      /* create 2 new boundary entries */
	      newbd1=(*m1).bd_nr;
	      newbd2=(*m1).bd_nr+1;
	      (*m1).bd_nr+=2;

	      /* bd -> edge */
	      (*m1).bound[newbd1*bd_w1+MCT1BDEDGE]=newe1;
	      (*m1).bound[newbd2*bd_w1+MCT1BDEDGE]=newe2;
	      /* bd -> type */
	      (*m1).bound[newbd1*bd_w1+MCT1BDTYPE]=
		(*m1).bound[oldbd*bd_w1+MCT1BDTYPE];
	      (*m1).bound[newbd2*bd_w1+MCT1BDTYPE]=
		(*m1).bound[oldbd*bd_w1+MCT1BDTYPE];
	      /* bd -> func */
	      (*m1).bound[newbd1*bd_w1+MCT1BDFNCT]=
		(*m1).bound[oldbd*bd_w1+MCT1BDFNCT];
	      (*m1).bound[newbd2*bd_w1+MCT1BDFNCT]=
		(*m1).bound[oldbd*bd_w1+MCT1BDFNCT];
	      /* bd -> orie */
	      (*m1).bound[newbd1*bd_w1+MCT1BDORIE]=
		(*m1).bound[oldbd*bd_w1+MCT1BDORIE];
	      (*m1).bound[newbd2*bd_w1+MCT1BDORIE]=
		-(*m1).bound[oldbd*bd_w1+MCT1BDORIE];
	      /* bd -> sseg */
	      (*m1).bound[newbd1*bd_w1+MCT1BDSSEG]=
		(*m1).bound[oldbd*bd_w1+MCT1BDSSEG];
	      (*m1).bound[newbd2*bd_w1+MCT1BDSSEG]=
		(*m1).bound[oldbd*bd_w1+MCT1BDSSEG];

	      (*m1).edge[newe1*eg_w1+MCT1EGBND]=newbd1; /* set to valid */
	      (*m1).edge[newe2*eg_w1+MCT1EGBND]=newbd2; 
	    }
	  (*m1).edge[newe1*eg_w1+MCT1EGLVL]=level; 
	  (*m1).edge[newe2*eg_w1+MCT1EGLVL]=level; 

	  (*m1).edge[newe1*eg_w1+MCT1EGPCSU]=-1; /* no pcsurf */
	  (*m1).edge[newe2*eg_w1+MCT1EGPCSU]=-1; /* by default  */
	  /* if pcsurf .. */
	  if ((*m2).edge[i*eg_w2+MCT2EGPCSU]!=-1)
	    {
	      FIDX newps1, newps2, oldps;

	      oldps=(*m2).edge[i*eg_w2+MCT2EGPCSU];

	      /* create 2 new pcsurf entries */
	      newps1=(*m1).ps_nr;
	      newps2=(*m1).ps_nr+1;
	      (*m1).ps_nr+=2;

	      /* ps -> surf */
	      (*m1).pcsurf[newps1*ps_w1+MCXXPSSURF]=newe1;
	      (*m1).pcsurf[newps2*ps_w1+MCXXPSSURF]=newe2;
	      /* ps -> vole is set later*/
	      (*m1).pcsurf[newps1*ps_w1+MCXXPSVOLE]=-1;
	      (*m1).pcsurf[newps2*ps_w1+MCXXPSVOLE]=-1;
	      /* ps -> orie */
	      (*m1).pcsurf[newps1*ps_w1+MCXXPSORIE]=
		(*m2).pcsurf[oldps*ps_w2+MCXXPSORIE];
	      (*m1).pcsurf[newps2*ps_w1+MCXXPSORIE]=
		-(*m2).pcsurf[oldps*ps_w2+MCXXPSORIE];
	      /* ps -> crit */
	      (*m1).pcsurf[newps1*ps_w1+MCXXPSCRIT]=
		(*m2).pcsurf[oldps*ps_w2+MCXXPSCRIT];
	      (*m1).pcsurf[newps2*ps_w1+MCXXPSCRIT]=
		(*m2).pcsurf[oldps*ps_w2+MCXXPSCRIT];

	      (*m1).edge[newe1*eg_w1+MCT1EGPCSU]=newps1; /* set to valid */
	      (*m1).edge[newe2*eg_w1+MCT1EGPCSU]=newps2;
	    }
	}
    }
  
  /* pcvol, just copy */
  for (i=0; i<(*m2).pv_nr; i++)
    {
      for (j=0; j<MCXXPVLN; j++)
	{
	  (*m1).pcvol[i*pv_w1 + j]= (*m2).pcvol[i*pv_w2 + j];
	}
    }
  (*m1).pv_nr=(*m2).pv_nr;

  /* faces and elements */
  for (i=0; i<(*m2).fc_nr; i++)
    {
      FIDX nodes[3], edges[3], nodiedge[3], newedges[3];

      for (j=0; j<3; j++)
	edges[j]=(*m2).face[i*fc_w2+MCT2FCEDG1+j];
      
      /* first, get the edges into order,
	 first edge stays first and defines the first two points
      */
      nodes[0]=(*m2).edge[edges[0]*eg_w2+MCT2EGNOD1];
      nodes[1]=(*m2).edge[edges[0]*eg_w2+MCT2EGNOD1+1];
      nodiedge[0]=0; /* edge 0 is first node first */
      
      /* now see which of the two other edges has node[1] as well */
      if ((nodes[1]!=(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1])&&
	  (nodes[1]!=(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1+1]))
	{
	  /* edges 2 and 3 have to be swaped */
	  j=edges[2];
	  edges[2]=edges[1];
	  edges[1]=j;
	}
      /* get the 3rd node */
      if (nodes[1]==(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1  ])
	{
	  nodiedge[1]=0; /* edge 1 is first node first */
	  /* ok, the second node of the second edge is the 3rd node of
	     the face
	  */
	  nodes[2]=(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1+1];
	}
      else if (nodes[1]==(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1+1])
	{
	  nodiedge[1]=1; /* edge 1 is second node first */
	  /* ok, the first node of the second edge is the 3rd node of
	     the face
	  */
	  nodes[2]=(*m2).edge[edges[1]*eg_w2+MCT2EGNOD1  ];
	}
      else
	{ 
	  /* cry */
	  fprintf(stderr,
		  "mesh_t2_to_t1: face with one node only in one edge!\n");
	  return FAIL;
	}

      if (nodes[2]==(*m2).edge[edges[2]*eg_w2+MCT2EGNOD1  ])
	{
	  nodiedge[2]=0; /* edge 2 is first node first */
	}
      else if (nodes[2]==(*m2).edge[edges[2]*eg_w2+MCT2EGNOD1+1])
	{
	  nodiedge[2]=1; /* edge 2 is second node first */
	}
      else
	{ 
	  /* cry */
	  fprintf(stderr,
		  "mesh_t2_to_t1: face with 3rd node not in face!\n");
	  return FAIL;
	}

      /* build first 3 faces + elements (outer 3) */
      for (j=0; j<3; j++)
	{
	  FIDX newedge,old1,old2,chld1,chld2;

	  /* outer edges */
	  old1=edges[j]; old2=edges[(j+2)%3];
	  chld1=(*m1).edge[old1*eg_w1+MCT1EGCHL1+ nodiedge[j]];
	  chld2=(*m1).edge[old2*eg_w1+MCT1EGCHL1+ nodiedge[(j+2)%3]];

	  /* new inner edge */
	  newedge=(*m1).eg_nr; (*m1).eg_nr++;
	  newedges[j]=newedge;
	  (*m1).edge[newedge*eg_w1+MCT1EGNOD1  ]=
	    (*m2).edge[old1*eg_w2+MCT2EGNODM];
	  (*m1).edge[newedge*eg_w1+MCT1EGNOD1+1]=
	    (*m2).edge[old2*eg_w2+MCT2EGNODM];
	  (*m1).edge[newedge*eg_w1+MCT1EGCHL1]=-1;
	  (*m1).edge[newedge*eg_w1+MCT1EGBND ]=-1;
	  (*m1).edge[newedge*eg_w1+MCT1EGPCSU]=-1;

	  /*build the face */
	  (*m1).face[(4*i+j)*fc_w1+MCT1FCEDG1  ]=chld1;
	  (*m1).face[(4*i+j)*fc_w1+MCT1FCEDG1+1]=newedge;
	  (*m1).face[(4*i+j)*fc_w1+MCT1FCEDG1+2]=chld2;
	  (*m1).face[(4*i+j)*fc_w1+MCT1FCRHSF]=
	    (*m2).face[i*fc_w2+MCT2FCRHSF];

	  /*if needed, build the pcvol */
	  if ((*m2).face[i*fc_w2+MCT2FCPCVL]!=-1)
	    {
	      FIDX new,old,bla;
	      new=(*m1).pv_nr;
	      old=(*m2).face[i*fc_w2+MCT2FCPCVL];

	      for (bla=0; bla<MCXXPVLN; bla++)
		{
		  (*m1).pcvol[new*pv_w1 + bla]= (*m2).pcvol[old*pv_w2 + bla];
		}
	      (*m1).pcvol[new*pv_w1 + MCXXPVVOLM]=(4*i+j);

	      (*m1).pv_nr++;
	      (*m1).face[(4*i+j)*fc_w1+MCT1FCPCVL]= new;
	    }
	  else
	    {
	      (*m1).face[(4*i+j)*fc_w1+MCT1FCPCVL]= -1;
	    }
	    
	  /* check if the edges of the new face are pcsurf edges, if so,
	     let them know the name of the new face */
	  for (elj=0; elj<3; elj++)
	    {
	      FIDX eg, pcsu;
	      eg=(*m1).face[(4*i+j)*fc_w1+MCT1FCEDG1+elj];
	      pcsu=(*m1).edge[eg*eg_w1+MCT1EGPCSU];
	      if (pcsu!=-1)
		{
		  if ((*m1).pcsurf[pcsu*ps_w1+MCXXPSVOLE]==-1)
		    {
		      (*m1).pcsurf[pcsu*ps_w1+MCXXPSVOLE]=(4*i+j);
		      (*m1).pcsurf[pcsu*ps_w1+MCXXPSVOLE+1]=-1;
		    }
		  else
		    {
		      (*m1).pcsurf[pcsu*ps_w1+MCXXPSVOLE+1]=(4*i+j);
		    }
		}
	    }
	  
	  /* build the element */
	  (*m1).elem[(4*i+j)*el_w1+MCT1ELNOD1  ]=nodes[j];
	  (*m1).elem[(4*i+j)*el_w1+MCT1ELNOD1+1]=
	    (*m2).edge[old1*eg_w2+MCT2EGNODM];
	  (*m1).elem[(4*i+j)*el_w1+MCT1ELNOD1+2]=
	    (*m2).edge[old2*eg_w2+MCT2EGNODM];

	  /* build the inner face + element */
	  (*m1).face[(4*i+3)*fc_w1+MCT1FCEDG1+j]=newedges[j];
	  (*m1).face[(4*i+3)*fc_w1+MCT1FCRHSF]=
	    (*m2).face[i*fc_w2+MCT2FCRHSF];
	  (*m1).face[(4*i+3)*fc_w1+MCT1FCPCVL]=
	    (*m2).face[i*fc_w2+MCT2FCPCVL];
	  (*m1).elem[(4*i+3)*el_w1+MCT1ELNOD1+j]=
	    (*m2).edge[old1*eg_w2+MCT2EGNODM];
	}
      
    }
  (*m1).fc_nr = 4*(*m2).fc_nr;
  (*m1).el_nr = (*m1).fc_nr;



  /* no volumes */

  /* pccrit, just copy */
  for (i=0; i<(*m2).pc_nr; i++)
    {
      for (j=0; j<MCXXPCLN; j++)
	{
	  (*m1).pccrit[i*pc_w1 + j]= (*m2).pccrit[i*pc_w2 + j];
	}
    }
  (*m1).pc_nr=(*m2).pc_nr;

  /* function, just copy */
  for (i=0; i<(*m2).fu_nr; i++)
    {
      for (j=0; j<MC2XFULN; j++)
	{
	  (*m1).func[i*fu_w1 + j] = (*m2).func[i*fu_w2 + j];
	}
    }
  (*m1).fu_nr=(*m2).fu_nr;

  /* parameter, just copy */
  for (i=0; i<(*m2).pa_nr; i++)
    {
      for (j=0; j<MC2XPALN; j++)
	{
	  (*m1).para[i*pa_w1 + j] = (*m2).para[i*pa_w2 + j];
	}
    }
  (*m1).pa_nr=(*m2).pa_nr;


  /* shape segments, just copy */
  for (i=0; i<(*m2).sg_nr; i++)
    {
      for (j=0; j<MC2XSGLN; j++)
	{
	  (*m1).sseg[i*sg_w1 + j] = (*m2).sseg[i*sg_w2 + j];
	}
    }
  (*m1).sg_nr=(*m2).sg_nr;


  /* shape parameter, just copy */
  for (i=0; i<(*m2).sp_nr; i++)
    {
      for (j=0; j<MC2XSPLN; j++)
	{
	  (*m1).spar[i*sp_w1 + j] = (*m2).spar[i*sp_w2 + j];
	}
    }
  (*m1).sp_nr=(*m2).sp_nr;

  /* shape parameter values for the nodes, just copy */
  for (i=0; i<(*m2).st_nr; i++)
    {
	  (*m1).st[i] = (*m2).st[i];
    }
  (*m1).st_nr=(*m2).st_nr;

  (*m1).meshgen = (*m2).meshgen;
  (*m1).meshgenDATA = (*m2).meshgenDATA;

  return SUCCESS;
}


/********************************************************************/
/*                                                                  */
/*         element type independend stuff                           */
/*                                                                  */
/********************************************************************/


/*FUNCTION*/
int mesh_func_eval ( struct mesh *m, FIDX tfu, double *point, double time,
		     FIDX reslen, double *result, double *gradf, double *dotf
/* evaluates a function as defined by the mesh

   Input:  m       - the mesh
           tfu     - id of the function, points to the function
                     within m.func
	   point   - point where the function has to be evaluated
	             (vector of length m.dim)
           time    - time when the function is to be evaluated,
                     just provide time=0.0 if stationary problem 
	   reslen  - length of the provided result vector,
	             if the function type doesn't match reslen
	             this function will return FAIL

   Output: result  - vector of length reslen (provided by reference),
                     holds the result of the function evaluation on
                     SUCCESSful return
           gradf   - gradient of the function, only modified if
                     gradf!=NULL, otherwise has to be array of length reslen*(m.dim)
		     gradf[i*dim+j] is the derivative of the i-th
		     component of the function wrt. the j-th variable
		     of space
	   dotf    - time derivative of the function, only modified if
	             dotf!=NULL, otherwise has to be vector of length reslen
                     
   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){
  FIDX i, j, type, len, fu_w, dim;
  int  err;

  fu_w=(*m).fu_w;

  if ((tfu>=(*m).fu_nr)||(tfu<0))
    {
      fprintf(stderr,
	      "mesh_func_eval: tfu=%"dFIDX" out of range(0<=tfu<%"dFIDX")!\n",
	       tfu,  (*m).fu_nr);
      return FAIL;
    }
  
  type=(FIDX)(*m).func[tfu*fu_w+MC2XFUTYPE];

  dim = (*m).dim;

  if (type<100)
    { len = 1; }
  else
    {
      len   = dim;
      type -= 100;
    }

  if (len!=reslen)
    {
      fprintf(stderr,
	      "mesh_func_eval: reslen doesn't match function type!\n");
      return FAIL;
    }

  /* for the individual components of the result */
  for (i=0; i<len; i++)
    {
      switch(type)
	{
	case 0:
	  /* constant function */
	  result[i]=(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX];
	  if (gradf!=NULL)
	    {
	      for (j=0; j<dim; j++)
		{
		  gradf[i*dim+j]=0.0;
		}
	    }
	  if (dotf!=NULL)
	    { 
	      dotf[i]=0.0;
	    }
	  break;
	case 1:
	  /* linear function */
	  /* constant part first, then x and y */
	  result[i] = (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
	    +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0]*point[0]
	    +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1]*point[1];
	  if (gradf!=NULL)
	    {
	      for (j=0; j<dim; j++)
		{
		  gradf[i*dim+j]=(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+j];
		}
	    }
	  if (dotf!=NULL)
	    { 
	      dotf[i]=0.0;
	    }
	  break;
	case 2:
	  /* quadratic function */
	  /* f(x,y)=c[5] + (c[4]+c[1]*y)*y+(c[2]*y+c[3]+c[0]*x)*x */
	  result[i] = (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+5]
	    +( (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+4]
	       +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1]*point[1]
	       )*point[1]
	    +( (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]*point[1]
	       +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]
	       +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0]*point[0]
	       )*point[0];
	  if (gradf!=NULL)
	    {
	      gradf[i*dim+0]=
		(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]*point[1]
		+(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]
		+0.5*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0]*point[0];
	      gradf[i*dim+1]=(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+4]
		+0.5*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1]*point[1]
		+(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]*point[0];
	    }
	  if (dotf!=NULL)
	    { 
	      dotf[i]=0.0;
	    }
	  break;
	case 8:
	  /* 2D Gaussian function (radial only) */
	  {
	    double exp_part;
	    /* f(x,y)=c[2]*exp(-((x-c[0])^2 +(y-c[1])^2)/(2*c[3]^2) )+ c[4] */
	    exp_part  = (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
	      * exp(-(
		      (point[0]-(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0])
		      *(point[0]-(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0])
		      +
		      (point[1]-(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1])
		      *(point[1]-(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1])
		      )/(2*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]
			 *(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]));
	    result[i] = exp_part +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+4];
	    if (gradf!=NULL)
	      {
		for (j=0; j<dim; j++)
		  {
		    gradf[i*dim+j]= -exp_part
		      *(point[j]-(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+j])
		      /((*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]
			*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+3]);
		  }
	      }
	    if (dotf!=NULL)
	      { 
		dotf[i]=0.0;
	      }
	  }
	  break;
	case 9:
	  /* 2D Gaussian function (radial only), center, intensity and
	     offset given by other function */
	  /* f(x,y)=f_1*exp(-(norm(point-f_0)^2)/(2*c[2]^2) )+ f_3 
	     where 
	     f_0 - a vector valued function specified by parameter c[0],
	     f_1 - a scalar valued function specified by parameter c[1],
	           c[2]- the standard deviation,
	     f_3 - a scalar valued function specified by parameter c[3],
	  */
	  { 
	    double center[2], intens, offset;

	    /*             pointers                memory for variables */
	    double *Jac_center, *dot_center,  Jac_c_array[4],   dot_c_array[2];
	    double *grad_intens, *dot_intens, grad_in_array[2], dot_in_double;
	    double *grad_offset, *dot_offset, grad_of_array[2], dot_of_double;

	    double exp_part;
	    
	    if(gradf==NULL) {Jac_center=NULL; grad_intens=NULL; grad_offset=NULL;}
	    else {Jac_center=Jac_c_array; grad_intens=grad_in_array;
	      grad_offset=grad_of_array;}

	    if(dotf==NULL) {dot_center=NULL; dot_intens=NULL; dot_offset=NULL;}
	    else {dot_center=dot_c_array; dot_intens=&dot_in_double;
	      dot_offset=&dot_of_double;} 
	      

	    err=mesh_func_eval( m, (FIDX) (*m).func[tfu*fu_w+MC2XFUDAT1
						    +i*MC2XFUDATX+0],
				point, time, (*m).dim, center, Jac_center, dot_center );
	    FUNCTION_FAILURE_HANDLE( err, mesh_func_eval,
				     mesh_func_eval);

	    err=mesh_func_eval( m, (FIDX) (*m).func[tfu*fu_w+MC2XFUDAT1
						    +i*MC2XFUDATX+1],
				point, time, 1, &intens, grad_intens, dot_intens );
	    FUNCTION_FAILURE_HANDLE( err, mesh_func_eval,
				     mesh_func_eval);


	    err=mesh_func_eval( m, (FIDX) (*m).func[tfu*fu_w+MC2XFUDAT1
						    +i*MC2XFUDATX+3],
				point, time, 1, &offset, grad_offset, dot_offset );
	    FUNCTION_FAILURE_HANDLE( err, mesh_func_eval,
				     mesh_func_eval);

	    exp_part = exp(-(
			     (point[0]-center[0])*(point[0]-center[0])
			     +
			     (point[1]-center[1])*(point[1]-center[1])
			     )/(2*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
				*(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]));
	    
	    result[i] = intens*exp_part  +offset;
	    if (gradf!=NULL)
	      {
		for (j=0; j<dim; j++)
		  {
		    gradf[i*dim+j]= grad_intens[j]*exp_part
		      + intens*exp_part*(
					 -(point[j]-center[j])
					 /((*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
					   *(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2])
					 )*(1-Jac_center[j*dim+j])
		      + grad_offset[j];
		  }
	      }
	    if (dotf!=NULL)
	      { 
		dotf[i]=  (*dot_intens)*exp_part 
		  + intens*exp_part*(
				     ((point[0]-center[0])*dot_center[0]
				      +(point[1]-center[1])*dot_center[1]
				      )/((*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
				       *(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2])
				     ) /* */
				     + (*dot_offset); 
	      }
	  }
	  break;
	case 10:
	  /* compund function 1 */
	  /* first, check which function to use */
	  if ((*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+2]
	      +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0]*point[0]
	      +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1]*point[1]
	      >=0)
	    {
	      double *gradf_chld, *dotf_chld;
	      if (gradf==NULL) {gradf_chld=NULL;} else {gradf_chld=&gradf[i*dim+0];}
	      if (dotf==NULL) {dotf_chld=NULL;} else {dotf_chld=&dotf[i];}
 
	      /* use function found in MC2XFUDATX+3 */
	      err=mesh_func_eval( m, (FIDX) (*m).func[tfu*fu_w+MC2XFUDAT1
						     +i*MC2XFUDATX+3],
				  point, time, 1, &result[i], gradf_chld, dotf_chld );
	      FUNCTION_FAILURE_HANDLE( err, mesh_func_eval,
				       mesh_func_eval);
	    }
	  else
	    {
	      double *gradf_chld, *dotf_chld;
	      if (gradf==NULL) {gradf_chld=NULL;} else {gradf_chld=&gradf[i*dim+0];}
	      if (dotf==NULL) {dotf_chld=NULL;} else {dotf_chld=&dotf[i];}

	      /* use function found in MC2XFUDATX+4 */
	      err=mesh_func_eval( m, (FIDX) (*m).func[tfu*fu_w+MC2XFUDAT1
						     +i*MC2XFUDATX+4],
				  point, time, 1, &result[i], gradf_chld, dotf_chld );
	      FUNCTION_FAILURE_HANDLE( err, mesh_func_eval,
				       mesh_func_eval);
	    }
	  break;
	case 31:
	  /* constant in space, linear in time */
	  result[i] = (*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+0]
	    +(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1]*time;
	  if (gradf!=NULL)
	    {
	      for (j=0; j<dim; j++)
		{
		  gradf[i*dim+j]=0.0;
		}
	    }
	  if (dotf!=NULL)
	    { 
	      dotf[i]=(*m).func[tfu*fu_w+MC2XFUDAT1+i*MC2XFUDATX+1];
	    }
	  break;
	default:
	  /* type not implemented yet, cry */
	  fprintf(stderr,  "mesh_func_eval: function %"dFIDX" is of unknow function "
		  "type=%"dFIDX"  !\n",  tfu,
		  (FIDX) (*m).func[tfu*fu_w+MC2XFUTYPE]);
	  return FAIL;
	}
    }
  return SUCCESS;
}


/*FUNCTION*/
int mesh_sseg_adjust_tx( struct mesh *m, int insert,
			 FIDX nI, struct vector *dIdx,
			 struct vector *dIdF,
			 int type
/* Adjusts the shape segment parts of the boundary to match the
   current parameters or evaluates the derivatives of the node
   positions and applies the chain rule to evlauate dIdF where x=x(F),
   and dIdx is given, dIdF:=dIdx*dxdF.

   In/Out: m       - the mesh, vertices are adjusted (moved), 
                     only modified if dIdF==NULL

   Input:  insert  - wether or not points may be inserted on the
                     curved boundary, ==1 may instert, ==0 not,
		     ==1 only usefull if in the pre-mesh-generator
		     stage, as this could destroy the connectivity
	   nI      - number of performance criteria, size of dIdF and
                     dIdx, ignored if dIdF==NULL
           dIdx    - derivatives of performance criteria wrt node
                     positions, i.e. dIdx[i].V[d*vx_nr+j] is
                     derivative of criterion I[i] (i=0..nI) wrt the
                     d-th coordinate of node j, ignored if dIdF==NULL
	   type    - type of the mesh, type==1 ==> T1, type==2 ==> T2

   Output: dIdF    - chain rule is applied as describe above, 
		     ignored if dIdF==NULL, otherwise, dIdF[i].V[j] is
		     the derivative of performance criterion I[i] wrt
		     spar[j]

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
			 ){
  FIDX i, j, rcnt, refines;
  int  err;
  FIDX seg, eg;

  FIDX vx_nr, eg_nr, eg_w, bd_nr, bd_w, sg_nr, sg_w;

  struct ilist **eglist; /* list of edges for each segment */
  FIDX *nd_nbs;  /* nd_nbs[i*2+j]=k,  j=1,2:  node i has neigbour k */
  FIDX *nodes;   /* list of nodes on the segment */
  FIDX *nd_segs; /* details the segment in which a node has been set,
		   -1 if it hasn't yet been set */
  FIDX ignore1, ignore2; /* at most two nodes may be necessary to
			   ignore because they have been set
			   previously */

  FIDX l_mcBDSSEG, l_mcBDEDGE, l_mcEGCHL1, l_mcVXSTRT,
		l_mcEGBND, l_mcEGNOD1;

  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT=MCT1VXSTRT;
      l_mcBDSSEG=MCT1BDSSEG;
      l_mcBDEDGE=MCT1BDEDGE;
      l_mcEGCHL1=MCT1EGCHL1;
      l_mcEGBND=MCT1EGBND;
      l_mcEGNOD1=MCT1EGNOD1;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT=MCT2VXSTRT;
      l_mcBDSSEG=MCT2BDSSEG;
      l_mcBDEDGE=MCT2BDEDGE;
      l_mcEGCHL1=MCT2EGCHL1;
      l_mcEGBND=MCT2EGBND;
      l_mcEGNOD1=MCT1EGNOD1;
      break;
    default:
      fprintf(stderr,"mesh_sseg_adjust_tx: unknown type=%d\n",type);
      return FAIL;
    }

  if (dIdF!=NULL)
    {
      if (insert!=0)
	{
	  fprintf(stderr,"mesh_sseg_adjust_tx: insert!=0 "
		  "does not make sense if the derivatives are "
		  "to be evaluated\n");
	  return FAIL;
	}
	
      /* init the derivatives */
      for (i=0; i<nI; i++)
	for (j=0; j< m->sp_nr; j++)
	  dIdF[i].V[j]=0.0;
    }


  sg_nr = m->sg_nr;

  eg_w  = m->eg_w;
  bd_w  = m->bd_w;
  sg_w  = m->sg_w;

  TRY_MALLOC( eglist, sg_nr, struct ilist*, mesh_sseg_adjust_tx);
  /* init the list */
  for (i=0; i<sg_nr; i++)
    eglist[i]=NULL;


  /* set the number of refines according to parameter insert */
  if (insert==1)
    refines=2;
  else
    refines=0;

  /******************************************************************/
  /* refine curved segments and build an edge list for each segment */
  /******************************************************************/
  for (rcnt=0; rcnt<=refines; rcnt++)
    {
      /* build a list of edges for each segment */
      for (i=0; i<(*m).bd_nr; i++)
	{
	  seg = m->bound[i*bd_w+l_mcBDSSEG];
	  eg  = m->bound[i*bd_w+l_mcBDEDGE];

	  if ((seg!=-1)&&(m->edge[eg*eg_w+l_mcEGCHL1]==-1))
	    {
	      ilist_sorted_insert( &eglist[seg], eg);
	    }
	}
      /* refine */
      if (rcnt<refines)
	{
	  for (seg=0; seg<sg_nr; seg++)
	    {
	      FIDX stype=m->sseg[seg*sg_w+MC2XSGTYPE];

	      if (stype==2)
		{
		  struct ilist *this;
		  
		  this=eglist[seg];

		  /* split each edge of this segment */
		  while (this!=NULL)
		    {
		      err=mesh_split_edge_t1( m, this->data);
		      FUNCTION_FAILURE_HANDLE(err, mesh_split_edge_t1,
					      mesh_sseg_adjust_tx);
		      this=this->next;
		    }
		}
	      /* the edge list is outdated now, so we start over */
	      ilist_free(&eglist[seg]);
	      eglist[seg]=NULL;
	    } /* end loop segments */
	  
	} /* end if still refine */
      
    }/* end loop possible refinements + build edge list */

  /* thevertex,  edge, ... number may have changed up to here */
  vx_nr = m->vx_nr;
  eg_nr = m->eg_nr;
  bd_nr = m->bd_nr;
  sg_nr = m->sg_nr;
  
  TRY_MALLOC( nd_nbs, 2*vx_nr, FIDX, mesh_sseg_adjust_tx);
  TRY_MALLOC( nodes, vx_nr, FIDX, mesh_sseg_adjust_tx);
  TRY_MALLOC( nd_segs, vx_nr, FIDX, mesh_sseg_adjust_tx);

  /* clear nd_segs, it will be set when the nodes of the segment are
     sorted, along with ignore1, and ignore2 */
  for (i=0; i<vx_nr; i++)
    nd_segs[i]=-1;

  for (seg=0; seg<sg_nr; seg++)
    {
      struct ilist *this;
      FIDX seg_nds, tnode, pnode;      

      /**********************************************************/
      /* for each node on the edges mark its neighbouring nodes */
      /**********************************************************/
      this=eglist[seg];

      /* clear nd_nbs and nd_edges*/
      for (i=0; i<2*vx_nr; i++)
	  nd_nbs[i]=-1;

      while (this!=NULL)
	{
	  FIDX eg=this->data;

	  if (type==1)
	    {
	      FIDX n1= m->edge[eg*eg_w+MCT1EGNOD1  ];
	      FIDX n2= m->edge[eg*eg_w+MCT1EGNOD1+1];

	      /* add the edge to nd_edges 
	      if (nd_edges[n1*2]==-1)
		nd_edges[n1*2]=eg;
	      else if (nd_edges[n1*2+1]==-1)
		nd_edges[n1*2+1]=eg;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node on more than two edges\n");
		  return FAIL;
		}		
	      if (nd_edges[n2*2]==-1)
		nd_edges[n2*2]=eg;
	      else if (nd_edges[n2*2+1]==-1)
		nd_edges[n2*2+1]=eg;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node on more than two edges\n");
		  return FAIL;
		}		
	      /* end adding edges to nd_edges */
	      
	      /* add the neighbours */
	      if (nd_nbs[n1*2]==-1)
		nd_nbs[n1*2]=n2;
	      else if (nd_nbs[n1*2+1]==-1)
		nd_nbs[n1*2+1]=n2;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}

	      if (nd_nbs[n2*2]==-1)
		nd_nbs[n2*2]=n1;
	      else if (nd_nbs[n2*2+1]==-1)
		nd_nbs[n2*2+1]=n1;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}
	    } /* end type=1 edges */
	  else if (type==2)
	    {
	      FIDX n1= m->edge[eg*eg_w+MCT2EGNOD1  ];
	      FIDX n2= m->edge[eg*eg_w+MCT2EGNOD1+1];
	      FIDX nm= m->edge[eg*eg_w+MCT2EGNODM];

	      /* add the edge to nd_edges 
	      if (nd_edges[n1*2]==-1)
		nd_edges[n1*2]=eg;
	      else if (nd_edges[n1*2+1]==-1)
		nd_edges[n1*2+1]=eg;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node on more than two edges\n");
		  return FAIL;
		}		
	      if (nd_edges[n2*2]==-1)
		nd_edges[n2*2]=eg;
	      else if (nd_edges[n2*2+1]==-1)
		nd_edges[n2*2+1]=eg;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node on more than two edges\n");
		  return FAIL;
		}
	      if (nd_edges[nm*2]==-1)
		nd_edges[nm*2]=eg;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "midnode on more than one edge\n");
		  return FAIL;
		}			
	      /* end adding edges to nd_edges */
		    
	      /* add the neighbours */
	      if (nd_nbs[n1*2]==-1)
		nd_nbs[n1*2]=nm;
	      else if (nd_nbs[n1*2+1]==-1)
		nd_nbs[n1*2+1]=nm;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}

	      if (nd_nbs[n2*2]==-1)
		nd_nbs[n2*2]=nm;
	      else if (nd_nbs[n2*2+1]==-1)
		nd_nbs[n2*2+1]=nm;
	      else 
		{
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}
	      /* the midnode really should have only two neighbours */
	      nd_nbs[nm*2  ]=n1;
	      nd_nbs[nm*2+1]=n2;
	    } /* end type=2 edges */
	  else
	    {
	      fprintf(stderr,"mesh_sseg_adjust_tx: "
		      "unknown type in node neigbours search\n");
	      return FAIL;
	    }

	  this=this->next;
	}


      /**********************************************************/
      /* order the segment                                      */
      /**********************************************************/
      ignore1=-1;                         /* no nodes to be ignored yet */
      ignore2=-1;
      seg_nds=0;                          /* nodes of this segment */
      pnode=-2;                           /* previous node */
      tnode=m->sseg[seg*sg_w+MC2XSGNOD1]; /* this node     */
      while ((tnode>=0)&&(seg_nds<vx_nr))
	{
	  FIDX nb;

	  nb=nd_nbs[tnode*2  ];
	  /* if this neighbour is the previous node, use other
	     neighbour */
	  if (nb==pnode)
	    nb=nd_nbs[tnode*2+1];

	  /* check if this node has already been set */
	  if (nd_segs[tnode]==-1)
	    { /* has not been set yet, mark as set, continue */
	      nd_segs[tnode]=seg;
	    }
	  else
	    { /* has already been set, mark to be ignored */
	      if (ignore1==-1)
		ignore1=seg_nds;
	      else if (ignore2==-1)
		ignore2=seg_nds;
	      else
		{ /* something is wrong, we've ran out of ignores */
		  fprintf(stderr,"mesh_sseg_adjust_tx: "
			  "node %"dFIDX" in > 2 segments\n",  tnode);
		  return FAIL;
		}
	    }
	  
	  /* add this node to the node list */
	  nodes[seg_nds]=tnode;
	  seg_nds++;

	  /* continue with the neighbour */
	  pnode=tnode;
	  tnode=nb;
	}
	
      /* printf("seg=%"dFIDX",  found %"dFIDX" nodes\n",  seg,  seg_nds); /* */
	
      /* check we didn't run into trouble */
      if (seg_nds>=vx_nr)
	{
	  fprintf(stderr,"mesh_sseg_adjust_tx: " 
		  "node ordering did not terminate\n");
	  return FAIL;
	}
      /* nodes holds now a list of the seg_nds nodes of this segment */

      /**********************************************************/
      /* adjust the nodes on the ordered segment or             */
      /* eval the derivatives                                   */
      /**********************************************************/
	
      err=mesh_sseg_adjust_nodes(m, seg, seg_nds, nodes,
 			ignore1, ignore2, l_mcVXSTRT, nI, dIdx, dIdF);
      FUNCTION_FAILURE_HANDLE(err, mesh_sseg_adjust_nodes,
			      mesh_sseg_adjust_tx);

    }/* end loop segments */
    
  /* free local data */
  for (i=0; i<sg_nr; i++)
    {
      ilist_free( &eglist[i]);
      eglist[i]=NULL;
    }
  free(eglist);
  free(nd_nbs);
  free(nodes);
  free(nd_segs);

  return SUCCESS;
}




/*FUNCTION*/
int mesh_sseg_adjust_nodes( struct mesh *m, FIDX seg, FIDX seg_nds,
			    FIDX *nodes, FIDX ignore1, 
			    FIDX ignore2, FIDX l_mcVXSTRT, 
			    FIDX nI, struct vector *dIdx,
			    struct vector *dIdF
/* Adjusts the nodes listed in order in nodes according to shape
   segment seg to match the current parameters or evaluates the
   derivatives of the node positions and applies the chain rule to
   evlauate dIdF where x=x(F), and dIdx is given. 

   In/Out: m       - the mesh, vertices are adjusted (moved), 
                     only modified if dIdF==NULL
           dIdF    - chain rule is applied, adding the contributions
                     of this segment, has to be initialised outside!,
		     ignored if dIdF==NULL, otherwise, dIdF[i].V[j] is
		     the derivative of performance criterion I[i] wrt
		     spar[j]

   Input:  seg     - id of the segment
	   seg_nds - number of nodes in the list nodes
           nodes   - an ordered list of nodes
	   ts      - t-value for each node
	   ignore1 - see ignore2
	   ignore2 - pointers into nodes for nodes which are to be
	             ignored, because they are set in another segment
	   l_mcVXSTRT
	           - start of the coordinates in a vertex entry in
	             this mesh
           nI      - number of performance criteria, size of dIdF and
                     dIdx, ignored if dIdF==NULL
           dIdx    - derivatives of performance criteria wrt node
                     positions, i.e. dIdx[i].V[d*vx_nr+j] is
                     derivative of criterion I[i] (i=0..nI) wrt the
                     d-th coordinate of node j, ignored if dIdF==NULL


   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
			 ){
  FIDX i, j, d;
  
  const FIDX vx_w=m->vx_w, sg_w=m->sg_w, sp_w=m->sp_w, vx_nr=m->vx_nr;
  const FIDX dim=m->dim;

  /* first set the t-values of first and last edge */	
  (*m).st[nodes[0]]=0.0;
  (*m).st[nodes[seg_nds-1]]=1.0;				 
  for (i=1; i<seg_nds; i++)
    if ( (*m).st[nodes[i-1]] >= (*m).st[nodes[i]])
      {
	printf("mesh_sseg_adjust_nodes: invalid t-values!\n");
	for (j=0; j<seg_nds; j++) printf("t%"dFIDX" = %f (node %"dFIDX")\n",j,(*m).st[nodes[j]], nodes[j]);
	return FAIL;
      }				 
      
  switch (m->sseg[seg*sg_w+MC2XSGTYPE])
    {
    case 1: /* line */
      {
	if (dIdF==NULL)
	  {
	    double lpar[4];
	    
	    for (i=0; i<4; i++)
	      lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

	    for (i=0; i<seg_nds; i++)
	      {
		const FIDX nd=nodes[i];
		const double t=(*m).st[nodes[i]];
		const double omt=1.0-t;

		if ((i!=ignore1)&&(i!=ignore2))
		  for (d=0; d<dim; d++)
		    {
		      m->vertex[nd*vx_w+l_mcVXSTRT+d]=
			omt*lpar[  d] +  t* lpar[2+d];
		    }
	      }
	  }
	else /* if deriv!=NULL */
	  {
	    for (j=0; j<nI; j++) /* loop criteria */
	      {
		double dlpar[4];
	    
		for (i=0; i<4; i++)
		  dlpar[i]= 0.0;

		for (i=0; i<seg_nds; i++)
		  {
		    const FIDX nd=nodes[i];
		    const double t=(*m).st[nodes[i]];
		    const double omt=1.0-t;

		    if ((i!=ignore1)&&(i!=ignore2))
		      for (d=0; d<dim; d++)
			{
			  dlpar[  d]+=omt*dIdx[j].V[d*vx_nr+nd];
			  dlpar[2+d]+=t*dIdx[j].V[d*vx_nr+nd];
			}
		  } /* end loop segment nodes / t */
		
		/* add up the derivative */
		for (i=0; i<4; i++)
		  dIdF[j].V[m->sseg[seg*sg_w+MC2XSGPAR1+i]]+=dlpar[i];

	      } /* end loop criteria */
	  } /* end if deriv!=NULL */
      }
      break;
    case 2: /* b-spline */
      {
	if (dIdF==NULL)
	  {
	    double lpar[8];
	    
	    for (i=0; i<8; i++)
	      lpar[i]= m->spar[m->sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

	    /* the sxi and syi need to be adjusted for easy evaluation */
	    /* the first two direction need just adding the point, to give
	       control points */
	    lpar[4]+=lpar[0];
	    lpar[5]+=lpar[1];
	    /* the later two require turning around and adding the point */
	    lpar[6]=lpar[2]-lpar[6];
	    lpar[7]=lpar[3]-lpar[7]; 
	    for (i=0; i<seg_nds; i++)
	      {
		const FIDX nd=nodes[i];
		const double t=(*m).st[nodes[i]];
		const double omt=1.0-t;
		
		if ((i!=ignore1)&&(i!=ignore2))
		  for (d=0; d<dim; d++)
		    {
		      double d11, d12, d13, d21, d22;
		
		      /* eval according to de Casteljau's algorithm */
		      d11=omt*lpar[  d]+t*lpar[4+d];
		      d12=omt*lpar[4+d]+t*lpar[6+d];
		      d13=omt*lpar[6+d]+t*lpar[2+d];
		      d21=omt*d11+t*d12;
		      d22=omt*d12+t*d13;
		      m->vertex[nd*vx_w+l_mcVXSTRT+d]=
			omt*d21+t*d22;
		    }
	      } /* end loop segment nodes / t */
	  }
	else /* if (deriv!=NULL) */
	  {
	    for (j=0; j<nI; j++) /* loop over criteria */
	      {
		double dlpar[8];
		for (i=0; i<8; i++)
		  dlpar[i]=0.0;

		for (i=0; i<seg_nds; i++)
		  {
		    const FIDX nd=nodes[i];
		    const double t=(*m).st[nodes[i]];
		    const double omt=1.0-t;

		    double dd21d1,dd21d2,dd21d3,  dd22d2,dd22d3,dd22d4;
		    double dxdl1,dxdl2,dxdl3,dxdl4;

		    /* derivatives of Casteljau's algorithm wrt parmeters */
		    dd21d1 = omt*omt; 
		    dd21d2 = 2*omt*t;
		    dd21d3 = t*t;

		    dd22d2 = omt*omt; 
		    dd22d3 = 2*omt*t;
		    dd22d4 = t*t;

		    dxdl1 = omt*dd21d1           ;
		    dxdl2 = omt*dd21d2 + t*dd22d2;
		    dxdl3 = omt*dd21d3 + t*dd22d3;
		    dxdl4 =              t*dd22d4;

		    if ((i!=ignore1)&&(i!=ignore2))
		      for (d=0; d<dim; d++)
			{
			  const double dIdnode = dIdx[j].V[d*vx_nr+nd];
			  dlpar[ +d] += dIdnode*dxdl1;
			  dlpar[4+d] += dIdnode*dxdl2;
			  dlpar[6+d] += dIdnode*dxdl3;
			  dlpar[2+d] += dIdnode*dxdl4;
			}
		  } /* end loop segment nodes / t */

		/* compensate for the interdependancy (chain rule bwd) */
		dlpar[0]+=dlpar[4];
		dlpar[1]+=dlpar[5];
		
		dlpar[2]+=dlpar[6]; dlpar[6]*=-1.0;
		dlpar[3]+=dlpar[7]; dlpar[7]*=-1.0;

		/* add up */
		for (i=0; i<8; i++)
		  dIdF[j].V[m->sseg[seg*sg_w+MC2XSGPAR1+i]] += dlpar[i];
	      } /* end loop over criteria */
	  } /* end if deriv!=NULL */
      } /* end b-spline block */
      break;
    default:
      fprintf(stderr,"mesh_sseg_adjust_nodes: "
	      "unknown segment type\n");
      return FAIL;
    }
  return SUCCESS;
}





/*FUNCTION*/
int mesh_more_vertices ( struct mesh *m, FIDX need
/* increases the number of vertices that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional vertices that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX vx_w, vx_max, newmax;

  vx_w=(*m).vx_w;
  vx_max=(*m).vx_max;

  /* just get n times as much */
  newmax=4*vx_max;
  if (newmax<vx_max+need) {newmax=vx_max+need;}

  TRY_REALLOC( (*m).vertex, newmax*vx_w, double, mesh_more_vertices);
			      
  (*m).vx_max=newmax;

  return SUCCESS;
}




/*FUNCTION*/
int mesh_more_edges ( struct mesh *m , FIDX need
/* increases the number of edges that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional edges that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX eg_w, eg_max, newmax;

  eg_w=(*m).eg_w;
  eg_max=(*m).eg_max;

  /* just get n times as much */
  newmax=4*eg_max;
  if (newmax<eg_max+need)
    {
      newmax=eg_max+need;
    }
  TRY_REALLOC( (*m).edge, newmax*eg_w, FIDX, mesh_more_edges);

  (*m).eg_max=newmax;

  return SUCCESS;
}



/*FUNCTION*/
int mesh_more_faces ( struct mesh *m , FIDX need
/* increases the number of faces that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional faces that are
                        required
   In/Out: m          - the mesh 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX fc_w, fc_max, newmax;

  fc_w=(*m).fc_w;
  fc_max=(*m).fc_max;

  /* just get n times as much */
  newmax=4*fc_max;
  if (newmax<fc_max+need) {newmax=fc_max+need;}

  TRY_REALLOC( (*m).face, newmax*fc_w, FIDX, mesh_more_faces);

  (*m).fc_max=newmax;

  return SUCCESS;
}

/*FUNCTION*/
int mesh_more_volumes ( struct mesh *m , FIDX need
/* increases the number of volumes that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional volumes that are
                        required
   In/Out: m          - the mesh 

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX vo_w, vo_max, newmax;

  vo_w=(*m).vo_w;
  vo_max=(*m).vo_max;

  /* just get n times as much */
  newmax=8*vo_max;
  if (vo_max+need>newmax) newmax=vo_max+need;
 
  TRY_REALLOC( (*m).vols, newmax*vo_w, FIDX, mesh_more_volumes);

  (*m).vo_max=newmax;

  return SUCCESS;
}



/*FUNCTION*/
int mesh_more_elems ( struct mesh *m , FIDX need
/* increases the number of elems that can be held in the mesh by
   reallocating memory

   Input:  need       - the number of additional elems that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX el_w, el_max, newmax;

  el_w=(*m).el_w;
  el_max=(*m).el_max;

  /* just get n times as much */
  newmax=4*el_max;
  if (newmax<el_max+need) {newmax=el_max+need;}

  TRY_REALLOC( (*m).elem, newmax*el_w, FIDX, mesh_more_elems);

  (*m).el_max=newmax;

  return SUCCESS;
}


/*FUNCTION*/
int mesh_more_boundary ( struct mesh *m , FIDX need
/* increases the number of boundary entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional boundary entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX bd_w, bd_max, newmax;

  bd_w=(*m).bd_w;
  bd_max=(*m).bd_max;

  /* just get n times as much */
  newmax=2*bd_max;
  if (newmax<bd_max+need) {newmax=bd_max+need;}

  TRY_REALLOC( (*m).bound, newmax*bd_w, FIDX, mesh_more_boundary);

  (*m).bd_max=newmax;

  return SUCCESS;
}

/*FUNCTION*/
int mesh_more_pcsurf ( struct mesh *m , FIDX need
/* increases the number of pcsurf entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional pcsurf entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX ps_w, ps_max, newmax;

  ps_w=(*m).ps_w;
  ps_max=(*m).ps_max;

  /* just get n times as much */
  newmax=3*ps_max;
  if (newmax<ps_max+need) {newmax=ps_max+need;}

  TRY_REALLOC( (*m).pcsurf, newmax*ps_w, FIDX, mesh_more_pcsurf);

  (*m).ps_max=newmax;

  return SUCCESS;
}

/*FUNCTION*/
int mesh_more_pcvol ( struct mesh *m , FIDX need
/* increases the number of pcvol entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional pcvol entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX pv_w, pv_max, newmax;

  pv_w=(*m).pv_w;
  pv_max=(*m).pv_max;

  /* just get n times as much */
  newmax=4*pv_max;
  if (newmax<pv_max+need) {newmax=pv_max+need;}

  TRY_REALLOC( (*m).pcvol, newmax*pv_w, FIDX, mesh_more_pcvol);

  (*m).pv_max=newmax;

  return SUCCESS;
}



/*FUNCTION*/
int mesh_more_hierarchy ( struct mesh *m , FIDX need
/* increases the number of hierarchy entries that can be held in the
   mesh by reallocating memory

   Input:  need       - the number of additional hierarchy entries that
                        are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX hi_w, hi_max, newmax;

  hi_w=(*m).hi_w;
  hi_max=(*m).hi_max;

  /* just get n times as much */
  newmax=4*hi_max;
  if (newmax<hi_max+need) {newmax=hi_max+need;}

  TRY_REALLOC( (*m).hier, newmax*hi_w, FIDX, mesh_more_hierarchy);

  (*m).hi_max=newmax;

  return SUCCESS;
}


/*FUNCTION*/
int mesh_more_elhiers ( struct mesh *m , FIDX need
/* increases the number of elementwise hierarchy entries that can be
   held in the mesh by reallocating memory

   Input:  need       - the number of additional elementwise hierarchy
                        entries that are required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX eh_w, eh_max, newmax;

  eh_w=(*m).eh_w;
  eh_max=(*m).eh_max;

  /* just get n times as much */
  newmax=5*eh_max;
  if (newmax<eh_max+need) {newmax=eh_max+need;}

  TRY_REALLOC( (*m).elhier, newmax*eh_w, FIDX, mesh_more_elhiers);

  (*m).eh_max=newmax;

  return SUCCESS;
}



/*FUNCTION*/
int mesh_more_st ( struct mesh *m, FIDX need
/* increases the number of t-values for the shape segmants that can 
   be held in the mesh by reallocating memory

   Input:  need       - the number of additional vertices that are
                        required
   In/Out: m          - the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		      ){
  FIDX st_max, newmax, i;

  st_max=(*m).st_max;
			      
  /* just get n times as much */
  newmax=4*st_max;
  if ((st_max+need)>newmax) newmax=st_max+need;
  TRY_REALLOC( (*m).st, newmax, double, mesh_more_st);

  for (i=st_max; i<newmax; i++)			      
   (*m).st[i]=-1.0;			      
  
  
  (*m).st_max=newmax;

  return SUCCESS;
}




/*FUNCTION*/
void mesh_free ( struct mesh *m
/* frees the memory internally allocated in the mesh data struct */
		 ){
  (*m).vx_nr  = 0;
  (*m).vx_max = 0;
  free((*m).vertex);
  (*m).vertex = NULL;
			 
  (*m).el_nr  = 0;
  (*m).el_max = 0;
  free((*m).elem);
  (*m).elem = NULL;

  (*m).eg_nr  = 0;
  (*m).eg_max = 0;
  free((*m).edge);
  (*m).edge = NULL;

  (*m).fc_nr  = 0;
  (*m).fc_max = 0;
  free((*m).face);
  (*m).face = NULL;

  (*m).vo_nr  = 0;
  (*m).vo_max = 0;
  free((*m).vols);
  (*m).vols = NULL;

  (*m).hi_nr  = 0;
  (*m).hi_max = 0;
  free((*m).hier);
  (*m).hier = NULL;

  (*m).eh_nr  = 0;
  (*m).eh_max = 0;
  if ( (*m).elhier !=NULL) free((*m).elhier);
  (*m).elhier = NULL;

  (*m).bd_nr  = 0;
  (*m).bd_max = 0;
  free((*m).bound);
  (*m).bound = NULL;

  (*m).ps_nr  = 0;
  (*m).ps_max = 0;
  free((*m).pcsurf);
  (*m).pcsurf = NULL;

  (*m).pv_nr  = 0;
  (*m).pv_max = 0;
  free((*m).pcvol);
  (*m).pcvol = NULL;

  (*m).pc_nr  = 0;
  (*m).pc_max = 0;
  free((*m).pccrit);
  (*m).pccrit = NULL;

  (*m).fu_nr  = 0;
  (*m).fu_max = 0;
  free((*m).func);
  (*m).func = NULL;

  (*m).pa_nr  = 0;
  (*m).pa_max = 0;
  free((*m).para);
  (*m).para = NULL;

  (*m).sg_nr  = 0;
  (*m).sg_max = 0;
  free((*m).sseg);
  (*m).sseg = NULL;
  
  (*m).st_nr  = 0;
  (*m).st_max = 0; 
  free((*m).st);
  (*m).st   = NULL;

  (*m).sp_nr  = 0;
  (*m).sp_max = 0;
  free((*m).spar);
  (*m).spar = NULL;

  (*m).meshgen=-1;
  free((*m).meshgenDATA);
  (*m).meshgenDATA=NULL;

  return;
}


/*FUNCTION*/
void ilist_free( struct ilist **head
/* empties ilist (deletes all entries and defines head to be NULL */
		){
  struct ilist *this, *next;
  
  this = *head;

  while (this != NULL)
    {
      next = this->next;

      free(this);

      this = next;
    }

  *head = NULL;
  
  return;
}


/*FUNCTION*/
int ilist_sorted_insert( struct ilist **head, FIDX newname
/* if the entry newname does not exist in the sorted list, it is
   inserted at the appropriate position */
		){
  struct ilist *this, *next, *hlp;

  /* printf("head=%p    newname=%"dFIDX"\n", *head, newname);
     printf("head->data = %"dFIDX"    head->next=%p\n",
     (*head)->data, (*head)->next);
     printf("this->data = %"dFIDX"    this->next=%p\n", this->data, next); */

  if ( *head == NULL)
    {
      /* first item in the list anyway */
      TRY_MALLOC( *head, 1, struct ilist, ilist_sorted_insert);
      
      (*head)->data = newname;
      (*head)->next = NULL;
      
      return SUCCESS;
    }

  /* otherwise we were given a non-empty list */
  this = *head;

  if ( (*head)->data > newname )
    {
      /* even the head has name > newname, so we need to insert right
	 at the head of the list */
      TRY_MALLOC( hlp, 1, struct ilist, ilist_sorted_insert);
	      
      hlp->data = newname;
      hlp->next = (*head);

      (*head) = hlp;
	      
      return SUCCESS;
    }


  /* we are really going inside the list */
  while (this != NULL)
    {
      next = this->next;

      if ( this->data == newname )
	{
	  /* exists alredy */
	  return SUCCESS;
	}

      if ( next != NULL ) 
	{
	  if ( next->data <= newname )
	    {
	      /* next one still <= newname  ==> just keep searching for
		 the appropriate spot */
	      this = next;
	    }
	  else
	    {
	      /* the next one is > newname, so this is the right spot
		 to insert */
	      TRY_MALLOC( hlp, 1, struct ilist, ilist_sorted_insert);
	      
	      hlp ->data = newname;
	      hlp ->next = next;

	      this->next = hlp;
	      
	      return SUCCESS;
	    }
	}
      else /* next == NULL */
	{
	  /* no next one, so append newname at the end */
	  TRY_MALLOC( hlp, 1, struct ilist, ilist_sorted_insert);
	      
	  hlp ->data = newname;
	  hlp ->next = NULL;

	  this->next = hlp;
	  
	  return SUCCESS;
	}

      this = next;
    }

  fprintf(stderr, "ilist_sorted_insert: unexpected end!\n");
  return FAIL;
}




/*FUNCTION*/
int comp_intdouble_d(const void *a1, const void *b1
/* helper function for qsort, compares the double part of the
   intdouble struct, designed for descending order

   Input:  a1      - pointer to intdouble a
           b1      - pointer to intdouble b

   Return: -1   if a.d >  b.d
            0   if a.d == b.d
            1   if a.d <  b.d
*/
		     ){
  struct intdouble *a, *b;

  a=(struct intdouble *) a1;
  b=(struct intdouble *) b1;

  if ((*a).d>(*b).d)
    {
      return -1;
    }
  else if ((*a).d<(*b).d)
    {
      return 1;
    }
  else
    {
      return 0;
    }
}


/*FUNCTION*/
int comp_intdouble_i(const void *a1, const void *b1
/* helper function for qsort, compares the FIDX part of the
   intdouble struct, designed for ascending order

   Input:  a1      - pointer to intdouble a
           b1      - pointer to intdouble b

   Return: -1   if a.i <  b.i
            0   if a.i == b.i
            1   if a.i >  b.i
*/
		     ){
  struct intdouble *a, *b;

  a=(struct intdouble *) a1;
  b=(struct intdouble *) b1;

  if ((*a).i<(*b).i)
    {
      return -1;
    }
  else if ((*a).i>(*b).i)
    {
      return 1;
    }
  else
    {
      return 0;
    }
}



/*FUNCTION*/
int mesh_write_solution_grape_t2(struct mesh *m,struct vector *x,
				FIDX ncompo,
				FIDX namlen, char *pathname
/* writes the mesh and the solution into a GRAPE readable
   file (for visualisation) 
   
   Input:  m         - the mesh
           x         - vector with data associated to the nodes
	   ncompo    - dimension of the solution vector per node which
	               shall be writen to the file, (or number of
	               solution components), the components of the
	               vector x have to be numbered such that
	               x[j*vx_nr+i] gives the j-th component in the
	               i-th node of the mesh
	   namlen    - maximal useable length of name
	   pathname  - pathname for the grape file post0.dat


   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		     ){

  FIDX i, j, k, el_w, vx_w, vx_nr,eg_nr, eg_w,el_nr, eg_id, nd_id,nd0,nd1,nd2, *E2F;
  int flag;

  int *clockwise;

  double d2[2];
  long l6[6];
  long l3[3];
  double p, a,b,c,d,det;

  char *prename, *postname;
  FILE *postfile;


  el_w =(*m).el_w;
  vx_w =(*m).vx_w;
  vx_nr=(*m).vx_nr;
  eg_nr=(*m).eg_nr;
  eg_w=(*m).eg_w;
  el_nr=(*m).el_nr;

  TRY_MALLOC( E2F,2*eg_nr, FIDX, mesh_write_solution_grape_t2);
  TRY_MALLOC( clockwise,el_nr, int, mesh_write_solution_grape_t2);

  /* initialise clockwise with 0 */
  for (i=0; i<el_nr; i++)
    {
      clockwise[i] = 0;
    }

  if (ncompo*vx_nr!=(*x).len)
    {
      fprintf( stderr,
	       "\nmesh_write_solution_grape_t2: nodenr in mesh and "
	       "dimension of vector must agree!\n");
      return FAIL;
    }

  /* prepare the names */
  TRY_MALLOC( prename, (namlen+8), char, 
	      mesh_write_solution_grape_t2 );
  TRY_MALLOC( postname, (namlen+9), char, 
	      mesh_write_solution_grape_t2 );
 
  strncpy( prename, pathname, namlen );
  strncpy( postname, pathname, namlen );

  /* make sure no overread happens */
  prename[namlen-1]='\0';
  postname[namlen-1]='\0';

  /* append ".dat" */
  strcat( prename, "pre0.dat" );
  strcat( postname, "post0.dat" );

  /* clear the buffer */
  fflush(stdout);
  
  /* open the post file */
  postfile=fopen(postname, "w");
  if (postfile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_grape_t2: error opening file %s\n",
	      postname);
      return FAIL;
    }

  /* write the header */
  flag = 1;    /* Flag=1 -> write geometry */
  fwrite(&flag,1,sizeof(int),postfile);
  fwrite(&vx_nr,1,sizeof(FIDX),postfile);
  fwrite(&el_nr,1,sizeof(FIDX),postfile);

  for (i=0; i<vx_nr; i++)
    {
      /* write the node positions */
      for (j=0; j<ncompo-1; j++)
	{
	  d2[j] = (*m).vertex[i*vx_w+MCT2VXSTRT+j];
	}
      fwrite(&d2,1,2*sizeof(double),postfile);

      /* write the velocities */
      for (j=0; j<ncompo-1; j++)
	{
	  d2[j] = (*x).V[j*vx_nr+i];
	}
      fwrite(&d2,1,2*sizeof(double),postfile);
     
      /* write the pressure */
      p = (*x).V[(ncompo-1)*vx_nr+i];
      fwrite(&p,1,sizeof(double),postfile);
    }

  putchar('.');
  fflush(stdout);

  /* write nodes of each element in counter-clockwise order */
  for (i=0; i<el_nr; i++)
    {
      for (j=0; j<6; j++)
	{
	  l6[j] = (*m).elem[i*el_w+MCT2ELNOD1+j];
	}
      nd0 = (*m).elem[i*el_w+MCT2ELNOD1+0];
      nd1 = (*m).elem[i*el_w+MCT2ELNOD1+1];
      nd2 = (*m).elem[i*el_w+MCT2ELNOD1+2];

      a = (*m).vertex[nd1*vx_w+MCT2VXSTRT+0] - (*m).vertex[nd0*vx_w+MCT2VXSTRT+0]; /* x(nd1) - x(nd0) */
      b = (*m).vertex[nd2*vx_w+MCT2VXSTRT+0] - (*m).vertex[nd0*vx_w+MCT2VXSTRT+0]; /* x(nd2) - x(nd0) */
      c = (*m).vertex[nd1*vx_w+MCT2VXSTRT+1] - (*m).vertex[nd0*vx_w+MCT2VXSTRT+1]; /* y(nd1) - y(nd0) */
      d = (*m).vertex[nd2*vx_w+MCT2VXSTRT+1] - (*m).vertex[nd0*vx_w+MCT2VXSTRT+1]; /* y(nd2) - y(nd0) */
      det = (a*d-b*c);

      if (det < 0.0) /* no counter-clockwise order  */
	{
	  /* swap vertices 0 and 1 */
	  nd0 = l6[0];
	  l6[0] = l6[1];
	  l6[1] = nd0;
	  /* and swap vertices 4 and 5 */
	  nd0 = l6[4];
	  l6[4] = l6[5];
	  l6[5] = nd0;
	  /* mark this element as clockwise oriented for later */
	  clockwise[i] = 1;
	}
      fwrite(&l6,1,6*sizeof(long),postfile);
    }

  /* fill E2F */
  for (i=0; i<2*eg_nr; i++)
    {
      E2F[i] = -1;
    }
  for (i=0; i<el_nr; i++)
    {
      for (j=0; j<3; j++)
	{
	  eg_id = (*m).face[i*(*m).fc_w+MCT2FCEDG1+j];
	  if (E2F[2*eg_id]==-1) /* no neighbor face found yet */
	    {
	      E2F[2*eg_id] = i;
	    }
	  else  /* already one neighbor face determined */
	    {
	      E2F[2*eg_id+1] = i;
	    }
	}
    }

  /* write neighbor faces of each edge of each element */
  for (i=0; i<el_nr; i++) /* loop over all elements */
    {
      for (j=0; j<3; j++) /* loop over all edges */ 
	{
	  for (k=0; k<3; k++) /* sort edges according to opposing node */
	    {
	      eg_id = (*m).face[i*(*m).fc_w+MCT2FCEDG1+k];
	      nd_id = (*m).elem[i*el_w+MCT2ELNOD1+j];
	      /* not the right edge, try next one */
	      if ((nd_id==(*m).edge[eg_id*eg_w+MCT2EGNOD1  ]) ||
		  (nd_id==(*m).edge[eg_id*eg_w+MCT2EGNOD1+1]))
		{
		  continue;
		}
	      else /* found right edge */
		{
		  if (E2F[2*eg_id]==i) /* find the edge's neighbor face */
		    {
		      l3[j] = E2F[2*eg_id+1] ;
		    }
		  else
		    {
		      l3[j] = E2F[2*eg_id] ;
		    }
		}
	    }
	}
      /* if this element has clockwise orientation -> swap faces */
      if (clockwise[i]==1)
	{
	  nd0 = l3[0];
	  l3[0] = l3[1];
	  l3[1] = nd0;
	}
      fwrite(&l3,1,3*sizeof(long),postfile);
    }

  putchar('.');
  fflush(stdout);

  /* close post file */
  fclose(postfile);

  /* free some memory  */
  free(postname);
  free(E2F);
  free(clockwise);


#ifdef ADAPTIVEMESH

  /* open the pre file */
  prefile=fopen(prename, "w");
  if (prefile==NULL)
    {
      fprintf(stderr,
	      "mesh_write_solution_grape_t2: error opening file %s\n",
	      prename);
      return FAIL;
    }

  /* write the header */
  flag = 0;   /* Flag=0 -> write no geometry */
  fwrite(&flag,1,sizeof,prefile); 
  fwrite(&vx_nr,1,sizeof(FIDX),prefile);
  fwrite(&el_nr,1,sizeof(FIDX),prefile);

  for (i=0; i<vx_nr; i++)
    {
      /* write the velocities */
      for (j=0; j<ncompo-1; j++)
	{
	  d2[j] = (*x).V[j*vx_nr+i];
	}
      fwrite(&d2,1,2*sizeof(double),prefile);
     
      /* write the pressure */
      p = (*x).V[(ncompo-1)*vx_nr+i];
      fwrite(&p,1,sizeof(double),prefile);
    }

  putchar('.');
  fflush(stdout);

  /* close pre file */
  fclose(prefile);

#endif

  free(prename);

  return SUCCESS;
}




/*FUNCTION*/
int mesh_spline_t_init_tx(struct mesh *m, int type
/* creates st (t-values (parameter of spline segment)
   for the shape parameters) for the nodes by distributing the
   t-values uniformly on [0,1] has to be applied before the
   adjust_sseg, 
   (so even before the mesh generator) 

   Input/Output 
           m       - the mesh, m.st will be generated

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
			  ){

  FIDX dim, seg, vx_nr, eg_nr, sg_nr, eg_w, sg_w, bd_w, bd_nr, i, j, eg;
  int  err;			      
			      
  struct ilist **eglist;  /* list of edges for each segment */
  FIDX *nd_nbs;           /* nd_nbs[i*2+j]=k,  j=1,2:  node i has neigbour k */
  FIDX *nodes;            /* list of nodes on the segment */
  FIDX *nd_segs;          /* details the segment in which a node has been set,
		             -1 if it hasn't yet been set */
  FIDX ignore1, ignore2;  /* at most two nodes may be necessary to
			     ignore because they have been set
			     previously */

  FIDX l_mcBDSSEG, l_mcBDEDGE, l_mcEGCHL1, l_mcVXSTRT,
	 l_mcEGNOD1, l_MCEGBND;

  dim=(*m).dim;
  vx_nr = (*m).vx_nr;
  eg_nr = (*m).eg_nr;
  eg_w = (*m).eg_w;
  sg_nr = (*m).sg_nr;
  sg_w = (*m).sg_w;
  bd_w = (*m).bd_w;
  bd_nr = (*m).bd_nr;

  TRY_MALLOC( nd_nbs, 2*vx_nr, FIDX, mesh_spline_t_init_tx);
  TRY_MALLOC( nodes, vx_nr, FIDX, mesh_spline_t_init_tx);
  TRY_MALLOC( nd_segs, vx_nr, FIDX, mesh_spline_t_init_tx);

  if(type!=1)
    { /* mesh_spline_t_init_tx isn't fully implemented for T2 yet */
      fprintf(stderr,"mesh_spline_t_init_tx " 
	"should be called for T1 meshes! (type=%d)\n", type);
      return FAIL;
    }
    
  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT=MCT1VXSTRT;
      l_mcBDSSEG=MCT1BDSSEG;
      l_mcBDEDGE=MCT1BDEDGE;
      l_mcEGCHL1=MCT1EGCHL1;
      l_mcEGNOD1=MCT1EGNOD1;
      l_MCEGBND=MCT1EGBND;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT=MCT2VXSTRT;
      l_mcBDSSEG=MCT2BDSSEG;
      l_mcBDEDGE=MCT2BDEDGE;
      l_mcEGCHL1=MCT2EGCHL1;
      l_mcEGNOD1=MCT2EGNOD1;
      l_MCEGBND=MCT2EGBND;
      break;
    default:
      fprintf(stderr,"mesh_spline_t_init_tx: unknown type=%d\n",type);
      return FAIL;
    }  
    
  TRY_MALLOC( eglist, sg_nr, struct ilist*, mesh_spline_t_init_tx);
  /* init the list */
  for (i=0; i<sg_nr; i++) eglist[i]=NULL;    

  /* build a list of edges for each segment */
  for (i=0; i<bd_nr; i++)
    {
      seg = m->bound[i*bd_w+l_mcBDSSEG];
      eg  = m->bound[i*bd_w+l_mcBDEDGE];

      if ((seg!=-1)&&(m->edge[eg*eg_w+l_mcEGCHL1]==-1))
	{
	  ilist_sorted_insert( &eglist[seg], eg);
	}
    }
  
  /* clear nd_segs, it will be set when the nodes of the segment are
     sorted, along with ignore1, and ignore2 */
  for (i=0; i<vx_nr; i++) nd_segs[i]=-1;

  for (seg=0; seg<sg_nr; seg++)
    {
      struct ilist *this;
      FIDX seg_nds, tnode, pnode;      

      /**********************************************************/
      /* for each node on the edges mark its neighbouring nodes */
      /**********************************************************/
      this=eglist[seg];

      /* clear nd_nbs and nd_edges*/
      for (i=0; i<2*vx_nr; i++)
	nd_nbs[i]=-1;

      while (this!=NULL)
	{
	  FIDX eg=this->data;

	  if ((type==1)||(type==2))
	    {
	      FIDX n1= m->edge[eg*eg_w+l_mcEGNOD1  ];
	      FIDX n2= m->edge[eg*eg_w+l_mcEGNOD1+1];
	      
	      /* add the neighbours */
	      if (nd_nbs[n1*2]==-1)
		nd_nbs[n1*2]=n2;
	      else if (nd_nbs[n1*2+1]==-1)
		nd_nbs[n1*2+1]=n2;
	      else 
		{
		  fprintf(stderr,"mesh_spline_t_init_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}

	      if (nd_nbs[n2*2]==-1)
		nd_nbs[n2*2]=n1;
	      else if (nd_nbs[n2*2+1]==-1)
		nd_nbs[n2*2+1]=n1;
	      else 
		{
		  fprintf(stderr,"mesh_spline_t_init_tx: "
			  "node multiplicity error\n");
		  return FAIL;
		}
	    } /* end type=1 or 2 edges */

	  else
	    {
	      fprintf(stderr,"mesh_spline_t_init_tx: "
		      "unknown type in node neigbours search\n");
	      return FAIL;
	    }

	  this=this->next;
	}


      /**********************************************************/
      /* order the segment                                      */
      /**********************************************************/
      ignore1=-1;                         /* no nodes to be ignored yet */
      ignore2=-1;
      seg_nds=0;                          /* nodes of this segment */
      pnode=-2;                           /* previous node */
      tnode=m->sseg[seg*sg_w+MC2XSGNOD1]; /* this node     */
      while ((tnode>=0)&&(seg_nds<vx_nr))
	{
	  FIDX nb;
	  nb=nd_nbs[tnode*2  ];
	  /* if this neighbour is the previous node, use other
	     neighbour */
	  if (nb==pnode)
	    nb=nd_nbs[tnode*2+1];

	  /* check if this node has already been set */
	  if (nd_segs[tnode]==-1)
	    { /* has not been set yet, mark as set, continue */
	      nd_segs[tnode]=seg;
	    }
	  else
	    { /* has already been set, mark to be ignored */
	      if (ignore1==-1)
		ignore1=seg_nds;
	      else if (ignore2==-1)
		ignore2=seg_nds;
	      else
		{ /* something is wrong, we've ran out of ignores */
		  fprintf(stderr,"mesh_spline_t_init_tx: "
			  "node %"dFIDX" in > 2 segments\n",  tnode);
		  return FAIL;
		}
	    }
	  
	  /* add this node to the node list */
	  nodes[seg_nds]=tnode;
	  seg_nds++;

	  /* continue with the neighbour */
	  pnode=tnode;
	  tnode=nb;
	}

      /* check we didn't run into trouble */
      if (seg_nds>=vx_nr)
	{
	  fprintf(stderr,"mesh_spline_t_init_tx: " 
		  "node ordering did not terminate\n");
	  return FAIL;
	}
      /* nodes holds now a list of the seg_nds nodes of this segment */

      if (type==1)
        {
          double dt, x[3];
	  dt=1.0/(seg_nds-1);  		
	  for(i=0; i<seg_nds; i++)
	    {
	      (*m).st[nodes[i]]=i*dt;	
	      err=mesh_seg_node_eval(m, seg, (*m).st[nodes[i]], x);
	      FUNCTION_FAILURE_HANDLE(err, 
		mesh_seg_node_eval, mesh_seg_node_eval);
	      for(j=0; j<dim; j++) 
		(*m).vertex[nodes[i]*(*m).vx_w+l_mcVXSTRT+j]=x[j];
	    }	  
	} /* T1 done */	      
      else 
	{
	  fprintf(stderr,"mesh_spline_t_init_tx: " 
	    "Unknown element type = %d\n", type);
	  return FAIL;
	}	
    } /* end loop over segments */
    
  /* free local data */
  for (i=0; i<sg_nr; i++)
    {
      ilist_free( &eglist[i]);
      eglist[i]=NULL;
    }
  free(eglist);
  free(nd_nbs);
  free(nodes);
  free(nd_segs);

  return SUCCESS;			      
}






/*FUNCTION*/
int mesh_spline_t_fix_new_nodes(struct mesh *m, int type
/* After the mesh generator, there may be new nodes on the boundary
   without a valid t-value for the shape segments. The m.st for these 
   nodes will be set.


  Input/Output 
          m        - the mesh, m.st_t will be adjusted

  Input:  type     - type of the mesh

  Return: SUCCESS  - success
          FAIL     - failure, see error message, output will not be
                     valid
*/
		      ){

  FIDX dim, seg, vx_nr, eg_nr, sg_nr, eg_w, sg_w, bd_w, bd_nr, i, eg;
  int  err;			      
			      
  struct ilist **eglist; /* list of edges for each segment */
  FIDX *nd_nbs;  /* nd_nbs[i*2+j]=k,  j=1,2:  node i has neigbour k */
  FIDX *nodes;   /* list of nodes on the segment */
  FIDX *nd_segs; /* details the segment in which a node has been set,
		   -1 if it hasn't yet been set */
  FIDX ignore1, ignore2; /* at most two nodes may be necessary to
			   ignore because they have been set
			   previously */
    
  FIDX l_mcBDSSEG, l_mcBDEDGE, l_mcEGCHL1, l_mcVXSTRT,
	 l_mcEGNOD1, l_MCEGBND;

  dim=(*m).dim;
  vx_nr = (*m).vx_nr;
  eg_nr = (*m).eg_nr;
  eg_w = (*m).eg_w;
  sg_nr = (*m).sg_nr;
  sg_w = (*m).sg_w;
  bd_w = (*m).bd_w;
  bd_nr = (*m).bd_nr;

  TRY_MALLOC( nd_nbs, 2*vx_nr, FIDX, mesh_spline_t_fix_new_nodes);
  TRY_MALLOC( nodes, vx_nr, FIDX, mesh_spline_t_fix_new_nodes);
  TRY_MALLOC( nd_segs, vx_nr, FIDX, mesh_spline_t_fix_new_nodes);
 

  if ((*m).vx_nr>(*m).st_max)
    {
      fprintf(stderr, "mesh_spline_t_fix_new_nodes: vx_nr=%"dFIDX">st_nr=%"dFIDX"\n",
	      (*m).vx_nr,(*m).st_nr);
      return FAIL;
    }

  /* all t-values will be set */
  (*m).st_nr = (*m).vx_nr;   

  if(type!=1)
    { /* mesh_spline_t_fix_new_nodes isn't fully implemented for T2 yet */
      fprintf(stderr,"mesh_spline_t_fix_new_nodes " 
	"should be called for T1 meshes! (type=%d)\n", type);
      return FAIL;
    }
    
  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT=MCT1VXSTRT;
      l_mcBDSSEG=MCT1BDSSEG;
      l_mcBDEDGE=MCT1BDEDGE;
      l_mcEGCHL1=MCT1EGCHL1;
      l_mcEGNOD1=MCT1EGNOD1;
      l_MCEGBND=MCT1EGBND;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT=MCT2VXSTRT;
      l_mcBDSSEG=MCT2BDSSEG;
      l_mcBDEDGE=MCT2BDEDGE;
      l_mcEGCHL1=MCT2EGCHL1;
      l_mcEGNOD1=MCT2EGNOD1;
      l_MCEGBND=MCT2EGBND;
      break;
    default:
      fprintf(stderr,"mesh_spline_t_fix_new_nodes: unknown type=%d\n",type);
      return FAIL;
    }  
    
  TRY_MALLOC( eglist, sg_nr, struct ilist*, mesh_spline_t_fix_new_nodes);
  /* init the list */
  for (i=0; i<sg_nr; i++) eglist[i]=NULL;    

  /* build a list of edges for each segment */
  for (i=0; i<bd_nr; i++)
    {
      seg = m->bound[i*bd_w+l_mcBDSSEG];
      eg  = m->bound[i*bd_w+l_mcBDEDGE];

      if ((seg!=-1)&&(m->edge[eg*eg_w+l_mcEGCHL1]==-1))
	{
	  ilist_sorted_insert( &eglist[seg], eg);
	}
    }
  
  /* clear nd_segs, it will be set when the nodes of the segment are
     sorted, along with ignore1, and ignore2 */
  for (i=0; i<vx_nr; i++) nd_segs[i]=-1;

  for (seg=0; seg<sg_nr; seg++)
    {
      struct ilist *this;
      FIDX seg_nds, tnode, pnode;      
	  int test;

      /**********************************************************/
      /* for each node on the edges mark its neighbouring nodes */
      /**********************************************************/
      this=eglist[seg];

      /* clear nd_nbs and nd_edges*/
      for (i=0; i<2*vx_nr; i++)
	nd_nbs[i]=-1;

      while (this!=NULL)
	{
	  FIDX eg=this->data;

	  if ((type==1)||(type==2))
	    {
	      FIDX n1= m->edge[eg*eg_w+l_mcEGNOD1  ];
	      FIDX n2= m->edge[eg*eg_w+l_mcEGNOD1+1];
	      
	      /* add the neighbours */
	      if (nd_nbs[n1*2]==-1)
		nd_nbs[n1*2]=n2;
	      else if (nd_nbs[n1*2+1]==-1)
		nd_nbs[n1*2+1]=n2;
	      else 
		{
		  fprintf(stderr,"mesh_spline_t_fix_new_nodes: "
			  "node multiplicity error\n");
		  return FAIL;
		}

	      if (nd_nbs[n2*2]==-1)
		nd_nbs[n2*2]=n1;
	      else if (nd_nbs[n2*2+1]==-1)
		nd_nbs[n2*2+1]=n1;
	      else 
		{
		  fprintf(stderr,"mesh_spline_t_fix_new_nodes: "
			  "node multiplicity error\n");
		  return FAIL;
		}
	    } /* end type=1 or 2 edges */

	  else
	    {
	      fprintf(stderr,"mesh_spline_t_fix_new_nodes: "
		      "unknown type in node neigbours search\n");
	      return FAIL;
	    }

	  this=this->next;
	}


      /**********************************************************/
      /* order the segment                                      */
      /**********************************************************/
      ignore1=-1;                         /* no nodes to be ignored yet */
      ignore2=-1;
      seg_nds=0;                          /* nodes of this segment */
      pnode=-2;                           /* previous node */
      tnode=m->sseg[seg*sg_w+MC2XSGNOD1]; /* this node     */
      while ((tnode>=0)&&(seg_nds<vx_nr))
	{
	  FIDX nb;
	  nb=nd_nbs[tnode*2  ];
	  /* if this neighbour is the previous node, use other
	     neighbour */
	  if (nb==pnode)
	    nb=nd_nbs[tnode*2+1];

	  /* check if this node has already been set */
	  if (nd_segs[tnode]==-1)
	    { /* has not been set yet, mark as set, continue */
	      nd_segs[tnode]=seg;
	    }
	  else
	    { /* has already been set, mark to be ignored */
	      if (ignore1==-1)
		ignore1=seg_nds;
	      else if (ignore2==-1)
		ignore2=seg_nds;
	      else
		{ /* something is wrong, we've ran out of ignores */
		  fprintf(stderr,"mesh_spline_t_fix_new_nodes: "
			  "node %"dFIDX" in > 2 segments\n",  tnode);
		  return FAIL;
		}
	    }
	  
	  /* add this node to the node list */
	  nodes[seg_nds]=tnode;
	  seg_nds++;

	  /* continue with the neighbour */
	  pnode=tnode;
	  tnode=nb;
	}

      /* check we didn't run into trouble */
      if (seg_nds>=vx_nr)
	{
	  fprintf(stderr,"mesh_spline_t_fix_new_nodes: " 
		  "node ordering did not terminate\n");
	  return FAIL;
	}
      /* check if there are uninitialized t-values ( <0.0)*/
      (*m).st[nodes[0]]=0.0;
      (*m).st[nodes[seg_nds-1]]=1.0;
      test=0;	
      for (i=0; i<seg_nds; i++)
	  if ((*m).st[nodes[i]]<0.0) test=1;
      if (test==1)
	{ /* there are nodes without a t-value yet */
	  FIDX n1=0;
	  while  (n1<seg_nds-1)
	    { /* find those nodes without t-values */
	      FIDX n2=n1+1;
	      while ((n2<seg_nds)&&((*m).st[nodes[n2]]<0.0))
		n2++;
	      if (n2>(seg_nds-1))
		{
		  fprintf(stderr,"mesh_spline_t_fix_new_nodes: " 
			  "end of segment has no t-value set???\n");
		  return FAIL;
		}
	      /* now the nodes n1+1...n2-1 have no t-value  */
	      for (i=n1+1; i<n2; i++)
	        {
		  err=mesh_spline_set_t(m, seg, 
			nodes[n1], nodes[n2], nodes[i], 0, 2);
		  FUNCTION_FAILURE_HANDLE( err, mesh_get_st,
			mesh_spline_t_fix_new_nodes);
		}
	      /* set n1 to n2 to move on to the next part */	
	      n1=n2;	
            }
	}
	
    } /* end loop over segments */

    
  /* free local data */
  for (i=0; i<sg_nr; i++)
    {
      ilist_free( &eglist[i]);
      eglist[i]=NULL;
    }
  free(eglist);
  free(nd_nbs);
  free(nodes);
  free(nd_segs);

  return SUCCESS;			      
}


/*FUNCTION*/
int mesh_spline_set_t( struct mesh *m, FIDX seg, FIDX n1, FIDX n2, FIDX nm,
		       int adjust, int type
/* determine the t-value for the bezier spline for the midnode of an edge
   by bisection to make sure that the node is near the mid (so the 
   interpolation/restriction will work better). The node position can be also
   adjusted

   INPUT/Output:
	    m		- the mesh:m.st[midnode] and 
	                  (optional) m.vertex[midonde] will be set
   INPUT:   seg	        - the shape segment
	    n1,
	    n2     	- start and end node of the edge under consideration
	    nm		- the midnode
	    adjust	- the vertex entry will be adjusted if adjust==1
			 
   Return:  SUCCESS 	- success
            FAIL   	- failure, see error message, output will not be
                          valid
*/
			 ){
  FIDX dim, i, j, done;
  int err;
  FIDX l_mcBDSSEG, l_mcBDEDGE, l_mcVXSTRT,
	 l_mcEGNOD1, l_MCEGBND, l_MCBDSSEG;
  double t, h, tol; 	/* stepsize and tolerance */
  double d1, d2;  /* distances d1=|x1-xm|² and d2=|x2-xm|² */
  double omega; 	/* relation factor between initial d1 and d2 */  
  double x1[3], x2[3], xm[3];
  int max_it;

  dim=(*m).dim;
  if (dim!=2)
    {
      printf("Error: mesh_spline_set_t is only implemented for 2d!\n");
      return FAIL;
    }
    
  switch (type)
    {
    case 1: /* T1 mesh */
      l_mcVXSTRT=MCT1VXSTRT;
      l_mcBDSSEG=MCT1BDSSEG;
      l_mcBDEDGE=MCT1BDEDGE;
      l_mcEGNOD1=MCT1EGNOD1;
      l_MCEGBND =MCT1EGBND;
      l_MCBDSSEG=MCT1BDSSEG;
      break;
    case 2: /* T2 mesh */
      l_mcVXSTRT=MCT2VXSTRT;
      l_mcBDSSEG=MCT2BDSSEG;
      l_mcBDEDGE=MCT2BDEDGE;
      l_mcEGNOD1=MCT2EGNOD1;
      l_MCEGBND =MCT2EGBND;
      l_MCBDSSEG=MCT2BDSSEG;
     break;
    default:
      fprintf(stderr,"mesh_spline_set_t: unknown type=%d\n", type);
      return FAIL;
    }  
    
  /* Check if node-numbers are in range */
  if (  (((n1<0)||(n2<0)) || ((nm<0)||(n1>=(*m).vx_nr)))
	|| (((n2>=(*m).vx_nr)||(nm>=(*m).vx_nr)) 
	    ||((n1>=(*m).st_nr)||(n2>=(*m).st_nr)))
	||(nm>=(*m).st_nr) )
    {
      fprintf(stderr, "Error in mesh_spline_set_t: "
	      "node numbers don't match mesh\n"
	      "  n1=%"dFIDX", n2=%"dFIDX", nm=%"dFIDX", vx_nr=%"dFIDX", st_nr=%"dFIDX"\n", n1, n2, nm, 
	      (*m).vx_nr,(*m).st_nr);
      return FAIL;
    }

  /* get nodes */
  for (i=0; i<dim; i++)
    {	  
      x1[i]=(*m).vertex[n1*(*m).vx_w+l_mcVXSTRT+i];
      x2[i]=(*m).vertex[n2*(*m).vx_w+l_mcVXSTRT+i];
      xm[i]=(*m).vertex[nm*(*m).vx_w+l_mcVXSTRT+i];	    
    }
  /* Check if t-values of x1 and x2 are valid */
  if ( ((*m).st[n1]<0.0)||((*m).st[n1]>1.0) )
    {
      fprintf(stderr, "Error in mesh_spline_set_t: "
	      "t-value for n1 is not valid (st=%8.3e)\n", (*m).st[n1]);
      return FAIL;
    }
  if ( ((*m).st[n2]<0.0)||((*m).st[n2]>1.0) )
    {
      fprintf(stderr, "Error in mesh_spline_set_t: "
	      "t-value for n2 is not valid (st=%8.3e)\n", (*m).st[n2]);
      fprintf(stderr, "seg = %"dFIDX" \n", seg);
      return FAIL;
    }
  
  /* get t-values of these nodes */
  if ( ((*m).st[n1]==0.0)||((*m).st[n1]==1.0) )
    {
      if ( n1==(*m).sseg[seg*(*m).sg_w+MC2XSGNOD1] ) (*m).st[n1]=0.0;
      else (*m).st[n1]=1.0;	      
    }
  if ( ((*m).st[n2]==0.0)||((*m).st[n2]==1.0) )
    {
      if ( n2==(*m).sseg[seg*(*m).sg_w+MC2XSGNOD1] ) (*m).st[n2]=0.0;
      else (*m).st[n2]=1.0;	      
    }
  /* init */
  max_it =100;  
  t=0.5*( (*m).st[n1]+(*m).st[n2] );
  h=0.25*( (*m).st[n2]-(*m).st[n1] );
  tol=0.99;
  done=0;
  j=0;
    
  /* determine omega, the ratio of the initial distances */
  d1=0.0;
  d2=0.0;    
  for (i=0; i<dim; i++)
    {	  
      d1+=(x1[i]-xm[i])*(x1[i]-xm[i]);
      d2+=(x2[i]-xm[i])*(x2[i]-xm[i]);	    
    } 
  omega = d1/d2;
     
  /* Bisection to determine t */
  while ( (done!=1)&&(j<max_it) )
    {
      err=mesh_seg_node_eval(m, seg, t, xm);
      FUNCTION_FAILURE_HANDLE(err, mesh_seg_node_eval, mesh_spline_set_t);

      /* calculate distances */  
      d1=0.0;
      d2=0.0;	     
      for (i=0; i<dim; i++)
        {	  
          d1+=(x1[i]-xm[i])*(x1[i]-xm[i]);
          d2+=(x2[i]-xm[i])*(x2[i]-xm[i]);	    
        }
	
      /* adjust d2*=omega for easier calculation */
      d2 *=omega;	
      /* Check if the distances are equal */
      if (  ( d2 > (d1*tol) ) && ( d1 > (d2*tol) )  )
        {  /* xm is okay */
          done=1;
        }   
      else
        {
	  if ( d1 < (d2*tol) ) 
	    {  /* that is xm is too close to x1 */
              t+=h;
            }
          else if ( d2 < (d1*tol) ) 
            {  /* that is xm is too close to x2 */
              t-=h;
            }
	  h*=0.5;
        }
      j++;
    } /* end of bisection */
   
  if (j>max_it-1)
    {
      fprintf(stderr,"mesh_spline_set_t: Bisection method failed\n");
      fprintf(stderr,"bisec: n1=%"dFIDX", n2=%"dFIDX", nm=%"dFIDX", omega=%f\n", 
	      n1, n2, nm, omega); 
      fprintf(stderr,"bisec: t1=%f, t2=%f, t=%f\n", 
	      (*m).st[n1], (*m).st[n2], t );
      fprintf(stderr,"bisec: x1=(%f,%f), x2=(%f,%f), xm=(%f,%f)\n", 
	      x1[0], x1[1],  x2[0], x2[1], xm[0], xm[1] );   
      fprintf(stderr,"bisec: d1=%f, omega*d2=%f, j=%"dFIDX" \n", d1, d2,j);	    
      return FAIL;
    }
    
  /* set t */  
  (*m).st[nm]=t;
    
  if(adjust==1)
    { /* node position will be adjusted */
      for (i=0; i<dim; i++)
	{	  
          (*m).vertex[nm*(*m).vx_w+l_mcVXSTRT+i]=xm[i];   
	}
      
    }
  return SUCCESS;			 
}




/*FUNCTION*/
int mesh_seg_node_eval( struct mesh *m, FIDX seg, 
			double t, double *x
/* evaluates the position of a point on the shape segment according to
   the given t-value

   INPUT:  seg     - the shape segment
	   m	   - the mesh 
	   t	   - t-value for the bezier spline
   OUTPUT: x	   - vector holding the coordinates
			 
   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
			 ){
  FIDX i, d;
  
  const FIDX dim=(*m).dim, sp_w=(*m).sp_w, sg_w=(*m).sg_w;				 
				 
  switch ((*m).sseg[seg*sg_w+MC2XSGTYPE])
    {
    case 1: /* line */
      {
	const double omt=1.0-t;
	double lpar[4];
	for (i=0; i<4; i++)
	  lpar[i]= (*m).spar[(*m).sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

	for (d=0; d<dim; d++)
	  x[d]=omt*lpar[  d] +  t * lpar[2+d];
      }
      break;
    case 2: /* b-spline */
      {
	const double omt=1.0-t;	
	double lpar[8];
	for (i=0; i<8; i++)
	  lpar[i]= (*m).spar[(*m).sseg[seg*sg_w+MC2XSGPAR1+i]*sp_w];

	/* the sxi and syi need to be adjusted for easy evaluation */
	/* the first two directions need just adding the point, to give
	    control points */
	lpar[4]+=lpar[0];
	lpar[5]+=lpar[1];
	/* the later two require turning around and adding the point */
	lpar[6]=lpar[2]-lpar[6];
	lpar[7]=lpar[3]-lpar[7]; 

	for (d=0; d<dim; d++)
	  {
	    double d11, d12, d13, d21, d22;
	    /* eval according to de Casteljau's algorithm */
	    d11=omt*lpar[  d]+t*lpar[4+d];
	    d12=omt*lpar[4+d]+t*lpar[6+d];
	    d13=omt*lpar[6+d]+t*lpar[2+d];
	    d21=omt*d11+t*d12;
	    d22=omt*d12+t*d13;
	    x[d]=omt*d21+t*d22;
	  }
      } /* end b-spline block */
      break;
    default:
      fprintf(stderr,"mesh_seg_node_eval: "
	      "unknown segment type\n");
      return FAIL;
    }    
  return SUCCESS;
}


