/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
linsolve_umfpack.h
*/
#ifndef __LINSOLVE_UMFPACK_H__
#define __LINSOLVE_UMFPACK_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"
#include "linsolve_struct.h"


int linsolver_init(struct linsolver *S
/* initialises S to be empty, such that it can be emptied or deleted without
   problems 
   
   Input:  (none)

   Output: S       - (S is given by reference), is initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int linsolver_free(struct linsolver *S  
/* frees memory in S 
   
   Input:  (none)

   Output: S       - (S is given by reference), is emptied/freed

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
);

int UMFPACK_factorize (struct sparse *A, struct linsolver *sol
/* calls UMFPACK to factorise the sparse matrix A, such equation
   systems with A can be solved by UMFPACK_solve 
   
   Input:  A       - sparse matrix, must be in compressed row storage,
                     and must be invertible

   Output: sol     - (sol is given by reference), 
                     sparse triangulation of A is computed and stored
                     in sol, neccessary fields in sol are allocated,

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int UMFPACK_solve(struct linsolver *sol, struct vector *b, struct vector *x
/* calls UMFPACK to solve a linear system with a previously by
   UMFPACK_factorize factorised sparse matrix
         A x = b
   
   Input:  sol     - sparse triangulation of A as computed by
                     UMFPACK_factorize 
	   b       - right-hand-side vector


   Output: x       - (x is given by reference), 
                     solution vector x of Ax=b, has to have right size
                     beforehand

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_solve_UMFPACK(struct vector *x,
			 struct sparse *K, struct vector *rhs, 
			 struct projector1 *P
/* wrapper to solve the linear system
     K*x=rhs
   with Dirichlet conditions as defined by the projector P

   In/Out: x       - solution vector, the values at the Dirichlet DOFs
                     as defined by the projector P are left unchanged,
                     the rest is set to the solution of the system
                     under the Dirichlet conditions 
	   K       - Stiffness matrix, rows and columns corresponding
                     to Dirichlet DOFs (see projector P) are modified
                     (for simplicity of implementation),
		     K must have symmetric connectivity structure, but
		     entries may be non-symmetric, ideally K should be
		     still in flexible storage format,
	   rhs     - right-hand-side vector, is modified to implement
                     Dirichlet conditions (for simplicity of
                     implementation) 


   Input:  P       - projector defining the Dirichlet DOFs

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_solve_UMFPACK_reuse(struct vector *x,
			       struct sparse *K, 
			       struct vector *rhs, 
			       struct projector1 *P,
			       struct linsolver *umfpackdata,
			       FIDX **isdiri,
			       struct vector *help, 
			       int task
/* wrapper to solve the linear system
     K*x=rhs
   with Dirichlet conditions as defined by the projector P, and allow
   reuse of the triangular factors of K in case multiple systems with
   identical K have to be solved

   In/Out: x       - solution vector, the values at the Dirichlet DOFs
                     as defined by the projector P are left unchanged,
                     the rest is set to the solution of the system
                     under the Dirichlet conditions 
	   K       - Stiffness matrix, rows and columns corresponding
                     to Dirichlet DOFs (see projector P) are modified
                     (for simplicity of implementation),
		     K must have symmetric connectivity structure, but
		     entries may be non-symmetric, ideally K should be
		     still in flexible storage format,
	   rhs     - right-hand-side vector, is modified to implement
                     Dirichlet conditions (for simplicity of
                     implementation) 
           umfpackdata
                   - internal data for UMFPACK linear solver, 
		     see description of task for input/output behaviour
           isdiri  - internal data specifying if nodes are Dirichlet, 
		     see description of task for input/output behaviour
           help    - internaly used vector, 
		     see description of task for input/output behaviour


   Input:  P       - projector defining the Dirichlet DOFs,
           task    - specify task:
                     task==1 => initial solve, umfpackdata, isdiri and
                                help are created
                     task==2 => follow up solve, umfpackdata, isdiri and
                                help are used as provided 
                     task==9 => finish, umfpackdata, isdiri and help are
                                destroyed, memory is freed

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);
#endif
