#include <stdio.h>
#include "Gmsh.h"
#include "GModel.h"
#include "MElement.h"

int main(int argc, char **argv)
{
  GmshInitialize(argc, argv);

  if (argc<5)
    {
      fprintf(stderr,"\n%s: needs at least 4 arguments, "
	      "usage\n  %s name.geo name.msh querry.qrx output.pnt\n", argv[0], argv[0]);
      exit(1);
    }

  GModel *m = new GModel();
  m->readGEO(argv[1]); 
  m->readMSH(argv[2]);

  /* open querry file for reading */
  FILE *querry = fopen(argv[3], "r");
  if (querry == NULL)
    {
      fprintf(stderr, "%s: couldn't open querry file: %s\n",
	      argv[0],argv[3]);
      exit(1);	        
    }

  FILE *output = fopen(argv[4], "w");
  if (output == NULL)
    {
      fprintf(stderr, "%s: couldn't open output file: %s\n",
	      argv[0],argv[4]);
      exit(1);	        
    }

  /* expected querry format:
  k-number-line-points
  index1 line-tag current-x current-y current-z
  ...
  indexk line-tag current-x current-y current-z
  m-number-face-points
  index1 face-tag current-x current-y current-z
  ...
  indexm face-tag current-x current-y current-z
  [EOF]

  output format is the same, but without face-tag
  */
  int i, numEdgePoints, numFacePoints, nnread;

  nnread = fscanf(querry,"%d", &numEdgePoints);
  if (nnread!=1)
    {
      fprintf(stderr, "%s: error reading number of edges in file: %s\n",
	      argv[0], argv[3]);
      exit(1);	        
    }
  fprintf(output,"%d\n", numEdgePoints);
  for (i=0; i<numEdgePoints; i++)
    {
      int id, lineTag, nread;
      double qx,qy,qz;
      
      nread = fscanf(querry,"%d %d %le %le %le ", &id, &lineTag, &qx, &qy, &qz);

      if (nread!=5)
	{
	  fprintf(stderr, "%s: error reading %d-th edge entry in file: %s\n",
		  argv[0], i+1, argv[3]);
	  exit(1);	        
	}

      GEdge *eg = m->getEdgeByTag(lineTag);
      SPoint3 qpoint(qx,qy,qz);
      double param;

      GPoint closest = eg->closestPoint(qpoint, param);

      fprintf(output,"%d  %25.16e %25.16e %25.16e\n",
	      id, closest.x(), closest.y(), closest.z());
    }

  nnread = fscanf(querry,"%d", &numFacePoints);
  if (nnread!=1)
    {
      fprintf(stderr, "%s: error reading number of face in file: %s\n",
	      argv[0], argv[3]);
      exit(1);	        
    }
  fprintf(output,"%d\n", numFacePoints);
  for (i=0; i<numFacePoints; i++)
    {
      int id, faceTag, nread;
      double qx,qy,qz;

      nread = fscanf(querry,"%d %d %le %le %le ", &id, &faceTag, &qx, &qy, &qz);

      if (nread!=5)
	{
	  fprintf(stderr, "%s: error reading %d-th face entry in file: %s\n",
		  argv[0], i+1, argv[3]);
	  exit(1);	        
	}

      GFace *fc = m->getFaceByTag(faceTag);
      SPoint3 qpoint(qx,qy,qz);

      double initialguess[2] = {0.0, 0.0};
      GPoint closest = fc->closestPoint(qpoint, initialguess);

      fprintf(output,"%d  %25.16e %25.16e %25.16e\n",
	      id, closest.x(), closest.y(), closest.z());
    }
  delete m;
  GmshFinalize();

  exit(0);
}
