/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

/* this file contains useful text macros, which save work during
   programming */
#ifndef FEINS_MACROS_H
#define FEINS_MACROS_H

#include <config.h>
#include <stdio.h>
#include <stdlib.h>

/* still in development ==> debug switch on */
/* DEBUGFEINS is now automatically switched on by CMAKE if CMAKE_BUILD_TYPE==Debug */

/* for gettimeofday */
/* with a little help from http://www.linuxjournal.com/article/5574 */
#ifdef _WIN32
#include <sys/timeb.h>
#include <sys/types.h>
#include <winsock.h>
struct timezone {
	void *bla;
};
#define isnan(x) _isnan(x)

#define gettimeofday(_tvin_,_tzin_)\
{       struct timeval* t=(_tvin_);\
		struct _timeb timebuffer;\
        _ftime( &timebuffer );\
        t->tv_sec=timebuffer.time;\
        t->tv_usec=1000*timebuffer.millitm;\
}
#else
#include <sys/types.h>
#include <sys/time.h>
#endif

#ifdef HAVE_OPENMP
#include <omp.h>
#endif


/* define the integer datatypes, */
/* FIDX is for indices into arrays/vectors, also general integer type */
#define FIDX long 
/* the corresponding format string (d for int, ld for long int...) */
#define dFIDX "ld"

/* new adjoint handling, without replacing rows in the matrices */
#define NEW_ADJOINT

/* type of coarse grid stabilisation:
   0=none,
   1=lower Re:    use lower Re on coarser grids
   2=crude SUPG:  2nd order elements, leave out all terms that vanish
                  in 1st order elements case, only add stab term for
                  advection 
   3=lin ixSUPG:  use linear elements in the coarse meshes, only add
                  stab term for advection
   4=lin SUPG AD: use only advection diffusion matrices (for linear
                  elements) on coarser meshes
   5=art diff:    define diffusion coefficient locally in each cell
                  such that stability is ensured  
   6=full SUPG:   all terms dictated by the subgrid idea of Hughes */
#define STABTYPE 2

#define RAMAGE_NOT_TEZDUYAR  /* use the stabilisation parameter
                                used by Ramage rather than that by Tezduyar */

/*#define TIMETERM /* include an artificial time term into the solution
		    process, hopefully improving performance of the
		    linear solvers at the cost of slight slowing down
		    of the nonlinear solve */

#define FP_VELOCITY_SOLVER (1) /* the type of velocity solver used in
				the F_p preconditioner, 
				0 - GMRES, no preco
				1 - GMRES, V-cycle_MG (uses
				    MG_FOURDIR_SWEEPS (if so,
				    MG_SMOOTHER_DAMPING as well), all
				    STABTYPEs, MG_MIN_EN) 
				2 - GMRES, W-cycle_MG (uses MG_W_ARMS,
				    MG_W_DAMPING, STABTYPES 0,1,2,5,
				    USE_SORTED_GS)
				3 - GMRES, AMG preconditioner
				4 - AMG V-cycles
			     */
/* #define MG_FOUR_DIR_SWEEPS /* do sorted Gauss-Seidel Sweeps,
			      left-right, bottom-top, right-left,
			      top-bottom */
/* #define MG_W_DAMPING  /* use damped coarse grid updates */
#define MG_W_ARMS 2   /* number of CGCs in each W-cycle-level
			 (1=V-cycle, 2=W-cycle,...) */
#define MG_SMOOTHER_DAMPING 1.0 /* damping in GS sweeps */
/* #define USE_SORTED_GS /* use sorted Gauss-Seidel sweeps in the
			 smoother (rather than unsorted) */
/* #define MG_MIN_EN /* switch for minimal energy MG step */

/* #define DIAG_ONLY_COARSE
/* */
	
/* #define MG_F_NO_CMAT     /* switch OFF exact coarse-grid solves */		 
/* #define MG_F_NO_DIAGPROJ /* switch OFF projection of the diagonals in
			    Gauss-Seidel Smoother */		 

/* switch for conservative formulation of the advective term (only in
   Newton type linearisation) */
/*#define CONSERV_ADV_NEWT
/* */


/* use a coarse grid solver in the multigrid solver for the pressure
   space Laplacian in the F_p preconditioner,
   !!! still problems with "no boundary conditions", therefore deactivated !!!
*/
/* #define LAPLACIAN_COARSE_GRID_SOLVER
/* */

/* test wave resolution on the mesh (for the
   advection-diffusion-reaction problems that arise from the Newton
   linearization of the Navier-Stokes system) */
/* #define WAVE_RESOLVE_TEST
/* */


/* in the stokes preconditioner, switch to Chebyshev semi iteration
   for M^-1 */
#define STOKES_PRECO_CHEBYSHEV
/* use block-triangular preconditioner (non-symmetric) */
//#define STOKES_NONSYMETRIC_PRECON


/***********************************************************************
 **                                                                   **
 **                                                                   **
 **   SDFEM/Box-Smoother options                                      **
 **                                                                   **
 **                                                                   **
 ***********************************************************************/
/*#define BOXSMOOTHER_SORTED          /* switch on sorting of the sweeps
                                       of local problems, sort by x,y */
/*#define BOXSMOOTHER_SORTED_2DIR     /* switch on a second sorted
				       sweep, sorted by -y,-x */
/*#define BOXSMOOTHER_ELEMBASED       /* switch on element-based smoother,
                                       otherwise p-node based */
/*#define BOXSMOOTHER_LOC_NEUMAN      /* switch to Neumann-like local
				       problems, in p-node based,
				       otherwise local Dirichlet */
/*#define BOXSMOOTHER_DIAG_ONLY_KLOC  /* use only diagonal of K_loc
				       velocity block, otherwise full
				       K_loc  */ 
#define BOXSMOOTHER_DAMPING (0.5)   /* damping factor for the smoother */
/* #include "box_smoother_damping.h" /* */
#define BOXSMOOTHER_BC_ELEMENT_ALL_V /*use all v-dofs for boundary
				       elements, otherwise as defined
				       by switch local
				       Dirichlet/Neumann problems */
#define BOXSMOOTHER_VEL_WEIGHTS     /* Weight the velocity dofs in the
				       local problems by
				       1/sqrt(#appearances) */





/* return value convention */
#define SUCCESS 0
#define FAIL    1

/* maximal refinement dept */
#define FEINSLEVELMAX 30


#ifdef HAVE_OPENMP
#define FEINS_FAILURE_ACTION {exit(EXIT_FAILURE);}
#else
#define FEINS_FAILURE_ACTION {return FAIL;}
#endif


/* macro for malloc failure returns */
#define TRY_MALLOC(__who__, __somuch__, __type__, __func__) {		\
    __who__ = (__type__*) malloc( (__somuch__)*sizeof(__type__) );	\
    if ((__who__)==NULL)						\
      {									\
	fprintf(stderr,							\
		"line-%d in %s, file %s: malloc for %d %s entries "	\
		"to %s failed\n",					\
		__LINE__, #__func__, __FILE__, (int)(__somuch__),	\
		#__type__, #__who__ );					\
	FEINS_FAILURE_ACTION						\
      }									\
  }

#ifdef __cplusplus
#define TRY_NEW(__who__, __somuch__, __type__, __func__) {		\
    __who__ = new __type__[__somuch__] ;				\
    if ((__who__)==NULL)						\
      {									\
	fprintf(stderr,							\
		"line-%d in %s, file %s: new for %d %s entries "	\
		"to %s failed\n",					\
		__LINE__, #__func__, __FILE__, (int)(__somuch__),	\
		#__type__, #__who__ );					\
	FEINS_FAILURE_ACTION						\
      }									\
  }
#endif


/* macro for malloc failure returns */
#define TRY_REALLOC(__who__, __somuch__, __type__, __func__) {		\
    __who__ = (__type__*) realloc( (__who__),				\
				   (__somuch__)*sizeof(__type__) );	\
    if ((__who__)==NULL)						\
      {									\
	fprintf(stderr,							\
		"line-%d in %s, file %s: realloc for %d %s entries "	\
		"to %s failed\n",					\
		__LINE__, #__func__, __FILE__, (int)(__somuch__),	\
		#__type__, #__who__ );					\
	FEINS_FAILURE_ACTION						\
      }									\
  }

/* macro to handle error returns by subroutines, 
   __where__ = function in which this happens
   __who__   = the subroutine called
   __err__   = variable holding the return value of the subroutine */
#define FUNCTION_FAILURE_HANDLE( __err__, __who__, __where__) {		\
    if ((__err__)!=SUCCESS)						\
      {									\
	fprintf(stderr,							\
		"line-%d in %s, file %s: call to %s "			\
		"returned with %d(!=SUCCESS)\n",			\
		__LINE__, #__where__, __FILE__, #__who__, __err__ );	\
	FEINS_FAILURE_ACTION						\
	  }								\
  }

/* too much trouble with fmax => define it ourself */
#define fmax(Val1, Val2)  (Val1>Val2)? Val1:Val2
#define fmin(Val1, Val2)  (Val1<Val2)? Val1:Val2

#endif
