/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE cubature.c
HEADER cubature.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"

*/



/* prototypes of external functions */
#include <math.h>
#include "elements.h"


/* function prototypes */
#include "cubature.h"

/*FUNCTION*/
int cubature_formula(FIDX dim, FIDX degree, enum elemtype type, 
		     struct int_data *form
/* set int_data form to contain a integration formula for given
   dimension such that the formula integrates polynomials up to degree
   degree exact on the master element of element type type 

   Input:  dim    - dimension of space
           degree - up to which the formula has to be exact
           type   - type of the master element (see element types)

   Output: form   - the quadrature formula, given via reference

   Return: SUCCESS- success
           FAIL   - failure, see error message

   after succesful return form contains a formula such that

   I_elem = sum[i=1..num_points] weight[i]*func(points[i])
          = integral[master element] func(x)

   exact for all polynomials "func" of degree up to degree

   this can also be used to approximate this integral for other
   functions "func".
*/
		     ){
  /* for each formula blocks like:
       elseif ((dim==)&&((degree==)&&(type==)))
         {
            set to something;
            return SUCCESS;
         }
       elseif ... */
  if ((dim==1)&&((degree==3)&&(type==inter)))
    {
      /* formula degree=3 on the inteval [0,1] */
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 2;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* weights = 1/2 */
      (*form).weights[0] = 0.5;
      (*form).weights[1] = 0.5;
      /* points = 1/2 +- 1/6 * sqrt(3) */
      (*form).points[0]  = 0.5 - 0.5/sqrt(3.0);
      (*form).points[1]  = 0.5 + 0.5/sqrt(3.0);

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==1)&&((degree==3)&&(type==tria)))
    {
      /* formula degree=3 on interval [0,1]x{0} as part of triangle 
         convex_hull( [0,0], [1,0], [0,1])
	 (usefull if gradient has to be integrated on boundary
      */
      /* formula degree=3 on the inteval [0,1] */
      int ldim=2;

	  (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 2;

	  TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, ldim*(*form).num_points, double,
		 cubature_formula );

      /* weights = 1/2 */
      (*form).weights[0] = 0.5;
      (*form).weights[1] = 0.5;
      /* points = 1/2 +- 1/6 * sqrt(3) */
      (*form).points[0*ldim+0]  = 0.5 - 0.5/sqrt(3.0);
      (*form).points[0*ldim+1]  = 0.0;

      (*form).points[1*ldim+0]  = 0.5 + 0.5/sqrt(3.0);
      (*form).points[1*ldim+1]  = 0.0;

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==2)&&((degree==1)&&(type==tria)))
    {
      /* formula degree=1 on simplices */
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 1;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* weight = 1/(dim!) */
      (*form).weights[0] = 1;
      for (i=0; i< dim; i++)
	{
	  (*form).weights[0] /= (double)(i+1);
	  (*form).points[i] = 1.0/(dim+1);
	}
      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==2)&&((degree==2)&&(type==tria)))
    {
      /* formula degree=2 on triangles */
      /* from the database:
	 Region: Simplex
	 Dimension: 2
	 Degree: 2
	 Points: 3
	 Structure: Fully symmetric
	 Rule struct: 0 0 0 0 1 0
	 Generator: [ Fully symmetric ]
	 ( 0.166666666666666666666666666666666, 
	   0.166666666666666666666666666666666, )
	 Corresponding weight:
	 0.166666666666666666666666666666666 */
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 3;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* all weights equal */
      for (i=0; i<(*form).num_points; i++)
	(*form).weights[i] = 1.0/6.0;
      /* points are: b=1/6,
	 (b,b), (b, 1-2b), (1-2b,b) */
      (*form).points[0*dim +0] = 1.0/6.0;
      (*form).points[0*dim +1] = 1.0/6.0;
      (*form).points[1*dim +0] = 1.0/6.0;
      (*form).points[1*dim +1] = 4.0/6.0;
      (*form).points[2*dim +0] = 4.0/6.0;
      (*form).points[2*dim +1] = 1.0/6.0;
	
      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==2)&&((degree==4)&&(type==tria)))
    {
      /* formula degree=4 on triangles */
      /* from the database:
	 Region: Simplex
	 Dimension: 2
	 Degree: 4
	 Points: 6
	 Structure: Fully symmetric
	 Rule struct: 0 0 0 0 2 0
	 Generator: [ Fully symmetric ]
	 ( 0.0915762135097707434595714634022015,
	 0.0915762135097707434595714634022015, )
	 Corresponding weight:
	 0.0549758718276609338191631624501052,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.445948490915964886318329253883051,
	 0.445948490915964886318329253883051, )
	 Corresponding weight:
	 0.111690794839005732847503504216561,
      */
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 6;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* first 3 weights */
      for (i=0; i<3; i++)
	(*form).weights[i] = 0.0549758718276609338191631624501052;
      /* last 3 weights */
      for (i=3; i<6; i++)
	(*form).weights[i] = 0.111690794839005732847503504216561;

      /* points are: 
	 (b,b), (b, 1-2b), (1-2b,b)
	 (c,c), (c, 1-2c), (1-2c,c)
	 b=0.0915762135097707434595714634022015
	 c=0.445948490915964886318329253883051
      */
#define B (0.0915762135097707434595714634022015)
#define C (0.445948490915964886318329253883051)
      (*form).points[0*dim +0] = B;
      (*form).points[0*dim +1] = B;
      (*form).points[1*dim +0] = B;
      (*form).points[1*dim +1] = 1.0-2*B;
      (*form).points[2*dim +0] = 1.0-2*B;
      (*form).points[2*dim +1] = B;

      (*form).points[3*dim +0] = C;
      (*form).points[3*dim +1] = C;
      (*form).points[4*dim +0] = C;
      (*form).points[4*dim +1] = 1.0-2*C;
      (*form).points[5*dim +0] = 1.0-2*C;
      (*form).points[5*dim +1] = C;
#undef B
#undef C


      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==2)&&((degree==7)&&(type==tria)))
    {
      /* formula degree=7 on triangles */
      /* from the database:
	 Region: Simplex
	 Dimension: 2
	 Degree: 7
	 Points: 12
	 Structure: Ro3 invariant
	 Rule struct: 0 0 0 0 0 4
	 
	 Generator: [ Ro3 invariant ]
	 ( 0.0623822650944021181736830009963499,
	   0.0675178670739160854425571310508685, )
	 Corresponding weight:
	 0.0265170281574362514287541804607391,

	 Generator: [ Ro3 invariant ]
	 ( 0.0552254566569266117374791902756449,
	   0.321502493851981822666307849199202, )
	 Corresponding weight:
	 0.0438814087144460550367699031392875,

	 Generator: [ Ro3 invariant ]
	 ( 0.0343243029450971464696306424839376,
	   0.660949196186735657611980310197799, )
	 Corresponding weight:
	 0.0287750427849815857384454969002185,

	 Generator: [ Ro3 invariant ]
	 ( 0.515842334353591779257463386826430,
	   0.277716166976391782569581871393723, )
	 Corresponding weight:
	 0.0674931870098027744626970861664214, 
      */
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 12;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* first 3 weights */
      for (i=0; i<3; i++)
	(*form).weights[i] = 0.0265170281574362514287541804607391;
      /* second 3 weights */
      for (i=3; i<6; i++)
	(*form).weights[i] = 0.0438814087144460550367699031392875;
      /* third  3 weights */
      for (i=6; i<9; i++)
	(*form).weights[i] = 0.0287750427849815857384454969002185;
      /* last   3 weights */
      for (i=9; i<12; i++)
	(*form).weights[i] = 0.0674931870098027744626970861664214;

      /* points are: 
	 (a,b), (b,c), (c,a)

	 where a,b are the coords of the generator as given by the
	 database and c=1-a-b, the third barycentric coordinate
      */
      /* first 3 points */
#define A (0.0623822650944021181736830009963499)
#define B (0.0675178670739160854425571310508685)
#define C (1.0-A-B)
      (*form).points[ 0*dim +0] = A;
      (*form).points[ 0*dim +1] = B;
      (*form).points[ 1*dim +0] = B;
      (*form).points[ 1*dim +1] = C;
      (*form).points[ 2*dim +0] = C;
      (*form).points[ 2*dim +1] = A;
#undef A
#undef B
#undef C

      /* second 3 points */
#define A (0.0552254566569266117374791902756449)
#define B (0.321502493851981822666307849199202)
#define C (1.0-A-B)
      (*form).points[ 3*dim +0] = A;
      (*form).points[ 3*dim +1] = B;
      (*form).points[ 4*dim +0] = B;
      (*form).points[ 4*dim +1] = C;
      (*form).points[ 5*dim +0] = C;
      (*form).points[ 5*dim +1] = A;
#undef A
#undef B
#undef C

      /* third 3 points */
#define A (0.0343243029450971464696306424839376)
#define B (0.660949196186735657611980310197799)
#define C (1.0-A-B)
      (*form).points[ 6*dim +0] = A;
      (*form).points[ 6*dim +1] = B;
      (*form).points[ 7*dim +0] = B;
      (*form).points[ 7*dim +1] = C;
      (*form).points[ 8*dim +0] = C;
      (*form).points[ 8*dim +1] = A;
#undef A
#undef B
#undef C

      /* last 3 points */
#define A (0.515842334353591779257463386826430)
#define B (0.277716166976391782569581871393723)
#define C (1.0-A-B)
      (*form).points[ 9*dim +0] = A;
      (*form).points[ 9*dim +1] = B;
      (*form).points[10*dim +0] = B;
      (*form).points[10*dim +1] = C;
      (*form).points[11*dim +0] = C;
      (*form).points[11*dim +1] = A;
#undef A
#undef B
#undef C

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==3)&&((degree==1)&&(type==tetra)))
    {
      /* formula degree=1 on simplices */
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 1;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* weight = 1/(dim!) */
      (*form).weights[0] = 1;
      for (i=0; i< dim; i++)
	{
	  (*form).weights[0] /= (double)(i+1);
	  (*form).points[i] = 1.0/(dim+1);
	}
      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==3)&&((degree==2)&&(type==tetra)))
      /* formula degree=2 on tetrahedra */
      /* from the database:
	 Region: Simplex
	 Dimension: 3
	 Degree: 2
	 Points: 4
	 Structure: Fully symmetric
	 Rule struct: 0 0 0 0 0 0 0 1 0 0 0
	 Generator: [ Fully symmetric ]
	 ( 0.138196601125010515179541316563436,
	 0.138196601125010515179541316563436,
	 0.138196601125010515179541316563436,
	 )
	 Corresponding weight:
	 0.0416666666666666666666666666666666,
      */
    {
      FIDX i;
      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 4;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* the 4 weights */
      for (i=0; i<4; i++)
	(*form).weights[i] = 0.0416666666666666666666666666666666;

      /* points are: ,
	 (b,b,b), (b,b,1-3b), (b,1-3b,b),(1-3b,b,b) */
#define B (0.138196601125010515179541316563436)
      (*form).points[0*dim +0] = B;
      (*form).points[0*dim +1] = B;
      (*form).points[0*dim +2] = B;

      (*form).points[1*dim +0] = B;
      (*form).points[1*dim +1] = B;
      (*form).points[1*dim +2] = 1.0-3*B;

      (*form).points[2*dim +0] = B;
      (*form).points[2*dim +1] = 1.0-3*B;
      (*form).points[2*dim +2] = B;

      (*form).points[3*dim +0] = 1.0-3*B;
      (*form).points[3*dim +1] = B;
      (*form).points[3*dim +2] = B;
#undef B

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==3)&&((degree==4)&&(type==tetra)))
      /* formula degree=4 on tetrahedra */
      /* from the database:
	 Region: Simplex
	 Dimension: 3
	 Degree: 4
	 Points: 14
	 Structure: Fully symmetric
	 Rule struct: 0 1 0 0 0 0 0 2 0 0 0
	 Generator: [ Fully symmetric ]
	 ( 0.5, 0.5, 0., )
	 Corresponding weight:
	 10 ^ -3 x 3.17460317460317460317460317460317,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.100526765225204479693840794395431,
	 0.100526765225204479693840794395431,
	 0.100526765225204479693840794395431,
	 )
	 Corresponding weight:
	 0.0147649707904967850722720270773448,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.314372873493192192750757129367461,
	 0.314372873493192192750757129367461,
	 0.314372873493192192750757129367461,
	 )
	 Corresponding weight:
	 0.0221397911142651196896327348274170, 
      */
    {
      FIDX i;

      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 14;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* first 6 weights */
      for (i=0; i<6; i++)
	(*form).weights[i] = 3.17460317460317460317460317460317e-3;

      /* next 4 weights */
      for (i=6; i<10; i++)
	(*form).weights[i] = 0.0147649707904967850722720270773448;

      /* next 4 weights */
      for (i=10; i<14; i++)
	(*form).weights[i] = 0.0221397911142651196896327348274170;

      /* points are: (a=1/2)
	 (a,a,0), (a,0,a), (0,a,a), (a,0,0), (0,a,0), (0,0,a),
	 (b,b,b), (b,b,1-3b), (b,1-3b,b),(1-3b,b,b),
	 (c,c,c), (c,c,1-3c), (c,1-3c,c),(1-3c,c,c)
      */
#define A (0.5)
#define B (0.100526765225204479693840794395431)
#define C (0.314372873493192192750757129367461)
      /* (a,a,0), (a,0,a), (0,a,a) */
      (*form).points[0*dim +0] = A;
      (*form).points[0*dim +1] = A;
      (*form).points[0*dim +2] = 0.0;

      (*form).points[1*dim +0] = A;
      (*form).points[1*dim +1] = 0.0;
      (*form).points[1*dim +2] = A;

      (*form).points[2*dim +0] = 0.0;
      (*form).points[2*dim +1] = A;
      (*form).points[2*dim +2] = A;

      /* (a,0,0), (0,a,0), (0,0,a) */
      (*form).points[3*dim +0] = A;
      (*form).points[3*dim +1] = 0.0;
      (*form).points[3*dim +2] = 0.0;

      (*form).points[4*dim +0] = 0.0;
      (*form).points[4*dim +1] = A;
      (*form).points[4*dim +2] = 0.0;

      (*form).points[5*dim +0] = 0.0;
      (*form).points[5*dim +1] = 0.0;
      (*form).points[5*dim +2] = A;

      /* (b,b,b), (b,b,1-3b), (b,1-3b,b),(1-3b,b,b) */
      (*form).points[6*dim +0] = B;
      (*form).points[6*dim +1] = B;
      (*form).points[6*dim +2] = B;

      (*form).points[7*dim +0] = B;
      (*form).points[7*dim +1] = B;
      (*form).points[7*dim +2] = 1.0-3*B;

      (*form).points[8*dim +0] = B;
      (*form).points[8*dim +1] = 1.0-3*B;
      (*form).points[8*dim +2] = B;

      (*form).points[9*dim +0] = 1.0-3*B;
      (*form).points[9*dim +1] = B;
      (*form).points[9*dim +2] = B;

      /* (c,c,c), (c,c,1-3c), (c,1-3c,c),(1-3c,c,c) */
      (*form).points[10*dim +0] = C;
      (*form).points[10*dim +1] = C;
      (*form).points[10*dim +2] = C;

      (*form).points[11*dim +0] = C;
      (*form).points[11*dim +1] = C;
      (*form).points[11*dim +2] = 1.0-3*C;

      (*form).points[12*dim +0] = C;
      (*form).points[12*dim +1] = 1.0-3*C;
      (*form).points[12*dim +2] = C;

      (*form).points[13*dim +0] = 1.0-3*C;
      (*form).points[13*dim +1] = C;
      (*form).points[13*dim +2] = C;
#undef A
#undef B
#undef C

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else if ((dim==3)&&((degree==6)&&(type==tetra)))
      /* formula degree=6 on tetrahedra */
      /* from the database:
	 Region: Simplex
	 Dimension: 3
	 Degree: 6
	 Points: 24
	 Structure: Fully symmetric
	 Rule struct: 0 0 0 0 0 0 0 3 0 1 0
	 Generator: [ Fully symmetric ]
	 ( 0.214602871259152029288839219386284,
	 0.214602871259152029288839219386284,
	 0.214602871259152029288839219386284,
	 )
	 Corresponding weight:
	 10 ^ -3 x 6.65379170969458201661510459291332,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.0406739585346113531155794489564100,
	 0.0406739585346113531155794489564100,
	 0.0406739585346113531155794489564100,
	 )
	 Corresponding weight:
	 10 ^ -3 x 1.67953517588677382466887290765614,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.322337890142275510343994470762492,
	 0.322337890142275510343994470762492,
	 0.322337890142275510343994470762492,
	 )
	 Corresponding weight:
	 10 ^ -3 x 9.22619692394245368252554630895433,
	 
	 Generator: [ Fully symmetric ]
	 ( 0.0636610018750175252992355276057269,
	 0.0636610018750175252992355276057269,
	 0.269672331458315808034097805727606,
	 )
	 Corresponding weight:
	 10 ^ -3 x 8.03571428571428571428571428571428, 
      */
    {
      FIDX i;

      (*form).dim        = dim;
      (*form).type       = type;
      (*form).degree     = degree;
      (*form).num_points = 24;
      TRY_MALLOC((*form).weights, (*form).num_points, double,
		 cubature_formula );
      TRY_MALLOC((*form).points, dim*(*form).num_points, double,
		 cubature_formula );

      /* first 4 weights */
      for (i=0; i<4; i++)
	(*form).weights[i] = 6.65379170969458201661510459291332e-3;

      /* next 4 weights */
      for (i=4; i<8; i++)
	(*form).weights[i] = 1.67953517588677382466887290765614e-3;

      /* next 4 weights */
      for (i=8; i<12; i++)
	(*form).weights[i] = 9.22619692394245368252554630895433e-3;

      /* next 12 weights */
      for (i=12; i<24; i++)
	(*form).weights[i] = 8.03571428571428571428571428571428e-3;
 
      /* points are: 
	 (a,a,a), (a,a,1-3a), (a,1-3a,a),(1-3a,a,a),
	 (b,b,b), (b,b,1-3b), (b,1-3b,b),(1-3b,b,b),
	 (c,c,c), (c,c,1-3c), (c,1-3c,c),(1-3c,c,c),
	 f:=1-2d-e
	 (d,d,e), (d,e,d), (e,d,d),
	 (d,d,f), (d,f,d), (f,d,d),
	 (d,e,f), (e,f,d), (f,d,e),
	 (d,f,e), (e,d,f), (f,e,d)
      */
#define A (0.214602871259152029288839219386284)
#define B (0.0406739585346113531155794489564100)
#define C (0.322337890142275510343994470762492)
#define D (0.0636610018750175252992355276057269)
#define E (0.269672331458315808034097805727606)
#define F (1.0-2*D-E)
      /* (a,a,a), (a,a,1-3a), (a,1-3a,a),(1-3a,a,a), */
      (*form).points[0*dim +0] = A;
      (*form).points[0*dim +1] = A;
      (*form).points[0*dim +2] = A;

      (*form).points[1*dim +0] = A;
      (*form).points[1*dim +1] = A;
      (*form).points[1*dim +2] = 1.0-3*A;

      (*form).points[2*dim +0] = A;
      (*form).points[2*dim +1] = 1.0-3*A;
      (*form).points[2*dim +2] = A;

      (*form).points[3*dim +0] = 1.0-3*A;
      (*form).points[3*dim +1] = A;
      (*form).points[3*dim +2] = A;

      /* (b,b,b), (b,b,1-3b), (b,1-3b,b),(1-3b,b,b), */
      (*form).points[4*dim +0] = B;
      (*form).points[4*dim +1] = B;
      (*form).points[4*dim +2] = B;

      (*form).points[5*dim +0] = B;
      (*form).points[5*dim +1] = B;
      (*form).points[5*dim +2] = 1.0-3*B;

      (*form).points[6*dim +0] = B;
      (*form).points[6*dim +1] = 1.0-3*B;
      (*form).points[6*dim +2] = B;

      (*form).points[7*dim +0] = 1.0-3*B;
      (*form).points[7*dim +1] = B;
      (*form).points[7*dim +2] = B;

      /* (c,c,c), (c,c,1-3c), (c,1-3c,c),(1-3c,c,c), */
      (*form).points[8*dim +0] = C;
      (*form).points[8*dim +1] = C;
      (*form).points[8*dim +2] = C;

      (*form).points[9*dim +0] = C;
      (*form).points[9*dim +1] = C;
      (*form).points[9*dim +2] = 1.0-3*C;

      (*form).points[10*dim +0] = C;
      (*form).points[10*dim +1] = 1.0-3*C;
      (*form).points[10*dim +2] = C;

      (*form).points[11*dim +0] = 1.0-3*C;
      (*form).points[11*dim +1] = C;
      (*form).points[11*dim +2] = C;

      /* (d,d,e), (d,e,d), (e,d,d), */
      (*form).points[12*dim +0] = D;
      (*form).points[12*dim +1] = D;
      (*form).points[12*dim +2] = E;

      (*form).points[13*dim +0] = D;
      (*form).points[13*dim +1] = E;
      (*form).points[13*dim +2] = D;

      (*form).points[14*dim +0] = E;
      (*form).points[14*dim +1] = D;
      (*form).points[14*dim +2] = D;

      /* (d,d,f), (d,f,d), (f,d,d), */
      (*form).points[15*dim +0] = D;
      (*form).points[15*dim +1] = D;
      (*form).points[15*dim +2] = F;

      (*form).points[16*dim +0] = D;
      (*form).points[16*dim +1] = F;
      (*form).points[16*dim +2] = D;

      (*form).points[17*dim +0] = F;
      (*form).points[17*dim +1] = D;
      (*form).points[17*dim +2] = D;

      /* (d,e,f), (e,f,d), (f,d,e), */
      (*form).points[18*dim +0] = D;
      (*form).points[18*dim +1] = E;
      (*form).points[18*dim +2] = F;

      (*form).points[19*dim +0] = E;
      (*form).points[19*dim +1] = F;
      (*form).points[19*dim +2] = D;

      (*form).points[20*dim +0] = F;
      (*form).points[20*dim +1] = D;
      (*form).points[20*dim +2] = E;

      /* (d,f,e), (e,d,f), (f,e,d)  */
      (*form).points[21*dim +0] = D;
      (*form).points[21*dim +1] = F;
      (*form).points[21*dim +2] = E;

      (*form).points[22*dim +0] = E;
      (*form).points[22*dim +1] = D;
      (*form).points[22*dim +2] = F;

      (*form).points[23*dim +0] = F;
      (*form).points[23*dim +1] = E;
      (*form).points[23*dim +2] = D;
#undef A
#undef B
#undef C
#undef D
#undef E
#undef F

      (*form).bases      = NULL;
      
      return SUCCESS;
    }
  else
    {
      fprintf(stderr,
	      "sorry, cubature_formula not implemented for (%d,%d,%d,*)\n",
	      (int) dim, (int) degree, type);
      return FAIL;
    }
}

/*FUNCTION*/
int cubature_bases(FIDX dim, FIDX degree, enum elemtype type,
		   FIDX num_subtypes, FIDX subtype[],  
		   struct int_data *form
/* set int_data form to contain a integration formula for given
   dimension such that the formula integrates polynomials up to degree
   degree exact on the master element of element type type, compute
   int_data_basis for the two element subtypes subtype1, subtype2

   Input:  dim    - dimension of space
           degree - up to which the formula has to be exact
           type   - type of the master element (see element types)
	   num_subtypes
                  - number of subtypes
	   subtype- vector of length num_subtypes, containing the
    	            subtypes to be considered, e.g. for triangular
    	            elements when the basis pair P_2 P_1 needed is
    	            then this would be the vector [2,1]

   Output: form   - the quadrature formula, given via reference
           form.bases 
	          - will contain a array of pointers to struct
              	    int_data_basis such that form.bases[i] points to a
              	    struct which holds the basis functions and their
              	    derivatives the subtype[i], evaluated at the
              	    quadrature points for 

   Return: SUCCESS- succes
           FAIL   - failure, see error message

   after succesful return form contains everything necessary to
   compute certain types of integrals on the elements by mapping the
   integrand to the master element and computing the integral there
*/
		   ){
  FIDX i;
  int err;
  
  /* get the formula */
  err=cubature_formula(dim, degree, type, form);
  if (err!=SUCCESS)
    {
      fprintf(stderr,"cubature_formula failed in cubature_bases!\n");
      return FAIL;
    }
  
  /* plug in the array form.bases */
  (*form).num_subtypes=num_subtypes;
  TRY_MALLOC((*form).bases, num_subtypes, struct int_data_basis*,
	     cubature_bases );
  for (i=0; i<num_subtypes; i++)
    {
      double **thisphi, **thisgrad, **thishess;
      FIDX *thisnum_b;

      TRY_MALLOC((*form).bases[i], 1, struct int_data_basis,
		 cubature_bases );
      (*form).bases[i]->subtype=subtype[i];
      thisphi  = &((*form).bases[i]->phi);
      thisgrad = &((*form).bases[i]->gradphi);
      thishess = &((*form).bases[i]->hessphi);
      thisnum_b= &((*form).bases[i]->num_basis);

      err = eval_basis(dim, type, subtype[i], (*form).num_points,
		       (*form).points, thisnum_b,
		       thisphi, thisgrad, thishess);
      if (err!=SUCCESS)
	{
	  fprintf(stderr,
		  "eval_basis failed in cubature_bases, basis %d\n",
		  (int) i);
	  return FAIL;
	}
    }
  return SUCCESS;
}


/*FUNCTION*/
int free_intdata( struct int_data *form 
/* free the elements of this data type such that the form itself can
   be deleted */
		  ){
  FIDX i;

  free((*form).weights);
  free((*form).points);
  for (i=0; i<(*form).num_subtypes; i++)
    {
      free(((*form).bases[i]->phi));
      free(((*form).bases[i]->gradphi));
      free(((*form).bases[i]->hessphi));
      free((*form).bases[i]);
    }
  free((*form).bases);
  
  return SUCCESS;
}
