/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2013, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
assembly.h
*/
#ifndef __ASSEMBLY_H__
#define __ASSEMBLY_H__ 1



#include "feins_macros.h"

#include "datastruc.h"
#include "sparse_struct.h"


int assem_poisson_t1(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P, 
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poisson equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poisson equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (only p1 triangles for now)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   In/Out: u0      - initial guess for the solution of the equation
 	             system, modified to satisfy the Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_poisson_t2(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P,
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poisson equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poisson equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (p2 triangles)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   In/Out: u0      - initial guess for the solution of the equation
 	             system, modified to satisfy the Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_poisson_t1_t2(struct sparse *K, FIDX vx_nr1, FIDX *dofs,
		       struct mesh *m
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Poisson equation with linear
   triangle elements (T1) on the (T2) mesh m

   Input:  m       - the mesh
           vx_nr1  - the number of DOFs, (the number of corner nodes of
                     the elements in the mesh)
	   dofs    - pointer to array of length m.vx_nr, such that 
	             dofs[i] tells which (if any) degree of freedom
		     (DOF) belongs to the node, 
		     dofs[i]=j ==> j-th DOF belongs to the i-th node
		     of the mesh

   Output: K       - stiffness matrix

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_poisson_e1(struct sparse *K, struct vector *rhs,
		    struct vector *u0, struct projector1 *P, 
		    struct mesh *m
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from finite element discretisation of
   the Poisson equation on the mesh m, such that 
       K u = rhs
   defines the approximate solution u of the Poisson equation
       Laplace(u) = f
   with boundary conditions as given in the mesh m    
   (only p1 tetrahedra for now)

   Input:  m       - the mesh

   Output: K       - stiffness matrix
           rhs     - righthand side vector
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC

   In/Out: u0      - initial guess for the solution of the equation
 	             system, modified to satisfy the Dirichlet BC

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_mass_tx(struct sparse *M,  
		  struct mesh *m,
		  FIDX type
/* performs the assembly of the mass matrix M for finite element
   discretisations 

   Input:  m       - the mesh
           type    - type of mesh, ==1 means linear (P1) elements, 
                     ==2 means quadratic (P2) elements,

   Output: M       - mass matrix

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_mass_gamma_t1( struct sparse *Mg, struct mesh *m
/* computes the mass matrix for the boundary, as for example required
   for Robin boundary conditions
   ==>
       Mg(i,j)=int_{Gamma} \phi_j \phi i dGamma

   Input:  m       - the mesh (2d T2), input

   Output: Mg      - Mass matrix of boundary integrals, see above

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_lame_tx(struct sparse *K, struct projector1 *P, struct mesh *msh, struct vector *rhs, struct vector *u0,
                  double lambda, double mu,
		  int gammas, int bcswitch, int type
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Lame' equation (linear
   elasticity) on the mesh msh, such that 
       K u = 0
   defines the approximate solution u of the Lame' equation
       mu*Laplace(u) + (mu+lambda)*grad div(u) = 0
   with boundary conditions as given in the mesh msh
   (p1 or p2 triangles for now)

   Input:  msh     - the mesh
           lambda  - Lame' constant 
           mu      - Lame' constant 
           gammas  - the stiffening parameter, usefull when this is
                     used for mesh deformation, then gammas=0,
		     FOR REAL LINEAR ELASTICITY USE gammas=1 !
           bcswitch- if bcswitch==0 only specified Dirichlet BC are used
                     if ==1 ALL types of boundary conditions
                     are treated as Dirichlet, ignoring their
                     specified type
                     if ==2 Dirichlet BC and Neumann BC are used
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: K       - stiffness matrix
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC, if not wanted
           	     give P=NULL
         rhs        - righthandside vector

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_lame_tx_tx(struct sparse *K, struct projector1 *P, 
		  struct mesh *msh, struct vector *rhs, struct vector *u0,
                  double lambda, double mu,
		  int gammas, int bcswitch, int type
/* performs the assembly of the stiffness matrix K which results from
   finite element discretisation of the Lame' equation (linear
   elasticity) on the mesh msh, such that 
       K u = 0
   defines the approximate solution u of the Lame' equation
       mu*Laplace(u) + (mu+lambda)*grad div(u) = 0
   with boundary conditions as given in the mesh msh
   (p1 or p2 triangles for now)

   Input:  msh     - the mesh
           lambda  - Lame' constant 
           mu      - Lame' constant 
           gammas  - the stiffening parameter, usefull when this is
                     used for mesh deformation, then gammas=0,
		     FOR REAL LINEAR ELASTICITY USE gammas=1 !
           bcswitch- if bcswitch==0 only specified Dirichlet BC are used
                     if ==1 ALL types of boundary conditions
                     are treated as Dirichlet, ignoring their
                     specified type
                     if ==2 Dirichlet BC and Neumann BC are used
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: K       - stiffness matrix
	   P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC, if not wanted
           	     give P=NULL
         rhs        - righthandside vector

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_neum_bc_t2( struct mesh *m, struct vector *rhs
/* adds the contribution of the Neumann boundary conditions to the rhs
       sigma(u)*n=g on Gamma_N
   ==>
       rhs(v) += int_{Gamma_N} +g*v ds

   Input:  m       - the mesh (2d T2), input

   Output: rhs     - rhighthandside vector, set according to the
                     integral formula above

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_Dirichlet_BC_tx(struct projector1 *P, struct mesh *msh, 
			  int ncomp, struct vector *u0, int type
/* collects the Dirichlet DOFs into the projector P and sets the
   values of the initial solution u0 at those nodes to the data
   specidied in the mesh msh (p1 or p2 triangles for now)

   Input:  msh     - the mesh
           ncomp   - number of components per node in the solution
                     vector u0, if ncomp>1, u0.V[j*vx_nr+i] must
                     specify the j-th solution component at the i-th
                     node
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: P       - data for a projector which projects onto the
           	     linear subspace which doesn't change the values
           	     of DOFs related to Dirichlet BC, 
		     has to be initialised and allocated by the
		     calling routine 
	   u0      - initial solution vector, values at Dirichlet
	             nodes are set

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int assem_elem_detA_tx(struct mesh *m, double *elem_detA, int type
/* computes the detA the determinate of the element linear mapping 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: elem_detA-detA for each element

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int max_angle_tx(struct mesh *m, double *maxal, int type
/* computes the maximum interior angle amon the triangular elements in
   the mesh 

   Input:  m       - the mesh
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles

   Output: maxal   - maximum interior angle for all elements, in
                     radiant (rather than degrees), given by reference

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);

int lame_error_est_residual_t2(struct mesh *m, struct vector *uh,
			       double lambda, double mu, FIDX *marker, 
			       FIDX *nrmarked, double *globest,
			       struct solver_settings *set, int type
/* Estimates the error with a residual-based error estimator for the
   Lame problem.

   Input:  m           - the mesh
	   uh	       - soltution vector of the discretisized lame problem
           lambda, mu  - Lame constants 
	   set	       - holding the settings for the process
           type        - defining which type of mesh this actually is
	                 type=1 ==> linear    triangles
			 type=2 ==> quadratic triangles

   Output: marker      - vector of length msh.el_nr with
                         marker[i]==1 -> element i isto be refined,
           nrmarked    - number of marked elements, i.e. number of
                         non-zero entries in marker (given by reference)
           globest     - value of the global estimate (given by reference)

   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
	                 valid
*/);
#endif
