% ***********************************************************************
%
%    This file is part of
%    FEINS, Finite Element Incompressible Navier-Stokes solver,
%    which is expanding to a more general FEM solver and toolbox,
%    Copyright (C) 2003--2008, Rene Schneider 
%    <rene.schneider@mathematik.tu-chemnitz.de>
% 
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
% 
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
% 
%    You should have received a copy of the GNU General Public License
%    along with this program. If not, see <http://www.gnu.org/licenses/>.
%
%    Minor contributions to this program (for example bug-fixes and
%    minor extensions) by third parties automatically transfer the
%    copyright to the general author of FEINS, to maintain the
%    possibility of commercial re-licensing. If you contribute but wish
%    to keep the copyright of your contribution, make that clear in
%    your contribution!
% 
%    Non-GPL licenses to this program are available upon request from
%    the author.
%
% ***********************************************************************


function [rpar0,Aspar,bspar,Crpar,drpar]=pedcrank2_gen(fname,level0,level,ref_stop,ref_vx_max,alpha,rpar_in,Aspar,bspar,Crpar,drpar);
%
% generates a mesh file fname for the pedal crank
%
%   ___+____ _____+____
%  /        +          \
% +  <>  (=======)   O  +
%  \___ ____+_____ ____/
%      +          +
%
%
% return:
%   rpar0  - the initial parameters of the geometry, the vector to be
%            optimised
%   Aspar  
%   bspar  - spar=Aspar*rpar+bspar gives the parameter in the meshfile
%            in dependance on the actual parameters
%   Crpar
%   drpar  - Crpar*rpar+drpar >=0 form the constraints on the actual
%            parameters 
diamond_shaft = 1.0;
radius_pedal  = 1.0;

if nargout>1
  rpar0=[
      -3.0
      +1.0
      +1.0
      +2.7
      +2.0
      +0.0
      +4.0
      +2.3
      +2.0
      +0.0
      +8.5
      +2.5
      +1.8
      +0.0
      +12.0
      -2.3
      +2.5
      +0.5
      +5.0
      +0.5
      +1.5
      +0.0
      +8.0
      -0.5
        ];

  circ_par = 0.551948775598299; % direction parameter for circles, radius=1
  
  nrpar=length(rpar0);
  nspar=29+4+1+2+12; %29 for outer shape, 
                     %4 for pedal left,right,top,bottom
                     %1 for horizontal pedal center
                     %2 for direction parameter of pedal circle (+-)
                     %12 for inner shape (hole)
  Aspar=[speye(nspar,16),spalloc(nspar,8,8*2)];
  bspar=zeros(nspar,1);
  % A=eye -> direct dependency of parameters 0--15 is done
  % symmetry for bottom side:
  Aspar(16+1,2+1) =1;
  Aspar(17+1,3+1) =-1;
  Aspar(18+1,4+1) =-1;
  Aspar(19+1,5+1) =+1;
  Aspar(20+1,6+1) =1;
  Aspar(21+1,7+1) =-1;
  Aspar(22+1,8+1) =-1;
  Aspar(23+1,9+1) =+1;
  Aspar(24+1,10+1)=1;
  Aspar(25+1,11+1)=-1;
  Aspar(26+1,12+1)=-1;
  Aspar(27+1,13+1)=+1;
  bspar(28+1)   =0.0;
  bspar(29+1)   =10.0-radius_pedal;
  bspar(30+1)   =10.0+radius_pedal;
  bspar(31+1)   =+radius_pedal;
  bspar(32+1)   =-radius_pedal;
  bspar(33+1)   =10.0;
  bspar(34+1)   =+circ_par*radius_pedal;
  bspar(35+1)   =-circ_par*radius_pedal;
  Aspar(36+1,16+1)=1;
  Aspar(37+1,17+1)=1;
  Aspar(38+1,18+1)=1;
  Aspar(39+1,19+1)=1;
  Aspar(40+1,20+1)=1;
  Aspar(41+1,21+1)=1;
  Aspar(42+1,22+1)=1;
  Aspar(43+1,23+1)=1;
  Aspar(44+1,18+1)=1;
  Aspar(45+1,19+1)=-1;
  Aspar(46+1,20+1)=-1;
  Aspar(47+1,21+1)=+1;



  Crpar=zeros(23,nrpar);
  drpar=zeros(size(Crpar,1),1);
  keepdist=0.3;
  kdhole  =0.01;
  min_width=max([diamond_shaft,radius_pedal])*(1+keepdist);
  
  % constraints
  % min thickness left and right:
  row=1;Crpar(row, 0+1)=-1;  drpar(row)=-diamond_shaft*(1+keepdist);
  row=2;Crpar(row,14+1)= 1;  drpar(row)=-radius_pedal *(1+keepdist)-10.0;
  % min_width of outer shape:
  for i=0:2
    row=i*3+3;Crpar(row,i*4+3+1)= 1;  drpar(row)=-min_width;
    row=i*3+4;Crpar(row,i*4+3+1)= 1;
              Crpar(row,i*4+5+1)= 1;  drpar(row)=-min_width;
    row=i*3+5;Crpar(row,i*4+3+1)= 1;
              Crpar(row,i*4+5+1)=-1;  drpar(row)=-min_width;
  end
  % avoid degenerate parameterization of the length:
  row=12;Crpar(row,6+1)= 1;
         Crpar(row,2+1)=-1;   drpar(row)=-keepdist;
  row=13;Crpar(row,10+1)= 1;
         Crpar(row,6 +1)=-1;  drpar(row)=-keepdist;
  row=14;Crpar(row, 1+1)= 1;  drpar(row)=-keepdist;
  row=15;Crpar(row,15+1)=-1;  drpar(row)=+keepdist;

  row=16;Crpar(row,16+1)= 1;  drpar(row)=-diamond_shaft*(1+keepdist);
  row=17;Crpar(row,22+1)=-1;  drpar(row)=-radius_pedal *(1+keepdist)+10.0;
  row=18;Crpar(row,17+1)= 1;  drpar(row)=-kdhole;
  row=19;Crpar(row,23+1)=-1;  drpar(row)=-kdhole;

  row=20;Crpar(row,19+1)= 1;  drpar(row)=-kdhole;  
  row=21;Crpar(row,19+1)= 1;
         Crpar(row,21+1)= 1;  drpar(row)=-kdhole;;
  row=22;Crpar(row,19+1)= 1;
         Crpar(row,21+1)=-1;  drpar(row)=-kdhole;;

  row=23;Crpar(row,22+1)= 1;
         Crpar(row,16+1)=-1;  drpar(row)=-keepdist;
else
  rpar=rpar_in;
  
  nspar=size(Aspar,1);
  spar = Aspar*rpar + bspar;
  
  % test constraints
  constr=Crpar*rpar+drpar;
  if sum(constr<-1e-7)>1
    violated_constr=(constr<0)'*constr
  end
  % open the meshfile
  mf=fopen(fname,'w');
  
  fprintf(mf,[...
      '<!-- mesh file for flow around an obstacle \n',...
      '-->\n',...
      '<header>\n',...y
      '  <format    0 >\n',...
      '  <dim       2 >\n',...
      '  <problem   3 >\n',...
      '  <meshgen   1 >\n',...
      '  <vertex    20 >\n',...
      '  <elements  0 >\n',...
      '  <boundary  20 >\n',...
      '  <holes     3 >\n',...
      '  <pcsurf    0 >\n',...
      '  <pcvol     0 >\n',...
      '  <pccrit    2 >\n',...
      '  <function  2 >\n',...
      '  <parameter 2 >\n',...
      '  <shape_seg 16 >\n',...
      '  <shape_par %d >\n',...
      '</header>\n',...
      '\n'],...
          nspar);
  fprintf(mf,[
      '<solver_settings>\n',...
      '  <refine_ini      %d >\n',...
      '  <refine_type     1 >\n',...
      '  <refine_steps    %d >\n',...
      '  <refine_stop     %e > \n', ...
      '  <refine_max_vx   %e > \n', ...
      '  <adap_mark       0 >\n',...
      '  <adap_mark_par   0.1 >\n',...
      '  <solver          3       >\n',...
      '  <solver_atol     1.0e-12 >\n',...
      '  <solver_ini_rtol 1.0e-6  >\n',...
      '  <solver_ref_rtol 1.0e-2  >\n',...
      '  <write_mesh      1 >\n',...
      '  <write_ssegs     1 >\n',...
      '</solver_settings>\n'],...
          level0, level, ref_stop, ref_vx_max);
  fprintf(mf,[
      '\n',...
      '<!-- syntax of a vertex entry is:\n',...
      '       id x y [z]\n',...
      '-->\n',...
      '<vertex>\n']);
%              id   x         y
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 0, -diamond_shaft,  0.0);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 1, 0.0, +diamond_shaft);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 2, +diamond_shaft,  0.0);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 3, 0.0, -diamond_shaft);

fprintf(mf,'  %3d  %22.15e  %22.15e\n', 4, spar(29+1),spar(28+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 5, spar(33+1),spar(31+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 6, spar(30+1),spar(28+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 7, spar(33+1),spar(32+1));

fprintf(mf,'  %3d  %22.15e  %22.15e\n', 8, spar(0+1),spar(28+1));
for id=9:11
  fprintf(mf,'  %3d  %22.15e  %22.15e\n',id, spar((id-9)*4+2+1), spar((id-9)*4+3+1));
end
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 12, spar(14+1),spar(28+1));
for id=13:15
  fprintf(mf,'  %3d  %22.15e  %22.15e\n',id, spar(-(id-13)*4+24+1), spar(-(id-13)*4+25+1));
end
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 16, spar(36+1),spar(28+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 17, spar(38+1),spar(39+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 18, spar(42+1),spar(28+1));
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 19, spar(44+1),spar(45+1));

fprintf(mf,[
  '</vertex>\n',...
  '\n',...
  '<!-- syntax of a element entry is:\n',...
  '       id type func nod1 nod2 ... nodk\n',...
  '-->\n',...
  '<elements>\n',...
  '</elements>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a boundary entry is:\n',...
  '       id bctype type func orient sseg nod1 ... nodk\n',...
  '-->\n',...
  '<boundary>\n']);

% diamond
for i=1:4
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1,   1,     0,   0,   1,    -1,  i-1, mod(i,4)  );
%     id bctype   type func orient sseg nod1 ... nodk
end
% circle
for i=1:4
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1+4, 2,     0,   1,   1,   i-1,  i+3, mod(i,4)+4  );
%     id   bctype type func orient sseg nod1 ... nodk
end
% outer boundary
for i=1:8
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1+8, 0,     0,  -1,   1,   i+3,  i+7, mod(i,8)+8  );
%     id   bctype type func orient sseg nod1 ... nodk
end
% inner boundary (hole)
for i=1:4
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1+16, 0,     0,  -1,   1, i+11, i+15, mod(i,4)+16  );
%     id   bctype type func orient sseg nod1 ... nodk
end

fprintf(mf,[
  '</boundary>\n',...
  '\n',...
  '<!--  syntax of a holes entry is:\n',...
  '       id x y [z]\n',...
  '-->\n',...
  '<holes>\n',...
  '  0    %22.15e   %22.15e\n',...
  '  1    %22.15e   %22.15e\n',...
  '  2    %22.15e   %22.15e\n',...
  '</holes>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a pcsurf entry is:\n',...
  '       id c_id orient nod1 ... nodk\n',...
  '-->\n',...
  '<pcsurf>\n'], 0.0,0.0,  10.0,0.0,  0.5*(spar(36+1)+spar(42+1)),0.0);

fprintf(mf,[
  '</pcsurf>\n',...
  '\n',...
  '<!-- syntax of a pcvol entry is:\n',...
  '       id c_id elem\n',...
  '-->\n',...
  '<pcvol>\n',...
  '</pcvol>\n',...
  '   \n',...
  '\n',...
  '<!-- syntax of a pccrit entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<pccrit>\n',...
  '   0    12   1.0\n',...
  '   1    11   %22.15e\n',...
  '</pccrit>\n',...
  '\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a function entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<function>\n',...
  '   0  100   0.0      0.0\n',...
  '   1  100   0.0   -100.0\n',...
  '</function>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of the parameter entry is:\n',...
  '       data1 ... datak\n',...
  '-->\n',...
  '<parameter>\n',...
  '   1.15e5  7.7e4 \n',...
  '</parameter>\n',...
  '\n\n',...
  '<!-- syntax of the shape_segm entry is:\n',...
  '       id   type  nod1   para1 ... parak\n',...
  '-->\n',...
    '<shape_seg>\n'],alpha);
fprintf(mf, '  %3d 2 %3d    %3d  %3d  %3d  %3d  %3d  %3d  %3d  %3d\n',... 
        0,   4,  29,28,  33,31,  28,34,  34,28, ... % circle
        1,   5,  33,31,  30,28,  34,28,  28,35, ...
        2,   6,  30,28,  33,32,  28,35,  35,28, ...
        3,   7,  33,32,  29,28,  35,28,  28,34, ...
        4,   8,   0,28,   2,3,   28,1,    4,5,  ... % outer boundary
        5,   9,   2,3,    6,7,    4,5,    8,9,  ...
        6,  10,   6,7,   10,11,   8,9,   12,13, ...
        7,  11,  10,11,  14,28,  12,13,  28,15, ...
        8,  12,  14,28,  24,25,  28,15,  26,27, ...
        9,  13,  24,25,  20,21,  26,27,  22,23, ...
        10, 14,  20,21,  16,17,  22,23,  18,19, ...
        11, 15,  16,17,   0,28,  18,19,  28,1,  ...
        12, 16,  36,28,  38,39,  28,37,  40,41, ... % inner boundary
        13, 17,  38,39,  42,28,  40,41,  28,43, ...
        14, 18,  42,28,  44,45,  28,43,  46,47, ...
        15, 19,  44,45,  36,28,  46,47,  28,37  ...
        );
%      id  nod1  x0,y0   x1,y1  sx0,sy0 sx1,sy1 

fprintf(mf,[
    '</shape_seg>\n',...
    '\n']);

  fprintf(mf,[
      '<!-- syntax of the shape_para entry is:\n',...
      '       id  data\n',...
      '-->\n',...
      '<shape_par>\n']);

  for i=1:nspar
    fprintf(mf, '   %3d   %22.15e\n', i-1,  spar(i)); 
  end

  fprintf(mf,[
      '</shape_par>\n',...
      '\n' ]);

  fclose(mf);
end

return;
