function convert_f1m_boundary_step_prism(basename, thickness)
% function convert_f1m_boundary_step_prism(basename, thickness)
%
% generates a basename.tcl file, which if processed with OpenCASCADE
% DRAWEXE will produce a basename.step file, that contains a boundary
% description of a prismatic body, which is the 3D prismatic extension
% of the 2D domain described by the boundary in basename.f1m
%
% arguments:
%   basename  - first part of the filenames 
%   thickness - thickness of the prismatic extension
%
  
% ***********************************************************************
%
%    This file is part of
%    FEINS, Finite Element Incompressible Navier-Stokes solver,
%    which is expanding to a more general FEM solver and toolbox,
%    Copyright (C) 2003--2013, Rene Schneider 
%    <rene.schneider@mathematik.tu-chemnitz.de>
% 
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
% 
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
% 
%    You should have received a copy of the GNU General Public License
%    along with this program. If not, see <http://www.gnu.org/licenses/>.
%
%    Minor contributions to this program (for example bug-fixes and
%    minor extensions) by third parties automatically transfer the
%    copyright to the general author of FEINS, to maintain the
%    possibility of commercial re-licensing. If you contribute but wish
%    to keep the copyright of your contribution, make that clear in
%    your contribution!
% 
%    Non-GPL licenses to this program are available upon request from
%    the author.
%
% ***********************************************************************


name_in=[basename,'.f1m']
fin=fopen(name_in,'r');

if (fin==-1)
  error(sprintf('could not open file %s for read',name_in));
end

name_step=[basename,'.step']

name_tcl=[basename,'.tcl']
ftcl=fopen(name_tcl,'w');

if (ftcl==-1)
  error(sprintf('could not open file %s for write',name_tcl));
end

% read the header, assumme order:
% vertex ... boundary ... shape_seg ... shape_par
function nr=read_header(name,hfin)
string=''; count=1;
while ((strcmp(string,['<',name])==0)&&(count~=0))
  [string,count]=fscanf(hfin,'%s',1);
end

if ((strcmp(string,['<',name])==1))
  [nr,count]=fscanf(fin,'%d',1);
else 
  error(['could not find ',name,' header']);
end
end % function read_header

vx_nr=read_header('vertex', fin);
bd_nr=read_header('boundary', fin);
sg_nr=read_header('shape_seg', fin);
sp_nr=read_header('shape_par', fin);

function goto_block(name,hfin)
string=''; count=1;
while ((strcmp(string,['<',name,'>'])==0)&&(count~=0))
  [string,count]=fscanf(hfin,'%s',1);
end

if ((strcmp(string,['<',name,'>'])==1))
  %
else 
  error(['could not find start of ',name,' block']);
end
end % function goto_block


% first search the vertex block, read it
goto_block('vertex',fin);
Cvertex=textscan(fin,'%d%f%f',vx_nr);
vertex=[Cvertex{2},Cvertex{3}];

% then the boundary block, read it
goto_block('boundary',fin);
Cboundary=textscan(fin,'%d%d%d%d%d%d%d%d',bd_nr);
boundary_nodes=[Cboundary{7},Cboundary{8}];
boundary_sseg =Cboundary{6};

% then the shape_seg block, read it
goto_block('shape_seg',fin);
shape_seg=fscanf(fin,'%d',[3+8,sg_nr])';
shape_seg_par=shape_seg(:,4:11);

% then the shape_par block, read it
goto_block('shape_par',fin);
shape_par_scan=fscanf(fin,'%f',[2,sp_nr])';
shape_par=shape_par_scan(:,2);


% analyse connectivity of the 2d boundary

node2bd=zeros(vx_nr,2);
for bd=1:bd_nr
  for j=1:2
    node=boundary_nodes(bd,j)+1;
    if (node2bd(node,1)==0)
      node2bd(node,1)=(-1)^(j-1)*bd; % =bd for first node, -bd for second
    elseif (node2bd(node,2)==0)
      node2bd(node,2)=(-1)^(j-1)*bd; % =bd for first node, -bd for second
    else
      error(sprintf('node %d in more than 2 boundary entries',...
                    node));
    end
  end
end


% find connected strings of boundary, these will become wires.
%
% take the one with the larges bounding box as the outermost,
% this will create fc1. All other are then cut out from fc1

found_all_strings=0;
nr_strings=0;
boundary_strings=zeros(bd_nr,1);
string   = { 0 };
str_bbox = { [ 0 0 0 0 ] };
while (found_all_strings==0)
  % find first boundary segment that is not on a string yet
  bd=1;
  while ((bd<=bd_nr)&&(boundary_strings(bd)~=0)) 
    bd=bd+1;
  end
  
  if (bd>bd_nr)
    found_all_strings=1;
  else
    % this is a new string
    nr_strings=nr_strings+1;
    tstring=nr_strings;
    
    s_first_node = boundary_nodes(bd,1)+1;
    s_next_node  = boundary_nodes(bd,2)+1;
    
    string{tstring} = bd;
    boundary_strings(bd) = tstring;
    
    str_bbox{tstring} = ...
        [ min([vertex(s_first_node,1),vertex(s_next_node,1)]),...
          min([vertex(s_first_node,2),vertex(s_next_node,2)]),...
          max([vertex(s_first_node,1),vertex(s_next_node,1)]),...
          max([vertex(s_first_node,2),vertex(s_next_node,2)]) ];
    while (s_next_node~=s_first_node)
      % find other bd with s_next_node, append it to the string
      if (abs(node2bd(s_next_node,1))~=bd)
        bd=node2bd(s_next_node,1);
      else
        bd=node2bd(s_next_node,2);
      end
      if (bd>0)
        s_next_node = boundary_nodes(bd,2)+1;
      else
        s_next_node = boundary_nodes(-bd,1)+1;
      end
      string{tstring} = [string{tstring},bd];
      boundary_strings(abs(bd)) = tstring;
    
      str_bbox{tstring} = ...
          [ min([str_bbox{tstring}(1),vertex(s_next_node,1)]),...
            min([str_bbox{tstring}(2),vertex(s_next_node,2)]),...
            max([str_bbox{tstring}(3),vertex(s_next_node,1)]),...
            max([str_bbox{tstring}(4),vertex(s_next_node,2)]) ];
    end % while s_next_node~=s_first_node
  end % if else (new string)
end % while found_all_strings==0

nr_strings,

vol_max=0;
for i=1:nr_strings
  bbox=str_bbox{i};
  vol = (bbox(3)-bbox(1))*(bbox(4)-bbox(2));
  %disp(sprintf('string %d, %d bd_entries, vol=%f\n',i, length(string{i}), ...
  %             vol));
  if (vol>vol_max)
    maxi=i;
    vol_max=vol;
  end
end

stringorder=[maxi,1:maxi-1,maxi+1:nr_strings];


% start the tcl file for opencascade DRAWEXE 
fprintf(ftcl,['# tcl file generated by ' ...
              'convert_f1m_boundary_step_prism.m\n']);
fprintf(ftcl,'# form %s\n\n', name_in);

fprintf(ftcl,'pload ALL ; \n\n');



% we only take those vertices into the geo file, that are used in
% boundary definition
point_nr=0;
vx2points=-ones(vx_nr,1);
for i=1:bd_nr
  if (boundary_sseg(i)==-1)
    for j=1:2
      node=boundary_nodes(i,j)+1;
      if (vx2points(node)==-1)
        point_nr=point_nr+1;
        fprintf(ftcl,'vertex v%d   %23.15e %23.15e 0.0 ;\n',...
                point_nr,vertex(node,1),vertex(node,2));
        vx2points(node)=point_nr;
      end
    end % for j
  end % if no sseg
end % loop over boundary

% the bezier-curves dont use vertices 

fprintf(ftcl,'# end vertex \n\n');

edge_nr=0;
face_nr=0;
for str=stringorder
  thestring=string{str};
  
  string_start=edge_nr+1;
  
  for bd=thestring
    sg=boundary_sseg(abs(bd))+1;
    
    edge_nr=edge_nr+1;
    if (sg==0)
      if bd>0
        fprintf(ftcl,'edge e%d v%d v%d ;\n',...
                edge_nr,...
                vx2points(boundary_nodes(bd,1)+1),...
                vx2points(boundary_nodes(bd,2)+1) );
      else
        fprintf(ftcl,'edge e%d v%d v%d ;\n',...
                edge_nr,...
                vx2points(boundary_nodes(-bd,2)+1),...
                vx2points(boundary_nodes(-bd,1)+1) );
      end
    else % if sseg
      % the first two pairs of parameters define endpoints, these
      % have to be added if not yet present.
      for j=0:1
        spix(j*3+1)=shape_par(shape_seg_par(sg,j*2+1)+1);
        spiy(j*3+1)=shape_par(shape_seg_par(sg,j*2+2)+1);
      end % for j
    
      % the third and fourth are differences to the points, so the
      % resulting points have to be added no matter what, since they
      % are used only once, in this
    
      % third pair, (first inner point) 
      spix(2)= shape_par(shape_seg_par(sg,0*2+1)+1)...
               + shape_par(shape_seg_par(sg,2*2+1)+1);
      spiy(2)= shape_par(shape_seg_par(sg,0*2+2)+1)...
               + shape_par(shape_seg_par(sg,2*2+2)+1);

      % fourth pair, (second inner point) 
      spix(3)= shape_par(shape_seg_par(sg,1*2+1)+1)...
               - shape_par(shape_seg_par(sg,3*2+1)+1);
      spiy(3)= shape_par(shape_seg_par(sg,1*2+2)+1)...
               - shape_par(shape_seg_par(sg,3*2+2)+1);

      
      if bd>0
        fprintf(ftcl,'beziercurve bc%d  4    %23.15e %23.15e 0.0   %23.15e %23.15e 0.0   %23.15e %23.15e 0.0    %23.15e %23.15e 0.0 ;\n',...
                edge_nr,...
                spix(1), spiy(1),...
                spix(2), spiy(2),...
                spix(3), spiy(3),...
                spix(4), spiy(4) );
      else
        fprintf(ftcl,'beziercurve bc%d  4    %23.15e %23.15e 0.0   %23.15e %23.15e 0.0   %23.15e %23.15e 0.0    %23.15e %23.15e 0.0 ;\n',...
                edge_nr,...
                spix(4), spiy(4),...
                spix(3), spiy(3),...
                spix(2), spiy(2),...
                spix(1), spiy(1) );
      end
      fprintf(ftcl,'mkedge e%d  bc%d ;\n', edge_nr, edge_nr);
    end % if sseg

  end % for bd=
  string_end=edge_nr;
  
  fprintf(ftcl,'\n\n');

  % create the wire for this string, and the face
  face_nr=face_nr+1;
  
  fprintf(ftcl,'wire w%d ', face_nr);
  for i=string_start:string_end
    fprintf(ftcl,'e%d ', i);
  end
  fprintf(ftcl,';\n');
  fprintf(ftcl,'mkplane fc%d w%d ;\n\n\n', face_nr, face_nr);
end % for str=

for i=2:nr_strings
  fprintf(ftcl,'cut fc1 fc1 fc%d ;\n',i);
end

fprintf(ftcl,'\n\n\n\n');

fprintf(ftcl,'prism vol1 fc1   0 0 %23.15e;\n\n\n',thickness);

fprintf(ftcl,'stepwrite 0 vol1 "%s" ;\n\n',name_step);

fprintf(ftcl,'# end tcl file\n');

fclose(fin);
fclose(ftcl);

disp('call: ');
disp(sprintf(['(echo "source \"%s\""; sleep 15; ' ...
                'killall DRAWEXE) |DRAWEXE'], name_tcl));
%ls(name_step)

end