/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "linsolve_umfpack.h"
#include "assembly.h"
#include "gen_aux.h"
#include "mesh_deform.h"
#include "lame_adj.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

/*#include <unistd.h>*/
#include <string.h>


int main(int argc, char *argv[])
{	
  struct sparse *Ks;
  struct projector1 P;
  struct coarse_mat K_inv;
  
  struct vector psi;    /* solution of K^T * Psi = dIdu^T */
  struct vector dIdu;   /* derivate of Is with respect to the solution u */
  struct vector dIdx;  /* gradient of Is with respect to the node positions */
  struct vector dIdF;  /* gradient of Is with respect to the shape parameters */
  struct vector rhs, nodem, dIdx_FD;
  struct mesh msh1;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;
  struct bpxdata bpx;

  struct solver_settings set;
	  
  int  err, stop;
  FIDX dim, vx_nr, i, j, d;

  FIDX level0, lvlm, lvl;

  double mu, lambda;
  double globest;
  FIDX nrmarked;
  int stiffening;
  int vcycles=1;
  int smooths=4;

  double Is, Ism,Isp;
  
  int iter;
  double resi;
  double res_start;

  FIDX *marker;

  char *buffer;
  FILE *out;

  res_start = -1.0;
  stiffening = 1;  /* ==1 for real Lame', 
                      ==0 for stiffening of small elements */
  
  if (argc>=1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  solver_settings_init( &set );
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);
    

  if(set.write_ssegs!=0) /* write shape segments for visualisation */
    {
      err=mesh_write_ssegs_svg_tx( &msh1, 18, "visual/shape_segs", 1);
      FUNCTION_FAILURE_HANDLE( err, mesh_write_ssegs_svg_tx, main);  
    }   


  /* set the goals and methods */
  level0  = set.refine_ini;
  lvlm   = set.refine_steps;

  dim=msh1.dim;

  lambda=msh1.para[0];
  mu    =msh1.para[1];
  /* Lame canstants for steel
  mu         = 7.7e4; 
  lambda     = 1.15e5; */


  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  msh1.lvl=-1;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

    
  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks,  lvlm+1, struct sparse, main);

  /* Apply deformation due to shape segments */
  if (msh2.sg_nr>0)
    {
      err=mesh_deform_start_t2(&msh2, Ks);
      FUNCTION_FAILURE_HANDLE( err, mesh_deform, main);
    }    

  vx_nr=msh2.vx_nr;     
  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);  
  for (i=0; i<dim*vx_nr; i++)
    {
      nodem.V[i]=0.0;    
    }
  printf("\n Level = %d : vx_nr = %"dFIDX"\n", 0,  msh2.vx_nr);	
      
  /* Mesh Refinement */
  stop=0;
  for (lvl=0; (lvl<lvlm)&&(stop==0); lvl++)
    {  
      if(msh2.lvl+1!=lvl)
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%"dFIDX"+1!= %"dFIDX")\n",  msh2.lvl,  lvl);
	  return FAIL;
        }
      err=sparse_flex_alloc( &Ks[lvl], dim*vx_nr);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
      err=projector1_alloc( &P, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
      err=vector_alloc( &rhs, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  
      for (i=0; i<dim*vx_nr; i++) rhs.V[i]=0.0;
	  
      /* Assembly stiffness matrix with projector for Dirichlet 
	 on the entire boundary */
      err=assem_lame_tx_tx( &Ks[lvl], &P, &msh2, &rhs, &nodem,
			 lambda, mu, stiffening, 2, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);
   
      /* solve the equation system */
      if (set.solver==0) /* variant 1: direct solve (good for coarse mesh) */
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvl], P.len, P.V,
			  1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++)
	    {
	      rhs.V[P.V[i]]=nodem.V[P.V[i]];
	    }
	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
      if (set.solver==3) /* variant 3: direct solve with UMFPACK */
	{
	  err=sparse_solve_UMFPACK( &nodem, &Ks[lvl], &rhs, &P); /* */
	  FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK, main);
	}
      if((set.solver==1)||(set.solver==2))
	{
	  struct coarse_mat *cmat;
	  FIDX *coarse_bc_nodes;
	  FIDX n_c_bc;

	  /* set multilevel and multigrid data for PCG_MG or PCG_BPX */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
	  
	  /* generate the coarse grid matrix */
	  n_c_bc=0;
	  TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
	  TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
	  for (i=0; i<P.len; i++)
	    {
	      FIDX dof, child;
	      dof=P.V[i];
	      MLVLFINDENTRY(child, dof, 0, ml);
	      if (child>=0)
		{
		  coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
		  n_c_bc++;
		}
            }
	  err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			      1, cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	  /* free old cmat and add new one */
	  if (mg.cmat != NULL)
	    {
	      coarse_mat_free(mg.cmat);
	      free(mg.cmat);
	    }
	  mg.cmat=cmat;
    	  free(coarse_bc_nodes);
	  mg.vcycles=vcycles;
	  mg.smooths=smooths;
	  if (stiffening==0)
	    {
	      mg.CGC_scale=0.25;
	    }
	  else
	    {
	      mg.CGC_scale=1.0;
	    }
	}
      if(set.solver==1) /* variant 2: PCG with multigrid */
	{
	  double rtol;
	  if (lvl==0)
	    { /* coarse level, use relative tolerance for "initial" */
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    { /* subsequent level, use refinement for "refinement" */
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, 
		&iter, sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvl], &rhs, &mg); /* */
	  FUNCTION_FAILURE_HANDLE( err, PCG, main); /*  */
	  printf("PCG_MG_refine |res|=%8.1e it=%3d\n", resi, iter);
	}
      if (set.solver==2) /* variant 3: PCG with BPX */
        {
	  int iter;
	  double resi, rtol;

	  err=bpx_init_tx(&bpx, &ml);
	  FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
	  bpx.msh = &msh2;
	  bpx.P   = &P;    
	  /* bpx.cmat = mg.cmat; /* */
	  if (lvl==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_bpx_tx,
		   &Ks[lvl], &rhs, &bpx); /* */
	  /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
	  /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_BPX_refine |res|=%8.1e it=%3d\n", resi, iter);
	  bpx_free(&bpx);
	}

      /* Apply mesh refinement */
      {
	FIDX *marker;
	TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
	switch(set.refine_type)
	  {
	  case 0:
	    /* Uniform mesh refinement */
	    /* OPTIONAL: estimating the error */
	    err=lame_error_est_residual_t2(&msh2, &nodem, lambda, mu,
					   marker, &nrmarked, &globest,
					   &set, 2); 
	    FUNCTION_FAILURE_HANDLE( err, lame_error_est_residual_t2, main);
	    if ((globest<set.refine_stop)||(4*vx_nr>set.refine_max_vx))
	      {
		stop=1;
		lvlm=lvl;
	      }
	    else
	      {
		/* refine the mesh */
		err=mesh_refine_uniform_t2( &msh2);
		FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	      }
	    break;
	  case 1:
	    /* Adaptive mesh refinement , residual error estimator */
	    err=lame_error_est_residual_t2(&msh2, &nodem, lambda, mu,
					   marker, &nrmarked, &globest,
					   &set, 2); 
	    FUNCTION_FAILURE_HANDLE( err, lame_error_est_residual_t2, main);

	    if ((globest<set.refine_stop)||(vx_nr+4*nrmarked>set.refine_max_vx))
	      {
		stop=1;
		lvlm=lvl;
	      }
	    else
	      {
		/* refine the mesh */
		err=mesh_refine_adaptive_t2( &msh2, marker);
		FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	      }
	    break;
	  case 2:
	    /* Adaptive mesh refinement, ZZ error estimator */
	    err=gen_error_est_ZZ_tx(&msh2, dim, &nodem, 
				    marker, &nrmarked, &globest, &set, 2); 
	    FUNCTION_FAILURE_HANDLE( err, gen_error_est_ZZ_tx, main);

	    if ((globest<set.refine_stop)||(vx_nr+4*nrmarked>set.refine_max_vx))
	      {
		stop=1;
		lvlm=lvl;
	      }
	    else
	      {
		/* refine the mesh */
		err=mesh_refine_adaptive_t2( &msh2, marker);
		FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	      }
	    break;
	  default:
	    printf("Error in main (test_lame.c): unknown refinement type!"
		   " type = %"dFIDX"\n",  set.refine_type);
	    return FAIL;
	  }
	free(marker);
      }
      if ( ((stop==0)&&(msh2.lvl!=lvl))
	   ||((stop==1)&&(msh2.lvl!=lvl-1)))
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%"dFIDX" != %"dFIDX")\n",  msh2.lvl,  lvl);
	  return FAIL;
        }
      printf("\n Level = %"dFIDX" : vx_nr = %"dFIDX"    stop = %d\n",
	      msh2.lvl,  msh2.vx_nr,  stop);


      if (stop==0)
        {
	  /* free all solver data that is not valid for the refined
	     mesh */
	  if((set.solver==1)||(set.solver==2))
	    {
	      /* multigrid data and multilevel are no longer required */
	      mg_free(&mg);
	      multilvl_free(&ml);
	    }
	  /* rhs and projector are no longer required */
	  projector1_free(&P);
	  vector_free(&rhs);
	}

      /* interpolate the solution vector nodem to the subsequent level */
      if (stop==0)
        {
	  FIDX node;
	  /* redefine multilvl data */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( NULL, &msh2, &ml, &mg, NULL);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
      
	  /* Remark: vx_nr is still the number of vertices on the coarser
	     mesh (lvl), msh2.vx_nr is the number on the finer mesh (lvl+1) */
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*msh2.vx_nr, lvl, ml);
		mg.xl[node]=nodem.V[i+d*vx_nr];
	      }
	      
	  err=mg_interpolate_t2( &mg, lvl, lvl+1, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2, main);  
	      
	  /* Now, Re-Alloc and copy back to the solution vector */    
	  vx_nr=msh2.vx_nr;
	  vector_free(&nodem);
	  err=vector_alloc( &nodem, dim*vx_nr );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main); 
	  for (i=0; i<dim*vx_nr; i++) 
	    {
	      nodem.V[i]=0.0;    
	    }
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*vx_nr, lvl+1, ml);
		nodem.V[i+d*vx_nr]=mg.xl[node];
	      }
	  multilvl_free(&ml);	
	  mg_free(&mg);

	  /* adjust the new nodes to math the shape segments */
	  err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
	  FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 
	}


    } /* End of mesh refinement */

  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);

  vx_nr=msh2.vx_nr; 

  if (stop==0)
    {
      /* reached finest level because max levels exhausted,
         need to solve once more */
      /* allocate memory for matrix Ks, P and the vectors */
      err=sparse_flex_alloc( &Ks[lvlm], dim*vx_nr);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
      err=projector1_alloc( &P, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
      err=vector_alloc( &rhs, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      /* set vectors to zero */
      for (i=0; i<dim*vx_nr; i++)
	{
	  rhs.V[i]  =0.0;
	}

      /* assemble the system on the fine mesh */
      err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
			    lambda, mu, stiffening, 2, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);


      /* solve the equation system */
      if (set.solver==0) /* variant 1: direct solve */
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			      1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
      if (set.solver==3) /* variant 3: direct solve with UMFPACK */
	{
	  err=sparse_solve_UMFPACK( &nodem, &Ks[lvlm], &rhs, &P); /* */
	  FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK, main);
	}
      if ((set.solver==1)||(set.solver==2))
	{ 
	  struct coarse_mat *cmat;
	  FIDX *coarse_bc_nodes;
	  FIDX n_c_bc;
      /* set multilevel and multigrid data for PCG_MG and PCG_BPX */
	  /* define the multigrid data */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
      
      
	  mg.vcycles=vcycles;
	  mg.smooths=smooths;
	  if (stiffening==0)
	    mg.CGC_scale=0.25;
	  else
	    mg.CGC_scale=1.0;

	  /* generate the coarse grid matrix */
	  n_c_bc=0;
	  TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
	  TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
      
	  for (i=0; i<P.len; i++)
	    {
	      FIDX dof, child;
	      dof=P.V[i];
	      MLVLFINDENTRY(child, dof, 0, ml);
	      if (child>=0)
		{
		  coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
		  n_c_bc++;
		}
	    }
	  err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			      1, cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	  mg.cmat=cmat;
      
	  free(coarse_bc_nodes);
	}
      if (set.solver==1) /* variant 2: PCG_MG */
	{
	  double rtol;
	  if (lvlm==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_MG_tx,
		   &Ks[lvlm], &rhs, &mg); /* */
	  FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_MG_main   |res|=%8.1e it=%3d\n", resi, iter);
	}
      if (set.solver==2) /* variant 3: PCG with BPX */
	{
	  int iter;
	  double resi, rtol;
      
	  err=bpx_init_tx(&bpx, &ml);
	  FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
	  bpx.msh = &msh2;
	  bpx.P   = &P;    
	  bpx.cmat = mg.cmat; /* */
	  if (lvlm==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_bpx_tx,
		   &Ks[lvl], &rhs, &bpx); /* */
	  /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
	  FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_BPX_refine |res|=%8.1e it=%3d\n", resi, iter);
	  bpx_free(&bpx);
	}
      /* OPTIONAL: estimate error */
      TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
      err=lame_error_est_residual_t2(&msh2, &nodem, lambda, mu, 
				     marker, &nrmarked, &globest, &set, 2);   
      FUNCTION_FAILURE_HANDLE( err, lame_error_est_residual_t2, main);
      free(marker); 
    } /* end if (stop==0) */

  if(set.write_mesh!=0) /* write data for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &nodem, dim, NULL, 0,
                              16, "visual/Lame_sol" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   


  err=vector_alloc( &dIdu, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &psi, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdx, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdF, msh2.sp_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  
  /* set vectors to zero */
  for (i=0; i<dim*vx_nr; i++)
    {
      psi.V[i]  =0.0;
      dIdu.V[i] =0.0;
      dIdx.V[i] =0.0;
    }
   
  /* Compute value of pccrits and the rhs for the adjoint problem */
  err=lame_dIdu_t2(&msh2, &nodem, &dIdu, &Is, lambda, mu);
  FUNCTION_FAILURE_HANDLE( err, lame_dIdu_t2, main);
  printf("I = %12.10e \n", Is);   
    

  if (set.solver==0) /* variant 1: direct solve */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* project rhs */
      for (i=0; i<P.len; i++) dIdu.V[P.V[i]]=0.0; 

      err=coarse_mat_solve( &K_inv, NoTrans, &dIdu, &psi);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      coarse_mat_free(&K_inv);
    }
  if (set.solver==3) /* variant 3: direct solve with UMFPACK */
    {
      /* need to re-assemble the matrix, because sparse_solve_UMFPACK
	 modified it */
      err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
			    lambda, mu, stiffening, 2, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

      err=sparse_solve_UMFPACK( &psi, &Ks[lvlm], &dIdu, &P); /* */
      FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK, main);
    }
  if (set.solver==1) /* variant 2: PCG */
    {
      /* Compute psi by solving K^T*psi = dIdu */
      err=PCG( 100, 3, set.solver_atol, set.solver_ini_rtol, 1, 
	       &psi, &resi, &iter,sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvlm], &dIdu, &mg);
      printf("PCG_MG_adj |res|=%8.1e it=%3d\n", resi, iter);
    }
  if (set.solver==2) /* variant 3: PCG with BPX */
    {
      int iter;
      double resi;
      
      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh2;
      bpx.P   = &P;    
      bpx.cmat = mg.cmat; /* */
      err=PCG( 100, 3, set.solver_atol, set.solver_ini_rtol, 1, 
	       &nodem, &resi, &iter, sparse_mul_mat_vec, gen_proj_bpx_tx,
	       &Ks[lvlm], &dIdu, &bpx); /* */
      /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
      /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      printf("PCG_bpx_refine |res|=%8.1e it=%3d\n", resi, iter);
      bpx_free(&bpx);
    }

  if(set.write_mesh!=0) /* write adjoint for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &psi, dim, NULL, 0,
                              16, "visual/Lame_psi" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   

  /* use psi for computing the dIdx of Is */
  err=lame_DIDx_t2(&msh2, lambda, mu, &psi, 
	    &nodem, &dIdx);
  FUNCTION_FAILURE_HANDLE( err, lame_DIDx_t2, main);

  if(set.write_mesh!=0) /* write mesh-gradient for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &dIdx, dim, NULL, 0,
                              17, "visual/Lame_DIDX" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   


  
  /* for tests compute derivatives also by formula from paper Allair
     et. al. (2004) */
  if (0==1)
    {
      struct vector dIdx_allaire, diff_allaire;
      if (0==1)
	{
	  /* compute J=0.5*(u^T K u) + rhs^T u to check implementation
	     of compliance functional */

	  struct vector hlp;
	  double J;

	  err=vector_alloc( &hlp, nodem.len );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);


	  for (i=0; i<rhs.len; i++)
	    {
	      rhs.V[i]  =0.0;
	      hlp.V[i]  =0.0;
	    }

	  err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &hlp,
				lambda, mu, stiffening, 2, 2);
	  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx_tx, main);

	  for (i=0; i<rhs.len; i++)
	    {
	      hlp.V[i]  =0.0;
	    }

	  /* multiply hlp= K*nodem */
	  err=sparse_mul_mat_vec(&Ks[lvlm], &nodem, &hlp);
	  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, main);

	  /* J=nodem'*(0.5*hlp+rhs) */
	  J=0.0;
	  for (i=0; i<rhs.len; i++)
	    {
	      J+=nodem.V[i]*(0.5*hlp.V[i]+rhs.V[i]); 
	    }
	  
	  printf("J=%e\n",J);
	  /* is OK, value also computed with FEMLAB:
	     Domain integral 
	      0.5*(sx_pn*ex_pn+sy_pn*ey_pn+2*sxy_pn*exy_pn)
	         =1.021971e-4
	     Boundary integral 
              u*1+v*0
                 =2.043942e-4
	     so  J=1.021971e-4+2.043942e-4
	       J = 3.0659e-04
	     FEINS gave 
               I = 3.0661644109e-04  (dIdu function)
               J = 3.066164e-04      (0.5*u'*K*u+u'*rhs)
	  */

	}


      err=vector_alloc( &dIdx_allaire, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      err=lame_Allaire_et_al_2004__formula_8_t2( 
	                  &msh2, lambda, mu, &nodem, &dIdx_allaire);
      FUNCTION_FAILURE_HANDLE( err, lame_Allaire_et_al_2004__formula_8_t2,
			       main);

      err=mesh_write_solution_vtk_t2( &msh2, &dIdx_allaire, dim, NULL, 0,
				      25, "visual/Lame_DIDX_allaire" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			       main);

      err=vector_alloc( &diff_allaire, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      for (i=0; i<dIdx_allaire.len; i++)
	{
	  diff_allaire.V[i]=(/*50.0/68.0* */ 1.5*dIdx_allaire.V[i]-dIdx.V[i]);
	  /*printf("allaire/DAM[%5d] = %16.7e\n", i,dIdx_allaire.V[i]/dIdx.V[i]);*/
	}
      err=mesh_write_solution_vtk_t2( &msh2, &diff_allaire, dim, NULL, 0,
				      30, "visual/Lame_DIDX_diff_allaire" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			       main);

      if (0==1)
	{
	  /* restrict the derivatives back to the level 0 mesh, to be
	     able to compare convergence */
	  FIDX node,bd;

	  /* redefine multilvl data */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( NULL, &msh2, &ml, &mg, NULL);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);

      
	  /* copy stuff from fine mesh into mg.xl, mg.bl */
	  for (i=0; i<dim*vx_nr; i++)
	    {
	      MLVLFINDENTRY(node, i, lvlm, ml);
	      mg.xl[node]=dIdx.V[i];
	      mg.bl[node]=dIdx_allaire.V[i];
	    }

	  /* restrict to coarsest mesh */
	  err=mg_restrict_t2( &mg, lvlm, 0, mg.xl);
	  FUNCTION_FAILURE_HANDLE( err, mg_restrict_t2, main);
	  err=mg_restrict_t2( &mg, lvlm, 0, mg.bl);
	  FUNCTION_FAILURE_HANDLE( err, mg_restrict_t2, main);

	  /* interpolate back to fine mesh for plotting */
	  err=mg_interpolate_t2( &mg, 0, lvlm, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2, main);  
	  err=mg_interpolate_t2( &mg, 0, lvlm, mg.bl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2, main);  
	      
	  /* copy back all boundary points */    
	  for (bd=0; bd<msh2.bd_nr; bd++)
	    {
	      FIDX eg=msh2.bound[bd*msh2.bd_w+MCT2BDEDGE];

	      /* work only on finest boundary edges */
	      if (msh2.edge[eg*msh2.eg_w + MCT2EGCHL1] == -1)
		{
		  for (i=0; i<3; i++)
		    {
		      FIDX nodei=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+i];
		      for (d=0; d<dim; d++)
			{
			  MLVLFINDENTRY(node, nodei+d*vx_nr, lvlm, ml);
			  dIdx.V[nodei+d*vx_nr]        =mg.xl[node];
			  dIdx_allaire.V[nodei+d*vx_nr]=mg.bl[node];

			  /* re-compute difference */
			  diff_allaire.V[nodei+d*vx_nr]=
			    dIdx.V[nodei+d*vx_nr]
			    -1.5*dIdx_allaire.V[nodei+d*vx_nr];
			}
		    }
		}
	    }

	  /* write those to files */
	  err=mesh_write_solution_vtk_t2( &msh2, &dIdx, dim, NULL, 0,
					  26, "visual/Lame_coarseBD_DIDX" );
	  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
				   main);

	  err=mesh_write_solution_vtk_t2( &msh2, &dIdx_allaire, dim, NULL, 0,
					  34, "visual/Lame_coarseBD_DIDX_allaire" );
	  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
				   main);

	  err=mesh_write_solution_vtk_t2( &msh2, &diff_allaire, dim, NULL, 0,
					  39, "visual/Lame_coarseBD_DIDX_diff_allaire" );
	  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
				   main);


	  multilvl_free(&ml);	
	  mg_free(&mg);
	} /* end restrict back to coarsest mesh */

      if (1==1)
	{
	  double Rs[]={1.0, 2.0};
	  /*
	    plot "visual/plot_phi_DIDX_allaire_R0.txt" with lines,"visual/plot_phi_DIDX_R0.txt" with lines, "visual/plot_phi_diff_allaire_R0.txt" with lines
	    
	    plot "visual/plot_phi_DIDX_allaire_R1.txt" with lines,"visual/plot_phi_DIDX_R1.txt" with lines, "visual/plot_phi_diff_allaire_R1.txt" with lines
	  */
	  err=mesh_gnuplot_boundary_circles_t2( 2, Rs, 0.01, &msh2, &dIdx, dim, 1, 
				    "visual/plot_phi_DIDX");
	  FUNCTION_FAILURE_HANDLE( err, mesh_gnuplot_boundary_circles_t2, main);
	  
	  err=mesh_gnuplot_boundary_circles_t2( 2, Rs, 0.01, &msh2, &dIdx_allaire, dim, 1, 
				    "visual/plot_phi_DIDX_allaire");
	  FUNCTION_FAILURE_HANDLE( err, mesh_gnuplot_boundary_circles_t2, main);

	  err=mesh_gnuplot_boundary_circles_t2( 2, Rs, 0.01, &msh2, &diff_allaire, dim, 1, 
				    "visual/plot_phi_diff_allaire");
	  FUNCTION_FAILURE_HANDLE( err, mesh_gnuplot_boundary_circles_t2, main);
	}

    
      vector_free(&dIdx_allaire);
      vector_free(&diff_allaire);
    } /* end test derivatives Allaire et al paper */

  /* test the Eppler conjecture */
  if (0==1)
    {
      double Rs[]={1.0, 2.0};
      struct vector sdifference;

      err=vector_alloc( &sdifference, vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      err=lame_eppler_conjecture_2010_rem3_3( &msh2, lambda, mu,
					      &nodem, &sdifference);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx_tx, main);

      err=mesh_write_solution_vtk_t2( &msh2, NULL, 0, &sdifference, 1,
				      31, "visual/Lame_eppler_sdifference" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
				   main);

      err=mesh_gnuplot_boundary_circles_t2( 2, Rs, 0.01, &msh2, &sdifference, 1, 1, 
				"visual/plot_phi_eppler_sdifference");
      FUNCTION_FAILURE_HANDLE( err, mesh_gnuplot_boundary_circles_t2, main);

      vector_free(&sdifference);
    }




  /* compute DIDF from DIDx */
  /*                       msh, insert, nI, dIdx,  dIdF, type */
  err=mesh_sseg_adjust_tx( &msh2, 0,     1, &dIdx, &dIdF, 2); 
  FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main);


    
  /* FINITE DIFFERENCE FOR dIdF_FD */
  if (0==1)
    {
      double a,b,c;

      err=vector_alloc( &dIdx_FD, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      for (j=0; j<0*(msh2.sp_nr); j++)
	{
	  double hlp=msh2.spar[j];
	  double h;
	  h= 1e-6;

	  /* +h */
	  msh2.spar[j] =hlp+h;

	  err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
	  FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 

	  for (i=0; i<nodem.len; i++)
	    {
	      nodem.V[i]=0.0;
	      rhs.V[i]  =0.0;
	    }

	  err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
				lambda, mu, stiffening, 2, 2);
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			      1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
	Isp=0.0;
	err=lame_dIdu_t2( &msh2, &nodem, &dIdu, &Isp, lambda, mu);

	/* -h */
	msh2.spar[j] =hlp-h;

	err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
	FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 

	for (i=0; i<nodem.len; i++)
	  {
	    nodem.V[i]=0.0;
	    rhs.V[i]  =0.0;
	  }

	err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
			      lambda, mu, stiffening, 2, 2);
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			      1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
	Ism=0.0;
	err=lame_dIdu_t2( &msh2, &nodem, &dIdu, &Ism, lambda, mu);

	/* +-0 */
	msh2.spar[j] =hlp;
      
	/* auswerten FD */
	dIdx_FD.V[j]=(Isp-Ism)/(2*h);
	
	a=dIdF.V[j];
	b=dIdx_FD.V[j];
	c=fabs((a-b)/b);
	if (c>1e-6)
	  {
	    printf("par %3"dFIDX" \t adj=%16.14f \t FD=%16.14f \t rel=%f\n",  j, a, b, c);
	  }
	}
      vector_free(&dIdx_FD);
      /* FINITE DIFFERENCE IS DONE */
    }

  /* write results to files */
  /* I */ 
  strcpy(buffer, argv[1]);
  strcat(buffer,"_crit");
  /* open the file */
  out=fopen(buffer, "w");
  if (out==NULL)
    {
      fprintf(stderr, "vector_write_file: "
	      "error opening file \"%s\"\n", buffer);
      return FAIL;
    }
  fprintf(out, "%+24.16e\n", Is);
  /* close the file */
  fclose(out);

  /* dIdF */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_grad");
  err=vector_write_file(&dIdF, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_n_write_file, main);

  /* free some stuff */
  if ((set.solver==1)||(set.solver==2))
    {
      mg_free(&mg);
      multilvl_free(&ml);
    }

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks[lvl]);
  free(Ks);

  projector1_free(&P);
  free(buffer);
  vector_free(&rhs);
  vector_free(&nodem);
  mesh_free(&msh1);
  mesh_free(&msh2);
  vector_free(&dIdu);
  vector_free(&psi);
  vector_free(&dIdx);
  vector_free(&dIdF);

  return SUCCESS;
}
