/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "linsolve_struct.h"
#include "linsolve_umfpack.h"
#include "assembly.h"
#include "gen_aux.h"
#include "mesh_deform.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

//#include <unistd.h>
#include <string.h>

int main(int argc, char *argv[])
{

  struct sparse *Ks;
  struct projector1 P;
  struct coarse_mat K_inv;
  struct bpxdata bpx;
  
  
  struct vector rhs, nodem, vx_old;
  struct mesh msh1;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;

  struct solver_settings set;
  
  int  err;
  FIDX dim, vx_nr, i, j, d;

  FIDX level0, lvlm, lvl;

  double res_start;
  double mu, lambda;
  int stiffening;
  int vcycles=1;
  int smooths=2;

  res_start=-1.0;
  stiffening = 1;  /* ==1 for real Lame', 
                      ==0 for stiffening of small elements */

  if (argc>1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* set refinement levels */
  level0  = set.refine_ini;
  lvlm   = set.refine_steps;

  dim=msh1.dim;

  lambda = msh1.para[0];
  mu     = msh1.para[1];
  /* Lame canstants for steel
  mu         = 7.7e4; 
  lambda     = 1.15e5; */

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  msh1.lvl=-1;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);
  mesh_free(&msh1);
    
  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks,  lvlm+1, struct sparse, main);
  
  /* Apply deformation due to shape segments */
  if (msh2.sg_nr>0)
    {
      err=mesh_deform_start_t2(&msh2, Ks);
      FUNCTION_FAILURE_HANDLE( err, mesh_deform, main); /* */
    }  

  if((set.write_mesh==1)||(set.write_mesh==2))/* WRITE INITIAL T2 MESH */
    {
      err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_ini", 2);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
    }
    
  if(set.write_mesh==3) /* WRITE INITIAL T1 MESH */
    {
      struct mesh m1;
      
      mesh_t2_to_t1(&msh2,&m1);	  
      
      err=mesh_write_mesh_gnuplot_tx( &m1, 27,
				    "visual/2d_mesh_gnuplot_ini", 1);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
      mesh_free(&m1);
    }

    
  vx_nr=msh2.vx_nr;     
  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);  
  for (i=0; i<dim*vx_nr; i++) nodem.V[i]=0.0;    
  printf("\nLevel %d, vx_nr = %"dFIDX"\n", 0, msh2.vx_nr);	

  /* Mesh Refinement */
  for (lvl=0; lvl<lvlm; lvl++)
    {  
      if(msh2.lvl+1!=lvl)
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%"dFIDX"+1!= %"dFIDX")\n", msh2.lvl, lvl);
	  return FAIL;
        }
      err=sparse_flex_alloc( &Ks[lvl], dim*vx_nr);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
      err=projector1_alloc( &P, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
      err=vector_alloc( &rhs, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
	
      for (i=0; i<dim*vx_nr; i++) rhs.V[i]=0.0;
	  
      /* Assembly stiffness matrix with projector for Dirichlet 
	 on the entire boundary */
      err=assem_lame_tx_tx( &Ks[lvl], &P, &msh2, &rhs, &nodem,
			    lambda, mu, stiffening, 2, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

      /* SOLVE */
      /* solve the equation system */
      if (set.solver==0) /* variant 1: direct solve */
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvl], P.len, P.V,
			      1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main ); 

	  /* project rhs */
	  for (i=0; i<P.len; i++)
	    {
	      rhs.V[P.V[i]]=0.0;
	    }
          err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
      if ((set.solver==1)||(set.solver==2))
        {
	  struct coarse_mat *cmat;
	  FIDX *coarse_bc_nodes;
	  FIDX n_c_bc=0;
      /* initialize multigrid data for PCG_MG or PCG_BPX */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
	
	  /* generate the coarse grid matrix */
	  TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
	  TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
	  for (i=0; i<P.len; i++)
	    {
	      FIDX dof, child;
	      dof=P.V[i];
	      MLVLFINDENTRY(child, dof, 0, ml);
	      if (child>=0)
		{
		  coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
		  n_c_bc++;
		}
            }
	  err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			      1, cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	  /* free old cmat and add new one */
	  if (mg.cmat != NULL)
	    {
	      coarse_mat_free(mg.cmat);
	      free(mg.cmat);
	    }
	  mg.cmat=cmat;
    	  free(coarse_bc_nodes);
  
	  mg.vcycles=vcycles;
	  mg.smooths=smooths;
	  if (stiffening==0) 
	    {
	      mg.CGC_scale=0.25;
	    }
	  else
	    {
	      mg.CGC_scale=1.0;
	    }
	}
      if (set.solver==1)
        {
	  int iter;
	  double resi;
	  double rtol;

	  if (lvl==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvl], &rhs, &mg); /* */
	  /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_MG_refine |res|=%8.1e it=%3d\n", resi, iter);
	}
      if (set.solver==2)
        {
	  int iter;
	  double resi, rtol;

	  err=bpx_init_tx(&bpx, &ml);
	  FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
	  bpx.msh = &msh2;
	  bpx.P   = &P;    
	  /* bpx.cmat = mg.cmat; /* */
	  if (lvl==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_bpx_tx,
		   &Ks[lvl], &rhs, &bpx); /* */
	  /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
	  /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_BPX_refine |res|=%8.1e it=%3d\n", resi, iter);
	  bpx_free(&bpx);
	}
      if ((set.solver==1)||(set.solver==2))
	{ /* multigrid data and mulitlvl are no longer valid */ 
	  mg_free(&mg);
	  multilvl_free(&ml);
	}   
      /* projector and rhs are no longer valid */  
      projector1_free(&P);
      vector_free(&rhs);

      if(0==1) /* WRITE INITIAL MESH */
	{
	  err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_ini", 2);
	  FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
	}	
      if(0==1) /* PRINT DEFORMATION IN GIVEN POINT */
	{
	  j = msh2.pccrit[MCXXPCDAT1];
	  printf("nodem (%"dFIDX"): %8.6e, %8.6e\n", 
		 j, nodem.V[0*vx_nr+j], nodem.V[1*vx_nr+j] );
	}
    
      /* Apply adaptive OR uniform mesh refinement */
      if (set.refine_type==1)
        {
	  /* error estimator */
	  FIDX *marker;
	  FIDX nrmarked;
	  double globest;

	  TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
          err=lame_error_est_residual_t2(&msh2, &nodem, lambda, mu, 
					 marker, &nrmarked, &globest,
					 &set, 2);  
          FUNCTION_FAILURE_HANDLE( err, error_est_lame_residual_t2, main);
	  // for (i=0; i<msh2.el_nr; i++) marker[i] = 1;
	  err=mesh_refine_adaptive_t2( &msh2, marker);
          FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	
	  free(marker);
        }
      if (set.refine_type==0)
        {
	  err=mesh_refine_uniform_t2( &msh2);
          FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
        }
	
      if(msh2.lvl!=lvl)
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%"dFIDX" != %"dFIDX")\n", msh2.lvl, lvl);
	  return FAIL;
        }
      printf("\nLevel %"dFIDX", vx_nr = %"dFIDX"\n", lvl+1, msh2.vx_nr);

      /* interpolate the solution vector nodem to the finer grid */
        {	
	  FIDX node;
	  /* redefine multilvl data */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( NULL, &msh2, &ml, &mg, NULL);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);


	  /* Remark: vx_nr is still the number of vertices on the
	     coarser mesh (lvl), msh2.vx_nr is the number on the 
	     finer mesh (lvl+1) */
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*msh2.vx_nr, lvl, ml);
		mg.xl[node]=nodem.V[i+d*vx_nr];
	      }
	      
	  err=mg_interpolate_t2( &mg, lvl, lvl+1, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2, main);  
	      
	  /* Now, Re-Alloc and copy back to the solution vector */    
	  vx_nr=msh2.vx_nr;
	  vector_free(&nodem);
	  err=vector_alloc( &nodem, dim*vx_nr );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
	  for (i=0; i<dim*vx_nr; i++) 
	    {
	      nodem.V[i]=0.0;
	    }
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*vx_nr, lvl+1, ml);
		nodem.V[i+d*vx_nr]=mg.xl[node];
	      }
	  multilvl_free(&ml);	
	  mg_free(&mg);
	}
	
      err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); /* */
	
    } /* END OF REFINEMENT */

  vx_nr=msh2.vx_nr;  
    
  if(0==1) /* WRITE INTERPOLATED SOLUTION */
    {
      struct mesh m1;

      double *vertex_tmp;
      TRY_MALLOC(vertex_tmp, dim*vx_nr, double, main);

      /* the computation of nodem is done, apply it to the mesh */
      for (i=0; i<vx_nr; i++)
	for (d=0; d<dim; d++)
	  {
	    vertex_tmp[d*vx_nr+i]=msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d];
	    msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += nodem.V[d*vx_nr+i];
	  }
      mesh_t2_to_t1(&msh2,&m1);	  
		  
      err=mesh_write_mesh_gnuplot_tx( &m1, 27,
				    "visual/2d_mesh_gnuplot_def", 1);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

      /* revert changes of vertex */
      for (i=0; i<vx_nr; i++)
	for (d=0; d<dim; d++)
	  {
	    msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d]=vertex_tmp[d*vx_nr+i];
	  }

      free(vertex_tmp);
    }
    
  /* allocate memory for matrix Ks and the vectors */
  err=sparse_flex_alloc( &Ks[lvlm], dim*vx_nr);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
  err=projector1_alloc( &P, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
  err=vector_alloc( &vx_old, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &rhs, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  /* set vector rhs to zero */
  for (i=0; i<dim*vx_nr; i++) rhs.V[i]=0.0;

  /* assemble the system on the fine mesh */
  err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
		     lambda, mu, stiffening, 2, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

  if (0==1) /* WRITE MATRIX AND VECTORS*/
    {
      char name[100];

      /* is already in copressed row format */
      sprintf(name,"visual/Lame_Mat_C_%07"dFIDX"_nodes.txt", vx_nr);
      err=sparse_mat_write_file(&Ks[lvlm], name);
      FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file, main);

      /* node positions */
      sprintf(name,"visual/Lame_vertes_%07"dFIDX"_nodes.txt", vx_nr);
      err=vector_write_file(&vx_old, name);
      FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);
    }
  if ((set.solver==1)||(set.solver==2))
    {  
      /* define the multigrid data */
      err=multilvl_init_tx( &msh2, dim, &ml, 2);
      FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
      err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
      FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);   
    }
  if(0==1) /* PRINT RESTRICTED VECTOR */
    { 
	  FIDX node1, node2;
	  FIDX *fathers; /* vector to check if other nodes 
				 than the fathers were changed */
	  TRY_MALLOC(fathers,vx_nr,FIDX,main);
	  printf("Test for restriction\n");  
	  /* Remark: vx_nr is the number on the finer mesh (lvl+1) */
	  for (i=0; i<vx_nr*dim; i++) nodem.V[i] = 1.0;
	  for (i=0; i<vx_nr; i++) fathers[i]=0;
	  nodem.V[333] = 1.0;
	  nodem.V[332+vx_nr] = 1.0;
	
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node1, i+d*msh2.vx_nr, lvlm, ml);
		if (node1>-1) mg.xl[node1]=nodem.V[i+d*vx_nr];
		else printf("Error1: node on last level not found!\n");
	      }
	      
	  err=mg_restrict_t2( &mg, lvlm, lvlm-1, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_restrict_t2, main);       
	      
	  for (j=ml.levlhindx[lvlm]; j<ml.levlhindx[lvlm+1]; j++)
            {   
	      printf("el_hier %"dFIDX"\n", j);	    
	      for (i=0; i<6; i++) 
		{
		  d=msh2.elhier[j*msh2.eh_w+MCT2EHFAT1+i];
		  fathers[d]=1;
		  /*MLVLFINDENTRY(node1, d+0*vx_nr, lvlm-1, ml);
		  MLVLFINDENTRY(node2, d+1*vx_nr, lvlm-1, ml);
		  if (node1>-1) 
		    {
		      printf("x%d \t%f, \t%f \n", d, mg.xl[node1],
		      mg.xl[node2]);
		    }
		  else 
		    {
		      printf("Warning: node on last level not found"
		             " (double refinement)!\n");
		    }
		  */
		}
	    }
	  for (i=0; i<vx_nr; i++) 
	    {  /* check if other nodes than the fathers were changed */
	      if (fathers[i]==0)
	        {
		  MLVLFINDENTRY(node1, i+0*vx_nr, lvlm-1, ml);
		  MLVLFINDENTRY(node2, i+1*vx_nr, lvlm-1, ml);
		  if ((node1>-1)&&(node2>-1))
		    {
		      printf("ERROR: node1=%"dFIDX", node2=%"dFIDX" \n"
			     "  for vertex i=%"dFIDX
			     " in level lvlm-1=%"dFIDX"\n",
			     node1, node2, i, lvlm-1);
		    }
		  else
		    {
		      if ( (abs(mg.xl[node1]-1)>1e-8)||(abs(mg.xl[node2])>1e-8) )
			printf("ERROR: x%"dFIDX" \thas changed!\t%f, \t%f\n", 
			       i, mg.xl[node1], mg.xl[node2]);
		    }	
		}
	    }		

	
	  for (i=0; i<vx_nr; i++) 
	    {
	      MLVLFINDENTRY(node1, i+0*vx_nr, lvlm-1, ml);
	      MLVLFINDENTRY(node2, i+1*vx_nr, lvlm-1, ml);
	      if ((node1>-1)&&(node2>-1))
		{
		  printf("restr[%3"dFIDX"]= %10f  %10f\n", i,
			 mg.xl[node1], mg.xl[node2]);
		}
	    }

	for (i=0; i<vx_nr*dim; i++) nodem.V[i] = 0.0;
    }  
    
    
  if(0==1) /* PRINT INTERPOLATED VECTOR */
    { 
	  FIDX node1, node2;
	  FIDX *fathers; /* vector to check if other nodes 
				 than the fathers were changed */
	  TRY_MALLOC(fathers,vx_nr,FIDX,main);
	  printf("Test for interpolation\n");  
	  
	  /* Remark: vx_nr is the number on the finer mesh (lvl+1) */
	  for (i=0; i<vx_nr*dim; i++) nodem.V[i] = 0.0;
	  for (i=0; i<vx_nr; i++) fathers[i]=0;
	  nodem.V[46] = 1.0;
	  nodem.V[123+vx_nr] = 1.0;
	
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node1, i+d*msh2.vx_nr, lvlm-1, ml);
		if (node1>-1) mg.xl[node1]=nodem.V[i+d*vx_nr];
	      }
	      
	  err=mg_interpolate_t2( &mg, lvlm-1, lvlm, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_restrict_t2, main);       
	      
	  for (j=ml.levlhindx[lvlm]; j<ml.levlhindx[lvlm+1]; j++)
            {   
	      printf("el_hier %"dFIDX"\n", j);	    
	      for (i=0; i<6; i++) 
		{
		  d=msh2.elhier[j*msh2.eh_w+MCT2EHFAT1+i];
		  fathers[d]=1;
		  /*MLVLFINDENTRY(node1, d+0*vx_nr, lvlm-1, ml);
		  MLVLFINDENTRY(node2, d+1*vx_nr, lvlm-1, ml);
		  if (node1>-1) 
		    {
		      printf("x%d \t%f, \t%f \n", d, 
		             mg.xl[node1], mg.xl[node2]);
		    }
		  else
		    {
		      printf("Warning: node on last level not found"
		             " (double refinement)!\n");
		    } */
		}
	      for (i=0; i<3; i++) 
		{
		  d=msh2.elhier[j*msh2.eh_w+MCT2EHCHL1+i];
		  fathers[d]=1;
		  /*MLVLFINDENTRY(node1, d+0*vx_nr, lvlm-1, ml);
		  MLVLFINDENTRY(node2, d+1*vx_nr, lvlm-1, ml);
		  if (node1>-1) 
		    {
		      printf("x%d \t%f, \t%f \n", d, 
		             mg.xl[node1], mg.xl[node2]);
		    }
		  else 
		    {
		      printf("Warning: node on last level not found"
		             " (double refinement)!\n");
		    } */
		}
	    }
	  for (i=0; i<vx_nr; i++) 
	    {  /* check if other nodes than the fathers were changed */
	      if (fathers[i]==1)
	        {
		  MLVLFINDENTRY(node1, i+0*vx_nr, lvlm-1, ml);
		  MLVLFINDENTRY(node2, i+1*vx_nr, lvlm-1, ml);
		  if ((node1>-1)&&(node2>-1))
		    {
		      if ( (abs(mg.xl[node1]-0.0)>1e-8)||(abs(mg.xl[node2]-0.0)>1e-8) )
			printf("ERROR: x%"dFIDX" \thas changed!\t%f, \t%f\n", 
			       i, mg.xl[node1], mg.xl[node2]);
		    }	
		}
	    }		

	
	  for (i=0; i<vx_nr; i++) 
	    {
	      MLVLFINDENTRY(node1, i+0*vx_nr, lvlm, ml);
	      MLVLFINDENTRY(node2, i+1*vx_nr, lvlm, ml);
	      if ((node1>-1)&&(node2>-1))
		{
		  if ( (abs(mg.xl[node1]-0.0)>1e-8)||(abs(mg.xl[node2]-0.0)>1e-8) )
		  printf("restr[%3"dFIDX"]= %10f  %10f\n", i,
			 mg.xl[node1], mg.xl[node2]);
		}
	    }

	for (i=0; i<vx_nr*dim; i++) nodem.V[i] = 0.0;
    }      
    
  if(0==1) /* PRINT ELEM HIERACHY */
  {
    for (i=ml.levlhindx[lvlm]; i<ml.levlhindx[lvlm+1]; i++)
      {   
	printf("\nelhier %"dFIDX":\nfathers  ",i);
	for (j=0; j<3; j++) printf("%"dFIDX" (%f %f), ", 
msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j], 
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j]*msh2.vx_w+MCT2VXSTRT+0],
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j]*msh2.vx_w+MCT2VXSTRT+1]);
	printf("\nfathers  ");
	for (j=3; j<6; j++) printf("%"dFIDX" (%f %f), ", 
msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j],
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j]*msh2.vx_w+MCT2VXSTRT+0],
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHFAT1+j]*msh2.vx_w+MCT2VXSTRT+1]);	
	printf("\nchildren ");
	for (j=0; j<3; j++) printf("%"dFIDX" (%f %f), ", 
msh2.elhier[i*msh2.eh_w+MCT2EHCHL1+j],
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHCHL1+j]*msh2.vx_w+MCT2VXSTRT+0],
msh2.vertex[msh2.elhier[i*msh2.eh_w+MCT2EHCHL1+j]*msh2.vx_w+MCT2VXSTRT+1]);
	printf("\n");
      }  
    printf("\n");
  } 
    
  
  for (i=0; i<0*vx_nr; i++)
    {
      printf("x%"dFIDX": (%f,%f)\n", i,
	msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+0],
	msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+1]);
    } 
    
  /* SOLVE the equation system */
  if (set.solver==0) /* variant 1: direct solve */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* project rhs */
      for (i=0; i<P.len; i++)
	{
	  FIDX dof = P.V[i];
	  rhs.V[dof]=0.0;
	}
      err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      coarse_mat_free(&K_inv);
    }
    
  if ((set.solver==1)||(set.solver==2))
    { /* initialize multigrid data for PCG_MG or PCG_BPX */
      /* generate the coarse grid matrix */
      struct coarse_mat *cmat;
      FIDX *coarse_bc_nodes;
      FIDX n_c_bc=0;

      TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
      TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
     
      for (i=0; i<P.len; i++)
        {
	  FIDX dof, child;
	  dof=P.V[i];
	  MLVLFINDENTRY(child, dof, 0, ml);
	  if (child>=0)
	    {
	      coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
	      n_c_bc++;
	    }
	}

      err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			1, cmat );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
      mg.cmat=cmat;
    
      free(coarse_bc_nodes);
	
      mg.vcycles=vcycles;
      mg.smooths=smooths;
      if (stiffening==0) mg.CGC_scale=0.25;
      else mg.CGC_scale=1.0;
    }
  if (set.solver==1) /* variant 2: PCG with multigrid */
    {
      int iter;
      double resi, rtol;	 
      if (lvlm==0)
	{
	  rtol=set.solver_ini_rtol;
	}
      else
	{
	  rtol=set.solver_ref_rtol;
	}
      err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvlm], &rhs, &mg); 	
      /* FUNCTION_FAILURE_HANDLE( err, PCG, main); */
      printf("PCG_MG_main |res|=%8.1e it=%3d\n", resi, iter);
    }
  if (set.solver==2) /* variant 3: PCG with BPX */
    {
      int iter;
      double resi, rtol;
      
      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh2;
      bpx.P   = &P;    
      bpx.cmat = mg.cmat; /* */
      if (lvlm==0)
	{
	  rtol=set.solver_ini_rtol;
	}
      else
	{
	  rtol=set.solver_ref_rtol;
	}
      err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_bpx_tx,
	       &Ks[lvlm], &rhs, &bpx); /* */
      /* sparse_mat_write_file(&Ks[lvlm], "visual/Lame_mat.txt"); /* */
      /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      printf("PCG_BPX_main |res|=%8.1e it=%3d\n", resi, iter);
      bpx_free(&bpx);
    }
  if (set.solver==3) /* variant 3: sparse direct solve with UMFPACK*/
    {}
      /* apply Dirichlet condition 
      err=linsolver_apply_diri(&Ks[lvlm], P.len, P.V);
      FUNCTION_FAILURE_HANDLE( err, sparse_apply_diri, main );
      /* project rhs 
      for (i=0; i<P.len; i++) rhs.V[P.V[i]]=0.0;
	      
      struct linsolver sol;
      linsolver_init(&sol);
	{
	  sol.type      = LINSOLVER_TYPE_UMFPACK;
	  sol.symbolic  = NULL;
	  sol.numeric   = NULL;
        }
      UMFPACK_factorize (&Ks[lvlm], &sol);
      UMFPACK_solve(&sol, &rhs, &nodem);

      linsolver_free(&sol);
    } /* */

  if(set.write_mesh==3)
    {
      struct mesh m1;
      
      mesh_t2_to_t1(&msh2,&m1);	  
      
      err=mesh_write_mesh_gnuplot_tx( &m1, 27,
				    "visual/2d_mesh_gnuplot_ini", 1);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
      mesh_free(&m1);
    }
    
  if (0==1) /* CALCULATE RESIDUAL VECTOR */
    {
      struct vector y;
      err=vector_alloc(&y,nodem.len);
      FUNCTION_FAILURE_HANDLE(err, vector_alloc, main);

      err=sparse_mul_mat_vec(&Ks[lvlm], &nodem, &y);
      FUNCTION_FAILURE_HANDLE(err, sparse_mul_mat_vec, main);

      for(i=0; i<y.len; i++)
	{
	  y.V[i]-=rhs.V[i];
	}
      for(i=0; i<P.len; i++)
	{
	  y.V[P.V[i]]=0.0;
	}

      /* add res to msh2.vertex for graphical output */
      for (i=0; i<vx_nr; i++)
	for (d=0; d<dim; d++)
	  {
	    msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += 5*y.V[d*vx_nr+i];
	  }

      if (1==1) /* WRITE DEFORMED T1 MESH */
	{
	  struct mesh m1;

	  mesh_t2_to_t1(&msh2,&m1);	  
	  
	  err=mesh_write_mesh_gnuplot_tx( &m1, 27,
					  "visual/2d_mesh_gnuplot_def", 1);
	  FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
				  main);
	}
      vector_free(&y);

      exit(1);
    }

    
 //   bpx_free(&bpx);
    
  if(set.write_mesh==2) /* WRITE REFINED T2 MESH */
    {
      err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_def", 2);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
    }  
    
    
  if(set.write_mesh==3) /* WRITE REFINED T1 MESH */
    {
      struct mesh m1;

      mesh_t2_to_t1(&msh2,&m1);	  
		  
      err=mesh_write_mesh_gnuplot_tx( &m1, 27,
				    "visual/2d_mesh_gnuplot_def", 1);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
      mesh_free(&m1);
    }   

  if(set.write_mesh!=0) /* write data for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &nodem, dim, NULL, 0,
                              16, "visual/Lame_sol" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   
    
  if(0==1) /* PRINT DEFORMATIONS */
  {
    double ux,uy;
    int node;
    for (node=0; node<8; node++)
      {
	ux=nodem.V[0*vx_nr+node];
	uy=nodem.V[1*vx_nr+node];
	
	printf("x%d: %f, %f  ",node,msh2.vertex[node*msh2.vx_w+MCT2VXSTRT+0],
	       msh2.vertex[node*msh2.vx_w+MCT2VXSTRT+1]);
      }

  }

  

  /* the computation of nodem is done, apply it to the mesh */
  for (i=0; i<vx_nr; i++)
     for (d=0; d<dim; d++)
      {
	msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += nodem.V[d*vx_nr+i];
      }

  if (set.write_mesh==1) /* WRITE DEFORMED T2 MESH */
    {
      err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_def", 2);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
    }
  
  if (0==1) /* WRITE DEFORMED T1 MESH */
    { 
      struct mesh m1;

      mesh_t2_to_t1(&msh2,&m1);	  
		  
      err=mesh_write_mesh_gnuplot_tx( &m1, 27,
				    "visual/2d_mesh_gnuplot_def", 1);
      FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);
      mesh_free(&m1);
    }
  
  if (set.write_ssegs==1) /* WRITE DEFORMED T2 MESH */
    {
      mesh_write_ssegs_svg_tx( &msh2, 8, "test.svg",2);
      FUNCTION_FAILURE_HANDLE( err, mesh_write_ssegs_svg_tx, main);
    }

  /* free some stuff */
  if ((set.solver==1)||(set.solver==2))
    { /* multigrid data and mulitlvl are no longer valid */ 
      mg_free(&mg);
      multilvl_free(&ml);
    }   

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks[lvl]);
  free(Ks);

  projector1_free(&P);
  vector_free(&rhs);
  vector_free(&nodem);
  vector_free(&vx_old);

  mesh_free(&msh2);


  return SUCCESS;


}
