/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2010, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "linsolve_umfpack.h"
#include "assembly.h"
#include "assem_conv_diff.h"
#include "gen_aux.h"
#include "mesh_deform.h"
#include "lame_adj.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

//#include <unistd.h>
#include <string.h>

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }

int main(int argc, char *argv[])
{

  struct sparse A, B;
  struct projector1 P;
  
  struct vector rhs, x, b;

  struct mesh msh1;
  struct mesh msh2;

  struct solver_settings set;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t0, ti, ta, tb;

  int  err;
  FIDX dim, vx_nr, i, j;

  FIDX level0, lvlm, lvl;

  double *vx_new;
  FIDX vx_max;

  double Tstrt, Tend;
  FIDX tstep, write_steps;

  struct linsolver umfpackdata;
  FIDX *umfpack_isdiri;
  struct vector umfpack_help;

  double maxPe, tau;
  //double eps[]={1e-1, 1e-6, 1e-6, 1e-7};
  //double eps[]={1e-4, 0, 0, 1e-4};
  double eps[]={30, 0, 0, 30};

  double *vx_tmp;
  
  int iter=0;
  double resi=0.0;

  char *buffer;
  
  
  if (argc>1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);


  /* set the goals and methods */
  level0  =  set.refine_ini;
  lvlm    =  set.refine_steps;
  tau         = set.instat_delta_t0;
  write_steps = set.instat_write_steps;
  Tstrt       = set.instat_Time0;
  Tend        = set.instat_Tend;

  dim=msh1.dim;

  // eps=msh1.para[0];

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
      
      tau         /= 4;
      write_steps *= 4;
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }


  /* initialise the deformation */
  err=mesh_deform_t1_t2(&msh1, lvlm, &vx_new, &vx_max, 0, NULL);
  FUNCTION_FAILURE_HANDLE( err, mesh_deform_t2_t1, main);

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

  /* refine */
  for (lvl=0; lvl<lvlm; lvl++)
    {
      err=mesh_refine_uniform_t2( &msh2 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);

      tau         /= 4;
      write_steps *= 4;
    }


  vx_nr=msh2.vx_nr; 

  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);



  /* allocate memory for matrix K and the vectors */
  err=sparse_flex_alloc( &A, vx_nr);
  err=sparse_flex_alloc( &B, vx_nr);
  err=projector1_alloc( &P, vx_nr );
  err=vector_alloc( &x, vx_nr );
  err=vector_alloc( &rhs, vx_nr );
  err=vector_alloc( &b, vx_nr );
  
  /* set vectors to zero */
  for (i=0; i<vx_nr; i++)
    {
      x.V[i]    =0.0;
      rhs.V[i]  =0.0;
    }

  vx_tmp=msh2.vertex;
  msh2.vertex=vx_new;

  /* assemble the system matrices for the time stepping  */
  {
    err=assem_conv_diff_tx( NULL, NULL,  &rhs, NULL, &maxPe, eps,
			    &A, &B, 1.0, 1.0/tau, 0.0, 1.0/tau,
			    &msh2, 1, Tstrt, 2);
    FUNCTION_FAILURE_HANDLE( err, assem_conv_diff_tx, main);
  }


  /* set the Dirichlet boundary conditions */
  err=assem_Dirichlet_BC_tx( &P, &msh2, 1, &x, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_Dirichlet_BC_tx, main);


  /* time loop */
  /* x=0 is given above */
  for (tstep=0; Tstrt+tstep*tau<Tend; tstep++)
    {
      double t=Tstrt+tstep*tau;


      if (tstep>0)
	{
	  int solver;

	  TIMEGET;
	  t0=ti;
	  
	  err=assem_conv_diff_tx( NULL, NULL,  &b, NULL, &maxPe, eps,
				  NULL, NULL, 1.0, 1.0/tau, 0.0, 1.0/tau,
				  &msh2, 1, t, 2); /* */
	  FUNCTION_FAILURE_HANDLE( err, assem_conv_diff_tx, main);

	  TIMEGET;
	  ta=ti-t0;
	  t0=ti;

	  /* b:= b+ B*x(k-1),   to use it as RHS for the linear solves */
	  err=sparse_mul_mat_vec_add( &B, &x, &b);
	  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec_add, main);


	  TIMEGET;
	  tb=ti-t0;

	  /* now solve A*x=b */

	  /* solve the equation system */
	  solver=3;
	  switch (solver)
	    {
	    case 1: /* variant 1: direct solve with cmat */
	      {
		struct coarse_mat A_inv;
		struct vector helpv;

		fprintf(stderr, "does the CMAT stuff modify the matrices/rhs?\n"
			"does it work with nonzero initial/dirichlet values?\n");
		return FAIL;

		TIMEGET;
		t0=ti;

		err=vector_alloc(&helpv, x.len);
		FUNCTION_FAILURE_HANDLE( err, vector_alloc, main );

		/* invert the stiffness matrix */
		err=coarse_mat_set( &A, P.len, P.V,
				    2, &A_inv );
		FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	
		/* modify the rhs */
		/* helpv=A*x */
		err=sparse_mul_mat_vec( &A, &x, &helpv);
		FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, main);  
		/* rhs=rhs-helpv */
		for (i=0; i<rhs.len; i++)
		  {
		    rhs.V[i]-=helpv.V[i];
		  }

		/* project rhs */
		for (i=0; i<P.len; i++)
		  {
		    FIDX dof = P.V[i];
		    rhs.V[dof]=0.0;
		  }

		/* solve the projected system */
		err=coarse_mat_solve( &A_inv, NoTrans, &rhs, &helpv);
		FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

		/* apply the solution */
		for (i=0; i<vx_nr; i++)
		  {
		    x.V[i]+=helpv.V[i];
		  }
		TIMEGET;
		printf("solver cmat,    size=%6"dFIDX", t_solver=%10.3e\n", rhs.len, ti-t0);


		coarse_mat_free(&A_inv);
		vector_free(&helpv);
	      }
	      break;
	    case 2:/* variant 2: PCG */
	      {
		TIMEGET;
		t0=ti;
		
		err=FAIL; /* PCG( 10000, 2, set.solver_atol,
			     set.solver_ini_rtol, 1, &nodem, &resi, &iter, 
			     sparse_mul_mat_vec, gen_proj_MG_tx,
			     &Ks[lvlm], &rhs, &mg); /* */
		FUNCTION_FAILURE_HANDLE( err, PCG, main);


		printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
		TIMEGET;
		printf("solver PCG MG,  size=%6"dFIDX", t_solver=%10.3e\n", rhs.len, ti-t0);
	      }
	      break;
	    case 3:/* variant 3: direct solver UMFPACK */
	      /* variant 3: direct solver UMFPACK */
	      {
		int task;
		
		TIMEGET;
		t0=ti;

		if (tstep<=1)
		  {
		    task=1;
		  }
		else
		  {
		    task=2;
		  }

		err=sparse_solve_UMFPACK_reuse( &x, &A, &b, &P, &umfpackdata,
						&umfpack_isdiri, &umfpack_help, task );
		FUNCTION_FAILURE_HANDLE( err, sparse_solve_UMFPACK_reuse, main); /* */
	
		TIMEGET;
		printf("tstep=%5"dFIDX"  t=%8.6f   solver UMFPACK, size=%6"dFIDX", t_assem=%10.3e, t_mul=%10.3e, t_solver=%10.3e\n", tstep, t, rhs.len, ta, tb, ti-t0);

		if (task==1)
		  {
		    /* recompute the system matrices, as they were
		       changed by the UMFPACK factorization phase */
		    err=assem_conv_diff_tx( NULL, NULL,  &b, NULL, &maxPe, eps,
					    &A, &B, 1.0, 1.0/tau, 0.0, 1.0/tau,
					    &msh2, 1, t, 2); /* */
		    FUNCTION_FAILURE_HANDLE( err, assem_conv_diff_tx, main);
		  }
	      }
	      break;
	    default:
	      fprintf(stderr,"main: unknown solver requested\n");
	      return FAIL;
	    }

	} /* end not first tstep */


      if ((tstep%write_steps)==0)
	{
	  char fname[100];
	  sprintf(fname, "visual/conv_diff_vtk_%05"dFIDX,
		  (tstep/write_steps));
	  /* write the solution for paraview visualisation */
	  err=mesh_write_solution_vtk_t2( &msh2, NULL, 0, &x, 1,
					  strlen(fname)+1,fname );
	  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t1,
				   main); /* */
	}


    } /* end time loop */

  /* clean up */
  sparse_free(&A);
  sparse_free(&B);

  /* free(vx_new) is not necessary, cause of mesh_free(&msh2) where the target of vx_new is included
     so only free(vx_tmp) is necessary */
  free(vx_tmp);

  projector1_free(&P);

  vector_free(&b);
  vector_free(&rhs);
  vector_free(&x);
  mesh_free(&msh2);
  mesh_free(&msh1);

  free(buffer);

  return SUCCESS;


}
