/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "assembly.h"
#include "sparse.h"
#include "lin_solver.h"
#include "gen_aux.h"

//#include <unistd.h>
#include <string.h>

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }


int main(int argc, char *argv[])
{
  char solver[50]="no_solver_set";
  struct sparse *Ks;
  struct vector rhs, x;
  struct mesh msh1, msh2;
  struct projector1 P;

  struct multilvl ml;
  struct mgdata mg;

  struct solver_settings set;

  int  err, iter, lmax, level0;
  FIDX i, level;

  double resi, atol, rtol, pointerror;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t0, ti, t0l;


  if (argc>1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  TIMEGET;
  t0=ti;


  /* set refinement levels */
  level0  = set.refine_ini;
  lmax    = set.refine_steps;

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  msh1.lvl=-1;
  for (i=0; i<msh1.eg_nr; i++)
    {
      msh1.edge[i*msh1.eg_w+MCT1EGLVL]=-1;
    }

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);


  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks,  lmax+1, struct sparse, main);

  for (level=0; level<=lmax; level ++)
    { 
      
      
      if (level != 0)
	{
	  /* uniform refinement by bisection */
	  FIDX *marker;
	  TRY_MALLOC( marker, msh2.el_nr, FIDX, main);

	  for (i=0; i<msh2.el_nr; i++)
	    marker[i]=1;

	  err=mesh_refine_adaptive_t2( &msh2, marker );
	  FUNCTION_FAILURE_HANDLE( err, mesh_refine_adaptive_t2, main);

	  free(marker);
	}

      sparse_alloc( &Ks[level], msh2.vx_nr, 9);
      vector_alloc( &rhs, msh2.vx_nr );
      vector_alloc( &x, msh2.vx_nr );
      projector1_alloc( &P, msh2.vx_nr );

      TIMEGET;
      t0l=ti;

      err=assem_poison_t2( &Ks[level], &rhs, &x, &P, &msh2 );
      FUNCTION_FAILURE_HANDLE( err, assem_poison_t2, main);


      /* define the multigrid data */
      err=multilvl_init_tx( &msh2, 1, &ml, 2);
      FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);

      err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
      FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);

      /* define solver parameter for the V-cycle */
      mg.vcycles=1;
      mg.smooths=1;
      mg.CGC_scale=1.0;

      atol = set.solver_atol;
      rtol = set.solver_ini_rtol;
      /* no interpolation of old solution yet, always use ini_rtol
	 if (level == 0)
	 {
	 rtol = set.solver_ini_rtol;
	 }
	 else
	 {
	 rtol = set.solver_ref_rtol;
	 } /* */

      /* linear solver */
      /* strcpy(solver,"PCG_no");
         err=PCG( 10000, 2, atol, rtol, 1, &x, &resi, &iter, sparse_mul_mat_vec, 
	 projector1_no_precon, &Ks[level], &rhs, &P );
	 FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */

      strcpy(solver,"PCG_MG");
      err=PCG( 10000, 2, atol, rtol, 1, &x, &resi, &iter, sparse_mul_mat_vec, 
	       gen_proj_MG_tx, &Ks[level], &rhs, &mg );
      FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      
      /* strcpy(solver,"PCR_no");
	 err=PCR( 10000, 2, atol, rtol, 1, &x, &resi, &iter, sparse_mul_mat_vec, 
	 projector1_no_precon, &Ks[level], &rhs, &P );
	 FUNCTION_FAILURE_HANDLE( err, PCR, main); /* */

      /*  nodeerrors=0.0;
	  for (i=0; i<x.len; i++)
	  {
	  double lx;
	  lx=msh2.vertex[i*msh2.vx_w+MCT1VXSTRT  ];
	  nodeerrors+=fabs(1.5*lx-0.5*lx*lx - x.V[i]);
	  }
	  nodeerrors*=1.0/x.len; /* */
       
      if (x.len>0)
	{
	  pointerror= x.V[0]; /* */
	  /*pointerror= fabs(x.V[16]-1.0); /* */
	}
      else
	{
	  pointerror=0.0;
	}


      TIMEGET;

      printf("%s: %4d iterations, |res|=%8.2e, vx_nr= %9"dFIDX",  "
	     "u_0=%16.9e, time_level=%10.3e, time_total=%10.3e\n",
	     solver, iter, resi, msh2.vx_nr, pointerror, ti-t0l, ti-t0); /* */

      /* printf("solution:\n");
	 for (i=0; i<x.len; i++)
	 printf("%7f\n", x.V[i]); /* */
      

      {
	/* output for CMESS */
	char basename[100], filename[105];
	struct sparse M;
	sprintf(basename,"visual/level_%02"dFIDX"__%08"dFIDX, level,
		msh2.vx_nr);
	

        sprintf(filename,"%s_stiffmat.txt",basename);
	err=sparse_mat_write_file(&Ks[level], filename);
	FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file, main);/* */

	sparse_flex_alloc( &M, msh2.vx_nr);
	err=assem_mass_tx( &M, &msh2, 2);
	FUNCTION_FAILURE_HANDLE( err, assem_mass_t2, main);
        sprintf(filename,"%s_massMat.txt",basename);
	err=sparse_mat_write_file(&M, filename);
	FUNCTION_FAILURE_HANDLE( err, sparse_mat_write_file, main);/* */
	sparse_free(&M);

        sprintf(filename,"%s_solution.txt",basename);
	err=vector_write_file(&x, filename);
	FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);/* */

	{
	  struct vector u0;
	  err=vector_alloc( &u0, x.len);
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

	  for (i=0; i<x.len; i++)
	    {
	      u0.V[i]=0.0;
	    }

	  for (i=0; i<P.len; i++)
	    {
	      u0.V[P.V[i]]=x.V[P.V[i]];
	    }

	  sprintf(filename,"%s_DiriData.txt",basename);
	  err=vector_write_file(&u0, filename);
	  FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);/* */
	
	  vector_free(&u0);
	}

	{
	  FILE *out;
	  sprintf(filename,"%s_DiriNodes.txt",basename);
	  
	  out=fopen(filename,"w");

	  if (out==NULL)
	    {
	      fprintf(stderr,
		      "main: error opening file %s\n",
	      filename);
	      return FAIL;
	    }
	  
	  for (i=0; i<P.len; i++)
	    {
	      fprintf(out,"%8"dFIDX"\n",P.V[i]);
	    }
	  fclose(out);

	}

        sprintf(filename,"%s_visualisation",basename);
	err= mesh_write_solution_vtk_t2(&msh2, NULL, 0, &x, 1,
					strlen(filename)+1,filename );
	FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2, main);/* */

	
      }

      

      mg_free(&mg);
      multilvl_free(&ml);


      vector_free(&rhs);
      projector1_free(&P);

      vector_free(&x);

    } /* */

  for (level=lmax; level>=0; level--)
    sparse_free(&Ks[level]);
  free(Ks);

  mesh_free(&msh2);
  mesh_free(&msh1);

  return 1;
}
