#ifndef SPARSE_STRUCT_H
#define SPARSE_STRUCT_H


/* sparse matrix data structure */
/* 23.03.2007 rework to compressed row structure,
   matrix will have two stages of buildup:
   1. ultra flexible datastructure that allows arbitrary addition and
      deletion of entries, based on simply connected lists
   2. compressed row structure, once the matrix is assembled it will
      be compressed to reduce indirect adressing issues */
#define SP_TYPE_UNDEF -1 
#define SP_TYPE_FLEX 1 
#define SP_TYPE_COMPROW 2 
struct sparse {
  int  type;          /* describes the actual storage type of the
			 current matrix:
			 type==SP_TYPE_FLEX: flexible type 
                                (for assembly, as long as connectivity
                                 of matrix is not yet known),
			 type==SP_TYPE_COMPROW: compressed row
			 storage, faster for calculations */
  FIDX row_nr;        /* actual number of rows */

  /* data structure for compressed row sotrage, only used when type==2 */
  FIDX *rows;         /* array containing the index of the begin of
			 each row in both cols and A, length= (row_nr+1),
			 meaning:
			 k:=rows[i], m:=rows[i+1], i<row_nr
			 => i-th row is stored in cols[k...m-1] and
			    A[k..m-1] */
  FIDX *cols;         /* array containing the columns to which entries
			 in the matrix correspond,
			 length=rows[row_nr], meaning:
			 j:=cols[rows[i]+k], k<rows[i+1]-rows[i],
			                     i<row_nr,
			 => A[rows[i]+k] contains A(i,j) */
  double *A;          /* array containing the actual matrix
			 entries, length= (rows[row_nr]), 
			 interpretation as described above in
			 comment on cols (and rows) */


  /* flexible storage stage, only used when type==1 */
  struct int_double_list *flex_rows;
                      /* array containing the roots of the simply
			 connected lists for each row,
		         each row will be sorted as it is created,
		         with the root element being the dioagonal
		         entry */
};


/* simply connected list of storing one FIDX and one double */
struct int_double_list
{
  FIDX   ix;          /* FIDX datum */
  double dx;          /* double data */
  struct int_double_list *succ;
                      /* successor element in the list,
		         if ==NULL no successor yet */
};


/* vector data structure */
struct vector {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  double *V;          /* pointer to array of length n_max, such that 
			 V[i] gives the i-th component of the vector
		      */
};

/* simple projector data structure */
struct projector1 {
  FIDX len;           /* actual length */
  FIDX n_max;         /* maximal length, length for which memory has
			 been allocated in V */
  FIDX *V;            /* pointer to array of length n_max, such that 
			 V[i] tells which components of a vector the
			 projector modifies (sets to zero),
			 V[i]=j ==> j-th component is set to zero
		      */
};


/* general transpose / notranspose stuff */
enum transposetype {
  NoTrans=0,  /* no transpose */
  Trans=1     /* transpose    */
};



/* data structure for coarse grid matrix factorizations */
struct coarse_mat
{
  int nr;             /* number of degrees of freedom (DOFs) covered
			 by this matrix */
  int band_w;         /* bandwidht of the matrix */
  int symm;           /* marking the symmetry properties of the
			 matrix */
  FIDX *nodes;        /* pointer to an array of node numbers to which
			 the DOFs correspond, represents a
			 permutation, which is used to reduce fill in
			 in the trinagulation */
  double *A;          /* the triangular factors of the matrix
			 themself, 
			 if symm==1 stored as band matrix of dimension
			       nr, with bandwidth band_w, stored in
			       the format as produced by the LAPACK
			       routine dpbtrf with upload='U',
			       LDAB=KD+1 and KD=band_w, note that the
			       i-th DOF of the system corresponds to
			       the node nodes[i] 
			 if symm==2 stored as by  LAPACK
			       routine dgbtrf with KU=KL=band_w,
			       LDAB=2*KL+KU+1, note that the i-th DOF
			       of the system corresponds to the node
			       nodes[i] */ 
  double *help;       /* a help vector of size dofs */
  int *ipiv;          /* integer vector of length dofs, to store the
			 pivot indices, only used if symm=2, otherwise
			 the pointer will be ==NULL */
};


/* data structure for adjency-, level- and similar lists of integers */
struct int_list
{
  FIDX ixn;           /* actual length of the index vector ix */
  FIDX ixnmax;        /* maximal length of the index vector ix */
  FIDX *ix;           /* index vector, j=ix[i] marks the begin of the
			 i-th part of el, k=ix[i+1]-1 marks the end of
			 the i-th part of el, i<ixn<=ixnmax */
  FIDX elnmax;        /* maximal length of el, therefore 
			 ix[ixn+1]-1<=elnmax has to be asured */
  FIDX *el;           /* list of elements, with j and k as described
			 in the ix part, el[j+t] is the t-th element
			 of the i-th part of the list, t<=k-j-1 */
};



#endif
