/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
sparse.h
*/
#ifndef __SPARSE_H__
#define __SPARSE_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"


int sparse_flex_alloc(struct sparse *K, FIDX nrows
/* creates a sparse matrix in the initial flexible format with nrows rows 
   
   Input:  nrows   - number of rows of the matrix,

   Output: K       - (K is given by reference), memory for the
                     neccessary fields in K is allocated,
		     dimensions of K initialised, type set to flexible

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_alloc(struct sparse *K, FIDX nrows, FIDX notused_density
/* OBSOLETE, kept for ease of switching to new matrix format;
   creates a sparse matrix in the initial flexible format with nrows rows 
   
   Input:  nrows   - number of rows of the matrix,
           density - not used!, only present for backward compatibility

   Output: K       - (K is given by reference), memory for the
                     neccessary fields in K is allocated,
		     dimensions of K initialised, type set to flexible

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void sparse_init(struct sparse *K
/* initialises K to be empty, such that it can be emptied without
   problems 
   
   Input:  nrows   - number of rows of the matrix,
           density - maximal number of nonzeros per row,

   Output: K       - (K is given by reference), memory for the
                     neccessary fields in K is allocated,
		     dimensions of K initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void sparse_free(struct sparse *K
/* memory allocated in K is freed, so that K can be freed

   Output: K       - (K is given by reference) inside memory is
                     released 
*/);

void sparse_empty(struct sparse *K
/* the sparse matrix K is set to zero, 
   if the matrix is in compressed row storage, the sparsity structure
   is preserved
   if the matrix is in flexible format, it will be reset to a diagonal
   matrix of zeros

   Output: K       - (K is given by reference) is set to zero matrix
*/);

int sparse_row_empty(struct sparse *K, FIDX row 
/* the row-th row of the sparse matrix K is set to zero

   Input:  row     - number of the row to be emptied

   Output: K       - (K is given by reference) is set to zero matrix

   Return: SUCCESS - success
           FAIL    - failure, the row doesn't belong to the matrix
*/);

int sparse_mat_write_file(struct sparse *A, char *name
/* writes the contents of the sparse matrix to a file with the
   specified name
   
   Input:  A       - sparse matrix, original
           name    - name for the file

   Output: (the file)

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_mat_read_file(struct sparse *A, char *name
/* writes the contents of the sparse matrix to a file with the
   specified name
   
   Input:  name    - name for the file

   Output: A       - sparse matrix in flexible storage format,
                     is initialised and storage allocated here
           
   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_mat_transp(struct sparse *A, struct sparse *B
/* copies a the transpose of the sparse matrix A into B,
   B = A^T;

   B is always created in flexible format

   Input:  A       - sparse matrix, original, may be flexible or
                     compressed row format
		     must be N-by-N matrix

   Output: B       - sparse matrix, transpose, internal sizes are set
                     and memory allocated,

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_mul_mat_vec(void *arg1, struct vector *vec,
		       struct vector *out
/* multiplies the sparse matrix  K from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - sparse matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_GS_sweep_fwd(struct sparse *K, struct vector *b,
			struct vector *invdiag, FIDX sweeps,
			struct vector *x
/* applies sweeps Gauss-Seidel (GS) sweeps to improve the x as
   solution of 

    K * x = b;

   Input:  K       - sparse matrix
           b       - right hand side vector
           invdiag - inverse of the diagonal elements (1/a_ii)
           sweeps  - number of GS sweeps to be performed during this call

   In/Out: x       - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_GS_sweep_bwd(struct sparse *K, struct vector *b,
			struct vector *invdiag, FIDX sweeps,
			struct vector *x
/* applies sweeps Gauss-Seidel (GS) sweeps to improve the x as
   solution of 

    K * x = b;

   Input:  K       - sparse matrix
           b       - right hand side vector
           invdiag - inverse of the diagonal elements (1/a_ii)
           sweeps  - number of GS sweeps to be performed during this call

   In/Out: x       - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_GS_sweep_sorted(struct sparse *K, struct vector *b,
			   struct vector *invdiag, 
			   FIDX *sorter, int dir, FIDX sweeps,
			   double scale, struct vector *x
/* applies sweeps Gauss-Seidel (GS) sweeps to improve the x as
   solution of 

    K * x = b;

   Input:  K       - sparse matrix
           b       - right hand side vector
           invdiag - inverse of the diagonal elements (1/a_ii)
	   sorter  - permutation of the dofs that sorts them in a
	             particular maner, e.g. stream wise
           dir     - direction of the sweeps dir=+1 =>fwd, dir=-1 =>bwd
           sweeps  - number of GS sweeps to be performed during this
                     call
	   scale   - multiply the update by this factor, used for
	             damping, 0<scale<1 is advisable
	   

   In/Out: x       - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_vec_mat_vec(struct sparse *K, struct vector *u,
		       struct vector *v, double *out
/* computes the product
     out = v^T *K* u
   for the sparse matrix K and the vectors u and v

   Input:  K       - sparse matrix
           u       - vector
           v       - vector

   Output: out     - scalar, given by reference

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_mul_mat_vec_add(struct sparse *K, struct vector *vec,
			   struct vector *out
/* multiplies the sparse matrix  K from left to the vector vec and
   adds the result to out,
   
   out = out + K * vec;

   Input:  K       - sparse matrix
           vec     - vector

   In/Out: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_row_tim_vec(struct sparse *K, FIDX rown, FIDX *rows, 
		       struct vector *vec, struct vector *out
/* multiplies the specified rows of the sparse matrix  K from left to
   the vector vec, 
   
   out[i] = K_rows[i] * vec;     

   Input:  K       - sparse matrix
           rown    - number of given rows
	   rows    - integer vector of length rown, specifying the
                     rows for which the product shall be computed
           vec     - vector

   Output: out     - vector containing the results,
                     out[i] = K_rows[i] * vec

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_row_tim_vec_add(struct sparse *K, FIDX rown, FIDX *rows, 
		       struct vector *vec, struct vector *out
/* multiplies the specified rows of the sparse matrix  K from left to
   the vector vec and adds the result to out,
   
   out[i] = out[i] + K_row[i] * vec;

   Input:  K       - sparse matrix
           rown    - number of given rows
	   rows    - integer vector of length rown, specifying the
                     rows for which the product shall be computed
           vec     - vector

   In/Out: out     - vector containing the results,
                     out[i] = out[i] + K_row[i] * vec

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_mul_mat_vec_add_trans(struct sparse *K, struct vector *vec,
				 struct vector *out
/* multiplies the transpose of the sparse matrix  K from left to the
   vector vec and adds the result to out,
   
   out = out + K^T * vec;

   Input:  K       - sparse matrix
           vec     - vector

   In/Out: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_add_local(struct sparse *K, enum transposetype trans,
		     FIDX m, FIDX row[], FIDX n,
		     FIDX col[], double local[], FIDX LDlocal
/* adds the matrix local to the by row and col defined submatrix of K,
   if trans==NoTrans:
   K(row[i],col[j])+=local[j*LDlocal+i];

   if trans==Trans:
   K(row[i],col[j])+=local[i*LDlocal+j];

   Input:  trans   - ==NoTrans  => normal add
                     ==Trans    => Transposed add
           m       - number of rows in local, length of row
           row     - array of row indices into K, length m
           n       - number of collumns in local, length of col
           col     - array of collumn indices into K, length n
	   local   - array to be added to K, size m*n
	   LDlocal - leading dimension of local

   In/Out: K       - sparse matrix to which values are added

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_get_entry(struct sparse *K, FIDX row, FIDX col,
		     int create, double **entry
/* if K contains an entry K(row,col) a pointer to this entry is
   returned, such that 
             (*entry)=K(row,col)
   if it doesn't contain the entry yet and create==1, it is created
   and a pointer to the entry is returned as before.

   Input:   row    - row index into the sparse matrix K
            col    - column index into the sparse matrix K
	    create - if ==1 entry will be created (if possible)
	             otherwise, no entry will be created, only
	             existing ones are returned

   In/Out:  K      - sparse matrix, eventually the entry K(row,col) is
                     created

   Output:  entry  - (by reference) pointer to a double such that 
                     (*entry)=K(row,col)

   Return: SUCCESS - success,
           FAIL    - failure, entry not found and not allowed to
                     create,
		     if allowed, creation failed, see error message
*/);

int sparse_convert_compressed_row(struct sparse *K
/* converts a matrix stored in flexible mode to compressed row mode

   In/Out:  K      - sparse matrix, storage type is converted

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int sparse_extract_invdiag(struct sparse *K, struct vector *Kinvdiag
/* the inverse of the diagonal of K is stored in the vector Kinvdiag

   Input:   K      - sparse matrix

   Output:  Kinvdiag
                   - (by reference) if size does not fit K, the vector
                     is reinitialised, on exit i-th component of
                     Kinvdiag is set to 1/(K_ii),
		     

   Return: SUCCESS - success,
           FAIL    - failure
*/);

void vector_init(struct vector *V 
/* V initialised to be empty

   Output: V       - (V is given by reference) defined to be empty
*/);

int vector_alloc(struct vector *V, FIDX len
/* allocates memory in V 
   
   Input:  len     - length of the vector

   Output: V       - (V is given by reference), memory in V is
                     allocated, dimensions of V initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void vector_free(struct vector *V
/* memory allocated in V is freed, so that V can be freed

   Output: V       - (V is given by reference) inside memory is
                     released 
*/);

int vector_diag_scale(void *notused, struct vector *in,
		      void *arg3, struct vector *out
/* performs
     out = D*in
   where D is a diagonal matrix with entries as specified in the
   vector diag, 

   Input:  notused - well, it is not used but included in the
                     interface to allow this function to be used as a
                     preconditioner (i.e. if diag is the inverse of
                     the diagonal of a matrix, this is the Jacobi
                     preconditioner)
           in      - input vector
	   arg3=
	   diag    - vector storing the diagonal entries of D,

   Output: out    - (given by reference), the D*in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int vector_read_file(struct vector *vec, char *name
/* reads the contents of the file into the vector, memory is allocated
   for that
      
   Input:  name    - name of the file

   Output: vec     - vector, memory is allocated and the contents
                     assigned 

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int vector_write_file(struct vector *vec, char *name
/* writes the contents of the vector to a file with the specified name
      
   Input:  vec     - vector
           name    - name for the file

   Output: (the file)

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int vector_n_write_file(FIDX nCol, struct vector *vec, char *name
/* writes the contents of an array of column vectors to a file with 
   the specified name
      
   Input:  nCol    - number of vectors in the array
           vec     - array of vectors, all vec[j] have to be of same
                     length j=0..nCol-1, the file will be written as a
                     matrix with nCol columns and vec[0].len rows,
           name    - name for the file

   Output: (the file)

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int projector1_no_precon(void *notused, struct vector *in,
			 void *arg3, struct vector *out
/* performs
     out = P*I*P^T *in
   where P is a projector, defined to set a number of components of
   the vector to zero and not change the others,
   therefore
     P*I*P^T = P 
   in this special case, which is the way how the computation is
   performed 

   Input:  notused - well, it is not used but included in the
                     interface to allow this function to be used as a
                     "preconditioner" 
           in      - input vector
	   arg3=
	   P       - projector data

   Output: out    - (given by reference), the projection of in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int projector1_alloc(struct projector1 *V, FIDX len
/* allocates memory in V 
   
   Input:  len     - length of the projector1

   Output: V       - (V is given by reference), memory in V is
                     allocated, dimensions of V initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

void projector1_free(struct projector1 *V
/* memory allocated in V is freed, so that V can be freed

   Output: V       - (V is given by reference) inside memory is
                     released 
*/);

void int_double_list_free( struct int_double_list *root
/* free all enties of a list given by the root entry
   
   Output: root    - pointer to an entry, this entry and all
                     successors will be freed
*/);

void coarse_mat_free( struct coarse_mat *cmat
/* the internal parts of cmat are freed, so that cmat itself can safely
   be freed
   
   Output: cmat    - (given by reference) on output it will be empty
*/);

void coarse_mat_null_def( struct coarse_mat *cmat
/* the internal parts of cmat are defined, so that cmat itself can
   be given to coarse_mat_free even though it has not been set 
   
   Output: cmat    - (given by reference) on output it will be empty
*/);

int coarse_mat_set( struct sparse *K, FIDX nr_diri, FIDX *diris,
		    int symm, struct coarse_mat *cmat
/* computes the triangular factorisation of the stiffness matrix K
   and stores it in cmat, such that this is suitable for the coarse
   grid solver
   
   Input:  K       - the stiffness matrix for the coarse grid
           nr_diri - number of degrees of freedom (DOFs) which are
	             subject to Dirichlet boundary conditions, such
	             that the according rows and columns of the
	             matrix have to be deleted 
           diris   - vector of length nr_diri, stating which DOFs
                     these are
	   symm    - marking the symmetry properties of the matrix,
	             if the matrix is known to be symmetric positive
	             definit, use symm=1 (so more efficient methods
	             can be used), 
		     if the matrix is known to have symmetric
		     connectivity but the matrix is not necessarily
		     symmetric, use symm=2 

   Output: cmat    - (given by reference) on output it will contain
                     the data for the coarse grid solver, internal
                     fields are initialised and memory allocated

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int coarse_mat_solve( struct coarse_mat *cmat, enum transposetype trans,
		      struct vector *rhs, struct vector *x
/* solves the equation system 

     A*x = rhs

   where a triangulation of A has been stored in cmat by the routine
   coarse_mat_set
   
   Input:  cmat    - (given by reference) on output it will contain
                     the data for the coarse grid solver, internal
                     fields are initialised and memory allocated
           trans   - ==NoTrans  => normal solve
                     ==Trans    => solve with transposed of the matrix
           rhs     - right hand side of the system, can be the same as
                     rhs, in which case rhs is overwriten with the
                     solution on exit

   Output: x       - solution of the system, can be the same as rhs,
                     in which case rhs is overwriten with the solution
                     on exit

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int int_list_alloc( struct int_list *L, FIDX ixnmax, FIDX elnmax
/* allocates memory in the int_list struct for given size
   
   Input:  ixnmax  - maximal number of parts of the list,
           elnmax  - maximal number of elements in the list,

   Output: L       - the list struct, memory is allocated and
                     dimensions initialised

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);

void int_list_free( struct int_list *L
/* frees the memory in the int_list struct, so it can be freed
   
   In/Out: L       - the list struct, memory is freed
*/);

int bandw_red_perm( FIDX nr, FIDX *adix, FIDX *adel,
		    FIDX *perm, FIDX *iperm,
		    FIDX *bandw
/* calls the Gibbs-Poole-Stockmeyer modification of the
   Cuthill-McKee algorithm to produce a bandwith reducing permutaion
   for a given adjency list of a sparce matrix
   
   Input:  nr      - nr of verticex in the graph
           adix    - index of the adjency list
           adel    - element vector of the adjency list (needs to be
                     full adjency list, i.e. contain all edges (i,j)
                     twice: as (i,j) and (j,i) )

   Output: perm    - bandwith reducing permutation, given as integer 
                     vector, (size=nr)
	   iperm   - inverse permutation of perm, given as integer 
                     vector, (size=nr)
           bandw   - resulting bandwidth

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);

int bandw_gipost( FIDX n1, FIDX n2, struct int_list *ad,
		  FIDX *perm, FIDX *iperm, FIDX *bandw
/* applies the Gibbs-Poole-Stockmeyer modification of the
   Cuthill-McKee algorithm to produce a bandwith reducing permutaion
   for a given full adjency list of a connected subgraph of the
   connectivity graph of a sparce matrix
   
   Input:  n1      - starting position of the connected subgraph
           n2      - end position(+1) of the connected subgraph
           ad      - full adjency list of the whole graph, only the n1
                     to n2-1 parts are read

   Output: perm    - bandwith reducing permutation, the n1 to n2-1
                     entries are written, given as integer vector
           iperm   - reverse of the bandwith reducing permutation,
	             the n1 to n2-1 entries are written, given as
	             integer vector 
           bandw   - resulting bandwidth for this subgraph

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);

int comp_fidx(const void *a1, const void *b1
/* helper function for qsort, compares the integers a and b

   Input:  a1      - pointer to int a
           b1      - pointer to int b

   Return: -1   if a >  b
            0   if a == b
            1   if a <  b
*/);

int bandw_level_struct( FIDX root, FIDX n1, FIDX n2, struct int_list *ad, 
			struct int_list *L, FIDX *hint
/* defines the level structure rooted at root, for a given subgraph

   Input:  root    - root of the level structure
           n1      - index of the first node of the connected subgraph
           n2      - index of the last node +1 of the connected subgraph
           ad      - full adjency list of the whole graph

   In/Out: hint    - integer help vector, lenght n2-n1, has to be
                     provided by the calling routine

   Output: L       - level structure rooted at root, only empty
                     int_list struct has to be given, it is
                     initialised and memory in it allocated

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);

int bandw_mark_connected( FIDX label, FIDX start, FIDX end, FIDX *adix, 
			  FIDX *adel, FIDX n1, FIDX *marked
/* marks the vertex with label, and all starts a recursion for all
   adjacent vertices i which are marked with marked[i]==0 to mark them
   with the same label, thus all nodes connected to start are marked
   with label, if a vertex i with marked[i]>0 and marked[i]!=label is
   found this is considered an error, thus return will be FAIL,

   Input:  label   - how vertex start and all connected shall be labelled
           start   - the vertex where the labeling starts
           end     - the vertex where the labeling ends
           adix    - vector marking the begin and end of each nodes
                     adjacency list in adel
           adel    - together with adix defines the adjency of the
                     nodes
	   n1      - ofset of the lowest nodenumber in the considered
	             subgraph 
	   

   In/Out: marked  - integer vector, lengh = number of vertices, start
                     and all vertices i connected to start with
                     marked[i]==0, will be marked with makred[i]=label

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);

int bandw_number_adj_vert(struct int_list *ad, struct int_list *Lv,
			  FIDX *level, FIDX l1, FIDX l2, FIDX n1, FIDX n2,
			  FIDX *perm, FIDX *iperm, FIDX *current,
			  FIDX *sorter
/* starts with the vertex in level l1 of lowest assigned number,
   assigns all adjecent nodes in level l2 (which have not been
   assigned a number yet) a number in order of increasing degree,
   moves to the next until no further are left

   Input:  ad      - full adjency list
           Lv      - level structure
           level   - integer verctor, specifying the level for each
                     vertex 
           l1      - level in which to start
	   l2      - ofset of the lowest nodenumber in the considered
	             subgraph 
           n1      - numbering offset (begin of the connected subgraph)
           n2      - maximal index of the block (end of the connected
                     subgraph) 
	   

   In/Out: perm    - permutation, which is defined by the assigned
                     numbers integer vector, lengh = number of
                     vertices, every i with perm[i]!=-1 is considered
                     already given a number
           iperm   - inverse permutation to perm, every iperm[i]!=-1
                     is considered already given a number to iperm[i]
           current - current highes assigned number+1 (the next number
                     to assign), updated every time a vertex is
                     assigned a number

   Worksp: sorter  - an integer vector of lenght 2*level_width

   Return: SUCCESS - success,
           FAIL    - failure, see error message, output will be
                     invalid,
*/);
#endif
