/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
gen_aux.h
*/
#ifndef __GEN_AUX_H__
#define __GEN_AUX_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"


int gen_proj_bpx_tx(void *notused, struct vector *in,
		    void *arg3, struct vector *out
/* performs boundary condition projection and BPX preconditioning 
     out = P*C^{-1}*P^T in
   where C^{-1} is the BPX preconditioner sum(l=level)Q_l*D_l*Q_l^T 
   and P a project
   
   Input:  notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner
           in      - input vector
	   arg3=
           bpx     - struct bpxdata containing pointers to the mesh
                     a multilevel struct, a multilevel help vector,
		     and a projector1

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int gen_proj_MG_tx(void *arg1, struct vector *in,
		   void *arg3, struct vector *out
/* performs the boudary condition projection and multigrid
   preconditioning, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there), and C^-1 repressents the action of the multigrid
   preconditioning, that is one V cycle
   
   Input:  arg1    - not used, in the interface for compatibility reasons
           in      - input vector
	   arg3=
	   mg      - mgdata struct, containing everything needed for
	             the projection and multigrid

   Output: out    - (given by reference), P*C^-1P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int gen_zz_interpolate_gradient_tx(struct mesh *m,
				   int type,
				   FIDX ncomp,
				   struct vector *u,
				   struct vector *gradu
/* averages the values of the gradient (or the jacobian) at the nodes
   of the mesh, thus interpolation of these provides smoothed
   gradients.
   this idea is for example used in the ZZ type error estimators

   ideally this will be done according to 

   O.C. Zienkiewicz and J.Z. Zhu, The Superconvergent patch recovery
   and a posteriori error estimators. Part 1. The recovery technique,
   Int. J. Numer. Methods Eng., 33, 1331-1364 (1992)

   but for now it only a simplified version of this idea.

   
   Input:  m       - the mesh, has to be T1 or T2 
           type    - defining which type of mesh this actually is
	             type=1 ==> linear,  type=2 quadratic triangles
           ncomp   - number of (solution) components in u, i.e. number
                     of variables per node in the mesh
	   u       - coefficient vector representing a function on
	             the current mesh (e.g. solution of PDE problem)
		     u.V[r*m->vx_nr+i] has to be the value of the r-th
		     component of the (possibly) vector valued
		     function u at node i of the mesh

   Output: gradu   - the gradient (or Jacobian) of u averaged at the
                     nodes of the mesh, has to be of size
                     dim*ncomp*vx_nr,
		     gradu.V[d*ncomp*vx_nr + r*vx_nr +i]
		     will be the averaged value of the partial derivative
		     wrt. to the d-th coordinate of the r-th component
		     of function u at the i-th node of the mesh

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int gen_error_est_ZZ_tx(struct mesh *m, FIDX ncomp, struct vector *uh,
			FIDX *marker, FIDX *nrmarked, double *globest,
			struct solver_settings *set, int type
/* estimates the interpolation error for a finite element function in
   the H1-semi-norm using the technique of Zienkiewicz and Zhu's
   superconvergent patch recovery, 


   a simplified approach similar to  
   [
   O.C. Zienkiewicz and J.Z. Zhu, The Superconvergent patch recovery
   and a posteriori error estimators. Part 1. The recovery technique,
   Int. J. Numer. Methods Eng., 33, 1331-1364 (1992)
   
   O.C. Zienkiewicz and J.Z. Zhu, The Superconvergent patch recovery
   and a posteriori error estimators. Part 2. Error estimates and
   adaptivity, Int. J. Numer. Methods Eng., 33, 1365-1382 (1992)
   ]


   Input:  m           - the mesh
           ncmop       - the number of components in uh (per node of
                         the mesh)
	   uh	       - coefficient vector representing a function on
	                 the current mesh (e.g. solution of PDE problem)
			 u.V[r*m->vx_nr+i] has to be the value of the r-th
			 component of the (possibly) vector valued
			 function u at node i of the mesh
	   set	       - holding the settings for the process
           type        - defining which type of mesh this actually is
	                 type=1 ==> linear    triangles
			 type=2 ==> quadratic triangles

   Output: marker      - vector of length msh.el_nr with
                         marker[i]==1 -> element i isto be refined,
           nrmarked    - number of marked elements, i.e. number of
                         non-zero entries in marker (given by reference)
           globest     - value of the global estimate (given by reference)

   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
	                 valid
*/);

int gen_error_est_marker(struct mesh *m, struct vector *nk,
			 FIDX *marker, FIDX *nrmarked, double *globest,
			 struct solver_settings *set
/* marks the elements based on the element wise error estimates nk


   Input:  m           - the mesh
           nk          - the element wise error estimates
	   set	       - holding the settings for the process
           type        - defining which type of mesh this actually is
	                 type=1 ==> linear    triangles
			 type=2 ==> quadratic triangles

   Output: marker      - vector of length msh.el_nr with
                         marker[i]==1 -> element i isto be refined,
           nrmarked    - number of marked elements, i.e. number of
                         non-zero entries in marker (given by reference)
           globest     - value of the global estimate (given by reference)

   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
	                 valid
*/);

int gen_band_marker_tx(struct mesh *m, 
		       FIDX *marker, FIDX *nrmarked, 
		       double *c_band, double d_band, double width_band,
		       int type
/* marks the elements based that are in a band of the geometric domain
   as described below

   a point x is said to be in the band if
   abs(c^T x + d) <= width

   all elements which have at least one vertex in the band or which
   have points on both sides of the band will be marked

   Input:  m           - the mesh, must be either T1 or T2 elements
           nk          - the element wise error estimates
           c_band      - the vector c in the above definition of band,
                         must be of dimension m.dim
           d_band      - the scalar d in the above definition of band
           width_band  - the scalar width in the above definition of
                         band, width_band>0
	   type        - type of mesh, i.e. type==1 for T1, type==2
	                 for T2

   Output: marker      - vector of length msh.el_nr with
                         marker[i]==1 -> element i isto be refined,
           nrmarked    - number of marked elements, i.e. number of
                         non-zero entries in marker (given by reference)
           
   Return: SUCCESS     - success
           FAIL        - failure, see error message, output will not be
	                 valid
*/);

void heapSort(double *a, int *b, int len
/* Sorts a vector with the Heap Sort algorithm

   INPUT:        len   - lenght of array a and b
   
   INPUT/OUTPUT: a     - array whose elements are to be sorted 
   
   OUTPUT:       b     - array holding the new positions of the ordered elements

*/);

void siftDown(double *a, int *b, int start, int end
/* part of the Heap Sort algorithm

  INPUT/OUTPUT: a   - an unordered array a of length count
		b   - vector holding the new positions of the ordered elements

  INPUT:        end - represents the limit of how far down the heap
			to sift.
*/);

void swap(double *a, int *b, int pos1, int pos2
/* swaps the two entries pos1 and pos2 in the array a and b

  INPUT/OUTPUT : a     - an unordered array a of length count
                 b     - array holding the new positions 
		         of the ordered elements

  INPUT:         pos1,
                 pos2  -positions o be swapped
*/);
#endif
