% ***********************************************************************
%
%    This file is part of
%    FEINS, Finite Element Incompressible Navier-Stokes solver,
%    which is expanding to a more general FEM solver and toolbox,
%    Copyright (C) 2003--2008, Rene Schneider 
%    <rene.schneider@mathematik.tu-chemnitz.de>
% 
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
% 
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
% 
%    You should have received a copy of the GNU General Public License
%    along with this program. If not, see <http://www.gnu.org/licenses/>.
%
%    Minor contributions to this program (for example bug-fixes and
%    minor extensions) by third parties automatically transfer the
%    copyright to the general author of FEINS, to maintain the
%    possibility of commercial re-licensing. If you contribute but wish
%    to keep the copyright of your contribution, make that clear in
%    your contribution!
% 
%    Non-GPL licenses to this program are available upon request from
%    the author.
%
% ***********************************************************************

function divider_n_msh(a,b,c,d,e,xi)
% parameter a,b,c are identical to those in the sketch in the
% thesis, BUT d,e not. (no e in sketch). thesis_d=5*code_e, 
% code_d=length(inlet)


n=length(xi);


if n<2
  error('n<2 not supported in this version');
end

% double and integer vars relating to the blocks
global blocks_d; global blocks_i; global write_fem_fvm;


blocks_d=zeros(0,4);
blocks_i=zeros(0,4);

if write_fem_fvm==2
  fid = fopen('divider_n.msh','w');
else
  % just send all output in the bin
  fid = fopen('/dev/null','w');
end

fprintf(fid,...
    '# auto generated for a=%8.1e, b=%8.1e, c=%8.1e, n=%d, #blocks=\n',...
    a,b,c,n);
% number of blocks
bid=14;
for i=1:n
  for j=2:i
    bid=bid+3;
  end
  if i<n
    bid=bid+11;
  end
end
blk_nr = bid+1;
fprintf(fid,'%d\n', blk_nr);

crit_marker=-ones(blk_nr,1);

fprintf(fid,'# block lines:  block_id x0 y0 x1 y1 Top Bottom Right Left\n');

% inlet:
blockline(fid, 0, 0.5-c, 0.5- 10*d, 0.5, 0.5-  5*d, 1, -2, -1, -1);
blockline(fid, 1, 0.5-c, 0.5-  5*d, 0.5, 0.5-2.5*d, 2,  0, -1, -1);
blockline(fid, 2, 0.5-c, 0.5-2.5*d, 0.5, 0.5-    d, 3,  1, -1, -1);
blockline(fid, 3, 0.5-c, 0.5-    d, 0.5, 0.5,       4,  2, -1, -1);

fprintf(fid,'\n');

% first layer + outlet
% left -> right
y0=0.5; y1=0.5+xi(1);
blockline(fid, 4, 0.5-c,     y0,       0.5,   y1, 15,  3,  5, -1);
blockline(fid, 5, 0.5,       y0,     0.5+e,   y1, 16, -1,  6,  4);
blockline(fid, 6, 0.5+e,     y0, 0.5+2.5*e,   y1, 17, -1,  7,  5);
blockline(fid, 7, 0.5+2.5*e, y0, 0.5+4.0*e,   y1, 18, -1,  8,  6);
blockline(fid, 8, 0.5+4.0*e, y0, 0.5+5.0*e,   y1, 19, -1,  9,  7);
blockline(fid, 9, 0.5+5.0*e, y0, 0.5+5.0*e+a, y1, 20, 10, -1,  8);

fprintf(fid,'\n');

%top->bottom
x0=0.5+5*e; x1=0.5+5*e+a;
blockline(fid,10, x0, 0.5-1.0*d, x1, 0.5-0.0*d, 9, 11, -1, -1);
blockline(fid,11, x0, 0.5-2.5*d, x1, 0.5-1.0*d,10, 12, -1, -1);
blockline(fid,12, x0, 0.5-4.5*d, x1, 0.5-2.5*d,11, 13, -1, -1);
blockline(fid,13, x0, 0.5-8.0*d, x1, 0.5-4.5*d,12, 14, -1, -1);
blockline(fid,14, x0, 0.5- 10*d, x1, 0.5-8.0*d,13, -3, -1, -1);

fprintf(fid,'\n');

% interior layers + their outlets
bid=15; % first bid of the next block
for i=2:n-1
  % left -> right
  y0=0.5+xi(i-1); y1=0.5+xi(i);
  su=11+(i-1)*3;
  sl=11+(i-2)*3;
  %#         block_id x0 y0 x1 y1               Top      Bottom   Right Left);
  blockline(fid,bid+0,0.5-c,    y0,      0.5,y1,bid+su+0,bid-sl+0,bid+1,   -1);
  blockline(fid,bid+1,0.5,      y0,    0.5+e,y1,bid+su+1,bid-sl+1,bid+2,bid+0);
  blockline(fid,bid+2,0.5+e,    y0,0.5+2.5*e,y1,bid+su+2,bid-sl+2,bid+3,bid+1);
  blockline(fid,bid+3,0.5+2.5*e,y0,0.5+4.0*e,y1,bid+su+3,bid-sl+3,bid+4,bid+2);
  blockline(fid,bid+4,0.5+4.0*e,y0,0.5+5.0*e,y1,bid+su+4,bid-sl+4,bid+5,bid+3);
  
  fprintf(fid,'\n');
  
  bid=bid+5;
  % 3 intermediate cells with upper and lower neighbours
  xa=0.5+5.0*e;
  for j=3:i
    blockline(fid,bid+0,xa,    y0,xa+a,      y1,bid+su+0,bid-sl+0,bid+1,bid-1);
    blockline(fid,bid+1,xa+a,  y0,xa+a+0.5*b,y1,bid+su+1,bid-sl+1,bid+2,bid+0);
    blockline(fid,bid+2,xa+a+0.5*b,y0,xa+a+b,y1,bid+su+2,bid-sl+2,bid+3,bid+1);
    bid=bid+3;
    xa=xa+a+b;
  end
  % 3 intermediate cells with only upper neighbours
  blockline(fid,bid+0,xa,    y0,xa+a,      y1,bid+su+0,bid-sl,bid+1,bid-1);
  blockline(fid,bid+1,xa+a,  y0,xa+a+0.5*b,y1,bid+su+1,    -1,bid+2,bid+0);
  blockline(fid,bid+2,xa+a+0.5*b,y0,xa+a+b,y1,bid+su+2,    -1,bid+3,bid+1);
  bid=bid+3;
  xa=xa+a+b;
  
  % final cell forming the bend down
  blockline(fid,bid+0,xa,    y0,xa+a,      y1,bid+su+0, bid+1,   -1,bid-1);
  bid=bid+1;
  
  fprintf(fid,'\n');

  
  %top->bottom
  x0=xa; x1=xa+a;
  blockline(fid,bid+0,x0, y0-1.0*d,x1, y0-0.0*d,bid-1,bid+1, -1, -1);
  blockline(fid,bid+1,x0, y0-2.5*d,x1, y0-1.0*d,bid+0,bid+2, -1, -1);
  blockline(fid,bid+2,x0, y0-4.5*d,x1, y0-2.5*d,bid+1,bid+3, -1, -1);
  blockline(fid,bid+3,x0,0.5-8.0*d,x1, y0-4.5*d,bid+2,bid+4, -1, -1);
  blockline(fid,bid+4,x0,0.5- 10*d,x1,0.5-8.0*d,bid+3,   -3, -1, -1);
  bid=bid+5;

  fprintf(fid,'\n');

end % next layer+outlet


% top layer + last outlet
i=n;
% left -> right
y0=0.5+xi(i-1); y1=0.5+xi(i);
su=11+(i-1)*3;
sl=11+(i-2)*3;
%#         block_id x0 y0 x1 y1             Top  Bottom   Right Left);
blockline(fid,bid+0,0.5-c,    y0,      0.5,y1,-1,bid-sl+0,bid+1,   -1);
blockline(fid,bid+1,0.5,      y0,    0.5+e,y1,-1,bid-sl+1,bid+2,bid+0);
blockline(fid,bid+2,0.5+e,    y0,0.5+2.5*e,y1,-1,bid-sl+2,bid+3,bid+1);
blockline(fid,bid+3,0.5+2.5*e,y0,0.5+4.0*e,y1,-1,bid-sl+3,bid+4,bid+2);
blockline(fid,bid+4,0.5+4.0*e,y0,0.5+5.0*e,y1,-1,bid-sl+4,bid+5,bid+3);
  
fprintf(fid,'\n');


bid=bid+5;
% 3 intermediate cells with lower neighbours
xa=0.5+5.0*e;
for j=3:i
  blockline(fid,bid+0,xa,    y0,xa+a,      y1,-1,bid-sl+0,bid+1,bid-1);
  blockline(fid,bid+1,xa+a,  y0,xa+a+0.5*b,y1,-1,bid-sl+1,bid+2,bid+0);
  blockline(fid,bid+2,xa+a+0.5*b,y0,xa+a+b,y1,-1,bid-sl+2,bid+3,bid+1);
  bid=bid+3;
  xa=xa+a+b;
end
% 3 intermediate cells with no lower neighbours
blockline(fid,bid+0,xa,    y0,xa+a,      y1,-1,bid-sl,bid+1,bid-1);
blockline(fid,bid+1,xa+a,  y0,xa+a+0.5*b,y1,-1,    -1,bid+2,bid+0);
blockline(fid,bid+2,xa+a+0.5*b,y0,xa+a+b,y1,-1,    -1,bid+3,bid+1);
bid=bid+3;
xa=xa+a+b;
  
% final cell forming the bend down
blockline(fid,bid+0,xa,    y0,xa+a,      y1,-1, bid+1,   -1,bid-1);
bid=bid+1;
  
fprintf(fid,'\n');

  
%top->bottom
x0=xa; x1=xa+a;
blockline(fid,bid+0,x0, y0-1.0*d,x1, y0-0.0*d,bid-1,bid+1, -1, -1);
blockline(fid,bid+1,x0, y0-2.5*d,x1, y0-1.0*d,bid+0,bid+2, -1, -1);
blockline(fid,bid+2,x0, y0-4.5*d,x1, y0-2.5*d,bid+1,bid+3, -1, -1);
blockline(fid,bid+3,x0,0.5-8.0*d,x1, y0-4.5*d,bid+2,bid+4, -1, -1);
blockline(fid,bid+4,x0,0.5- 10*d,x1,0.5-8.0*d,bid+3,   -3, -1, -1);
bid=bid+5;


%end blocks


fprintf(fid,'# number of criteria\n');
fprintf(fid,'%d\n', n+1+n);

rec_voli=1/(n*2*a*a);

fprintf(fid,'# criteria: crit_id  block_id  type  coeff\n');
% first the mean pressure bit:
bid=14;
for i=1:n
  for j=2:i
    bid=bid+3;
  end
  fprintf(fid,'%2d   %3d    %2d    %21.16e\n',...
      i-1, bid, 3, rec_voli);
  
  crit_marker(bid+1)=1;
  
  bid=bid+11;
end

fprintf(fid,'\n');
% the reset switch
fprintf(fid,'%2d   %3d    %2d    %21.16e\n',...
      n, -1, 2, 0.0);
fprintf(fid,'\n');

% now the mean deviation bit:
bid=14;
for i=1:n
  for j=2:i
    bid=bid+3;
  end
  fprintf(fid,'%2d   %3d    %2d    %21.16e\n',...
      n+i, bid, 13, rec_voli);
  
  bid=bid+11;
end

fclose(fid);

if write_fem_fvm==1
  % convert into FEINS mesh
  blocks_convert_fem(blocks_d, blocks_i, crit_marker);
end

return;


function blockline(fid, bid, x0, y0, x1, y1, top,bot,rig,lef)

% double and integer vars relating to the blocks
global blocks_d; global blocks_i;

fprintf(fid,...
    '%3d  %+21.15e  %+21.15e  %+21.15e  %+21.15e    %2d  %2d  %2d  %2d\n',...
    bid, x0, y0, x1, y1, top,bot,rig,lef);

blocks_d=[blocks_d; [x0, y0, x1, y1]];
blocks_i=[blocks_i; [top,bot,rig,lef]];

return;





function blocks_convert_fem(blk_d, blk_i, crit_marker);
% convert blocks into FEM mesh

global fem_par; global nu;

doplots=1;

nblk=size(blk_d,1);

% init 
vertex   = zeros(9*nblk,2); % at most 9 new nodes per block
elements = zeros(8*nblk,3); % exaclty 8 elements per block

bound=zeros(0,4);           % 4 data per bound entry: nod1, nod2, func, sseg
                            % func==0 => noslip, func==-1 =>no BC,
                            % just shape segment
func = zeros(0,3);          % a*x^2+b*x+c
segs = zeros(0,5);          % 5 data: nod1, px0,py0, px1,py1

bl_vx      = zeros(nblk,9); % the nine nodes for each block
crit_elems = [];            % the elements that have to go into pcvol

% build the interdependancy info for the blocks
[block_par, par_val]=para_inter_dep(blk_d, blk_i);

fem_par=block_par;

vx_nr=0;
% loop over the blocks, define elements and required vertices
for bl=1:nblk
  top=blk_i(bl,1)+1;
  bot=blk_i(bl,2)+1;
  rig=blk_i(bl,3)+1;
  lef=blk_i(bl,4)+1;
  
  lnods=zeros(9,1);
  % local nodes are ordered  7 6 5
  %                          8 9 4
  %                          1 2 3
  
  % node 1, bottom left
  if ((lef<bl)&&(bot<bl) && (lef>0)&&(bot>0))
    lnods(1)=bl_vx(lef,3);
    if lnods(1)~=bl_vx(bot,7); error('1,1'); end
  elseif (lef<bl) && (lef>0)
    lnods(1)=bl_vx(lef,3);
  elseif (bot<bl) && (bot>0)
    lnods(1)=bl_vx(bot,7);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    vertex(vx_nr,:) = blk_d(bl,[1,2]); %[x0, y0, x1, y1]
    lnods(1)=vx_nr;
  end

  % node 3, bottom right
  if ((rig<bl)&&(bot<bl) && (rig>0)&&(bot>0))
    lnods(3)=bl_vx(rig,1);
    if lnods(3)~=bl_vx(bot,5); error('1,1'); end
  elseif (rig<bl) && (rig>0)
    lnods(3)=bl_vx(rig,1);
  elseif (bot<bl) && (bot>0)
    lnods(3)=bl_vx(bot,5);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    vertex(vx_nr,:) = blk_d(bl,[3,2]); %[x0, y0, x1, y1]
    lnods(3)=vx_nr;
  end

  % node 5, top right
  if ((rig<bl)&&(top<bl) && (rig>0)&&(top>0))
    lnods(5)=bl_vx(rig,7);
    if lnods(5)~=bl_vx(top,3); error('1,1'); end
  elseif (rig<bl) && (rig>0)
    lnods(5)=bl_vx(rig,7);
  elseif (top<bl) && (top>0)
    lnods(5)=bl_vx(top,3);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    vertex(vx_nr,:) = blk_d(bl,[3,4]); %[x0, y0, x1, y1]
    lnods(5)=vx_nr;
  end

  % node 7, top left
  if ((lef<bl)&&(top<bl) && (lef>0)&&(top>0))
    lnods(7)=bl_vx(lef,5);
    if lnods(7)~=bl_vx(top,1); error('1,1'); end
  elseif (lef<bl) && (lef>0)
    lnods(7)=bl_vx(lef,5);
  elseif (top<bl) && (top>0)
    lnods(7)=bl_vx(top,1);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    vertex(vx_nr,:) = blk_d(bl,[1,4]); %[x0, y0, x1, y1]
    lnods(7)=vx_nr;
  end

  % node 2, bottom
  if (bot<bl) && (bot>0)
    lnods(2)=bl_vx(bot,6);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    %[x0, y0, x1, y1]
    vertex(vx_nr,:) = 0.5* ( blk_d(bl,[1,2])+blk_d(bl,[3,2]) ); 
    lnods(2)=vx_nr;
  end

  % node 6, top
  if (top<bl) && (top>0)
    lnods(6)=bl_vx(top,2);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    %[x0, y0, x1, y1]
    vertex(vx_nr,:) = 0.5* ( blk_d(bl,[1,4])+blk_d(bl,[3,4]) ); 
    lnods(6)=vx_nr;
  end

  % node 4, right
  if (rig<bl) && (rig>0)
    lnods(4)=bl_vx(rig,8);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    %[x0, y0, x1, y1]
    vertex(vx_nr,:) = 0.5* ( blk_d(bl,[3,2])+blk_d(bl,[3,4]) ); 
    lnods(4)=vx_nr;
  end

  % node 8, left
  if (lef<bl) && (lef>0)
    lnods(8)=bl_vx(lef,4);
  else
    % create new vertex
    vx_nr=vx_nr+1;
    %[x0, y0, x1, y1]
    vertex(vx_nr,:) = 0.5* ( blk_d(bl,[1,2])+blk_d(bl,[1,4]) ); 
    lnods(8)=vx_nr;
  end

  % node 9, create new vertex
  vx_nr=vx_nr+1;
  %[x0, y0, x1, y1]
  vertex(vx_nr,:) = 0.5* ( blk_d(bl,[1,2])+blk_d(bl,[3,4]) ); 
  lnods(9)=vx_nr;
  
  % all nodes known now, set bl_vx
  bl_vx(bl,:)=lnods';
  
  % create the elements
  elements((bl-1)*8+1,:)=lnods([1,2,9])';
  elements((bl-1)*8+2,:)=lnods([2,3,9])';

  elements((bl-1)*8+3,:)=lnods([3,4,9])';
  elements((bl-1)*8+4,:)=lnods([4,5,9])';

  elements((bl-1)*8+5,:)=lnods([5,6,9])';
  elements((bl-1)*8+6,:)=lnods([6,7,9])';

  elements((bl-1)*8+7,:)=lnods([7,8,9])';
  elements((bl-1)*8+8,:)=lnods([8,1,9])';

  % create pcvol markers
  if (crit_marker(bl)==1)
    crit_elems=[crit_elems, (bl-1)*8+(1:8)]; 
  end

  % do the boundary conditions
  % the local node for each edge
  bnodes=[ 5,6,7; 1,2,3; 3,4,5; 7,8,1];
  % the local parameters for each side
  % sides [t,b,r,l] parameters are [t,b,r,l], need [x0,y0,x1,y1]
  side_seg_pars = [
    3,1,4,1;...
    4,2,3,2;...
    3,2,3,1;...
    4,1,4,2]; 
    
    
  % data for the quadratic functions for this block
  x0=blk_d(bl,1);   x1=blk_d(bl,3);
  fscale=4/(x1-x0)^2;
  fqdat=fscale*[ -1.0, x0+x1, -x0*x1];
  
  % loop all sides of the block (top,bot,rig,lef)
  for i=1:4 
    switch(blk_i(bl,i))
      case -1  % noslip
	fid=0;
      case -2  % inlet
	% create the function entry
	func=[func; fqdat];
	fid=size(func,1);
      case -3  % outlet
	% create the function entry
	func=[func; -fqdat];
	fid=size(func,1);
      otherwise
	fid=-1; % no real boundary
	if (blk_i(bl,i)<0)
	  this_blk_i=[blk_i(bl,i),i],
	  error('something wrong')
	end
    end % end switch
    
    % only create a new boundary segment if there is no neigbour or the
    % neigbour has a higher block id
    if (blk_i(bl,i)+1>bl)||(blk_i(bl,i)+1<=0)
      segs=[segs; ...
	[lnods(bnodes(i,1)), block_par(bl,side_seg_pars(i,:)) ] ];
      
      segid=size(segs,1);
      
      bound=[bound; ...
	[lnods(bnodes(i,1:2))', fid, segid]; ...
	[lnods(bnodes(i,2:3))', fid, segid]  ]; 
      
    end

  end % end loop side of block
	

end % loop over blocks

% plot the mesh
if (doplots==1)
  figure(1);
  clf; hold on;

  for i=1:8*nblk
    xy=vertex(elements(i,[1,2,3,1]),:);
    %  plot(xy(:,1),xy(:,2),'m-');
  end
  
  bd_nr=size(bound,1);
  for i=1:bd_nr
    if bound(i,3)>0
      % must be horrizontal for this mesh 
      fid=bound(i,3);
      x0=vertex(bound(i,1),1);
      x1=vertex(bound(i,2),1);
      y= vertex(bound(i,1),2);
      
      px=vertex(bound(i,1:2),1);
      py=vertex(bound(i,1:2),2);
      plot(px,py,'k-')

      px=x0:(x1-x0)/32:x1;
      py=y+func(fid,1)*px.^2+func(fid,2)*px+func(fid,3);
      col='r-';
    else
      px=vertex(bound(i,1:2),1);
      py=vertex(bound(i,1:2),2);
      if (bound(i,3)==0)
	% still boundary
	col='k-';
      else
	% only parametric shape segment
	col='b-';
      end
    end
    plot(px,py,col);
  end
  pause(0);
end

el_nr = 8*nblk;
bd_nr = size(bound,1);
pv_nr = length(crit_elems);
fu_nr = size(func,1)+1;
sg_nr = size(segs,1);
sp_nr = length(par_val);

% everything is in place now to write the FEINS mesh to a file
mf = fopen('divider_n.f1m','w');
fprintf(mf,[
  '<!-- mesh file for flow divider  \n',...
  '-->\n',...
  '<header>\n',...y
  '  <format    0 >\n',...
  '  <dim       2 >\n',...
  '  <problem   2 >\n',...
  '  <meshgen   0 >\n',...
  '  <vertex    %d >\n',...
  '  <elements  %d >\n',...
  '  <boundary  %d >\n',...
  '  <holes     0 >\n',...
  '  <pcsurf    0 >\n',...
  '  <pcvol     %d >\n',...
  '  <pccrit    1 >\n',...
  '  <function  %d >\n',...
  '  <parameter 1 >\n',...
  '  <shape_seg %d >\n',...
  '  <shape_par %d >\n',...
  '</header>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a vertex entry is:\n',...
  '       id x y [z]\n',...
  '-->\n',...
  '<vertex>\n'],...
  vx_nr, el_nr, bd_nr, pv_nr, fu_nr, sg_nr, sp_nr);

% the vertices
for i=1:vx_nr
  fprintf(mf,'  %3d  %22.15e  %22.15e\n',i-1, vertex(i,1), vertex(i,2) );
end

fprintf(mf,[
  '</vertex>\n',...
  '\n',...
  '<!-- syntax of a element entry is:\n',...
  '       id type func nod1 nod2 ... nodk\n',...
  '-->\n',...
  '<elements>\n']);

% the elements
for i=1:el_nr
  fprintf(mf,'  %3d   1  -1   %3d %3d %3d\n', i-1, ...
      elements(i,1)-1, elements(i,2)-1, elements(i,3)-1);
end


fprintf(mf,[
  '</elements>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a boundary entry is:\n',...
  '       id bctype type func orient sseg nod1 ... nodk\n',...
  '-->\n',...
  '<boundary>\n']);

% boundary
for i=1:bd_nr
  if bound(i,3)>=0
    bct = 1;   % proper Diri BC
  else
    bct = -1;  % just segment
  end
  bcf  = bound(i,3); % in that one function 0 is really allready used 
  sid  = bound(i,4)-1;
  nod1 = bound(i,1)-1;
  nod2 = bound(i,2)-1;
  
  fprintf(mf,  '   %3d   %2d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1, bct,   0, bcf,   1,   sid, nod1,    nod2  );
%     id bctype type func orient sseg nod1 ... nodk
end

fprintf(mf,[
  '</boundary>\n',...
  '\n',...
  '<!--  syntax of a holes entry is:\n',...
  '       id x y [z]\n',...
  '-->\n',...
  '<holes>\n',...
  '</holes>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a pcsurf entry is:\n',...
  '       id c_id orient nod1 ... nodk\n',...
  '-->\n',...
  '<pcsurf>\n',...
  '</pcsurf>\n',...
  '\n',...
  '<!-- syntax of a pcvol entry is:\n',...
  '       id c_id elem\n',...
  '-->\n',...
  '<pcvol>\n',...
  ]);

% pcvol
for i=1:pv_nr
  fprintf(mf,  '   %3d   %d   %3d\n',...
      i-1,  0,   crit_elems(i)-1 );
%     id  c_id   element
end

fprintf(mf,[
  '</pcvol>\n',...
  '   \n',...
  '\n',...
  '<!-- syntax of a pccrit entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<pccrit>\n',...
  '   0    5    2   1.0\n',...
  '</pccrit>\n',...
  '\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a function entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<function>\n',...
  '   0  100   0.0   0.0\n']);

% function part, all inlets prescribe a quadratic velocity profile in y dirtection 
for i=2:fu_nr
  fprintf(mf,'  %2d  102   0 0 0 0 0 0    %22.15e   0 0  %22.15e  0   %22.15e\n',...
          i-1, func(i-1,1), func(i-1,2), func(i-1,3) );
end

fprintf(mf,[
  '</function>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of the parameter entry is:\n',...
  '       data1 ... datak\n',...
  '-->\n',...
  '<parameter>\n',...
  '   %8.1e \n',...
  '</parameter>\n',...
  '\n\n',...
  '<!-- syntax of the shape_segm entry is:\n',...
  '       id   type  nod1   para1 ... parak\n',...
  '-->\n',...
  '<shape_seg>\n'],nu);

for i=1:sg_nr
  fprintf(mf, '  %3d   1 %3d    %3d  %3d  %3d  %3d\n',... 
      i-1,    segs(i,1)-1, segs(i,2)-1,segs(i,3)-1,segs(i,4)-1,segs(i,5)-1);
  %   id  type  nod1       x0  y0 x1 y1
end

fprintf(mf,[
  '</shape_seg>\n',...
  '\n',...
  '<!-- syntax of the shape_para entry is:\n',...
  '       id  data\n',...
  '-->\n',...
  '<shape_par>\n']);

for i=1:sp_nr
  fprintf(mf, '   %3d   %22.15e\n', i-1,  par_val(i)); 
end

fprintf(mf,[
  '</shape_par>\n',...
  '\n' ]);

fclose(mf);


return;














function [bp, pv]=para_inter_dep(blk_d, blk_i)
%
% [top,bot,rig,lef] = bp(i,:)     parameter id's for each block i
%                     pv          values of these parameters

nblk=size(blk_i,1);

bp = - ones(nblk, 4); % initialise as no parameters defined for anything

pv = zeros(0,1);

% for each parameter the coparameter (eg top-bottom => 1-2) 
copar= [ 2; 1; 4; 3 ];
% direction of propagation e.g. top -> horizontal=3, left -> vertical=1
prdir= [ 3; 3; 1; 1 ];

% position of the paramter in the block coordinates
% [top,bot,rig,lef]  --> position in [x0,y0,x1,y1]
side_xi= [ 4; 2; 3; 1];

% loop all blocks, 
% check for each parameter if it is already defined
% if it is do nothing
% if it isn't, define a new parameter, propagate this parameter
% definition to all neigbours
npar=0;
for bl=1:nblk
  for side=1:4
    pid=bp(bl,side);
  
    % if this block was not known already then define a
    % new parameter 
    if (pid<0)
      % new parameter
      npar=npar+1;
      pid=npar;
      
      pv=[pv; blk_d(bl, side_xi(side))];
    end
    %  propagate this definition to all neigbours
    % propagate in both directions, eg. first right then left
    bp=para_propagate(bl,pid, side,copar(side),prdir(side),   bp,blk_i);
    bp=para_propagate(bl,pid, side,copar(side),prdir(side)+1, bp,blk_i);

    %bp, error('stop')
  end
end

%pv, size(pv), bp

return;








function bo =para_propagate(bl, pid, par, copar, pdir, bi, blk_i )

% copy in to out
bo=bi;

% define this parameter id
if ( bo(bl,par)<0 )||( bo(bl,par)==pid )
  bo(bl,par)=pid;
  %plotpars(bl,par,pid);
else
  bo,
  [bl, par, pid, bo(bl,par)]
  error('conflicting parameter definition');
end

% define the id of the copar
neighb = blk_i(bl,par)+1;
if neighb>0
  if ( bo(neighb,copar)<0 )||( bo(neighb,copar)==pid )
    bo(neighb,copar)=pid;
    %plotpars(neighb,copar,pid);
  else
    error('conflicting parameter definition (copar)');
  end
end


% check if there is a neigbour in directio pdir, if so, propagate to that
neighb = blk_i(bl,pdir)+1;
if neighb>0
  bo=para_propagate(neighb, pid, par, copar, pdir, bo, blk_i );
end

return;


function plotpars(bl,par,pid)

global blocks_d; global blocks_i;

% top,bot,rig,lef,    x0,y0,x1,y1

%     x0,y0, x1,y1  which are to be averaged
idx=[ 1, 4, 3, 4;...
  1, 2, 3, 2;...
  3, 2, 3, 4;...
  1, 2, 1, 4];

x=0.5*( blocks_d(bl,idx(par,1)) + blocks_d(bl,idx(par,3)) );
y=0.5*( blocks_d(bl,idx(par,2)) + blocks_d(bl,idx(par,4)) );


text(x,y,sprintf('%d',pid));
hold on;

return;


