/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "sparse.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

#include <unistd.h>
#include <string.h>

/* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>
#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }

int main(int argc, char *argv[])
{

  struct sparse K;

  struct vector x, y, invdiag;

  int  err, num;
  FIDX i;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double ti, t1, t_mult, t_gauss_seidel_fwd, t_gauss_seidel_bwd;

  err=sparse_mat_read_file(&K, argv[1]);
  FUNCTION_FAILURE_HANDLE( err, sparse_mat_read_file, main);

  err=vector_alloc(&x, K.row_nr);
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc(&y, K.row_nr);
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err = sparse_convert_compressed_row( &K );
  FUNCTION_FAILURE_HANDLE( err, sparse_convert_compressed_row, 
			       main );


  num=100;

  /* first matrix-vector multiply */
  for (i=0; i<K.row_nr; i++)
    {
      x.V[i]=i;
      y.V[i]=i*i-5*i;
    }

  TIMEGET;
  t1=ti;
  for (i=0; i<num; i++)
    {
      err=sparse_mul_mat_vec_add( &K, &x, &y);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec_add,
			       main);  
    }
  TIMEGET;
  t_mult=ti-t1;


  /* then Gauss-Seidel sweeps */
  vector_init(&invdiag); 
  err=sparse_extract_invdiag(&K, &invdiag);
  FUNCTION_FAILURE_HANDLE( err, sparse_extract_invdiag,
			   main);  
  for (i=0; i<K.row_nr; i++)
    {
      x.V[i]=i;
      y.V[i]=i*i-5*i;
    }
  

  TIMEGET;
  t1=ti;
  for (i=0; i<num; i++)
    {
      err=sparse_GS_sweep_fwd( &K, &y, &invdiag, 1, &x);
      FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_fwd,
			       main);  
    }
  TIMEGET;
  t_gauss_seidel_fwd=ti-t1;

  TIMEGET;
  t1=ti;
  for (i=0; i<num; i++)
    {
      err=sparse_GS_sweep_bwd( &K, &y, &invdiag, 1, &x);
      FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_bwd,
			       main);  
    }
  TIMEGET;
  t_gauss_seidel_bwd=ti-t1;
  
  printf("time/multiply= %16.8e  time/GS_fwd=%16.8e time/GS_bwd=%16.8e "
	 " size= %7d  file= %s\n",
	 t_mult/num, t_gauss_seidel_fwd/num, t_gauss_seidel_bwd/num, 
	 (int) K.row_nr, argv[1]);


  vector_free(&y);
  vector_free(&x);

  sparse_free(&K);

  return SUCCESS;


}
