/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "navsto_solver.h"

#include <unistd.h>
#include <string.h>

  /* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>


int main(int argc, char *argv[])
{
  struct mesh msh1;

  struct vector solref;
  struct vector pccrit;
  struct vector spar;
  struct vector *DIDF;

  struct solver_settings set;

  char *buffer;

  int  err, ihlp;
  FIDX i, lmax, level0, lvl;

  if (argc<1+1) 
    {
      printf("\n"
	     "%s: not enough arguments\n\n"
	     "usage:\n\n"
	     "   %s meshfile.f1m [para_file]\n"
	     "\n", argv[0], argv[0]);
      printf("  para_file - if provided the shape parameters in the\n"
	     "              mesh file are overriden\n"
	     "\n");
      printf("The computed solution will be in meshfile.f1m_sol,\n"
	     "the criteria in meshfile.f1m_crit and the sensitivities\n"
	     "in meshfile.f1m_grad\n");
      return FAIL;
    }

  printf("meshfile: %s      STABTYPE:%d ", argv[1], STABTYPE);
#ifdef RAMAGE_NOT_TEZDUYAR
  printf("  stab-par: Ramage\n");
#else
  printf("  stab-par: Tezduyar\n");
#endif
  printf("          FIDX size= %d     pointer size = %d\n",
	 sizeof(FIDX), sizeof(FIDX *) );

  vector_init(&spar);
  
  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);

  if (argc>=2+1)
    { 
      /* 2-nd argument is spar override file */
      err=vector_read_file(&spar, argv[2]);
      FUNCTION_FAILURE_HANDLE( err, vector_read_file, main);
    }
  /* tell navsto_solver we don't have a reference solution */
  solref.len=0;

  /* { 
     err=vector_read_file(&solref, "divider_n.f1m_sol");
     FUNCTION_FAILURE_HANDLE( err, vector_read_file, main);
     } /* */
  
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* set refinement levels */
  level0  = set.refine_ini;
  lmax    = set.refine_steps;

  fprintf(stderr, "debug st_nr problems: vx_nr=%d >?> st_nr=%d\n",
	  msh1.vx_nr,msh1.st_nr);

  for (lvl=0; lvl<level0 ; lvl++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }


  /* reset hierarchy */
  msh1.hi_nr=0;
  msh1.lvl=-1;
  for (i=0; i<msh1.eg_nr; i++)
    {
      msh1.edge[i*msh1.eg_w+MCT1EGLVL]=-1;
    }

  /* if spar is set, see if it is compatible, and replace spar in the
     mesh */
  if (spar.len!=0)
    {
      if (spar.len==msh1.sp_nr)
	{
	  free(msh1.spar);
	  msh1.spar=spar.V;
	  msh1.sp_max=msh1.sp_nr;
	  vector_init(&spar);
	}
      else
	{
	  fprintf(stderr,"main: the provided shape parameter vector "
		  "has wrong size\n");
	  return FAIL;
	}
    }

  /* err=mesh_write_ssegs_svg_tx( &msh1, 18, "visual/shape_segs", 1);
     FUNCTION_FAILURE_HANDLE( err, mesh_write_ssegs_svg_tx, main);  
     fprintf(stderr,"plot only --> stop!\n"); exit(1); /* */

  /* allocate space for the performance criteria */
  err=vector_alloc(&pccrit, msh1.pc_nr);
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  /* allocate memory for the sensitivities */
  TRY_MALLOC(DIDF, msh1.pc_nr, struct vector, main);


  /* solve */
  err=navsto_solver(&msh1, lmax, &set, &solref, &pccrit, DIDF);
  FUNCTION_FAILURE_HANDLE(err, navsto_solver, main);

  /* finite difference check of DIDF */
  if (0==1)
  { 
    double h=1e-5;
    FIDX j, sp_min_test, sp_max_test;
    struct vector *DIDF_FD;
    struct vector pccrit_ph, pccrit_mh;

    sp_min_test=0; sp_max_test=2; 

    TRY_MALLOC(DIDF_FD, msh1.pc_nr, struct vector, main);
    for (i=0; i<msh1.pc_nr; i++)
      {
	err=vector_alloc(&DIDF_FD[i], msh1.sp_nr);
	FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
      }
    err=vector_alloc(&pccrit_ph, msh1.pc_nr);
    FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
    err=vector_alloc(&pccrit_mh, msh1.pc_nr);
    FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

    for (i=sp_min_test; i<sp_max_test; i++)
      {
	double par_save;
	par_save=msh1.spar[i*MC2XSPLN+MC2XSPDATA];

	msh1.spar[i*MC2XSPLN+MC2XSPDATA] = par_save + h;

	err=navsto_solver(&msh1, lmax, &set, &solref, &pccrit_ph, NULL);
	FUNCTION_FAILURE_HANDLE(err, navsto_solver, main);


	msh1.spar[i*MC2XSPLN+MC2XSPDATA] = par_save - h;

	err=navsto_solver(&msh1, lmax, &set, &solref, &pccrit_mh, NULL);
	FUNCTION_FAILURE_HANDLE(err, navsto_solver, main);

	for (j=0; j<msh1.pc_nr; j++)
	  {
	    DIDF_FD[j].V[i]=(pccrit_ph.V[j]-pccrit_mh.V[j])/(2.0*h);
	  }
    
	msh1.spar[i*MC2XSPLN+MC2XSPDATA] = par_save;
      }

    for(j=0; j<msh1.pc_nr; j++)
      for (i=sp_min_test; i<sp_max_test; i++)
	{
	  printf("[%d,%4d]  DIDF=%10.3e   FD=%10.3e   diff=%8.1e  "
		 " rel=%8.1e\n",
		 (int) j, (int) i,
		 DIDF[j].V[i],DIDF_FD[j].V[i],
		 DIDF[j].V[i]-DIDF_FD[j].V[i],
		 (DIDF[j].V[i]-DIDF_FD[j].V[i])/fabs(DIDF[j].V[i]));
	}

    printf("\n\nif results differ check first that "
	   "nu = nu_file in all calls\n");

    for (i=0; i<msh1.pc_nr; i++)
      {
	vector_free(&DIDF_FD[i]);
      }
    free(DIDF_FD);
    vector_free(&pccrit_ph);
    vector_free(&pccrit_mh);
  }


  /* write results to files */
  /* criteria */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_crit");
  err=vector_write_file(&pccrit, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);

  /* derivatives */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_grad");
  err=vector_n_write_file(msh1.pc_nr, DIDF, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_n_write_file, main);

  /* reference solution */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_sol");
  err=vector_write_file(&solref, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);


  vector_free(&pccrit);
  vector_free(&solref);

  for (i=0; i<msh1.pc_nr; i++)
    {
      vector_free(&DIDF[i]);
    }
  free(DIDF);

  mesh_free(&msh1);
  free(buffer);

  return SUCCESS;
}
