/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "assembly.h"
#include "gen_aux.h"
#include "mesh_deform.h"
#include "lame_adj.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

#include <unistd.h>
#include <string.h>


int main(int argc, char *argv[])
{	
  struct sparse *Ks;
  struct projector1 P;
  struct coarse_mat K_inv;
  
  struct vector psi;    /* solution of K^T * Psi = dIdu^T */
  struct vector dIdu;   /* derivate of Is with respect to the solution u */
  struct vector dIdx;  /* gradient of Is with respect to the node positions */
  struct vector dIdF;  /* gradient of Is with respect to the shape parameters */
  struct vector rhs, nodem, dIdx_FD;
  struct mesh msh1;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;
  struct bpxdata bpx;

  struct solver_settings set;
	  
  int  err;
  FIDX dim, vx_nr, i, j, d;

  FIDX level0, lvlm, lvl;

  double mu, lambda;
  int stiffening;
  int vcycles=1;
  int smooths=4;

  double Is, Ism,Isp;
  
  int iter;
  double resi;
  double res_start;

  char *buffer;
  
  res_start = -1.0;
  stiffening = 1;  /* ==1 for real Lame', 
                      ==0 for stiffening of small elements */
  
  if (argc>=1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  solver_settings_init( &set );
  err=mesh_read_file_t1( &msh1, &set, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);
    
  /* set the goals and methods */
  level0  = set.refine_ini;
  lvlm   = set.refine_steps;

  dim=msh1.dim;

  lambda=msh1.para[0];
  mu    =msh1.para[1];
  /* Lame canstants for steel
  mu         = 7.7e4; 
  lambda     = 1.15e5; */


  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  msh1.lvl=-1;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

    
  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks,  lvlm+1, struct sparse, main);

  /* Apply deformation due to shape segments */
  if (msh2.sg_nr>0)
    {
      err=mesh_deform_start_t2(&msh2, Ks);
      FUNCTION_FAILURE_HANDLE( err, mesh_deform, main);
    }    

  vx_nr=msh2.vx_nr;     
  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);  
  for (i=0; i<dim*vx_nr; i++)
    {
      nodem.V[i]=0.0;    
    }
  printf("\n Level = %d : vx_nr = %d\n", 0, msh2.vx_nr);	
      
  /* Mesh Refinement */
  for (lvl=0; lvl<lvlm; lvl++)
    {  
      if(msh2.lvl+1!=lvl)
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%d+1!= %d)\n", msh2.lvl, lvl);
	  return FAIL;
        }
      err=sparse_flex_alloc( &Ks[lvl], dim*vx_nr);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
      err=projector1_alloc( &P, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
      err=vector_alloc( &rhs, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  
      for (i=0; i<dim*vx_nr; i++) rhs.V[i]=0.0;
	  
      /* Assembly stiffness matrix with projector for Dirichlet 
	 on the entire boundary */
      err=assem_lame_tx_tx( &Ks[lvl], &P, &msh2, &rhs, &nodem,
			 lambda, mu, stiffening, 2, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);
   
      /* solve the equation system */
      if (set.solver==0) /* variant 1: direct solve (good for coarse mesh) */
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvl], P.len, P.V,
			  1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++)
	    {
	      rhs.V[P.V[i]]=nodem.V[P.V[i]];
	    }
	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
      if((set.solver==1)||(set.solver==2))
	{/* set multilevel and multigrid data for PCG_MG or PCG_BPX */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
	  
	  /* generate the coarse grid matrix */
	  struct coarse_mat *cmat;
	  FIDX *coarse_bc_nodes;
	  FIDX n_c_bc=0;
	  TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
	  TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
	  for (i=0; i<P.len; i++)
	    {
	      FIDX dof, child;
	      dof=P.V[i];
	      MLVLFINDENTRY(child, dof, 0, ml);
	      if (child>=0)
		{
		  coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
		  n_c_bc++;
		}
            }
	  err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			      1, cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
	  /* free old cmat and add new one */
	  if (mg.cmat != NULL)
	    {
	      coarse_mat_free(mg.cmat);
	      free(mg.cmat);
	    }
	  mg.cmat=cmat;
    	  free(coarse_bc_nodes);
	  mg.vcycles=vcycles;
	  mg.smooths=smooths;
	  if (stiffening==0)
	    {
	      mg.CGC_scale=0.25;
	    }
	  else
	    {
	      mg.CGC_scale=1.0;
	    }
	}
      if(set.solver==1) /* variant 2: PCG with multigrid */
	{
	  double rtol;
	  if (lvl==0)
	    { /* coarse level, use relative tolerance for "initial" */
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    { /* subsequent level, use refinement for "refinement" */
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, 
		&iter, sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvl], &rhs, &mg); /* */
	  FUNCTION_FAILURE_HANDLE( err, PCG, main); /*  */
	  printf("PCG_MG_refine |res|=%8.1e it=%3d\n", resi, iter);
	}
      if (set.solver==2) /* variant 3: PCG with BPX */
        {
	  int iter;
	  double resi, rtol;

	  err=bpx_init_tx(&bpx, &ml);
	  FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
	  bpx.msh = &msh2;
	  bpx.P   = &P;    
	  /* bpx.cmat = mg.cmat; /* */
	  if (lvl==0)
	    {
	      rtol=set.solver_ini_rtol;
	    }
	  else
	    {
	      rtol=set.solver_ref_rtol;
	    }
	  err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
		   sparse_mul_mat_vec, gen_proj_bpx_tx,
		   &Ks[lvl], &rhs, &bpx); /* */
	  /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
	  /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
	  printf("PCG_BPX_refine |res|=%8.1e it=%3d\n", resi, iter);
	  bpx_free(&bpx);
	}
      if((set.solver==1)||(set.solver==2))
	{
	  /* multigrid data and multilevel are no longer required */
	  mg_free(&mg);
	  multilvl_free(&ml);
	}
      /* rhs and projector are no longer required */
      projector1_free(&P);
      vector_free(&rhs);

      /* Apply mesh refinement */
      if (set.refine_type==1)
        { /* Adaptive mesh refinement */
	  /* error estimator */
	  FIDX *marker;
	  TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
	  err=error_est_lame_residual_t2(&msh2, &nodem, lambda, mu, marker,
					 &set, 2); 
	  FUNCTION_FAILURE_HANDLE( err, error_est_lame_residual_t2, main);
	  /* refine the mesh */
	  err=mesh_refine_adaptive_t2( &msh2, marker);
          FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	  free(marker);
        }
      else if (set.refine_type==0)
        { /* Uniform mesh refinement */
	  /* OPTIONAL: estimating the error */
	  FIDX *marker;
	  TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
	  err=error_est_lame_residual_t2(&msh2, &nodem, lambda, mu,
					 marker, &set, 2); 
	  FUNCTION_FAILURE_HANDLE( err, error_est_lame_residual_t2, main);
	  free(marker);
	  /* refine the mesh */
	  err=mesh_refine_uniform_t2( &msh2);
          FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
        }
      else
	{
	  printf("Error in main (test_lame.c): unknown refinement type!"
		 " type = %d\n", set.refine_type);
	  return FAIL;
	}
      if(msh2.lvl!=lvl)
        {
	  printf("Level of Mesh doesn't match with refinement level!"
		 " (%d != %d)\n", msh2.lvl, lvl);
	  return FAIL;
        }
      printf("\n Level = %d : vx_nr = %d\n", lvl+1, msh2.vx_nr);

      /* interpolate the solution vector nodem to the subsequent level */
        {
	  /* redefine multilvl data */
	  err=multilvl_init_tx( &msh2, dim, &ml, 2);
	  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
	  err=mg_init_tx( NULL, &msh2, &ml, &mg, NULL);
	  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
      
	  FIDX node;
	  /* Remark: vx_nr is still the number of vertices on the coarser
	     mesh (lvl), msh2.vx_nr is the number on the finer mesh (lvl+1) */
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*msh2.vx_nr, lvl, ml);
		mg.xl[node]=nodem.V[i+d*vx_nr];
	      }
	      
	  err=mg_interpolate_t2( &mg, lvl, lvl+1, mg.xl);	
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2, main);  
	      
	  /* Now, Re-Alloc and copy back to the solution vector */    
	  vx_nr=msh2.vx_nr;
	  vector_free(&nodem);
	  err=vector_alloc( &nodem, dim*vx_nr );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main); 
	  for (i=0; i<dim*vx_nr; i++) 
	    {
	      nodem.V[i]=0.0;    
	    }
	  for (i=0; i<vx_nr; i++)
	    for (d=0; d<dim; d++)
	      {
		MLVLFINDENTRY(node, i+d*vx_nr, lvl+1, ml);
		nodem.V[i+d*vx_nr]=mg.xl[node];
	      }
	  multilvl_free(&ml);	
	  mg_free(&mg);
	}

	/* Simply adjust the new nodes to math the shape segments */
	err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
	FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 

    } /* End of mesh refinement */


  vx_nr=msh2.vx_nr; 
  
  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);

  /* allocate memory for matrix Ks, P and the vectors */
  err=sparse_flex_alloc( &Ks[lvlm], dim*vx_nr);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);
  err=projector1_alloc( &P, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
  err=vector_alloc( &rhs, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdu, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &psi, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdx, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdx_FD, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  err=vector_alloc( &dIdF, msh2.sp_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);
  
  /* set vectors to zero */
  for (i=0; i<dim*vx_nr; i++)
    {
      rhs.V[i]  =0.0;
      psi.V[i]  =0.0;
      dIdu.V[i] =0.0;
      dIdx.V[i]=0.0;
      dIdx_FD.V[i]=0.0;
    }

  /* assemble the system on the fine mesh */
  err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
		     lambda, mu, stiffening, 2, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);


  /* solve the equation system */
  if (set.solver==0) /* variant 1: direct solve */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* project rhs */
      for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

      err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      coarse_mat_free(&K_inv);
    }
  if ((set.solver==1)||(set.solver==2))
    { /* set multilevel and multigrid data for PCG_MG and PCG_BPX */
      /* define the multigrid data */
      err=multilvl_init_tx( &msh2, dim, &ml, 2);
      FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);
      err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
      FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);
      
      
      mg.vcycles=vcycles;
      mg.smooths=smooths;
      if (stiffening==0)
	mg.CGC_scale=0.25;
      else
	mg.CGC_scale=1.0;

      /* generate the coarse grid matrix */
      struct coarse_mat *cmat;
      FIDX *coarse_bc_nodes;
      FIDX n_c_bc=0;

      TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
      TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
      
      for (i=0; i<P.len; i++)
	{
	  FIDX dof, child;
	  dof=P.V[i];
	  MLVLFINDENTRY(child, dof, 0, ml);
	  if (child>=0)
	    {
	      coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
	      n_c_bc++;
	    }
	}
      err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			  1, cmat );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
      mg.cmat=cmat;
      
      free(coarse_bc_nodes);
    }
  if (set.solver==1) /* variant 2: PCG_MG */
    {
      double rtol;
      if (lvlm==0)
	{
	  rtol=set.solver_ini_rtol;
	}
      else
	{
	  rtol=set.solver_ref_rtol;
	}
      err=PCG( 100, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvlm], &rhs, &mg); /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      printf("PCG_MG_main   |res|=%8.1e it=%3d\n", resi, iter);
    }
  if (set.solver==2) /* variant 3: PCG with BPX */
    {
      int iter;
      double resi, rtol;
      
      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh2;
      bpx.P   = &P;    
      bpx.cmat = mg.cmat; /* */
      if (lvlm==0)
	{
	  rtol=set.solver_ini_rtol;
	}
      else
	{
	  rtol=set.solver_ref_rtol;
	}
      err=PCG( 1000, 3, set.solver_atol, rtol, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_bpx_tx,
	       &Ks[lvl], &rhs, &bpx); /* */
      /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      printf("PCG_BPX_refine |res|=%8.1e it=%3d\n", resi, iter);
      bpx_free(&bpx);
    }
  /* OPTIONAL: estimate error */
  FIDX *marker;
  TRY_MALLOC( marker, msh2.el_nr, FIDX, main);
  err=error_est_lame_residual_t2(&msh2, &nodem, lambda, mu, marker, &set, 2);  
  FUNCTION_FAILURE_HANDLE( err, error_est_lame_residual_t2, main);
  free(marker); 

  if(set.write_mesh!=0) /* write data for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &nodem, dim, NULL, 0,
                              16, "visual/Lame_sol" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   

   
  /* Compute value of pccrits and the rhs for the adjoint problem */
  err=lame_dIdu_t2(&msh2, &nodem, &dIdu, &Is, lambda, mu);
  FUNCTION_FAILURE_HANDLE( err, lame_dIdu_t2, main);
  printf("I = %12.10e \n", Is);   
    

  if (set.solver==0) /* variant 1: direct solve */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* project rhs */
      for (i=0; i<P.len; i++) dIdu.V[P.V[i]]=0.0; 

      err=coarse_mat_solve( &K_inv, NoTrans, &dIdu, &psi);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      coarse_mat_free(&K_inv);
    }
  if (set.solver==1) /* variant 2: PCG */
    {
      /* Compute psi by solving K^T*psi = dIdu */
      err=PCG( 100, 3, set.solver_atol, set.solver_ini_rtol, 1, 
	       &psi, &resi, &iter,sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvlm], &dIdu, &mg);
      printf("PCG_MG_adj |res|=%8.1e it=%3d\n", resi, iter);
    }
  if (set.solver==2) /* variant 3: PCG with BPX */
    {
      int iter;
      double resi;
      
      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh2;
      bpx.P   = &P;    
      bpx.cmat = mg.cmat; /* */
      err=PCG( 100, 3, set.solver_atol, set.solver_ini_rtol, 1, 
	       &nodem, &resi, &iter, sparse_mul_mat_vec, gen_proj_bpx_tx,
	       &Ks[lvlm], &dIdu, &bpx); /* */
      /* sparse_mat_write_file(&Ks[lvl], "visual/Lame_mat.txt"); /* */
      /* FUNCTION_FAILURE_HANDLE( err, PCG, main); /* */
      printf("PCG_bpx_refine |res|=%8.1e it=%3d\n", resi, iter);
      bpx_free(&bpx);
    }

  if(set.write_mesh!=0) /* write adjoint for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &psi, dim, NULL, 0,
                              16, "visual/Lame_psi" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   

  /* use psi for computing the dIdx of Is */
  err=lame_DIDx_t2(&msh2, lambda, mu, &psi, 
	    &nodem, &dIdx);
  FUNCTION_FAILURE_HANDLE( err, lame_DIDx_t2, main);

  if(set.write_mesh!=0) /* write mesh-gradient for visualisation */
    {
      err=mesh_write_solution_vtk_t2( &msh2, &dIdx, dim, NULL, 0,
                              17, "visual/Lame_DIDX" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			    main);
    }   


  
  /* for tests compute derivatives also by formula from paper Allair
     et. al. (2004) */
  if (0==1)
    {
      struct vector dIdx_allaire, diff_allaire;

      err=vector_alloc( &dIdx_allaire, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      err=lame_Allaire_et_al_2004__formula_8_t2( 
	                  &msh2, lambda, mu, &nodem, &dIdx_allaire);
      FUNCTION_FAILURE_HANDLE( err, lame_Allaire_et_al_2004__formula_8_t2,
			       main);

      err=mesh_write_solution_vtk_t2( &msh2, &dIdx_allaire, dim, NULL, 0,
				      25, "visual/Lame_DIDX_allaire" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			       main);

      err=vector_alloc( &diff_allaire, dim*vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

      for (i=0; i<dIdx_allaire.len; i++)
	{
	  diff_allaire.V[i]=(50.0/68.0*dIdx_allaire.V[i]-dIdx.V[i]);
	  /*printf("allaire/DAM[%5d] = %16.7e\n",(int) i,dIdx_allaire.V[i]/dIdx.V[i]);*/
	}
      err=mesh_write_solution_vtk_t2( &msh2, &diff_allaire, dim, NULL, 0,
				      30, "visual/Lame_DIDX_diff_allaire" );
      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t2,
			       main);
      
    }






  /* compute DIDF from DIDx */
  /*                       msh, insert, nI, dIdx,  dIdF, type */
  err=mesh_sseg_adjust_tx( &msh2, 0,     1, &dIdx, &dIdF, 2); 
  FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main);


    
  /* FINITE DIFFERENCE FOR dIdF_FD */
  for (j=0; j<0*(msh2.sp_nr); j++)
    {
      double hlp=msh2.spar[j];
      double h;
      h= 1e-4;

      /* +h */
      msh2.spar[j] =hlp+h;

      err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 

      err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
	lambda, mu, stiffening, 2, 2);
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
	Isp=0.0;
      err=lame_dIdu_t2( &msh2, &nodem, &dIdu, &Isp, lambda, mu);

      /* -h */
      msh2.spar[j] =hlp-h;

      err=mesh_sseg_adjust_tx( &msh2,  0,     0, NULL, NULL, 2);
      FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, main); 

      err=assem_lame_tx_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
	lambda, mu, stiffening, 2, 2);
	{
	  /* invert the stiffness matrix */
	  err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

	  /* project rhs */
	  for (i=0; i<P.len; i++) rhs.V[P.V[i]]=nodem.V[P.V[i]];

	  err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &nodem);
	  FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

	  coarse_mat_free(&K_inv);
	}
	Ism=0.0;
      err=lame_dIdu_t2( &msh2, &nodem, &dIdu, &Ism, lambda, mu);

      /* +-0 */
      msh2.spar[j] =hlp;
      
      /* auswerten FD */
      dIdx_FD.V[j]=(Isp-Ism)/(2*h);
    
      double a,b,c;
      a=dIdF.V[j];
      b=dIdx_FD.V[j];
      c=fabs((a-b)/b);
      if (c>0.01)
	{
	  printf("%16.14f\t%16.14f\t%f\n", a, b, c);
	}
    }
  /* FINITE DIFFERENCE IS DONE */

  /* write results to files */
  /* I */ 
  strcpy(buffer, argv[1]);
  strcat(buffer,"_crit");
  FILE *out;
  /* open the file */
  out=fopen(buffer, "w");
  if (out==NULL)
    {
      fprintf(stderr, "vector_write_file: "
	      "error opening file \"%s\"\n", buffer);
      return FAIL;
    }
  fprintf(out, "%+24.16e\n", Is);
  /* close the file */
  fclose(out);

  /* dIdF */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_grad");
  err=vector_write_file(&dIdF, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_n_write_file, main);

  /* free some stuff */
  if ((set.solver==1)||(set.solver==2))
    {
      mg_free(&mg);
      multilvl_free(&ml);
    }

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks[lvl]);
  free(Ks);

  projector1_free(&P);
  free(buffer);
  vector_free(&rhs);
  vector_free(&nodem);
  mesh_free(&msh1);
  mesh_free(&msh2);
  vector_free(&dIdu);
  vector_free(&psi);
  vector_free(&dIdx);
  vector_free(&dIdx_FD);
  vector_free(&dIdF);

  return SUCCESS;
}
