/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
stokes_aux.h
*/
#ifndef __STOKES_AUX_H__
#define __STOKES_AUX_H__ 1



#include "feins_macros.h"

#include "sparse_struct.h"
#include "navsto_struct.h"


int stokes_projector_part(void *arg1, struct vector *in,
			  void *notused,
			  struct vector *out
/* performs the boudary condition projection for the subproblems in
     the preconditioner for Stokes problems,
     out = P*I*P^T * in
   where P projects the values for boundary nodes to zero,
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner" 

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_projector_part_yser(void *arg1, struct vector *in,
			       void *notused,
			       struct vector *out
/* performs Yserentant preconditioning and the boudary condition
   projection for the subproblems in the preconditioner for Stokes
   problems,
                out = P*C^{-1}*P^T * in
   where C^{-1} is the hierarchical basis preconditioner Q*Q^T by
   Yserentant
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner 

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_projector_part_bpx(void *arg1, struct vector *in,
			      void *notused,
			      struct vector *out
/* performs BPX preconditioning and the boudary condition
   projection for the subproblems in the preconditioner for Stokes
   problems, 
     out = P*C^{-1}*P^T * in
   where C^{-1} is the BPX preconditioner sum(l=level)Q_l*D_l*Q_l^T and 
   P projects the values for boundary nodes to zero, 
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and precondtioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_projector_part_MG(void *arg1, struct vector *in,
			     void *notused,
			     struct vector *out
/* performs a number of V cycles as (multigrid) preconditioning and
   the boudary condition projection for the subproblems in the
   preconditioner for Stokes problems,
                out = P*C^{-1}*P^T * in
   where C^{-1} is the preconditioner consisting of a number of
   multigrid V cycles
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and mutigrid data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner"

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_projector_w_precon(void *arg1, struct vector *in,
			      void *notused,
			      struct vector *out
/* performs preconditioning and the boudary condition projection for
   stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there) and projects the pressure such that it has zero
   mean 
   
   C^-1 is the Wathen block preconditioner:
        
   C^-1=[I           ] [  I          ] [A^-1        ]  velocity space 1
        [  I         ]*[       I     ]*[     A^-1   ]  velocity space 2
        [    nu*Mp^-1] ]-B_1 -B_2  I ] [           I]  pressure space

   where A is the stiffness matrix of the Laplace operator, nu is 1/Re
   and Mp is the pressure space mass matrix, 

   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_matrix_tim_vec(void *arg1, struct vector *vec,
			  struct vector *out
/* multiplies the sparse matrix  K from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_matrix_part_tim_vec(void *arg1, struct vector *vec,
			       struct vector *out
/* multiplies the sparse matrix K.Fs[lvl] from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/);

int stokes_write_streamfunc_t2( struct mesh *m, struct vector *usol,
				FIDX namlen, char *name
/* recovers the stream function of a 2d velocity field and writes it
   with the mesh into a NAG IRIS EXPLORER readable file (for
   visualisation)  
   
   Input:  m         - the mesh
           usol      - vector with containing the velocity field,
                       stored as usol[j*vx_nr+i] gives the j-th
                       velocity component in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/);
#endif
