/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE stokes_aux.c
HEADER stokes_aux.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "sparse_struct.h"
#include "navsto_struct.h"

*/



/* function prototypes */
#include "stokes_aux.h"

/* prototypes of external functions */
#include <math.h>
#include "feins_lapack.h"
#include "sparse.h"
#include "mesh.h"
#include "stokesassem.h"
#include "elements.h"
#include "cubature.h"
#include "lin_solver.h"
#include "navsto_aux.h"



/*FUNCTION*/
int stokes_projector_part(void *arg1, struct vector *in,
			  void *notused,
			  struct vector *out
/* performs the boudary condition projection for the subproblems in
     the preconditioner for Stokes problems,
     out = P*I*P^T * in
   where P projects the values for boundary nodes to zero,
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner" 

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, dim, vx_nr;
  struct navsto_matrix *K;

  K=arg1;
  vx_nr= (*K).vx_nr;
  dim  = (*K).dim;

  if (((*in).len!=(*out).len)||((*in).len!=vx_nr))
    {
      fprintf(stderr,
	      "stokes_projector_part: dimensions make no sense!\n");
      return FAIL;
    }

  /* copy in to out */
  for (i=0; i<(*in).len; i++)
    {
      (*out).V[i]=(*in).V[i];
    }

  /* apply P=P*P^T */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* done */

  return SUCCESS;
}



/*FUNCTION*/
int stokes_projector_part_yser(void *arg1, struct vector *in,
			       void *notused,
			       struct vector *out
/* performs Yserentant preconditioning and the boudary condition
   projection for the subproblems in the preconditioner for Stokes
   problems,
                out = P*C^{-1}*P^T * in
   where C^{-1} is the hierarchical basis preconditioner Q*Q^T by
   Yserentant
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner 

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, dim, vx_nr, hi_w, hi_nr, fath1, fath2, child, eg_w, eg_nr;
  struct mesh *m;
  struct navsto_matrix *K;

  K=arg1;
  m=(*K).msh;

  vx_nr= (*K).vx_nr;
  dim  = (*K).dim;
  eg_nr= (*m).eg_nr;
  eg_w = (*m).eg_w;
  hi_nr= (*m).hi_nr;
  hi_w = (*m).hi_w;

  if (((*in).len!=(*out).len)||((*in).len!=vx_nr))
    {
      fprintf(stderr,
	      "stokes_projector_part_yser: dimensions make no sense!\n");
      return FAIL;
    }

  /* copy in to out */
  for (i=0; i<(*in).len; i++)
    {
      (*out).V[i]=(*in).V[i];
    }

  /* apply P^T */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* apply Q^T */
  for (i=hi_nr-1; i>=0; i--)
    {
      child=(*m).hier[i*hi_w+MCT2HICHLD];
      fath1=(*m).hier[i*hi_w+MCT2HIFAT1  ];
      fath2=(*m).hier[i*hi_w+MCT2HIFAT1+1];

      (*out).V[fath1]+=0.5*(*out).V[child];
      (*out).V[fath2]+=0.5*(*out).V[child];
    }

  /* apply Q   */
  for (i=0; i<hi_nr; i++)
    {
      child=(*m).hier[i*hi_w+MCT2HICHLD];
      fath1=(*m).hier[i*hi_w+MCT2HIFAT1  ];
      fath2=(*m).hier[i*hi_w+MCT2HIFAT1+1];

      (*out).V[child]+=0.5*( (*out).V[fath1] + (*out).V[fath2] );
    }

  /* apply P */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* done */

  return SUCCESS;

}



/*FUNCTION*/
int stokes_projector_part_bpx(void *arg1, struct vector *in,
			      void *notused,
			      struct vector *out
/* performs BPX preconditioning and the boudary condition
   projection for the subproblems in the preconditioner for Stokes
   problems, 
     out = P*C^{-1}*P^T * in
   where C^{-1} is the BPX preconditioner sum(l=level)Q_l*D_l*Q_l^T and 
   P projects the values for boundary nodes to zero, 
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and precondtioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, j, dim, vx_nr, hi_w, hi_nr, fath1, fath2, child, eg_w, eg_nr;
  FIDX level, level_old; 
  struct navsto_matrix *K;
  struct mesh *m;
  struct multilvl *ml;
  struct bpxdata *bpx;

  K=arg1;
  m=(*K).msh;
  ml=(*K).ml1;
  bpx=(*K).bpx;


  vx_nr= (*K).vx_nr;
  dim  = (*K).dim;
  eg_nr= (*m).eg_nr;
  eg_w = (*m).eg_w;
  hi_nr= (*m).hi_nr;
  hi_w = (*m).hi_w;

  if (((*in).len!=(*out).len)||((*in).len!=vx_nr))
    {
      fprintf(stderr,
	      "stokes_projector_part_bpx: dimensions make no sense!\n");
      return FAIL;
    }

  /* copy in to hvec */
  for (i=0; i<(*in).len; i++)
    {
      (*bpx).hvec[i]=(*in).V[i];
    }

  /* apply P^T */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*bpx).hvec[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* calculate r_l = Q_l^T r = [I P_l^T] r_(l+1), r_lmax=r for all
     levels l, r_l is stored in hvec[ nlevl[l+1]...nlevl[l]-1 ], */
  level_old=(*ml).lmax+1;
  for (i=hi_nr-1; i>=0; i--)
    {
      level=(*m).hier[i*hi_w+MCT2HILVL];
      if (level<=0) printf("error, level=%d\n", (int) level);
      if (level_old!=level)
	{
	  /* initialise the new (coarser) level */
	  for (j=(*ml).nlevl[level+1]; j<(*ml).nlevl[level]; j++)
	    {
	      /* get the entry number of the coarser level */
#ifdef USEOLDMLVL
	      fath2 = (*ml).hindx[j];
#else
	      fath2 = j-(*ml).nlevl[level+1];
	      if (fath2>=(*ml).nlevl[level-1]-(*ml).nlevl[level]) fath2=-1;
#endif
	      if (fath2!=-1)
		{
		  /* the I (identity) part of the multiplication,
		     r_l = r_(l+1) */
		  (*bpx).hvec[fath2+(*ml).nlevl[level]]=(*bpx).hvec[j];
		}
	    }
	  level_old=level;
	}

      child=(*m).hier[i*hi_w+MCT2HICHLD];
      fath1=(*m).hier[i*hi_w+MCT2HIFAT1  ];
      fath2=(*m).hier[i*hi_w+MCT2HIFAT1+1];      

      MLVLFINDENTRY(fath1, fath1, level-1, *ml);
      MLVLFINDENTRY(fath2, fath2, level-1, *ml);
      MLVLFINDENTRY(child, child, level,   *ml);

      /* the P_l^T part */
      (*bpx).hvec[fath1] += 0.5*(*bpx).hvec[child];
      (*bpx).hvec[fath2] += 0.5*(*bpx).hvec[child];
    }


  /* perform the boundary projection on all levels */
  for (i=0; i<(*K).bn_nr; i++)
    {
      FIDX node, nodel;
      node=(*K).nodes[i];
      for (j=0; j<=(*ml).lmax; j++)
	{
	  MLVLFINDENTRY(nodel, node, j, *ml);
	  if (nodel>=0)
	    {
	      (*bpx).hvec[nodel]=0.0;
	    }
	}
    }



  /* calculate w_l = [I ; P_l] w_(l-1) + D_l r_l, w_0=K_0^(-1) r_0 */
  /* initialise the coarsest level, 
     appy the coarse grid solver, if available,
     otherwise nothing has to be done in 2d as D_0 = I */
  if ((*bpx).cmat!=NULL) 
    {
      int  err;
      FIDX dof;
      struct vector rhs_x;

      /* here we suppose the coarse grid nodes have consecutive
	 numbers and start from 0 */
      dof=0;
      MLVLFINDENTRY(dof, dof, 0, *ml);

      rhs_x.V=&(*bpx).hvec[dof];
      rhs_x.len=(*ml).nlevl[0]-(*ml).nlevl[1];
      rhs_x.n_max=rhs_x.len;
      
      /* the dirichlet dofs are set to zero by the above projection */

      /* coarse grid solve */
      err=coarse_mat_solve( (*bpx).cmat, NoTrans, &rhs_x, &rhs_x);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve,
			      stokes_projector_part_bpx);

      /* the solution is in w_0 */
    }

  /* calculate w_l = [I ; P_l] w_(l-1) + D_l r_l, */
  level_old=0; 
  for (i=0; i<hi_nr; i++)
    {
      level=(*m).hier[i*hi_w+MCT2HILVL];

      if (level_old!=level)
	{
	  /* initialise the new (finer) level */
	  for (j=(*ml).nlevl[level+1]; j<(*ml).nlevl[level]; j++)
	    {
	      /* get the entry number of the coarser level */
	      fath2 = j-(*ml).nlevl[level+1];
	      if (fath2>=(*ml).nlevl[level-1]-(*ml).nlevl[level]) fath2=-1;
	      if (fath2!=-1)
		{
		  /* the I (identity) part of the multiplication,
		     w_l = w_(l-1) + D_l r_l,
		     D_l=I in 2d, r_l=hvec[j], so just add w_(l-1)
		  */
		  (*bpx).hvec[j]+=(*bpx).hvec[fath2+(*ml).nlevl[level]];
		}
	    }
	  level_old=level;
	}
      
      child=(*m).hier[i*hi_w+MCT2HICHLD];
      fath1=(*m).hier[i*hi_w+MCT2HIFAT1  ];
      fath2=(*m).hier[i*hi_w+MCT2HIFAT1+1];      
      
      MLVLFINDENTRY(fath1, fath1, level-1, *ml);
      MLVLFINDENTRY(fath2, fath2, level-1, *ml);
      MLVLFINDENTRY(child, child, level,   *ml);
      
      /* the P_(l-1) part */
      (*bpx).hvec[child]+=0.5*( (*bpx).hvec[fath1] + (*bpx).hvec[fath2] );
    }

  /* copy to out */
  for (i=0; i<vx_nr; i++)
    {
      (*out).V[i]=(*bpx).hvec[i];
    }


  /* apply P */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* done */

  return SUCCESS;
}


/*FUNCTION*/
int stokes_projector_part_MG(void *arg1, struct vector *in,
			     void *notused,
			     struct vector *out
/* performs a number of V cycles as (multigrid) preconditioning and
   the boudary condition projection for the subproblems in the
   preconditioner for Stokes problems,
                out = P*C^{-1}*P^T * in
   where C^{-1} is the preconditioner consisting of a number of
   multigrid V cycles
   
   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and mutigrid data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     "preconditioner"

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  int  err;
  FIDX i, j;
  FIDX dim, vx_nr, eh_w, eh_nr, eg_w, eg_nr;
  struct vector xi, dxi, bi, invdiagi;
  struct mesh *m;
  struct navsto_matrix *K;
  struct multilvl *ml;
  struct mgdata *mg;
  double *xl, *dxl, *bl, *invdiag;

  FIDX lvl, lmax, lvl_vx, smooths, vcycles, vccount;

  double normres, normb, stop_eps;

  double drow;

  K  = arg1;
  m  = (*K).msh;
  ml = (*K).ml1;
  mg = (*K).mg;

  xl   = (*mg).xl;
  dxl  = (*mg).dxl;
  bl   = (*mg).bl;
  invdiag = (*mg).invdiag;

  smooths  = (*mg).smooths;
  vcycles  = (*mg).vcycles;
  stop_eps = (*mg).stop_eps;

  vx_nr= (*K).vx_nr;
  dim  = (*K).dim;
  eg_nr= (*m).eg_nr;
  eg_w = (*m).eg_w;
  eh_nr= (*m).eh_nr;
  eh_w = (*m).eh_w;

  lmax=(*ml).lmax;

  if (lmax != K->lvl)
    {
      fprintf(stderr,
	      "stokes_projector_part_MG: lvl info contradicts!\n");
      return FAIL;
    }

  if (((*in).len!=(*out).len)||((*in).len!=vx_nr))
    {
      fprintf(stderr,
	      "stokes_projector_part_MG: dimensions make no sense!\n");
      return FAIL;
    }

  /* copy in to bl */
  for (i=0; i<(*in).len; i++)
    {
      bl[i]=(*in).V[i];
    }

  /* apply P^T */
  for (i=0; i<(*K).bn_nr; i++)
    {
      bl[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* apply C^-1 */

  /* set the vectors for multilpication with the stiffness matrix */
  dxi.V     = &drow;
  dxi.n_max = 1;
  dxi.len   = 1;

  if (vcycles==0)
    {
      fprintf(stderr, "stokes_projector_part_MG: vcycles==0 not "
	      "defined\n");
      return FAIL;
    }
  if (stop_eps<0.0)
    {
      fprintf(stderr, "stokes_projector_part_MG: stop_eps<0 not "
	      "sensible\n");
      return FAIL;
    }
 
  if (vcycles<0)
    {
      /* compute normb */
      normb=0.0;
      for (i=0; i<(*in).len; i++)
	{
	  normb+=bl[i]*bl[i];
	}
      normb=sqrt(normb);
    }
  else
    {
      normb=1.0;
      stop_eps=0.0;
    }
  /* make sure the process starts */
  normres=stop_eps*normb+1.0;

  /* perform at most vcycles V-cycles */
  for (vccount=0; (vccount<abs(vcycles))&&(normres>stop_eps*normb);
       vccount++)
    {
      /* V-cycle downward */
      for (lvl=lmax; lvl>=0; lvl--)
	{ 
	  lvl_vx=(*ml).nlevl[lvl]-(*ml).nlevl[lvl+1];

	  if ((vccount==0)||(lvl<lmax))
	    {
	      /* set xl=0 */
	      for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
		xl[j]=0.0;
	    }

	  /* make x_l, b_l, invdiag_l accessible as vector */
	  xi.len=lvl_vx;
	  xi.n_max=lvl_vx;
	  xi.V=&xl[(*ml).nlevl[lvl+1]];

	  bi.len=lvl_vx;
	  bi.n_max=lvl_vx;
	  bi.V=&bl[(*ml).nlevl[lvl+1]];

	  invdiagi.len=lvl_vx;
	  invdiagi.n_max=lvl_vx;
	  invdiagi.V=&invdiag[(*ml).nlevl[lvl+1]];

	  if ((lvl==0)&&((*(*K).bpx).cmat!=NULL))
	    {
	      /* use the coarse grid solver */
	      struct vector bcoarse;

	      /* apply the projector first */
	      for (i=0; i<(*K).bn_nr; i++)
		{
		  FIDX node, nodel;
		  node=(*K).nodes[i];
		  MLVLFINDENTRY(nodel, node, lvl, *ml);
		  if (nodel>=0)
		    {
		      bl[nodel]=0.0;
		    }
		}

	      bcoarse.len=lvl_vx;
	      bcoarse.n_max=lvl_vx;
	      bcoarse.V=&bl[(*ml).nlevl[lvl+1]];
	      err=coarse_mat_solve( (*(*K).bpx).cmat, NoTrans,
				    &bcoarse, &xi);
	      FUNCTION_FAILURE_HANDLE( err, coarse_mat_solve,
				       stokes_projector_part_MG);
	    }
	  else
	    {
	      /* do smooths Gauss-Seidel sweeps forward */
	      err=sparse_GS_sweep_fwd( &(*K).Fs[lvl], &bi,
				       &invdiagi, smooths, &xi);
	      FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_fwd,
				       stokes_projector_part_MG);

	      if (lvl==0)
		{
		  /* do smooths Gauss-Seidel sweeps backward */
		  err=sparse_GS_sweep_bwd( &(*K).Fs[lvl], &bi,
					   &invdiagi, smooths, &xi);
		  FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_bwd,
					   stokes_projector_part_MG);

		}
	    }

	  if (lvl>0)
	    {
	      /* compute the residual on this lvl */
	      /* compute the matrix vector product */
	      dxi.V   = &dxl[(*ml).nlevl[lvl+1]];
	      dxi.len = lvl_vx;
	      dxi.n_max = lvl_vx;
	      err=sparse_mul_mat_vec( &(*K).Fs[lvl], &xi, &dxi);
	      FUNCTION_FAILURE_HANDLE( err, sparse_row_tim_vec,
				       stokes_projector_part_MG);

	      /* now change dxl to rl=bl-A*xl */
	      for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
		dxl[j]=bl[j]-dxl[j];

	      /* restrict rl to r_l-1 */
	      err=mg_restrict_t2( mg, lvl, lvl-1, dxl);
	      FUNCTION_FAILURE_HANDLE( err, mg_restrict_2,
				       stokes_projector_part_MG);
	      /* copy r_l-1 to b_l-1 */
	      for (j=(*ml).nlevl[lvl]; j<(*ml).nlevl[lvl-1]; j++)
		{
		  bl[j]=dxl[j];
		}

	      dxi.V     = &drow;
	      dxi.n_max = 1;
	      dxi.len   = 1;
	    }
	} /* end V-cycle downward */

      /* V-cycle upward */
      for (lvl=1; lvl<=lmax; lvl++)
	{ 
	  lvl_vx=(*ml).nlevl[lvl]-(*ml).nlevl[lvl+1];
	  
	  /* apply the update computed in the lower level */
	  /* copy the update to dx */
	  for (j=(*ml).nlevl[lvl]; j<(*ml).nlevl[lvl-1]; j++)
	    dxl[j]=xl[j];
	  /* interpolate dx to lvl */
	  err=mg_interpolate_t2( mg, lvl-1, lvl, dxl);
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_t2,
				   stokes_projector_part_MG);

	  /* apply the update to xl */
	  for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
	    xl[j]+=dxl[j];

	  /* make x_l, b_l, invdiag_l accessible as vector */
	  xi.len=lvl_vx;
	  xi.n_max=lvl_vx;
	  xi.V=&xl[(*ml).nlevl[lvl+1]];

	  bi.len=lvl_vx;
	  bi.n_max=lvl_vx;
	  bi.V=&bl[(*ml).nlevl[lvl+1]];

	  invdiagi.len=lvl_vx;
	  invdiagi.n_max=lvl_vx;
	  invdiagi.V=&invdiag[(*ml).nlevl[lvl+1]];
	  
	  /* do smooths Gauss-Seidel sweeps backward */
	  err=sparse_GS_sweep_bwd( &(*K).Fs[lvl], &bi,
				   &invdiagi, smooths, &xi);
	  FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_bwd,
				   stokes_projector_part_MG);
	} /* end V-cycle upward */

      if (vcycles<0)
	{
	  /* compute the residual norm */
	  /* out=A*x */
	  err=sparse_mul_mat_vec( &(*K).Fs[lmax], &xi, out);
	  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec,
				   stokes_projector_part_MG);
	  /* out -= rhs = in */
	  for (i=0; i<(*out).len; i++)
	    {
	      (*out).V[i]-=bl[i];
	    }
	  /* apply P */
	  for (i=0; i<(*K).bn_nr; i++)
	    {
	      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
	    }
	  normres=0.0;
	  for (i=0; i<(*out).len; i++)
	    {
	      normres+=(*out).V[i]*(*out).V[i];
	    }
	  normres=sqrt(normres);
	}
    } /* end V-cycles loop */
  
  (*mg).vccount+=vccount;

  /* copy xl to out */
  for (i=0; i<(*out).len; i++)
    {
      (*out).V[i]=xl[i];
    }

  /* apply P */
  for (i=0; i<(*K).bn_nr; i++)
    {
      (*out).V[ (*K).nodes[i] ]   = 0.0 ;
    }

  /* done */

  return SUCCESS;

}


/*FUNCTION*/
int stokes_projector_w_precon(void *arg1, struct vector *in,
			      void *notused,
			      struct vector *out
/* performs preconditioning and the boudary condition projection for
   stokes problems, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there) and projects the pressure such that it has zero
   mean 
   
   C^-1 is the Wathen block preconditioner:
        
   C^-1=[I           ] [  I          ] [A^-1        ]  velocity space 1
        [  I         ]*[       I     ]*[     A^-1   ]  velocity space 2
        [    nu*Mp^-1] ]-B_1 -B_2  I ] [           I]  pressure space

   where A is the stiffness matrix of the Laplace operator, nu is 1/Re
   and Mp is the pressure space mass matrix, 

   Input:  arg1=
           K       - navsto_matrix struct holding the boundary
                     projection and preconditioner data 
	   in      - input vector
	   notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner

   Output: out    - (given by reference), P*C^-1*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  int  err, iter;
  FIDX i, j, dim, lvl, vx_nr, pvx_nr;
  double pmean, sumw, resi;
  struct vector rhs, xi;
  struct navsto_matrix *K;

  K=arg1;

  lvl    = (*K).lvl;
  vx_nr  = (*K).vx_nr;
  pvx_nr = (*K).pvx_nr;
  dim    = (*K).dim;
  sumw   = 0.0;

  if (((*in).len!=(*out).len)||((*in).len<dim*vx_nr+(*K).pvx_nr))
    {
      fprintf(stderr,
	      "stokes_projector_w_precon: dimensions make no sense!\n");
      return FAIL;
    }

  err=vector_alloc( &rhs, vx_nr);
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, stokes_projector_w_precon);

  /* velocity components */
  for (j=0; j<dim; j++)
    {
      /* apply P^T, rhs = P^T*in  */
      /* copy in to rhs */
      for (i=0; i<vx_nr; i++)
	{
	  rhs.V[i]=(*in).V[j*vx_nr+i];
	}
      for (i=0; i<(*K).bn_nr; i++)
	{
	  rhs.V[ (*K).nodes[i] ]   = 0.0 ;
	}

      /* prepare to solve F_i out = rhs */
      xi.len=vx_nr;
      xi.n_max=vx_nr;
      xi.V=&(*out).V[j*vx_nr];
      /* prepare the coarse grid data */
      if ((*K).cmat.nr!=0)
	{
	  (*(*K).bpx).cmat=&(*K).cmat;
	}

      /* solve F_i out = rhs */
      /***************************************************************/
      (*(*K).mg).vccount=0; 
      err= stokes_projector_part_MG( K, &rhs,  NULL, &xi);
      iter=(*(*K).mg).vccount; 
      if (((*(*K).mg).vcycles<0)&&((*(*K).mg).vccount>0))
	(*(*K).mg).vcycles=(*(*K).mg).vccount; /* */
      /***************************************************************/
      /* if ( K->innersteps[0]<0)
	 {
	 err=PCG( 10000, 2, 0.0, (*K).innereps, 0, &xi, &resi, &iter,
	 stokes_matrix_part_tim_vec, stokes_projector_part_bpx,
	 K, &rhs, NULL);
	 K->innersteps[0]=iter;
	 }
	 else
	 {
	 err=PCG( K->innersteps[0], 1, 0.0, 0.0, 0, &xi, &resi, &iter,
	 stokes_matrix_part_tim_vec, stokes_projector_part_bpx,
	 K, &rhs, NULL);
	 if (err==10) err=SUCCESS;
	 } /* */
      /***************************************************************/
      /* err=PCG( 6, 2, 0.0, 0.0, 0, &xi, &resi, &iter,
	 sparse_mul_mat_vec, stokes_projector_part_MG,
	 &(*K).Fs[lvl], &rhs, P); /* */
      /***************************************************************/
      /* err=PCG( 10000, 2, 0.0, (*K).innereps, 0, &xi, &resi, &iter,
	 sparse_mul_mat_vec, stokes_projector_part_MG,
	 &(*K).Fs[lvl], &rhs, P); /* */
      /***************************************************************/
      /* err=PCG( 10, 2, 0.0, 0.0, 0, &xi, &resi, &iter,
	 sparse_mul_mat_vec, stokes_projector_part_bpx,
	 &(*K).Fs[lvl], &rhs, P); /* */
      /***************************************************************/
      /* err=PCG( 10000, 2, 0.0, (*K).innereps, 0, &xi, &resi, &iter,
	 sparse_mul_mat_vec, stokes_projector_part_bpx,
	 &(*K).Fs[lvl], &rhs, P); /* */
      /***************************************************************/
      /* if (err==10) err=SUCCESS; /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, stokes_projector_w_precon);
      (*K).innercount[j]+=iter;

      /* revert the change to the bpx.cmat */
      (*(*K).bpx).cmat=NULL;
      
      /* apply P, out = P*out  */
      for (i=0; i<(*K).bn_nr; i++)
	{
	  (*out).V[ j*vx_nr+ (*K).nodes[i] ]   = 0.0 ;
	}
    }


  /* pressure components */
  /* apply P^T, rhs = -P^T*in  */
  /* copy in to rhs */
  for (i=0; i<pvx_nr; i++)
    {
      rhs.V[i]=- (*in).V[dim*vx_nr+i];
    }
  rhs.len=pvx_nr;
  /* get the summ of all pressure components */
  pmean=0.0;
  for (i=0; i<vx_nr; i++)
    {
      sumw+= (*K).weight[i];
      if ((*K).pdof[i]!=-1)
	{
	  pmean+= rhs.V[ (*K).pdof[i] ];
	}
    }

  /* subtract weight[i] times the pressure summ */
  for (i=0; i<vx_nr; i++)
    {
      if ((*K).pdof[i]!=-1)
	{
	  rhs.V[(*K).pdof[i] ] -= pmean * (*K).weight[i];
	}
    }


  /* P^T done, now C^{-1} */

  /* rhs=[B -I]*[out.vel; rsh] */
  for (j=0; j<dim; j++)
    {
      xi.len=vx_nr;
      xi.n_max=vx_nr;
      xi.V=&(*out).V[j*vx_nr];
      err=sparse_mul_mat_vec_add( &(*K).Bs[j], &xi, &rhs);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec_add,
			       stokes_projector_w_precon);
    }
  /* rhs=-rhs (= [-B P^T]*[out.vel; in.p]) */
  for (i=0; i<vx_nr; i++)
    {
      if ((*K).pdof[i]!=-1)
	{
	  rhs.V[(*K).pdof[i] ]= -rhs.V[(*K).pdof[i] ];
	}
    }
  
  /* solve M out = rhs */
  xi.len=pvx_nr;
  xi.n_max=pvx_nr;
  xi.V=&(*out).V[dim*vx_nr];
  if ( K->innersteps[dim]<0 )
    {
      err=PCG( 10000, 2, 0.0, (*K).innereps, 0, &xi, &resi, &iter,
	       sparse_mul_mat_vec, NULL, &(*K).M, &rhs, NULL);
      FUNCTION_FAILURE_HANDLE( err, PCG, stokes_projector_w_precon);
      K->innersteps[dim]=iter;
    }
  else
    {
      err=PCG( K->innersteps[dim], 0, 0.0, 0.0, 0, &xi, &resi, &iter,
	       sparse_mul_mat_vec, NULL, &(*K).M, &rhs, NULL);
      if (err==10) err=SUCCESS;
      FUNCTION_FAILURE_HANDLE( err, PCG, stokes_projector_w_precon);
    }
  (*K).innercount[dim]+=iter;

  /* C^{-1} done, now P */
  /* get the mean pressure */
  pmean=0.0;
  for (i=0; i<vx_nr; i++)
    {
      if ((*K).pdof[i]!=-1)
	{
	  pmean+= (*out).V[dim*vx_nr + (*K).pdof[i] ] * (*K).weight[i];
	}
    }

  /* subtract the mean pressure */
  for (i=0; i<vx_nr; i++)
    {
      if ((*K).pdof[i]!=-1)
	{
	  (*out).V[dim*vx_nr + (*K).pdof[i] ] -= pmean;
	}
    }

  /* done */
  vector_free( &rhs );

  return SUCCESS;
}


/*FUNCTION*/
int stokes_matrix_tim_vec(void *arg1, struct vector *vec,
			  struct vector *out
/* multiplies the sparse matrix  K from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		     ){
  struct vector x, y;
  struct navsto_matrix *K;

  int  err;
  FIDX i, dim, lvl, vx_nr, pvx_nr;

  K=arg1;
  dim    = (*K).dim;
  lvl    = (*K).lvl;
  vx_nr  = (*K).vx_nr;
  pvx_nr = (*K).pvx_nr;

  for (i=0; i<dim; i++)
    {
      /* out_i = F_i * vec_i  +  B_i^T * vec_p */
      x.len   = vx_nr;
      x.n_max = vx_nr;
      x.V     = &(*vec).V[i*vx_nr];

      y.len   = vx_nr;
      y.n_max = vx_nr;
      y.V     = &(*out).V[i*vx_nr];

      err=sparse_mul_mat_vec( &(*K).Fs[lvl], &x, &y);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, 
			       stokes_matrix_tim_vec);

      x.len   = pvx_nr;
      x.n_max = pvx_nr;
      x.V     = &(*vec).V[dim*vx_nr];

      err=sparse_mul_mat_vec_add_trans( &(*K).Bs[i], &x, &y);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec_add_trans, 
			       stokes_matrix_tim_vec);

    }
  
  /* out_p =  sum (B_i * vec_i) */

  /* i=0 */
  x.len   = vx_nr;
  x.n_max = vx_nr;
  x.V     = &(*vec).V[0];

  y.len   = pvx_nr;
  y.n_max = pvx_nr;
  y.V     = &(*out).V[dim*vx_nr];

  err=sparse_mul_mat_vec( &(*K).Bs[0], &x, &y);
  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, 
			   stokes_matrix_tim_vec);


  /* i=1..dim-1 */
  for (i=1; i<dim; i++)
    {
      x.len   = vx_nr;
      x.n_max = vx_nr;
      x.V     = &(*vec).V[i*vx_nr];

      err=sparse_mul_mat_vec_add( &(*K).Bs[i], &x, &y);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, 
			       stokes_matrix_tim_vec);
    }

  return SUCCESS;
}

/*FUNCTION*/
int stokes_matrix_part_tim_vec(void *arg1, struct vector *vec,
			       struct vector *out
/* multiplies the sparse matrix K.Fs[lvl] from left to the vector vec,
   
   out = K * vec;

   Input:  arg1=
           K       - Stokes-problem stiffness matrix
           vec     - vector

   Output: out     - resulting vector

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		     ){
  struct navsto_matrix *K;

  int  err;
  FIDX lvl;

  K=arg1;
  lvl    = (*K).lvl;

  err=sparse_mul_mat_vec( &(*K).Fs[lvl], vec, out);
  FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, 
			   stokes_matrix_part_tim_vec);

  return SUCCESS;
}


/*FUNCTION*/
int stokes_write_streamfunc_t2( struct mesh *m, struct vector *usol,
				FIDX namlen, char *name
/* recovers the stream function of a 2d velocity field and writes it
   with the mesh into a NAG IRIS EXPLORER readable file (for
   visualisation)  
   
   Input:  m         - the mesh
           usol      - vector with containing the velocity field,
                       stored as usol[j*vx_nr+i] gives the j-th
                       velocity component in the i-th node of the mesh
	   namlen    - maximal useable length of name
	   name      - basename of the files, one will be
	               <name>.pyr and the other <name>.lat

   Output: (writes the files)

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
				){
  int err, iter;
  double resi;
  struct sparse K;
  struct vector rhs, strf;
  struct projector1 P;

  if ((*m).dim!=2)
    {
      fprintf(stderr,"stokes_write_streamfunc_t21: only for 2d!\n");
      return FAIL;
    }

  err=sparse_alloc( &K, (*m).vx_nr, 32);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, stokes_write_streamfunc_t21);
  err=vector_alloc( &rhs, (*m).vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, stokes_write_streamfunc_t21);
  err=vector_alloc( &strf, (*m).vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, stokes_write_streamfunc_t21);
  err=projector1_alloc( &P, (*m).vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, stokes_write_streamfunc_t21);

  err=stokes_ass_streamfunc_poison_t2( &K, &rhs, &strf, &P, m, usol );
  FUNCTION_FAILURE_HANDLE( err, stokes_stream_func_poison_t2,
			   stokes_write_streamfunc_t21);

  err=PCG( 10000, 1, 1e-7, 0.0, 1, &strf, &resi, &iter, sparse_mul_mat_vec, 
	   projector1_no_precon, &K, &rhs, &P );
  FUNCTION_FAILURE_HANDLE( err, PCG, stokes_write_streamfunc_t21);
  printf(" psi_PCG: it=%3d  ", iter); /* */

  err=mesh_write_solution_exp_t2( m, &strf, 1, namlen, name );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
			   stokes_write_streamfunc_t21);

  sparse_free(&K);
  vector_free(&rhs);
  vector_free(&strf);
  projector1_free(&P);

  return SUCCESS;
}

