/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE navsto_solver.c
HEADER navsto_solver.h

TO_HEADER:

*/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "sparse.h"
#include "lin_solver.h"

#include "navsto_struct.h"
#include "stokes_aux.h"
#include "navsto_aux.h"
#include "navsto_adj.h"
#include "navstoassem.h"

#include "assembly.h"
#include "mesh_deform.h"

#include <unistd.h>
#include <string.h>

  /* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>


#define VERBOSE 4

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }

/*FUNCTION*/
int navsto_solver_bcontrol(struct mesh *msh1, FIDX lvl_max,
			   struct vector *ref_sol,
			   struct vector *pccrit,
			   struct vector *DIDB
/* the outer solver loop, controlls refinement, Reynolds number besed
   on the mesh Peclet number, mesh deformation, adjoint solve
   
   goal is to solve the discrete Navier-Stokes equations on uniformly
   refined T2 versions of the T1 mesh msh1, evaluate the
   performance criteria, solve the adjoint equations and evaluate the
   derivatives wrt to Dirichlet boundary data

   Input:  msh1    - the coarse mesh (T1),  hierarchy has to be reset,
                     i.e. msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1
		     for all j
           lvl_max - the desired refinement level
           
	   ref_sol - reference solution on the finest mesh, an error
	             is returned if it is of wrong size,
		     this is passed to navsto_dIdX_t2

   Output: pccrit  - vector of performance criteria, length
                     msh1.pc_nr, if pccrit==NULL it is not used
	   DIDB    - array of msh1.pc_nr vectors of length ??,
	             containing the derivatives of the pccrit wrt the
	             Dirichlet data, not used if 
	             DIDB==NULL, the DIDB[i] are initialised here
	             otherwise

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid */
		  ){
  struct navsto_matrix K;
  struct vector rhs_stokes, rhs, x, x_old, y, z, sol, soli;
  struct mesh msh2;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t1, t2, t3, ti, tstart, tsolve, tadj;
  FILE *mark;

  char solver[15];
  FIDX dim, vx_nr, i, j, d, step, linmeth, nu_chng;
  int  iter, err;

  FIDX lvl, lvl_min;

  double resi, resnl_old, resnl, normy, deltax;
  double leps, nleps, nleps_coarse, nleps_final, nleps_oldresnl;

  double impeps, innereps;
  double adjeps, adjinnereps;
  
  double nu, nu0, nu_akt, dnu, maxPe;


  int m=100;

  int use_SDFEM=0;

  if (use_SDFEM) { printf("using SDFEM and box-smoother\n"); m = 200; } /* */

  TIMEGET;
  tstart=ti;


  nu=msh1->para[MC2XPANUPO];

  /* set the goals and methods */
  linmeth = 1;

  /* #warning "linmeth=2!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"
     linmeth = 2; /* */

  nleps_coarse   = 1e-1;
  nleps_final    = 1e-6;
  nleps_oldresnl = 1e-8;
  impeps   = 1e-5;
  innereps = 1e-6;

  adjeps      = 1e-6; 
  adjinnereps = 1e-6;
  //  fprintf(stderr,"for tests: adjeps=1e-5\n"); adjeps      = 1e-5; 

  dnu     = 0.5;
  dim=msh1->dim;


  /* null define msh2 */
  mesh_init( &msh2);

  /* init the stiffness matrix */
  err=navsto_matrix_init( &K, dim, lvl_max+1);
  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_init, navsto_solver_bcontrol);

  
  /* start from scratch */
  lvl_min=0;

  for (lvl=lvl_min; lvl<=lvl_max; lvl++)
    {

      if (lvl==lvl_min)
	{
	  /* initial assem init */
	  if (!use_SDFEM)
	    {
	      /* get the maximal stable Re on the coarse mesh */
	      err=navsto_maxnu_init_t21(msh1, &nu0);
	      FUNCTION_FAILURE_HANDLE(err, navsto_maxnu_init_t21,
				      navsto_solver_bcontrol);
	      
	      nu_akt=fmax(nu, nu0); 
	      msh1->para[MC2XPANUPO]=nu_akt; 
	      if (VERBOSE>0)
		{
		  printf("nu=%8.1e     nu0=%8.1e     nu_akt=%8.1e\n",
			 nu, nu0, nu_akt);
		}
	      nu_chng=0;

	      err=navsto_MG_init_assem_t21( msh1, lvl, &K, &rhs_stokes,
					    &rhs, &x, &msh2,
					    NULL, 0 );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_init_assem_t21,
				       navsto_solver_bcontrol);

	    }
	  else
	    {
	      nu_akt  = nu;
	      err=navsto_MG_init_assem_SDFEM_t21( msh1, lvl, &K, &rhs,
						  &x, &msh2,
						  NULL, 0 );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_init_assem_SDFEM_t21,
				       navsto_solver_bcontrol);
	      
	      err=vector_alloc( &rhs_stokes, x.len );
	      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
	      for (i=0; i<rhs_stokes.len; i++) { rhs_stokes.V[i]=0.0; }
	    }
	  vx_nr=msh2.vx_nr;

	  err=vector_alloc( &x_old, x.len );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
	}
      else /* if (lvl!=lvl_min) */
	{
	  /* copy the old solution, used for the interpolation */
	  for (i=0; i<x.len; i++)
	    { x_old.V[i]=x.V[i];}

	  /* update nu */
	  nu_akt=fmax(nu, maxPe*1.3*nu_akt*dnu);
	  /* #warning "higher Pe"
	     nu_akt=fmax(nu, 0.125*maxPe*1.3*nu_akt*dnu); /* */
	  if (msh1->para[MC2XPANUPO]!=nu_akt)
	    {
	      msh1->para[MC2XPANUPO]=nu_akt; 
	      nu_chng=1;
	    }

	  /* refine, init on finer mesh, with larger Re */
	  err=navsto_MG_ref_intp_ass_t21( msh1, lvl, &msh2, NULL, 0,
					  &K, &rhs_stokes, &rhs,
					  &x_old, &x );
	  FUNCTION_FAILURE_HANDLE( err, navsto_MG_ref_intp_ass_t21,
				   navsto_solver_bcontrol); 

	  vx_nr=msh2.vx_nr;
	} /* end if (lvl>0) */
    
      err=vector_alloc( &y, x.len );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
      err=vector_alloc( &z, x.len );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
      err=vector_alloc( &sol, (dim+1)*K.vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);

      if (lvl==lvl_max)
	nleps=nleps_final;
      else
	nleps=nleps_coarse;

      printf("nleps=%e\n",nleps);

      /* the nonlinear loop */
      resi=1.0; resnl=1.0; deltax=1.0; step=0; resnl_old=1e300;
      while (((deltax>nleps)||(resi>nleps))
	     &&((resnl>nleps_oldresnl)&&(step<100)))
	{
	  /* save the old x for the computation of deltax */
	  for (i=0; i<x.len; i++)
	    { x_old.V[i]=x.V[i]; }

	  TIMEGET;
	  t1=ti;

	  if (!use_SDFEM)
	    {
	      err=navsto_MG_C_assem_t21( msh1, lvl, &K, &rhs, &maxPe, 
					 &rhs_stokes, &x, &msh2, 
					 NULL, 0,
					 linmeth );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_C_assem_t21,
				       navsto_solver_bcontrol);
	    }
	  else
	    {      
	      err=navsto_MG_assem_SDFEM_t21( msh1, lvl, &K, &rhs, &x, 
					     &msh2, NULL, 0 );
	      FUNCTION_FAILURE_HANDLE( err, navsto_assem_SDFEM_t21,
				       navsto_solver_bcontrol);
	      maxPe=0.0;
	    }

	  TIMEGET;
	  t2=ti;

	  /* test the residual */
	  /* y=K*x */
	  err=navsto_matrix_tim_vec( &K, &x, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec,
				   navsto_solver_bcontrol);  
	  /* z=K*x-rhs, get norm(y) */
	  for (i=0; i<x.len; i++)
	    {
	      z.V[i]=y.V[i]-rhs.V[i];
	    }
	  /* apply the projector */
	  err=navsto_projector_no_precon( &K, &z, NULL, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon, 
				   navsto_solver_bcontrol);  
	  normy=0.0;
	  for (i=0; i<x.len; i++)
	    {
	      normy+= y.V[i]*y.V[i];
	    }
	  resnl=sqrt(normy); /* */
	  
	  if (VERBOSE>3)
	    {
	      printf("maxPe= %8.1e ", maxPe);
	      printf("|res_nl|=%12.6e, ", resnl); /* */
	      printf("t_a=%8.2e ", t2-t1); /* */
	      printf("\n"); /* */
	    }

	  /* before we solve, output the matrices to files (for
	     external analysis */
	  /* { 
	     char matname[50];
	     sprintf(matname,  "visual/savemat_%d",lvl); 
	     err=navsto_matrices_write_files( &K, matname);
	     FUNCTION_FAILURE_HANDLE(err, navsto_matrices_write_files,
	     navsto_solver_bcontrol);
	     }/* */


	  resi=0.0;
	  iter=0;

	  /* set stoping criterion for linear solver */
	  leps=fmax(impeps, 1e-2 * nleps_oldresnl/ resnl); 

	  if (!use_SDFEM)
	    {
	      K.innereps=innereps;
	      K.mg->vcycles=1;
	      K.mg->smooths=1;
	      K.mg->stop_eps=0.0;

	      strcpy(solver, "GMRES_Fp");
	      err=GMRES( m, 100, 2, 0.0, leps, 1, &x, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_projector_w_precon,
			 &K, &rhs, NULL ); /* */
	    }
	  else
	    {
	      strcpy(solver, "GMRES_mg");
	      err=GMRES( m, 100, 2, 0.0, leps, 1, &x, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_mg_SDFEM_precon,
			 &K, &rhs, NULL ); /* */
	    }
	  FUNCTION_FAILURE_HANDLE(err, GMRES, navsto_solver_bcontrol);
	  

	  TIMEGET;
	  t3=ti;

	  /* test the residual */
	  /* y=K*x */
	  err=navsto_matrix_tim_vec( &K, &x, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				   navsto_solver_bcontrol);  
	  /* z=K*x-rhs, get norm(y) */
	  normy=0.0;
	  for (i=0; i<x.len; i++)
	    {
	      z.V[i]=y.V[i]-rhs.V[i];
	    }
	  /* apply the projector */
	  err=navsto_projector_no_precon( &K, &z, NULL, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon, 
				   navsto_solver_bcontrol);  
	  for (i=0; i<x.len; i++)
	    {
	      normy+= y.V[i]*y.V[i];
	    }
	  resi=sqrt(normy); /* */


	  /* compute deltax */
	  deltax=0.0;
	  for (i=0; i<x.len; i++)
	    { deltax+=(x_old.V[i]-x.V[i])*(x_old.V[i]-x.V[i]); }
	  deltax=sqrt(deltax);

	  /* increase the step counter */
	  step++;

	  if ((VERBOSE>3)
	      ||((VERBOSE>1)
		  &&( (deltax<=nleps)&&(resi<=nleps) ) ))
	    {
	      printf("nu %8.1e st %3d  %8s: N=%6d, it=%4d, "
		     "i_it: %4d %4d %4d %4d err=%2d, |res_l|=%12.6e, "
		     "|dx|=%9.3e, ",
		     nu_akt, (int) step, solver, (int) x.len, iter,
		     K.innercount[0], K.innercount[1],
		     K.innercount[2], K.innercount[3],
		     err, resi, deltax); /* */
	      printf("t_s=%8.2e ", t3-t2); /* */
	    }

	  if (VERBOSE>-1) /* write the solution for visualization */
	    {
	      for (i=0; i<dim*vx_nr; i++)
		{ sol.V[i]=x.V[i]; }
	      for (i=0; i<vx_nr; i++)
		{ sol.V[dim*vx_nr+i]=0.0; }
	      for (i=0; i<vx_nr; i++)
		{ 
		  if (K.pdof[i]!=-1)
		    { sol.V[dim*vx_nr+i]= x.V[dim*vx_nr+K.pdof[i]]; }
		}
	      /* correct the pressure values of the edge-mid-nodes */
	      {
		FIDX eg, nod1, nod2, nodm;
		for (eg=0; eg<msh2.eg_nr; eg++)
		  {
		    if (msh2.edge[eg*msh2.eg_w+MCT2EGCHL1]==-1)
		      {
			nod1=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1  ];
			nod2=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+1];
			nodm=msh2.edge[eg*msh2.eg_w+MCT2EGNODM  ];
		    
			sol.V[2*msh2.vx_nr+nodm]= 0.5 * 
			  (sol.V[2*msh2.vx_nr+nod1] 
			   + sol.V[2*msh2.vx_nr+nod2]); 
		      }
		  }
	      }
	      /* write the solution with grape */
	      soli.V=&sol.V[0];
	      soli.len=(dim+1)*vx_nr;
	      soli.n_max=(dim+1)*vx_nr;
	      err=mesh_write_solution_grape_t2( &msh2, &soli, dim+1, 22,
					      "visual/2d_mesh_grape_" );
	      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_grape_t2,
				       navsto_solver_bcontrol); /* */

	      /* write the velocity components */
	      soli.V=&sol.V[0];
	      soli.len=dim*vx_nr;
	      soli.n_max=dim*vx_nr;
	      err=mesh_write_solution_exp_t2( &msh2, &soli, dim, 20,
					      "visual/2d_mesh_v" );
	      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
				       navsto_solver_bcontrol); /* */
	  
	      /* write the streamfunction */
	      /* err=stokes_write_streamfunc_t2( &msh2, &soli, 20,
		 "visual/2d_mesh_psi" );
		 FUNCTION_FAILURE_HANDLE( err, stokes_write_streamfunc_t2,
		 navsto_solver_bcontrol); /* */
	  
	      /* write the solution values for comparison */
	      {
		char fname[300];
		sprintf(fname, "visual/line.x.%d.txt", (int) lvl);
		err=navsto_write_line_u_t2( &msh2, &soli, 0, 0.5, fname );
		FUNCTION_FAILURE_HANDLE( err,
					 mesh_write_line_u_t2,
					 navsto_solver_bcontrol); 
		sprintf(fname, "visual/line.y.%d.txt", (int) lvl);
		err=navsto_write_line_u_t2( &msh2, &soli, 1, 0.5,
					    fname ); 
		FUNCTION_FAILURE_HANDLE( err,
					 mesh_write_line_u_t2,
					 navsto_solver_bcontrol);
	      } /* */ 
	  
	      /* write the pressure components */
	      soli.V=&sol.V[dim*vx_nr];
	      soli.len=vx_nr;
	      soli.n_max=vx_nr;
	      err=mesh_write_solution_exp_t2( &msh2, &soli, 1, 20,
					      "visual/2d_mesh_p" );
	      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
				       navsto_solver_bcontrol); /* */

	      /* write the magnitude of the velocity  */
	      for (i=0; i<vx_nr;i++)
		{
		  double normi=0.0;
		  for (j=0; j<dim; j++)
		    normi+=sol.V[i+j*vx_nr]*sol.V[i+j*vx_nr];
		  sol.V[i]=sqrt(normi);
		}
	      soli.V=&sol.V[0];
	      soli.len=vx_nr;
	      soli.n_max=vx_nr;
	      err=mesh_write_solution_exp_t2( &msh2, &soli, 1, 20,
					      "visual/2d_mesh_absv" );
	      FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
				       navsto_solver_bcontrol); /* */

	      /* touch the marker file */
	      mark=fopen("visual/2d_mesh_mark", "w");
	      fprintf(mark, "next please!\n");
	      fclose(mark);
	    }
	}


      /* final check of nonlinear convergence */
      TIMEGET;
      t1=ti;
      
      if (!use_SDFEM)
	{
	  err=navsto_MG_C_assem_t21( msh1, lvl, &K, &rhs, &maxPe,
				     &rhs_stokes,&x, &msh2,
				     NULL, 0, linmeth );
	  FUNCTION_FAILURE_HANDLE( err, navsto_MG_C_assem_t21,
				   navsto_solver_bcontrol);
	}
      else
	{
	  err=navsto_MG_assem_SDFEM_t21( msh1, lvl, &K, &rhs, &x,
					 &msh2, NULL, 0 );
	  FUNCTION_FAILURE_HANDLE( err, navsto_assem_SDFEM_t21, 
				   navsto_solver_bcontrol);
	}
  
      TIMEGET;
      t2=ti;

      /* test the residual */
      /* y=K*x */
      err=navsto_matrix_tim_vec( &K, &x, &y);
      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, navsto_solver_bcontrol);  
      /* z=K*x-rhs, get norm(y) */
      normy=0.0;
      for (i=0; i<x.len; i++)
	{
	  z.V[i]=y.V[i]-rhs.V[i];
	}
      /* apply the projector */
      err=navsto_projector_no_precon( &K, &z, NULL, &y);
      FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon,
			       navsto_solver_bcontrol);  
      for (i=0; i<x.len; i++)
	{
	  normy+= y.V[i]*y.V[i];
	}
      normy=sqrt(normy); /* */

      if (VERBOSE>3)
	{
	  printf("maxPe= %8.1e ", maxPe);
	  printf("|res_nl|=%12.6e, ", normy); /* */
	  printf("t_a=%8.2e ", t2-t1); /* */
	  printf("\n");
	}

      vector_free(&y);
      vector_free(&z);
      vector_free(&sol);

      /* do the postprocessing */ 
      if ((pccrit!=NULL)&&(lvl==lvl_max))
	{
	  err=navsto_dIdX_t21( &msh2, &K, &x, msh2.pc_nr,
			       pccrit, NULL, NULL, ref_sol);
	  FUNCTION_FAILURE_HANDLE( err, navsto_dIdX_t21, navsto_solver_bcontrol);  

	  if (VERBOSE>2)
	    {
	      printf("\nperformance criteria eval: err=%d\n",err);
	      for (i=0; i<msh2.pc_nr; i++)
		{
		  printf("crit %d: %e\n", (int) i, pccrit->V[i]);
		}
	    }
	}

    } /* end mesh refinement + interpolation */




  /*********************************************************/
  /*                                                       */
  /* navier-stokes system is now solved on the finest mesh */
  /*                                                       */
  /*********************************************************/




  TIMEGET;
  tsolve=ti;

  /* if required, do the adjoint solve */
  if ((DIDB != NULL)&&(msh1->pc_nr>0))
    {

      FIDX tc;
      struct vector *DIDuB, *dIdu, *dIduI, *dIduB;
      struct vector Psi, y;
      
      FIDX bc_sections;

      /* count the number of quadratic Dirichlet function sections */
      bc_sections=0;
      for (i=0; i<msh2.fu_nr; i++)
	{
	  if (msh2.func[i*msh2.fu_w+MC2XFUTYPE]==102) bc_sections++;
	}

      TRY_MALLOC(DIDuB, msh2.pc_nr, struct vector, navsto_solver_bcontrol);
      TRY_MALLOC(dIdu, msh2.pc_nr, struct vector, navsto_solver_bcontrol);
      TRY_MALLOC(dIduI, msh2.pc_nr, struct vector, navsto_solver_bcontrol);
      TRY_MALLOC(dIduB, msh2.pc_nr, struct vector, navsto_solver_bcontrol);

      for (tc=0; tc<msh2.pc_nr; tc++)
	{
	  err=vector_alloc( &DIDuB[tc], x.len); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);

	  err=vector_alloc( &DIDB[tc], bc_sections); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);

	  err=vector_alloc( &dIdu[tc], x.len); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);

	  err=vector_alloc( &dIduI[tc], x.len); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);

	  err=vector_alloc( &dIduB[tc], x.len); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
	}

      err=navsto_dIdX_t21( &msh2, &K, &x, msh2.pc_nr,
			   pccrit, dIdu, NULL, ref_sol);
      FUNCTION_FAILURE_HANDLE( err, navsto_dIdX_t21, navsto_solver_bcontrol);

      /* transpose the system matrix */
      err=navsto_adj_convert( &K);
      FUNCTION_FAILURE_HANDLE( err, navsto_adj_convert, navsto_solver_bcontrol);

      /* allocate the adjoint solution vector Psi and a help vector y */
      err=vector_alloc( &Psi, x.len);
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);
      err=vector_alloc( &y, x.len);
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver_bcontrol);


      /* prepare to solve */
      K.innereps=adjinnereps;
      K.mg->vcycles=1;
      K.mg->smooths=1;
      K.mg->stop_eps=0.0;

      
      /* for each criterion */
      for (tc=0; tc<msh2.pc_nr; tc++)
	{
	  /* copy dIdu to dIduI, set dIduB to zero */
	  for (i=0; i<dIdu[tc].len; i++)
	    {
	      dIduI[tc].V[i] = dIdu[tc].V[i];
	      dIduB[tc].V[i] = 0.0;
	    }

	  /* split dIdu into dIduI and dIduB */
	  for  (i=0; i<K.bn_nr; i++)
	    {
	      FIDX bcnode;

	      bcnode= K.nodes[i];

	      for (j=0; j<dim; j++)
		{
		  FIDX dof=bcnode+j*vx_nr;

		  dIduB[tc].V[dof] = dIduI[tc].V[dof];
		  dIduI[tc].V[dof] = 0.0;
		}	      
	    }

	  double norm_dIduI, norm_adjres;
	  /* compute the norm of the adjoint rhs */
	  normy=0.0;
	  for (i=0; i<dIduI[tc].len; i++)
	    normy += dIduI[tc].V[i] * dIduI[tc].V[i];
	  norm_dIduI=sqrt(normy);

	  if (norm_dIduI!=0.0)
	    {
	      /* actualy need to solve */

	      /* reset iteration counters */
	      for (i=0; i<5; i++) K.innercount[i]=0;

	      /* solve the adjoint system */
#ifdef NEW_ADJOINT
	      /* new adjoint handling: same preconditioner as for
		 forward solve, but do a projection step first */
	      {
		double Psi_lam;

		Psi_lam=0.0;
		for (i=0; i<vx_nr; i++)
		  {
		    if (K.pdof[i]!=-1)
		      {
			Psi_lam += dIduI[tc].V[dim*vx_nr+ K.pdof[i] ];
		      }
		  }

		printf("Psi_lambda=%e\n", Psi_lam);

		for (i=0; i<vx_nr; i++)
		  {
		    if ( K.pdof[i]!=-1)
		      {
			dIduI[tc].V[ dim*vx_nr+K.pdof[i] ] -= 
			  Psi_lam * K.weight[i] ;
		      }
		  }

		if (Psi_lam > 1e-8)
		  {
		    fprintf(stderr,"navsto_solver_bcontrol: "
			    "current adjoint handling can not deal\n"
			    "with e*dI/dp > 0, something wrong with\n"
			    "the criterion?\n");
		    return FAIL;
		  }
	      }

	      err=GMRES( m, 100, 2, 0.0, adjeps, 0, &Psi, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_projector_w_precon,
			 &K, &dIduI[tc], NULL ); /* */ 
#else
	      /* old adjoint handling, special multiplication and
		 preconditioner routines */
	      err=GMRES( m, 100, 2, 0.0, adjeps, 0, &Psi, &resi, &iter,
			 navsto_adjmat_tim_vec, navsto_adj_w_precon,
			 &K, &dIduI[tc], NULL ); /* */ 
#endif
	      FUNCTION_FAILURE_HANDLE(err, GMRES, navsto_solver_bcontrol);

	      /* check the actual residual */
#ifdef NEW_ADJOINT
	      err=navsto_matrix_tim_vec( &K, &Psi, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				       navsto_solver_bcontrol);

#else
	      err=navsto_adjmat_tim_vec( &K, &Psi, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_adjmat_tim_vec, 
				       navsto_solver_bcontrol);
#endif

	      for (i=0; i<y.len; i++)
		{
		  y.V[i]-=dIduI[tc].V[i];
		}
#ifdef NEW_ADJOINT
	      /* residual has to be projected */
	      for (i=0; i<y.len; i++)
		{
		  dIduI[tc].V[i]=y.V[i];
		}

	      err=navsto_projector_no_precon( &K, &dIduI[tc], NULL, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				       navsto_solver_bcontrol);

#endif
	      normy=0.0;
	      for (i=0; i<y.len; i++)
		{
		  normy+=y.V[i]*y.V[i];
		}


	      norm_adjres=sqrt(normy);
	      printf("adjoint solved!  iter=%d, |rhs|=%8.1e, "
		     "|C^-1*res|=%8.1e  |res|=%8.1e"
		     " inner-it:  %4d %4d %4d %4d\n",
		     iter, norm_dIduI, resi,
		     norm_adjres, K.innercount[0], K.innercount[1],
		     K.innercount[2], K.innercount[3]);

	      /* plot the adjoint solution */
		{
		  struct vector sol, soli;
		  FIDX eg, nod1, nod2, nodm;
		  
		  err=vector_alloc(&sol, 3*vx_nr);
		  FUNCTION_FAILURE_HANDLE( err, vector_alloc, 
					   navsto_solver_bcontrol);
		  for (i=0; i<dim*vx_nr; i++)
		    { sol.V[i]=Psi.V[i]; }
		  for (i=0; i<vx_nr; i++)
		    { sol.V[dim*vx_nr+i]=0.0; }
		  for (i=0; i<vx_nr; i++)
		    { 
		      if (K.pdof[i]!=-1)
			{ sol.V[dim*vx_nr+i]= Psi.V[dim*vx_nr+K.pdof[i]]; }
		    }

		  for (eg=0; eg<msh2.eg_nr; eg++)
		    {
		      if (msh2.edge[eg*msh2.eg_w+MCT2EGCHL1]==-1)
			{
			  nod1=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1  ];
			  nod2=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+1];
			  nodm=msh2.edge[eg*msh2.eg_w+MCT2EGNODM  ];
		    
			  sol.V[2*msh2.vx_nr+nodm]= 0.5 * 
			    (sol.V[2*msh2.vx_nr+nod1] 
			     + sol.V[2*msh2.vx_nr+nod2]); 
			}
		    }
		
		  /* write the velocity components */
		  soli.V=&sol.V[0];
		  soli.len=dim*vx_nr;
		  soli.n_max=dim*vx_nr;
		  err=mesh_write_solution_exp_t2( &msh2, &soli, dim, 20,
						  "visual/2d_mesh_v" );
		  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
					   navsto_solver_bcontrol); /* */
	  
	  
		  /* write the pressure components */
		  soli.V=&sol.V[dim*vx_nr];
		  soli.len=vx_nr;
		  soli.n_max=vx_nr;
		  err=mesh_write_solution_exp_t2( &msh2, &soli, 1, 20,
						  "visual/2d_mesh_p" );
		  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
					   navsto_solver_bcontrol); /* */
		  /* touch the marker file */
		  mark=fopen("visual/2d_mesh_mark", "w");
		  fprintf(mark, "next please!\n");
		  fclose(mark);

		  printf("wrote adjoint\n");
		}
	    }
	  else
	    {
	      /* don't really need to compute Psi as Psi=0 anyway */
	      for (i=0; i<Psi.len; i++)
		Psi.V[i]=0.0;
	    }

	  /* now evaluate DIDx from the adjoint representation,
	        DI     dI            dR
		--   = --  - Psi_I^T -- ,
		DuB    duB           duB  

	     while 
  
	     dR
             --   = K_IB
             duB
 
	     such that
                      dR
	     Psi_I^T  --  = (K^T Psi)^T_B
                      duB

	  */

	  err=navsto_matrix_tim_vec( &K, &Psi, &y );
	  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				   navsto_solver_bcontrol);

	  /* split out the boundary part of y  */
	  for (i=0; i<y.len; i++)
	    {
	      DIDuB[tc].V[i]=0.0;
	    }
	  for  (i=0; i<K.bn_nr; i++)
	    {
	      FIDX bcnode;
	      bcnode= K.nodes[i];
	      for (j=0; j<dim; j++)
		{
		  FIDX dof=bcnode+j*vx_nr;

		  DIDuB[tc].V[dof] = dIduB[tc].V[dof]-y.V[dof];
		}	      
	    }

	  /* now need to split DIDuB according to the boundary
	     segments into DIDB */
	  for (i=0; i<bc_sections; i++)
	    {
	      DIDB[tc].V[i]=0.0;
	    }

	  FIDX icount;
	  FIDX *fctn_id2bc_sections;

	  TRY_MALLOC(fctn_id2bc_sections,msh2.fu_nr, FIDX, navsto_solver_bcontrol);

	  /* create mapping from function id to bc_sections */
	  icount = 0;
	  for (i=0; i<msh2.fu_nr; i++)
	    {
	      if (msh2.func[i*msh2.fu_w+MC2XFUTYPE]==102)
		{
		  fctn_id2bc_sections[i] = icount;
		  icount++;
		}
	      else
		{
		  fctn_id2bc_sections[i] = -1;
		}
	    }

	  for (j=0; j<msh2.bd_nr; j++)
	    {
	      FIDX cur_edge, cur_node, fctn_id;

	      /* only for quadratic Dirichlet BCs  */
	      fctn_id = msh2.bound[j*msh2.bd_w+MCT2BDFNCT];
	      if (msh2.func[fctn_id*msh2.fu_w+MC2XFUTYPE]==102)
		{
		  cur_edge = msh2.bound[j*msh2.bd_w+MCT2BDEDGE];

		  /* take first node of current edge */
		  cur_node = msh2.edge[cur_edge*msh2.eg_w+MCT2EGNOD1+0];
		  /* add current node's contribution to scalar product  */
		  for (i=0; i<dim; i++)
		    {
		      DIDB[tc].V[fctn_id2bc_sections[fctn_id]] += 
			DIDuB[tc].V[cur_node+i*vx_nr]*x.V[cur_node+i*vx_nr];
		    }

		  /* take second node of current edge */
		  cur_node = msh2.edge[cur_edge*msh2.eg_w+MCT2EGNOD1+1];
		  /* add current node's contribution to scalar product  */
		  for (i=0; i<dim; i++)
		    {
		      DIDB[tc].V[fctn_id2bc_sections[fctn_id]] += 
			DIDuB[tc].V[cur_node+i*vx_nr]*x.V[cur_node+i*vx_nr];
		    }

		  /* take midnode of current edge */
		  cur_node = msh2.edge[cur_edge*msh2.eg_w+MCT2EGNODM];
		  /* add current node's contribution to scalar product (TWICE!)  */
		  for (i=0; i<dim; i++)
		    {
		      DIDB[tc].V[fctn_id2bc_sections[fctn_id]] += 
			2.0*DIDuB[tc].V[cur_node+i*vx_nr]*x.V[cur_node+i*vx_nr];
		    }
		}
	    }

	  /* correct calculated scalar product by dividing by 2
	     (since each node has been added twice except the two outermost ones 
	      but there u=0 anyway) */
	  for (i=0; i<bc_sections; i++)  DIDB[tc].V[i] = 0.5*DIDB[tc].V[i];

	  free(fctn_id2bc_sections);

	  vector_free(&dIduI[tc]);
	  vector_free(&dIduB[tc]);
	  vector_free(&dIdu[tc]);
	} /* end loop over cirteria */

      /* free intermediate help vectors */
      vector_free(&Psi);
      vector_free(&y);
      free(dIduB);
      free(dIduI);
      free(dIdu);

    }
  TIMEGET;
  tadj=ti;

  printf("\n\n times: solve=%8.1e   adj=%8.1e \n\n",
	 tsolve-tstart, tadj-tsolve);

  /* free local data */
  navsto_matrix_free(&K);
  vector_free(&rhs_stokes);
  vector_free(&rhs);
  vector_free(&x);

  vector_free(&x_old);

  mesh_free(&msh2);

  if (nu==nu_akt)
    return SUCCESS;
  else
    {
      fprintf(stderr,"navsto_solver_bcontrol: "
	      "unable to use desired nu on this mesh\n");
      return FAIL;
    }
}
