/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE navsto_solver.c
HEADER navsto_solver.h

TO_HEADER:

*/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "sparse.h"
#include "lin_solver.h"

#include "navsto_struct.h"
#include "stokes_aux.h"
#include "navsto_aux.h"
#include "navsto_adj.h"
#include "navstoassem.h"

#include "assembly.h"
#include "mesh_deform.h"

#include <unistd.h>
#include <string.h>

  /* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>


#define VERBOSE 4

#define TIMEGET {\
   gettimeofday(&tv,&tz); \
   sec=tv.tv_sec; \
   musec=tv.tv_usec; \
   ti=((double)sec+((double)musec)*1e-6); }

/*FUNCTION*/
int navsto_solver(struct mesh *msh1, FIDX lvl_max,
		  struct solver_settings *set,
		  struct vector *solref,
		  struct vector *pccrit,
		  struct vector *DIDF
/* the outer solver loop, controlls refinement, Reynolds number besed
   on the mesh Peclet number, mesh deformation, adjoint solve
   
   goal is to solve the discrete Navier-Stokes equations on uniformly
   refined, deformed T2 versions of the T1 mesh msh1, evaluate the
   performance criteria, solve the adjoint equations and evaluate the
   derivatives wrt to the shape parameters

   Input:  msh1    - the coarse mesh (T1),  hierarchy has to be reset,
                     i.e. msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1
		     for all j
           lvl_max - the desired refinement level
           set     - solver settings, the levels are ignored, because
                     they are supposed to be handled by the calling
                     routine
           
   In/Out: solref  - reference solution on the finest mesh, if
                     solref.len==0, this is initialised here,
                     otherwise it is used as initial solution for
                     Newtons method to solve the nonlinear flow
                     equations, or an error is returned if it is of
                     wrong size

   Output: pccrit  - vector of performance criteria, length
                     msh1.pc_nr, if pccrit==NULL it is not used
	   DIDF    - array of msh1.pc_nr vectors of length msh1.sp_nr,
	             containing the derivatives of the pccrit wrt the
	             shape paramters, not used if 
	             DIDF==NULL, the DIDF[i] are initialised here
	             otherwise

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid */
		  ){
  struct navsto_matrix K;
  struct vector rhs_stokes, rhs, x, x_old, y, z, sol, soli;
  struct mesh msh2;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t1, t2, t3, ti, tstart, tsolve, tadj, 
    tAD_I_0, tAD_I_1, tAD_R_0, tAD_R_1;
  FILE *mark;

  char solver[15];
  FIDX dim, vx_nr, i, j, d, step, linmeth, nu_chng;
  int  iter, err;

  FIDX lvl, lvl_min;

  double resi, resnl_old, resnl, normy, deltax;
  double leps, nleps, nleps_coarse, nleps_final, nleps_oldresnl;

  double impeps, innereps;
  double adjeps, adjinnereps;
  
  double nu, nu0, nu_akt, dnu, maxPe;

  double *vx_new;
  FIDX    vx_max;


  int m=100;

  int use_SDFEM=0;

  if (use_SDFEM) { printf("using SDFEM and box-smoother\n"); m = 200; } /* */

  TIMEGET;
  tstart=ti;


  nu=msh1->para[MC2XPANUPO];

  /* set the goals and methods */
  linmeth = 1;

  /* #warning "linmeth=2!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!"
     linmeth = 2; /* */

  nleps_coarse   = 1e-1;
  nleps_final    = 1e-6;
  nleps_oldresnl = 1e-8;
  impeps   = 1e-3;
  innereps = 1e-6; /* 1e-6 appears to be more robust for high Re than 1e-3 */ 

  adjeps      = 1e-6; 
  adjinnereps = 1e-6;
  /* fprintf(stderr,"for tests: adjeps=1e-5\n"); adjeps      = 1e-5; */

  dnu     = 0.5;
  dim=msh1->dim;

  /* initialise the deformation */
  err=mesh_deform_t1_t2(msh1, lvl_max, &vx_new, &vx_max, 0, NULL);
  FUNCTION_FAILURE_HANDLE( err, mesh_deform_t2_t1, navsto_solver);

  /* null define msh2 */
  mesh_init( &msh2);

  /* init the stiffness matrix */
  err=navsto_matrix_init( &K, dim, lvl_max+1);
  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_init, navsto_solver);

  
  if (solref->len==0)
    {
      /* start from scratch */
      lvl_min=0;
    }
  else
    {
      /* use the the provided initial solution */
      lvl_min=lvl_max;
    }

  for (lvl=lvl_min; lvl<=lvl_max; lvl++)
    {

      if (lvl==lvl_min)
	{
	  /* initial assem init */
	  if (!use_SDFEM)
	    {

	      if (solref->len==0)
		{
		  /* get the maximal stable Re on the coarse mesh */
		  err=navsto_maxnu_init_t21(msh1, &nu0);
		  FUNCTION_FAILURE_HANDLE(err, navsto_maxnu_init_t21,
					  navsto_solver);
	      
		  nu_akt=fmax(nu, nu0); 
		  msh1->para[MC2XPANUPO]=nu_akt; 
		  if (VERBOSE>0)
		    {
		      printf("nu=%8.1e     nu0=%8.1e     nu_akt=%8.1e\n",
			     nu, nu0, nu_akt);
		    }
		  nu_chng=0;
		}
	      else 
		{
		  nu_akt  = nu;
		  nu0     = nu;
		  nu_chng = 0;
		}

	      err=navsto_MG_init_assem_t21( msh1, lvl, &K, &rhs_stokes,
					    &rhs, &x, &msh2,
					    vx_new, vx_max );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_init_assem_t21,
				       navsto_solver);
	    }
	  else
	    {
	      nu_akt  = nu;
	      err=navsto_MG_init_assem_SDFEM_t21( msh1, lvl, &K, &rhs,
						  &x, &msh2,
						  vx_new, vx_max );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_init_assem_SDFEM_t21,
				       navsto_solver);
	      
	      err=vector_alloc( &rhs_stokes, x.len );
	      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);
	      for (i=0; i<rhs_stokes.len; i++) { rhs_stokes.V[i]=0.0; }
	    }
	  vx_nr=msh2.vx_nr;

	  err=vector_alloc( &x_old, x.len );
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

	  /* if solref is provided do
	     x_old       = sol_ref,
	     x_old(diri) = x(diri) (just to be sure)
	     x           = x_old */
	  if (solref->len!=0)
	    {
	      if (solref->len!=x.len)
		{
		  fprintf(stderr, "navsto_solver: "
			  "provided solref has wrong length\n");
		  return FAIL;
		}
	      /* x_old       = sol_ref */
	      for (i=0; i<x.len; i++)
		x_old.V[i]=solref->V[i];

	      /* x_old(diri) = x(diri) (just to be sure) */
	      for (i=0; i<K.bn_nr; i++)
		for (d=0; d<dim; d++)
		  {
		    if (x_old.V[K.nodes[i]+d*vx_nr]!=x.V[K.nodes[i]+d*vx_nr])
		      printf("xref node %d, dim %d is strange\n",
			     (int) K.nodes[i], (int) d);
		    x_old.V[K.nodes[i]+d*vx_nr]=x.V[K.nodes[i]+d*vx_nr];
		  }

	      /* x           = x_old */
	      for (i=0; i<x.len; i++)
		x.V[i]=x_old.V[i];
	    } /* end init with solref */
	}
      else /* if (lvl!=lvl_min) */
	{
	  /* copy the old solution, used for the interpolation */
	  for (i=0; i<x.len; i++)
	    { x_old.V[i]=x.V[i];}

	  /* update nu */
	  nu_akt=fmax(nu, maxPe*1.3*nu_akt*dnu);
	  /* #warning "higher Pe"
	     nu_akt=fmax(nu, 0.125*maxPe*1.3*nu_akt*dnu); /* */
	  if (msh1->para[MC2XPANUPO]!=nu_akt)
	    {
	      msh1->para[MC2XPANUPO]=nu_akt; 
	      nu_chng=1;
	    }

	  /* refine, init on finer mesh, with larger Re */
	  err=navsto_MG_ref_intp_ass_t21( msh1, lvl, &msh2, vx_new, vx_max,
					  &K, &rhs_stokes, &rhs,
					  &x_old, &x );
	  FUNCTION_FAILURE_HANDLE( err, navsto_MG_ref_intp_ass_t21,
				   navsto_solver); 

	  vx_nr=msh2.vx_nr;
	} /* end if (lvl>0) */
    
      err=vector_alloc( &y, x.len );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);
      err=vector_alloc( &z, x.len );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);
      err=vector_alloc( &sol, (dim+1)*K.vx_nr );
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

      if (lvl==lvl_max)
	nleps=nleps_final;
      else
	nleps=nleps_coarse;

      printf("nleps=%e\n",nleps);

      /* the nonlinear loop */
      resi=1.0; resnl=1.0; deltax=1.0; step=0; resnl_old=1e300;
      while (((deltax>nleps)||(resi>nleps))
	     &&((resnl>nleps_oldresnl)&&(step<100)))
	{
	  /* save the old x for the computation of deltax */
	  for (i=0; i<x.len; i++)
	    { x_old.V[i]=x.V[i]; }

	  TIMEGET;
	  t1=ti;

	  if (!use_SDFEM)
	    {
	      err=navsto_MG_C_assem_t21( msh1, lvl, &K, &rhs, &maxPe, 
					 &rhs_stokes, &x, &msh2, 
					 vx_new, vx_max,
					 linmeth );
	      FUNCTION_FAILURE_HANDLE( err, navsto_MG_C_assem_t21,
				       navsto_solver);
	    }
	  else
	    {      
	      err=navsto_MG_assem_SDFEM_t21( msh1, lvl, &K, &rhs, &x, 
					     &msh2, vx_new, vx_max );
	      FUNCTION_FAILURE_HANDLE( err, navsto_assem_SDFEM_t21,
				       navsto_solver);
	      maxPe=0.0;
	    }

	  TIMEGET;
	  t2=ti;

	  /* test the residual */
	  /* y=K*x */
	  err=navsto_matrix_tim_vec( &K, &x, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec,
				   navsto_solver);  
	  /* z=K*x-rhs, get norm(y) */
	  for (i=0; i<x.len; i++)
	    {
	      z.V[i]=y.V[i]-rhs.V[i];
	    }
	  /* apply the projector */
	  err=navsto_projector_no_precon( &K, &z, NULL, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon, 
				   navsto_solver);  
	  normy=0.0;
	  for (i=0; i<x.len; i++)
	    {
	      normy+= y.V[i]*y.V[i];
	    }
	  resnl=sqrt(normy); /* */
	  
	  if (VERBOSE>3)
	    {
	      printf("maxPe= %8.1e ", maxPe);
	      printf("|res_nl|=%12.6e, ", resnl); /* */
	      printf("t_a=%8.2e ", t2-t1); /* */
	      printf("\n"); /* */
	    }

	  /* before we solve, output the matrices to files (for
	     external analysis */
	  /* { 
	     char matname[50];
	     sprintf(matname,  "visual/savemat_%d",lvl); 
	     err=navsto_matrices_write_files( &K, matname);
	     FUNCTION_FAILURE_HANDLE(err, navsto_matrices_write_files,
	     navsto_solver);
	     }/* */


	  resi=0.0;
	  iter=0;

	  /* set stoping criterion for linear solver */
	  leps=fmax(impeps, 1e-2 * nleps_oldresnl/ resnl); 

	  if (!use_SDFEM)
	    {
	      K.innereps=innereps;
	      K.mg->vcycles=1;
	      K.mg->smooths=1;
	      K.mg->stop_eps=0.0;

	      strcpy(solver, "GMRES_Fp");
	      err=GMRES( m, 100, 2, 0.0, leps, 1, &x, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_projector_w_precon,
			 &K, &rhs, NULL ); /* */
	    }
	  else
	    {
	      strcpy(solver, "GMRES_mg");
	      err=GMRES( m, 100, 2, 0.0, leps, 1, &x, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_mg_SDFEM_precon,
			 &K, &rhs, NULL ); /* */
	    }
	  FUNCTION_FAILURE_HANDLE(err, GMRES, navsto_solver);
	  

	  TIMEGET;
	  t3=ti;

	  /* test the residual */
	  /* y=K*x */
	  err=navsto_matrix_tim_vec( &K, &x, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				   navsto_solver);  
	  /* z=K*x-rhs, get norm(y) */
	  normy=0.0;
	  for (i=0; i<x.len; i++)
	    {
	      z.V[i]=y.V[i]-rhs.V[i];
	    }
	  /* apply the projector */
	  err=navsto_projector_no_precon( &K, &z, NULL, &y);
	  FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon, 
				   navsto_solver);  
	  for (i=0; i<x.len; i++)
	    {
	      normy+= y.V[i]*y.V[i];
	    }
	  resi=sqrt(normy); /* */


	  /* compute deltax */
	  deltax=0.0;
	  for (i=0; i<x.len; i++)
	    { deltax+=(x_old.V[i]-x.V[i])*(x_old.V[i]-x.V[i]); }
	  deltax=sqrt(deltax);

	  /* increase the step counter */
	  step++;

	  if ((VERBOSE>3)
	      ||((VERBOSE>1)
		  &&( (deltax<=nleps)&&(resi<=nleps) ) ))
	    {
	      printf("nu %8.1e st %3d  %8s: N=%6d, it=%4d, "
		     "i_it: %4d %4d %4d %4d err=%2d, |res_l|=%12.6e, "
		     "|dx|=%9.3e, ",
		     nu_akt, (int) step, solver, (int) x.len, iter,
		     K.innercount[0], K.innercount[1],
		     K.innercount[2], K.innercount[3],
		     err, resi, deltax); /* */
	      printf("t_s=%8.2e ", t3-t2); /* */
	    }

	  if (VERBOSE>-1) /* write the solution for visualization */
	    {
	      for (i=0; i<dim*vx_nr; i++)
		{ sol.V[i]=x.V[i]; }
	      for (i=0; i<vx_nr; i++)
		{ sol.V[dim*vx_nr+i]=0.0; }
	      for (i=0; i<vx_nr; i++)
		{ 
		  if (K.pdof[i]!=-1)
		    { sol.V[dim*vx_nr+i]= x.V[dim*vx_nr+K.pdof[i]]; }
		}
	      /* correct the pressure values of the edge-mid-nodes */
	      {
		FIDX eg, nod1, nod2, nodm;
		for (eg=0; eg<msh2.eg_nr; eg++)
		  {
		    if (msh2.edge[eg*msh2.eg_w+MCT2EGCHL1]==-1)
		      {
			nod1=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1  ];
			nod2=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+1];
			nodm=msh2.edge[eg*msh2.eg_w+MCT2EGNODM  ];
		    
			sol.V[2*msh2.vx_nr+nodm]= 0.5 * 
			  (sol.V[2*msh2.vx_nr+nod1] 
			   + sol.V[2*msh2.vx_nr+nod2]); 
		      }
		  }
	      }
	      /* write the solution for paraview */
	      { struct vector sol_u, sol_p;
		sol_u.V     = &sol.V[0];
		sol_u.len   = (dim)*vx_nr;
		sol_u.n_max = (dim)*vx_nr;
		sol_p.V     = &sol.V[dim*vx_nr];
		sol_p.len   = vx_nr;
		sol_p.n_max = vx_nr;
		err=mesh_write_solution_vtk_t2( &msh2, &sol_u, dim, &sol_p, 1,
						20, "visual/2d_mesh_vtk_" );
		FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_vtk_t1,
					 navsto_solver); /* */
		err=mesh_write_solution_femplot_t2( &msh2, &sol_u, dim, 
						20, "visual/navsto_vec__" );
		FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_femplot_t2,
					 navsto_solver); /* */
		err=mesh_write_solution_femplot_t2( &msh2, &sol, dim+1, 
						20, "visual/navsto_all__" );
		FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_femplot_t2,
					 navsto_solver); /* */
	      }

	      /* write the solution values for comparison */
	      {
		soli.V     = &sol.V[0];
		soli.len   = (dim)*vx_nr;
		soli.n_max = (dim)*vx_nr;
		char fname[300];
		sprintf(fname, "visual/line.x.%d.txt", (int) lvl);
		err=navsto_write_line_u_t2( &msh2, &soli, 0, 0.5, fname );
		FUNCTION_FAILURE_HANDLE( err,
					 mesh_write_line_u_t2,
					 navsto_solver); 
		sprintf(fname, "visual/line.y.%d.txt", (int) lvl);
		err=navsto_write_line_u_t2( &msh2, &soli, 1, 0.5,
					    fname ); 
		FUNCTION_FAILURE_HANDLE( err,
					 mesh_write_line_u_t2,
					 navsto_solver);
	      } /* */ 
	  

	      /* touch the marker file */
	      mark=fopen("visual/2d_mesh_mark", "w");
	      fprintf(mark, "next please!\n");
	      fclose(mark);
	    }
	}


      /* final check of nonlinear convergence */
      TIMEGET;
      t1=ti;
      
      if (!use_SDFEM)
	{
	  err=navsto_MG_C_assem_t21( msh1, lvl, &K, &rhs, &maxPe,
				     &rhs_stokes,&x, &msh2,
				     vx_new, vx_max, linmeth );
	  FUNCTION_FAILURE_HANDLE( err, navsto_MG_C_assem_t21,
				   navsto_solver);
	}
      else
	{
	  err=navsto_MG_assem_SDFEM_t21( msh1, lvl, &K, &rhs, &x,
					 &msh2, vx_new, vx_max );
	  FUNCTION_FAILURE_HANDLE( err, navsto_assem_SDFEM_t21, 
				   navsto_solver);
	}
  
      TIMEGET;
      t2=ti;

      /* test the residual */
      /* y=K*x */
      err=navsto_matrix_tim_vec( &K, &x, &y);
      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, navsto_solver);  
      /* z=K*x-rhs, get norm(y) */
      normy=0.0;
      for (i=0; i<x.len; i++)
	{
	  z.V[i]=y.V[i]-rhs.V[i];
	}
      /* apply the projector */
      err=navsto_projector_no_precon( &K, &z, NULL, &y);
      FUNCTION_FAILURE_HANDLE( err, navsto_projector_no_precon,
			       navsto_solver);  
      for (i=0; i<x.len; i++)
	{
	  normy+= y.V[i]*y.V[i];
	}
      normy=sqrt(normy); /* */

      if (VERBOSE>3)
	{
	  printf("maxPe= %8.1e ", maxPe);
	  printf("|res_nl|=%12.6e, ", normy); /* */
	  printf("t_a=%8.2e ", t2-t1); /* */
	  printf("\n");
	}

      vector_free(&y);
      vector_free(&z);
      vector_free(&sol);

      /* do the postprocessing */ 
      if (pccrit!=NULL)
	{
	  err=navsto_dIdX_t21( &msh2, &K, &x, msh2.pc_nr,
			       pccrit, NULL, NULL, NULL);
	  FUNCTION_FAILURE_HANDLE( err, navsto_dIdX_t21, navsto_solver);  

	  if (VERBOSE>2)
	    {
	      printf("\nperformance criteria eval: err=%d\n",err);
	      for (i=0; i<msh2.pc_nr; i++)
		{
		  printf("crit %d: %e\n", (int) i, pccrit->V[i]);
		}
	    }
	}

    } /* end mesh refinement + interpolation */




  /*********************************************************/
  /*                                                       */
  /* navier-stokes system is now solved on the finest mesh */
  /*                                                       */
  /*********************************************************/



  
  /* if no reference solution was provided, return one to the
     calling routine */
  if (solref->len==0)
    {
      err=vector_alloc(solref, x.len);
      FUNCTION_FAILURE_HANDLE( err, vecto_alloc, navsto_solver);
      /* copy x to solref */
      for (i=0; i<x.len; i++)
	solref->V[i]=x.V[i];
    }

  TIMEGET;
  tsolve=ti;

  /* if required, do the adjoint solve */
  if ((DIDF != NULL)&&(msh1->pc_nr>0))
    {
      FIDX tc;
      struct vector *DIDx, *dIdx, *dIdsol;
      struct vector Psi, y, Psi_dRdx;

      TRY_MALLOC(DIDx,   msh2.pc_nr, struct vector, navsto_solver);
      TRY_MALLOC(dIdx,   msh2.pc_nr, struct vector, navsto_solver);
      TRY_MALLOC(dIdsol, msh2.pc_nr, struct vector, navsto_solver);

      for (tc=0; tc<msh2.pc_nr; tc++)
	{
	  err=vector_alloc( &DIDx[tc], dim*vx_nr); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

	  err=vector_alloc( &dIdx[tc], dim*vx_nr); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

	  err=vector_alloc( &DIDF[tc], msh1->sp_nr); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

	  err=vector_alloc( &dIdsol[tc], x.len); 
	  FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);
	}
      TIMEGET;
      tAD_I_0=ti;

      err=navsto_dIdX_t21( &msh2, &K, &x, msh2.pc_nr,
			   pccrit, dIdsol, dIdx, NULL);
      FUNCTION_FAILURE_HANDLE( err, navsto_dIdX_t21, navsto_solver);

      TIMEGET;
      tAD_I_1=ti;

      printf("\n tAD_I=%9.2e\n",tAD_I_1-tAD_I_0);

      /* transpose the system matrix */
      err=navsto_adj_convert( &K);
      FUNCTION_FAILURE_HANDLE( err, navsto_adj_convert, navsto_solver);

      /* allocate the adjoint solution vector Psi and a help vector y */
      err=vector_alloc( &Psi, x.len);
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);
      err=vector_alloc( &y, x.len);
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

      /* allocate space for Psi_dRdx */
      err=vector_alloc( &Psi_dRdx, dim*vx_nr);
      FUNCTION_FAILURE_HANDLE( err, vector_alloc, navsto_solver);

      /* prepare to solve */
      K.innereps=adjinnereps;
      K.mg->vcycles=1;
      K.mg->smooths=1;
      K.mg->stop_eps=0.0;

      /* for each criterion */
      for (tc=0; tc<msh2.pc_nr; tc++)
	{
	  double norm_dIdsol, norm_adjres;
	  /* compute the norm of the adjoint rhs */
	  normy=0.0;
	  for (i=0; i<dIdsol[tc].len; i++)
	    normy += dIdsol[tc].V[i] * dIdsol[tc].V[i];
	  norm_dIdsol=sqrt(normy);

	  if (norm_dIdsol!=0.0)
	    {
	      /* actualy need to solve */

	      /* reset iteration counters */
	      for (i=0; i<5; i++) K.innercount[i]=0;

	      /* solve the adjoint system */
#ifdef NEW_ADJOINT
	      /* new adjoint handling: same preconditioner as for
		 forward solve, but do a projection step first */
	      {
		double Psi_lam;

		Psi_lam=0.0;
		for (i=0; i<vx_nr; i++)
		  {
		    if (K.pdof[i]!=-1)
		      {
			Psi_lam += dIdsol[tc].V[dim*vx_nr+ K.pdof[i] ];
		      }
		  }

		printf("Psi_lambda=%e\n", Psi_lam);

		for (i=0; i<vx_nr; i++)
		  {
		    if ( K.pdof[i]!=-1)
		      {
			dIdsol[tc].V[ dim*vx_nr+K.pdof[i] ] -= 
			  Psi_lam * K.weight[i] ;
		      }
		  }

		if (Psi_lam > 1e-8)
		  {
		    fprintf(stderr,"navsto_solver: "
			    "current adjoint handling can not deal\n"
			    "with e*dI/dp > 0, something wrong with\n"
			    "the criterion?\n");
		    return FAIL;
		  }
	      }

	      err=GMRES( m, 100, 2, 0.0, adjeps, 0, &Psi, &resi, &iter,
			 navsto_matrix_tim_vec, navsto_projector_w_precon,
			 &K, &dIdsol[tc], NULL ); /* */ 
#else
	      /* old adjoint handling, special multiplication and
		 preconditioner routines */
	      err=GMRES( m, 100, 2, 0.0, adjeps, 0, &Psi, &resi, &iter,
			 navsto_adjmat_tim_vec, navsto_adj_w_precon,
			 &K, &dIdsol[tc], NULL ); /* */ 
#endif
	      FUNCTION_FAILURE_HANDLE(err, GMRES, navsto_solver);

	      /* check the actual residual */
#ifdef NEW_ADJOINT
	      err=navsto_matrix_tim_vec( &K, &Psi, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				       navsto_solver);

#else
	      err=navsto_adjmat_tim_vec( &K, &Psi, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_adjmat_tim_vec, 
				       navsto_solver);
#endif

	      for (i=0; i<y.len; i++)
		{
		  y.V[i]-=dIdsol[tc].V[i];
		}
#ifdef NEW_ADJOINT
	      /* residual has to be projected */
	      for (i=0; i<y.len; i++)
		{
		  dIdsol[tc].V[i]=y.V[i];
		}

	      err=navsto_projector_no_precon( &K, &dIdsol[tc], NULL, &y );
	      FUNCTION_FAILURE_HANDLE( err, navsto_matrix_tim_vec, 
				       navsto_solver);

#endif
	      normy=0.0;
	      for (i=0; i<y.len; i++)
		{
		  normy+=y.V[i]*y.V[i];
		}


	      norm_adjres=sqrt(normy);
	      printf("adjoint solved!  iter=%d, |rhs|=%8.1e, "
		     "|C^-1*res|=%8.1e  |res|=%8.1e"
		     " inner-it:  %4d %4d %4d %4d\n",
		     iter, norm_dIdsol, resi,
		     norm_adjres, K.innercount[0], K.innercount[1],
		     K.innercount[2], K.innercount[3]);

	      /* plot the adjoint solution */
		{
		  struct vector sol, soli;
		  FIDX eg, nod1, nod2, nodm;
		  
		  err=vector_alloc(&sol, 3*vx_nr);
		  FUNCTION_FAILURE_HANDLE( err, vector_alloc, 
					   navsto_solver);
		  for (i=0; i<dim*vx_nr; i++)
		    { sol.V[i]=Psi.V[i]; }
		  for (i=0; i<vx_nr; i++)
		    { sol.V[dim*vx_nr+i]=0.0; }
		  for (i=0; i<vx_nr; i++)
		    { 
		      if (K.pdof[i]!=-1)
			{ sol.V[dim*vx_nr+i]= Psi.V[dim*vx_nr+K.pdof[i]]; }
		    }

		  for (eg=0; eg<msh2.eg_nr; eg++)
		    {
		      if (msh2.edge[eg*msh2.eg_w+MCT2EGCHL1]==-1)
			{
			  nod1=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1  ];
			  nod2=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+1];
			  nodm=msh2.edge[eg*msh2.eg_w+MCT2EGNODM  ];
		    
			  sol.V[2*msh2.vx_nr+nodm]= 0.5 * 
			    (sol.V[2*msh2.vx_nr+nod1] 
			     + sol.V[2*msh2.vx_nr+nod2]); 
			}
		    }
		
		  /* write the velocity components */
		  soli.V=&sol.V[0];
		  soli.len=dim*vx_nr;
		  soli.n_max=dim*vx_nr;
		  err=mesh_write_solution_exp_t2( &msh2, &soli, dim, 20,
						  "visual/2d_mesh_v" );
		  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
					   navsto_solver); /* */
	  
	  
		  /* write the pressure components */
		  soli.V=&sol.V[dim*vx_nr];
		  soli.len=vx_nr;
		  soli.n_max=vx_nr;
		  err=mesh_write_solution_exp_t2( &msh2, &soli, 1, 20,
						  "visual/2d_mesh_p" );
		  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2,
					   navsto_solver); /* */
		  /* touch the marker file */
		  mark=fopen("visual/2d_mesh_mark", "w");
		  fprintf(mark, "next please!\n");
		  fclose(mark);

		  printf("wrote adjoint\n");

		  vector_free(&sol);
		}
	    }
	  else
	    {
	      /* don't really need to compute Psi as Psi=0 anyway */
	      for (i=0; i<Psi.len; i++)
		Psi.V[i]=0.0;
	    }

	  /* now evaluate DIDx from the adjoint representation,
	        DI   dI         dR
		-- = -- - Psi^T -- .
		Dx   dx         dx                                */

          TIMEGET;
	  tAD_R_0=ti;

	  err=navsto_Psi_dRdF_t21(&Psi_dRdx, &Psi, &x, &K, &msh2);
	  FUNCTION_FAILURE_HANDLE( err, navsto_Psi_dRdF_t21,
				   navsto_solver);

          TIMEGET;
	  tAD_R_1=ti;
	  printf("\n tAD_R=%9.2e\n",tAD_R_1-tAD_R_0);

	  for (i=0; i<dim*vx_nr; i++)
	    {
	      DIDx[tc].V[i] = dIdx[tc].V[i] - Psi_dRdx.V[i];
	    }

	  vector_free(&dIdx[tc]);
	  vector_free(&dIdsol[tc]);
	} /* end loop over cirteria */

      /* free intermediate help vectors */
      vector_free(&Psi);
      vector_free(&y);
      vector_free(&Psi_dRdx);
      free(dIdx);
      free(dIdsol);

      /* modify DIDx to account for that interior nodes are linearly
	 dependant on position of boundary nodes */
      err=mesh_deform_t1_t2(msh1, lvl_max, NULL, NULL, msh1->pc_nr,
			    DIDx);
      FUNCTION_FAILURE_HANDLE( err, mesh_deform_t2_t1, navsto_solver);

      /* compute DIDF from DIDx */
      /*                       msh, insert, nI,       dIdx, dIdF, type */
      err=mesh_sseg_adjust_tx( &msh2, 0, msh1->pc_nr, DIDx, DIDF, 2); 
      FUNCTION_FAILURE_HANDLE( err, mesh_sseg_adjust_tx, navsto_solver);

      /* free DIDx */
      for (tc=0; tc<msh2.pc_nr; tc++)
	vector_free(&DIDx[tc]);
      free(DIDx);
    }
  TIMEGET;
  tadj=ti;

  printf("\n\n times: solve=%8.1e   adj=%8.1e \n\n",
	 tsolve-tstart, tadj-tsolve);

  /* free local data */
  navsto_matrix_free(&K);
  vector_free(&rhs_stokes);
  vector_free(&rhs);
  vector_free(&x);

  vector_free(&x_old);

  mesh_free(&msh2);

  if (nu==nu_akt)
    return SUCCESS;
  else
    {
      fprintf(stderr,"navsto_solver: "
	      "unable to use desired nu on this mesh\n");
      return FAIL;
    }
}
