/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE gen_aux.c
HEADER gen_aux.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "sparse_struct.h"

*/



/* function prototypes */
#include "gen_aux.h"

/* prototypes of external functions */
#include <math.h>

#include "mesh.h"


/*FUNCTION*/
int gen_proj_bpx_tx(void *notused, struct vector *in,
		    void *arg3, struct vector *out
/* performs boundary condition projection and BPX preconditioning 
     out = P*C^{-1}*P^T in
   where C^{-1} is the BPX preconditioner sum(l=level)Q_l*D_l*Q_l^T 
   and P a project
   
   Input:  notused - well, it is not used but in the interface to
                     allow this function to be used as a
                     preconditioner
           in      - input vector
	   arg3=
           bpx     - struct bpxdata containing pointers to the mesh
                     a multilevel struct, a multilevel help vector,
		     and a projector1

   Output: out    - (given by reference), P*P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, j, d;
  int err;
  FIDX vx_nr, hi_w, hi_nr, fath1, fath2, child, eg_w, eg_nr;
  FIDX level, level_old, lvlmax;
  FIDX ml_type, mldim;
  FIDX l_mctxhilvl, l_mctxhichld,  l_mctxhifat1;

  struct bpxdata       *bpx;
  struct mesh          *m;
  struct multilvl      *ml;
  struct projector1    *P;

  bpx    = arg3;
  m      = (*bpx).msh;
  ml     = (*bpx).ml;
  lvlmax = (*ml).lmax;
  ml_type= (*ml).type;
  mldim  = (*ml).dim;
  P      = (*bpx).P;

  vx_nr  = (*m).vx_nr;
  eg_nr  = (*m).eg_nr;
  eg_w   = (*m).eg_w;
  hi_nr  = (*m).hi_nr;
  hi_w   = (*m).hi_w;

  switch (ml_type)
    {
    case 1:
      l_mctxhilvl  = MCT1HILVL;
      l_mctxhichld = MCT1HICHLD;
      l_mctxhifat1 = MCT1HIFAT1;
      break;
    case 2:
      l_mctxhilvl  = MCT2HILVL;
      l_mctxhichld = MCT2HICHLD;
      l_mctxhifat1 = MCT2HIFAT1;
      break;
    default:
      fprintf(stderr,"gen_proj_bpx_tx: unknown type=%d\n",
	      (int) ml_type);
      return FAIL;
    }
    

  if ( (((*in).len!=(*out).len)||((*in).len!=vx_nr*mldim))||
       (*ml).nlevl[(*ml).lmax]-(*ml).nlevl[(*ml).lmax+1]!=vx_nr*mldim)
    {
      fprintf(stderr,
	      "gen_proj_bpx_tx: dimensions don't make sense!\n");
      return FAIL;
    }

  /* copy in to hvec */
  for (i=0; i<(*in).len; i++)
    {
      (*bpx).hvec[i]=(*in).V[i];
    }

  if (P!=NULL)
    {
      /* apply P^T */
      for (i=0; i<(*P).len; i++)
	{
	  (*bpx).hvec[ (*P).V[i] ]   = 0.0 ;
	}
    }

  /* calculate r_l = Q_l^T r = [I P_l^T] r_(l+1), r_lmax=r for all
     levels l, r_l is stored in hvec[ nlevl[l+1]...nlevl[l]-1 ], */
  level_old=(*ml).lmax+1;
  for (i=hi_nr-1; i>=0; i--)
    {
      level=(*m).hier[i*hi_w+l_mctxhilvl];
      if (level<=0) printf("error, level=%d\n", (int) level);
      if (level_old!=level)
	{
	  FIDX l_vx_nr=((*ml).nlevl[level]-(*ml).nlevl[level+1])/mldim;
      
	  /* initialise the new (coarser) level */
	  for (j=0; j<l_vx_nr; j++)
	    for (d=0; d<mldim; d++)
	      {
		/* get the entry number of the coarser level */
		child = j+d*vx_nr;
		MLVLFINDENTRY(fath2, child, level-1, *ml);
		if (fath2>=0) 
		  {
		    MLVLFINDENTRY(child, child, level, *ml);
		    /* the I (identity) part, r_l = r_(l+1) */
		    (*bpx).hvec[fath2]=(*bpx).hvec[child];
		  }
	      }
	  level_old=level;
	}

      for (d=0; d<mldim; d++)
	{
	  child=(*m).hier[i*hi_w+l_mctxhichld  ]+d*vx_nr;
	  fath1=(*m).hier[i*hi_w+l_mctxhifat1  ]+d*vx_nr;
	  fath2=(*m).hier[i*hi_w+l_mctxhifat1+1]+d*vx_nr;

	  MLVLFINDENTRY(fath1, fath1, level-1, *ml);
	  MLVLFINDENTRY(fath2, fath2, level-1, *ml);
	  MLVLFINDENTRY(child, child, level,   *ml);

	  //printf("i=%d   d=%d  f1=%d f2=%d c=%d\n",i,d,fath1,fath2,child);

	  /* the P_l^T part */
	  (*bpx).hvec[fath1] += 0.5*(*bpx).hvec[child];
	  (*bpx).hvec[fath2] += 0.5*(*bpx).hvec[child]; 
	}
    }


#ifdef DO_PROJ_ALL_LVL_BPX
  if (P!=NULL)
    {
      /* perform the boundary projection on all levels */
      for (i=0; i<(*P).len; i++)
	{
	  FIDX node, nodel;
	  node=(*P).V[i];
	  for (j=0; j<=(*ml).lmax; j++)
	    {
	      MLVLFINDENTRY(nodel, node, j, *ml);
	      if (nodel>=0)
		{
		  (*bpx).hvec[nodel]=0.0;
		}
	    }
	}
    }
#endif



  /* calculate w_l = [I ; P_l] w_(l-1) + D_l r_l, w_0=K_0^(-1) r_0 */
  /* initialise the coarsest level, 
     appy the coarse grid solver, if available,
     otherwise nothing has to be done in 2d as D_0 = I */
  if ((*bpx).cmat!=NULL) 
    {
      FIDX dof;
      struct vector rhs_x;

      /* here we suppose the coarse grid nodes have consecutive
	 numbers and start from 0 */
      dof=0;
      MLVLFINDENTRY(dof, dof, 0, *ml);

      rhs_x.V=&(*bpx).hvec[dof];
      rhs_x.len=(*ml).nlevl[0]-(*ml).nlevl[1];
      rhs_x.n_max=rhs_x.len;

      /* the dirichlet dofs are set to zero by the above projection */
#warning "requires projection of residual first?"

      /* coarse grid solve */
      err=coarse_mat_solve( (*bpx).cmat, NoTrans, &rhs_x, &rhs_x);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve,
			      stokes_projector_part_bpx);

      /* the solution is in w_0 */
    }

  /* calculate w_l = [I ; P_l] w_(l-1) + D_l r_l, */
  level_old=0; 
  for (i=0; i<hi_nr; i++)
    {
      level=(*m).hier[i*hi_w+l_mctxhilvl];

      if (level_old!=level)
	{
	  FIDX lm1_vx_nr=((*ml).nlevl[level-1]-(*ml).nlevl[level])/mldim;
      
	  /* initialise the new (finer) level */
	  for (j=0; j<lm1_vx_nr; j++)
	    for (d=0; d<mldim; d++)
	      {
		/* get the entry number of the coarser level */
		fath2 = j+d*vx_nr;
		MLVLFINDENTRY(child, fath2, level, *ml);
		MLVLFINDENTRY(fath2, fath2, level-1, *ml);
		/* the I (identity) part */
		(*bpx).hvec[child]+=(*bpx).hvec[fath2];
	      }
	  level_old=level;
	}
      
      for (d=0; d<mldim; d++)
	{
	  child=(*m).hier[i*hi_w+l_mctxhichld  ]+d*vx_nr;
	  fath1=(*m).hier[i*hi_w+l_mctxhifat1  ]+d*vx_nr;
	  fath2=(*m).hier[i*hi_w+l_mctxhifat1+1]+d*vx_nr;
      
	  MLVLFINDENTRY(fath1, fath1, level-1, *ml);
	  MLVLFINDENTRY(fath2, fath2, level-1, *ml);
	  MLVLFINDENTRY(child, child, level,   *ml);
      
	  /* the P_(l-1) part */
	  (*bpx).hvec[child]+=0.5*( (*bpx).hvec[fath1] 
				    + (*bpx).hvec[fath2] );
	}
    }

  /* copy to out */
  for (i=0; i<(*out).len; i++)
    {
      (*out).V[i]=(*bpx).hvec[i];
    }

  if (P!=NULL)
    {
      /* apply P */
      for (i=0; i<(*P).len; i++)
	{
	  (*out).V[ (*P).V[i] ]   = 0.0 ;
	}
    }

  /* done */
  return SUCCESS;
}


/*FUNCTION*/
int gen_proj_MG_tx(void *arg1, struct vector *in,
		   void *arg3, struct vector *out
/* performs the boudary condition projection and multigrid
   preconditioning, 
     out = P*C^-1*P^T * in
   where P projects the velocity components of boundary nodes to zero,
   (such that addition of a projected vector doesn't change the
   velocity there), and C^-1 repressents the action of the multigrid
   preconditioning, that is one V cycle
   
   Input:  arg1    - not used, in the interface for compatibility reasons
           in      - input vector
	   arg3=
	   mg      - mgdata struct, containing everything needed for
	             the projection and multigrid

   Output: out    - (given by reference), P*C^-1P^T* in

   Return: SUCCESS - success,
           FAIL    - failure, see error message
*/
		 ){
  FIDX i, j, len;
  int err;
  FIDX dim, vx_nr;

  struct vector xi, bi, invdiagi, dxi;

  struct mgdata *mg;
  struct mesh *m;
  struct multilvl *ml;

  struct sparse *Ks;
  struct projector1 *P;

  double *xl, *dxl, *bl, *invdiag;

  double alpha;

  FIDX lvl, lmax, lmin, lvl_vx, smooths, vcycles, vccount;


  mg     = arg3;

  m      = (*mg).msh;
  ml     = (*mg).ml;

  Ks     = (*mg).Ks;
  P      = (*mg).P;

  dim    = (*ml).dim;
  vx_nr  = (*mg).vx_nr;
  len    = dim * vx_nr;

  xl      = (*mg).xl;
  dxl     = (*mg).dxl;
  bl      = (*mg).bl;
  invdiag = (*mg).invdiag;

  smooths  = (*mg).smooths;
  vcycles  = (*mg).vcycles;
  alpha    = (*mg).CGC_scale;

  if ((len!=(*in).len)||(len!=(*out).len))
    {
      fprintf(stderr,
	      "gen_proj_MG_tx: in or out length doesn't match\n");
      return FAIL;
    }


  lmax = (*ml).lmax;

  lmin=0; 


  /* copy in to bl */
  for (i=0; i<(*in).len; i++)
    {
      bl[i]=(*in).V[i];
    }

  /* apply P^T */
  for (i=0; i<(*P).len; i++)
    {
      bl[ (*P).V[i] ] = 0.0 ;
    }

  /* apply C^-1 */

  if (vcycles<=0)
    {
      fprintf(stderr, "gen_proj_MG_tx: vcycles<=0 not defined\n");
      return FAIL;
    }
 

  /* perform vcycles V-cycles */
  for (vccount=0; vccount<vcycles; vccount++)
    {
      /* V-cycle downward */
      for (lvl=lmax; lvl>=lmin; lvl--)
	{ 
	  lvl_vx=(*ml).nlevl[lvl]-(*ml).nlevl[lvl+1];
	  
	  if ((vccount==0)||(lvl<lmax))
	    {
	      /* set xl=0 */
	      for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
		xl[j]=0.0;
	    }
	  
	  /* make x_l, b_l, invdiag_l accessible as vector */
	  xi.len=lvl_vx;
	  xi.n_max=lvl_vx;
	  xi.V=&xl[(*ml).nlevl[lvl+1]];

	  bi.len=lvl_vx;
	  bi.n_max=lvl_vx;
	  bi.V=&bl[(*ml).nlevl[lvl+1]];

	  invdiagi.len=lvl_vx;
	  invdiagi.n_max=lvl_vx;
	  invdiagi.V=&invdiag[(*ml).nlevl[lvl+1]];

	  if ((lvl==0)&&((*mg).cmat!=NULL))
	    {
	      /* use the coarse grid solver */
	      struct vector bcoarse;
	      bcoarse.len=lvl_vx;
	      bcoarse.n_max=lvl_vx;
	      bcoarse.V=&bl[(*ml).nlevl[lvl+1]];

	      /* projection */
	      for (i=0; i<(*P).len; i++)
		{
		  FIDX dof, child;
		  dof=(*P).V[i];
		  MLVLFINDENTRY(child, dof, lvl, *ml);
		  if (child>=0)
		    {
		      bl[child]=0.0;
		    }
		}


	      err=coarse_mat_solve( (*mg).cmat, NoTrans,
				    &bcoarse, &xi);
	      FUNCTION_FAILURE_HANDLE( err, coarse_mat_solve,
				       gen_proj_MG_tx);    
	    }
	  else
	    {
	      /* do smooths Gauss-Seidel sweeps forward */
	      err=sparse_GS_sweep_fwd( &Ks[lvl], &bi,
				       &invdiagi, smooths, &xi);
	      FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_fwd,
				       gen_proj_MG_tx);
		    
	      /* do smooths Gauss-Seidel sweeps backward */
	      err=sparse_GS_sweep_bwd( &Ks[lvl], &bi,
				       &invdiagi, smooths, &xi);
	      FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_bwd,
				       gen_proj_MG_tx);
	    }

	  
	  if (lvl>0)
	    {
	      /* compute the residual on this lvl */
	      /* compute the matrix vector product,
		 dxl=K*xl */
	      dxi.V   = &dxl[(*ml).nlevl[lvl+1]];
	      dxi.len = lvl_vx;
	      dxi.n_max = lvl_vx;
	      err=sparse_mul_mat_vec( &Ks[lvl], &xi, &dxi);
	      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec,
				       gen_proj_MG_tx);

	      /* now change dxl to rl=bl-K*xl */
	      for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
		dxl[j]=bl[j]-dxl[j];

	      /* restrict rl to r_l-1 */
	      err=mg_restrict_tx( mg, lvl, lvl-1, dxl);
	      FUNCTION_FAILURE_HANDLE( err, mg_restrict_tx,
				       gen_proj_MG_tx);
	      /* copy r_l-1 to b_l-1 */
	      for (j=(*ml).nlevl[lvl]; j<(*ml).nlevl[lvl-1]; j++)
		bl[j]=dxl[j];
	    }
	} /* end V-cycle downward */

      /* V-cycle upward */
      for (lvl=lmin+1; lvl<=lmax; lvl++)
	{ 
	  lvl_vx=(*ml).nlevl[lvl]-(*ml).nlevl[lvl+1];

	  /* apply the update computed in the lower level */
	  /* copy the update to dx */
	  for (j=(*ml).nlevl[lvl]; j<(*ml).nlevl[lvl-1]; j++)
	      dxl[j]=xl[j];


	  /* interpolate dx to lvl */
	  err=mg_interpolate_tx( mg, lvl-1, lvl, dxl);
	  FUNCTION_FAILURE_HANDLE( err, mg_interpolate_tx,
				   gen_proj_MG_tx);


	  /* apply the update to xl */
	  for (j=(*ml).nlevl[lvl+1]; j<(*ml).nlevl[lvl]; j++)
	    xl[j]+=alpha*dxl[j];

	  /* make x_l, b_l, invdiag_l accessible as vector */
	  xi.len=lvl_vx;
	  xi.n_max=lvl_vx;
	  xi.V=&xl[(*ml).nlevl[lvl+1]];

	  bi.len=lvl_vx;
	  bi.n_max=lvl_vx;
	  bi.V=&bl[(*ml).nlevl[lvl+1]];

	  invdiagi.len=lvl_vx;
	  invdiagi.n_max=lvl_vx;
	  invdiagi.V=&invdiag[(*ml).nlevl[lvl+1]];
	  
	  /* do smooths Gauss-Seidel sweeps backward */
	  err=sparse_GS_sweep_bwd( &Ks[lvl], &bi,
				   &invdiagi, smooths, &xi);
	  FUNCTION_FAILURE_HANDLE( err, sparse_GS_sweep_bwd,
				   gen_proj_MG_tx);
	} /* end V-cycle upward */
    } /* end V-cycles loop */
  
  (*mg).vccount+=vccount;



  /* copy xl to out */
  for (i=0; i<(*out).len; i++)
    {
      (*out).V[i]=xl[i];
    }

  /* apply P */
  for (i=0; i<(*P).len; i++)
    {
      (*out).V[ (*P).V[i] ] = 0.0 ;
    }

  /* done */

  return SUCCESS;
}

