/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2010, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE assem_conv_diff.c
HEADER assem_conv_diff.h

TO_HEADER:


// useful text macros
#include "feins_macros.h"
// data structures 
#include "datastruc.h"
#include "sparse_struct.h"

*/



/* function prototypes */
#include "assem_conv_diff.h"

/* prototypes of external functions */
#include <math.h>
#include "elements.h"
#include "cubature.h"
#include "sparse.h"
#include "mesh.h"
#include "feins_lapack.h"    /* link against LAPACK! */


/*FUNCTION*/
int assem_conv_diff_tx(struct sparse *K, struct vector *rhs, double *maxPe,
		       struct mesh *m, int stab, FIDX type
/* performs the assembly of the stiffness matrix K and the right hand
   side vector rhs which result from SUPG stabelised finite element
   discretisation of the convection-diffusion equation on the
   triangular mesh m, such that

           K x = rhs

   defines the (approximate) solution x=(u) of

         -eps Laplace(u) + b * grad u = f

   with eps=m.para[MC2XPANUPO], b=m.function[0] and boundary conditions as
   given in the mesh m 

   Input:  m       - the mesh
	   stab    - to indicate wether stabilization shall be used
	             for the advection term or not, stab==0: use no
	             stabilisation, stab==1: use SUPG stabilisation
           type    - type of mesh, ==1 means linear (P1) elements, 
                     ==2 means quadratic (P2) elements
           
   In/Out: K       - stiffness matrix as struct sparse, initialisation
                     and finalisation should be done by the caling
                     routine  
	   rhs     - righthand side vector, has to be initialised by
                     the calling routine

   Output: maxPe   - maximum Peclet number observed in the mesh

   Return: SUCCESS - success
           FAIL    - failure, see error message, output will not be
                     valid
*/
		){
  FIDX el, d, i, j, k, l, r, line, col;
  int  err;

  FIDX dim, bas_n, bas_n_2, vx_nr, eg_nr, el_nr,
    el_w, vx_w, eg_w, bd_w, fu_w, fc_w;

  /* for FORTRAN calls: */
  char fTrans='T', fNoTrans='N';
  double dhelp, done=1.0, dzero=0.0;
  int fdim, fbas_n; 

  FIDX l_mctxVXSTRT, l_mctxEGNOD1, l_mctxELNOD1, l_mctxBDTYPE,
    l_mctxBDEDGE;
  int cubat_order;



  struct int_data iform; /* integration formula 2d   */
  double *x_k;            /* the integration point */
  double *bk;            /* velocity vector at the integration point */
  double fx_k;           /* rhs function f(.) at the integration point */
  double *elK;           /* element stiffness matrix */
  double *Jac, *Jacinv;  /* Jacobian of the element mapping and its
			    inverse */
  double *Jinvgrad;      /* inverse Jacobian times gradphi */
  double detJac;         /* determinant of the Jacobian */

  double *phi, *gradphi;
  double eps, weight;

  double *xc;            /* the center point of the element */
  double *bc;            /* velocity at the element center */
  double h, abs_bc, heg, h_max, hbc, hbcmax, pe, beta;
                         /* values used to determine the mesh peclet
			    number and the stabilization term */ 
  double *phic, *gphic, *hphic;
                         /* the basis function values at this point */

  /****************   init ******************************************/
  dim   = (*m).dim;
  vx_nr = (*m).vx_nr;
  eg_nr = (*m).eg_nr;
  el_nr = (*m).el_nr;
  vx_w  = (*m).vx_w;
  el_w  = (*m).el_w;
  eg_w  = (*m).eg_w;
  bd_w  = (*m).bd_w;
  fu_w  = (*m).fu_w;
  fc_w  = (*m).fc_w;

  eps= (*m).para[MC2XPANUPO];

  /* get integration formula */
  /* required order: 
      type=1:  grad u * grad v     order 0
               (b' * grad u)*v     order 1+order(b)
               f v                 order 1+order(f)
               (b' * grad u)*(b' * grad v)
                                   order 0+2*order(b)
                                   ------------------
                     use:          >=order 2*2, say 4
     --------------------------------------------------
      type=2:  grad u * grad v     order 2
               (b' * grad u)*v     order 3+order(b)
               f v                 order 2+order(f)
               (b' * grad u)*(b' * grad v)
                                   order 2+2*order(b)
                                   -----------------
                     use:          >=order 2*3, say 7
  */
  switch(type)
    {
    case 1:
      l_mctxVXSTRT = MCT1VXSTRT;
      l_mctxEGNOD1 = MCT1EGNOD1;
      l_mctxELNOD1 = MCT1ELNOD1;
      l_mctxBDTYPE = MCT1BDTYPE;
      l_mctxBDEDGE = MCT1BDEDGE;
      cubat_order=4;
      break;
    case 2:
      l_mctxVXSTRT = MCT2VXSTRT;
      l_mctxEGNOD1 = MCT2EGNOD1;
      l_mctxELNOD1 = MCT2ELNOD1;
      l_mctxBDTYPE = MCT2BDTYPE;
      l_mctxBDEDGE = MCT2BDEDGE;
      cubat_order=7;
      break;
    default:
      fprintf(stderr,"assem_conv_diff_tx: unknown type=%d\n",type);
      return FAIL;
    }

  err=cubature_bases( dim, cubat_order, tria, 1, &type, &iform); 
  FUNCTION_FAILURE_HANDLE( err, cubature_bases, assem_conv_diff_tx);

  /* make phi and gradphi better accessible */
  phi    = (iform.bases[0]->phi);
  gradphi= (iform.bases[0]->gradphi);
  bas_n  = (iform.bases[0]->num_basis);

  fbas_n = (int) bas_n;
  fdim   = (int) dim;



  if (dim > 2)
    {
      /* cry */
      fprintf(stderr,
	      "assem_conv_diff_tx: dim >2 not implemented (Jacinv+?)\n");
      return FAIL;
    }

  if ( (*K).row_nr!=vx_nr )
    {
      /* cry */
      fprintf(stderr,
	      "assem_conv_diff_tx: size of K wrong???\n");
      return FAIL; 
    }
    

  bas_n_2=bas_n*bas_n;    /* bas_n^2 */

  /* allocate memory for local data */
  TRY_MALLOC( x_k, dim, double, assem_conv_diff_tx);
  TRY_MALLOC( bk, dim, double, assem_conv_diff_tx);
  TRY_MALLOC( xc, dim, double, assem_conv_diff_tx);
  TRY_MALLOC( bc, dim, double, assem_conv_diff_tx);
  TRY_MALLOC( elK, bas_n_2, double, assem_conv_diff_tx);

  TRY_MALLOC( Jac, dim*dim, double, assem_conv_diff_tx);
  TRY_MALLOC( Jacinv, dim*dim, double, assem_conv_diff_tx);

  TRY_MALLOC( Jinvgrad, dim*bas_n, double, assem_conv_diff_tx);


  /* the center of gravity of the master element is
		     (1/dim,...,1/dim) */
  xc[0]=1.0/( (double)dim);  
  for (i=1; i<dim; i++)
    {
      xc[i]=xc[0];
    }
  /* evaluate the basis functions for this point */
  err=eval_basis( dim, tria, type, 1, xc, &i, &phic, &gphic, &hphic);
  FUNCTION_FAILURE_HANDLE( err, eval_basis, assem_conv_diff_tx);
  /* free unneeded info */
  free(hphic);

  /* clear K */
  sparse_empty(K);
  /* clear rhs */
  for (i=0; i<vx_nr; i++)
    {
      rhs->V[i]=0.0;
    }

  *maxPe=0.0;

  /* loop over all elements */
  for (el=0; el<el_nr; el++)
    {
      /* determine the mesh peclet number for this element */
      /* first get xc, the coordinates of the center of the element,
	 then evaluate bc, the velocity vector there */
      for (j=0; j<dim; j++)
	{
	  xc[j]=0.0;
	  for (i=0; i<bas_n; i++)
	    xc[j]+=phic[i]
	      *(*m).vertex[(*m).elem[el*el_w+l_mctxELNOD1+i]*vx_w
			   +l_mctxVXSTRT+j];
	}
      /* velocity always given by b=m.function[0] */
      err=mesh_func_eval( m, 0, xc, dim, bc);
      FUNCTION_FAILURE_HANDLE( err, mesh_func_eval, assem_conv_diff_tx);
      
      /* determine norm of the velocity vector */
      abs_bc=0;
      for (j=0; j<dim; j++)
	abs_bc+=bc[j]*bc[j];
      abs_bc=sqrt(abs_bc);

      /* now get the length of the edge which coincides best with the
	 direction of bc */
      hbcmax=-1.0;
      h_max=-1.0;
      for (i=0; i<bas_n; i++)
	for (j=i+1; j<bas_n; j++)
	  {
#warning "assem_conv_diff: use better approximation of cell size for stabilization!"
	    FIDX nodei, nodej;
	    nodei=(*m).elem[el*el_w+l_mctxELNOD1+i];
	    nodej=(*m).elem[el*el_w+l_mctxELNOD1+j];
	    hbc=0.0;
	    for(d=0; d<dim; d++)
	      hbc+=bc[d]*( (*m).vertex[nodei*vx_w+l_mctxVXSTRT+d]
			   -(*m).vertex[nodej*vx_w+l_mctxVXSTRT+d] );
		
	    heg=0.0; /* h_edge */
	    for(d=0; d<dim; d++)
	      heg+=( (*m).vertex[nodei*vx_w+l_mctxVXSTRT+d]
		     -(*m).vertex[nodej*vx_w+l_mctxVXSTRT+d] )
		*( (*m).vertex[nodei*vx_w+l_mctxVXSTRT+d]
		   -(*m).vertex[nodej*vx_w+l_mctxVXSTRT+d] );
	    heg=sqrt(heg);
		
	    if (heg>h_max) h_max=heg; /* max(h_edge) */

	    hbc=fabs(hbc);
	    if (hbc>hbcmax)         /* max(h_edge) in direction of b */
	      {
		hbcmax=hbc;
		h=heg;
	      }
	  }
      /* this elements Peclet number Pe */
      pe=0.5*h*abs_bc/eps;
      /* update the maximum Pe */
      if (*maxPe<pe) *maxPe=pe;
      /* stabilization parameter beta */
      if (abs_bc!=0.0)
	{
#ifdef RAMAGE_NOT_TEZDUYAR
	  /* Ramage stab parameter */
	  beta=(0.5*h-eps)/abs_bc;
#else
	  /* Tezduyar stab parameter */
	  beta=(0.5*h/abs_bc)*fmin(1.0, 0.33333333*pe);
#endif
	}
      else
	{
	  beta=0.0;
	}
      if ((beta<0.0)||(stab!=1)) beta=0.0;


      /* set elK to zero */
      for (i=0; i<bas_n_2; i++) 
	{
	  elK[i]=0.0;
	}

      /* loop over all integration points */
      for (k=0; k<iform.num_points; k++)
	{
	  /* compute the Jacobian for this element */
	  /* Jac=0 */
	  for (i=0; i<dim*dim; i++)
	    Jac[i]=0.0;
	  
	  /* Jac = sum_{i=nodes} vertex(i)*gradphi_i^T */
	  for (i=0;i<bas_n; i++)
	    {
	      for (j=0; j<dim; j++)
		{
		  for (r=0; r<dim; r++)
		    {
		      Jac[j*dim+r]+= 
			(*m).vertex[(*m).elem[el*el_w+l_mctxELNOD1+i]*vx_w
				    +l_mctxVXSTRT+j]
			* gradphi[k*bas_n*dim +i*dim +r];
		    }
		}
	    }

	  /* get detJac */
	  detJac=Jac[0]*Jac[3]-Jac[2]*Jac[1];
	  
	  /* get Jacinv (here direct) */
	  Jacinv[0]=1.0/detJac*Jac[3];
	  Jacinv[1]=-1.0/detJac*Jac[1];
	  Jacinv[2]=-1.0/detJac*Jac[2];
	  Jacinv[3]=1.0/detJac*Jac[0];


	  /* Jinvgrad= Jacinv * gradphi[k,:,:] 
	     (=real world gradient T1)
	  */
	  dgemm_( &fNoTrans, &fNoTrans, &fdim, &fbas_n, &fdim,
		  &done, Jacinv, &fdim, &(gradphi[k*bas_n*dim]), &fdim,
		  &dzero, Jinvgrad, &fdim );


	  /* the real wolrd weight of this integration point */
	  weight=fabs(detJac) * iform.weights[k];

	  dhelp=weight*eps;
	  /* elK += |detJac|*weigth[k] * eps* Jinvgrad^T*Jinvgrad */
	  dgemm_( &fTrans, &fNoTrans, &fbas_n, &fbas_n, &fdim,
		  &dhelp, Jinvgrad, &fdim, Jinvgrad, &fdim,
		  &done, elK, &fbas_n );



	  /* define bk[], velocity vector of convection term */
	  for (j=0; j<dim; j++)
	    {
	      xc[j]=0.0;
	      for (i=0; i<bas_n; i++)
		xc[j]+=phi[k*bas_n+i]
		  *(*m).vertex[(*m).elem[el*el_w+l_mctxELNOD1+i]*vx_w
			       +l_mctxVXSTRT+j];
	    }
	  /* velocity always given by b=m.function[0] */
	  err=mesh_func_eval( m, 0, xc, dim, bk);
	  FUNCTION_FAILURE_HANDLE( err, mesh_func_eval, assem_conv_diff_tx);


	  /* evaluate rhs function f(x_k) */
	  err=mesh_func_eval ( m, (*m).face[el*fc_w+MCT1FCRHSF],
			       x_k, 1, &fx_k);
	  FUNCTION_FAILURE_HANDLE(err,mesh_func_eval, assem_conv_diff_tx);

	  /* elK(i,j) += int(element)(
	            sum_d ( bk[d]* (d phi_j/d x_d)) 
	            * (phi_i + beta* sum_d (bk[d] *d phi_i/d x_d)))
	     (beta>0 --> SUPG-stabilization)

	     the integration for each entry of elK and rhs :
	  */
	  for (i=0; i<bas_n; i++)
	    {
	      double b_gradphi_i=0.0;
	      for (d=0; d<dim; d++)
		{
		  b_gradphi_i+=bk[d]*Jinvgrad[i*dim+d];
		}

	      for (j=0; j<bas_n; j++)
		{
		  double b_gradphi_j=0.0;
		  for (d=0; d<dim; d++)
		    {
		      b_gradphi_j+=bk[d]*Jinvgrad[j*dim+d];
		    }
		  elK[j*bas_n+i] +=
		    b_gradphi_j *( phi[k*bas_n+i] + beta*b_gradphi_i)
		    * weight ; /* */
		}
	      /* compute rhs */
	      (*rhs).V[(*m).elem[el*el_w+i]] +=
		fx_k *( phi[k*bas_n+i] + beta*b_gradphi_i)
		* weight ;
	    }
	} /* end loop intergration points */
      
      /***************************************************************
       * elK is ready, now assemble K                                *
       ***************************************************************/

      err=sparse_add_local( K, NoTrans,
			   bas_n, &(*m).elem[el*el_w+l_mctxELNOD1], 
			   bas_n, &(*m).elem[el*el_w+l_mctxELNOD1],
			   elK, bas_n );
      FUNCTION_FAILURE_HANDLE( err, sparse_add_local, assem_conv_diff_tx);

    } /* end loop elements */

  /* printf("lvl=%d,  maxPe=%8.1e\n", (int) lvl, *maxPe); /* */

  /* free local data */
  free(phic);
  free(gphic);
  free(Jinvgrad);
  free(Jacinv);
  free(Jac);
  free(elK);
  free(bc);
  free(xc);
  free(bk);
  free(x_k);
  free_intdata (&iform);


  return SUCCESS;
}
