% ***********************************************************************
%
%    This file is part of
%    FEINS, Finite Element Incompressible Navier-Stokes solver,
%    which is expanding to a more general FEM solver and toolbox,
%    Copyright (C) 2003--2008, Rene Schneider 
%    <rene.schneider@mathematik.tu-chemnitz.de>
% 
%    This program is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
% 
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
% 
%    You should have received a copy of the GNU General Public License
%    along with this program. If not, see <http://www.gnu.org/licenses/>.
%
%    Minor contributions to this program (for example bug-fixes and
%    minor extensions) by third parties automatically transfer the
%    copyright to the general author of FEINS, to maintain the
%    possibility of commercial re-licensing. If you contribute but wish
%    to keep the copyright of your contribution, make that clear in
%    your contribution!
% 
%    Non-GPL licenses to this program are available upon request from
%    the author.
%
% ***********************************************************************


function [rpar,Aspar,bspar,Crpar,drpar]=tobst_gen(fname,nu,rpar_in,level0,level);
%
% generates a mesh file fname for a wing, with viscosity parameter nu
%
%
% return:
%   rpar   - the actual parameters of the geometry, the vector to be
%            optimised
%   Aspar  
%   bspar  - spar=Aspar*rpar+bspar gives the parameter in the meshfile
%            in dependance on the actual parameters
%   Crpar
%   drpar  - Crpar*rpar+drpar >=0 form the constraints on the actual
%            parameters 
%



% initial profile is NACA0040 :
t=0.4; % t=0.xx  where xx are the last two digits of the NACA 4 digits

dx=0.001;
x=(0:dx:1)';

bbx0=-2; bbx1=6; bby=2; 
%bbx0=-1; bbx1=4; bby=1; 

global na; global nb;
if (isempty(na)||isempty(nb))
  error('na or nb not set (global vars)')
end
%na=1; nb=1;
%na=2; nb=4;
%na=10; nb=20;


xa=0.1;xb=0.9;

dxa=xa/na; dxb=(xb-xa)/nb;
xp=[(0:1/na:1).^2*xa,xa+dxb:dxb:xb,1]';

b_val=0.02;
xd=[0,xa,xb,1]';
ydt=[0,b_val,b_val,0];

% initial profile is NACA0040 :
yt  = t*5*(0.2929*sqrt(x)-0.1260*x-0.3516*x.^2+0.2843*x.^3-0.1015*x.^4);
ypt = t*5*(0.2929*sqrt(xp)-0.1260*xp-0.3516*xp.^2+0.2843*xp.^3-0.1015*xp.^4);

warning('off','MATLAB:divideByZero');
dypt= t*5*(0.2929*0.5./sqrt(xp)-0.1260-0.3516*2*xp+0.2843*3*xp.^2 ...
    -0.1015*4*xp.^3);
warning('on','MATLAB:divideByZero');

yu=yt;
yl=-yt;

clf;

plot(xd,ydt,'rd-'); hold on; plot(xd,-ydt,'rd-')
plot(x,yu); plot(x,yl)
plot(xp,ypt,'ks-'); plot(xp,-ypt,'ks-')
axis([0,1,-1,1])

% xp contains np=1+na+nb+1 points
% parameters/mesh points will be np,np-1,...,2,1,2,...,np
np=2+na+nb;
ypt(np)=0;

% vertices = first parameters for the b-splines
vx=[xp([np:-1:1,2:1:np]),[ypt(np:-1:1);-ypt(2:1:np)] ];

% directions = second parameters for the b-splines
dxp=zeros(np,1);
dyp=zeros(np,1);
for i=np-1:-1:2
  dxp(i)=0.45*min([xp(i+1)-xp(i),xp(i)-xp(i-1)]);
end
dxp(np) = 0.45*(xp(np)-xp(np-1));
dxp(1)  = 0;

dyp(1)  = 0.45*ypt(2);

dyp(2:np) = dxp(2:np).*dypt(2:np);


nspar=2*2*(np-1)+2*2*(np-1)+2;

% $$$ spar=zeros(nspar,1);
% $$$ % the first shape parameter are just the vertices
% $$$ for i=1:2*(np-1)
% $$$   spar((i-1)*2+1)=vx(i,1);
% $$$   spar((i-1)*2+2)=vx(i,2);
% $$$ end
% $$$ % the direction parameter are derived from dxp and dyp
% $$$ for i=1:np
% $$$   spar( (i-1)*2  +2*2*(np-1)+1)= -dxp(np-i+1);
% $$$   spar( (i-1)*2+1+2*2*(np-1)+1)= -dyp(np-i+1);
% $$$ end
% $$$ for i=np+2:2*np
% $$$   spar( (i-2)*2  +2*2*(np-1)+1)=  dxp(i-np);
% $$$   spar( (i-2)*2+1+2*2*(np-1)+1)= -dyp(i-np);
% $$$ end

nrp_p= na+nb;   % real position paramters
nrp_d= na+nb+2; % real direction parameters

Aspar=zeros(nspar,nrp_p+nrp_d);
bspar=zeros(nspar,1);

rpar=zeros(nrp_p+nrp_d,1);

% the first shape parameter are just the vertices,
% all x-ones are fixed
for i=1:2*(np-1)
  bspar((i-1)*2+1)=vx(i,1);
end
% tail and tip y-pos are fixed as well
bspar(2)             = vx(1,2);
bspar(2*(na+nb+1)+2) = vx(na+nb+1+1,2);

% the remaining y-pos are parameters
for i=1:nb+na
  rpar(i) = vx(i+1,2);              % parameter

  Aspar(2*i+2,i)               =  1;  % top half 
  Aspar(2*(2*(nb+na+1)-i)+2,i) = -1;  % bootom half 
end

% base for the numbering of dir parameters
d0= 2*2*(np-1);

% the direction parameter are derived from dxp and dyp
% all x-dir are fixed
for i=1:na+nb+1
  bspar((i-1)*2            +1 +d0) = -dxp(na+nb+2-i+1); % top half
  bspar((2*(na+nb+1)-i+1)*2+1 +d0) =  dxp(na+nb+2-i+1); % bottom half
end
% all y-dir are parameter, appart from tip
for i=1:na+nb+1
  rpar(nrp_p+i)                    =  dyp(na+nb+2-i+1); % parameter
  
  Aspar((i-1)*2            +2 +d0,nrp_p+i) = -1; % top half
  Aspar((2*(na+nb+1)-i+1)*2+2 +d0,nrp_p+i) = -1; % bottom half
end
i=na+nb+2;
rpar(nrp_p+i)                 =  dyp(na+nb+2-i+1); % parameter
Aspar((i-1)*2         +2 +d0,nrp_p+i) = -1; % tip


if (~isempty(rpar_in))
  rpar=rpar_in;
end

spar=Aspar*rpar+bspar;

% now the constraints, there are 
nconst= na+nb +2*na+2*nb+2;

Crpar=zeros(nconst, nrp_p+nrp_d);
drpar=zeros(nconst, 1);


% b_val is the min thickness at the center

% y-pos constraints   xa<=x<=xb
for i=1:nb+1
  Crpar(i, i) = 1;      % y-pos
  drpar(i)    = -b_val; % >=b_val

  %write=i
end
for i=2:na
  x=vx(i+nb+1,1);
  Crpar(i+nb,i+nb)= 1;           % y-pos
  drpar(i+nb)     =-b_val/xa*x;  % >= linear

  %write=i+nb
end

% dir constraints
cd0=na+nb;
% first constraint: slope at the end is >= slope to get b
Crpar(1+cd0,nrp_p+1)=1.0/bspar(1+d0);
drpar(1+cd0)        =-b_val/(1.0-xb);

% next ones in the area xa<=x<=xb
% y-pos+y-dir>=b_val
for i=2:nb+2
  Crpar(i     +cd0,      i-1)= 1;        % y-pos
  Crpar(i     +cd0,nrp_p+i  )= 1;        % y-dir
  drpar(i     +cd0)          =-b_val;
  
  %write=i+cd0

  Crpar(i+nb+1+cd0,      i-1)= 1;        % y-pos
  Crpar(i+nb+1+cd0,nrp_p+i  )=-1;        % y-dir
  drpar(i+nb+1+cd0)          =-b_val;

  %write=i+nb+1+cd0
end

% next ones in the area 0<x<xa
% y-pos+y-dir>=b_val
for i=2:na
  x=vx(i+nb+1,1);
  Crpar(i+2*(nb+1)     +cd0,      nb+i  )= 1;        % y-pos
  Crpar(i+2*(nb+1)     +cd0,nrp_p+nb+i+1)= 1;        % y-dir
  drpar(i+2*(nb+1)     +cd0)             =-b_val/xa*x; 

  %write=i+2*(nb+1)+cd0
  
  Crpar(i+2*(nb+1)+na-1+cd0,      nb+i  )= 1;        % y-pos
  Crpar(i+2*(nb+1)+na-1+cd0,nrp_p+nb+i+1)=-1;        % y-dir
  drpar(i+2*(nb+1)+na-1+cd0)             =-b_val/xa*x; 
  
  %write=i+2*(nb+1)+na-1+cd0
end

Crpar(2*(nb+na)+2+cd0,nrp_p+nrp_d)=1;        % y-dir
drpar(2*(nb+na)+2+cd0)            =-b_val/2; % >=b_val/2
%write=2*(nb+na)+2+cd0

% last: tip y-dir >= 0.5*b_val


% test constraints
constr=Crpar*rpar+drpar;
if sum(constr<0)>1
  violated_constr=(constr<0)'*constr
end


% write the meshfile
mf=fopen(fname,'w');
fprintf(mf,[
  '<!-- mesh file for flow around an obstacle \n',...
  '-->\n',...
  '<header>\n',...y
  '  <format    0 >\n',...
  '  <dim       2 >\n',...
  '  <problem   2 >\n',...
  '  <meshgen   1 >\n',...
  '  <vertex    %d >\n',...
  '  <elements  0 >\n',...
  '  <boundary  %d >\n',...
  '  <holes     1 >\n',...
  '  <pcsurf    %d >\n',...
  '  <pcvol     0 >\n',...
  '  <pccrit    2 >\n',...
  '  <function  2 >\n',...
  '  <parameter 1 >\n',...
  '  <shape_seg %d >\n',...
  '  <shape_par %d >\n',...
  '</header>\n',...
  '\n'],...
  2*(np-1)+4, 2*(np-1)+4, 2*(np-1), 2*(np-1),    nspar);
fprintf(mf,[
    '<solver_settings>\n',...
    '  <refine_ini      %d >\n',...
    '  <refine_type     1 >\n',...
    '  <refine_steps    %d >\n',...
    '  <adap_mark       0 >\n',...
    '  <adap_mark_par   0.6 >\n',...
    '  <solver          1       >\n',...
    '  <solver_atol     1.0e-12 >\n',...
    '  <solver_ini_rtol 1.0e-6  >\n',...
    '  <solver_ref_rtol 1.0e-2  >\n',...
    '  <write_mesh      1 >\n',...
    '  <write_ssegs     1 >\n',...
    '</solver_settings>\n'],...
  level0, level);
fprintf(mf,[
  '\n',...
  '<!-- syntax of a vertex entry is:\n',...
  '       id x y [z]\n',...
  '-->\n',...
  '<vertex>\n'],...
  2*(np-1)+4, 2*(np-1)+4, 2*(np-1), 2*(np-1),    nspar);

%   vx_nr        bd_nr       ps_nr     sg_nr     sp_nr
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 0, bbx0,  bby);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 1, bbx0, -bby);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 2, bbx1, -bby);
fprintf(mf,'  %3d  %22.15e  %22.15e\n', 3, bbx1,  bby);
for i=1:2*(np-1)
  fprintf(mf,'  %3d  %22.15e  %22.15e\n',i+3, vx(i,1), vx(i,2) );
end

fprintf(mf,[
  '</vertex>\n',...
  '\n',...
  '<!-- syntax of a element entry is:\n',...
  '       id type func nod1 nod2 ... nodk\n',...
  '-->\n',...
  '<elements>\n',...
  '</elements>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a boundary entry is:\n',...
  '       id bctype type func orient sseg nod1 ... nodk\n',...
  '-->\n',...
  '<boundary>\n']);

% outer boundary
for i=1:4
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i-1, 1,     0,   1,   1,    -1,  i-1, mod(i,4)  );
%     id bctype type func orient sseg nod1 ... nodk
end
% inner all around
for i=1:2*(np-1)
  fprintf(mf,  '   %3d   %d   %d   %2d  %+d   %3d      %3d  %3d\n',...
      i+3, 1,     0,   0,  -1,   i-1,  i+3, mod(i,2*(np-1))+4  );
%     id bctype type func orient sseg nod1 ... nodk
end

fprintf(mf,[
  '</boundary>\n',...
  '\n',...
  '<!--  syntax of a holes entry is:\n',...
  '       id x y [z]\n',...
  '-->\n',...
  '<holes>\n',...
  '  0    %22.15e   %22.15e\n',...
  '</holes>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a pcsurf entry is:\n',...
  '       id c_id orient nod1 ... nodk\n',...
  '-->\n',...
  '<pcsurf>\n'], (xa+xb)/2,0);

% inner all around
for i=1:2*(np-1)
  fprintf(mf,  '   %3d   %d   %+d     %3d  %3d\n',...
      i-1,  0,   -1,   i+3, mod(i,2*(np-1))+4  );
%     id  c_id orient nod1 ... nodk
end

fprintf(mf,[
  '</pcsurf>\n',...
  '\n',...
  '<!-- syntax of a pcvol entry is:\n',...
  '       id c_id elem\n',...
  '-->\n',...
  '<pcvol>\n',...
  '</pcvol>\n',...
  '   \n',...
  '\n',...
  '<!-- syntax of a pccrit entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<pccrit>\n',...
  '   0    1   1.0   0.0\n',...
  '   1    3\n',...
  '</pccrit>\n',...
  '\n',...
  '\n',...
  '\n',...
  '<!-- syntax of a function entry is:\n',...
  '       id type data1 ... datak\n',...
  '-->\n',...
  '<function>\n',...
  '   0  100   0.0   0.0\n',...
  '   1  100   1.0   0.0\n',...
  '</function>\n',...
  '\n',...
  '\n',...
  '<!-- syntax of the parameter entry is:\n',...
  '       data1 ... datak\n',...
  '-->\n',...
  '<parameter>\n',...
  '   %8.1e \n',...
  '</parameter>\n',...
  '\n\n',...
  '<!-- syntax of the shape_segm entry is:\n',...
  '       id   type  nod1   para1 ... parak\n',...
  '-->\n',...
  '<shape_seg>\n'],nu);
for i=1:2*(np-1)
  fprintf(mf, '  %3d 2 %3d    %3d  %3d  %3d  %3d  %3d  %3d  %3d  %3d\n',... 
      i-1,             i+3,  (i-1)*2, (i-1)*2+1, ...
      mod(i,2*(np-1))*2, mod(i,2*(np-1))*2+1,...
      (i-1)*2  +2*2*(np-1),(i-1)*2+1 +2*2*(np-1),...
      i    *2  +2*2*(np-1),    i*2+1 +2*2*(np-1) );
  %   id       type  nod1      x0      y0 
  %            x1            y1
  %           sx0            sy0  
  %           sx1            sy1  
end

fprintf(mf,[
  '</shape_seg>\n',...
  '\n',...
  '<!-- syntax of the shape_para entry is:\n',...
  '       id  data\n',...
  '-->\n',...
  '<shape_par>\n']);

for i=1:nspar
  fprintf(mf, '   %3d   %22.15e\n', i-1,  spar(i)); 
end

fprintf(mf,[
  '</shape_par>\n',...
  '\n' ]);

fclose(mf);

return;
