/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
/*
FILE test_stokes2.c
HEADER solver.h

TO_HEADER:

*/

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "assembly.h"
#include "stokesassem.h"
#include "sparse.h"
#include "lin_solver.h"
#include "navsto_struct.h"
#include "stokes_aux.h"
#include "navsto_aux.h"
#ifndef __USE_GNU
#define __USE_GNU
#endif
#include <string.h>
#include <math.h>

#include <sys/types.h>
#include <sys/time.h>

#define TIMEGET {\
      gettimeofday(&tv,&tz); \
      sec=tv.tv_sec; \
      musec=tv.tv_usec; \
      ti=((double)sec+((double)musec)*1e-6); }

int main(int argc, char *argv[])
{
  struct navsto_matrix K;
  struct vector rhs, x, y, sol, soli;
  struct mesh msh1, msh2;

  struct timeval tv;
  struct timezone tz;
  int sec, musec;
  double t1, t2, t3, ti;

  char solver[15], *buffer;
  int  err, iter, ihlp;
  FIDX dim, i, j, lvl, level, levelmax, coarseset, index;

  double nu, resi, normy;

  int m=30;
  double this_eps=1e-2;

  if (argc>=4)
    {
      printf("meshfile: %s\n", argv[1]);
    }
  else
    {
      printf("usage: %s  meshfile.f1m lvl0 lvl1 \n", argv[0]);
      return FAIL;
    }
  levelmax   = -1;
  if (sscanf(argv[3],"%d",&ihlp)==1) levelmax =(FIDX) ihlp;
  printf("levelmax=%d\n", levelmax);

  for (level=0; level<=levelmax; level ++)
      { /* never do more than level 8 ! */

  err=mesh_read_file_t1( &msh1, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  dim=msh1.dim;

  /*printf("file read, vx_nr=%3d, eg_nr=%3d, fc_nr=%3d\n", msh1.vx_nr,
    msh1.eg_nr, msh1.fc_nr);*/

  err=mesh_t1_to_t2( &msh1, &msh2 );
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

  for (lvl=0; lvl<0; lvl++) 
    {
      err=mesh_refine_uniform_t2(&msh2);
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
    }

  /* reset hierarchy */
  msh2.hi_nr=0;
  msh2.eh_nr=0;
  for (j=0; j<msh2.eg_nr; j++)
    {
      msh2.edge[j*msh2.eg_w+MCT2EGLVL]=-1;
    }

  err=navsto_matrix_init( &K, dim, level+1 );
  FUNCTION_FAILURE_HANDLE( err, navsto_matrix_init, main);

  TIMEGET;
  t1=ti;

  for (lvl=0; lvl<=level ; lvl++)
    {
      if (lvl>0)
	{
	  err=mesh_refine_uniform_t2( &msh2 );
	  FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
	}

      /* printf("refined,   vx_nr=%3d, eg_nr=%3d, fc_nr=%3d\n", msh2.vx_nr,
	 msh2.eg_nr, msh2.fc_nr);*/

      err=stokes_assem_t21( &K, &rhs, &x, &msh2, lvl, 0);
      FUNCTION_FAILURE_HANDLE( err, assem_stokes_t2, main);

      if (lvl==0)
	{
	  err=coarse_mat_set( &K.Fs[0], K.bn_nr, K.nodes, 1,
			      &K.cmat );
	  FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, 
				   main );
	}

      /* free unnecessary data */
      if (lvl<level)
	{
	  vector_free(&rhs);
	  vector_free(&x);
	}
    }


  TIMEGET;
  t2=ti;

  vector_alloc( &y, x.len );

  resi=0.0;
  iter=0;
  
  for (i=0; i<5; i++) K.innercount[i]=0;

  /* strcpy(solver,"PCR_no");
     err=PCR( x.len, 2, 1e-5, 1, &x, &resi, &iter, stokes_matrix_tim_vec, 
     stokes_projector_no_precon2, &K, &rhs, NULL ); /* */

  /*strcpy(solver,"PCR_w");
    P.innereps=1e-10;
    err=PCR( x.len, 2, 1e-8, 1, &x, &resi, &iter, stokes_matrix_tim_vec, 
    stokes_projector_w_precon, &K, &rhs, NULL ); /* */

  
  /* strcpy(solver, "GMRES_no");
     err=GMRES( 200, 100, 2, 1e-10, 1, &x, &resi, &iter,
     stokes_matrix_tim_vec, stokes_projector_no_precon2,
     &K, &rhs, NULL ); /* */

  K.innereps=this_eps;
  for (i=0; i<5; i++) K.innersteps[i]=-1;
  K.mg->smooths=1;
  //  K.mg->vcycles=-30;
  K.mg->vcycles=1;
  K.mg->vccount=0;
  K.mg->stop_eps=K.innereps;

  /* strcpy(solver,"PCR_w");
     err=PCR( 100, 2, 1e-5, 1, &x, &resi, &iter, stokes_matrix_tim_vec, 
     stokes_projector_w_precon, &K, &rhs, NULL ); /* */

  strcpy(solver, "GMRES_w");
     err=GMRES( m, 100, 2, 1e-5, 1, &x, &resi, &iter,
     stokes_matrix_tim_vec, stokes_projector_w_precon,
     &K, &rhs, NULL); /* */

  TIMEGET;
  t3=ti;

  if (err==FAIL) exit(0);
  /* test true residual */
  /* y=K*x */
  stokes_matrix_tim_vec( &K, &x, &y);
  /* y=K*x-rhs, get norm(y) */
  normy=0.0;
  for (i=0; i<x.len; i++)
    {
      rhs.V[i]=y.V[i]-rhs.V[i];
    }
  /* apply the projector */
  err=navsto_projector_no_precon( &K, &rhs, NULL, &y);
  FUNCTION_FAILURE_HANDLE(err, navsto_projector_no_precon, main);
  for (i=0; i<x.len; i++)
    {
      normy+= y.V[i]*y.V[i];
    }
  normy=sqrt(normy); /* */
  
  printf("%8s: N=%6d, it=%4d, err=%2d, |res_A|=%12.6e, |res_B|=%12.6e, ",
	 solver, (int) x.len, iter, err, resi, normy); /* */
  /* printf("%8s: N=%6d, it=%4d, err=%2d, |res_B|=%12.6e, ",
     solver, x.len, iter, err, normy); /* */
  printf("sub_CG_it: F_1=%5d, F_2=%5d, M=%5d ", K.innercount[0],
	 K.innercount[1], K.innercount[2]); /* */
  printf("time: a=%8.2e  s=%8.2e  ", t2-t1, t3-t2); /* */
  printf("\n");
     
  vector_alloc( &sol, (dim+1)*K.vx_nr );
  for (i=0; i<dim*K.vx_nr; i++)
    { sol.V[i]=x.V[i]; }
  for (i=0; i<K.vx_nr; i++)
    { sol.V[dim*K.vx_nr+i]=0.0; }
  for (i=0; i<K.vx_nr; i++)
    { 
      if (K.pdof[i]!=-1)
	{ sol.V[dim*K.vx_nr+i]=x.V[dim*K.vx_nr+K.pdof[i]]; }
    }
  /* correct the pressure values of the edge-mid-nodes */
  {
    FIDX eg, nod1, nod2, nodm;
    for (eg=0; eg<msh2.eg_nr; eg++)
      {
	if (msh2.edge[eg*msh2.eg_w+MCT2EGCHL1]==-1)
	  {
	    nod1=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1  ];
	    nod2=msh2.edge[eg*msh2.eg_w+MCT2EGNOD1+1];
	    nodm=msh2.edge[eg*msh2.eg_w+MCT2EGNODM  ];
	    
	    sol.V[2*msh2.vx_nr+nodm]= 0.5 * 
	      (sol.V[2*msh2.vx_nr+nod1] + sol.V[2*msh2.vx_nr+nod2]); 
	  }
      }
  }

  /* write the solution with grape */
  soli.V=&sol.V[0];
  soli.len=(dim+1)*msh2.vx_nr;
  soli.n_max=(dim+1)*msh2.vx_nr;
  err=mesh_write_solution_grape_t2( &msh2, &soli, dim+1, 20,
				    "visual/stokes_grape_" );
  FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_grape_t2,
			   test_stokes); /* */

  /* reference solution */
  TRY_MALLOC(buffer, strlen(argv[1])+7, char, main);
  strcpy(buffer, argv[1]);
  strcat(buffer,"_stokes");
  err=vector_write_file(&x, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);

  /* write the velocity components 
     soli.V=&sol.V[0];
     soli.len=2*K.vx_nr;
     soli.n_max=2*K.vx_nr;
     err=mesh_write_solution_exp_t2( &msh2, &soli, 2, 20,
     "visual/2d_mesh_v" );
     FUNCTION_FAILURE_HANDLE( err, mesh_write_solution_exp_t2, main); /* */

  /* */
  navsto_matrix_free(&K);
  vector_free(&rhs);
  vector_free(&x);
  vector_free(&y);
  vector_free(&sol);
  mesh_free(&msh1);
  mesh_free(&msh2);

  } /* */


  return 1;
}
