/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
#include "navsto_solver.h"

#include <unistd.h>
#include <string.h>

  /* for gettimeofday */
#include <sys/types.h>
#include <sys/time.h>


int main(int argc, char *argv[])
{
  struct mesh msh1;

  struct vector ref_sol;
  struct vector pccrit;
  struct vector *DIDB;

  char *buffer;

  int  err, ihlp;
  FIDX i, level, level0, lvl;

  if (argc<3+1) 
    {
      printf("\n"
	     "%s: not enough arguments\n\n"
	     "usage:\n\n"
	     "   %s meshfile.f1m lvl0 lvl1 [para_file]\n"
	     "\n", argv[0], argv[0]);
      printf("       lvl0 - number of refinements before seting base mesh\n"
	     "       lvl1 - total number of refinements, "
	     "lvl1>lvl0 is required\n"
	     "  para_file - if provided the shape parameters in the\n"
	     "              mesh file are overriden\n"
	     "\n");
      printf("The computed solution will be in meshfile.f1m_sol,\n"
	     "the criteria in meshfile.f1m_crit and the sensitivities\n"
	     "in meshfile.f1m_grad\n");
      return FAIL;
    }

  printf("meshfile: %s      STABTYPE:%d ", argv[1], STABTYPE);
#ifdef RAMAGE_NOT_TEZDUYAR
  printf("  stab-par: Ramage\n");
#else
  printf("  stab-par: Tezduyar\n");
#endif
  printf("          FIDX size= %d     pointer size = %d\n",
	 sizeof(FIDX), sizeof(FIDX *) );

  
  TRY_MALLOC(buffer, strlen(argv[1])+10, char, main);

  level0  = -1;
  level   = -1;
  if (sscanf(argv[2],"%d",&ihlp)==1) level0=(FIDX) ihlp;
  if (sscanf(argv[3],"%d",&ihlp)==1) level =(FIDX) ihlp;

  if ((level0<0)||(level<=level0))
    {
      fprintf(stderr, "test_navsto_solver:"
	      "lvl0 and/or lvl1 invalid\n"
	      "(call without arguments for help)\n");
      return FAIL;
    }


  
  /* read the reference solution */
  strcpy(buffer,argv[1]);
  strcat(buffer,"_stokes");
  err=vector_read_file(&ref_sol, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_read_file, main);
  /* */
  
  err=mesh_read_file_t1( &msh1, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);


  for (lvl=0; lvl<level0 ; lvl++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  for (i=0; i<msh1.eg_nr; i++)
    {
      msh1.edge[i*msh1.eg_w+MCT1EGLVL]=-1;
    }

  
  /* allocate space for the performance criteria */
  err=vector_alloc(&pccrit, msh1.pc_nr);
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  /* allocate memory for the sensitivities */
  TRY_MALLOC(DIDB, msh1.pc_nr, struct vector, main);


  /* solve */
  err=navsto_solver_bcontrol(&msh1, level-level0, &ref_sol, &pccrit, DIDB);
  FUNCTION_FAILURE_HANDLE(err, navsto_solver, main);

  /* write results to files */
  /* criteria */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_crit");
  err=vector_write_file(&pccrit, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_write_file, main);

  /* derivatives */
  strcpy(buffer, argv[1]);
  strcat(buffer,"_grad");
  err=vector_n_write_file(msh1.pc_nr, DIDB, buffer);
  FUNCTION_FAILURE_HANDLE( err, vector_n_write_file, main);


  vector_free(&pccrit);
  vector_free(&ref_sol);

  for (i=0; i<msh1.pc_nr; i++)
    {
      vector_free(&DIDB[i]);
    }
  free(DIDB);

  mesh_free(&msh1);
  free(buffer);

  return SUCCESS;
}
