/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "assembly.h"
#include "gen_aux.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

#include <unistd.h>
#include <string.h>

#define K_LINE_DENSE 27

int main(int argc, char *argv[])
{

  struct sparse *Ks_linel;
  struct projector1 P_linel;
  struct coarse_mat K_linel_inv;

  struct vector rhs_linel, x_linel, nodem, vx_old;
  struct mesh msh1;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;

  int  err;
  FIDX dim, vx_nr, i, j, d;

  FIDX level0, level, lvlm, lvl;
  int ihlp;

  double mu, lambda, alpha;
  int stiffening;

  /* Lame canstants for steel */
  mu         = 7.7e4;  /* 7.7e4; */
  lambda     = 1.15e5; /* 1.15e5; */
  stiffening = 0;
  /* mark stuff */
  /* mu     = 0.1; lambda = 100; /* */

  if (argc>=1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* set the goals and methods */
  level0  = 0;
  printf("level0=? "); if (scanf("%d",&ihlp)==1) level0=ihlp;
  level   = 1;
  printf("level=? "); if (scanf("%d",&ihlp)==1) level=ihlp;

  lvlm=level-level0;

  alpha=1e-1;
  printf("alpha=? "); scanf("%lg",&alpha);
  printf("\nalpha= %+8.1e \n", alpha);

  dim=msh1.dim;

  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }

  
  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks_linel,  lvlm+1, struct sparse, main);
  
  /* assemble the coarse meshes + refine */
  for (lvl=0; lvl<lvlm; lvl++)
    {
      vx_nr=msh2.vx_nr; 

      err=sparse_alloc( &Ks_linel[lvl], dim*vx_nr, dim*K_LINE_DENSE);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);

      err=assem_lame_tx( &Ks_linel[lvl], NULL, &msh2,
			 lambda, mu, stiffening, 1, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

      err=mesh_refine_uniform_t2( &msh2 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
    }
     
  vx_nr=msh2.vx_nr; 

  err=sparse_alloc( &Ks_linel[lvlm], dim*vx_nr, dim*K_LINE_DENSE);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);

  err=projector1_alloc( &P_linel, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);

  /* assemble the system on the fine mesh*/
  err=assem_lame_tx( &Ks_linel[lvlm], &P_linel, &msh2,
		     lambda, mu, stiffening, 1, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

  /* define the multigrid data */
  err=multilvl_init_tx( &msh2, dim, &ml, 2);
  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);

  err=mg_init_tx( Ks_linel, &msh2, &ml, &mg, &P_linel);
  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);

  /* generate the coarse grid matrix */
  {
    struct coarse_mat *cmat;
    FIDX *coarse_bc_nodes;
    FIDX n_c_bc=0;

    TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
    TRY_MALLOC(coarse_bc_nodes, P_linel.len, FIDX, main);
     
    for (i=0; i<P_linel.len; i++)
      {
	FIDX dof, child;
	dof=P_linel.V[i];
	MLVLFINDENTRY(child, dof, 0, ml);
	if (child>=0)
	  {
	    coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
	    n_c_bc++;
	  }
      }
    printf("n_c_bc=%d\n", (int) n_c_bc);

    err=coarse_mat_set( &Ks_linel[0], n_c_bc, coarse_bc_nodes,
			1, cmat );
    FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
    mg.cmat=cmat;
    
    free(coarse_bc_nodes);
  } /* */
  
  
  mg.vcycles=1;
  mg.smooths=1;
  if (stiffening==0)
    mg.CGC_scale=0.25;
  else
    mg.CGC_scale=1.0;

  {
    /* write the initial mesh */
    struct vector step;
    FILE *mark;

    err=vector_alloc(&step, vx_nr);
    FUNCTION_FAILURE_HANDLE(err, vector_alloc, main);

    for (i=0; i<step.len; i++)
      step.V[i]=0.0;
    err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_ini", 2);
    FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

    /* touch the marker file */
    mark=fopen("visual/2d_new_mesh_mark", "w");
    fprintf(mark, "next please!\n");
    fclose(mark);
    vector_free(&step);

    sleep(3);
  }


  err=vector_alloc( &vx_old, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &x_linel, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &rhs_linel, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  /* copy old vertices */
  for (d=0; d<dim; d++)
    for (i=0; i<vx_nr; i++)
      {
	vx_old.V[d*vx_nr+i]=msh2.vertex[i*MCT2VXLN+MCT2VXSTRT+d];
      }

  /* set vectors to zero */
  for (i=0; i<dim*vx_nr; i++)
    {
      x_linel.V[i]  =0.0;
      nodem.V[i]    =0.0;
      rhs_linel.V[i]=0.0;
    }


  /* sparse_mat_write_file( &K_linel, "visual/Klinel.txt"); /* */


  /* now define the BC-data for the linear-elasticity problem */
  for (i=0; i<P_linel.len; i++)
    {
      FIDX dof, node, ndim;
      double xn,yn;

      dof  = P_linel.V[i];
      node = dof%vx_nr;
      ndim = dof/vx_nr;

      xn = vx_old.V[      node];
      yn = vx_old.V[vx_nr+node];

      if (1==0) 
	{ /* general nonlinear data */
	  if (ndim==0)
	    nodem.V[dof]=alpha*0.6*(xn+yn)*(xn+yn);
	  else
	    nodem.V[dof]=alpha*sin(2.3*(xn+yn));
	}
      else
	{ /* taylored for obstacle problem */
	  if (((xn>0.9)&&(xn<1.4))&&((yn>0.3)&&(yn<0.7)))
	    { /* the obstacle itself */
	      if (ndim==0)
		nodem.V[dof]=0.0; /* don't move in x direction */
	      else
		{
		  double fy;
		  int flipp=1;
		  if (yn<0.5) flipp=-1;

		  /* fy=(sin((xn-1.0)*M_PI/0.3)
		     +0.25*sin(2*(xn-1.0)*M_PI/0.3)
		     +0.115*sin(4*(xn-1.0)*M_PI/0.3));
		     if (xn<=1.1)
		     fy=0.1*sqrt(abs(1.0000001-(xn-1.1)*(xn-1.1)/(0.1*0.1)));
		     else if (xn<=1.12)
		     fy=0.1;
		     else
		     fy=0.1*sqrt(1.0000001-(xn-1.12)*(xn-1.12)
		     /((1.3-1.12)*(1.3-1.12)));
		     nodem.V[dof]=0.5
		     +flipp*(alpha*fy)-yn;*/
		     
		  fy=(yn-0.5)*sin(2*(xn-1)/0.3*M_PI);
		  nodem.V[dof]=flipp*(alpha*fy);
		}
	    }
	  else
	    { /* the outer boundary */
	      nodem.V[dof]=0.0;
	    }
	}
    }


  /* solve the equation system */
  if (0==1) /* variant 1: direct solve (good for coarse mesh) */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks_linel[lvlm], P_linel.len, P_linel.V,
			  1, &K_linel_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* get the rhs */
      /* rhs=K*x */
      err=sparse_mul_mat_vec( &Ks_linel[lvlm], &nodem, &rhs_linel);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, main);  

      /* project rhs */
      for (i=0; i<P_linel.len; i++)
	{
	  FIDX dof = P_linel.V[i];
	  rhs_linel.V[dof]=0.0;
	}
      err=coarse_mat_solve( &K_linel_inv, NoTrans, &rhs_linel, &x_linel);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      /* construct the step */
      for (i=0; i<dim*vx_nr; i++)
	{
	  nodem.V[i]-=x_linel.V[i];
	}

      coarse_mat_free(&K_linel_inv);
    }
  else /* variant 2: PCG */
    {
      int iter;
      double resi;

      struct bpxdata bpx;

      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh1;
      bpx.P   = &P_linel;

      /* rhs is just zero, use nodem directly */
      /*err=PCG( 10000, 2, 1e-5, 1, &nodem, &resi, &iter,
	sparse_mul_mat_vec, projector1_no_precon,
	&K_linel, &rhs_linel, &P_linel); /* */
      err=PCG( 10000, 2, 1e-5, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks_linel[lvlm], &rhs_linel, &mg); /* */
      /* err=PCG( 10000, 2, 1e-5, 1, &nodem, &resi, &iter,
	 sparse_mul_mat_vec, gen_proj_bpx_tx,
	 &Ks_linel[lvlm], &rhs_linel, &bpx); /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, main);
      
      printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
      bpx_free(&bpx);
    }
		  
  /* the node movement computation is done, allpy it to the mesh */
  for (i=0; i<vx_nr; i++)
    for (d=0; d<dim; d++)
      {
	msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += nodem.V[d*vx_nr+i];
      }


  {
    /* write the resulting mesh */
    struct vector step;
    FILE *mark;

    err=vector_alloc(&step, vx_nr);
    FUNCTION_FAILURE_HANDLE(err, vector_alloc, main);

    for (i=0; i<step.len; i++)
      step.V[i]=0.0;
    err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_def", 2);
    FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

    /* touch the marker file */
    mark=fopen("visual/2d_new_mesh_mark", "w");
    fprintf(mark, "next please!\n");
    fclose(mark);
    vector_free(&step);
  }
  


  /* free some stuff */
  mg_free(&mg);
  multilvl_free(&ml);

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks_linel[lvl]);
  free(Ks_linel);

  projector1_free(&P_linel);

  
  vector_free(&rhs_linel);
  vector_free(&nodem);
  vector_free(&vx_old);
  vector_free(&x_linel);

  mesh_free(&msh2);
  mesh_free(&msh1);

  return SUCCESS;
}
