/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/

#include <math.h>
#ifdef fmax
#warning "math.h"
#endif

#include "feins_macros.h"
#include "datastruc.h"
#include "mesh.h"
/*#include "meshdetails.h"*/
#include "sparse.h"
#include "lin_solver.h"
#include "assembly.h"
#include "gen_aux.h"
#include "mesh_deform.h"

#ifndef __USE_GNU
#define __USE_GNU
#endif

#include <unistd.h>
#include <string.h>

#define K_LINE_DENSE 27

int main(int argc, char *argv[])
{

  struct sparse *Ks;
  struct projector1 P;
  struct coarse_mat K_inv;

  struct vector rhs, x, nodem, vx_old;
  struct mesh msh1;
  struct mesh msh2;

  struct multilvl ml;
  struct mgdata mg;

  int  err;
  FIDX dim, vx_nr, i, j, d;

  FIDX level0, level, lvlm, lvl;
  int ihlp;

  double *vx_new;
  FIDX *vx_max;

  double mu, lambda;
  int stiffening;

  double *vx_tmp;

  /* Lame canstants for steel */
  mu         = 7.7e4;  /* 7.7e4; */
  lambda     = 1.15e5; /* 1.15e5; */
  stiffening = 1;  /* ==1 for real Lame', 
                      ==0 for stiffening of small elements */

  if (argc>=1)
    {
      printf("meshfile: %s \n", argv[1]);
    }
  else
    {
      printf("main: no mesh specified!\n");
      return FAIL;
    }
  err=mesh_read_file_t1( &msh1, argv[1] ); /* */
  FUNCTION_FAILURE_HANDLE( err, mesh_read_file_t1, main);

  /* set the goals and methods */
  level0  = 0;
  printf("level0=? "); if (scanf("%d",&ihlp)==1) level0=ihlp;
  level   = 1;
  printf("level=? "); if (scanf("%d",&ihlp)==1) level=ihlp;

  lvlm=level-level0;

  dim=msh1.dim;


  for (i=0; i<level0 ; i++)
    {
      err=mesh_refine_uniform_t1( &msh1 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t1, main);
    }

  /* reset hierarchy */
  msh1.hi_nr=0;
  for (j=0; j<msh1.eg_nr; j++)
    {
      msh1.edge[j*msh1.eg_w+MCT1EGLVL]=-1;
    }



    /* write the initial mesh */
  {
    FILE *mark;

    err=mesh_write_mesh_gnuplot_tx( &msh1, 27,
				    "visual/2d_mesh_gnuplot_ini", 1);
    FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

    /* touch the marker file */
    mark=fopen("visual/2d_new_mesh_mark", "w");
    fprintf(mark, "next please!\n");
    fclose(mark);

    sleep(3);
  }



  /* initialise the deformation */
  err=mesh_deform_t1_t2(&msh1, lvlm, &vx_new, &vx_max, 0, NULL);
  FUNCTION_FAILURE_HANDLE( err, mesh_deform_t2_t1, main);
fprintf( stderr,"mesh_deform klappt\n");

  /* convert to coarse t2 mesh */
  err=mesh_t1_to_t2(&msh1, &msh2);
  FUNCTION_FAILURE_HANDLE( err, mesh_t1_to_t2, main);

  /* allocate memory for the sparse matrices */
  TRY_MALLOC( Ks,  lvlm+1, struct sparse, main);

  err=projector1_alloc( &P, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, projector1_alloc, main);
  /* assemble the coarse meshes + refine */
  for (lvl=0; lvl<lvlm; lvl++)
    {
      /* before assembly, push in the moved node positions */
      vx_tmp=msh2.vertex;
      msh2.vertex=vx_new;

      vx_nr=msh2.vx_nr; 

      err=sparse_alloc( &Ks[lvl], dim*vx_nr, dim*K_LINE_DENSE);
      FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);


      err=assem_lame_tx( &Ks[lvl], NULL, &msh2, NULL, NULL,
			 lambda, mu, stiffening, 1, 2);
      FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);

      /* after assembly and before mesh refinement, put in the old
	 unmodified node positions */
      msh2.vertex=vx_tmp;
      

      err=mesh_refine_uniform_t2( &msh2 );
      FUNCTION_FAILURE_HANDLE( err, mesh_refine_uniform_t2, main);
    }
     
  vx_nr=msh2.vx_nr; 


  /* allocate memory for matrix Ks and the vectors */
  err=sparse_alloc( &Ks[lvlm], dim*vx_nr, dim*K_LINE_DENSE);
  FUNCTION_FAILURE_HANDLE( err, sparse_alloc, main);

  err=vector_alloc( &vx_old, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &x, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &rhs, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  err=vector_alloc( &nodem, dim*vx_nr );
  FUNCTION_FAILURE_HANDLE( err, vector_alloc, main);

  /* set vectors to zero */
  for (i=0; i<dim*vx_nr; i++)
    {
      x.V[i]  =0.0;
      nodem.V[i]    =0.0;
      rhs.V[i]=0.0;
    }


  vx_tmp=msh2.vertex;
  msh2.vertex=vx_new;

  /* assemble the system on the fine mesh */
  err=assem_lame_tx( &Ks[lvlm], &P, &msh2, &rhs, &nodem,
		     lambda, mu, stiffening, 2, 2);
  FUNCTION_FAILURE_HANDLE( err, assem_lame_tx, main);



  /* copy old vertices */
  for (d=0; d<dim; d++)
    for (i=0; i<vx_nr; i++)
      {
	vx_old.V[d*vx_nr+i]=msh2.vertex[i*MCT2VXLN+MCT2VXSTRT+d];
      }

  /* define the multigrid data */
  err=multilvl_init_tx( &msh2, dim, &ml, 2);
  FUNCTION_FAILURE_HANDLE( err, multilvl_init_tx, main);

  err=mg_init_tx( Ks, &msh2, &ml, &mg, &P);
  FUNCTION_FAILURE_HANDLE( err, mg_init_tx, main);

  /* generate the coarse grid matrix */
  {
    struct coarse_mat *cmat;
    FIDX *coarse_bc_nodes;
    FIDX n_c_bc=0;

    TRY_MALLOC(cmat, 1,  struct coarse_mat, main);
    TRY_MALLOC(coarse_bc_nodes, P.len, FIDX, main);
     
    for (i=0; i<P.len; i++)
      {
	FIDX dof, child;
	dof=P.V[i];
	MLVLFINDENTRY(child, dof, 0, ml);
	if (child>=0)
	  {
	    coarse_bc_nodes[n_c_bc]=child-ml.nlevl[0+1];
	    n_c_bc++;
	  }
      }
    printf("n_c_bc=%d\n", (int) n_c_bc);

    err=coarse_mat_set( &Ks[0], n_c_bc, coarse_bc_nodes,
			1, cmat );
    FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );
    mg.cmat=cmat;
    
    free(coarse_bc_nodes);
  } /* */
  
  
  mg.vcycles=1;
  mg.smooths=1;
  if (stiffening==0)
    mg.CGC_scale=0.25;
  else
    mg.CGC_scale=1.0;

  {
    /* write the initial mesh */
    FILE *mark;

    err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_ini", 2);
    FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

    /* touch the marker file */
    mark=fopen("visual/2d_new_mesh_mark", "w");
    fprintf(mark, "next please!\n");
    fclose(mark);

    sleep(3);
  }

  // sparse_mat_write_file( &K, "visual/Klinel.txt"); 


  /* solve the equation system */
  if (0==1) /* variant 1: direct solve (good for coarse mesh) */
    {
      /* invert the stiffness matrix */
      err=coarse_mat_set( &Ks[lvlm], P.len, P.V,
			  1, &K_inv );
      FUNCTION_FAILURE_HANDLE( err, coarse_mat_set, main );

      /* get the rhs */
      /* rhs=K*x */
      err=sparse_mul_mat_vec( &Ks[lvlm], &nodem, &rhs);
      FUNCTION_FAILURE_HANDLE( err, sparse_mul_mat_vec, main);  

      /* project rhs */
      for (i=0; i<P.len; i++)
	{
	  FIDX dof = P.V[i];
	  rhs.V[dof]=0.0;
	}
      err=coarse_mat_solve( &K_inv, NoTrans, &rhs, &x);
      FUNCTION_FAILURE_HANDLE(err, coarse_mat_solve, main);

      /* construct the step */
      for (i=0; i<dim*vx_nr; i++)
	{
	  nodem.V[i]-=x.V[i];
	}

      coarse_mat_free(&K_inv);
    }
  else /* variant 2: PCG */
    {
      int iter;
      double resi;

      struct bpxdata bpx;

      err=bpx_init_tx(&bpx, &ml);
      FUNCTION_FAILURE_HANDLE( err, bpx_init_tx, main);
      bpx.msh = &msh1;
      bpx.P   = &P;

      /* rhs is just zero, use nodem directly */
      /*err=PCG( 10000, 2, 1e-5, 1, &nodem, &resi, &iter,
	sparse_mul_mat_vec, projector1_no_precon,
	&K, &rhs, &P); /* */
      err=PCG( 10000, 2, 1e-8, 1, &nodem, &resi, &iter,
	       sparse_mul_mat_vec, gen_proj_MG_tx,
	       &Ks[lvlm], &rhs, &mg); /* */
      /* err=PCG( 10000, 2, 1e-5, 1, &nodem, &resi, &iter,
	 sparse_mul_mat_vec, gen_proj_bpx_tx,
	 &Ks[lvlm], &rhs, &bpx); /* */
      FUNCTION_FAILURE_HANDLE( err, PCG, main);
      
      printf("PCG_lin_elast:  it=%3d  |res|=%8.1e\n", iter, resi);
      bpx_free(&bpx);
    }
		  
  /* the node movement computation is done, allpy it to the mesh */
  for (i=0; i<vx_nr; i++)
    for (d=0; d<dim; d++)
      {
	msh2.vertex[i*msh2.vx_w+MCT2VXSTRT+d] += nodem.V[d*vx_nr+i];
      }


  {
    /* write the resulting mesh */
    FILE *mark;

    err=mesh_write_mesh_gnuplot_tx( &msh2, 27,
				    "visual/2d_mesh_gnuplot_def", 2);
    FUNCTION_FAILURE_HANDLE(err, mesh_write_mesh_gnuplot_tx,
			    main);

    /* touch the marker file */
    mark=fopen("visual/2d_new_mesh_mark", "w");
    fprintf(mark, "next please!\n");
    fclose(mark);
  }
  


  /* free some stuff */
  mg_free(&mg);
  multilvl_free(&ml);

  for (lvl=0; lvl<=lvlm; lvl++)
    sparse_free(&Ks[lvl]);
  free(Ks);

  /* free(vx_new) is not necessary, cause of mesh_free(&msh2) where the target of vx_new is included
     so only free(vx_tmp) is necessary */
  free(vx_tmp);

  projector1_free(&P);

  vector_free(&rhs);
  vector_free(&nodem);
  vector_free(&vx_old);
  vector_free(&x);

  mesh_free(&msh2);
  mesh_free(&msh1);

  return SUCCESS;


}
