/***********************************************************************

    This file is part of
    FEINS, Finite Element Incompressible Navier-Stokes solver,
    which is expanding to a more general FEM solver and toolbox,
    Copyright (C) 2003--2008, Rene Schneider 
    <rene.schneider@mathematik.tu-chemnitz.de>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.

    Minor contributions to this program (for example bug-fixes and
    minor extensions) by third parties automatically transfer the
    copyright to the general author of FEINS, to maintain the
    possibility of commercial re-licensing. If you contribute but wish
    to keep the copyright of your contribution, make that clear in
    your contribution!

    Non-GPL licenses to this program are available upon request from
    the author.

************************************************************************/
#ifdef PARALLEL
#include <mpi.h>
#endif

#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "feins_macros.h"
#include "solver.h"

#define RESLEN 400
#define PAR_NR 5
#define MAX_NR 4096

#define WORKTAG         1
#define DIETAG          2
#define ERRORTAG        3

void master( int argc, char *argv[] );
void slave();

int main(int argc, char *argv[])
{
#ifdef PARALLEL
    int size,myrank;
    
    MPI_Init(&argc, &argv);     /* initialize MPI */
    MPI_Comm_size(MPI_COMM_WORLD,&size);

    if (size<=1)
    {
        /* the master works only if he has slaves. */
        fprintf(stderr,
                "This program works only with at least 2 nodes.\n");
        exit(0);
    }

    MPI_Comm_rank(
        MPI_COMM_WORLD,     /* always use this */
        &myrank);           /* process rank, 0 thru N-1 */

    if (myrank == 0) {    
        master(argc, argv);
    } else {
        slave();
    }
    
    MPI_Finalize();             /* cleanup MPI */
}

void master( int argc, char *argv[] )
{
#endif
    int    ntasks, rank;
    int    sent,received;

    /* variables specifying the work */
    int tot_tasks, i;
    double work[MAX_NR*PAR_NR];
    FIDX level;
    double nu;

    /* variables holding the result */
    double results[RESLEN];

    char *name,*repname;   /* Zum puffern der Dateinamen */
    FILE *report,*out; /* zwei Dateien: Protokoll,Ausgabe */

#ifdef PARALLEL
    MPI_Status          status;
    MPI_Comm_size(
        MPI_COMM_WORLD,     /* always use this */
        &ntasks);           /* #processes in application */
    
    printf("#proc=%d \n#arguments: %d \n", ntasks, argc);
#endif    
    if (argc>1)
    {
        /* the argument is used as filename */
        repname=(char *) malloc(strlen(argv[argc-1])+5);
        name   =(char *) malloc(strlen(argv[argc-1])+5);
        if (name==NULL)
        {
            fprintf(stderr,"unable to malloc string for filename\n");
#ifdef PARALLEL
            /*
             * Tell all the slaves to exit.
             */
            for (rank = 1; rank < ntasks; ++rank)
            {
                MPI_Send(0, 0, MPI_INT, rank, DIETAG, MPI_COMM_WORLD);
            }
#endif
            return ;
        }
        strcpy(repname, argv[argc-1]);
        strcat(repname,".rep");
        printf("reportfile: %s\n",repname);
        report=fopen(repname,"w");

        strcpy(name, argv[argc-1]);
        strcat(name,".out");
        out=fopen(name,"w");
    }
    else
    {
        report=NULL;
        out=NULL;
        fprintf(stderr,"no filename given!\n");
    }
    
        
        
    if ((out==NULL)||(report==NULL))
    {
        fprintf(stderr,"error opening one of the files \nEND\n");
#ifdef PARALLEL
        /*
         * Tell all the slaves to exit.
         */
        for (rank = 1; rank < ntasks; ++rank)
        {
            MPI_Send(0, 0, MPI_INT, rank, DIETAG, MPI_COMM_WORLD);
        }
#endif
        return ;
    }

    /*
     * define the work
     */
    tot_tasks=0;
    for (level=8; level>=3; level--)
      for (nu=1.0/128.0; nu<=0.13; nu*=2)
	{
	  work[tot_tasks*PAR_NR+0]=(double)level; /* level */
	  work[tot_tasks*PAR_NR+1]=1e-5;          /* impeps */
	  work[tot_tasks*PAR_NR+2]=1e-6;          /* innereps */
	  work[tot_tasks*PAR_NR+3]=1e-6;          /* adjinnereps */
	  work[tot_tasks*PAR_NR+4]=nu;            /* nu */
	  tot_tasks++;

	  if (tot_tasks>MAX_NR)
	    {
	      fprintf(stderr,"tot_tasks>MAX_NR, die\n");
#ifdef PARALLEL
	      for (rank = 1; rank < ntasks; ++rank)
		{
		  MPI_Send(0, 0, MPI_INT, rank, DIETAG, MPI_COMM_WORLD);
		}
#endif
	      return ;
	    }
	}
        
    /*
     * Seed the slaves / do the work.
     */
    sent=0;
    received=0;
    rank=1;


#ifdef PARALLEL
#else
    /* do the work */
    while (sent <tot_tasks)
    {
      int err;
      err=navstosolver( PAR_NR, &work[sent*PAR_NR], RESLEN, results);
      
      if (err==SUCCESS)
	{
	  fprintf(report,"OK...\n");
	  for (i=0; i<RESLEN; i++)
	    fprintf(out,"%e   ",results[i]);
	  fprintf(out,"\n");
	}
      else
	{
	  fprintf(report,"error reported, work #%d\n", sent);
	  return;
	}

      fprintf(report,"task %3d done \n",sent);
      { 
	fclose(out);
	fclose(report);
	report=fopen(repname,"a");
	out=fopen(name,"a");
      }

      sent++;
    }
    fprintf(report,"all tasks done => END\n");
    fclose(report);
    fclose(out);
    free(name);
    return;
#endif




#ifdef PARALLEL
    while ((rank< ntasks)&&(sent<tot_tasks))
    {
        /* got_work */

        fprintf(report,"send %2d tast %3d\n",rank,sent+1);

        MPI_Send(&work[sent*PAR_NR],    /* message buffer */
                 PAR_NR,           /* nr data items */
                 MPI_DOUBLE,       /* data items are double */
                 rank,             /* destination process rank */
                 WORKTAG,          /* user chosen message tag */
                 MPI_COMM_WORLD);  /* always use this */
        rank++;
        sent++;
    }

    /*
     * Receive a result from any slave and dispatch a new work
     * request until work requests have been exhausted.
     */
    while (sent <tot_tasks)
    {/* valid new work request */

        MPI_Recv(results,        /* message buffer */
		 RESLEN,         /* nr data items */
		 MPI_DOUBLE,     /* of type ? */
		 MPI_ANY_SOURCE, /* receive from any sender */
		 MPI_ANY_TAG,    /* any type of message */
		 MPI_COMM_WORLD, /* always use this */
		 &status);       /* received message info */
	received++;
        
        fprintf(report,"from  %2d message tag %d\n",status.MPI_SOURCE,
                status.MPI_TAG);
        if (status.MPI_TAG==ERRORTAG)
        {
            fprintf(report,"error reported, from %d\n",
		    status.MPI_SOURCE);
        }
        else 
        {
            fprintf(report,"OK...\n");
	    for (i=0; i<RESLEN; i++)
	      fprintf(out,"%e   ",results[i]);
            fprintf(out,"\n");
        }

        fprintf(report,"send %2d task %3d\n",status.MPI_SOURCE,sent+1);
        { 
            fclose(out);
            fclose(report);
            report=fopen(repname,"a");
            out=fopen(name,"a");
        }


        MPI_Send(&work[sent*PAR_NR],    /* message buffer */
                 PAR_NR,           /* nr data items */
                 MPI_DOUBLE,       /* data items are double */
                 status.MPI_SOURCE,/* destination process rank */
                 WORKTAG,          /* user chosen message tag */
                 MPI_COMM_WORLD);  /* always use this */
        sent++;

        /* get_next_work_request */
    }
    fprintf(report, "all tasks sent.\n");
    
    /*
     * Receive results for outstanding work requests.
     */
    while (received<sent)
    {
        MPI_Recv(results,        /* message buffer */
                 RESLEN,         /* nr data items */
                 MPI_DOUBLE,     /* of type ? */
                 MPI_ANY_SOURCE, /* receive from any sender */
                 MPI_ANY_TAG,    /* any type of message */
                 MPI_COMM_WORLD, /* always use this */
                 &status);       /* received message info */
        received++;
 
       fprintf(report,"from   %2d message tag %d\n",status.MPI_SOURCE,
                status.MPI_TAG);        

        if (status.MPI_TAG==ERRORTAG)
        {
            fprintf(report,"error reported, from %d\n",
		    status.MPI_SOURCE);
        }
        else 
        {
            fprintf(report,"OK...\n");
	    for (i=0; i<RESLEN; i++)
	      fprintf(out,"%e   ",results[i]);
            fprintf(out,"\n");
        }

        fclose(out);
        fclose(report);
        report=fopen(repname,"a");
        out=fopen(name,"a");
    }

    /*
     * Tell all the slaves to exit.
     */
    for (rank = 1; rank < ntasks; ++rank) {
        MPI_Send(0, 0, MPI_INT, rank, DIETAG, MPI_COMM_WORLD);
    }
    fprintf(report,"all tasks done => END\n");
    fclose(report);
    fclose(out);
    free(name);
}

void slave()
{
    /* variables specifying the work */
    int tot_tasks;
    double this_work[PAR_NR];

    double results[RESLEN];

    MPI_Status          status;
    int err;

    for (;;) {
        MPI_Recv(this_work, PAR_NR, MPI_DOUBLE, 0, MPI_ANY_TAG,
                 MPI_COMM_WORLD, &status);
        /*
         * Check the tag of the received message.
         */
        if (status.MPI_TAG == DIETAG) {
            return;
        }
	
	/* decode the work */

	/* do the work */
        err=navstosolver( PAR_NR, this_work, RESLEN, results);
        
        if (err==SUCCESS)
        {
            MPI_Send(results, RESLEN, MPI_DOUBLE, 0, 0, MPI_COMM_WORLD);
        }
        else
        {
            MPI_Send(results, RESLEN, MPI_DOUBLE, 0, ERRORTAG, MPI_COMM_WORLD);
        }
    }
#endif
}

